# Sine

Sine, abbreviated *sin*, is a [trigonometric](trigonometry.md) [function](function.md) that, simply stated, models a smooth oscillation, it is one of the most important and basic functions in geometry, [mathematics](math.md) and [physics](physics.md), and of course in [programming](programming.md) as well. Along with [cosine](cos.md), [tangent](tan.md) and [cotangent](cot.md) it belongs to a group of functions that can be defined by ratios of sides of a right [triangle](triangle.md) depending on one of the angles in it (hence *trigonometric* -- "triangle measuring"). If some measurement looks like a sine function, we say it is *harmonic*. This observation is very common in nature and technology, e.g. a weight on a spring goes up and down following this function's shape, [alternating current](ac.md) voltage has the sine shape (because it is generated by a circular motion), the amount of light falling onto a surface depends on the light angle via sine function etc.

The function is most commonly defined using a right triangle as follows. Consider the following triangle:

```
          /|
         / |
        /  |
      c/   |
      /    |a
     /     |
    /     _|
   /A____|_|
       b
```

*Sin(A)*, where *A* is the angle between side *b* and *c*, is the ratio *a* / *c*. The function can be defined in many other ways, for example it is the curve we get when tracking only one direction (e.g. horizontal) of a point moving along a circular path (well, in the end it's actually related to the triangle definition) -- imagine something making a circular motion while casting a shadow onto a nearby wall (in a direction that's in the plane of rotation), the shadow will go up and down by sine of time. It can also be defined as a solution to some [differential equations](differential_equation.md) etc.

The graph of the sine function is following:

```
                  ^ sin(x)
                  |
                1_|_
                  |     .--'''--.
      -1/2 pi     | _.''         ''._        3/2 pi
.________|________.'________|________'|________|________.' --> x
 '._     |     _.'|0        |         |'._     |     _.'|
    ''--___--''  _|_      1/2 pi      pi  ''--___--''  2 pi
               -1 |
```

Commonly we'll observe a sine function in places where variable oscillates naturally, gradually, i.e. where something goes there and back repeatedly but in a gradual, smooth manner, slowing down as it's approaching one end, coming to stop for a while, then reversing direction, slowly accelerating the other way and so on. Very rare in nature something comes there and back with constant speed, coming to halt immediately and reversing directions instantly.

**Why the fuck are there these [pi](pi.md) values on the x line???** Nubs often can't comprehend this. These pi values are values in **[radians](radian.md)**, units of measuring angles where *2 pi* is the full angle (360 degrees). In fact sine is sometimes shown with [degrees](degree.md) instead of radians (so imagine 90 degrees on the line where there is 1/2 pi etc.), but mathematicians prefer radians. **But why are there angles in the first place???** Why doesn't it go e.g. from 0 to 1 like all other nice functions? Well, it's because of the relation to geometry and circles, remember the fucking triangle above... if you define sine with a circle it all repeats after *2 pi*. Imagine the shadow on the wall going up and down -- the cycle will repeat when the rotating thing makes 360 degrees all round. Just draw some picture if you don't get it.

Some additional facts and properties regarding the sine functions are:

- The domain are all [real numbers](real_number.md), the [codomain](codomain.md) are real numbers in interval <-1,1> (including both bounds).
- It is an [odd function](odd_function.md) (*-sin(x) = sin(-x)*).
- It is periodic, with a period of 2 [pi](pi.md).
- Sine is just shifted [cosine](cos.md), i.e. *sin(x) = cos(x - 1/2 pi)*
- Its inverse function is [arcus sine](asin.md), abbreviated *asin*, also written as *sin^-1* -- this function tells you what argument you need to give to sin to get a specific result number. It's actually an inverse of only part of the sine function because the whole sine function can't be inverted, it isn't [bijective](bijection.md).
- [Derivative](derivative.md) of *sin(x)* is *cos(x)*, the [integral](integral.md) of *sin(x)* is *-cos(x)*.
- By adding many differently shifted and scaled sine functions we can create basically any other function, see e.g. [cosine transform](cosine_transform.md).
- Sine and [cosine](cos.md) functions are used to draw [circles](circle.md). If you plot points with *x* coordinate equal to *sin(t)* and *y* coordinate equal to *cos(t)* for *t* going from 0 to *2 * pi*, you'll get a unit circle.
- *sin(x)^2 + cos(x)^2 = 1*

Some values of the sine (and cosine) function are:

| x (rad)   | x (deg)  | sin(x)             | cos(x)               |
|-----------|----------|--------------------|----------------------|
| -2 * pi   | -360     | 0                  | 1                    |
| -pi * 3/2 | -270     | 1                  | 0                    | 
| -pi       | -180     | 0                  | -1                   |
| -pi / 2   | -90      | -1                 | 0                    |
| 0         | 0        | 0                  | 1                    |
| pi / 12   | 15       | ~0.258             | ~0.965               |
| pi / 6    | 30       | 0.5                | sqrt(3)/2 ~= 0.866   |
| pi / 4    | 45       | 1/sqrt(2) ~= 0.707 | 1/sqrt(2) ~= 0.707   |
| pi / 3    | 60       | sqrt(3)/2 ~= 0.866 | 0.5                  |
| pi * 5/12 | 75       | ~0.965             | ~0.258               |
| pi / 2    | 90       | 1                  | 0                    |
| pi * 7/12 | 105      | ~0.965             | ~-0.258              |
| pi * 2/3  | 120      | sqrt(3)/2 ~= 0.866 | -0.5                 |
| pi * 3/4  | 135      | 1/sqrt(2) ~= 0.707 | -1/sqrt(2) ~= -0.707 |
| pi * 5/6  | 150      | 0.5                | -sqrt(3)/2 ~= -0.866 |
| pi * 11/12| 165      | ~0.258             | ~-0.965              |
| pi        | 180      | 0                  | -1                   |
| pi * 3/2  | 270      | -1                 | 0                    |
| 2 * pi    | 360      | 0                  | 1                    |

## Programming

In [programming languages](programming_language.md) the sine function is generally available in some math [library](library.md), for example in [C](c.md) the function `sin` is in `math.h`. Spare yourself bugs, **always check if your sin function expects [radians](radian.md) or degrees!**

**Want to make your own sine function for whatever reason (performance, curiosity, ...)?** Then firstly consider what you expect from it. If you want a small, fast and perhaps integer only `sin` function (the one we'd prefer in [LRS](lrs.md)) that doesn't need extreme accuracy, consider using a **[look up table](lut.md)**. You simply precompute the values of the sine function into a static table in memory and the function just retrieves them when called -- this is super fast. Note that you can save a lot of space by **only storing sine values between 0 and 1/2 pi**, the remaining parts of the function are just different transformations of this part. You can further save space and/or make the function work with [floats](float.md) by further [interpolating](interpolation.md) (even just linearly) between the stored values, for example if `sin(3.45)` is called and you only have values stored for `sin(3.4)` and `sin(3.5)`, you simply average them.

Lot of times, e.g. in many calculators where speed isn't really critical, sine is computed using [Taylor series](taylor_series.md) (no, it's not named after Taylor Swift) -- a sum of infinitely many terms of which if we take the first *N*, we get an [approximation](approximation.md) of the function (the more terms we add, the more precise we get). For sine the series is

*sin(x) = x - x^3 / 3! + x^5 / 5! - x^7 / 7! + ...*

Adding just the first 3 terms (*x - x^3 / 6 + x^5 / 120*) already gives a very accurate approximation in range <-pi/2,pi/2> (error < 0.5 %). Here is a [C](c.md) function that uses this to compute an 8bit sine (the magic numbers are made so as to incorporate pi while using power of two divisors, also note the use of many operations that will make the function relatively slow):

```
// x = 255 means full angle, returns 0 to 255
unsigned char sin8(unsigned char x)
{
  int a = x;
  char flip = 0;

  if (a > 127)
  {
    a -= 128;
    flip = 1;
  }

  if (a > 63)
    a = 128 - a;

  int result = (411999 * a) - (a * a * a * 41);

  a /= 4;

  a = a * a * a * a * a;

  result = (a + result) / 131072;
  return flip ? (127 - result) : (127 + result);
}
```

If you just need a super fast and very rough sine-like value, there exists an **ugly engineering approximation** of sine that can be useful sometimes, it says that

*sin(x) = x, for small x*

Indeed, sine looks similar to a mere line near 0, but you can see it quickly diverges.

Very rough and fast approximations e.g. for primitive music synthesis can be done with the traditional very basic [square](square_function.md) or [triangle](triangle_function.md) functions. The following is a simple 8bit linear approximation that's more accurate than square or triangle (approximates sine with a linear function in each octant):

```
unsigned char sinA(unsigned char x)
{
  unsigned char quadrant = x / 64;

  x %= 64;

  if (quadrant % 2 == 1)
    x = 63 - x;

  x = x < 32 ? (2 * x + x) : (64 + x);

  return quadrant <= 1 ? (128 + x) : (127 - x);
}
```

Similar approximation can be made with a quadratic curve, the following is a modification of the above function that does this (notice that now we need at least 16 bits for the computation so the data type changed to int): { I quickly made this just now, maybe it can be improved. ~drummyfish }

```
int sinA(int x)
{
  unsigned char quadrant = x / 64;

  x %= 64;

  if (quadrant % 2 == 1)
    x = 63 - x;

  x -= 63;
  x = (x * x) / 32;

  return quadrant <= 1 ? (255 - x) : x;
}
```

Sine can also be surprisingly accurately approximated with the [smoothstep](smoothstep.md) function, which is just a polynomial *3 * x^2 - 2 * x^3*.

TODO: code for that

Furthermore there exist other nice approximations, such as the extremely accurate **Bhaskara I's approximation** (angle in radians): *sin(x) ~= (16 * x * (pi - x)) / (5 * pi^2 - 4 * x * (pi - x))*. (This formula is actually more elegant for cosine, so it may be even better to consider using that.) Here is a [C](c.md) [fixed point](fixed_point.md) implementation:

```
#define UNIT 1024
#define PI ((int) (UNIT * 3.14159265))

/* Integer sine using Bhaskara's approx. Returns a number
in <-UNIT, UNIT> interval. Argument is in radians * UNIT. */

int sinInt(int x)
{
  int sign = 1;

  if (x < 0) // odd function
  {
    x *= -1;
    sign = -1;
  }

  x %= 2 * PI;

  if (x > PI)
  {
    x -= PI;
    sign *= -1;
  }

  x *= PI - x;

  return sign * (16 * x) / ((5 * PI * PI - 4 * x) / UNIT);
}
```

Mainstream way of implementing [floating point](float.md) sine (but potentially fixed point too) is through [Taylor series](taylor_series.md), i.e. with a [polynomial](polynomial.md) of order *N* that has first *N* [derivatives](derivative.md) identical to the approximated function near some given point. For *sin(x)* near *x = 0* this series is:

*sin(x) = x/1! - x^(3)/3! + x^(5)/5! - x^(7)/7! + ...*

Here is a simple implementation using fixed number of terms, which gives quite precise results:

```
double sinF(double x)
{
#define _PI 3.141593
  // adjust argument:

  if (x < -1 * _PI / 2)
    x = -1 * x + _PI;

  if (x > _PI / 2)
  {
    x -= _PI / 2;
    int part = x / _PI;
    x -= part * _PI;
    x = (part % 2) ? (-1 * _PI / 2 + x) : (_PI / 2 - x);
  }
#undef _PI

  // Taylor series:

  double x2 = x * x, r = x;

  x *= x2;
  r -= x / 6;
  x *= x2;
  r += x / 120;
  x *= x2;
  r -= x / 5040;
  x *= x2;
  r += x / 362880;
  return r;
}
```

