#include <stdio.h>
#include <stdlib.h>
#include <gccore.h>
#include <ogc/machine/processor.h>
#include <fat.h>
#include <sdcard/wiisd_io.h>

#include <unistd.h>
#include <string.h>

// for directory parsing and low-level file I/O
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>

#include "input.h"

#include "loader_bin.h"

// to start, all we need to do is jump to the Devolution blob
#define LAUNCH() ((void(*)(void))loader_bin)()

// constant value for identification purposes
#define CONFIG_SIG        0x3EF9DB23

// version may change when future options are added
#define CONFIG_VERSION    0x0201

// option flags
// v1.10
#define CONFIG_WIFILOG    (1<<0)
#define CONFIG_WIDE       (1<<1)
#define CONFIG_NOLED      (1<<2)
// v1.11
// patch F-Zero GX to F-Zero AX
#define CONFIG_FZERO_AX   (1<<3)
// This option may fix games that run faster than they are meant to, but it may also degrade performance
// below the original speed.
// DO NOT USE THIS OPTION UNLESS YOU HAVE CONFIRMED THE GAME YOU ARE TESTING RUNS TOO FAST WITH
// THE *CURRENT* BUILD OF DEVOLUTION. IT _WILL_ CAUSE SLOWDOWNS - THAT'S THE POINT OF IT.
#define CONFIG_TIMER_FIX  (1<<4)
// v1.12
// Use direct mapping of CC/PS3/OUYA buttons (a=a, b=b, etc.)
// otherwise b=a, y=b, x=y, a=x
#define CONFIG_D_BUTTONS  (1<<5)
// v2.00
// setting memcard_cluster to 1 means use NAND ("/shared2/sys/memcard.bin")
// NAND file must already be present and set to the desired memcard size
//
// scale up horizontally by 32 pixels to reduce large overscan borders at left/right sides
// no effect if the overscan area is already reduced (eg. most first party games already do this)
#define CONFIG_CROP_OVERSCAN (1<<6)
// apply a minimum delay to large reads to emulate the real disc drive speed
#define CONFIG_DISC_DELAY    (1<<7)
// write usage times to the system menu's playlog
// this will force Devolution to launch title 1-2 (system menu) on exit,
// which is required to allow the system playlog to update itself
#define CONFIG_PLAYLOG       (1<<8)
// v2.01
// force 60Hz mode to use progressive mode if component cable is used
#define CONFIG_FORCE_480P    (1<<9)
// force 50Hz mode to use progressive mode if component cable is used
#define CONFIG_FORCE_576P    (1<<10)

typedef struct global_config
{
	uint32_t signature;
	uint16_t version;
	uint16_t device_signature;
	uint32_t memcard_cluster;
	uint32_t disc1_cluster;
	uint32_t disc2_cluster;
	uint32_t options;
} gconfig;

// this is where Devolution looks for the config struct
static gconfig *DEVO_CONFIG = (gconfig*)0x80000020;

// basic video setup, nothing special here
static void initialize(GXRModeObj *rmode)
{
	static void *xfb = NULL;

	if (xfb)
		free(MEM_K1_TO_K0(xfb));

	xfb = SYS_AllocateFramebuffer(rmode);
	VIDEO_ClearFrameBuffer(rmode, xfb, COLOR_BLACK);
	xfb = MEM_K0_TO_K1(xfb);
	console_init(xfb,20,20,rmode->fbWidth,rmode->xfbHeight,rmode->fbWidth*VI_DISPLAY_PIX_SZ);
	VIDEO_Configure(rmode);
	VIDEO_SetNextFramebuffer(xfb);
	VIDEO_SetBlack(FALSE);
	VIDEO_Flush();
	VIDEO_WaitVSync();
	VIDEO_WaitVSync();
}

// creates a list of *.iso files in the specified directory
// allows the user to select one and reads required information from it
static void select_game(const char *dirname, int real_memcard)
{
	u32 count=0;
	int select=0;
	DIR *dir;
	struct dirent *entry;
	char full_path[256];
	int data_fd;
	u8 *lowmem = (u8*)0x80000000;
	struct stat st;
	FILE *iso_file;

	sprintf(full_path, "disk:/%s", dirname);

	dir = opendir(full_path);
	if (dir==NULL)
		return;

	// count how many .iso files exist
	while ((entry = readdir(dir)))
	{
		size_t length = strlen(entry->d_name);
		if (length > 4 && stricmp(entry->d_name+length-4, ".ISO")==0)
		{
			if (!count)
				strcpy(full_path, entry->d_name);
			count++;
		}
	}

	printf("Select Game: <%s>", full_path);

	if (count)
	{
		u32 pressed;
		u32 index;
		do
		{
			VIDEO_WaitVSync();
			get_input(NULL, &pressed);

			if (!pressed)
				continue;

			if (pressed & INPUT_BUTTON_CANCEL)
			{
				printf("\nExiting...\n");
				exit(0);
			}

			if (pressed & INPUT_BUTTON_RIGHT)
				select++;
			else if (pressed & INPUT_BUTTON_LEFT)
				select--;
			else if (pressed & INPUT_BUTTON_DOWN)
				select += 5;
			else if (pressed & INPUT_BUTTON_UP)
				select -= 5;

			while (select < 0)
				select += count;
			while (select >= count)
				select -= count;

			rewinddir(dir);
			index = 0;
			while ((entry = readdir(dir)))
			{
				size_t length = strlen(entry->d_name);
				if (length > 4 && stricmp(entry->d_name+length-4, ".ISO")==0 && select==index++)
					break;
			}
			printf("\rSelect Game: <%s>\x1b[K", entry->d_name);
		} while (!(pressed & INPUT_BUTTON_OK));
		printf("\n");

		// fill out the Devolution config struct
		memset(DEVO_CONFIG, 0, sizeof(*DEVO_CONFIG));
		DEVO_CONFIG->signature = CONFIG_SIG;
		DEVO_CONFIG->version = CONFIG_VERSION;

		// use wifi logging if USB gecko is not found in slot B
		DEVO_CONFIG->options |= CONFIG_WIFILOG;

		// Add GC playtimes to the system menu's daily activity log
		// (this forces "return to menu" instead of "return to loader")
//		DEVO_CONFIG->options |= CONFIG_PLAYLOG;

		// Force progressive modes if wii is set to 480P
		// this option won't work if a component(/HDMI) cable is not connected
		if (CONF_GetProgressiveScan() > 0)
		{
			DEVO_CONFIG->options |= CONFIG_FORCE_480P;
			DEVO_CONFIG->options |= CONFIG_FORCE_576P;
		}

		// assume widescreen TVs have small horizontal overscan
		// i.e. if wii widescreen mode is on, reduce overscan area
		// (user might want to force this if playing on the wiiu gamepad regardless of TV behaviour?)
		if (CONF_GetAspectRatio() == CONF_ASPECT_16_9)
			DEVO_CONFIG->options |= CONFIG_CROP_OVERSCAN;

		sprintf(full_path, "disk:/%s/%s", dirname, entry->d_name);

		// use direct mapping of CC/PS3/OUYA buttons (A=a, B=b, etc.)
		DEVO_CONFIG->options |= CONFIG_D_BUTTONS;
		
		//Turn off LED lights by default
		DEVO_CONFIG->options |= CONFIG_NOLED;
		
		// get the FAT starting cluster (and device ID) for the ISO file
		stat(full_path, &st);

		// read 32 bytes of disc ID to the start of MEM1
		iso_file = fopen(full_path, "rb");
		fread(lowmem, 1, 32, iso_file);
		fclose(iso_file);

		// set the device so Devolution knows what to use. This is a 4-byte ID in libogc ('WISD' or 'WUSB')
		// but devkitpro is made of fail and st.st_dev is only 2 bytes wide, so that's what we use.
		// THIS DRIVE IS USED FOR BOTH DISK AND MEMCARD EMULATION, IT IS NOT POSSIBLE TO USE SEPARATE DRIVES
		DEVO_CONFIG->device_signature = st.st_dev;       // 'SD' or 'SB', anything else will not be recognized here

		// set the starting FAT cluster of the disc 1 ISO file
		// if disc2 is accidentally used here, Devolution will search for disc1 in the /games/ directory
		// disc2 cannot be used for initially launching the game
		DEVO_CONFIG->disc1_cluster = st.st_ino;          // set starting cluster for first disc ISO file

		// if there is an ISO file for the second disc, the cluster for it should be set here.
		// otherwise if it is zero Devolution will search the /games/ directory for the ISO file
		// when it has to change discs.
		//DEVO_CONFIG->disc2_cluster = second_iso_cluster;

		// if the clusters for the ISO files aren't set Devolution will search all .ISO files in the /games/
		// directory for any that match the game ID found in memory.

		// make sure these directories exist ON THE DRIVE THAT HOLDS THE ISO FILE, they are required for Devolution to function correctly
		mkdir("disk:/apps", 0777);
		mkdir("disk:/apps/gc_devo", 0777);

		if (!real_memcard)
		{
			// find or create a memcard file for emulation
			// this file can be located anywhere since it's passed by cluster, not name
			// it must be at least 512KB (smallest possible memcard = 59 blocks)

			// IT MUST BE LOCATED ON THE SAME DRIVE AS THE ISO FILE!
			// IF YOU FUCK THIS UP (I'M LOOKING AT YOU, CFG-LOADER) YOU RISK DATA CORRUPTION
			strcpy(full_path, "disk:/apps/gc_devo/memcard.bin");

			// check if file doesn't exist
			if (stat(full_path, &st) == -1 || st.st_size < (1<<19))
			{
				// need to enlarge or create it
				data_fd = open(full_path, O_WRONLY|O_CREAT);
				if (data_fd>=0)
				{
					// try to make it 16MB (largest possible memcard = 2043 blocks)
					printf("Resizing memcard file...\n");
					ftruncate(data_fd, 16<<20);
					if (fstat(data_fd, &st)==-1 || st.st_size < (1<<19))
					{
						// it isn't big enough. Give up.
						st.st_ino = 0;
					}
					close(data_fd);
				}
				else
				{
					// couldn't open or create the memory card file
					st.st_ino = 0;
				}
			}

			// set FAT cluster for start of memory card file
			// if this is 0 memory card emulation will not be used
			// if this is 1 NAND memory card emulation will be used ("/shared2/sys/memcard.bin")
			// no real files can occupy clusters 0 or 1 on a FAT partition so these "reserved"
			// values cannot conflict with real files
			DEVO_CONFIG->memcard_cluster = st.st_ino;
		}
	}

	// flush disc ID and Devolution config to physical memory
	DCFlushRange(lowmem, 64);

	closedir(dir);
}

int main(int argc, char **argv)
{
	GXRModeObj *rmode, *vidmode;
	int real_memcard;
	s32 card_sector_size, card_size;

//  Enable for USB gecko logging
//	CON_EnableGecko(1, 1);

	VIDEO_Init();

	rmode = VIDEO_GetPreferredMode(NULL);

	initialize(rmode);

	printf("\n\n\n"); // move past screen border

	input_startup();

	// the Devolution blob has an ID string at offset 4
	printf("%s\n\n", (const char*)loader_bin+4);

	// make sure AHBPROT is disabled
	if ((read32(0xCD800038) | read32(0xCD80003C))==0)
	{
		printf("This app requires AHBPROT access!\nPlease run it from HBC using the correct meta.xml file.\n");
		goto wait_for_exit;
	}

	printf("Devolution Launcher CCBR ---\n\n");
	printf("Classic Controllers will flash when inactive.\nSelect controller slot with PLUS and confirm with HOME to activate controller.\n\n");
	printf("Commands for inactive Classic Controllers:\nMINUS&DPAD-RIGHT toggles slot LED setting. MINUS&A toggles rumble on/off.\n\n");

	CARD_Init(NULL, NULL);

	real_memcard = CARD_ProbeEx(CARD_SLOTA, &card_size, &card_sector_size);
	if (real_memcard != CARD_ERROR_WRONGDEVICE && real_memcard != CARD_ERROR_NOCARD)
	{
		card_size <<= 17;
		printf("Memory card found in slot A, %d sectors of %dKB each (%d bytes).\n", card_size/card_sector_size, card_sector_size>>10, card_size);
		printf("Memory card emulation will be disabled.\n\n");
		real_memcard = 1;
	}
	else
	{
		printf("No memory card found in slot A (%d), memory card emulation will be used.\n\n", real_memcard);
		real_memcard = 0;
	}

	// set game ID to zero
	write32(0, 0);

	// Code for selecting game loading source. 
	// Set one of the booleans below to 1 in order to skip source prompt go straight to USB or SD
	u32 straightToUSB = 0;
	u32 straightToSD = 0;
	
	if( straightToUSB == 0 && straightToSD == 0 )
		printf("Press A to mount USB drive, B to mount SD card or Y or 1 to exit.\n");

	while (1)
	{
		u32 pressed;
		get_input(NULL, &pressed);
		VIDEO_WaitVSync();

		if (pressed & INPUT_BUTTON_OK || straightToUSB == 1)
		{
			if (fatMountSimple("disk", &__io_usbstorage)==true)
				break;
			printf("Failed to mount USB drive\n");
		}

		if (pressed & INPUT_BUTTON_CANCEL || straightToSD == 1)
		{
			if (fatMountSimple("disk", &__io_wiisd)==true)
				break;
			printf("Failed to mount SD card\n");
		}

		if (pressed & INPUT_BUTTON_1)
		{
			printf("Exiting...\n");
			goto finish;
		}
	}

	// find an .ISO file in /games/ to run
	// this directory can be changed, but /games/ is the default if Devolution needs to search for a disc
	select_game("/games", real_memcard);

	// unmount disk to be sure any changes are flushed
	fatUnmount("disk:");

	printf("Disc ID: %08X\n", read32(0));

	// set the video mode based on the ISO's region
	// DO NOT SKIP THIS STEP OR TRY TO FORCE A DIFFERENT MODE
	// if you don't use the native video mode the game will probably complain (by crashing)
	if ((read32(0)&0xFF)=='E' || (read32(0)&0xFF)=='J')
	{
		// set 60Hz mode
		if (CONF_GetVideo() == CONF_VIDEO_PAL)
		{
			// PAL60 is the same as NTSC except it works properly with SCART cables
			// (NTSC assumes S-Video output = red picture with a SCART cable)
			vidmode = &TVEurgb60Hz480IntDf;
		}
		else
			vidmode = &TVNtsc480IntDf;
	}
	else
	{
		//TODO: NOT WORKING! Offer choice of video mode for 480p compatible games e.g. PAL TimeSplitters (G3FB) 
		if ((read32(0)&0xFFFFFFFF)==0x47334650)
		{
			printf("Press A for 576i/480i or B for 480p/480i\n");
			
			while (1)
			{
				u32 pressed;
				get_input(NULL, &pressed);
				VIDEO_WaitVSync();

				if (pressed & INPUT_BUTTON_OK)
				{
					vidmode = &TVPal528IntDf;
					break;
				}

				if (pressed & INPUT_BUTTON_CANCEL)
				{
					vidmode = &TVNtsc480Prog;
					break;
				}
			}
		}
		else
		{
			vidmode = &TVPal528IntDf;
		}
	}

	// make sure we loaded an ISO successfully
	if (read32(0))
	{
		input_shutdown();
		// switch video mode if necessary
		if (vidmode != rmode)
		{
			printf("Switching video mode...\n");
			rmode = vidmode;
			initialize(rmode);
		}
		// Go!
		LAUNCH();
		printf("What!? Devolution engine failed to launch.\n");
		input_startup();
	}

wait_for_exit:
	printf("Press START to exit\n");
	while(1) {
		u32 pressed;
		get_input(NULL, &pressed);
		if (pressed & INPUT_BUTTON_START) break;
		VIDEO_WaitVSync();
	}

finish:
	return 0;
}
