#include <stdint.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <math.h>
#include <vector>

#ifdef WIN32
#include <windows.h>
#else
#include <inttypes.h>
#endif

#include "primegen.h"

#define NO_PHASE  999

typedef struct
{
   int32_t minN;
   int32_t maxN;
} phase_t;

int32_t     gi_base = 0;
int32_t     gi_c = 0;
int32_t     gi_recoveryPhase;
int32_t     gi_recoverySubPhase;
int32_t     gi_recoveryStep = 0;
uint32_t    gt_start_time = 0;
uint64_t    gl_minK = 0;
uint64_t    gl_maxK = 0;
uint64_t    gl_remainingK = 0;
uint32_t    gi_maxNfbncsieve = 0;
uint32_t    gi_maxKsrsieve2 = 0;
phase_t     gs_phases[105];
int32_t     gi_phaseCount = 0;
int32_t     gi_currentPhase = 0;
int32_t     gi_currentSubPhase = 0;

std::vector<bool> gb_KTerms;

#define BIT(k)       ((k) - gl_minK)

#ifdef WIN32
#define atoll  _atoi64

#define PRIu64 "I64u"
#define PRId64 "I64d"
#define PRIx64 "I64x"
#define SCNu64 "I64u"
#define SCNd64 "I64d"
#define SCNx64 "I64x"
#endif

#define STEP_PHASE_STARTED                11
#define STEP_SIEVING                      21
#define STEP_PRP_TESTING                  31
#define STEP_PRIMALITY_TESTING            41
#define STEP_MERGING_RESULTS              91
#define STEP_PHASE_DONE                   99

void getInputs(void);
void checkRecovery(void);
bool buildRemainingBitMap(void);
void buildNewBitMap(void);
void removeTrivial(void);
void removeGFN(void);
void removeMOB(void);
void runfbncsieve(uint32_t n);
void processFbncAbcdFile(uint32_t n, char *fileName);
void doPhase(int phase);
void do_sieving(uint64_t max_k);
void do_prp_testing(void);
void do_primality_testing(void);
void process_results(const char *inFileName, int isPrime);
void merge_results(void);
void delete_temp_files(void);
void output_remain(const char *message);
void prepare_recovery(uint32_t step);
void delete_file(const char *fileName);
void checkForProgram(const char *programName);
double getAverageTestTime(uint64_t k, uint32_t n);

void error(const char *fmt, ...)
{
   va_list args;

   fprintf(stderr, "Error: ");
   va_start(args, fmt);
   vfprintf(stderr, fmt, args);
   va_end(args);
   fprintf(stderr, "\n");
   exit(0);
}

void report(const char *fmt, ...)
{
   FILE *fPtr = fopen("srbsieve.log", "a+");

   uint32_t report_time = (uint32_t) time(NULL);
            
   int32_t totalSeconds = (report_time - gt_start_time);
   int32_t hours = totalSeconds / 3600;
   int32_t minutes = (totalSeconds - hours * 3600) / 60;
   int32_t seconds = totalSeconds % 60;
   char    buffer1[500], buffer2[500];

   va_list args;
   
   va_start(args, fmt);
   vsprintf(buffer2, fmt, args);
   va_end(args);

   sprintf(buffer1, "Status (%02d:%02d:%02d): ", hours, minutes, seconds);
   
   printf("%s %s\n", buffer1, buffer2);

   fprintf(fPtr, "%s %s\n", buffer1, buffer2);

   fclose(fPtr);
}

void reportWithDateTime(const char *fmt, ...)
{
   FILE *fPtr = fopen("srbsieve.log", "a+");

   time_t report_time;
   time(&report_time);
   struct tm *stm = localtime(&report_time);
   char    buffer1[500], buffer2[500];

   va_list args;

   va_start(args, fmt);
   vsprintf(buffer2, fmt, args);
   va_end(args);

   sprintf(buffer1, "Status (%04d-%02d-%02d %02d:%02d:%02d): ", stm->tm_year+1900, stm->tm_mon+1, stm->tm_mday, stm->tm_hour, stm->tm_min, stm->tm_sec);
   
   printf("%s %s\n", buffer1, buffer2);

   fprintf(fPtr, "%s %s\n", buffer1, buffer2);

   fclose(fPtr);
}

int main(int argc, char **argv)
{
   int32_t  phase;

   gi_recoveryPhase = NO_PHASE;
   gi_recoverySubPhase = 0;
   
   gt_start_time = (uint32_t) time(NULL);

   getInputs();

   if (argc > 1)
   {
      if (atol(argv[1]) < 1 || atol(argv[1]) > gi_phaseCount)
         error("Invalid phase specified");
      
      phase = gi_recoveryPhase = atol(argv[1]);
      FILE    *ckpt = fopen("srbsieve.ckpt", "r");
      if (ckpt)
         error("srbsieve.ckpt was found.  Either delete that file or do not specify the starting phase on the command line");
   }
   else
      checkRecovery();

   if (gi_recoveryPhase != NO_PHASE)
   {
      if (!buildRemainingBitMap()) 
      {
         printf("pl_remain.txt not found.  Starting from the beginning\n");
         gi_recoveryPhase = NO_PHASE;
      }
      else
      {
         if (gi_recoverySubPhase == 0)
            reportWithDateTime("Started phase %u.  %" PRIu64" k remaining", gi_recoveryPhase, gl_remainingK);
         else
            reportWithDateTime("Continuing phase %u subphase %u.  %" PRIu64" k remaining at start of phase", gi_recoveryPhase, gi_recoverySubPhase, gl_remainingK);
      }
   }
   
   if (gi_recoveryPhase == NO_PHASE)
   {         
      if (buildRemainingBitMap()) 
         error("pl_remain.txt was found.  Either delete or specify the phase to start with");
      
      delete_file("pl_prime.txt");
      delete_file("pl_prp.txt");
      delete_file("pl_MOB.txt");
      delete_file("pl_GFN.txt");
      delete_file("pl_remain.txt");
      delete_file("pl_trivial.txt");

      buildNewBitMap();

      removeTrivial();
      removeGFN();
      removeMOB();

      for (uint32_t n=1; n<=gi_maxNfbncsieve; n++)
         runfbncsieve(n);

      output_remain("Finished all initial steps.  Phase 1 is next");

      phase = 1;
   }

#ifdef WIN32
   checkForProgram("srsieve2.exe");
   checkForProgram("pfgw64.exe");
#else
   checkForProgram("srsieve2");
   checkForProgram("pfgw64");
#endif

   for ( ; phase<=gi_phaseCount; phase++)
      doPhase(phase);

   delete_file("srbsieve.ckpt");
   exit(0);
}

void  getInputs(void)
{
   char  buffer[100], *pos;
   FILE *ini = fopen("srbsieve.ini", "r");
   int   index;

   if (!ini)
      error("srbsieve.ini not found");

   gi_base = 0;
   gi_c = 0;
   gl_minK = 0;
   gl_maxK = 0;
   gi_phaseCount = 0;
   gi_maxNfbncsieve = 0;

   while (fgets(buffer, 100, ini) != NULL)
   {
      if (!memcmp(buffer, "base=", 5))
         gi_base = atoi(buffer+5);
      if (!memcmp(buffer, "c=", 2))
         gi_c = atoi(buffer+2);
      if (!memcmp(buffer, "mink=", 5))
         gl_minK = atoll(buffer+5);
      if (!memcmp(buffer, "maxk=", 5))
         gl_maxK = atoll(buffer+5);
      if (!memcmp(buffer, "maxNfbncsieve=", 14))
         gi_maxNfbncsieve = atoll(buffer+14);
      if (!memcmp(buffer, "maxKsrsieve2=", 13))
         gi_maxKsrsieve2 = atoll(buffer+13);

      if (!memcmp(buffer, "phase=", 6))
      {
         gi_phaseCount++;

         index = gi_phaseCount;

         if (gi_phaseCount > 100)
            error("Reached limit of 100 phase entries");

         // gs_phases[0] will be empty
         if (gi_phaseCount == 1)
            gs_phases[1].minN = 1;

         
         pos = buffer+6;

         if (sscanf(pos, "%d", &gs_phases[index].maxN) != 1)
            error("Could not process line %s", buffer);

         // Set minN for next phase
         gs_phases[index+1].minN = gs_phases[index].maxN + 1;
      }
   }

   fclose(ini);

   if (gi_base == 0)
      error("base must be specified in srbsieve.ini");
   if (gi_c == 0)
      error("c must be specified in srbsieve.ini");
   if (gl_minK == 0)
      error("mink must be specified in srbsieve.ini");
   if (gl_maxK == 0)
      error("maxK must be specified in srbsieve.ini");
   
   if (gi_base < 2)
      error("base cannot be less than 2");
   
   if (gi_c != 1 && gi_c != -1)
      error("c must be -1 or 1");

   if (gl_minK >= gl_maxK)
      error("mink must be less than maxK");
   
   if (gi_maxNfbncsieve < 2)
      error("maxNfbncsieve must be at least 2 in srbsieve.ini");
   
   if (gi_maxKsrsieve2 < 1000)
      error("gi_maxKsrsieve2 must be at least 1000 in srbsieve.ini");
   
   gs_phases[1].minN = gi_maxNfbncsieve + 1;
   
   if (gi_base & 1)
   {
      // Both need to be even if the base is odd
      if (gl_minK & 1)
         gl_minK++;
      
      if (gl_maxK & 1)
         gl_maxK--;
   }
   else
   {
      // Both need to be odd if the base is even
      if (!(gl_minK & 1))
         gl_minK++;
      
      if (!(gl_maxK & 1))
         gl_maxK--;
   }
}

void  checkRecovery(void)
{
   char     buffer[1000];
   FILE    *ckpt = fopen("srbsieve.ckpt", "r");
   int32_t  minN;
   int32_t  maxN;
   int32_t  index;

   if (!ckpt)
      return;

   while (fgets(buffer, 100, ckpt) != NULL)
   {
      if (!memcmp(buffer, "phaseInProgress=", 16))
         gi_recoveryPhase = atoi(buffer+16);
      if (!memcmp(buffer, "subphaseInProgress=", 19)) 
         gi_recoverySubPhase = atoi(buffer+19);
      if (!memcmp(buffer, "phase=", 6))
         if (sscanf(buffer, "phase=%d,%d", &minN, &maxN) != 2)
            error("Could not process line %s", buffer);
      if (!memcmp(buffer, "currentStep=", 12))
         gi_recoveryStep = atoi(buffer+12);
   }

   fclose(ckpt);

   if (gi_recoveryPhase < 1 || gi_recoveryPhase > gi_phaseCount)
      error("Invalid phase for recovery");
   if (gi_recoverySubPhase < 1)
      error("Invalid subphase for recovery");
   if (gi_recoveryStep < 9 || gi_recoveryStep > 99)
      error("Invalid recovery step");
   if (minN != gs_phases[gi_recoveryPhase].minN || maxN != gs_phases[gi_recoveryPhase].maxN)
      error("checkpoint phase does not match input phase");
}

void  buildNewBitMap(void)
{
   uint64_t k;
   uint32_t removed = 0;

   gl_remainingK = gl_maxK - gl_minK + 1;

   gb_KTerms.resize(gl_remainingK);
   std::fill(gb_KTerms.begin(), gb_KTerms.end(), true);

   if (gi_base & 1)
   {
      // For odd bases remove odd k
      for (k=gl_maxK+1; k<=gl_maxK-1; k+=2)
      {
         gb_KTerms[BIT(k)] = false;
         gl_remainingK--;
      }
   }

   report("Started with %d terms", gl_remainingK);
}

bool  buildRemainingBitMap(void)
{
   FILE *remain = fopen("pl_remain.txt", "r");
   char     buffer[1000];
   uint64_t k;

   if (!remain)
      return false;

   gl_remainingK = gl_maxK - gl_minK + 1;

   gb_KTerms.resize(gl_remainingK);
   std::fill(gb_KTerms.begin(), gb_KTerms.end(), false);

   gl_remainingK = 0;

   while (fgets(buffer, sizeof(buffer), remain) != NULL)
   {
      if (buffer[0] == '\n')
         continue;

      k = atoll(buffer);

      if (k < gl_minK || k > gl_maxK)
         continue;
      
      gb_KTerms[BIT(k)] = true;
      gl_remainingK++;
   }

   fclose(remain);
      
   return true;
}

void  removeTrivial(void)
{
   int32_t  left = gi_base - 1;
   int32_t  p;
   int32_t  factors[10];
   int32_t  factorCount = 0;
   uint64_t k;
   uint32_t removed = 0;
   primegen pg;
   FILE    *trivials = NULL;

   // Find all prime divisors of the base
   primegen_init(&pg);
   do
   {
      p = primegen_next(&pg);

      if (left % p == 0)
      {
         factors[factorCount++] = p;
         while (left % p == 0 && left > 0)
            left /= p;
      }
   } while (left > p);

   for (int f=0; f<factorCount; f++)
   {
      k = gl_minK - (gl_minK % factors[f]);

      if (gi_c == -1)
         k += 1;
      else if (k == 0)
         k = factors[f] - 1;
      else
         k -= 1;

      while (k < gl_minK)
         k += factors[f];

      for ( ; k <= gl_maxK; k += factors[f])
      {         
         if (gb_KTerms[BIT(k)])
         {
            gb_KTerms[BIT(k)] = false;
            gl_remainingK--;
            removed++;
            
            if (!trivials)
               trivials = fopen("pl_trivial.txt", "w");
            
            fprintf(trivials, "%" PRId64"\n", k);
         }
      }
   }

   if (trivials)
      fclose(trivials);

   reportWithDateTime("Removed %u terms due to trivial factorization", removed);
}

void  removeGFN(void)
{
   // Only b^n+1 can be GFNs, and b must be even
   // otherwise b^n+1 is always even.
   if (gi_base % 2 == 1 || gi_c != 1)
      return;

   FILE *gfns = NULL;
   int32_t  root;
   uint32_t removed = 0;
   uint64_t k;

   // Find the lowest root for the base, e.g. if base = 216, then root = 6 (216 = 6^3).
   for (int i=2; i<=gi_base; i+=2)
   {
      root = i;

      while (root < gi_base)
         root *= i;

      if (root == gi_base)
         break;
   }

   // Find all k where k*b^n+1 = b^m+1 for some integer m.
   // Those will be removed as those are potential GFNs.  Any GFNs for
   // these bases are already searched up to very high n.
   k = root;
   while (k <= gl_maxK)
   {
      if (k >= gl_minK)
      {
         if (gb_KTerms[BIT(k)])
         {
            gb_KTerms[BIT(k)] = false;
            gl_remainingK--;
            removed++;

            if (!gfns)
               gfns = fopen("pl_GFN.txt", "w");
            
            fprintf(gfns, "%" PRId64"*%d^n%+d\n", k, gi_base, gi_c);
         }
      }

      k *= root;
   }

   if (gfns)
      fclose(gfns);

   reportWithDateTime("Removed %d terms due to GFN", removed);
}

void  removeMOB(void)
{
   uint32_t lowPrimeCount = 1000000;
   primegen pg;
   uint64_t *lowPrimes;
   uint64_t  rangeMinK, rangemaxK;
   uint64_t k;
   uint32_t removed = 0, adder;
   uint32_t bit, byte;
   uint8_t *sieveBitMap;
   FILE    *mobs = fopen("pl_MOB.txt", "w");

   lowPrimes = (uint64_t *) malloc((lowPrimeCount + 1 ) * sizeof(uint64_t));

   primegen_init(&pg);
   for (int pi=0; pi<lowPrimeCount; pi++)
      lowPrimes[pi] = primegen_next(&pg);
   lowPrimes[lowPrimeCount] = 0;

   rangeMinK = gl_minK + gi_c;
   rangemaxK = gl_maxK + gi_c;
   sieveBitMap = (uint8_t *) malloc(1 + (size_t) 1 + (gl_maxK - gl_minK + 1 / 8));

   memset(sieveBitMap, 0xff, 1 + (size_t) 1 + (gl_maxK - gl_minK + 1 / 8));

   // Sieve through all k between minK and maxK that are divisible
   // by the base to find those that are composite.
   for (int i=0; i<lowPrimeCount; i++)
   {
      if (lowPrimes[i] * lowPrimes[i] > rangemaxK)
         break;

      if (gi_base % lowPrimes[i] == 0)
         continue;

      // Find the smallest k <= minK divisible by the base
      k = rangeMinK - (rangeMinK % (lowPrimes[i] * gi_base));

      // Since k is unsigned, we need to ensure that k-gi_c is > 0
      // otherwise k-gi_c will become a large positive value rather than
      // a negative value.
      if (k == 0 && gi_c == 1)
         k += lowPrimes[i];

      while ((k-gi_c) % gi_base != 0)
         k += lowPrimes[i];

      // Make sure that k >= minK
      if (k < rangeMinK)
         k += (lowPrimes[i] * gi_base);

      if (k == lowPrimes[i])
         k += (lowPrimes[i] * gi_base);

      // k is now the smallest k >= mink where k is divisible by lowPrimes[i]
      while (k <= rangemaxK)
      {
         byte = (uint32_t) ((k-rangeMinK) >> 3);
         bit = (uint8_t) (1 << ((k-rangeMinK) & 7));

         // Mark as composite
         sieveBitMap[byte] &= ~bit;

         k += (lowPrimes[i] * gi_base);
      }
   }

   k = gl_minK - (gl_minK % gi_base);
   while (k < gl_minK)
      k += gi_base;

   if (k & 1)
   {
      // Make sure that k is even
      if (gi_base & 1)
         k += gi_base;
   }

   // If the gi_base is odd, then we add bsae*2 to get the next k
   adder = (gi_base & 1 ? (gi_base*2) : gi_base);
   
   // For each k between minK and maxK that are divisible by the base, if k+c is
   // composite, then k*base^n+c is composite for all n so we can remove that k.
   while (k <= gl_maxK)
   {
      byte = (uint32_t) (((k+gi_c)-rangeMinK) >> 3);
      bit = (uint8_t) (1 << (((k+gi_c)-rangeMinK) & 7));

      if (!(sieveBitMap[byte] & bit))
      {
         if (gb_KTerms[BIT(k)])
         {
            gb_KTerms[BIT(k)] = false;
            gl_remainingK--;
            removed++;
            
            if (!mobs)
               fopen("pl_MOB.txt", "w");
            
            fprintf(mobs, "%" PRId64"\n", k);
         }
      }

      k += adder;
   }

   free(lowPrimes);
   free(sieveBitMap);
   
   if (mobs)
      fclose(mobs);
   
   reportWithDateTime("Removed %d terms due to MOB", removed);
}

void runfbncsieve(uint32_t n)
{
   char     program[50];
   char     sequence[50];
   char     command[200];
   char     fileName[100];
   FILE    *fPtr;
   
   sprintf(fileName, "n%u.abcd", n);
   
#ifdef WIN32
   fPtr = fopen("fbncsieve.exe", "r");
   
   sprintf(program, "fbncsieve");
   sprintf(sequence, "k*%u^^%u%+d", gi_base, n, gi_c);
#else
   fPtr = fopen("fbncsieve", "r");

   sprintf(program, "./fbncsieve");
   sprintf(sequence, "k*%u^%u%+d", gi_base, n, gi_c);
#endif

   if (!fPtr)
      error("fbncsieve not found");
   
   fclose(fPtr);

   sprintf(command, "%s -r -fD -k%llu -K%llu -s%s -o %s", program, gl_minK, gl_maxK, sequence, fileName);

   system(command);
   
   processFbncAbcdFile(n, fileName);
   
   delete_file(fileName);
}

void processFbncAbcdFile(uint32_t n, char *fileName)
{
   char  buffer[100], message[200];
   char *pos;
   uint64_t k, diff, pmax;
   uint32_t removed = 0;
   uint32_t base;
   int32_t  c;
   
   FILE *fptr = fopen(fileName, "r");
   FILE *primes = fopen("pl_prime.txt", "a");

   if (!fptr) 
      error("Could not open file %s", fileName);

   if (n > gs_phases[0].minN)
      gs_phases[0].minN = n + 1;

   // Read the first line
   fgets(buffer, 100, fptr);

   if (sscanf(buffer, "ABCD $a*%u^%d%d  [%" SCNu64"] // Sieved to %" SCNu64"", &base, &n, &c, &k, &pmax) != 5)
      error("Invalid first line of file %s", fileName);

   if (base != gi_base)
      error("Read base %d from %s", base, fileName);

   if (c != +1 || c != -1)
      error("Read c %+d from %s", c, fileName);

   // Note that fbncsieve will only output even k if the base is odd.
   // Fortunately sbrsieve will already have set odd k to false in the vector.
   if (gb_KTerms[BIT(k)])
   {
      gb_KTerms[BIT(k)] = false;
      gl_remainingK--;
      removed++;
      
      fprintf(primes, "%" PRId64"*%d^%d%+d\n", k, gi_base, n, gi_c);;
   }

   while (fgets(buffer, sizeof(buffer), fptr) != NULL)
   {
      if (sscanf(buffer, "%" SCNu64 , &diff) != 1)
         error("Line %s is malformed", buffer);
      
      k += diff;

      if (gb_KTerms[BIT(k)])
      {
         gb_KTerms[BIT(k)] = false;
         gl_remainingK--;
         removed++;
         
         fprintf(primes, "%" PRId64"*%d^%d%+d\n", k, gi_base, n, gi_c);;
      }
   }
   
   fclose(fptr);
   fclose(primes);

   sprintf(message, "Removed %d terms from file for n = %d", removed, n);
   reportWithDateTime("%s: %d remaining", message, gl_remainingK);
}

void doPhase(int32_t phase)
{
   FILE     *sieve_in = NULL;
   uint32_t  count = 0;
   uint64_t  startingK = 0, completedK = 0;
   int32_t   subPhase = 0;
   uint64_t  k, max_k, min_k;
   int64_t   index;
   uint32_t  xOfY, chunks;
   
   gi_currentPhase = phase;
   gi_currentSubPhase = 0;

   gt_start_time = (uint32_t) time(NULL);

   k = gl_minK;

   // Skip over all k tested so far
   if (gi_recoveryPhase > 0)
   {
      if (gi_recoverySubPhase == 0)
      {
         printf("Recovering phase %d with %" PRIu64" terms\n", gi_recoveryPhase, gl_remainingK);
         gi_recoveryPhase = NO_PHASE;
      }
      else 
      {
         printf("Recovering from phase %d, subphase %d with %" PRIu64" terms\n", gi_recoveryPhase, gi_recoverySubPhase, gl_remainingK);

         index = (gi_recoverySubPhase - 1) * gi_maxKsrsieve2 + 1;
         max_k = 0;

         for (; ; k++)
         {         
            if (gb_KTerms[BIT(k)])
            {
               max_k++;
               
               if (index == max_k)
                  break;
            }
         }

         gi_currentSubPhase = gi_recoverySubPhase - 1;
         gi_recoverySubPhase = 0;
         completedK = (index - 1);
      }
   }
   
   startingK = gl_remainingK;
   xOfY = 0;
   chunks = 1 + (gl_remainingK / gi_maxKsrsieve2);

   for ( ; k<=gl_maxK; k++)
   {
      if (gb_KTerms[BIT(k)])
      {
         max_k = k;

         if (count == 0)
         {
            gi_currentSubPhase++;
            sieve_in = fopen("sieve.in", "w");
            min_k = k;
         }

         fprintf(sieve_in, "%" PRId64"*%d^n%+d\n", k, gi_base, gi_c);
         count++;

         if (count == gi_maxKsrsieve2)
         {
            fclose(sieve_in);
            sieve_in = NULL;
            
            xOfY++;
            
            printf("Phase %d:  Processing k from %" PRIu64" to %" PRIu64". Chunk %u of %u.\n", gi_currentPhase, min_k, max_k, xOfY, chunks);

            do_sieving(max_k);
            do_prp_testing();
            do_primality_testing();
            delete_temp_files();
            
            completedK += count;
            count = 0;

            report("Completed %.02f pct of phase %d.", 100.0 * ((double) completedK) / ((double) startingK), phase);
            printf("\n\n");
         }
      }
   }

   if (sieve_in)
      fclose(sieve_in);

   if (count > 0)
   {
      printf("Phase %d:  Processing k from %" PRIu64" to %" PRIu64"\n", gi_currentPhase, min_k, max_k);

      do_sieving(max_k);
      do_prp_testing();
      do_primality_testing();
      delete_temp_files();

      completedK += count;
   }

   merge_results();

   reportWithDateTime("Completed phase %u.  %" PRIu64" k remaining", phase, gl_remainingK);

   prepare_recovery(STEP_PHASE_DONE);
}

void do_sieving(uint64_t max_k)
{
   char      command[500];
   char      program[100];
   uint32_t  n_min = gs_phases[gi_currentPhase].minN;
   uint32_t  n_max = gs_phases[gi_currentPhase].maxN;

   if (gi_recoveryPhase != NO_PHASE)
      if (gi_recoveryStep != STEP_SIEVING && gi_recoveryStep != STEP_PHASE_STARTED)
         return;

   double averageTestTime = getAverageTestTime(max_k, n_min + 2*((n_max - n_min) / 3));

   prepare_recovery(STEP_SIEVING);

#ifdef WIN32
   strcpy(program, "srsieve2");
   checkForProgram("srsieve2.exe");
#else
   strcpy(program, "./srsieve2");
   checkForProgram("srsieve2");
#endif
   
   if (averageTestTime < 1.0)
      sprintf(command, "%s -n%d -N%d -s sieve.in -fP -4%f -w1e4 -osr_b.pfgw", program, n_min, n_max, 1.0 / averageTestTime);
   else
      sprintf(command, "%s -n%d -N%d -s sieve.in -fP -5%f -w1e4 -osr_b.pfgw", program, n_min, n_max, averageTestTime);

   printf("command: %s\n", command);
   system(command);

   // Failure between here and next step requires restarting of the entire step
   prepare_recovery(STEP_PHASE_STARTED);
}

double getAverageTestTime(uint64_t k, uint32_t n)
{
   char command[500];
   char buffer[1000];
   
   delete_file("pfgw.ini");
   delete_file("rate.out");
   delete_file("rate.in");
   
   FILE *rate_in = fopen("rate.in", "w");

   // Take the average of 10 tests with varying k because each k could use
   // a differnt FFT size.
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-9, gi_base, n, gi_c);
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-8, gi_base, n, gi_c);
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-7, gi_base, n, gi_c);
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-6, gi_base, n, gi_c);
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-5, gi_base, n, gi_c);
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-4, gi_base, n, gi_c);
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-3, gi_base, n, gi_c);
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-2, gi_base, n, gi_c);
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-1, gi_base, n, gi_c);
   fprintf(rate_in, "%" PRIu64"*%u^%u%+d\n", k-0, gi_base, n, gi_c);

   fclose(rate_in);

#ifdef WIN32
   sprintf(command, "pfgw64 -Cquiet -lrate.out -f0 rate.in > nul");
   checkForProgram("pfgw64.exe");
#else
   sprintf(command, ".\pfgw64 -Cquiet -lrate.out -f0 rate.in > nul");
   checkForProgram("pfgw64");
#endif
   
   system(command);
   
   FILE *rate_out = fopen("rate.out", "r");
   double testTime, totalTime = 0.0;
   double count = 0.0;
   while (fgets(buffer, 1000, rate_out) != NULL)
   {
      char *pos1 = strchr(buffer, '(');
      if (pos1 == NULL)
         error("could not get rate from rate.out (cannot find start of time)");
      
      char *pos2 = strchr(pos1, 's');
      if (pos2 == NULL)
         error("could not get rate from rate.out (cannot find end of time)");
      
      *pos1 = *pos2 = 0;
      if (sscanf(pos1+1, "%lf", &testTime) != 1)
         error("could not get rate from rate.out (failed scan)");
      
      totalTime += testTime;
      count += 1.0;
   }

   fclose(rate_out);
   
   delete_file("pfgw.ini");
   delete_file("rate.out");
   delete_file("rate.in");
   
   return totalTime / count;
}

void do_prp_testing(void)
{
   FILE *pfgw_in;
   FILE *pfgw_out;
   char  inFileName[300];
   char  buffer[1000];
   char  command[500];
   char *pos;
   
   if (gi_recoveryPhase != NO_PHASE)
      if (gi_recoveryStep != STEP_PRP_TESTING)
         return;

   sprintf(inFileName, "sr_%d.pfgw", gi_base);
   pfgw_in = fopen(inFileName, "r");

   if (pfgw_in)
   {
      pfgw_out = fopen("sr_b.pfgw", "w");

      fgets(buffer, 1000, pfgw_in);
      pos = strchr(buffer, 'c');
      pos++;
      *pos = 0;
      fprintf(pfgw_out, "%s // {number_primes,$a,1}\n", buffer);

      while (fgets(buffer, 1000, pfgw_in) != NULL) 
         fprintf(pfgw_out, "%s", buffer);
  
      fclose(pfgw_in);
      fclose(pfgw_out);
   
      delete_file(inFileName);
   }

   delete_file("pfgw.ini");
   
   gi_recoveryPhase = NO_PHASE;   
   prepare_recovery(STEP_PRP_TESTING);

#ifdef WIN32
   sprintf(command, "pfgw64 -Cquiet -f0 sr_b.pfgw > nul");
   checkForProgram("pfgw64.exe");
#else
   sprintf(command, "./pfgw64 -Cquiet -f0 sr_b.pfgw > /dev/null");
   checkForProgram("pfgw64");
#endif
   
   printf("command: %s\n", command);
   system(command);

   // Failure between here and next step requires restarting of the entire step
   prepare_recovery(STEP_PHASE_STARTED);
}

void do_primality_testing(void)
{
   char  command[500];
   FILE *prpFile;

   if (gi_recoveryPhase != NO_PHASE)
      if (gi_recoveryStep != STEP_PRIMALITY_TESTING)
         return;
   
   prpFile = fopen("sr_b.prp", "r");

   if (!prpFile)
   {
      prpFile = fopen("pfgw.log", "r");

      // This could happen if no PRPs were found
      if (!prpFile)
         return;

      fclose(prpFile);

      rename("pfgw.log", "sr_b.prp");
   }
   else
      fclose(prpFile);
   
   // Yes, this will cause pfgw to start from the beginning of the file
   // but it shouldn't cost too much time to retest whatever is in the file.
   delete_file("pfgw.ini");
   delete_file("pfgw-prime.log");

   prepare_recovery(STEP_PRIMALITY_TESTING);

#ifdef WIN32
   sprintf(command, "pfgw64 -Cquiet -f0 -t%c sr_b.prp > nul", (gi_c == 1 ? 'm' : 'p'));
#else
   system("./pfgw64 -Cquiet -f0 -t%c sr_b.prp > /dev/null", (gi_c == 1 ? 'm' : 'p'));
#endif
   
   printf("command: %s\n", command);
   system(command);

   // Failure between here and next step requires restarting of the entire step
   prepare_recovery(STEP_PHASE_STARTED);

   process_results("pfgw.log", 0);
   process_results("pfgw-prime.log", 1);

   delete_file("sr_b.prp");
}

void process_results(const char *inFileName, int isPrime)
{
   char buffer[1000];
   FILE *in;
   FILE *out;

   in = fopen(inFileName, "r");
   if (!in) return;
   out = fopen("results.tmp", "a");

   while (fgets(buffer, 1000, in) != NULL) 
      fprintf(out, "%d,%s", isPrime, buffer);

   fclose(in);
   fclose(out);
}

void merge_results(void)
{
   FILE    *in;
   FILE    *out1, *out2, *out3;
   char     buffer[1000], *pos;
   uint64_t k;
   uint32_t prp = 0;

   in = fopen("results.tmp", "r");
   if (!in) return;

   out2 = fopen("pl_prime.txt", "a");
   out3 = fopen("pl_prp.txt", "a");
   
   while (fgets(buffer, 1000, in) != NULL)
   {
      if (buffer[0] == '1')
         fprintf(out2, "%s", buffer+2);
      else
      {
         fprintf(out3, "%s", buffer+2);
         prp++;
      }

      // Remove k from bitmap
      pos = strchr(buffer, '*');
      if (!pos)
         error("Missing multiplier");

      *pos = 0;
      sscanf(buffer+2, "%" SCNu64"", &k);

      gb_KTerms[BIT(k)] = false;
   }
   
   fclose(in);
   fclose(out2);
   fclose(out3);

   gl_remainingK = 0;
   out1 = fopen("pl_remain.tmp", "w");

   for (uint64_t k=gl_minK; k<=gl_maxK ; k++)
   {
      if (gb_KTerms[BIT(k)])
      {
         fprintf(out1, "%" PRIu64"\n", k);
         gl_remainingK++;
      }
   }

   fclose(out1);

   if (!prp)
      delete_file("pl_prp.txt");

   delete_file("results.tmp");
   delete_file("pl_remain.txt");
   rename("pl_remain.tmp", "pl_remain.txt");
}

void  output_remain(const char *message)
{
   FILE *remain = fopen("pl_remain.txt", "w");

   for (uint64_t k=gl_minK; k<=gl_maxK; k++)
   {
      if (gb_KTerms[BIT(k)])
         fprintf(remain, "%" PRId64"\n", k);
   }

   fclose(remain);

   reportWithDateTime("%s: %d remaining", message, gl_remainingK);
}

void  delete_temp_files()
{
   delete_file("sieve.in");
   delete_file("sr_b.pfgw");
   delete_file("pfgw.ini");
   delete_file("pfgw-prime.log");
   delete_file("pfgw.log");
}

void  prepare_recovery(uint32_t currentStep)
{
   FILE    *ckpt = fopen("srbsieve.ckpt_tmp", "w");

   fprintf(ckpt, "phaseInProgress=%d\n", gi_currentPhase);
   fprintf(ckpt, "subphaseInProgress=%d\n", gi_currentSubPhase);
   fprintf(ckpt, "currentStep=%u\n", currentStep);
   fprintf(ckpt, "phase=%d,%d\n", gs_phases[gi_currentPhase].minN, gs_phases[gi_currentPhase].maxN);

   fclose(ckpt);

   delete_file("srbsieve.ckpt");
   rename("srbsieve.ckpt_tmp", "srbsieve.ckpt");

   gi_recoveryPhase = NO_PHASE;
}

void delete_file(const char *fileName)
{   
   FILE *fPtr = fopen(fileName, "r");
   if (!fPtr) return;
   fclose(fPtr);

   remove(fileName);
}

void checkForProgram(const char *programName)
{
   FILE *fPtr = fopen(programName, "r");
   
   if (fPtr)
   {
      fclose(fPtr);
      return;
   }
   
   error("Program %s was not found", programName);
}