/* legendre.c -- (C) 2016 Mark Rodenkirch, Geoffrey Reynolds

   Generate lookup tables to replace the Legendre(-ckb^n,p) function.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
*/

#include <assert.h>
#include <inttypes.h>
#include <math.h>
#include <stdlib.h>
#include "arithmetic.h"

// Determine if there is a value x such that x^2 = 2 (mod p).
// Note that this does not find x.  That is what findRoot() will do.
// Since findRoot() is more computationally expensive this can
// quickly eliminate p from further consideration.
int32_t	   isQuadraticResidue(uint64_t n, uint64_t p)
{
	int32_t	   j = 1;
	uint64_t	   swap;

	while (n > 1)
	{
		if (n & 1)
		{
			swap = n;
			n = p;
			p = swap;
			if (((n & 3) == 3) && ((p & 3) == 3))
				j = -j;
			n = n % p;
		}
		else
		{
			n >>= 1;
			if (((p & 7) == 3) || ((p & 7) == 5))
				j = -j;
		}
	}

   return (j == 1);
}


// Find x such that x^2 = 2 mod p
// This is solved using Hensel's Lemma
uint64_t	   findRoot(uint64_t p)
{
	uint64_t	   i, s, t, d, m, rem, A, D;

	if ((p & 7) == 7)
		return powmod64(2, (p+1) >> 2, p);

	t = p - 1;
	s = 0;
	while (!(t & 1))
	{
		s++;
		t >>= 1;
	}

	A = powmod64(2, t, p);

	// Find value d where Lengendre Symbol is -1
	for (d=3; d<p; d++)
		if (!isQuadraticResidue(d, p))
			break;

	D = powmod64(d, t, p);

	m = 0;
	for (i=0; i<s; i++)
	{
		if (m == 0)
			rem = 1;
		else
			rem = powmod64(D, m, p);

		rem = mulmod64(rem, A, p);
		rem = powmod64(rem, 1 << (s - 1 - i), p);
		
		if (rem == p - 1)
			m += (1 << i);
	}

	if (m == 0)
		rem = 1;
	else
		rem = powmod64(D, m >> 1, p);

	i = powmod64(2, (t+1) >> 1, p);
	return mulmod64(rem, i, p);
}
