/* files.c -- (C) 2016 Mark Rodenkirch

   Routines for reading and writing sieve files and for building initial bitmap.
*/

#include <assert.h>
#include <ctype.h>
#include <inttypes.h>
#include <stdio.h>
#include <string.h>
#include "cksieve.h"

uint32_t b_term;

int32_t c_carol;
int32_t c_kynea;

uint64_t p_min;
uint64_t p_max;

char abc_buf[70];

void add_nc(uint32_t n, int32_t c);

static int line_counter = 0;
static void line_error(const char *msg, const char *file_name)
{
   error("Line %d: %s in input file `%s'.", line_counter, msg, file_name);
}


/* Reads a line from file, ignoring blank lines and comments. Returns a
   pointer to the first non-whitespace character in the line.
*/
#define MAX_LINE_LENGTH 256
#define COMMENT_CHAR '#'
static const char *read_line(FILE *file)
{
   static char buf[MAX_LINE_LENGTH];
   const char *ptr;

   while ((ptr = fgets(buf,MAX_LINE_LENGTH,file)) != NULL)
   {
      line_counter++;
      while (isspace(*ptr))
         ptr++;
      if (*ptr != COMMENT_CHAR && *ptr != '\0')
         break;
   }

   return ptr;
}

FILE *xfopen(const char *fn, const char *mode, void (*fun)(const char *,...))
{
   FILE *ret;

   ret = fopen(fn,mode);
   if (ret == NULL && fun != NULL)
      fun("Failed to open %sput file `%s'.", (*mode == 'r') ? "in" : "out", fn);

   line_counter = 0;
   return ret;
}

const char *abc_str()
{
   sprintf(abc_buf, "(%"PRIu32"^n+/-c)^2-2", b_term);

   return abc_buf;
}

void xfclose(FILE *file, const char *fn)
{
   if (file != NULL && fclose(file))
      warning("Problem closing file `%s'.", fn);
}

void read_abc_file(const char *file_name)
{
   FILE *file;
   const char *line;
   uint64_t k;
   uint32_t n;
   int32_t  c;

   /* Try to read header */
   file = xfopen(file_name,"r",error);
   line = read_line(file);

   if (line == NULL || sscanf(line, "ABC (%"PRIu32"^$a$b)^2-2 // CK Sieved to: %"PRIu64"", &b_term, &k) != 2)
      line_error("Invalid header",file_name);

   if (b_term < 2)
      line_error("Invalid base",file_name);

   n_min = n_max = 0;

   if (p_min == 0)
      p_min = k;
   else if (p_min != k)
      warning("--pmin=%"PRIu64" from command line overrides pmin=%"PRIu64" from `%s'", p_min, k, file_name);

   while ((line = read_line(file)) != NULL)
   {
      if (sscanf(line, "%" SCNu32 " %" SCNu32, &n, &c) != 2)
         line_error("Malformed line",file_name);
      add_nc(n, c);
   }

   if (ferror(file))
      line_error("Read error",file_name);
   
   fclose(file);

   report(1,"Read %"PRIu32" term%s for %s from ABC file `%s'.", n_count, plural(n_count), abc_str(), file_name);
}

static void write_term(uint32_t n, int32_t c, void *file)
{
   fprintf((FILE *)file,"%d %+d\n", n, c);
}

void write_abc_file(int scroll, uint64_t p, const char *file_name)
{
   if (file_name != NULL)
   {
      FILE *file;

      if ((file = xfopen(file_name,"w",warning)) != NULL)
      {
         uint32_t count;

         fprintf(file, "ABC (%"PRIu32"^$a$b)^2-2 // CK Sieved to: %"PRIu64"\n", b_term, p);

         count = for_each_term(write_term, file);

         xfclose(file,file_name);
         report(scroll,"Wrote %"PRIu32" term%s to ABC file `%s'.", count, plural(count), file_name);
      }
   }
}


