/* cksieve.h -- (C) 2016 Mark Rodenkirch

   cksieve specialised for the Carol/Kynea prime search project.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
*/

#ifndef _cksIEVE_H
#define _cksIEVE_H

#include <limits.h>
#include <stdint.h>
#include "config.h"

#define XSTR(ARG) STR(ARG)
#define STR(ARG) #ARG

/* Also check config.h for compiler settings.
 */

/* Set DEFAULT_L1_CACHE_SIZE,DEFAULT_L2_CACHE_SIZE to the L1,L2 data cache
   size, in Kb. (If not set in the Makefile).
*/
#ifndef DEFAULT_L1_CACHE_SIZE
#define DEFAULT_L1_CACHE_SIZE 16
#endif
#ifndef DEFAULT_L2_CACHE_SIZE
#define DEFAULT_L2_CACHE_SIZE 256
#endif

/* For a hash table expected to hold M elements, use a main table of at
   least M/HASH_MAX_DENSITY and at most M/HASH_MIN_DENSITY elements. The
   size will be increased further to minimise density within this range,
   if L1_cache_size is high enough.
*/
#define HASH_MAX_DENSITY 0.60
#define HASH_MIN_DENSITY 0.10

/* Files to write log reports and report factors to.
 */
#define LOG_FILE_NAME "cksieve.log"

/* Date format used for log file entries (see date --help)
 */
#define LOG_STRFTIME_FORMAT "%c "

/* Report period in seconds.
 */
#define REPORT_PERIOD 60

/* Save period in seconds.
 */
#define DEFAULT_SAVE_PERIOD 3600

/* Set CHECK_FACTORS=1 to double check found factors.
 */
#define CHECK_FACTORS 1

/* Nothing below here should normally need adjustment. */

#ifdef __GNUC__
/* macros that evaluate their arguments only once. From the GCC manual.
 */
#define MIN(a,b) \
   ({ typeof (a) _a = (a); \
      typeof (b) _b = (b); \
      _a < _b ? _a : _b; })
#define MAX(a,b) \
   ({ typeof (a) _a = (a); \
      typeof (b) _b = (b); \
      _a > _b ? _a : _b; })
#define ABS(a) \
   ({ typeof (a) _a = (a); \
      _a < 0 ? -_a : _a; })
#else
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define ABS(a) ((a) < 0 ? -(a) : (a))
#endif

#if (MULTI_PATH && CODE_PATH==2)
# define CODE_PATH_NAME(name) name##_sse2
#elif (MULTI_PATH && CODE_PATH==3)
# define CODE_PATH_NAME(name) name##_cmov
#else
# define CODE_PATH_NAME(name) name
#endif


/* bsgs.c */

void sieve(void);

#if MULTI_PATH
void sieve_sse2(void);
void sieve_cmov(void);
#endif


/* cpu.c */
void set_cache_sizes(uint32_t L1_opt, uint32_t L2_opt);
#if defined(__i386__) || defined(__x86_64__)
int have_sse2(void);
int have_cmov(void);
int is_amd(void);
#endif

/* clock.c */
double get_accumulated_cpu(void);
void set_accumulated_cpu(double seconds);
uint32_t millisec_elapsed_time(void);
double get_accumulated_time(void);
void set_accumulated_time(double seconds);
uint64_t timestamp(void);

/* events.c */
extern uint64_t total_primes_tested;

typedef enum
{
  initialise_events,
  received_sigterm,
  received_sigint,
  received_sighup,
  report_due,
  save_due,

  factor_found,

  /* add more events here */

  last_event
} event_t;

static inline void check_events(uint64_t current_prime)
{
  extern volatile int event_happened;
  extern void process_events(uint64_t);

  if (event_happened)
  {
    process_events(current_prime);
  }
}
void handle_signal(int signum);
void notify_event(event_t event);
void check_progress(void);
void notify_factor(void);

/* factors.c */

void save_factor(uint32_t n, int32_t c, uint64_t p);
int is_factor(uint32_t n, int32_t c, uint64_t p);


/* files.c */

#ifdef EOF
FILE *xfopen(const char *fn, const char *mode, void (*fun)(const char *,...));
void xfclose(FILE *f, const char *fn);
#endif
void read_abc_file(const char *file_name);
void write_abc_file(int scroll, uint64_t p, const char *file_name);
#if USE_COMMAND_LINE_FILE
void read_argc_argv(int *argc, char ***argv, const char *file_name);
#endif

/* primes.c */

void init_prime_sieve(uint64_t pmax);
void prime_sieve(uint64_t low_prime, uint64_t high_prime, void (*bsgs_fun)(uint64_t));
void fini_prime_sieve(void);


/* priority.c */

void set_process_priority(int level);
void set_cpu_affinity(int cpu_number);


/* sequences.c */

extern uint32_t b_term;

extern uint64_t p_min;
extern uint64_t p_max;
extern uint32_t n_min;
extern uint32_t n_max;
extern uint32_t n_count;
extern uint32_t factor_count;

const char *bnc_str(uint32_t n, int32_t c);

int32_t    isQuadraticResidue(uint64_t n, uint64_t p);
uint64_t   findRoot(uint64_t p);

/* cksieve.c */

extern const char *factors_file_name;
extern const char *output_file_name;
extern uint32_t save_period;
extern int verbose_opt;
extern int quiet_opt;

void print_status(uint64_t p, uint32_t p_per_sec, uint32_t secs_per_factor);
void start_cksieve(void);
void finish_cksieve(const char *reason, uint64_t p);

void eliminate_term(uint32_t n, int32_t c, uint64_t p);
void eliminate_repeating_term(uint32_t n, uint32_t next_n, int32_t c, uint64_t p);
uint32_t for_each_term(void (*fun)(uint32_t,int32_t,void *), void *arg);

void build_terms();

/* util.c */

void error(const char *fmt, ...) attribute ((noreturn,format(printf,1,2)));
void warning(const char *fmt, ...) attribute ((format(printf,1,2)));
void logger(int print, const char *fmt, ...) attribute ((format(printf,2,3)));
void *xmalloc(uint32_t sz) attribute ((malloc));
void *xrealloc(void *d, uint32_t sz);
void *xmemalign(uint32_t align, uint32_t size) attribute ((malloc));
void xfreealign(void *mem);
void report(int scroll, const char *fmt, ...);
static inline const char *plural(int n)
{
  return (n == 1) ? "" : "s";
}

#endif /* _cksIEVE_H */
