/* cksieve.c -- (C) 2016 Mark Rodenkirch

   cksieve specialised for the Carol/Kynea prime search project.
      Carol -> (b^n-1)^2-2
      Kynea -> (b^n+1)^2-2

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
*/

#include <assert.h>
#include <inttypes.h>
#include <math.h>
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <getopt.h>
#include <errno.h>
#include "cksieve.h"
#include "version.h"
#include "arithmetic.h"

#if HAVE_MALLOPT
#include <malloc.h>
#endif

#define NAME "cksieve"
#define DESC "A sieve for Carol (b^n-1)^2-2 and Kynea (b^n+1)^2-2 numbers"

char default_output_file_name[20];
const char *output_file_name = NULL;
const char *factors_file_name = NULL;
uint32_t save_period = DEFAULT_SAVE_PERIOD;
int verbose_opt = 0;
int quiet_opt = 0;
#if HASHTABLE_SIZE_OPT
uint32_t hashtable_size = 0;
#endif

static void banner(void)
{
  printf(NAME " " XSTR(MAJOR_VER) "." XSTR(MINOR_VER) "." XSTR(PATCH_VER)
         " -- " DESC ".\n");
#ifdef __GNUC__
  if (verbose_opt)
    printf("Compiled on " __DATE__ " with GCC " __VERSION__ ".\n");
#endif
}

static void help(void)
{
  printf("Usage: %s -P P1 -i FILE <-o FILE | -f FILE> [OPTION ...]\n", NAME);
  printf(" -b --base B\n");
  printf(" -n --nmin N0\n");
  printf(" -N --nmax N1\n");
  printf(" -p --pmin P0\n");
  printf(" -P --pmax P1          "
         "Sieve for factors p in the range P0 <= p <= P1\n");
  printf(" -i --input FILE       "
         "Read sieve from ABC format file FILE.\n");
  printf(" -o --output FILE      "
         "Write sieve to ABC format file FILE.\n");
  printf(" -f --factors FILE     "
         "Append new factors to file FILE.\n");
  printf(" -s --save TIME        "
         "Update output file every TIME (default %u) minutes.\n",
         (unsigned int)DEFAULT_SAVE_PERIOD/60);
  printf(" -z --lower-priority   "
         "Run at low priority. (-zz lower).\n");
  printf(" -Z --raise-priority   "
         "Run at high priority. (-ZZ higher).\n");
  printf(" -A --affinity N       "
         "Set affinity to CPU number N.\n");
  printf(" -q --quiet            "
         "Don't print found factors.\n");
  printf(" -v --verbose          "
         "Print some extra messages. -vv prints more.\n");
  printf(" -h --help             "
         "Print this help.\n\n");

  exit(EXIT_SUCCESS);
}

static const char *short_opts =
  "b:n:N:p:P:i:o:f:s:l:L:B:G:xzZA:dqvh";

static const struct option long_opts[] =
  {
    {"base",       required_argument, 0, 'b'},
    {"nmin",       required_argument, 0, 'n'},
    {"nmax",       required_argument, 0, 'N'},
    {"pmin",       required_argument, 0, 'p'},
    {"pmax",       required_argument, 0, 'P'},
    {"carol",      no_argument,       0, 'C'},
    {"kynea",      no_argument,       0, 'K'},
    {"input",      required_argument, 0, 'i'},
    {"output",     required_argument, 0, 'o'},
    {"factors",    required_argument, 0, 'f'},
    {"save",       required_argument, 0, 's'},
    {"lower-priority", no_argument,   0, 'z'},
    {"raise-priority", no_argument,   0, 'Z'},
    {"affinity",   required_argument, 0, 'A'},
    {"quiet",      no_argument,       0, 'q'},
    {"verbose",    no_argument,       0, 'v'},
    {"help",       no_argument,       0, 'h'},
    {0, 0, 0, 0}
  };


static time_t start_date;
static int opt_ind, opt_c;

static void attribute ((noreturn)) argerror(const char *str)
{
  if (long_opts[opt_ind].val == opt_c)
    error("--%s %s: argument %s.", long_opts[opt_ind].name, optarg, str);
  else
    error("-%c %s: argument %s.", opt_c, optarg, str);
}

static uint64_t parse_uint(uint64_t limit)
{
  uint64_t num;
  uint32_t expt;
  char *tail;

  errno = 0;
  num = strtoull(optarg,&tail,0);

  if (errno == 0 && num <= limit)
  {
    switch (*tail)
    {
      case 'e':
      case 'E':
        expt = strtoul(tail+1,&tail,0);
        if (errno != 0)
          goto range_error;
        if (*tail != '\0')
          break;
        while (expt-- > 0)
          if (num > limit/10)
            goto range_error;
          else
            num *= 10;
      case '\0':
        return num;
    }

    argerror("is malformed");
  }

 range_error:
  argerror("is out of range");
}

int main(int argc, char **argv)
{
  const char *input_file_name = NULL;
  int priority_opt =  -2;
  int want_help = 0;

  set_accumulated_time(0.0);

#if HAVE_MALLOPT
  /* All memory allocation takes place during initialization, so it doesn't
     have to be fast. Reducing this threshold allows more memory to be
     returned to the system when free() is called.
  */
  mallopt(M_MMAP_THRESHOLD,16000);
#endif

#if USE_COMMAND_LINE_FILE
  /* If no comand line arguments are given, attempt to read them from file.
   */
  if (argc == 1)
    read_argc_argv(&argc,&argv,NAME COMMAND_LINE_FILE_NAME_SUFFIX);
#endif

  while ((opt_c = getopt_long(argc,argv,short_opts,long_opts,&opt_ind)) != -1)
    switch (opt_c)
    {
      case 'b':
        b_term = parse_uint(UINT32_MAX);
        break;
      case 'n':
        n_min = parse_uint(UINT32_MAX);
        break;
      case 'N':
        n_max = parse_uint(UINT32_MAX);
        break;
      case 'p':
        p_min = parse_uint(UINT64_MAX);
        break;
      case 'P':
        p_max = parse_uint(UINT64_MAX);
        break;
      case 'i':
        input_file_name = optarg;
        break;
      case 'o':
        output_file_name = optarg;
        break;
      case 'f':
        factors_file_name = optarg;
        break;
      case 's':
        save_period = strtoul(optarg,NULL,0) * 60;
        break;
      case 'z':
        priority_opt--;
        break;
      case 'Z':
        priority_opt++;
        break;
      case 'A':
        set_cpu_affinity(strtol(optarg,NULL,0));
        break;
      case 'q':
        quiet_opt++;
        break;
      case 'v':
        verbose_opt++;
        break;
      case 'h':
        want_help = 1;
        break;
      default:
        return 1;
    }

  banner();

  if (want_help || argc < 2)
    help();


  /* We install these signal handlers early because the default handler
     terminates the program. SIGTERM and SIGINT handlers are not installed
     until sieving begins, but termination is the right default for them.
  */
#ifdef SIGUSR1
  signal(SIGUSR1,handle_signal);
#endif
#ifdef SIGUSR2
  signal(SIGUSR2,handle_signal);
#endif

  if (priority_opt)
    set_process_priority(priority_opt);

  if (optind < argc)
    error("Unhandled argument %s.", argv[optind]);
 
  if (input_file_name == NULL)
  {
    if (b_term == 0)
      error("Must specify input file name or base.");
    if (b_term % 2 == 1)
      error("--base must be even.");
    if (n_min == 0)
      error("--nmax is a required argument.");
    if (n_max == 0)
      error("--nmin is a required argument.");
    build_terms();
  }
  
  if (input_file_name != NULL)
  {
     if (b_term != 0)
       error("Can only specify one of input file name or base.");
    if (input_file_name != NULL)
      read_abc_file(input_file_name);
  }

   if (p_max == 0)
      error("--pmax is a required argument.");
   if (output_file_name == NULL)
   {
      sprintf(default_output_file_name, "ck_%d.pfgw", b_term);
      output_file_name = default_output_file_name;
   }

  if (p_min>=p_max || p_max >= UINT64_C(1)<<MOD64_MAX_BITS)
    error("Sieve range P0 <= p <= P1 must be in 3< P0 < P1 < 2^%u.\n",
          (unsigned int)MOD64_MAX_BITS);

   sieve();

  return EXIT_SUCCESS;
}

/* Return the fraction of the current range that has been done.
 */
double frac_done(uint64_t p)
{
  return (double)(p-p_min)/(p_max-p_min);
}

#define STRFTIME_FORMAT "ETA %d %b %H:%M"
void print_status(uint64_t p, uint32_t p_per_sec, uint32_t secs_per_factor)
{
  static int toggle = 0;
  char buf1[32], buf2[32];

  toggle = !toggle; /* toggle between reporting ETA and factor rate. */

  //if (toggle && factor_count && p_per_sec)
  {
    // uint64_t p_per_factor = (p-work_pmin)/factor_count;
    snprintf(buf1,31,"%d sec/factor", secs_per_factor); //p_per_factor/p_per_sec);
  }
  //else
  {
    double done = (double)(p-p_min)/(p_max-p_min);
    buf2[0] = '\0';
    if (done > 0.0) /* Avoid division by zero */
    {
      time_t finish_date = start_date+(time(NULL)-start_date)/done;
      struct tm *finish_tm = localtime(&finish_date);
      if (!finish_tm || !strftime(buf2,sizeof(buf2),STRFTIME_FORMAT,finish_tm))
        buf2[0] = '\0';
    }
  }
  buf2[31] = '\0';

  report(0,"p=%"PRIu64", %"PRIu32" p/sec, %"PRIu32" factor%s, %.1f%% done, %s, %s",
         p,p_per_sec,factor_count,plural(factor_count),100.0*frac_done(p),buf1,buf2);
}

static double expected_factors(uint32_t n, uint64_t p0, uint64_t p1)
{
  // TODO: Use a more accurate formula. This one is only reasonable when
  // p0/p1 is close to 1.

  return n*(1-log(p0)/log(p1));
}

void start_cksieve(void)
{
  if (verbose_opt)
    report(1,"Expecting to find factors for about %.2f terms.",
           expected_factors(n_count,p_min,p_max));

  logger(1,"%s started: %"PRIu32" <= n <= %"PRIu32", %"PRIu64" <= p <= %"PRIu64,
         NAME " " XSTR(MAJOR_VER) "." XSTR(MINOR_VER) "." XSTR(PATCH_VER),
         n_min, n_max, p_min, p_max);

  start_date = time(NULL);
}

void finish_cksieve(const char *reason, uint64_t p)
{
  logger(1,"%s stopped: at p=%"PRIu64" because %s.",
         NAME " " XSTR(MAJOR_VER) "." XSTR(MINOR_VER) "." XSTR(PATCH_VER),
         p, reason);

  write_abc_file(1, p, output_file_name);

  logger(verbose_opt,"Found factors for %"PRIu32" term%s in %.3f sec."
         " (expected about %.2f)",factor_count,plural(factor_count),
         get_accumulated_time(),expected_factors(n_count,p_min,p));
}
