from __future__ import (unicode_literals, division, absolute_import, print_function)

from PIL import Image
import io
import posixpath
import re

from .message_logging import log
from .utilities import (
        disable_debug_log, EXTS_OF_MIMETYPE, RESOURCE_TYPE_OF_EXT, convert_jxr_to_tiff, convert_pdf_to_jpeg,
        font_file_ext, image_file_ext, root_filename, urlrelpath)
from .yj_structure import SYMBOL_FORMATS

from .python_transition import (IS_PYTHON2)
if IS_PYTHON2:
    from .python_transition import (repr, urllib)
else:
    import urllib.parse


__license__ = "GPL v3"
__copyright__ = "2021, John Howell <jhowell@acm.org>"


USE_HIGHEST_RESOLUTION_IMAGE_VARIANT = True
FIX_JPEG_XR = True
FIX_PDF = True
IMAGE_QUALITY = 90


class Obj(object):
    def __init__(self, **kwargs):
        self.__dict__.update(kwargs)


class KFX_EPUB_Resources(object):
    def process_external_resource(self, resource_name, save=True, process_referred=False, save_referred=False,
                                  is_referred=False, is_variant=False, is_plugin=False):

        if is_referred and (not save) and resource_name in self.used_external_resources:
            return None

        if save_referred:
            process_referred = True

        self.used_external_resources.add(resource_name)

        resource = self.get_fragment(ftype="$164", fid=resource_name, delete=False).copy()

        if resource.pop("$175", "") != resource_name:
            raise Exception("Name of resource %s is incorrect" % resource_name)

        format = resource.pop("$161", None)

        if format in SYMBOL_FORMATS:
            extension = "." + SYMBOL_FORMATS[format]
        elif format is not None:
            log.error("Resource %s has unknown format: %s" % (resource_name, format))
            extension = ".bin"

        if is_plugin and format not in ["$287", "$284"]:
            log.error("Unexpected plugin resource format %s for %s" % (format, resource_name))
        elif (is_plugin is False) and format == "$287":
            log.error("Unexpected non-plugin resource format %s for %s" % (format, resource_name))

        fixed_height = resource.pop("$67", None)
        fixed_width = resource.pop("$66", None)

        resource_height = resource.pop("$423", None) or fixed_height
        resource_width = resource.pop("$422", None) or fixed_width

        if "$636" in resource:
            tile_height = resource.pop("$638")
            tile_width = resource.pop("$637")

            with disable_debug_log():
                full_image = Image.new("RGB", (resource_width, resource_height))

                for y, row in enumerate(resource.pop("$636")):
                    for x, location in enumerate(row):
                        tile_raw_media = self.locate_raw_media(location)
                        if tile_raw_media is not None:
                            tile = Image.open(io.BytesIO(tile_raw_media))
                            full_image.paste(tile, (x * tile_width, y * tile_height))
                            tile.close()

                if full_image.size != (resource_width, resource_height):
                    log.error("Combined tiled image size is (%d, %d) but should be (%d, %d)" % (
                            full_image.size[0], full_image.size[1], resource_width, resource_height))

                outfile = io.BytesIO()
                full_image.save(outfile, "jpeg" if extension == ".jpg" else extension[1:], quality=IMAGE_QUALITY)
                raw_media = outfile.getvalue()

            location = location.partition("-tile")[0]
        else:
            location = resource.pop("$165")
            search_path = resource.pop("$166", location)
            if search_path != location:
                log.error("Image resource %s has location %s != search_path %s" % (resource_name, location, search_path))

            raw_media = self.locate_raw_media(location)

        mime = resource.pop("$162", None)

        if mime in EXTS_OF_MIMETYPE:
            if extension == ".pobject" or extension == ".bin":
                if mime == "figure":
                    extension = image_file_ext(raw_media)
                else:
                    extension = EXTS_OF_MIMETYPE[mime][0]
        elif mime is not None:
            log.error("Resource %s has unknown mime type: %s" % (resource_name, mime))

        location_fn = location

        location_fn = resource.pop("yj.conversion.source_resource_filename", location_fn)
        location_fn = resource.pop("yj.authoring.source_file_name", location_fn)

        if (extension == ".pobject" or extension == ".bin") and "." in location_fn:
            extension = "." + location_fn.rpartition(".")[2]

        if not location_fn.endswith(extension):
            location_fn = location_fn.partition(".")[0] + extension

        resource.pop("$597", None)
        resource.pop("$57", None)
        resource.pop("$56", None)
        resource.pop("$499", None)
        resource.pop("$500", None)
        resource.pop("$137", None)
        resource.pop("$136", None)

        if process_referred:
            for rr in resource.pop("$167", []):
                self.process_external_resource(rr, is_referred=True, save=save_referred, is_plugin=None)
        else:
            resource.pop("$167", None)

        if "$214" in resource:
            self.process_external_resource(resource.pop("$214"), is_referred=is_referred, save=False)

        if FIX_JPEG_XR and (format == "$548") and (raw_media is not None):
            try:
                tiff_data = convert_jxr_to_tiff(raw_media, location_fn)
            except Exception as e:
                log.error("Exception during conversion of JPEG-XR '%s' to TIFF: %s" % (location_fn, repr(e)))
            else:
                with disable_debug_log():
                    img = Image.open(io.BytesIO(tiff_data))
                    ofmt, extension = ("PNG", ".png") if img.mode == "RGBA" else ("JPEG", ".jpg")
                    outfile = io.BytesIO()
                    img.save(outfile, ofmt, quality=IMAGE_QUALITY)
                    img.close()

                raw_media = outfile.getvalue()
                location_fn = location_fn.rpartition(".")[0] + extension

        suffix = ""
        if FIX_PDF and format == "$565" and raw_media is not None and "$564" in resource:
            page_num = resource["$564"] + 1
            try:
                jpeg_data = convert_pdf_to_jpeg(raw_media, page_num, reported_errors=self.reported_pdf_errors)
            except Exception as e:
                log.error("Exception during conversion of PDF \"%s\" page %d to JPEG: %s" % (location_fn, page_num, repr(e)))
            else:
                raw_media = jpeg_data
                extension = ".jpg"
                location_fn = location_fn.rpartition(".")[0] + extension
                suffix = "-page%d" % page_num
                resource.pop("$564")

        if is_referred:
            filename = root_filename(location)
        else:
            filename = self.resource_location_filename(location_fn, suffix, self.epub.IMAGE_FILEPATH)

        if not is_variant:
            for rr in resource.pop("$635", []):
                variant = self.process_external_resource(rr, is_referred=is_referred, is_variant=True, save=False)
                if (
                        USE_HIGHEST_RESOLUTION_IMAGE_VARIANT and save and variant is not None and
                        variant.width > resource_width and variant.height > resource_height):
                    if self.DEBUG:
                        log.info("Replacing image %s (%dx%d) with variant %s (%dx%d)" % (
                                filename, resource_width, resource_height, variant.filename, variant.width, variant.height))

                    filename, raw_media = variant.filename, variant.raw_media
                    resource_width, resource_height = variant.width, variant.height

        if save and (raw_media is not None):
            base_filename = filename
            cnt = 0
            while filename in self.epub.oebps_files:
                if self.epub.oebps_files[filename].binary_data == raw_media:
                    manifest_entry = self.epub.manifest_files[filename]
                    break

                if is_referred:
                    log.error("Multiple resources exist with location %s" % location)

                fn, ext = posixpath.splitext(base_filename)
                filename = "%s_%d%s" % (fn, cnt, ext)
                cnt += 1
            else:
                manifest_entry = self.epub.manifest_resource(
                        filename, data=raw_media, height=resource_height, width=resource_width,
                        mimetype=mime if is_referred else None)
        else:
            manifest_entry = None

        if "$564" in resource:
            filename += "#page=%d" % (resource.pop("$564") + 1)

        self.check_empty(resource, "resource %s" % resource_name)

        return Obj(filename=filename, raw_media=raw_media, format=format, mime=mime, width=resource_width, height=resource_height,
                   manifest_entry=manifest_entry)

    def locate_raw_media(self, location, report_missing=True):
        try:
            raw_media = self.book_data["$417"][location]
            self.used_raw_media.add(location)
        except Exception:
            if report_missing:
                log.error("Missing bcRawMedia %s" % location)

            raw_media = None

        return raw_media

    def resource_location_filename(self, location, suffix, filepath_template):

        if (location, suffix) in self.location_filenames:
            return self.location_filenames[(location, suffix)]

        if location.startswith("/"):
            location = "_" + location[1:]

        safe_location = re.sub(r"[^A-Za-z0-9_/.-]", "_", location)
        safe_location = safe_location.replace("//", "/x/")

        path, sep, name = safe_location.rpartition("/")
        path += sep

        root, sep, ext = name.rpartition(".")
        ext = sep + ext
        resource_type = RESOURCE_TYPE_OF_EXT.get(ext, "resource")

        unique_part = self.unique_part_of_local_symbol(root)
        root = self.prefix_unique_part_of_symbol(unique_part, resource_type)

        for prefix in ["resource/", filepath_template[1:].partition("/")[0] + "/"]:
            if path.startswith(prefix):
                path = path[len(prefix):]

        safe_filename = filepath_template % ("%s%s%s%s" % (path, root, suffix, ext))

        unique_count = 0
        oebps_files_lower = set([n.lower() for n in self.epub.oebps_files.keys()])

        while safe_filename.lower() in oebps_files_lower:
            safe_filename = filepath_template % ("%s%s%s-%d%s" % (path, root, suffix, unique_count, ext))
            unique_count += 1

        self.location_filenames[(location, suffix)] = safe_filename
        return safe_filename

    def process_fonts(self):
        fonts = self.book_data.pop("$262", {})
        raw_fonts = self.book_data.pop("$418", {})
        raw_media = self.book_data.get("$417", {})
        used_fonts = {}

        for font in fonts.values():
            location = font.pop("$165")

            if location in used_fonts:
                font["src"] = "url(\"%s\")" % urllib.parse.quote(urlrelpath(used_fonts[location], ref_from=self.epub.STYLES_CSS_FILEPATH))
            elif location in raw_fonts or (self.book.is_kpf_prepub and location in raw_media):
                raw_font = raw_fonts.pop(location, None) or raw_media.pop(location)

                filename = location
                if "." not in filename:
                    ext = font_file_ext(raw_font)
                    if not ext:
                        log.error("Font %s has unknown type (possibly obfuscated)" % filename)
                        ext = ".font"

                    filename = "%s%s" % (filename, ext)

                filename = self.resource_location_filename(filename, "", self.epub.FONT_FILEPATH)

                if filename not in self.epub.oebps_files:
                    self.epub.manifest_resource(filename, data=raw_font)

                font["src"] = "url(\"%s\")" % urlrelpath(urllib.parse.quote(filename), ref_from=self.epub.STYLES_CSS_FILEPATH)
                used_fonts[location] = filename
            else:
                log.error("Missing bcRawFont %s" % location)

            for prop in ["$15", "$12", "$13"]:
                if prop in font and font[prop] == "$350":
                    font.pop(prop)

            self.fix_font_name(font["$11"], add=True)
            self.font_faces.append(self.convert_yj_properties(font))

        for location in raw_fonts:
            log.warning("Unused font file: %s" % location)
            filename = self.resource_location_filename(location, "", self.epub.FONT_FILEPATH)
            self.epub.manifest_resource(filename, data=raw_fonts[location])

    def uri_reference(self, uri, save=True, save_referred=None, manifest_external_refs=False):
        purl = urllib.parse.urlparse(uri)

        if purl.scheme == "kfx":
            return self.process_external_resource(
                urllib.parse.unquote(purl.netloc + purl.path), is_plugin=None, save=save, save_referred=save_referred).filename

        if purl.scheme in ["navto", "navt"]:
            anchor = self.navto_anchor.get((urllib.parse.unquote(purl.netloc), float(purl.fragment) if purl.fragment else 0.0))
            if anchor is not None:
                return self.anchor_as_uri(anchor)
            else:
                log.error("Failed to locate anchor for %s" % uri)
                return "/MISSING_NAVTO#%s_%s" % (urllib.parse.unquote(purl.netloc), purl.fragment)

        if purl.scheme in ["http", "https"]:
            if manifest_external_refs:
                self.epub.manifest_resource(uri, external=True, report_dupe=False)

            return uri

        if purl.scheme != "mailto":
            log.error("Unexpected URI scheme: %s" % uri)

        return uri

    def unique_file_id(self, filename):
        if filename in self.file_ids:
            return self.file_ids[filename]

        id = re.sub(r"[^A-Za-z0-9.-]", "_", filename.rpartition("/")[2][:64])

        if not re.match(r"^[A-Za-z]", id[0]):
            id = "id_" + id

        if id in self.file_ids.values():
            base_id = id
            unique_count = 0
            while id in self.file_ids.values():
                id = "%s_%d" % (base_id, unique_count)
                unique_count += 1

        self.file_ids[filename] = id
        return id
