/* Copyright (C) 2012 Kristian Lauszus, TKJ Electronics. All rights reserved.

This software may be distributed and modified under the terms of the GNU
General Public License version 2 (GPL2) as published by the Free Software
Foundation and appearing in the file GPL2.TXT included in the packaging of
this file. Please note that GPL2 Section 2[b] requires that all works based
on this software must also be made publicly available under the terms of
the GPL2 ("Copyleft").

Contact information
-------------------

Kristian Lauszus, TKJ Electronics
Web      :  http://www.tkjelectronics.com
e-mail   :  kristianl@tkjelectronics.com
*/

#include "XBOXUSB.h"
//To enable serial debugging see "settings.h"
//#define EXTRADEBUG // Uncomment to get even more debugging data
//#define PRINTREPORT // Uncomment to print the report send by the Xbox 360 Controller

XBOXUSB::XBOXUSB(USB *p) : pUsb(p),     // pointer to USB class instance - mandatory
                           bAddress(0), // device address - mandatory
                           bPollEnable(false)
{ // don't start polling before dongle is connected
    for (uint8_t i = 0; i < 3; i++)
    {
        epInfo[i].epAddr = 0;
        epInfo[i].maxPktSize = (i) ? 0 : 8;
        epInfo[i].bmSndToggle = 0;
        epInfo[i].bmRcvToggle = 0;
        epInfo[i].bmNakPower = (i) ? USB_NAK_NOWAIT : USB_NAK_MAX_POWER;
    }

    if (pUsb)                            // register in USB subsystem
        pUsb->RegisterDeviceClass(this); //set devConfig[] entry
}

uint8_t XBOXUSB::Init(uint8_t parent, uint8_t port, bool lowspeed)
{
    const uint8_t dev_desc = sizeof(USB_DEVICE_DESCRIPTOR);
    uint8_t dev_desc_buf[dev_desc];

    //Hardcoded length is not ideal but should be enough to encompass interface and endpoint descriptors.
    //Not enough RAM in this chip to properly malloc this stuff.
    uint8_t conf_desc_buff[64];
    uint8_t num_endpoints;
    uint8_t rcode;
    UsbDevice *p = NULL;
    EpInfo *oldep_ptr = NULL;
    uint8_t interface_subclass, interface_protocol;
    USB_DEVICE_DESCRIPTOR *udd = reinterpret_cast<USB_DEVICE_DESCRIPTOR *>(dev_desc_buf);
    USB_INTERFACE_DESCRIPTOR *uid;
    //Config descriptor parser variables
    uint8_t cd_len = 0, cd_type = 0, cd_pos = 0;

    AddressPool &addrPool = pUsb->GetAddressPool();

    if (bAddress)
        return USB_ERROR_CLASS_INSTANCE_ALREADY_IN_USE;

    p = addrPool.GetUsbDevicePtr(0);

    if (!p)
        return USB_ERROR_ADDRESS_NOT_FOUND_IN_POOL;

    if (!p->epinfo)
        return USB_ERROR_EPINFO_IS_NULL;

    oldep_ptr = p->epinfo;
    p->epinfo = epInfo;
    p->lowspeed = lowspeed;

    rcode = pUsb->getDevDescr(0, 0, dev_desc, (uint8_t *)dev_desc_buf);
    if (rcode)
        goto FailGetDevDescr;

    rcode = pUsb->getConfDescr(0, 0, sizeof(conf_desc_buff), 0, conf_desc_buff);
    if (rcode)
        goto FailGetDevDescr;

    p->epinfo = oldep_ptr;

    uid = reinterpret_cast<USB_INTERFACE_DESCRIPTOR *>(conf_desc_buff +
                                                       sizeof(USB_CONFIGURATION_DESCRIPTOR));
    interface_subclass = uid->bInterfaceSubClass;
    interface_protocol = uid->bInterfaceProtocol;

    if ((interface_subclass != 0x5D ||  //Xbox360 wired bInterfaceSubClass
	     interface_protocol != 0x01))   //Xbox360 wired bInterfaceProtocol
        goto FailUnknownDevice;

    num_endpoints = uid->bNumEndpoints;

    if (num_endpoints != 2)
        goto FailUnknownDevice;

    // Allocate new address according to device class
    bAddress = addrPool.AllocAddress(parent, false, port);

    if (!bAddress)
        return USB_ERROR_OUT_OF_ADDRESS_SPACE_IN_POOL;

    // Extract Max Packet Size from device descriptor
    epInfo[0].maxPktSize = udd->bMaxPacketSize0;

    epInfo[XBOX_INPUT_PIPE].epAddr = 0;
    epInfo[XBOX_OUTPUT_PIPE].epAddr = 0;

    //Parse the configuration descriptor to find the two endpoint addresses
    while (epInfo[XBOX_INPUT_PIPE].epAddr == 0 || epInfo[XBOX_OUTPUT_PIPE].epAddr == 0)
    {
        if (cd_pos >= sizeof(conf_desc_buff) - 1)
            break;

        cd_len = conf_desc_buff[cd_pos];
        cd_type = conf_desc_buff[cd_pos + 1];

        if (cd_type == USB_ENDPOINT_DESCRIPTOR_TYPE)
        {
            USB_ENDPOINT_DESCRIPTOR *uepd = reinterpret_cast<USB_ENDPOINT_DESCRIPTOR *>(&conf_desc_buff[cd_pos]);
            if (uepd->bmAttributes == USB_TRANSFER_TYPE_INTERRUPT)
            {
                int pipe;
                (uepd->bEndpointAddress & 0x80) ? (pipe = XBOX_INPUT_PIPE) : (pipe = XBOX_OUTPUT_PIPE);
                epInfo[pipe].epAddr = uepd->bEndpointAddress & 0x7F;
                epInfo[pipe].epAttribs = uepd->bmAttributes;
            }
        }
        cd_pos += cd_len;
    }

    if (epInfo[XBOX_INPUT_PIPE].epAddr == 0 || epInfo[XBOX_OUTPUT_PIPE].epAddr == 0)
    {
        goto FailGetDevDescr;
    }

    // Assign new address to the device
    rcode = pUsb->setAddr(0, 0, bAddress);
    if (rcode)
    {
        p->lowspeed = false;
        addrPool.FreeAddress(bAddress);
        bAddress = 0;
#ifdef DEBUG_USB_HOST
        Notify(PSTR("\r\nsetAddr: "), 0x80);
        D_PrintHex<uint8_t>(rcode, 0x80);
#endif
        return rcode;
    }
#ifdef EXTRADEBUG
    Notify(PSTR("\r\nAddr: "), 0x80);
    D_PrintHex<uint8_t>(bAddress, 0x80);
#endif
    //delay(300); // Spec says you should wait at least 200ms

    p->lowspeed = false;

    //get pointer to assigned address record
    p = addrPool.GetUsbDevicePtr(bAddress);
    if (!p)
        return USB_ERROR_ADDRESS_NOT_FOUND_IN_POOL;

    p->lowspeed = lowspeed;

    // Assign epInfo to epinfo pointer - only EP0 is known
    rcode = pUsb->setEpInfoEntry(bAddress, 1, epInfo);
    if (rcode)
        goto FailSetDevTblEntry;

    /* The application will work in reduced host mode, so we can save program and data
    memory space. After verifying the VID we will use known values for the
    configuration values for device, interface, endpoints and HID for the XBOX360 Controllers */

    /* Initialize data structures for endpoints of device */
    epInfo[XBOX_INPUT_PIPE].epAttribs = USB_TRANSFER_TYPE_INTERRUPT;
    epInfo[XBOX_INPUT_PIPE].bmNakPower = USB_NAK_NOWAIT; // Only poll once for interrupt endpoints
    epInfo[XBOX_INPUT_PIPE].maxPktSize = EP_MAXPKTSIZE;
    epInfo[XBOX_INPUT_PIPE].bmSndToggle = 0;
    epInfo[XBOX_INPUT_PIPE].bmRcvToggle = 0;
    epInfo[XBOX_OUTPUT_PIPE].epAttribs = USB_TRANSFER_TYPE_INTERRUPT;
    epInfo[XBOX_OUTPUT_PIPE].bmNakPower = USB_NAK_NOWAIT; // Only poll once for interrupt endpoints
    epInfo[XBOX_OUTPUT_PIPE].maxPktSize = EP_MAXPKTSIZE;
    epInfo[XBOX_OUTPUT_PIPE].bmSndToggle = 0;
    epInfo[XBOX_OUTPUT_PIPE].bmRcvToggle = 0;

    rcode = pUsb->setEpInfoEntry(bAddress, 3, epInfo);
    if (rcode)
        goto FailSetDevTblEntry;

    delay(200); // Give time for address change

    rcode = pUsb->setConf(bAddress, epInfo[XBOX_CONTROL_PIPE].epAddr, 1);
    if (rcode)
        goto FailSetConfDescr;

#ifdef DEBUG_USB_HOST
    Notify(PSTR("\r\nXbox 360 Controller Connected\r\n"), 0x80);
#endif
    onInit();
    Xbox360Connected = true;
    bPollEnable = true;
    return 0; // Successful configuration

/* Diagnostic messages */
FailGetDevDescr:
#ifdef DEBUG_USB_HOST
    NotifyFailGetDevDescr();
    goto Fail;
#endif

FailSetDevTblEntry:
#ifdef DEBUG_USB_HOST
    NotifyFailSetDevTblEntry();
    goto Fail;
#endif

FailSetConfDescr:
#ifdef DEBUG_USB_HOST
    NotifyFailSetConfDescr();
#endif
    goto Fail;

FailUnknownDevice:
#ifdef DEBUG_USB_HOST
    NotifyFailUnknownDevice(VID, PID);
#endif
    rcode = USB_DEV_CONFIG_ERROR_DEVICE_NOT_SUPPORTED;

Fail:
#ifdef DEBUG_USB_HOST
    Notify(PSTR("\r\nXbox 360 Init Failed, error code: "), 0x80);
    NotifyFail(rcode);
#endif
    Release();
    return rcode;
}

/* Performs a cleanup after failed Init() attempt */
uint8_t XBOXUSB::Release()
{
    Xbox360Connected = false;
    pUsb->GetAddressPool().FreeAddress(bAddress);
    bAddress = 0;
    bPollEnable = false;
    return 0;
}

uint8_t XBOXUSB::Poll()
{
    if (!bPollEnable)
        return 0;
    uint16_t BUFFER_SIZE = EP_MAXPKTSIZE;
    pUsb->inTransfer(bAddress, epInfo[XBOX_INPUT_PIPE].epAddr, &BUFFER_SIZE, readBuf); // input on endpoint 1
    readReport();
#ifdef PRINTREPORT
    printReport(); // Uncomment "#define PRINTREPORT" to print the report send by the Xbox 360 Controller
#endif
    return 0;
}

void XBOXUSB::readReport()
{
    if (readBuf == NULL)
        return;
    if (readBuf[0] != 0x00 || readBuf[1] != 0x14)
    { // Check if it's the correct report - the controller also sends different status reports
        return;
    }

    ButtonState = (uint32_t)(readBuf[5] | ((uint16_t)readBuf[4] << 8) | ((uint32_t)readBuf[3] << 16) | ((uint32_t)readBuf[2] << 24));

    hatValue[LeftHatX] = (int16_t)(((uint16_t)readBuf[7] << 8) | readBuf[6]);
    hatValue[LeftHatY] = (int16_t)(((uint16_t)readBuf[9] << 8) | readBuf[8]);
    hatValue[RightHatX] = (int16_t)(((uint16_t)readBuf[11] << 8) | readBuf[10]);
    hatValue[RightHatY] = (int16_t)(((uint16_t)readBuf[13] << 8) | readBuf[12]);

    if (ButtonState != OldButtonState)
    {
        ButtonClickState = (ButtonState >> 16) & ((~OldButtonState) >> 16);
        if (((uint8_t)OldButtonState) == 0 && ((uint8_t)ButtonState) != 0) // The L2 and R2 buttons are special as they are analog buttons
            R2Clicked = true;
        if ((uint8_t)(OldButtonState >> 8) == 0 && (uint8_t)(ButtonState >> 8) != 0)
            L2Clicked = true;
        OldButtonState = ButtonState;
    }
}

void XBOXUSB::printReport()
{ //Uncomment "#define PRINTREPORT" to print the report send by the Xbox 360 Controller
#ifdef PRINTREPORT
    if (readBuf == NULL)
        return;
    for (uint8_t i = 0; i < XBOX_REPORT_BUFFER_SIZE; i++)
    {
        D_PrintHex<uint8_t>(readBuf[i], 0x80);
        Notify(PSTR(" "), 0x80);
    }
    Notify(PSTR("\r\n"), 0x80);
#endif
}

uint8_t XBOXUSB::getButtonPress(ButtonEnum b)
{
    if (b == L2) // These are analog buttons
        return (uint8_t)(ButtonState >> 8);
    else if (b == R2)
        return (uint8_t)ButtonState;
    return (bool)(ButtonState & ((uint32_t)pgm_read_word(&XBOX_BUTTONS[(uint8_t)b]) << 16));
}

bool XBOXUSB::getButtonClick(ButtonEnum b)
{
    if (b == L2)
    {
        if (L2Clicked)
        {
            L2Clicked = false;
            return true;
        }
        return false;
    }
    else if (b == R2)
    {
        if (R2Clicked)
        {
            R2Clicked = false;
            return true;
        }
        return false;
    }
    uint16_t button = pgm_read_word(&XBOX_BUTTONS[(uint8_t)b]);
    bool click = (ButtonClickState & button);
    ButtonClickState &= ~button; // clear "click" event
    return click;
}

int16_t XBOXUSB::getAnalogHat(AnalogHatEnum a)
{
    return hatValue[a];
}

/* Xbox Controller commands */
void XBOXUSB::XboxCommand(uint8_t *data, uint16_t nbytes)
{
    uint32_t timeout;

    uint8_t rcode = hrNAK;
    while (millis() - outPipeTimer < 1);

    timeout= millis();
    while (rcode != hrSUCCESS && (millis() - timeout) < 50)
        rcode = pUsb->outTransfer(bAddress, epInfo[XBOX_OUTPUT_PIPE].epAddr, nbytes, data);

    //Readback any response
    rcode = hrSUCCESS;
    timeout = millis();
    while (rcode != hrNAK && (millis() - timeout) < 50)
    {
        uint16_t bufferSize = EP_MAXPKTSIZE;
        rcode = pUsb->inTransfer(bAddress, epInfo[XBOX_INPUT_PIPE].epAddr, &bufferSize, readBuf);
        if (bufferSize > 0)
            readReport();
    }
    outPipeTimer = millis();
}

void XBOXUSB::setLedRaw(uint8_t value)
{
    writeBuf[0] = 0x01;
    writeBuf[1] = 0x03;
    writeBuf[2] = value;

    XboxCommand(writeBuf, 3);
}

void XBOXUSB::setLedOn(LEDEnum led)
{
    if (led == OFF)
        setLedRaw(0);
    else if (led != ALL) // All LEDs can't be on a the same time
        setLedRaw(pgm_read_byte(&XBOX_LEDS[(uint8_t)led]) + 4);
}

void XBOXUSB::setLedBlink(LEDEnum led)
{
    setLedRaw(pgm_read_byte(&XBOX_LEDS[(uint8_t)led]));
}

void XBOXUSB::setLedMode(LEDModeEnum ledMode)
{ // This function is used to do some special LED stuff the controller supports
    setLedRaw((uint8_t)ledMode);
}

void XBOXUSB::setRumbleOn(uint8_t lValue, uint8_t rValue)
{
    writeBuf[0] = 0x00;
    writeBuf[1] = 0x08;
    writeBuf[2] = 0x00;
    writeBuf[3] = lValue; // big weight
    writeBuf[4] = rValue; // small weight
    writeBuf[5] = 0x00;
    writeBuf[6] = 0x00;
    writeBuf[7] = 0x00;

    XboxCommand(writeBuf, 8);
}

void XBOXUSB::onInit()
{
    uint8_t stringDescriptor[10];
    //8bit-do appears as a Wired Xbox 360 controller, but will quickly change to a switch controller if you do not request a string descriptor
    //Request string descriptors
    pUsb->ctrlReq(bAddress, epInfo[XBOX_CONTROL_PIPE].epAddr, 0x80, 0x06, 0x02, 0x03, 0x0409, 0x0002, 2, stringDescriptor, NULL);
    delay(1);
    //Request string descriptors
    pUsb->ctrlReq(bAddress, epInfo[XBOX_CONTROL_PIPE].epAddr, 0x80, 0x06, 0x02, 0x03, 0x0409, 0x0022, 10, stringDescriptor, NULL);
    delay(1);


    uint8_t outBuf0[3] = {0x01, 0x03, 0x02};
    uint8_t outBuf1[3] = {0x01, 0x03, 0x06};
    uint8_t outBuf2[3] = {0x02, 0x08, 0x03}; //Not sure what this. Seen in windows driver
    uint8_t outBuf3[3] = {0x00, 0x03, 0x00}; //Turn off rumble?
    XboxCommand(outBuf0, 3);
    XboxCommand(outBuf1, 3);
    XboxCommand(outBuf2, 3);
    XboxCommand(outBuf3, 3);

    if (pFuncOnInit)
        pFuncOnInit(); // Call the user function
    else
        setLedOn(static_cast<LEDEnum>(LED1));
}
