const SteamUser = require("steam-user");
const SteamID = require("steamid");
const unzipper = require("unzipper");
const path = require("path");
const fs = require("fs");
const request = require("request");
const Coordinator = require("./Coordinator.js");

module.exports = class Helper {
	static decodeUids(input) {
		let bytes = [];

		let byte = input.readUint8();
		while (byte !== 0x00) {
			bytes.push(byte);
			byte = input.readUint8();
		}
		bytes.push(0);

		let b = Buffer.from(bytes);
		let results = [];
		let position = 0;

		while (position < b.length) {
			let lookahead = 0;
			let token = b.readInt8(position);
			let backup = token;
			let v5 = 0;
			let v7 = 0;
			let v9 = 0;

			do {
				backup = token;
				lookahead += 1;
				v9 = (token & 0x7F) << v7;
				v7 += 7;
				v5 |= v9;

				if (token === 0 || backup >= 0) {
					break;
				}

				token = b.readInt8(position + lookahead);
			} while (v7 < 35);

			if (backup < 0) {
				break;
			}

			position += lookahead;

			if (v5 === 0) {
				continue;
			}

			results.push(SteamID.fromIndividualAccountID(v5));
		}

		return results;
	}

	static encodeUids(steamids) {
		let outputBuffer = [];

		for (let id of steamids.map(s => s.accountid)) {
			while (id > 0x7F) {
				outputBuffer.push((id | 0x80) & 0xFF);
				id >>= 7;
			}

			outputBuffer.push(id);
		}

		outputBuffer.push(0x00);

		return Buffer.from(outputBuffer);
	}

	static IPStrToInt(ipString) {
		// Copied from: DoctorMcKay/node-stdlib
		// https://github.com/DoctorMcKay/node-stdlib/blob/3a65f4116116fb8a0a82239a9cc0db35c44558d9/components/ipv4.js
		let buf = Buffer.alloc(4);
		let octets = ipString.split(".");

		for (let i = 0; i < 4; i++) {
			buf[i] = parseInt(octets[i], 10);
		}

		return buf.readUInt32BE(0);
	}

	static deleteRecursive(dir) {
		return new Promise((resolve, reject) => {
			fs.readdir(dir, async (err, files) => {
				if (err) {
					reject(err);
					return;
				}

				for (let file of files) {
					let filePath = path.join(dir, file);
					let stat = fs.statSync(filePath);

					if (stat.isDirectory()) {
						await this.deleteRecursive(filePath);
					} else {
						await new Promise((res, rej) => {
							fs.unlink(filePath, (err) => {
								if (err) {
									rej(err);
									return;
								}

								res();
							});
						});
					}
				}

				fs.rmdir(dir, (err) => {
					if (err) {
						reject(err);
						return;
					}

					resolve();
				});
			});
		});
	}

	static downloadProtobufs(dir) {
		return new Promise(async (resolve, reject) => {
			let newProDir = path.join(dir, "Protobufs-master");
			if (fs.existsSync(newProDir)) {
				await this.deleteRecursive(newProDir);
			}

			// Yes I know the ones I download here are technically not the same as the ones in the submodule
			// but that doesn't really matter, I doubt Valve will do any major changes with the protobufs I use here anyways
			let r = request("https://github.com/SteamDatabase/Protobufs/archive/master.zip");
			let pipe = r.pipe(unzipper.Extract({ path: dir }));
			pipe.on("close", async () => {
				let proDir = path.join(dir, "protobufs");
				if (fs.existsSync(proDir)) {
					await this.deleteRecursive(proDir);
				}

				fs.rename(newProDir, proDir, (err) => {
					if (err) {
						reject(err);
						return;
					}

					resolve();
				});
			});
			pipe.on("error", reject);
		});
	}

	static verifyProtobufs() {
		let user = new SteamUser();
		let coord = new Coordinator(user);

		try {
			return typeof coord.Protos.csgo.EGCBaseClientMsg.k_EMsgGCClientHello === "number";
		} catch (e) {
			return false;
		}
	}

}
