#pragma once

#include <cstdint>
#include <unordered_map>
#include <type_traits>

struct Vector3
{
	float x;
	float y;
	float z;
};

typedef uintptr_t(*GetNativeAddressFunc)(uint64_t hash);
namespace Native
{
	class Context
	{
		// Internal RAGE stuff
		uint64_t* retVal = stack;
		uint64_t argCount = 0;
		uint64_t* stackPtr = stack;
		uint64_t dataCount = 0;
		uint64_t spaceForResults[24];

		// Our stack
		uint64_t stack[24]{ 0 };

	public:
		template<class T>
		T& At(uint32_t idx)
		{
			static_assert(sizeof(T) <= 8, "Argument is too big");

			return *reinterpret_cast<T*>(stack + idx);
		}

		uint32_t GetArgsCount()
		{
			return argCount;
		}

		void SetArgsCount(uint32_t idx)
		{
			argCount = idx;
		}

		template<class T, class... Args>
		void Push(T arg, Args... args)
		{
			static_assert(sizeof(T) <= 8, "Argument is too big");

			*(T*)(stack + argCount++) = arg;

			if constexpr (sizeof...(Args) > 0)
				Push(args...);
		}

		template<class T>
		T Result()
		{
			return *reinterpret_cast<T*>(retVal);
		}
		template<>
		void Result<void>() { }

		template<>
		Vector3 Result<Vector3>()
		{
			Vector3 vec;
			vec.x = *(float*)((uintptr_t)retVal + 0);
			vec.y = *(float*)((uintptr_t)retVal + 8);
			vec.z = *(float*)((uintptr_t)retVal + 16);
			return vec;
		}

		void Reset()
		{
			argCount = 0;
			dataCount = 0;
		}

		void CopyResults()
		{
			uint64_t a1 = (uint64_t)this;

			uint64_t result;

			for (; *(uint32_t*)(a1 + 24); *(uint32_t*)(*(uint64_t*)(a1 + 8i64 * *(signed int*)(a1 + 24) + 32) + 16i64) = result)
			{
				--* (uint32_t*)(a1 + 24);
				**(uint32_t * *)(a1 + 8i64 * *(signed int*)(a1 + 24) + 32) = *(uint32_t*)(a1 + 16 * (*(signed int*)(a1 + 24) + 4i64));
				*(uint32_t*)(*(uint64_t*)(a1 + 8i64 * *(signed int*)(a1 + 24) + 32) + 8i64) = *(uint32_t*)(a1
					+ 16i64
					* *(signed int*)(a1 + 24)
					+ 68);
				result = *(unsigned int*)(a1 + 16i64 * *(signed int*)(a1 + 24) + 72);
			}
			-- * (uint32_t*)(a1 + 24);
		}
	};

	typedef void(__cdecl * Handler)(Context * context);

	inline GetNativeAddressFunc & GetFunc()
	{
		static GetNativeAddressFunc _nativeAddressGetFunc;
		return _nativeAddressGetFunc;
	}

	inline void SetEssentialFunction(GetNativeAddressFunc func)
	{
		GetFunc() = func;
	}

	inline Handler GetHandler(uint64_t hashName)
	{
		return (Handler)GetFunc()(hashName);
	}

	void Invoke(Handler fn, Context& ctx)
	{
		if (!fn)
			return;

		__try {
			fn(&ctx);
		}
		__except (EXCEPTION_EXECUTE_HANDLER) {
		}
		__try {
			ctx.CopyResults();
		}
		__except (EXCEPTION_EXECUTE_HANDLER) {
		}
	}


	template<class Retn = uint64_t, class... Args>
	Retn Invoke(Handler fn, Args... args)
	{
		static Context ctx;

		if (!fn) return Retn();

		ctx.Reset();

		if constexpr (sizeof...(Args) > 0)
			ctx.Push(args...);

		Invoke(fn, ctx);
		return ctx.Result<Retn>();
	}

	template<class Retn = uint64_t, class... Args>
	Retn Invoke(uint64_t hashName, Args... args)
	{
		return Invoke<Retn>(GetHandler(hashName), args...);
	}

	inline void Invoke(uint64_t hashName, Context & ctx)
	{
		Invoke(GetHandler(hashName), ctx);
	}
};

struct native_dumper_t
{
	std::string name;
	uint64_t hash;
};

extern std::vector< native_dumper_t > native_dump_list = {
{  "WAIT", 0x4EDE34FBADD967A6 },
{  "START_NEW_SCRIPT", 0xE81651AD79516E48 },
{  "START_NEW_SCRIPT_WITH_ARGS", 0xB8BA7F44DF1575E1 },
{  "START_NEW_SCRIPT_WITH_NAME_HASH", 0xEB1C67C3A5333A92 },
{  "START_NEW_SCRIPT_WITH_NAME_HASH_AND_ARGS", 0xC4BB298BD441BE78 },
{  "TIMERA", 0x83666F9FB8FEBD4B },
{  "TIMERB", 0xC9D9444186B5A374 },
{  "SETTIMERA", 0xC1B1E9A034A63A62 },
{  "SETTIMERB", 0x5AE11BC36633DE4E },
{  "TIMESTEP", 0x0000000050597EE2 },
{  "SIN", 0x0BADBFA3B172435F },
{  "COS", 0xD0FFB162F40A139C },
{  "SQRT", 0x71D93B57D07F9804 },
{  "POW", 0xE3621CC40F31FE2E },
{  "VMAG", 0x652D2EEEF1D3E62C },
{  "VMAG2", 0xA8CEACB4F35AE058 },
{  "VDIST", 0x2A488C176D52CCA5 },
{  "VDIST2", 0xB7A628320EFF8E47 },
{  "SHIFT_LEFT", 0xEDD95A39E5544DE8 },
{  "SHIFT_RIGHT", 0x97EF1E5BCE9DC075 },
{  "FLOOR", 0xF34EE736CF047844 },
{  "CEIL", 0x11E019C8F43ACC8A },
{  "ROUND", 0xF2DB717A73826179 },
{  "TO_FLOAT", 0xBBDA792448DB5A89 },
{  "_SET_THREAD_PRIORITY", 0x42B65DEEF2EDF2A1 },
// Category: _NAMESPACE1
// Category: _NAMESPACE2
// Category: _NAMESPACE3
// Category: _NAMESPACE4
// Category: _NAMESPACE5
// Category: AUDIO
{  "CREATE_NEW_SCRIPTED_CONVERSATION", 0xD2C91A0B572AAE56 },
{  "ADD_PED_TO_CONVERSATION", 0x95D9F4BC443956E7 },
{  "START_SCRIPT_CONVERSATION", 0x6B17C62C9635D2DC },
{  "PRELOAD_SCRIPT_CONVERSATION", 0x3B3CAD6166916D87 },
{  "START_PRELOADED_CONVERSATION", 0x23641AFE870AF385 },
{  "IS_SCRIPTED_CONVERSATION_LOADED", 0xDF0D54BE7A776737 },
{  "GET_CURRENT_SCRIPTED_CONVERSATION_LINE", 0x480357EE890C295A },
{  "PAUSE_SCRIPTED_CONVERSATION", 0x8530AD776CD72B12 },
{  "RESTART_SCRIPTED_CONVERSATION", 0x9AEB285D1818C9AC },
{  "STOP_SCRIPTED_CONVERSATION", 0xD79DEEFB53455EBA },
{  "SKIP_TO_NEXT_SCRIPTED_CONVERSATION_LINE", 0x9663FE6B7A61EB00 },
{  "REGISTER_SCRIPT_WITH_AUDIO", 0xC6ED9D5092438D91 },
{  "UNREGISTER_SCRIPT_WITH_AUDIO", 0xA8638BE228D4751A },
{  "REQUEST_SCRIPT_AUDIO_BANK", 0x2F844A8B08D76685 },
{  "RELEASE_NAMED_SCRIPT_AUDIO_BANK", 0x77ED170667F50170 },
{  "RELEASE_SCRIPT_AUDIO_BANK", 0x7A2D8AD0A9EB9C3F },
{  "GET_SOUND_ID", 0x430386FE9BF80B45 },
{  "RELEASE_SOUND_ID", 0x353FC880830B88FA },
{  "PLAY_SOUND", 0x7FF4944CC209192D },
{  "PLAY_SOUND_FRONTEND", 0x67C540AA08E4A6F5 },
{  "_PLAY_AMBIENT_SPEECH1", 0x8E04FEDD28D42462 },
{  "_PLAY_AMBIENT_SPEECH_AT_COORDS", 0xED640017ED337E45 },
{  "PLAY_PAIN", 0xBC9AE166038A5CEC },
{  "SET_AMBIENT_VOICE_NAME", 0x6C8065A3B780185B },
{  "STOP_CURRENT_PLAYING_SPEECH", 0x79D2F0E66F81D90D },
{  "STOP_CURRENT_PLAYING_AMBIENT_SPEECH", 0xB8BEC0CA6F0EDB0F },
{  "IS_AMBIENT_SPEECH_PLAYING", 0x9072C8B49907BFAD },
{  "IS_SCRIPTED_SPEECH_PLAYING", 0xCC9AA18DCC7084F4 },
{  "IS_ANY_SPEECH_PLAYING", 0x729072355FA39EC9 },
{  "_CAN_PED_SPEAK", 0x49B99BF3FDA89A7A },
{  "IS_PED_IN_CURRENT_CONVERSATION", 0x049E937F18F4020C },
{  "SET_PED_IS_DRUNK", 0x95D2D383D5396B8A },
{  "PLAY_ANIMAL_VOCALIZATION", 0xEE066C7006C49C0A },
{  "IS_ANIMAL_VOCALIZATION_PLAYING", 0xC265DF9FB44A9FBD },
{  "SET_ANIMAL_MOOD", 0xCC97B29285B1DC3B },
{  "SET_STATIC_EMITTER_ENABLED", 0x399D2D3B33F1B8EB },
{  "PLAY_END_CREDITS_MUSIC", 0xCD536C4D33DCC900 },
{  "SET_AMBIENT_ZONE_STATE", 0xBDA07E5950085E46 },
{  "CLEAR_AMBIENT_ZONE_STATE", 0x218DD44AAAC964FF },
{  "SET_AMBIENT_ZONE_LIST_STATE", 0x9748FA4DE50CCE3E },
{  "CLEAR_AMBIENT_ZONE_LIST_STATE", 0x120C48C614909FA4 },
{  "SET_AMBIENT_ZONE_STATE_PERSISTENT", 0x1D6650420CEC9D3B },
{  "SET_AMBIENT_ZONE_LIST_STATE_PERSISTENT", 0xF3638DAE8C4045E1 },
{  "IS_HORN_ACTIVE", 0x9D6BFC12B05C6121 },
{  "IS_STREAM_PLAYING", 0xD11FA52EB849D978 },
{  "LOAD_STREAM", 0x1F1F957154EC51DF },
{  "PLAY_STREAM_FROM_PED", 0x89049DD63C08B5D1 },
{  "PLAY_STREAM_FRONTEND", 0x58FCE43488F9F5F4 },
{  "PLAY_STREAM_FROM_POSITION", 0x21442F412E8DE56B },
{  "STOP_STREAM", 0xA4718A1419D18151 },
{  "STOP_PED_SPEAKING", 0x9D64D7405520E3D3 },
{  "DISABLE_PED_PAIN_AUDIO", 0xA9A41C1E940FB0E8 },
{  "IS_AMBIENT_SPEECH_DISABLED", 0x932C2D096A2C3FFF },
{  "SET_HORN_ENABLED", 0x76D683C108594D0E },
{  "SET_AUDIO_VEHICLE_PRIORITY", 0xE5564483E407F914 },
{  "_FORCE_VEHICLE_ENGINE_AUDIO", 0x4F0C413926060B38 },
{  "SET_GPS_ACTIVE", 0x3BD3F52BA9B1E4E8 },
{  "START_AUDIO_SCENE", 0x013A80FC08F6E4F2 },
{  "STOP_AUDIO_SCENE", 0xDFE8422B3B94E688 },
{  "STOP_AUDIO_SCENES", 0xBAC7FC81A75EC1A1 },
{  "IS_AUDIO_SCENE_ACTIVE", 0xB65B60556E2A9225 },
{  "SET_AUDIO_SCENE_VARIABLE", 0xEF21A9EF089A2668 },
{  "ADD_ENTITY_TO_AUDIO_MIX_GROUP", 0x153973AB99FE8980 },
{  "REMOVE_ENTITY_FROM_AUDIO_MIX_GROUP", 0x18EB48CFC41F2EA0 },
{  "AUDIO_IS_SCRIPTED_MUSIC_PLAYING", 0x845FFC3A4FEEFA3E },
{  "PREPARE_MUSIC_EVENT", 0x1E5185B72EF5158A },
{  "CANCEL_MUSIC_EVENT", 0x5B17A90291133DA5 },
{  "TRIGGER_MUSIC_EVENT", 0x706D57B0F50DA710 },
{  "GET_MUSIC_PLAYTIME", 0xE7A0D23DC414507B },
{  "SET_AUDIO_FLAG", 0xB9EFD5C25018725A },
{  "SET_PORTAL_SETTINGS_OVERRIDE", 0x044DBAD7A7FA2BE5 },
{  "REMOVE_PORTAL_SETTINGS_OVERRIDE", 0xB4BBFD9CD8B3922B },
// Category: _NAMESPACE7
// Category: BRAIN
{  "REGISTER_OBJECT_SCRIPT_BRAIN", 0x16AF9B4EEAC3B305 },
{  "ENABLE_SCRIPT_BRAIN_SET", 0x1CF6E5C6750EADBD },
{  "DISABLE_SCRIPT_BRAIN_SET", 0x3F44EA613A5B2676 },
// Category: CAM
{  "RENDER_SCRIPT_CAMS", 0x33281167E4942E4F },
{  "_RENDER_FIRST_PERSON_CAM", 0x8C7C7FF7CF0E5153 },
{  "CREATE_CAM", 0xE72CDBA7F0A02DD6 },
{  "CREATE_CAM_WITH_PARAMS", 0x40C23491CE83708E },
{  "CREATE_CAMERA", 0x57CDF879EA466C46 },
{  "CREATE_CAMERA_WITH_PARAMS", 0x98B99B9F27E2D60B },
{  "DESTROY_CAM", 0x4E67E0B6D7FD5145 },
{  "DESTROY_ALL_CAMS", 0x163600D6E136C9F8 },
{  "DOES_CAM_EXIST", 0x153AD457764FD704 },
{  "SET_CAM_ACTIVE", 0x87295BCA613800C8 },
{  "IS_CAM_ACTIVE", 0x63EFCC7E1810B8E6 },
{  "IS_CAM_RENDERING", 0x4415F8A6C536D39F },
{  "GET_RENDERING_CAM", 0x03A8931ECC8015D6 },
{  "GET_CAM_COORD", 0x6B12F11C2A9F0344 },
{  "GET_CAM_ROT", 0x9BF96B57254E7889 },
{  "GET_CAM_FOV", 0x8101D32A0A6B0F60 },
{  "SET_CAM_PARAMS", 0xA47BBFFFB83D4D0A },
{  "SET_CAM_COORD", 0xF9EE7D419EE49DE6 },
{  "SET_CAM_ROT", 0x63DFA6810AD78719 },
{  "SET_CAM_FOV", 0x27666E5988D9D429 },
{  "SET_CAM_NEAR_CLIP", 0xA924028272A61364 },
{  "SET_CAM_FAR_CLIP", 0x5E32817BF6302111 },
{  "SET_CAM_MOTION_BLUR_STRENGTH", 0x45FD891364181F9E },
{  "ATTACH_CAM_TO_ENTITY", 0xFDC0DF7F6FB0A592 },
{  "ATTACH_CAM_TO_PED_BONE", 0xDFC1E4A44C0324CA },
{  "DETACH_CAM", 0x05B41DDBEB559556 },
{  "POINT_CAM_AT_COORD", 0x948B39341C3A40C2 },
{  "POINT_CAM_AT_ENTITY", 0xFC2867E6074D3A61 },
{  "STOP_CAM_POINTING", 0xCA1B30A3357C71F1 },
{  "SET_CAM_AFFECTS_AIMING", 0x3CB9E8BDE5E76F33 },
{  "ADD_CAM_SPLINE_NODE", 0xF1F57F9D230F9CD1 },
{  "SET_CAM_SPLINE_PHASE", 0xF1898A68E7C15636 },
{  "GET_CAM_SPLINE_PHASE", 0x095EDCD24D90033A },
{  "SET_CAM_SPLINE_DURATION", 0xFF6311652CA91015 },
{  "SET_CAM_SPLINE_SMOOTHING_STYLE", 0x84B3645618E726B0 },
{  "SET_CAM_ACTIVE_WITH_INTERP", 0x8B15AE2987C1AC8F },
{  "IS_CAM_INTERPOLATING", 0x578F8F1CAA17BD2B },
{  "SHAKE_CAM", 0xF9A7BCF5D050D4E7 },
{  "IS_CAM_SHAKING", 0x2EEB402BD7320159 },
{  "STOP_CAM_SHAKING", 0xB78CC4B4706614B0 },
{  "PLAY_CAM_ANIM", 0xA263DDF694D563F6 },
{  "IS_SCREEN_FADED_OUT", 0xF5472C80DF2FF847 },
{  "IS_SCREEN_FADED_IN", 0x37F9A426FBCF4AF2 },
{  "IS_SCREEN_FADING_OUT", 0x02F39BEFE7B88D00 },
{  "IS_SCREEN_FADING_IN", 0x0CECCC63FFA2EF24 },
{  "DO_SCREEN_FADE_IN", 0x6A053CF596F67DF7 },
{  "DO_SCREEN_FADE_OUT", 0x40C719A5E410B9E4 },
{  "SET_WIDESCREEN_BORDERS", 0xD7F4D54CF80AFA34 },
{  "GET_GAMEPLAY_CAM_COORD", 0x595320200B98596E },
{  "GET_GAMEPLAY_CAM_ROT", 0x0252D2B5582957A6 },
{  "GET_GAMEPLAY_CAM_FOV", 0xF6A96E5ACEEC6E50 },
{  "GET_GAMEPLAY_CAM_RELATIVE_HEADING", 0xC4ABF536048998AA },
{  "SET_GAMEPLAY_CAM_RELATIVE_HEADING", 0x5D1EB123EAC5D071 },
{  "GET_GAMEPLAY_CAM_RELATIVE_PITCH", 0x99AADEBBA803F827 },
{  "SET_GAMEPLAY_CAM_RELATIVE_PITCH", 0xFB760AF4F537B8BF },
{  "SHAKE_GAMEPLAY_CAM", 0xD9B31B4650520529 },
{  "IS_GAMEPLAY_CAM_SHAKING", 0xEA4C5F4AA0A4DBEF },
{  "SET_GAMEPLAY_CAM_SHAKE_AMPLITUDE", 0x570E35F5C4A44838 },
{  "STOP_GAMEPLAY_CAM_SHAKING", 0xE0DE43D290FB65F9 },
{  "IS_GAMEPLAY_CAM_RENDERING", 0x8660EA714834E412 },
{  "IS_GAMEPLAY_CAM_LOOKING_BEHIND", 0x8FE0D24FFD04D5A2 },
{  "IS_SPHERE_VISIBLE", 0x2E941B5FFA2989C6 },
{  "_CLAMP_GAMEPLAY_CAM_YAW", 0x14F3947318CA8AD2 },
{  "_CLAMP_GAMEPLAY_CAM_PITCH", 0x326C7AA308F3DF6A },
{  "_ANIMATE_GAMEPLAY_CAM_ZOOM", 0x2126C740A4AC370B },
{  "_DISABLE_FIRST_PERSON_CAM_THIS_FRAME", 0x9C473089A934C930 },
{  "IS_FOLLOW_VEHICLE_CAM_ACTIVE", 0xA40C2F51FB589E9A },
{  "IS_FIRST_PERSON_AIM_CAM_ACTIVE", 0xF63134C54B6EC212 },
{  "_GET_GAMEPLAY_CAM_ZOOM", 0xB4132CA1B0EE1365 },
{  "_SET_FIRST_PERSON_CAM_PITCH_RANGE", 0x715B7F5E8BED32A2 },
{  "_GET_GAMEPLAY_CAM_COORDS", 0x5352E025EC2B416F },
{  "_GET_GAMEPLAY_CAM_ROT", 0x602685BD85DD26CA },
{  "SET_GAMEPLAY_COORD_HINT", 0xFA33B8C69A4A6A0F },
{  "SET_GAMEPLAY_PED_HINT", 0x90FB951648851733 },
{  "SET_GAMEPLAY_VEHICLE_HINT", 0xE2B2BB7DAC280515 },
{  "SET_GAMEPLAY_OBJECT_HINT", 0xC40551D65F2BF297 },
{  "SET_GAMEPLAY_ENTITY_HINT", 0xD1F7F32640ADFD12 },
{  "IS_GAMEPLAY_HINT_ACTIVE", 0x2E04AB5FEE042D4A },
{  "STOP_GAMEPLAY_HINT", 0x1BCEC33D54CFCA8A },
{  "SET_GAMEPLAY_HINT_FOV", 0x661E58BC6F00A49A },
{  "SET_CINEMATIC_BUTTON_ACTIVE", 0xB90411F480457A6C },
{  "IS_CINEMATIC_CAM_RENDERING", 0xBF7C780731AADBF8 },
{  "_DISABLE_VEHICLE_FIRST_PERSON_CAM_THIS_FRAME", 0xA5929C2E57AC90D1 },
{  "SET_CINEMATIC_MODE_ACTIVE", 0xCE7A90B160F75046 },
// Category: CLOCK
{  "SET_CLOCK_TIME", 0x3A52C59FFB2DEED8 },
{  "PAUSE_CLOCK", 0x4D1A590C92BF377E },
{  "ADVANCE_CLOCK_TIME_TO", 0x0184750AE88D0B1D },
{  "ADD_TO_CLOCK_TIME", 0xAB7C251C7701D336 },
{  "GET_CLOCK_HOURS", 0xC82CF208C2B19199 },
{  "GET_CLOCK_MINUTES", 0x4E162231B823DBBF },
{  "GET_CLOCK_SECONDS", 0xB6101ABE62B5F080 },
{  "SET_CLOCK_DATE", 0x02AD3092562941E2 },
{  "GET_CLOCK_DAY_OF_WEEK", 0x4DD02D4C7FB30076 },
{  "GET_CLOCK_DAY_OF_MONTH", 0xDF2FD796C54480A5 },
{  "GET_CLOCK_MONTH", 0x2D44E8FC79EAB1AC },
{  "GET_CLOCK_YEAR", 0xE136DCA28C4A48BA },
{  "GET_MILLISECONDS_PER_GAME_MINUTE", 0xE4CB8D126501EC52 },
{  "GET_POSIX_TIME", 0x90338AD4A784E455 },
// Category: _NAMESPACE11
// Category: _NAMESPACE12
// Category: _NAMESPACE13
// Category: _NAMESPACE14
// Category: _NAMESPACE15
{  "NETWORK_CLAN_SERVICE_IS_VALID", 0x579CCED0265D4896 },
{  "NETWORK_CLAN_PLAYER_IS_ACTIVE", 0xB124B57F571D8F18 },
{  "NETWORK_CLAN_PLAYER_GET_DESC", 0xEEE6EACBE8874FBA },
{  "_NETWORK_CLAN_GET_NUM_MEMBERSHIP_DESC", 0x1F471B79ACC90BEF },
{  "NETWORK_CLAN_GET_MEMBERSHIP_DESC", 0x48DE78AF2C8885B8 },
{  "NETWORK_GET_PRIMARY_CLAN_DATA_CLEAR", 0x9AA46BADAD0E27ED },
{  "NETWORK_GET_PRIMARY_CLAN_DATA_START", 0xCE86D8191B762107 },
{  "NETWORK_GET_PRIMARY_CLAN_DATA_PENDING", 0xB5074DB804E28CE7 },
{  "NETWORK_GET_PRIMARY_CLAN_DATA_SUCCESS", 0x5B4F04F19376A0BA },
{  "NETWORK_GET_PRIMARY_CLAN_DATA_NEW", 0xC080FF658B2E41DA },
// Category: DATABINDING
{  "_GET_DATA_CONTAINER_FROM_PATH", 0x0C827D175F1292F2 },
{  "_GET_DATA_CONTAINER_FROM_CHILD_INDEX", 0x0C827D175F1292F3 },
{  "_ADD_DATA_CONTAINER_FROM_PATH", 0x0C827D175F1292F4 },
{  "_ADD_DATA_CONTAINER_BY_HASH", 0x98BB14345BB68257 },
{  "_ADD_DATA_CONTAINER", 0xEB4F9A3537EEABCD },
{  "_ADD_DATA_BOOL_FROM_PATH", 0x37BB86A751148A6A },
{  "_ADD_DATA_BOOL_BY_HASH", 0xBC95D3AE2ECA70D6 },
{  "_ADD_DATA_BOOL", 0x58BAA5F635DA2FF4 },
{  "_ADD_DATA_INT_BY_HASH", 0x267F9527F4350ADE },
{  "_ADD_DATA_INT", 0x307A3247C5457BDE },
{  "_ADD_DATA_FLOAT", 0x5154228273ADB9A6 },
{  "_ADD_DATA_HASH_BY_HASH", 0x8E173DFB041993C6 },
{  "_ADD_DATA_HASH", 0x8538F1205D60ECA6 },
{  "_ADD_DATA_STRING_BY_HASH", 0xEAD09E76E22630C3 },
{  "_ADD_DATA_STRING", 0x617FCA1C5652BBAD },
{  "_ADD_DATA_GANG_ID", 0x7D0F2014DB28DD00 },
{  "_ADD_UI_ITEM_LIST_FROM_PATH", 0xDB5B9A474148F699 },
{  "_ADD_UI_ITEM_LIST_BY_HASH", 0x3C7799283325181B },
{  "_ADD_UI_ITEM_LIST", 0xFE74FA57E0CE6824 },
{  "_INSERT_UI_ITEM_TO_LIST_FROM_CONTEXT_STRING_ALIAS", 0x5859E970794D92F3 },
{  "_INSERT_UI_ITEM_TO_LIST_FROM_CONTEXT_HASH_ALIAS", 0xEE97A05C05F16E41 },
{  "_INSERT_UI_ITEM_TO_LIST_FROM_PATH_STRING_ALIAS", 0x5740774F608E4FC8 },
{  "_REMOVE_BINDING_ARRAY_ITEM_BY_DATA_CONTEXT_ID", 0xF68B1726EAF7B285 },
{  "_CLEAR_BINDING_ARRAY", 0xA1F15C1D03DF802D },
{  "_GET_ITEM_CONTEXT_BY_INDEX", 0xE96D7F9FEFCC105F },
{  "_SET_TEMPLATED_UI_ITEM_LIST_SIZE", 0xFE74FA57E0CE6825 },
{  "_SET_TEMPLATED_UI_ITEM_HASH_ALIAS", 0x0AE7138D0541F2DE },
{  "_ADD_HASH_ARRAY", 0x52F5F08278EA5D75 },
{  "_ADD_STRING_ARRAY", 0x1B23E0627BDBFE85 },
{  "_GET_ARRAY_COUNT", 0xD23F5DE04FE717E2 },
{  "_WRITE_DATA_SCRIPT_VARIABLES", 0xAB888B4B91046771 },
{  "_WRITE_DATA_BOOL", 0xAB888B4B91046770 },
{  "_WRITE_DATA_BOOL_FROM_PARENT", 0xBDFE546E4C2D0E21 },
{  "_WRITE_DATA_INT", 0x335C3F6B3766B8D9 },
{  "_WRITE_DATA_INT_FROM_PARENT", 0x9EFA98238BA08FC4 },
{  "_WRITE_DATA_INT_FROM_PARENT_BY_HASH", 0x9D6E10A41D6ED6EC },
{  "_WRITE_DATA_FLOAT", 0xDF504BECEB15DA93 },
{  "_WRITE_DATA_HASH_STRING", 0xACDEF586BD71B1FD },
{  "_WRITE_DATA_HASH_STRING_FROM_PARENT", 0x0971F04E1EAA7AE8 },
{  "_WRITE_DATA_HASH_STRING_FROM_PARENT_BY_HASH", 0x20209529689E0953 },
{  "_WRITE_DATA_STRING", 0xE1BD342F2872AEE9 },
{  "_WRITE_DATA_STRING_FROM_PARENT", 0x4FF713B2F17A391E },
{  "_READ_DATA_BOOL", 0x5EEFBD4B6D7CD6EB },
{  "_READ_DATA_BOOL_FROM_PARENT", 0xA8EDE09FE07BD77F },
{  "_READ_DATA_BOOL_FROM_PARENT_BY_HASH", 0x4CDC3FDDFAE07EB3 },
{  "_READ_DATA_INT", 0x570784D782597512 },
{  "_READ_DATA_INT_FROM_PARENT", 0xFFC566A4801F6B40 },
{  "_READ_DATA_INT_FROM_PARENT_BY_HASH", 0xB5F668B648EC0970 },
{  "_READ_DATA_HASH_STRING_FROM_PARENT", 0x9B535990B01B62DE },
{  "_READ_DATA_HASH_STRING_FROM_PARENT_BY_HASH", 0x1F43BC25A119B252 },
{  "_READ_DATA_STRING", 0x3D290B5FFA7C5151 },
{  "_READ_DATA_STRING_FROM_PARENT", 0x6323AD277C4A2AFB },
{  "_IS_DATA_ID_VALID", 0x1E7130793AAAAB8D },
{  "_REMOVE_DATA_ENTRY", 0x0AE9938D0541F2DA },
{  "_VIRTUAL_COLLECTION_SET_SIZE", 0x9DCE9B01A93B58BC },
{  "_VIRTUAL_COLLECTION_SET_INTEREST_INDEX", 0x49A8447533308BCF },
{  "_VIRTUAL_COLLECTION_ADD_ITEM", 0x6DCBF187221CF73D },
{  "_VIRTUAL_COLLECTION_RESET", 0x09D95666ED2B5F60 },
{  "_DOES_VIRTUAL_COLLECTION_EXIST", 0x37963B56755BFB35 },
// Category: DATAFILE
{  "DATAFILE_WATCH_REQUEST_ID", 0xA5834834CA8FD7FC },
{  "DATAFILE_CREATE", 0x56B7291FB953DD51 },
{  "DATAFILE_DELETE", 0x9FB90EEDEA9F2D5C },
{  "DATAFILE_GET_FILE_DICT", 0xBBD8CF823CAE557C },
{  "_OBJECT_VALUE_ADD_INTEGER", 0x26FDF5E99AA2F3E9 },
{  "_OBJECT_VALUE_GET_BOOLEAN", 0x175E915A486EE548 },
{  "_OBJECT_VALUE_GET_INTEGER", 0x9D896A3B87D96E2B },
{  "_OBJECT_VALUE_GET_FLOAT", 0x814643ECA258ADF5 },
{  "_OBJECT_VALUE_GET_STRING", 0xE37B38C0B4E95DFA },
{  "_OBJECT_VALUE_GET_VECTOR3", 0xE459C941431E0FFA },
{  "_OBJECT_VALUE_GET_OBJECT", 0x4D7A30130F46AC9C },
{  "_OBJECT_VALUE_GET_ARRAY", 0x1B5447CF18544B18 },
{  "_OBJECT_VALUE_GET_TYPE", 0x92E11E3CA4C7CDF0 },
{  "_ARRAY_VALUE_GET_BOOLEAN", 0xAB1231D2DE52F2D3 },
{  "_ARRAY_VALUE_GET_INTEGER", 0x96DEA500B6EBBE53 },
{  "_ARRAY_VALUE_GET_FLOAT", 0xA9D003CF419CB81E },
{  "_ARRAY_VALUE_GET_STRING", 0xB6790A8FF80F889F },
{  "_ARRAY_VALUE_GET_VECTOR3", 0x850DA2750DA14E9A },
{  "_ARRAY_VALUE_GET_OBJECT", 0xA010655985853485 },
{  "_ARRAY_VALUE_GET_SIZE", 0x6A885BF69239E539 },
{  "_ARRAY_VALUE_GET_TYPE", 0x151DAFE6B3B9888F },
{  "_DATAFILE_UNLOAD", 0x129567F0C05F81B9 },
{  "_DATAFILE_REGISTER_QUERY", 0xAE156A747C39A741 },
{  "_DATAFILE_GET_DATA_NODE_INDEX", 0x83C3ED532B6E5D07 },
{  "_DATAFILE_GET_NUM_NODES", 0xDF01B1F7A886B42D },
{  "_DATAFILE_GET_NUM_CHILDREN", 0x6BEB168D5195E7AB },
{  "_DATAFILE_GET_HASH", 0xFBFF3FF2F5E80C0B },
{  "_DATAFILE_GET_STRING", 0x08EAF8E9F2EB7B2E },
{  "_DATAFILE_GET_VECTOR", 0x06FBF89B12DA279C },
{  "_DATAFILE_GET_FLOAT", 0x7F034FC3E891B57A },
{  "_DATAFILE_GET_INT", 0xEF44ACC657352A35 },
{  "_DATAFILE_GET_BOOL", 0x0D9138F3F8261DF7 },
// Category: _NAMESPACE18
// Category: DECORATOR
{  "DECOR_SET_BOOL", 0xFE26E4609B1C3772 },
{  "DECOR_SET_FLOAT", 0x238F8B0C1C7FE834 },
{  "DECOR_SET_INT", 0xE88F4D7F52A6090F },
{  "DECOR_GET_BOOL", 0xDEF3F1B071ABB197 },
{  "DECOR_GET_FLOAT", 0xE5FF70CD842CA9D4 },
{  "DECOR_GET_INT", 0x44DB62727762FD9B },
{  "DECOR_EXIST_ON", 0xD9D1CDBF3464DCDF },
{  "DECOR_REMOVE", 0x2BA7F5877A088A1D },
{  "DECOR_REGISTER", 0x0B253D644E3C36B3 },
{  "DECOR_IS_REGISTERED_AS_TYPE", 0x72355278C069F272 },
// Category: DLC
{  "IS_DLC_PRESENT", 0x2763DC12BBE2BB6F },
{  "GET_IS_LOADING_SCREEN_ACTIVE", 0x71D4BF5890659B0C },
// Category: ENTITY
{  "DOES_ENTITY_EXIST", 0xD42BD6EB2E0F1677 },
{  "DOES_ENTITY_BELONG_TO_THIS_SCRIPT", 0x622B1980CBE13332 },
{  "DOES_ENTITY_HAVE_DRAWABLE", 0x20487F0DA9AF164A },
{  "DOES_ENTITY_HAVE_PHYSICS", 0xA512B3F1B2A0B51C },
{  "HAS_ENTITY_BEEN_DAMAGED_BY_ANY_OBJECT", 0x73BB763880CD23A6 },
{  "HAS_ENTITY_BEEN_DAMAGED_BY_ANY_PED", 0x9934E9C42D52D87E },
{  "HAS_ENTITY_BEEN_DAMAGED_BY_ANY_VEHICLE", 0x695D7C26DE65C423 },
{  "HAS_ENTITY_BEEN_DAMAGED_BY_ENTITY", 0x7B6E7BEC1143AC86 },
{  "HAS_ENTITY_CLEAR_LOS_TO_ENTITY", 0xFCDFF7B72D23A1AC },
{  "HAS_ENTITY_CLEAR_LOS_TO_ENTITY_IN_FRONT", 0xE88F19660651D566 },
{  "HAS_ENTITY_COLLIDED_WITH_ANYTHING", 0xDF18751EC74F90FF },
{  "GET_ANIM_DURATION", 0x9FFAF4940A54CC09 },
{  "GET_ENTITY_ATTACHED_TO", 0x56D713888A566481 },
{  "GET_ENTITY_COORDS", 0xA86D5F069399F44D },
{  "GET_ENTITY_FORWARD_VECTOR", 0x2412D9C05BB09B97 },
{  "GET_ENTITY_FORWARD_X", 0xDB0954E9960F6457 },
{  "GET_ENTITY_FORWARD_Y", 0x9A5C073ECBDA7EE7 },
{  "GET_ENTITY_HEADING", 0xC230DD956E2F5507 },
{  "GET_ENTITY_HEALTH", 0x82368787EA73C0F7 },
{  "GET_ENTITY_MAX_HEALTH", 0x15D757606D170C3C },
{  "SET_ENTITY_MAX_HEALTH", 0x166E7CF68597D8B5 },
{  "GET_ENTITY_HEIGHT", 0x296DEBC84474B375 },
{  "GET_ENTITY_HEIGHT_ABOVE_GROUND", 0x0D3B5BAEA08F63E9 },
{  "GET_ENTITY_MATRIX", 0x3A9B1120AF13FBF2 },
{  "GET_ENTITY_MODEL", 0xDA76A9F39210D365 },
{  "GET_OFFSET_FROM_ENTITY_GIVEN_WORLD_COORDS", 0x497C6B1A2C9AE69C },
{  "GET_OFFSET_FROM_ENTITY_IN_WORLD_COORDS", 0x1899F328B0E12848 },
{  "GET_ENTITY_PITCH", 0xEF355ABEFF7F5005 },
{  "GET_ENTITY_ROLL", 0xBF966536FA8B6879 },
{  "GET_ENTITY_ROTATION", 0xE09CAF86C32CB48F },
{  "GET_ENTITY_SPEED", 0xFB6BA510A533DF81 },
{  "GET_ENTITY_SPEED_VECTOR", 0xF2DB09816A419DC5 },
{  "GET_ENTITY_UPRIGHT_VALUE", 0x56398BE65160C3BE },
{  "GET_ENTITY_VELOCITY", 0x4805D2B1D8CF94A9 },
{  "GET_OBJECT_INDEX_FROM_ENTITY_INDEX", 0x280BBE5601EAA983 },
{  "GET_PED_INDEX_FROM_ENTITY_INDEX", 0x0F16D042BD640EA3 },
{  "GET_VEHICLE_INDEX_FROM_ENTITY_INDEX", 0xDF1E5AAC561AFC59 },
{  "GET_WORLD_POSITION_OF_ENTITY_BONE", 0x82CFA50E34681CA5 },
{  "GET_NEAREST_PLAYER_TO_ENTITY", 0x990E294FC387FB88 },
{  "GET_NEAREST_PLAYER_TO_ENTITY_ON_TEAM", 0xB2C30C3B4AFF718C },
{  "GET_ENTITY_TYPE", 0x97F696ACA466B4E0 },
{  "GET_ENTITY_POPULATION_TYPE", 0xADE28862B6D7B85B },
{  "IS_AN_ENTITY", 0x27CFF3E5A286D3DF },
{  "IS_ENTITY_A_PED", 0xCF8176912DDA4EA5 },
{  "IS_ENTITY_A_MISSION_ENTITY", 0x138190F64DB4BBD1 },
{  "IS_ENTITY_A_VEHICLE", 0xC3D96AF45FCCEC4C },
{  "IS_ENTITY_AN_OBJECT", 0x0A27A546A375FDEF },
{  "IS_ENTITY_AT_COORD", 0x5E58342602E94718 },
{  "IS_ENTITY_AT_ENTITY", 0xC057F02B837A27F6 },
{  "IS_ENTITY_ATTACHED", 0xEE6AD63ABF59C0B7 },
{  "IS_ENTITY_ATTACHED_TO_ANY_OBJECT", 0x306C1F6178F01AB3 },
{  "IS_ENTITY_ATTACHED_TO_ANY_PED", 0xC841153DED2CA89A },
{  "IS_ENTITY_ATTACHED_TO_ANY_VEHICLE", 0x12DF6E0D2E736749 },
{  "IS_ENTITY_ATTACHED_TO_ENTITY", 0x154A3C529497053E },
{  "IS_ENTITY_DEAD", 0x7D5B1F88E7504BBA },
{  "IS_ENTITY_IN_AIR", 0x886E37EC497200B6 },
{  "IS_ENTITY_IN_ANGLED_AREA", 0xD3151E53134595E5 },
{  "IS_ENTITY_IN_AREA", 0x0C2634C40A16193E },
{  "IS_ENTITY_IN_WATER", 0xDDE5C125AC446723 },
{  "GET_ENTITY_SUBMERGED_LEVEL", 0x4A77C3F73FD9E831 },
{  "IS_ENTITY_ON_SCREEN", 0x613C15D5D8DB781F },
{  "IS_ENTITY_PLAYING_ANIM", 0xDEE49D5CA6C49148 },
{  "IS_ENTITY_STATIC", 0x86468ADFA0F6B861 },
{  "IS_ENTITY_TOUCHING_ENTITY", 0x9A2304A64C3C8423 },
{  "IS_ENTITY_TOUCHING_MODEL", 0x2AE3EBC8DEB9768B },
{  "IS_ENTITY_UPRIGHT", 0xF6F6AFD8D4FB2658 },
{  "IS_ENTITY_UPSIDEDOWN", 0x109DE3DA41AAD94A },
{  "IS_ENTITY_VISIBLE", 0xFFC96ECB7FA404CA },
{  "IS_ENTITY_VISIBLE_TO_SCRIPT", 0xF213C724E77F321A },
{  "IS_ENTITY_OCCLUDED", 0x140188E884645624 },
{  "WOULD_ENTITY_BE_OCCLUDED", 0x3546FAB293FF2981 },
{  "IS_ENTITY_WAITING_FOR_WORLD_COLLISION", 0x5E1CC2E8DC3111DD },
{  "APPLY_FORCE_TO_ENTITY_CENTER_OF_MASS", 0x31DA7CEC5334DB37 },
{  "APPLY_FORCE_TO_ENTITY", 0xF15E8F5D333F09C4 },
{  "ATTACH_ENTITY_TO_ENTITY", 0x6B9BBD38AB0796DF },
{  "ATTACH_ENTITY_TO_ENTITY_PHYSICALLY", 0xB629A43CA1643481 },
{  "GET_ENTITY_BONE_INDEX_BY_NAME", 0xBACA8FE9C76C124E },
{  "CLEAR_ENTITY_LAST_DAMAGE_ENTITY", 0xBB19AC7D4DCEFD0F },
{  "DELETE_ENTITY", 0x4CD38C78BD19A497 },
{  "DETACH_ENTITY", 0x64CDE9D6BF8ECAD3 },
{  "FREEZE_ENTITY_POSITION", 0x7D9EFB7AD6B19754 },
{  "_SET_ENTITY_SOMETHING", 0x740CB4F3F602C9F4 },
{  "SET_ENTITY_AS_MISSION_ENTITY", 0xDC19C288082E586E },
{  "SET_ENTITY_AS_NO_LONGER_NEEDED", 0x4971D2F8162B9674 },
{  "SET_PED_AS_NO_LONGER_NEEDED", 0x2595DD4236549CE3 },
{  "SET_VEHICLE_AS_NO_LONGER_NEEDED", 0x629BFA74418D6239 },
{  "SET_OBJECT_AS_NO_LONGER_NEEDED", 0x3AE22DEB5BA5A3E6 },
{  "SET_ENTITY_CAN_BE_DAMAGED", 0x0D06D522B90E861F },
{  "SET_ENTITY_CAN_BE_DAMAGED_BY_RELATIONSHIP_GROUP", 0x0EF1AFB18649E015 },
{  "SET_ENTITY_CAN_BE_TARGETED_WITHOUT_LOS", 0x6D09F32E284D0FB7 },
{  "GET_ENTITY_COLLISION_DISABLED", 0xAA2FADD30F45A9DA },
{  "SET_ENTITY_COLLISION", 0xF66F820909453B8C },
{  "SET_ENTITY_COMPLETELY_DISABLE_COLLISION", 0xE0580EC84813875A },
{  "SET_ENTITY_COORDS", 0x06843DA7060A026B },
{  "SET_ENTITY_COORDS_NO_OFFSET", 0x239A3351AC1DA385 },
{  "SET_ENTITY_DYNAMIC", 0xFBFC4473F66CE344 },
{  "SET_ENTITY_HEADING", 0xCF2B9C0645C4651B },
{  "SET_ENTITY_INVINCIBLE", 0xA5C38736C426FCB8 },
{  "SET_ENTITY_IS_TARGET_PRIORITY", 0x0A5D170C44CB2189 },
{  "SET_ENTITY_LOAD_COLLISION_FLAG", 0x9B9EE31AED48072E },
{  "HAS_COLLISION_LOADED_AROUND_ENTITY", 0xBEB1600952B9CF5C },
{  "SET_ENTITY_ONLY_DAMAGED_BY_PLAYER", 0x473598683095D430 },
{  "SET_ENTITY_ONLY_DAMAGED_BY_RELATIONSHIP_GROUP", 0x6C1F6AA2F0ADD104 },
{  "SET_ENTITY_PROOFS", 0xFAEE099C6F890BB8 },
{  "SET_ENTITY_QUATERNION", 0x100E7007D13E3687 },
{  "SET_ENTITY_ROTATION", 0x9CC8314DFEDE441E },
{  "SET_ENTITY_VISIBLE", 0x1794B4FCC84D812F },
{  "SET_ENTITY_VELOCITY", 0x1C99BB7B6E96D16F },
{  "SET_ENTITY_HAS_GRAVITY", 0x0CEDB728A1083FA7 },
{  "SET_ENTITY_LOD_DIST", 0x5FB407F0A7C877BF },
{  "GET_ENTITY_LOD_DIST", 0xDF240D0C2A948683 },
{  "SET_ENTITY_ALPHA", 0x0DF7692B1D9E7BA7 },
{  "GET_ENTITY_ALPHA", 0x1BB501624FAF2BEA },
{  "RESET_ENTITY_ALPHA", 0x744B9EF44779D9AB },
{  "SET_ENTITY_ALWAYS_PRERENDER", 0xACAD101E1FB66689 },
{  "SET_ENTITY_RENDER_SCORCHED", 0x85B8A7534E44BC23 },
{  "CREATE_MODEL_SWAP", 0x10B2218320B6F5AC },
{  "REMOVE_MODEL_SWAP", 0x824E1C26A14CB817 },
{  "CREATE_MODEL_HIDE", 0x069848B3FB3C4426 },
{  "CREATE_MODEL_HIDE_EXCLUDING_SCRIPT_OBJECTS", 0xD136090A9AAAB17D },
{  "REMOVE_MODEL_HIDE", 0x3F38A98576F6213A },
{  "CREATE_FORCED_OBJECT", 0x0961A905AFBC34C7 },
{  "REMOVE_FORCED_OBJECT", 0x553FA683F2BCD814 },
{  "SET_ENTITY_NO_COLLISION_ENTITY", 0xE037BF068223C38D },
{  "SET_ENTITY_MOTION_BLUR", 0x516C6ABD18322B63 },
{  "FIND_ANIM_EVENT_PHASE", 0x42718CC559BD7776 },
{  "FORCE_ENTITY_AI_AND_ANIMATION_UPDATE", 0x4C9E96473D4F1A88 },
{  "HAS_ANIM_EVENT_FIRED", 0x5851CC48405F4A07 },
{  "HAS_ENTITY_ANIM_FINISHED", 0xAEB40615337EF1E3 },
{  "PLAY_ENTITY_ANIM", 0xDC6D22FAB76D4874 },
{  "STOP_ENTITY_ANIM", 0x786591D986DE9159 },
// Category: EVENT
{  "SET_DECISION_MAKER", 0x8AE2F981CDDB8FA4 },
{  "ADD_SHOCKING_EVENT_AT_POSITION", 0xD9F8455409B525E9 },
{  "ADD_SHOCKING_EVENT_FOR_ENTITY", 0x7FD8F3BE76F89422 },
{  "IS_SHOCKING_EVENT_IN_SPHERE", 0x9DB47E16060D6354 },
{  "REMOVE_SHOCKING_EVENT", 0xE8BB3CC253A34559 },
{  "REMOVE_ALL_SHOCKING_EVENTS", 0xD47A168C2AB90DC4 },
{  "REMOVE_SHOCKING_EVENT_SPAWN_BLOCKING_AREAS", 0xDB249021652420C5 },
{  "SUPPRESS_SHOCKING_EVENTS_NEXT_FRAME", 0x84994FAD4E4E4E69 },
// Category: FIRE
{  "START_SCRIPT_FIRE", 0x6B83617E04503888 },
{  "REMOVE_SCRIPT_FIRE", 0x790125C36E194069 },
{  "START_ENTITY_FIRE", 0xC4DC7418A44D6822 },
{  "STOP_ENTITY_FIRE", 0x8390751DC40C1E98 },
{  "IS_ENTITY_ON_FIRE", 0x1BD7C371CE257C3E },
{  "GET_NUMBER_OF_FIRES_IN_RANGE", 0xF9617BC6FAE61E08 },
{  "STOP_FIRE_IN_RANGE", 0xDB38F247BD421708 },
{  "GET_CLOSEST_FIRE_POS", 0xB646FB657F448261 },
{  "ADD_EXPLOSION", 0x7D6F58F69DA92530 },
{  "ADD_OWNED_EXPLOSION", 0xD84A917A64D4D016 },
{  "ADD_EXPLOSION_WITH_USER_VFX", 0x53BA259F3A67A99E },
{  "IS_EXPLOSION_IN_AREA", 0x8391BA4313A25AD3 },
{  "IS_EXPLOSION_ACTIVE_IN_AREA", 0xD96E82AEBFFAAFF0 },
{  "IS_EXPLOSION_IN_SPHERE", 0xD62DD846D82CBB90 },
{  "IS_EXPLOSION_IN_ANGLED_AREA", 0x5AE661ECD18524C9 },
{  "_GET_ENTITY_INSIDE_EXPLOSION_AREA", 0x8002DDAB58594D78 },
// Category: _NAMESPACE24
// Category: GRAPHICS
{  "DRAW_LIGHT_WITH_RANGE", 0xD2D9E04C0DF927F4 },
{  "_UPDATE_ENTITY_SHIT_OR_SOMETHING", 0xBDBACB52A03CC760 },
{  "SET_CHECKPOINT_RGBA", 0xCAAFC225E33B1D15 },
{  "_SET_CHECKPOINT_ICON_RGBA", 0x99AFF17222D4DEB4 },
{  "DELETE_CHECKPOINT", 0x0DED5B0C8EBAAE12 },
{  "DRAW_RECT", 0x405224591DF02025 },
{  "SET_SCRIPT_GFX_DRAW_BEHIND_PAUSEMENU", 0x906B86E6D7896B9E },
{  "SET_SCRIPT_GFX_DRAW_ORDER", 0xCFCC78391C8B3814 },
{  "DRAW_SPRITE", 0xC9884ECADE94CB34 },
{  "ATTACH_TV_AUDIO_TO_ENTITY", 0x40866A418EB8EFDE },
{  "SET_TV_AUDIO_FRONTEND", 0x64437C98FCC5F291 },
{  "GET_SCREEN_RESOLUTION", 0x66773C92835D0909 },
{  "GET_SCREEN_COORD_FROM_WORLD_COORD", 0xCB50D7AFCC8B0EC6 },
{  "SET_ARTIFICIAL_LIGHTS_STATE", 0xB2797619A7C7747B },
{  "CREATE_TRACKED_POINT", 0xFB405CB357C69CB9 },
{  "SET_TRACKED_POINT_INFO", 0xF6FDA3D4404D4F2C },
{  "IS_TRACKED_POINT_VISIBLE", 0xCBB056BA159FB48D },
{  "DESTROY_TRACKED_POINT", 0x37A59922109F8F1C },
{  "_SET_FAR_SHADOWS_SUPPRESSED", 0x8FBFD2AEB196B369 },
{  "_CASCADESHADOWS_SET_TYPE", 0xCE4774E0F9AD48D1 },
{  "_CASCADESHADOWS_RESET_TYPE", 0xF7C29D7C12C36F03 },
{  "TOGGLE_PAUSED_RENDERPHASES", 0xEF9E1C45732F55FA },
{  "GET_TOGGLE_PAUSED_RENDERPHASES_STATUS", 0x86ED21BDB2791CE8 },
{  "START_PARTICLE_FX_NON_LOOPED_AT_COORD", 0x2E80BF72EF7C87AC },
{  "START_NETWORKED_PARTICLE_FX_NON_LOOPED_AT_COORD", 0xFB97618457994A62 },
{  "START_PARTICLE_FX_NON_LOOPED_ON_PED_BONE", 0x3FAA72BD940C3AC0 },
{  "START_PARTICLE_FX_NON_LOOPED_ON_ENTITY", 0xFF4C64C513388C12 },
{  "START_NETWORKED_PARTICLE_FX_NON_LOOPED_ON_ENTITY", 0xE6CFE43937061143 },
{  "SET_PARTICLE_FX_NON_LOOPED_COLOUR", 0x60B85BED6577A35B },
{  "START_PARTICLE_FX_LOOPED_AT_COORD", 0xBA32867E86125D3A },
{  "START_PARTICLE_FX_LOOPED_ON_PED_BONE", 0xE689C1B1432BB8AF },
{  "START_PARTICLE_FX_LOOPED_ON_ENTITY", 0xBD41E1440CE39800 },
{  "START_PARTICLE_FX_LOOPED_ON_ENTITY_BONE", 0xD3BA6EC7F2FBD5E9 },
{  "START_NETWORKED_PARTICLE_FX_LOOPED_ON_ENTITY", 0x8F90AB32E1944BDE },
{  "START_NETWORKED_PARTICLE_FX_LOOPED_ON_ENTITY_BONE", 0x9C56621462FFE7A6 },
{  "STOP_PARTICLE_FX_LOOPED", 0x22970F3A088B133B },
{  "DOES_PARTICLE_FX_LOOPED_EXIST", 0x9DD5AFF561E88F2A },
{  "SET_PARTICLE_FX_LOOPED_OFFSETS", 0xD3A4A95FC94FE83B },
{  "SET_PARTICLE_FX_LOOPED_EVOLUTION", 0x3674F389B0FACD80 },
{  "SET_PARTICLE_FX_LOOPED_COLOUR", 0x239879FC61C610CC },
{  "SET_PARTICLE_FX_LOOPED_ALPHA", 0x88786E76234F7054 },
{  "SET_PARTICLE_FX_LOOPED_SCALE", 0x1A9E1C0D98D093B7 },
{  "SET_PARTICLE_FX_LOOPED_FAR_CLIP_DIST", 0x9B04D471DA0AD7AA },
{  "REMOVE_PARTICLE_FX", 0x459598F579C98929 },
{  "REMOVE_PARTICLE_FX_FROM_ENTITY", 0x92884B4A49D81325 },
{  "REMOVE_PARTICLE_FX_IN_RANGE", 0x87B5905ECA623B68 },
{  "USE_PARTICLE_FX_ASSET", 0xA10DB07FC234DD12 },
{  "SET_PARTICLE_FX_OVERRIDE", 0xBE711A169E9C7E95 },
{  "RESET_PARTICLE_FX_OVERRIDE", 0x274B3DABF7E72DEF },
{  "REMOVE_DECALS_IN_RANGE", 0x86DE59FA02902B40 },
{  "REMOVE_DECALS_FROM_OBJECT", 0xFB8972BAE0013140 },
{  "ADD_DECAL", 0x57CB267624EF85C0 },
{  "START_PETROL_TRAIL_DECALS", 0x46F246D6504F0031 },
{  "ADD_PETROL_TRAIL_DECAL_INFO", 0x73354FB6D03D2E8A },
{  "END_PETROL_TRAIL_DECALS", 0x0E126AAE933F3B56 },
{  "REMOVE_DECAL", 0x49A720552EB0BB88 },
{  "IS_DECAL_ALIVE", 0x3E4B4E5CF5D3EEB5 },
{  "SET_TIMECYCLE_MODIFIER", 0xFA08722A5EA82DA7 },
{  "SET_TIMECYCLE_MODIFIER_STRENGTH", 0xFDB74C9CC54C3F37 },
{  "SET_TRANSITION_TIMECYCLE_MODIFIER", 0xFF927A09F481D80C },
{  "CLEAR_TIMECYCLE_MODIFIER", 0x0E3F4AF2D63491FB },
{  "GET_TIMECYCLE_MODIFIER_INDEX", 0xA705394293E2B3D3 },
{  "GET_TIMECYCLE_TRANSITION_MODIFIER_INDEX", 0x2DA67BA3C8A6755D },
{  "_SET_EXTRA_TIMECYCLE_MODIFIER_STRENGTH", 0x6FE93BCC7BF12B63 },
{  "SET_TV_CHANNEL", 0x593FAF7FC9401A56 },
{  "GET_TV_CHANNEL", 0xF90FBFD68F3C59AE },
{  "SET_TV_VOLUME", 0x73A97068787D7231 },
{  "DRAW_TV_CHANNEL", 0xC0A145540254A840 },
{  "SET_TV_CHANNEL_PLAYLIST", 0xDEC6B25F5DC8925B },
{  "_IS_TV_PLAYLIST_ITEM_PLAYING", 0x4D562223E0EB65F3 },
{  "ENABLE_MOVIE_SUBTITLES", 0x6FC9B065229C0787 },
{  "ANIMPOSTFX_PLAY", 0x4102732DF6B4005F },
{  "ANIMPOSTFX_STOP", 0xB4FD7446BAB2F394 },
{  "ANIMPOSTFX_IS_RUNNING", 0x4A123E85D7C4CA0B },
{  "ANIMPOSTFX_STOP_ALL", 0x66560A0D4C64FD21 },
// Category: _NAMESPACE26
// Category: GOOGLE_ANALYTICS
{  "_GOOGLE_ANALYTICS_PUSH_PAGE", 0xD43A616AE3AC4EF6 },
{  "_GOOGLE_ANALYTICS_POP_PAGE", 0xC6DE040378364798 },
// Category: HUD
{  "BUSYSPINNER_OFF", 0x58F441B90EA84D06 },
{  "BUSYSPINNER_IS_ON", 0x823BF7B1DF613A21 },
{  "_GET_TEXT_SUBSTRING", 0x9D7E12EC6A1EE4E5 },
{  "_GET_LABEL_TEXT", 0xCFEDCCAD3C5BA90D },
{  "CLEAR_ALL_HELP_MESSAGES", 0x916ED8321F087059 },
{  "DOES_TEXT_LABEL_EXIST", 0x73C258C68D6F55B6 },
{  "GET_LENGTH_OF_LITERAL_STRING", 0x481FBF588B0B76DB },
{  "GET_LENGTH_OF_LITERAL_STRING_IN_BYTES", 0xDC5AD6B7AB8184F5 },
{  "IS_RADAR_PREFERENCE_SWITCHED_ON", 0x81E47F0EE1F2B21E },
{  "DISPLAY_HUD", 0xD63FE3AF9FB3D53F },
{  "IS_HUD_HIDDEN", 0x71B72B478F8189DC },
{  "IS_RADAR_HIDDEN", 0x1B82FD5FFA4D666E },
{  "SET_TEXT_SCALE", 0x4170B650590B3B00 },
{  "SET_TEXT_CENTRE", 0xBE5261939FBECB8C },
{  "SET_TEXT_DROPSHADOW", 0x1BE39DBAA7263CA5 },
{  "SET_TEXT_RENDER_ID", 0xE550CDE128D56757 },
{  "REGISTER_NAMED_RENDERTARGET", 0x98AF2BB6F62BD588 },
{  "IS_NAMED_RENDERTARGET_REGISTERED", 0x3EE32F7964C40FE6 },
{  "RELEASE_NAMED_RENDERTARGET", 0x0E692EE61761361F },
{  "LINK_NAMED_RENDERTARGET", 0x2F506B8556242DDB },
{  "GET_NAMED_RENDERTARGET_RENDER_ID", 0xB6762A85EE29AA60 },
{  "IS_NAMED_RENDERTARGET_LINKED", 0x707032835FF09AE7 },
{  "HIDE_LOADING_ON_FADE_THIS_FRAME", 0xEA600AABAF4B9084 },
{  "HIDE_HUD_AND_RADAR_THIS_FRAME", 0x36CDD81627A6FCD2 },
{  "SET_MISSION_NAME", 0x402669A4BDAA72DA },
{  "_SET_MISSION_NAME_2", 0xD98630CE73C61E98 },
{  "GET_HUD_SCREEN_POSITION_FROM_WORLD_POSITION", 0xB39C81628EF10B42 },
{  "DISABLE_FRONTEND_THIS_FRAME", 0x56CE42A528156A67 },
{  "_ALLOW_PAUSE_MENU_WHEN_DEAD_THIS_FRAME", 0x30996422DF1EE561 },
{  "SET_FRONTEND_ACTIVE", 0xCE47C21C0687EBC2 },
{  "IS_PAUSE_MENU_ACTIVE", 0x535384D6067BA42E },
{  "_FORCE_CLOSE_TEXT_INPUT_BOX", 0xFBBF5943E4C2C992 },
{  "_IS_MULTIPLAYER_CHAT_ACTIVE", 0xBC19069D5FBB5C46 },
{  "_JOURNAL_WRITE_ENTRY", 0x6DFDD665E416B093 },
{  "_JOURNAL_CAN_WRITE_ENTRY", 0xCF782691D91F270E },
{  "_JOURNAL_GET_ENTRY_COUNT", 0xE65B5DE53351BE22 },
{  "_JOURNAL_GET_ENTRY_AT_INDEX", 0x3D16ABD7A1FD8C96 },
{  "_JOURNAL_GET_GRIME_AT_INDEX", 0xCB5945E1B855852F },
{  "_JOURNAL_GET_ENTRY_INFO", 0x5514C3E60673530F },
{  "_JORUNAL_GET_TEXTURE_WITH_LAYOUT", 0x62CC549B3B8EA2AA },
{  "_JORUNAL_MARK_READ", 0xE4509BABE59BD24E },
{  "_JORUNAL_CLEAR_ALL_PROGRESS", 0xF402978DE6F88D6E },
{  "_CREATE_PROMPT", 0x29FA7910726C3889 },
{  "_BEGIN_REGISTER_PROMPT", 0x04F97DE45A519419 },
{  "_END_REGISTER_PROMPT", 0xF7AA2696A22AD8B9 },
{  "_PROMPT_SET_PRIORITY", 0xCA24F528D0D16289 },
{  "_PROMPT_SET_CONTROL_ACTION", 0xB5352B7494A08258 },
{  "_PROMPT_SET_ALLOWED_ACTION", 0x565C1CE183CB0EAF },
{  "_PROMPT_SET_STANDARD_MODE", 0xCC6656799977741B },
{  "_PROMPT_HAS_STANDARD_MODE_COMPLETED", 0xC92AC953F0A982AE },
{  "_PROMPT_SET_PRESSED_TIMED_MODE", 0x1473D3AF51D54276 },
{  "_PROMPT_HAS_PRESSED_TIMED_MODE_COMPLETED", 0x3CE854D250A88DAF },
{  "_PROMPT_HAS_PRESSED_TIMED_MODE_FAILED", 0x1A17B9ECFF617562 },
{  "_PROMPT_SET_HOLD_MODE", 0x94073D5CA3F16B7B },
{  "_PROMPT_SET_STANDARDIZED_HOLD_MODE", 0x74C7D7B72ED0D3CF },
{  "_PROMPT_SET_HOLD_INDEFINITELY_MODE", 0xEA5CCF4EEB2F82D1 },
{  "_PROMPT_SET_HOLD_AUTO_FILL_MODE", 0x3CE932E737C145D6 },
{  "_PROMPT_HAS_HOLD_AUTO_FILL_MODE", 0x8010BEBD0D5ED5BC },
{  "_PROMPT_IS_HOLD_MODE_RUNNING", 0xC7D70EAEF92EFF48 },
{  "_PROMPT_HAS_HOLD_MODE_COMPLETED", 0xE0F65F0640EF0617 },
{  "_PROMPT_SET_MASH_MODE", 0xDF6423BF071C7F71 },
{  "_PROMPT_SET_MASH_INDEFINITELY_MODE", 0x7B66E89312727274 },
{  "_PROMPT_SET_MASH_WITH_RESISTANCE_MODE", 0xCD1BDFF15EFA79F5 },
{  "_PROMPT_SET_MASH_WITH_RESISTANCE_CAN_FAIL_MODE", 0xDC0CB602DEADBA53 },
{  "_PROMPT_SET_MASH_AUTO_FILL_MODE", 0x6C39587D7CC66801 },
{  "_PROMPT_SET_MASH_MANUAL_MODE", 0x32DF729D8BD3C1C6 },
{  "_PROMPT_SET_MASH_MANUAL_CAN_FAIL_MODE", 0x179DCF71F705DA20 },
{  "_PROMPT_HAS_MASH_MODE_COMPLETED", 0x845CE958416DC473 },
{  "_PROMPT_HAS_MASH_MODE_FAILED", 0x25B18E530CF39D6F },
{  "_PROMPT_GET_MASH_MODE_PROGRESS", 0x8A9585293863B8A5 },
{  "_PROMPT_SET_MASH_MANUAL_MODE_INCREASE_PER_PRESS", 0xA0D1D79C6036A855 },
{  "_PROMPT_SET_MASH_MANUAL_MODE_DECAY_SPEED", 0x7D393C247FB9B431 },
{  "_PROMPT_SET_MASH_MANUAL_MODE_PRESSED_GROWTH_SPEED", 0x56DBB26F98582C29 },
{  "_PROMPT_SET_ROTATE_MODE", 0x7ABE7095FB3D2581 },
{  "_PROMPT_SET_TARGET_MODE", 0x5F6503D9CD2754EB },
{  "_PROMPT_SET_TARGET_MODE_TARGET", 0x5E019C45DD3B6A14 },
{  "_PROMPT_SET_TARGET_MODE_PROGRESS", 0x00123054BEC8A30F },
{  "_PROMPT_SET_BEAT_MODE", 0xF957A1654C6322FE },
{  "_PROMPT_WAS_BEAT_MODE_PRESSED_IN_TIME_WINDOW", 0x1FE4788AB1430C55 },
{  "_PROMPT_SET_BEAT_MODE_GRAYED_OUT", 0xB487A4936FBF40AC },
{  "_PROMPT_RESTART_MODES", 0xDC6C55DFA2C24EE5 },
{  "_PROMPT_SET_SPINNER_SPEED", 0xAC6586A7FDCD4B68 },
{  "_PROMPT_SET_SPINNER_POSITION", 0x832CB510DE546282 },
{  "_PROMPT_SET_URGENT_PULSING_ENABLED", 0xC5F428EE08FA7F2C },
{  "_PROMPT_GET_URGENT_PULSING_ENABLED", 0x1FBA0DABECDDB52B },
{  "_PROMPT_SET_TAG", 0xDEC85C174751292B },
{  "_PROMPT_GET_GROUP_ID_FOR_TARGET_ENTITY", 0xB796970BD125FCE8 },
{  "_PROMPT_GET_GROUP_ID_FOR_SCENARIO_POINT", 0xCB73D7521E7103F0 },
{  "_PROMPT_SET_GROUP", 0x2F11D3A254169EA4 },
{  "_PROMPT_REMOVE_GROUP", 0x4E52C800A28F7BE8 },
{  "_PROMPT_SET_ACTIVE_GROUP_THIS_FRAME", 0xC65A45D4453C2627 },
{  "_PROMPT_GET_GROUP_ACTIVE_PAGE", 0xC1FCC36C3F7286C8 },
{  "_PROMPT_SET_AMBIENT_GROUP_THIS_FRAME", 0x315C81D760609108 },
{  "_PROMPT_DOES_AMBIENT_GROUP_EXIST", 0xEB550B927B34A1BB },
{  "_PROMPT_ADD_GROUP_LINK", 0x684C96CC7C66E8EF },
{  "_PROMPT_ADD_GROUP_RETURN_LINK", 0x837972ED28159536 },
{  "_PROMPT_SET_TRANSPORT_MODE", 0x876E4A35C73A6655 },
{  "_PROMPT_DELETE", 0x00EDE88D4D13CF59 },
{  "_PROMPT_IS_VALID", 0x347469FBDD1589A9 },
{  "_PROMPT_IS_ACTIVE", 0x546E342E01DE71CF },
{  "_PROMPT_SET_VISIBLE", 0x71215ACCFDE075EE },
{  "_PROMPT_SET_ENABLED", 0x8A0FB4D03A630D21 },
{  "_PROMPT_SET_TEXT", 0x5DD02A8318420DD7 },
{  "_PROMPT_SET_ATTRIBUTE", 0x560E76D5E2E1803F },
{  "_PROMPT_SET_POSITION", 0xAE84C5EE2C384FB3 },
{  "_PROMPT_IS_PRESSED", 0x21E60E230086697F },
{  "_PROMPT_IS_JUST_PRESSED", 0x2787CC611D3FACC5 },
{  "_PROMPT_IS_JUST_RELEASED", 0x635CC82FA297A827 },
{  "_PROMPT_IS_RELEASED", 0xAFC887BA7A7756D6 },
{  "_PROMPT_GLOBAL_ENABLE_PROMPT_TYPE_THIS_FRAME", 0x06565032897BA861 },
{  "_PROMPT_GLOBAL_DISABLE_PROMPT_TYPE_THIS_FRAME", 0xFC094EF26DD153FA },
{  "_PROMPT_GLOBAL_FILTER_CLEAR", 0x6A2F820452017EA2 },
{  "_PROMPT_GLOBAL_SET_FAVOURED_PED_FOR_CONFLICT_RESOLUTION", 0x530A428705BE5DEF },
{  "_PROMPT_GLOBAL_CLEAR_FAVOURED_PED_FOR_CONFLICT_RESOLUTION", 0x51259AE5C72D4A1B },
{  "_PROMPT_IS_ENABLED", 0x0D00EDDFB58B7F28 },
{  "_PROMPT_SET_ORDERING_AS_INPUT_TYPE", 0x2F385ECC5200938D },
{  "CREATE_FAKE_MP_GAMER_TAG", 0x53CB4B502E1C57EA },
{  "REMOVE_MP_GAMER_TAG", 0x839BFD7D7E49FE09 },
{  "IS_MP_GAMER_TAG_ACTIVE", 0x6E1C31E14C7A5F97 },
{  "SET_MP_GAMER_TAG_NAME", 0xEA6F4B8D4B4B5B3E },
{  "SET_MP_GAMER_TAG_BIG_TEXT", 0xA0D7CE5F83259663 },
// Category: _NAMESPACE29
// Category: _NAMESPACE30
{  "_SET_MOUSE_CURSOR_ACTIVE_THIS_FRAME", 0xF12E4CCAF249DC10 },
{  "_SET_MOUSE_CURSOR_SPRITE", 0x7F5858AAB5A58CCE },
// Category: INTERIOR
{  "IS_VALID_INTERIOR", 0x017C1B3159F79F6C },
{  "IS_INTERIOR_SCENE", 0x4200F14D6F840A9A },
{  "CLEAR_ROOM_FOR_ENTITY", 0xA1762D5BBFCA13A8 },
{  "FORCE_ROOM_FOR_ENTITY", 0xBC29A9894C976945 },
{  "GET_ROOM_KEY_FROM_ENTITY", 0x076E46E0EB52AFC6 },
{  "GET_KEY_FOR_ENTITY_IN_ROOM", 0x27D7B6F79E1F4603 },
{  "GET_INTERIOR_FROM_ENTITY", 0xB417689857646F61 },
{  "FORCE_ROOM_FOR_GAME_VIEWPORT", 0x115B4AA8FB28AB43 },
{  "CLEAR_ROOM_FOR_GAME_VIEWPORT", 0x951A049765E0D450 },
{  "GET_INTERIOR_AT_COORDS", 0xCDD36C9E5C469070 },
{  "PIN_INTERIOR_IN_MEMORY", 0xBD3D33EABF680168 },
{  "UNPIN_INTERIOR", 0x07FD1A0B814F6055 },
{  "IS_INTERIOR_READY", 0x941560D2D45DBFC8 },
{  "GET_INTERIOR_AT_COORDS_WITH_TYPE", 0xAAD6170AA33B13C0 },
{  "GET_INTERIOR_AT_COORDS_WITH_TYPEHASH", 0x3543AEA1816D1D2B },
{  "_ARE_COORDS_COLLIDING_WITH_EXTERIOR", 0xF291396B517E25B2 },
{  "GET_INTERIOR_FROM_COLLISION", 0x5054D1A5218FA696 },
{  "ACTIVATE_INTERIOR_ENTITY_SET", 0x174D0AAB11CED739 },
{  "DEACTIVATE_INTERIOR_ENTITY_SET", 0x33B81A2C07A51FFF },
{  "IS_INTERIOR_ENTITY_SET_ACTIVE", 0x32810CA2125F5842 },
{  "DISABLE_INTERIOR", 0x3C2B92A1A07D4FCE },
// Category: _NAMESPACE32
// Category: ITEMDATABASE
{  "_ITEM_DATABASE_CAN_EQUIP_ITEM_ON_CATEGORY", 0x856FF92C57742AE5 },
{  "_ITEM_DATABASE_GET_FITS_SLOT_COUNT", 0x2970D1D6BFCF9B46 },
{  "_ITEM_DATABASE_GET_FITS_SLOT_INFO", 0x77210C146CED5261 },
{  "_ITEM_DATABASE_GET_HAS_SLOT_COUNT", 0x44915068579D7710 },
{  "_ITEM_DATABASE_GET_HAS_SLOT_INFO", 0x8A9BD0DB7E8376CF },
{  "_ITEM_DATABASE_FILLOUT_ITEM_INFO", 0xFE90ABBCBFDC13B2 },
{  "_ITEM_DATABASE_FILLOUT_ACQUIRE_COST", 0x74F7928816E4E181 },
{  "_ITEM_DATABASE_FILLOUT_SELL_PRICE", 0x7A62A2EEDE1C3766 },
{  "_ITEM_DATABASE_FILLOUT_SATCHEL_DATA", 0x4776EFD78F75C23F },
{  "_ITEM_DATABASE_FILLOUT_UI_DATA", 0xB86F7CC2DC67AC60 },
{  "_ITEM_DATABASE_FILLOUT_ITEM_EFFECTS_IDS", 0x9379BE60DC55BBE6 },
{  "_ITEM_DATABASE_FILLOUT_ITEM_EFFECTS_ID_INFO", 0xCF2D360D27FD1ABF },
{  "_ITEM_DATABASE_FILLOUT_TAG_DATA", 0x5A11D6EEA17165B0 },
{  "_ITEM_DATABASE_GET_ACQUIRE_COSTS_COUNT", 0x01FDDAD392D04144 },
{  "_ITEM_DATABASE_GET_ACQUIRE_COST", 0x6772A83C67A25775 },
{  "_ITEM_DATABASE_CREATE_ITEM_COLLECTION", 0x71EFA7999AE79408 },
{  "_ITEM_DATABASE_IS_INTRINSIC_ITEM", 0x337F88E3A063995E },
{  "_ITEM_DATABASE_IS_OVERPOWERED_ITEM", 0x337F88E3A063995F },
{  "_ITEM_DATABASE_IS_KEY_VALID", 0x6D5D51B188333FD1 },
{  "_ITEM_DATABASE_GET_BUNDLE_ID", 0x891A45960B6B768A },
{  "_ITEM_DATABASE_GET_BUNDLE_ITEM_COUNT", 0x3332695B01015DF9 },
{  "_ITEM_DATABASE_GET_BUNDLE_ITEM_INFO", 0x5D48A77E4B668B57 },
{  "_ITEM_DATABASE_IS_SHOP_KEY_VALID", 0x00B9507D8E1D8716 },
{  "_ITEM_DATABASE_GET_SHOP_INVENTORIES_ITEMS_COUNT", 0xC568B1A0F17C7025 },
{  "_ITEM_DATABASE_GET_SHOP_INVENTORIES_ITEM_INFO", 0x4A79B41B4EB91F4E },
{  "_ITEM_DATABASE_GET_SHOP_INVENTORIES_ITEM_INFO_BY_KEY", 0xCFB06801F5099B25 },
{  "_ITEM_DATABASE_GET_SHOP_INVENTORIES_REQUIREMENT_GROUP_INFO", 0x76C752D788A76813 },
{  "_ITEM_DATABASE_GET_SHOP_INVENTORIES_REQUIREMENT_INFO", 0xE0EA5C031AE5539F },
{  "_ITEM_DATABASE_IS_SHOP_LAYOUT_KEY_VALID", 0x3AFE5182C45A84F6 },
{  "_ITEM_DATABASE_GET_SHOP_LAYOUT_INFO", 0x66A6D76B6BB999B4 },
{  "_ITEM_DATABASE_GET_SHOP_LAYOUT_ROOT_MENU_INFO", 0x86FCB565CCA0CFA7 },
{  "_ITEM_DATABASE_GET_SHOP_LAYOUT_MENU_INFO_BY_ID", 0xD66114469978B55B },
{  "_ITEM_DATABASE_GET_SHOP_LAYOUT_MENU_INFO_BY_INDEX", 0xF04247092F193B75 },
{  "_ITEM_DATABASE_GET_SHOP_LAYOUT_MENU_PAGE_KEY", 0x9A60570657A7B635 },
{  "_ITEM_DATABASE_GET_SHOP_LAYOUT_PAGE_INFO_BY_KEY", 0xB347C100DF0C9B7F },
{  "_ITEM_DATABASE_GET_SHOP_LAYOUT_PAGE_ITEM_KEY", 0xF32BEF578B3DBAE8 },
{  "_ITEM_DATABASE_IS_BUY_AWARD_KEY_VALID", 0x4CE753203FA42214 },
{  "_ITEM_DATABASE_FILLOUT_BUY_AWARD_ACQUIRE_COSTS", 0xB52E20F6767A09A2 },
{  "_ITEM_DATABASE_GET_ITEM_PRICE_MODIFIERS", 0x4EB37AAB79AB0C48 },
{  "_ITEM_DATABASE_FILLOUT_PRICE_MODIFIER_BY_KEY", 0x40C5D95818823C94 },
{  "_ITEM_DATABASE_GET_NUMBER_OF_MODIFIED_PRICES", 0x5AAAF40E9B224F5E },
{  "_ITEM_DATABASE_GET_MODIFIED_PRICE", 0xCB92EC9C004732B4 },
{  "_ITEM_DATABASE_GET_NUMBER_OF_MODIFIERS", 0x1289D8315235856D },
{  "_ITEM_DATABASE_FILLOUT_MODIFIER", 0x60614A0AB580A2B5 },
{  "_ITEM_DATABASE_DOES_ITEM_HAVE_TAG", 0xFF5FB5605AD56856 },
{  "_ITEM_DATABASE_GET_ITEM_TAG_TYPE", 0x6111B8F9413F413A },
{  "_ITEM_DATABASE_LOCALIZATION_GET_NUM_LABEL_TYPES", 0xCEC6A41E8910486A },
{  "_ITEM_DATABASE_LOCALIZATION_GET_NUM_VALUES", 0x49885D82A13EEAEA },
{  "_ITEM_DATABASE_LOCALIZATION_GET_TYPE", 0xCABF5D41D0073D4A },
{  "_ITEM_DATABASE_LOCALIZATION_GET_VALUE", 0x9AE5610FDCED6EA7 },
// Category: ITEMSET
{  "CREATE_ITEMSET", 0xA1AF16083320065A },
{  "DESTROY_ITEMSET", 0x712BC69F10549B92 },
{  "IS_ITEMSET_VALID", 0xD30765D153EF5C76 },
{  "ADD_TO_ITEMSET", 0xABE74510883C7950 },
{  "REMOVE_FROM_ITEMSET", 0xC5BAA432B429DC24 },
{  "GET_ITEMSET_SIZE", 0x55F2E375AC6018A9 },
{  "GET_INDEXED_ITEM_IN_ITEMSET", 0x275A2E2C0FAB7612 },
{  "IS_IN_ITEMSET", 0xD1503C2EE2FE688C },
{  "CLEAN_ITEMSET", 0x85F3A86CA9021FB0 },
// Category: _NAMESPACE35
// Category: LOCALIZATION
{  "_GET_SYSTEM_LANGUAGE", 0x3C1A05F86AE6ACB5 },
{  "GET_CURRENT_LANGUAGE", 0xDB917DA5C6835FCC },
{  "_LOCALIZATION_GET_USER_LANGUAGE", 0x76E30B799EBEEA0F },
// Category: MAP
{  "GET_BLIP_FROM_ENTITY", 0x6D2C41A8BD6D6FD0 },
{  "GET_MAIN_PLAYER_BLIP_ID", 0x5CD2889B2B381D45 },
{  "SET_BLIP_FLASH_TIMER", 0x02FF4CF43B7209D1 },
{  "SET_BLIP_FLASHES", 0x0DF2B55F717DDB10 },
{  "TRIGGER_SONAR_BLIP", 0x72DD432F3CDFC0EE },
{  "SET_BLIP_COORDS", 0x4FF674F5E23D49CE },
{  "GET_BLIP_COORDS", 0x201C319797BDA603 },
{  "SET_BLIP_SPRITE", 0x74F74D3207ED525C },
{  "SET_BLIP_NAME_FROM_TEXT_FILE", 0x0A062D6D7C0B2C2C },
{  "SET_BLIP_NAME_TO_PLAYER_NAME", 0x093DD5A31BC2B459 },
{  "SET_BLIP_ROTATION", 0x6049966A94FBE706 },
{  "SET_BLIP_SCALE", 0xD38744167B2FA257 },
{  "REMOVE_BLIP", 0xF2C3C9DA47AAA54A },
{  "DOES_BLIP_EXIST", 0xCD82FA174080B3B1 },
{  "SET_RADAR_ZOOM", 0xCAF6489DA2C8DD9E },
{  "IS_BLIP_ON_MINIMAP", 0x46534526B9CD2D17 },
{  "FORCE_SONAR_BLIPS_THIS_FRAME", 0xEE1C7BA69BB74B08 },
{  "SET_GPS_FLAGS", 0x5DE61C90DDECFA2D },
{  "CLEAR_GPS_FLAGS", 0x4D3771237C79FF41 },
{  "SET_GPS_CUSTOM_ROUTE_RENDER", 0xF6CEF599FC470B33 },
{  "CLEAR_GPS_CUSTOM_ROUTE", 0x1EAA5674B4D181C5 },
{  "START_GPS_MULTI_ROUTE", 0x3D3D15AF7BCAAF83 },
{  "ADD_POINT_TO_GPS_MULTI_ROUTE", 0x64C59DD6834FA942 },
{  "SET_GPS_MULTI_ROUTE_RENDER", 0x4426D65E029A4DC0 },
{  "CLEAR_GPS_MULTI_ROUTE", 0x9E0AB9AAEE87CE28 },
{  "CLEAR_GPS_PLAYER_WAYPOINT", 0x08FDC6F796E350D1 },
{  "SET_WAYPOINT_OFF", 0xFA8C41E8020D3439 },
{  "IS_WAYPOINT_ACTIVE", 0x202B1BBFC6AB5EE4 },
{  "DISPLAY_RADAR", 0x1B3DA717B9AFF828 },
{  "SET_RADAR_AS_EXTERIOR_THIS_FRAME", 0xA8EBBAE986FB5457 },
{  "_SET_MINIMAP_REVEALED", 0x4B8F743A4A6D2FF8 },
{  "LOCK_MINIMAP_ANGLE", 0x0BFD145EF819FB3A },
{  "UNLOCK_MINIMAP_ANGLE", 0x5373DE8E179BC2A0 },
// Category: MINIGAME
{  "_MINIGAME_IS_CONNECTED_TO_SERVER", 0x2A0C4736AC5AF0CE },
{  "_MINIGAME_LEAVE_TABLE", 0xF5446E47941E654C },
{  "_MINIGAME_GET_NEXT_EVENT_TYPE", 0x578907F59BA01B6C },
{  "_MINIGAME_GET_NEXT_EVENT", 0xDF728C5AE137FC13 },
{  "_MINIGAME_POP_NEXT_EVENT", 0x833E03BAEBADC4B0 },
{  "_MINIGAME_IS_REQUEST_PENDING", 0x9105A4A2556FA937 },
{  "_POKER_GET_GAME_SETTINGS_FOR_ID", 0x2D20E12E1990D584 },
{  "_POKER_BUY_IN", 0xB4D610EA5A1FDE74 },
{  "_POKER_CALL", 0x8DED681B161EBD78 },
{  "_POKER_CHECK", 0x49A045628D9B1B86 },
{  "_POKER_FOLD", 0x3DFAB7D9BB45B5BE },
{  "_POKER_RAISE", 0xECCF45A79A17BB96 },
{  "_POKER_REVEAL", 0x2F2131DB0A8B02DC },
{  "_DOMINOES_BUY_IN", 0x399E6CD12FC8CA89 },
{  "_DOMINOES_PLACE_DOMINO", 0xB79A29B33BF29BA5 },
{  "_DOMINOES_REQUEST_VALID_PLACEMENTS", 0xE26AEE7E67D9E21D },
// Category: MISC
{  "GET_NUMBER_OF_FREE_STACKS_OF_THIS_SIZE", 0x40DC2907A9697EF7 },
{  "SET_RANDOM_SEED", 0x5CD7A49104AFCB6B },
{  "SET_TIME_SCALE", 0x9682AF6050854856 },
{  "SET_MISSION_FLAG", 0x36694B456BE80D0A },
{  "GET_MISSION_FLAG", 0xB15CD1CF58771DE1 },
{  "SET_RANDOM_EVENT_FLAG", 0xB1ADCCC4150C6473 },
{  "GET_RANDOM_EVENT_FLAG", 0x924D54E5698AE3E0 },
{  "SET_RANDOM_WEATHER_TYPE", 0x6E5A7FBEECAB3C72 },
{  "CLEAR_WEATHER_TYPE_PERSIST", 0xD85DFE5C131E4AE9 },
{  "_GET_WEATHER_TYPE_TRANSITION", 0x0AC679B2342F14F2 },
{  "_SET_WEATHER_TYPE_TRANSITION", 0xFA3E3CA8A1DE6D5D },
{  "CLEAR_OVERRIDE_WEATHER", 0x80A398F16FFE3CC3 },
{  "SET_WIND_SPEED", 0xD00C2D82DC04A99F },
{  "GET_WIND_SPEED", 0xFFB7E74E041150A4 },
{  "SET_WIND_DIRECTION", 0xB56C4F5F57A45600 },
{  "GET_WIND_DIRECTION", 0xF703E82F3FE14A5F },
{  "_SET_RAIN_FX_INTENSITY", 0x193DFC0526830FD6 },
{  "GET_RAIN_LEVEL", 0x931B5F4CC130224B },
{  "GET_SNOW_LEVEL", 0x1E5D727041BE1709 },
{  "FORCE_LIGHTNING_FLASH", 0x369DB5B2510FA080 },
{  "GET_GAME_TIMER", 0x4F67E8ECA7D3F667 },
{  "_GET_SYSTEM_TIME", 0xBE7F225417E35A7C },
{  "_GET_NUMBER_OF_MICROSECONDS_SINCE_LAST_CALL", 0xB0CE5E5ED8BB3581 },
{  "_GET_SCRIPT_TIME_WITHIN_FRAME_IN_MICROSECONDS", 0x63219768C586667C },
{  "_RESET_SCRIPT_TIME_WITHIN_FRAME", 0x1411A7CBC3A6EB7B },
{  "GET_FRAME_TIME", 0x5E72022914CE3C38 },
{  "_GET_BENCHMARK_TIME", 0x3F3172FEAE3AFE1C },
{  "GET_FRAME_COUNT", 0x77DFA958FCF100C1 },
{  "GET_RANDOM_FLOAT_IN_RANGE", 0xE29F927A961F8AAA },
{  "GET_RANDOM_INT_IN_RANGE", 0xD53343AA4FB7DD28 },
{  "GET_GROUND_Z_FOR_3D_COORD", 0x24FA4267BB8D2431 },
{  "GET_GROUND_Z_AND_NORMAL_FOR_3D_COORD", 0x2A29CA9A6319E6AB },
{  "ASIN", 0x6E3C15D296C15583 },
{  "ACOS", 0x586690F0176DC575 },
{  "TAN", 0x8C13DB96497B7ABF },
{  "ATAN", 0x503054DED0B78027 },
{  "ATAN2", 0x965B220A066E3F07 },
{  "GET_DISTANCE_BETWEEN_COORDS", 0x0BE7F4E3CDBAFB28 },
{  "GET_ANGLE_BETWEEN_2D_VECTORS", 0xD0DFE1C486097BBB },
{  "GET_HEADING_FROM_VECTOR_2D", 0x38D5202FF9271C62 },
{  "SET_BIT", 0xF73FBE4845C43B5B },
{  "CLEAR_BIT", 0x7D1D4A3602B6AD4E },
{  "GET_HASH_KEY", 0xFD340785ADF8CFB7 },
{  "IS_POSITION_OCCUPIED", 0x825CA3ED43831015 },
{  "CLEAR_AREA", 0x3B882A96EA77D5B1 },
{  "CLEAR_ANGLED_AREA_OF_VEHICLES", 0xA4D83115C1E02F8A },
{  "SET_CREDITS_ACTIVE", 0xD37BECF862DA726F },
{  "NETWORK_SET_SCRIPT_IS_SAFE_FOR_NETWORK_GAME", 0x3D0EAC6385DD6100 },
{  "PAUSE_DEATH_ARREST_RESTART", 0x66AB6B6C7E72F393 },
{  "IGNORE_NEXT_RESTART", 0x6C9FF40FF1B69F8F },
{  "SET_FADE_IN_AFTER_DEATH_ARREST", 0xDF3B5846DE5904AF },
{  "SET_FADE_IN_AFTER_LOAD", 0xAC806C4CAB973517 },
{  "OVERRIDE_SAVE_HOUSE", 0xB2C69E11A37B5AF0 },
{  "SHOOT_SINGLE_BULLET_BETWEEN_COORDS", 0x867654CBC7606F2C },
{  "GET_MODEL_DIMENSIONS", 0xDCB8DDD5D054A7E7 },
{  "IS_BIT_SET", 0x4ED6CFDFE8D4131A },
{  "IS_MINIGAME_IN_PROGRESS", 0xF4D8BCD052E7EA1B },
{  "SHOULD_USE_METRIC_MEASUREMENTS", 0x4FB556ACEFA93098 },
{  "COMPARE_STRINGS", 0xBFBB74A15EFC149B },
{  "ABSI", 0x0C214D5B8A38C828 },
{  "ABSF", 0x134549B388167CBF },
{  "IS_PROJECTILE_IN_AREA", 0x05B0061EFDFC8941 },
{  "IS_PROJECTILE_TYPE_IN_AREA", 0x04965FB9E14235C7 },
{  "IS_PROJECTILE_TYPE_IN_ANGLED_AREA", 0x928431F4133CD3D4 },
{  "_IS_PROJECTILE_TYPE_IN_RADIUS", 0xF51C9BAAD9ED64C4 },
{  "_GET_PROJECTILE_NEAR_PED_COORDS", 0xD73C960A681052DF },
{  "_GET_PROJECTILE_NEAR_PED", 0x9578986A6105A6AD },
{  "IS_BULLET_IN_ANGLED_AREA", 0x9D09D8493747CF02 },
{  "IS_BULLET_IN_AREA", 0xC652FD308772D79E },
{  "IS_BULLET_IN_BOX", 0xC128137C52152741 },
{  "HAS_BULLET_IMPACTED_IN_AREA", 0xC153E5BCCF411814 },
{  "HAS_BULLET_IMPACTED_IN_BOX", 0x3B6A4C05FB2B33AC },
{  "IS_ORBIS_VERSION", 0x88CFAE250D3E0C71 },
{  "IS_DURANGO_VERSION", 0xD1CCC2A2639D325F },
{  "IS_PC_VERSION", 0xB0FB6CFAA5A1C833 },
{  "IS_STRING_NULL", 0x602102324604D96B },
{  "IS_STRING_NULL_OR_EMPTY", 0x2CF12F9ACF18F048 },
{  "ARE_STRINGS_EQUAL", 0xD3852F22AB713A1F },
{  "_DOES_STRING_EXIST_IN_STRING", 0x9382D5D43D2AA6FF },
{  "STRING_TO_INT", 0xF2DD2298B3AF23E2 },
{  "_STRING_FORMAT_INT", 0xCF11C0CEB40C401B },
{  "_CREATE_VAR_STRING", 0xFA925AC00EB830B9 },
{  "SET_BITS_IN_RANGE", 0x324DC1CEF57F31E6 },
{  "GET_BITS_IN_RANGE", 0x68E1352AF48F905D },
{  "SET_GAME_PAUSED", 0xFAEC088D28B1DE4A },
{  "SET_THIS_SCRIPT_CAN_BE_PAUSED", 0x3215376E79F6EA18 },
{  "SET_THIS_SCRIPT_CAN_REMOVE_BLIPS_CREATED_BY_ANY_SCRIPT", 0x8ABD939C2E5D00ED },
{  "POPULATE_NOW", 0xEA6DC3A8ADD2005F },
{  "_COPY_MEMORY", 0xF7AC7DC0DEE7C9BE },
{  "ENABLE_DISPATCH_SERVICE", 0x50E52637EF70EF77 },
{  "BLOCK_DISPATCH_SERVICE_RESOURCE_CREATION", 0x66947E61A44DE2C6 },
{  "CREATE_INCIDENT", 0x3F892CAF67444AE7 },
{  "DELETE_INCIDENT", 0x5CFD0F0D6AAE0AEE },
{  "IS_INCIDENT_VALID", 0x39F2B1BAD412246A },
{  "ADD_POP_MULTIPLIER_AREA", 0x5EBDA1A3B8CB5EF7 },
{  "DOES_POP_MULTIPLIER_AREA_EXIST", 0x03BA619C81A646B3 },
{  "REMOVE_POP_MULTIPLIER_AREA", 0x88CB484364EFB37A },
{  "RESET_DISPATCH_IDEAL_SPAWN_DISTANCE", 0xC7817264BC4B6377 },
{  "SET_DISPATCH_IDEAL_SPAWN_DISTANCE", 0xEAB6823B82FBD283 },
{  "REMOVE_DISPATCH_SPAWN_BLOCKING_AREA", 0x49F751F6868DDC5B },
{  "DISPLAY_ONSCREEN_KEYBOARD", 0x044131118D8DB3CD },
{  "UPDATE_ONSCREEN_KEYBOARD", 0x37DF360F235A3893 },
{  "GET_ONSCREEN_KEYBOARD_RESULT", 0xAFB4CF58A4A292B1 },
{  "_CANCEL_ONSCREEN_KEYBOARD", 0x58A39BE597CE99CD },
{  "_REMOVE_STEALTH_KILL", 0x7ACF124C12A2B045 },
{  "SET_SUPER_JUMP_THIS_FRAME", 0xB3E9BE963F10C445 },
{  "_START_BENCHMARK_RECORDING", 0x29D1F6DF864A094E },
{  "_STOP_BENCHMARK_RECORDING", 0xB89AEC71AFF2B599 },
{  "_RESET_BENCHMARK_RECORDING", 0xECBABD0307FB216F },
{  "_SAVE_BENCHMARK_RECORDING", 0xF4743E2ECC02B3DA },
{  "_UI_IS_SINGLEPLAYER_PAUSE_MENU_ACTIVE", 0x4FFA0386A6216113 },
{  "_GET_BENCHMARK_ITERATIONS_FROM_COMMAND_LINE", 0x22FC52CF470CC98D },
{  "_GET_BENCHMARK_PASS_FROM_COMMAND_LINE", 0x9297DACF3A2CDFF7 },
// Category: _NAMESPACE40
// Category: _NAMESPACE41
// Category: NETSHOPPING
{  "_CASH_INVENTORY_TRANSACTION_READY_FOR_TRANSITIONING", 0xFCC24220FDDAC929 },
{  "_CASH_INVENTORY_TRANSACTION_FIRE_AND_FORGET_ITEM", 0xFFEA09CCEC4AF32F },
{  "_CASH_INVENTORY_TRANSACTION_START", 0xF039EC27F4490E96 },
{  "_CASH_INVENTORY_TRANSACTION_FIND", 0x52A226ADF4A270D2 },
{  "_CASH_INVENTORY_TRANSACTION_DELETE", 0x59EF5D516E2D96B9 },
{  "_CASH_INVENTORY_TRANSACTION_ADD_ITEM_WITH_INGREDIENTS", 0x3745C002F5A21C45 },
{  "_CASH_INVENTORY_TRANSACTION_GET_NUMBER_ITEMS", 0xCF2D04D076847478 },
{  "_CASH_INVENTORY_TRANSACTION_CHECKOUT", 0x592BC00BF6629BE7 },
{  "_CASH_INVENTORY_TRANSACTION_CHECKOUT_STATUS", 0x26C008791D066F37 },
// Category: NETWORK
{  "NETWORK_IS_SIGNED_ONLINE", 0x1077788E268557C2 },
{  "_NETWORK_ARE_ROS_AVAILABLE", 0x85443FF4C328F53B },
{  "NETWORK_IS_CLOUD_AVAILABLE", 0x9A4CF4F48AD77302 },
{  "NETWORK_IS_HOST", 0x8DB296B814EDDA07 },
{  "NETWORK_HAVE_ONLINE_PRIVILEGES", 0x25CB5A9F37BFD063 },
{  "NETWORK_CAN_ACCESS_MULTIPLAYER", 0xAF50DA1A3F8B1BA4 },
{  "NETWORK_SESSION_IS_PRIVATE", 0xCEF70AA5B3F89BA1 },
{  "_NETWORK_GET_SERVER_TIME", 0x6D03BFBD643B2A02 },
{  "NETWORK_CLEAR_CLOCK_TIME_OVERRIDE", 0xD972DF67326F966E },
{  "NETWORK_IS_CLOCK_TIME_OVERRIDDEN", 0xD7C95D322FF57522 },
{  "NETWORK_IS_FINDING_GAMERS", 0xDDDF64C91BFCF0AA },
{  "NETWORK_CLEAR_FOUND_GAMERS", 0x6D14CCEE1B40381A },
{  "NETWORK_CLEAR_GET_GAMER_STATUS", 0x86E0660E4F5C956D },
{  "NETWORK_IS_GAME_IN_PROGRESS", 0x10FAB35428CCC9D7 },
{  "NETWORK_IS_SESSION_ACTIVE", 0xD83C2B94E7508980 },
{  "NETWORK_IS_IN_SESSION", 0xCA97246103B63917 },
{  "NETWORK_IS_SESSION_STARTED", 0x9DE624D2FC4B603F },
{  "NETWORK_CAN_SESSION_END", 0x4EEBC3694E49C572 },
{  "NETWORK_IS_IN_PLATFORM_PARTY", 0x2FC5650B0271CB57 },
{  "NETWORK_GET_RANDOM_INT_RANGED", 0xE30CF56F1EFA5F43 },
{  "NETWORK_SET_THIS_SCRIPT_IS_NETWORK_SCRIPT", 0x1CA59E306ECB80A5 },
{  "NETWORK_GET_THIS_SCRIPT_IS_NETWORK_SCRIPT", 0x2910669969E9535E },
{  "_NETWORK_GET_NUM_PARTICIPANTS_HOST", 0xA6C90FBC38E395EE },
{  "NETWORK_GET_NUM_PARTICIPANTS", 0x18D0456E86604654 },
{  "NETWORK_GET_SCRIPT_STATUS", 0x57D158647A6BFABF },
{  "NETWORK_REGISTER_HOST_BROADCAST_VARIABLES", 0x3E9B2F01C50DF595 },
{  "NETWORK_REGISTER_PLAYER_BROADCAST_VARIABLES", 0x3364AA97340CA215 },
{  "NETWORK_GET_PLAYER_INDEX", 0x24FB80D107371267 },
{  "NETWORK_GET_PARTICIPANT_INDEX", 0x1B84DF6AF2A46938 },
{  "NETWORK_GET_PLAYER_INDEX_FROM_PED", 0x6C0E2E0125610278 },
{  "NETWORK_GET_NUM_CONNECTED_PLAYERS", 0xA4A79DD2D9600654 },
{  "NETWORK_IS_PLAYER_CONNECTED", 0x93DC1BE4E1ABE9D1 },
{  "NETWORK_GET_TOTAL_NUM_PLAYERS", 0xCF61D4B4702EE9EB },
{  "NETWORK_IS_PARTICIPANT_ACTIVE", 0x6FF8FF40B6357D45 },
{  "NETWORK_IS_PLAYER_ACTIVE", 0xB8DFD30D6973E135 },
{  "NETWORK_IS_PLAYER_A_PARTICIPANT", 0x3CA58F6CB7CBD784 },
{  "NETWORK_IS_HOST_OF_THIS_SCRIPT", 0x83CD99A1E6061AB5 },
{  "NETWORK_SET_MISSION_FINISHED", 0x3B3D11CD9FFCDFC9 },
{  "NETWORK_IS_SCRIPT_ACTIVE", 0x9D40DF90FAD26098 },
{  "NETWORK_GET_NUM_SCRIPT_PARTICIPANTS", 0x3658E8CD94FC121A },
{  "PARTICIPANT_ID", 0x90986E8876CE0A83 },
{  "PARTICIPANT_ID_TO_INT", 0x57A3BDDAD8E5AA0A },
{  "NETWORK_GET_DESTROYER_OF_NETWORK_ID", 0x7A1ADEEF01740A24 },
{  "_NETWORK_GET_DESTROYER_OF_ENTITY", 0x4CACA84440FA26F6 },
{  "NETWORK_GET_ENTITY_KILLER_OF_PLAYER", 0x42B2DAA6B596F5F8 },
{  "NETWORK_RESURRECT_LOCAL_PLAYER", 0xEA23C49EAA83ACFB },
{  "NETWORK_SET_LOCAL_PLAYER_INVINCIBLE_TIME", 0x2D95C7E2D7E07307 },
{  "NETWORK_SET_LOCAL_PLAYER_SYNC_LOOK_AT", 0x524FF0AEFF9C3973 },
{  "NETWORK_GET_NETWORK_ID_FROM_ENTITY", 0xA11700682F3AD45C },
{  "NETWORK_GET_ENTITY_FROM_NETWORK_ID", 0xCE4E5D9B0A4FF560 },
{  "NETWORK_GET_ENTITY_IS_NETWORKED", 0xC7827959479DCC78 },
{  "NETWORK_REGISTER_ENTITY_AS_NETWORKED", 0x06FAACD625D80CAA },
{  "NETWORK_DOES_NETWORK_ID_EXIST", 0x38CE16C96BD11344 },
{  "NETWORK_REQUEST_CONTROL_OF_NETWORK_ID", 0xA670B3662FAFFBD0 },
{  "NETWORK_HAS_CONTROL_OF_NETWORK_ID", 0x4D36070FE0215186 },
{  "NETWORK_REQUEST_CONTROL_OF_ENTITY", 0xB69317BF5E782347 },
{  "NETWORK_HAS_CONTROL_OF_ENTITY", 0x01BF60A500E28887 },
{  "NETWORK_HAS_CONTROL_OF_PICKUP", 0x5BC9495F0B3B6FA6 },
{  "VEH_TO_NET", 0xB4C94523F023419C },
{  "PED_TO_NET", 0x0EDEC3C276198689 },
{  "OBJ_TO_NET", 0x99BFDC94A603E541 },
{  "NET_TO_VEH", 0x367B936610BA360C },
{  "NET_TO_PED", 0xBDCD95FC216A8B3E },
{  "NET_TO_OBJ", 0xD8515F5FEA14CB3F },
{  "NET_TO_ENT", 0xBFFEAB45A9A9094A },
{  "NETWORK_GET_LOCAL_HANDLE", 0xE86051786B66CD8E },
{  "NETWORK_HANDLE_FROM_PLAYER", 0x388EB2B86C73B6B3 },
{  "NETWORK_HASH_FROM_PLAYER_HANDLE", 0xBC1D768F2F5D6C05 },
{  "NETWORK_HANDLE_FROM_FRIEND", 0xD45CB817D7E177D2 },
{  "NETWORK_GET_GAMERTAG_FROM_HANDLE", 0x426141162EBE5CDB },
{  "NETWORK_ARE_HANDLES_THE_SAME", 0x57DBA049E110F217 },
{  "NETWORK_IS_HANDLE_VALID", 0x6F79B93B0A8E4133 },
{  "NETWORK_GET_PLAYER_FROM_GAMER_HANDLE", 0xCE5F689CF5A0A49D },
{  "NETWORK_IS_GAMER_IN_MY_SESSION", 0x0F10B05DDF8D16E9 },
{  "NETWORK_SHOW_PROFILE_UI", 0x859ED1CEA343FCA8 },
{  "NETWORK_IS_FRIEND", 0x1A24A179F9B31654 },
{  "NETWORK_IS_PENDING_FRIEND", 0x0BE73DA6984A6E33 },
{  "NETWORK_ADD_FRIEND", 0x8E02D73914064223 },
{  "NETWORK_SET_FRIENDLY_FIRE_OPTION", 0xF808475FA571D823 },
{  "NETWORK_SET_RICH_PRESENCE", 0x1DCCACDCFC569362 },
{  "NETWORK_GET_TIMEOUT_TIME", 0x5ED0356A0CE3A34F },
{  "SET_NETWORK_ID_EXISTS_ON_ALL_MACHINES", 0xE05E81A888FA63C8 },
{  "_SET_NETWORK_ID_SYNC_TO_PLAYER", 0xA8A024587329F36A },
{  "_NETWORK_SET_ENTITY_INVISIBLE_TO_NETWORK", 0xF1CA12B18AEF5298 },
{  "SET_NETWORK_ID_VISIBLE_IN_CUTSCENE", 0xA6928482543022B4 },
{  "_NETWORK_CAN_NETWORK_ID_BE_SEEN", 0xA1607996431332DF },
{  "SET_LOCAL_PLAYER_VISIBLE_IN_CUTSCENE", 0xD1065D68947E7B6E },
{  "SET_LOCAL_PLAYER_INVISIBLE_LOCALLY", 0xE5F773C1A1D9D168 },
{  "SET_PLAYER_INVISIBLE_LOCALLY", 0x12B37D54667DB0B8 },
{  "SET_PLAYER_VISIBLE_LOCALLY", 0xFAA10F1FAFB11AF2 },
{  "SET_ENTITY_VISIBLE_IN_CUTSCENE", 0xE0031D3C8F36AB82 },
{  "IS_DAMAGE_TRACKER_ACTIVE_ON_NETWORK_ID", 0x6E192E33AD436366 },
{  "ACTIVATE_DAMAGE_TRACKER_ON_NETWORK_ID", 0xD45B1FFCCD52FF19 },
{  "IS_SPHERE_VISIBLE_TO_ANOTHER_MACHINE", 0xD82CF8E64C8729D8 },
{  "IS_SPHERE_VISIBLE_TO_PLAYER", 0xDC3A310219E5DA62 },
{  "RESERVE_NETWORK_MISSION_OBJECTS", 0x4E5C93BD0C32FBF8 },
{  "RESERVE_NETWORK_MISSION_PEDS", 0xB60FEBA45333D36F },
{  "RESERVE_NETWORK_MISSION_VEHICLES", 0x76B02E21ED27A469 },
{  "CAN_REGISTER_MISSION_OBJECTS", 0x800DD4721A8B008B },
{  "CAN_REGISTER_MISSION_PEDS", 0xBCBF4FEF9FA5D781 },
{  "CAN_REGISTER_MISSION_VEHICLES", 0x7277F1F2E085EE74 },
{  "CAN_REGISTER_MISSION_ENTITIES", 0x69778E7564BADE6D },
{  "GET_NUM_RESERVED_MISSION_OBJECTS", 0xAA81B5F10BC43AC2 },
{  "GET_NUM_RESERVED_MISSION_PEDS", 0x1F13D5AE5CB17E17 },
{  "GET_NUM_RESERVED_MISSION_VEHICLES", 0xCF3A965906452031 },
{  "GET_NETWORK_TIME", 0x7A5487FE9FAA6B48 },
{  "GET_NETWORK_TIME_ACCURATE", 0x89023FBBF9200E9F },
{  "HAS_NETWORK_TIME_STARTED", 0x46718ACEEDEAFC84 },
{  "GET_TIME_OFFSET", 0x017008CCDAD48503 },
{  "IS_TIME_LESS_THAN", 0xCB2CF5148012C8D0 },
{  "IS_TIME_MORE_THAN", 0xDE350F8651E4346C },
{  "GET_TIME_DIFFERENCE", 0xA2C6FC031D46FFF0 },
{  "GET_CLOUD_TIME_AS_INT", 0x9A73240B49945C76 },
{  "_GET_DATE_AND_TIME_FROM_UNIX_EPOCH", 0xAC97AF97FA68E5D5 },
{  "NETWORK_SET_IN_SPECTATOR_MODE", 0x423DE3854BB50894 },
{  "NETWORK_IS_IN_SPECTATOR_MODE", 0x048746E388762E11 },
{  "NETWORK_SET_IN_MP_CUTSCENE", 0x9CA5DE655269FEC4 },
{  "NETWORK_IS_IN_MP_CUTSCENE", 0x6CC27C9FA2040220 },
{  "NETWORK_IS_PLAYER_IN_MP_CUTSCENE", 0x63F9EE203C3619F2 },
{  "USE_PLAYER_COLOUR_INSTEAD_OF_TEAM_COLOUR", 0x5FFE9B4144F9712F },
{  "NETWORK_END_TUTORIAL_SESSION", 0xD0AFAFF5A51D72F7 },
{  "NETWORK_IS_IN_TUTORIAL_SESSION", 0xADA24309FE08DACF },
{  "_NETWORK_IS_PLAYER_EQUAL_TO_INDEX", 0x9DE986FC9A87C474 },
{  "NETWORK_CONCEAL_PLAYER", 0xBBDF066252829606 },
{  "NETWORK_IS_PLAYER_CONCEALED", 0x919B3C98ED8292F9 },
{  "NETWORK_IS_TUNABLE_CLOUD_REQUEST_PENDING", 0x0467C11ED88B7D28 },
{  "NETWORK_DOES_TUNABLE_EXIST", 0x85E5F8B9B898B20A },
{  "NETWORK_ACCESS_TUNABLE_INT", 0x8BE1146DFD5D4468 },
{  "NETWORK_ACCESS_TUNABLE_BOOL", 0xAA6A47A573ABB75A },
{  "UGC_REQUEST_CONTENT_DATA_FROM_PARAMS", 0x7FD2990AF016795E },
{  "UGC_REQUEST_CACHED_DESCRIPTION", 0x5E0165278F6339EE },
{  "TEXTURE_DOWNLOAD_REQUEST", 0x16160DA74A8E74A2 },
{  "UGC_TEXTURE_DOWNLOAD_REQUEST", 0x308F96458B7087CC },
{  "TEXTURE_DOWNLOAD_RELEASE", 0x487EB90B98E9FB19 },
{  "TEXTURE_DOWNLOAD_GET_NAME", 0x3448505B6E35262D },
{  "_GET_STATUS_OF_TEXTURE_DOWNLOAD", 0x8BD6C6DEA20E82C6 },
{  "_NETWORK_HAS_PLAYER_BEEN_BANNED", 0x8020A73847E0CA7D },
{  "NETWORK_HAS_ROS_PRIVILEGE", 0xA699957E60D80214 },
{  "_NETWORK_GET_NUM_UNACKED_FOR_PLAYER", 0xFF8FCF9FFC458A1C },
{  "_NETWORK_GET_OLDEST_RESEND_COUNT_FOR_PLAYER", 0x52C1EADAF7B10302 },
// Category: OBJECT
{  "CREATE_OBJECT", 0x509D5878EB39E842 },
{  "CREATE_OBJECT_NO_OFFSET", 0x9A294B2138ABB884 },
{  "DELETE_OBJECT", 0x931914268722C263 },
{  "PLACE_OBJECT_ON_GROUND_PROPERLY", 0x58A850EAEE20FAA3 },
{  "SLIDE_OBJECT", 0x2FDFF4107B8C1147 },
{  "SET_OBJECT_TARGETTABLE", 0x8A7391690F5AFD81 },
{  "GET_CLOSEST_OBJECT_OF_TYPE", 0xE143FA2249364369 },
{  "HAS_OBJECT_BEEN_BROKEN", 0x8ABFB70C49CC43E2 },
{  "HAS_CLOSEST_OBJECT_OF_TYPE_BEEN_BROKEN", 0x761B0E69AC4D007E },
{  "_GET_OBJECT_OFFSET_FROM_COORDS", 0x163E252DE035A133 },
{  "REMOVE_DOOR_FROM_SYSTEM", 0x464D8E1427156FE4 },
{  "DOOR_SYSTEM_SET_DOOR_STATE", 0x6BAB9442830C7F53 },
{  "DOOR_SYSTEM_GET_DOOR_STATE", 0x160AA1B32F6139B8 },
{  "DOOR_SYSTEM_SET_AUTOMATIC_RATE", 0x03C27E13B42A0E82 },
{  "DOOR_SYSTEM_SET_AUTOMATIC_DISTANCE", 0x9BA001CB45CBF627 },
{  "DOOR_SYSTEM_SET_OPEN_RATIO", 0xB6E6FBA95C7324AC },
{  "DOOR_SYSTEM_GET_OPEN_RATIO", 0x65499865FCA6E5EC },
{  "IS_DOOR_REGISTERED_WITH_SYSTEM", 0xC153C43EA202C8C1 },
{  "IS_DOOR_CLOSED", 0xC531EE8A1145A149 },
{  "DOES_OBJECT_OF_TYPE_EXIST_AT_COORDS", 0xBFA48E2FF417213F },
{  "IS_POINT_IN_ANGLED_AREA", 0x2A70BAE8883E4C81 },
{  "SET_OBJECT_PHYSICS_PARAMS", 0xF6DF6E90DE7DF90F },
{  "GET_OBJECT_FRAGMENT_DAMAGE_HEALTH", 0xB6FBFD079B8D0596 },
{  "SET_ACTIVATE_OBJECT_PHYSICS_AS_SOON_AS_IT_IS_UNFROZEN", 0x406137F8EF90EAF5 },
{  "TRACK_OBJECT_VISIBILITY", 0xB252BC036B525623 },
{  "IS_OBJECT_VISIBLE", 0x8B32ACE6326A7546 },
{  "GET_RAYFIRE_MAP_OBJECT", 0xB48FCED898292E52 },
{  "SET_STATE_OF_RAYFIRE_MAP_OBJECT", 0x5C29F698D404C5E1 },
{  "GET_STATE_OF_RAYFIRE_MAP_OBJECT", 0x899BA936634A322E },
{  "DOES_RAYFIRE_MAP_OBJECT_EXIST", 0x52AF537A0C5B8AAD },
{  "GET_RAYFIRE_MAP_OBJECT_ANIM_PHASE", 0x260EE4FDBDF4DB01 },
{  "CREATE_PICKUP", 0xFBA08C503DD5FA58 },
{  "CREATE_PICKUP_ROTATE", 0x891804727E0A98B7 },
{  "CREATE_AMBIENT_PICKUP", 0x673966A0C0FD7171 },
{  "CREATE_PORTABLE_PICKUP", 0x2EAF1FDB2FB55698 },
{  "ATTACH_PORTABLE_PICKUP_TO_PED", 0x8DC39368BDD57755 },
{  "DETACH_PORTABLE_PICKUP_FROM_PED", 0xCF463D1E9A0AECB1 },
{  "GET_SAFE_PICKUP_COORDS", 0x6E16BC2503FF1FF0 },
{  "GET_PICKUP_COORDS", 0x225B8B35C88029B3 },
{  "REMOVE_ALL_PICKUPS_OF_TYPE", 0x27F9D613092159CF },
{  "HAS_PICKUP_BEEN_COLLECTED", 0x80EC48E6679313F9 },
{  "REMOVE_PICKUP", 0x3288D8ACAECD2AB2 },
{  "DOES_PICKUP_EXIST", 0xAFC1CA75AD4074D1 },
{  "DOES_PICKUP_OBJECT_EXIST", 0xD9EFB6DBF7DAAEA3 },
{  "GET_PICKUP_OBJECT", 0x5099BC55630B25AE },
{  "DOES_PICKUP_OF_TYPE_EXIST_IN_AREA", 0xF9C36251F6E48E33 },
{  "SET_PICKUP_REGENERATION_TIME", 0x78015C9B4B3ECC9D },
{  "_SET_LOCAL_PLAYER_CAN_USE_PICKUPS_WITH_THIS_MODEL", 0x88EAEC617CD26926 },
{  "SET_TEAM_PICKUP_OBJECT", 0x53E0DF1A2A3CF0CA },
{  "SET_PICKUP_GENERATION_RANGE_MULTIPLIER", 0x318516E02DE3ECE2 },
{  "GET_WEAPON_TYPE_FROM_PICKUP_TYPE", 0x08F96CA6C551AD51 },
{  "_SET_OBJECT_TEXTURE_VARIATION", 0x971DA0055324D033 },
{  "_GET_PICKUP_HASH", 0x5EAAD83F8CFB4575 },
{  "SET_FORCE_OBJECT_THIS_FRAME", 0xF538081986E49E9D },
{  "_MARK_OBJECT_FOR_DELETION", 0xADBE4809F19F927A },
// Category: PAD
{  "IS_CONTROL_ENABLED", 0x1CEA6BFDF248E5D9 },
{  "IS_CONTROL_PRESSED", 0xF3A21BCD95725A4A },
{  "IS_CONTROL_RELEASED", 0x648EE3E7F38877DD },
{  "IS_CONTROL_JUST_PRESSED", 0x580417101DDB492F },
{  "IS_CONTROL_JUST_RELEASED", 0x50F940259D3841E6 },
{  "GET_CONTROL_VALUE", 0xD95E79E8686D2C27 },
{  "GET_CONTROL_NORMAL", 0xEC3C9B8D5327B563 },
{  "GET_CONTROL_UNBOUND_NORMAL", 0x5B84D09CEC5209C5 },
{  "_SET_CONTROL_NORMAL", 0xE8A25867FBA3B05E },
{  "IS_DISABLED_CONTROL_PRESSED", 0xE2587F8CBBD87B1D },
{  "IS_DISABLED_CONTROL_JUST_PRESSED", 0x91AEF906BCA88877 },
{  "IS_DISABLED_CONTROL_JUST_RELEASED", 0x305C8DCD79DA8B0F },
{  "GET_DISABLED_CONTROL_NORMAL", 0x11E65974A982637C },
{  "GET_DISABLED_CONTROL_UNBOUND_NORMAL", 0x4F8A26A890FD62FB },
{  "_IS_INPUT_DISABLED", 0xA571D46727E2B718 },
{  "SET_PAD_SHAKE", 0x48B3886C1358D0D5 },
{  "STOP_PAD_SHAKE", 0x38C16A305E8CDC8D },
{  "IS_LOOK_INVERTED", 0x77B612531280010D },
{  "SET_INPUT_EXCLUSIVE", 0xEDE476E5EE29EDB1 },
{  "DISABLE_CONTROL_ACTION", 0xFE99B66D079CF6BC },
{  "ENABLE_CONTROL_ACTION", 0x351220255D64C155 },
{  "DISABLE_ALL_CONTROL_ACTIONS", 0x5F4B6931816E599B },
// Category: PATHFIND
{  "SET_ROADS_IN_AREA", 0xBF1A602B5BA52FEE },
{  "SET_ROADS_IN_ANGLED_AREA", 0x1A5AA1208AF5DB59 },
{  "SET_ROADS_BACK_TO_ORIGINAL", 0x1EE7063B80FFC77C },
{  "SET_ROADS_BACK_TO_ORIGINAL_IN_ANGLED_AREA", 0x0027501B9F3B407E },
{  "GET_CLOSEST_VEHICLE_NODE", 0x240A18690AE96513 },
{  "GET_CLOSEST_VEHICLE_NODE_WITH_HEADING", 0x23CFFD4CCB243354 },
{  "GET_NTH_CLOSEST_VEHICLE_NODE", 0x5A6D8DF6FBC5D0C4 },
{  "GET_NTH_CLOSEST_VEHICLE_NODE_ID", 0x116443008E5CEFC3 },
{  "GET_NTH_CLOSEST_VEHICLE_NODE_WITH_HEADING", 0x591B40D4390DB54A },
{  "GET_NTH_CLOSEST_VEHICLE_NODE_ID_WITH_HEADING", 0x4114EAA8A7F7766D },
{  "GET_NTH_CLOSEST_VEHICLE_NODE_FAVOUR_DIRECTION", 0x2FAC235A6062F14A },
{  "IS_VEHICLE_NODE_ID_VALID", 0x5829A02AF4F0B3CB },
{  "GET_VEHICLE_NODE_POSITION", 0x8E8D72FF24DEE1FB },
{  "GET_VEHICLE_NODE_IS_SWITCHED_OFF", 0x28533DBDDF7C2C97 },
{  "GET_CLOSEST_ROAD", 0x132F52BBA570FE92 },
{  "_ARE_PATH_NODES_LOADED_IN_AREA", 0xF7B79A50B905A30D },
{  "GET_RANDOM_VEHICLE_NODE", 0x93E0DB8440B73A7D },
{  "IS_POINT_ON_ROAD", 0x125BF4ABFC536B09 },
{  "SET_PED_PATHS_IN_AREA", 0x34F060F4BF92E018 },
{  "GET_SAFE_COORD_FOR_PED", 0xB61C8E878A4199CA },
{  "SET_PED_PATHS_BACK_TO_ORIGINAL", 0xE04B48F2CC926253 },
{  "ADD_NAVMESH_REQUIRED_REGION", 0x387EAD7EE42F6685 },
{  "IS_NAVMESH_LOADED_IN_AREA", 0xF813C7E63F9062A5 },
{  "ADD_NAVMESH_BLOCKING_OBJECT", 0xFCD5C8E06E502F5A },
{  "REMOVE_NAVMESH_BLOCKING_OBJECT", 0x46399A7895957C0E },
{  "DOES_NAVMESH_BLOCKING_OBJECT_EXIST", 0x0EAEB0DB4B132399 },
{  "SET_AMBIENT_PED_RANGE_MULTIPLIER_THIS_FRAME", 0x0B919E1FB47CC4E0 },
{  "SET_IGNORE_NO_GPS_FLAG", 0x72751156E7678833 },
// Category: PED
{  "CREATE_PED", 0xD49F9B0955C367DE },
{  "DELETE_PED", 0xCC0EF140F99365C5 },
{  "CLONE_PED", 0xEF29A16337FACADB },
{  "CLONE_PED_TO_TARGET", 0xE952D6431689AD9A },
{  "IS_PED_IN_VEHICLE", 0xA3EE4A07279BB9DB },
{  "IS_PED_IN_MODEL", 0x796D90EFB19AA332 },
{  "IS_PED_IN_ANY_VEHICLE", 0x997ABD671D25CA0B },
{  "IS_PED_INJURED", 0x84A2DD9AC37C35C1 },
{  "IS_PED_FATALLY_INJURED", 0xD839450756ED5A80 },
{  "IS_PED_DEAD_OR_DYING", 0x3317DEDB88C95038 },
{  "IS_PED_AIMING_FROM_COVER", 0x3998B1276A3300E5 },
{  "IS_PED_RELOADING", 0x24B100C68C645951 },
{  "IS_PED_A_PLAYER", 0x12534C348C6CB68B },
{  "CREATE_PED_INSIDE_VEHICLE", 0x7DD959874C1FD534 },
{  "SET_PED_DESIRED_HEADING", 0xAA5A7ECE2AA8FE70 },
{  "_FREEZE_PED_CAMERA_ROTATION", 0xFF287323B0E2C69A },
{  "IS_PED_FACING_PED", 0xD71649DB0A545AA3 },
{  "IS_PED_IN_MELEE_COMBAT", 0x4E209B2C1EAD5159 },
{  "IS_PED_STOPPED", 0x530944F6F4B8A214 },
{  "IS_ANY_PED_SHOOTING_IN_AREA", 0xA0D3D71EA1086C55 },
{  "IS_PED_SHOOTING", 0x34616828CD07F1A1 },
{  "SET_PED_ACCURACY", 0x7AEFB85C1D49DEB6 },
{  "GET_PED_ACCURACY", 0x37F4AD56ECBC0CD6 },
{  "IS_PED_MODEL", 0xC9D55B1A358A5BF7 },
{  "EXPLODE_PED_HEAD", 0x2D05CED3A38D0F3A },
{  "ADD_ARMOUR_TO_PED", 0x5BA652A0CD14DF2F },
{  "SET_PED_INTO_VEHICLE", 0xF75B0D629E1C063D },
{  "SET_PED_MOVE_ANIMS_BLEND_OUT", 0x9E8C908F41584ECD },
{  "IS_PED_MALE", 0x6D9F5FAA7488BA46 },
{  "IS_PED_HUMAN", 0xB980061DA992779D },
{  "GET_VEHICLE_PED_IS_IN", 0x9A9112A0FE9A4713 },
{  "RESET_PED_LAST_VEHICLE", 0xBB8DE8CF6A8DD8BB },
{  "SET_SCENARIO_PED_DENSITY_MULTIPLIER_THIS_FRAME", 0x7A556143A1C03898 },
{  "SET_PED_NON_CREATION_AREA", 0xEE01041D559983EA },
{  "CLEAR_PED_NON_CREATION_AREA", 0x2E05208086BA0651 },
{  "IS_PED_ON_MOUNT", 0x460BC76A0E10655E },
{  "GET_MOUNT", 0xE7E11B8DCBED1058 },
{  "IS_PED_ON_VEHICLE", 0x67722AEB798E5FAB },
{  "IS_PED_ON_SPECIFIC_VEHICLE", 0xEC5F66E459AF3BB2 },
{  "SET_PED_MONEY", 0xA9C8960E8684C1B5 },
{  "GET_PED_MONEY", 0x3F69145BBA87BAE7 },
{  "IS_PED_SITTING_IN_VEHICLE", 0xA808AA1D79230FC2 },
{  "IS_PED_SITTING_IN_ANY_VEHICLE", 0x826AA586EDB9FEF8 },
{  "IS_PED_ON_FOOT", 0x01FEE67DB37F59B2 },
{  "IS_PED_PLANTING_BOMB", 0xC70B5FAE151982D8 },
{  "GET_DEAD_PED_PICKUP_COORDS", 0xCD5003B097200F36 },
{  "IS_PED_IN_ANY_BOAT", 0x2E0E1C2B4F6CB339 },
{  "IS_PED_IN_ANY_HELI", 0x298B91AE825E5705 },
{  "IS_PED_IN_ANY_PLANE", 0x5FFF4CFC74D8FB80 },
{  "IS_PED_IN_FLYING_VEHICLE", 0x9134873537FA419C },
{  "GET_PED_LAST_DAMAGE_BONE", 0xD75960F6BD9EA49C },
{  "CLEAR_PED_LAST_DAMAGE_BONE", 0x8EF6B7AC68E2F01B },
{  "SET_AI_WEAPON_DAMAGE_MODIFIER", 0x1B1E2A40A65B8521 },
{  "RESET_AI_WEAPON_DAMAGE_MODIFIER", 0xEA16670E7BA4743C },
{  "SET_AI_MELEE_WEAPON_DAMAGE_MODIFIER", 0x66460DEDDD417254 },
{  "SET_PED_CAN_BE_TARGETTED", 0x63F58F7C80513AAD },
{  "SET_PED_CAN_BE_TARGETTED_BY_TEAM", 0xBF1CA77833E58F2C },
{  "SET_PED_CAN_BE_TARGETTED_BY_PLAYER", 0x66B57B72E0836A76 },
{  "IS_PED_FALLING", 0xFB92A102F1C4DFA3 },
{  "IS_PED_JUMPING", 0xCEDABC5900A0BF97 },
{  "IS_PED_CLIMBING", 0x53E8CB4F48BFE623 },
{  "IS_PED_VAULTING", 0x117C70D1F5730B5E },
{  "IS_PED_DIVING", 0x5527B8246FEF9B11 },
{  "_IS_PED_OPENING_A_DOOR", 0x26AF0E8E30BD2A2C },
{  "IS_PED_IN_ANY_TAXI", 0x6E575D6A898AB852 },
{  "SET_PED_ID_RANGE", 0xF107E836A70DCE05 },
{  "SET_PED_HIGHLY_PERCEPTIVE", 0x52D59AB61DDC05DD },
{  "SET_PED_SEEING_RANGE", 0xF29CF591C4BF6CEE },
{  "SET_PED_HEARING_RANGE", 0x33A8F7F7D5F7F33C },
{  "SET_PED_VISUAL_FIELD_MIN_ANGLE", 0x2DB492222FB21E26 },
{  "SET_PED_VISUAL_FIELD_MAX_ANGLE", 0x70793BDCA1E854D4 },
{  "SET_PED_VISUAL_FIELD_PERIPHERAL_RANGE", 0x9C74B0BC831B753A },
{  "SET_PED_VISUAL_FIELD_CENTER_ANGLE", 0x3B6405E8AB34A907 },
{  "SET_PED_STEALTH_MOVEMENT", 0x88CBB5CEB96B7BD2 },
{  "GET_PED_STEALTH_MOVEMENT", 0x7C2AC9CA66575FBF },
{  "CREATE_GROUP", 0x90370EBE0FEE1A3D },
{  "SET_PED_AS_GROUP_LEADER", 0x2A7819605465FBCE },
{  "SET_PED_AS_GROUP_MEMBER", 0x9F3480FE65DB31B5 },
{  "SET_PED_CAN_TELEPORT_TO_GROUP_LEADER", 0x2E2F4240B3F24647 },
{  "REMOVE_GROUP", 0x8EB2F69076AF7053 },
{  "REMOVE_PED_FROM_GROUP", 0xED74007FFB146BC2 },
{  "IS_PED_GROUP_MEMBER", 0x9BB01E3834671191 },
{  "IS_PED_HANGING_ON_TO_VEHICLE", 0x1C86D8AEF8254B78 },
{  "SET_GROUP_SEPARATION_RANGE", 0x4102C7858CFEE4E4 },
{  "IS_PED_PRONE", 0xD6A86331A537A7B9 },
{  "IS_PED_IN_COMBAT", 0x4859F1FC66A6278E },
{  "CAN_PED_IN_COMBAT_SEE_TARGET", 0xEAD42DE3610D0721 },
{  "IS_PED_JACKING", 0x4AE4FF911DFB61DA },
{  "IS_PED_BEING_JACKED", 0x9A497FE2DF198913 },
{  "IS_PED_BEING_STUNNED", 0x4FBACCE3B4138EE8 },
{  "GET_PEDS_JACKER", 0x9B128DC36C1E04CF },
{  "GET_JACK_TARGET", 0x5486A79D9FBD342D },
{  "IS_PED_FLEEING", 0xBBCCE00B381F8482 },
{  "IS_PED_IN_COVER", 0x60DFD0691A170B88 },
{  "IS_PED_IN_COVER_FACING_LEFT", 0x845333B3150583AB },
{  "IS_PED_GOING_INTO_COVER", 0x9F65DBC537E59AD5 },
{  "GET_SEAT_PED_IS_TRYING_TO_ENTER", 0x6F4C85ACD641BCD2 },
{  "GET_PED_SOURCE_OF_DEATH", 0x93C8B64DEB84728C },
{  "GET_PED_CAUSE_OF_DEATH", 0x16FFE42AB2D2DC59 },
{  "GET_PED_TIME_OF_DEATH", 0x1E98817B311AE98A },
{  "SET_PED_RELATIONSHIP_GROUP_DEFAULT_HASH", 0xADB3F206518799E8 },
{  "SET_PED_RELATIONSHIP_GROUP_HASH", 0xC80A74AC829DDD92 },
{  "SET_RELATIONSHIP_BETWEEN_GROUPS", 0xBF25EB89375A37AD },
{  "CLEAR_RELATIONSHIP_BETWEEN_GROUPS", 0x5E29243FB56FC6D4 },
{  "ADD_RELATIONSHIP_GROUP", 0xF372BC22FCB88606 },
{  "REMOVE_RELATIONSHIP_GROUP", 0xB6BA2444AB393DA2 },
{  "GET_RELATIONSHIP_BETWEEN_PEDS", 0xEBA5AD3A0EAF7121 },
{  "GET_PED_RELATIONSHIP_GROUP_DEFAULT_HASH", 0x42FDD0F017B1E38E },
{  "GET_PED_RELATIONSHIP_GROUP_HASH", 0x7DBDD04862D95F04 },
{  "GET_RELATIONSHIP_BETWEEN_GROUPS", 0x9E6B70061662AE5C },
{  "SET_PED_TO_INFORM_RESPECTED_FRIENDS", 0x112942C6E708F70B },
{  "IS_PED_RESPONDING_TO_EVENT", 0x625B774D75C87068 },
{  "SET_PED_FIRING_PATTERN", 0x9AC577F5A12AD8A9 },
{  "SET_PED_SHOOT_RATE", 0x614DA022990752DC },
{  "SET_COMBAT_FLOAT", 0xFF41B4B141ED981C },
{  "GET_COMBAT_FLOAT", 0x52DFF8A10508090A },
{  "GET_GROUP_SIZE", 0x8DE69FE35CA09A45 },
{  "DOES_GROUP_EXIST", 0x7C6B0C22F9F40BBE },
{  "GET_PED_GROUP_INDEX", 0xF162E133B4E7A675 },
{  "IS_PED_IN_GROUP", 0x5891CAC5D4ACFF74 },
{  "GET_PLAYER_PED_IS_FOLLOWING", 0x6A3975DEA89F9A17 },
{  "SET_GROUP_FORMATION", 0xCE2F5FC3AF7E8C1E },
{  "SET_GROUP_FORMATION_SPACING", 0x1D9D45004C28C916 },
{  "RESET_GROUP_FORMATION_DEFAULT_SPACING", 0x63DAB4CCB3273205 },
{  "GET_VEHICLE_PED_IS_USING", 0x6094AD011A2EA87D },
{  "GET_VEHICLE_PED_IS_ENTERING", 0xF92691AED837A5FC },
{  "SET_PED_GRAVITY", 0x9FF447B6B6AD960A },
{  "APPLY_DAMAGE_TO_PED", 0x697157CED63F18D4 },
{  "GET_PED_TYPE", 0xFF059E1E4C01E63C },
{  "SET_PED_AS_COP", 0xBB03C38DD3FB7FFD },
{  "SET_PED_MAX_HEALTH", 0xF5F6378C4F3419D3 },
{  "GET_PED_MAX_HEALTH", 0x4700A416E8324EF3 },
{  "SET_PED_MAX_TIME_IN_WATER", 0x43C851690662113D },
{  "SET_PED_MAX_TIME_UNDERWATER", 0x6BA428C528D9E522 },
{  "SET_PED_CAN_BE_KNOCKED_OFF_VEHICLE", 0x7A6535691B477C48 },
{  "CAN_KNOCK_PED_OFF_VEHICLE", 0x51AC07A44D4F5B8A },
{  "KNOCK_PED_OFF_VEHICLE", 0x45BBCBA77C29A841 },
{  "GET_PED_AS_GROUP_MEMBER", 0x51455483CF23ED97 },
{  "GET_PED_AS_GROUP_LEADER", 0x5CCE68DBD5FE93EC },
{  "SET_PED_KEEP_TASK", 0x971D38760FBC02EF },
{  "IS_PED_SWIMMING", 0x9DE327631295B4C2 },
{  "IS_PED_SWIMMING_UNDER_WATER", 0xC024869A53992F34 },
{  "SET_CREATE_RANDOM_COPS", 0x102E68B2024D536D },
{  "IS_PED_IN_ANY_TRAIN", 0x6F972C1AB75A1ED0 },
{  "IS_PED_GETTING_INTO_A_VEHICLE", 0xBB062B2B5722478E },
{  "SET_ENABLE_HANDCUFFS", 0xDF1AF8B5D56542FA },
{  "SET_ENABLE_BOUND_ANKLES", 0xC52E0F855C58FC2E },
{  "RESET_PED_WEAPON_MOVEMENT_CLIPSET", 0x97B0DB5B4AA74E77 },
{  "RESET_PED_IN_VEHICLE_CONTEXT", 0x22EF8FF8778030EB },
{  "SET_PED_GESTURE_GROUP", 0xDDF803377F94AAA8 },
{  "GET_ANIM_INITIAL_OFFSET_POSITION", 0xBE22B26DD764C040 },
{  "GET_ANIM_INITIAL_OFFSET_ROTATION", 0x4B805E6046EE9E47 },
{  "SET_PED_RANDOM_COMPONENT_VARIATION", 0xC8A9481A01E63C28 },
{  "KNOCK_OFF_PED_PROP", 0x6FD7816A36615F48 },
{  "SET_BLOCKING_OF_NON_TEMPORARY_EVENTS", 0x9F8AA94D6D97DBF4 },
{  "REGISTER_TARGET", 0x2F25D9AEFA34FBA2 },
{  "REGISTER_HATED_TARGETS_AROUND_PED", 0x9222F300BF8354FE },
{  "GET_CLOSEST_PED", 0xC33AB876A77F8164 },
{  "CAN_PED_RAGDOLL", 0x128F79EDCECE4FD5 },
{  "SET_PED_TO_RAGDOLL", 0xAE99FB955581844A },
{  "SET_PED_TO_RAGDOLL_WITH_FALL", 0xD76632D99E4966C8 },
{  "SET_PED_RAGDOLL_ON_COLLISION", 0xF0A4F1BBF4FA7497 },
{  "IS_PED_RAGDOLL", 0x47E4E977581C5B55 },
{  "IS_PED_RUNNING_RAGDOLL_TASK", 0xE3B6097CC25AA69E },
{  "SET_PED_RAGDOLL_FORCE_FALL", 0x01F6594B923B9251 },
{  "RESET_PED_RAGDOLL_TIMER", 0x9FA4664CF62E47E8 },
{  "SET_PED_CAN_RAGDOLL", 0xB128377056A54E2A },
{  "IS_PED_RUNNING_MOBILE_PHONE_TASK", 0x2AFE52F782F25775 },
{  "SET_RAGDOLL_BLOCKING_FLAGS", 0x26695EC767728D84 },
{  "CLEAR_RAGDOLL_BLOCKING_FLAGS", 0xD86D101FCFD00A4B },
{  "SET_PED_SPHERE_DEFENSIVE_AREA", 0x9D3151A373974804 },
{  "SET_PED_DEFENSIVE_AREA_DIRECTION", 0x413C6C763A4AFFAD },
{  "REMOVE_PED_DEFENSIVE_AREA", 0x74D4E028107450A9 },
{  "GET_PED_DEFENSIVE_AREA_POSITION", 0x3C06B8786DD94CD1 },
{  "IS_PED_DEFENSIVE_AREA_ACTIVE", 0xBA63D9FE45412247 },
{  "REVIVE_INJURED_PED", 0x8D8ACD8388CD99CE },
{  "RESURRECT_PED", 0x71BC8E838B9C6035 },
{  "SET_PED_NAME_DEBUG", 0x98EFA132A4117BE1 },
{  "APPLY_PED_BLOOD_SPECIFIC", 0xEF0D582CBF2D9B0F },
{  "APPLY_PED_DAMAGE_PACK", 0x46DF918788CB093F },
{  "CLEAR_PED_BLOOD_DAMAGE", 0x8FE22675A5A45817 },
{  "CLEAR_PED_BLOOD_DAMAGE_BY_ZONE", 0x56E3B78C5408D9F4 },
{  "CLEAR_PED_DAMAGE_DECAL_BY_ZONE", 0x523C79AEEFCC4A2A },
{  "CLEAR_PED_WETNESS", 0x9C720776DAA43E7E },
{  "SET_PED_WETNESS_HEIGHT", 0x44CB6447D2571AA0 },
{  "SET_PED_WETNESS_ENABLED_THIS_FRAME", 0xB5485E4907B53019 },
{  "CLEAR_PED_ENV_DIRT", 0x6585D955A68452A5 },
{  "SET_PED_SWEAT", 0x27B0405F59637D1F },
{  "CLEAR_PED_DECORATIONS", 0x0E5173C163976E38 },
{  "WAS_PED_SKELETON_UPDATED", 0x11B499C1E0FF8559 },
{  "GET_PED_BONE_COORDS", 0x17C07FC640E86B4E },
{  "ADD_SCENARIO_BLOCKING_AREA", 0x1B5C85C612E5256E },
{  "REMOVE_SCENARIO_BLOCKING_AREAS", 0xD37401D78A929A49 },
{  "REMOVE_SCENARIO_BLOCKING_AREA", 0x31D16B74C6E29D66 },
{  "IS_PED_USING_ANY_SCENARIO", 0x57AB4A3080F85143 },
{  "SET_FACIAL_IDLE_ANIM_OVERRIDE", 0xFFC24B988B938B38 },
{  "CLEAR_FACIAL_IDLE_ANIM_OVERRIDE", 0x726256CC1EEB182F },
{  "SET_PED_CAN_PLAY_GESTURE_ANIMS", 0xBAF20C5432058024 },
{  "SET_PED_CAN_PLAY_AMBIENT_ANIMS", 0x6373D1349925A70E },
{  "SET_PED_CAN_PLAY_AMBIENT_BASE_ANIMS", 0x0EB0585D15254740 },
{  "SET_PED_CAN_ARM_IK", 0x6C3B4D6D13B4C841 },
{  "SET_PED_CAN_HEAD_IK", 0xC11C18092C5530DC },
{  "SET_PED_CAN_LEG_IK", 0x73518ECE2485412B },
{  "SET_PED_CAN_TORSO_IK", 0xF2B7106D37947CE0 },
{  "SET_PED_CAN_USE_AUTO_CONVERSATION_LOOKAT", 0xEC4686EC06434678 },
{  "IS_PED_HEADTRACKING_PED", 0x5CD3CB88A7F8850D },
{  "IS_PED_HEADTRACKING_ENTITY", 0x813A0A7C9D2E831F },
{  "SET_PED_CONFIG_FLAG", 0x1913FE4CBF41C463 },
{  "SET_PED_RESET_FLAG", 0xC1E8A365BF3B29F2 },
{  "GET_PED_CONFIG_FLAG", 0x7EE53118C892B513 },
{  "GET_PED_RESET_FLAG", 0xAF9E59B1B1FBF2A0 },
{  "SET_PED_GROUP_MEMBER_PASSENGER_INDEX", 0x0BDDB8D9EC6BCF3C },
{  "IS_PED_EVASIVE_DIVING", 0x414641C26E105898 },
{  "SET_PED_MODEL_IS_SUPPRESSED", 0xE163A4BCE4DE6F11 },
{  "SET_PED_CAN_RAGDOLL_FROM_PLAYER_IMPACT", 0xDF993EE5E90ABA25 },
{  "SET_PED_LEG_IK_MODE", 0xC396F5B86FF9FEBD },
{  "SET_PED_COMBAT_MOVEMENT", 0x4D9CA1009AFBD057 },
{  "GET_PED_COMBAT_MOVEMENT", 0xDEA92412FCAEB3F5 },
{  "SET_PED_COMBAT_ABILITY", 0xC7622C0D36B2FDA8 },
{  "SET_PED_COMBAT_RANGE", 0x3C606747B23E497B },
{  "SET_PED_COMBAT_ATTRIBUTES", 0x9F7794730795E019 },
{  "SET_PED_TARGET_LOSS_RESPONSE", 0x0703B9079823DA4A },
{  "IS_PED_PERFORMING_MELEE_ACTION", 0xDCCA191DF9980FD7 },
{  "IS_PED_BEING_STEALTH_KILLED", 0x863B23EFDE9C5DF2 },
{  "GET_MELEE_TARGET_FOR_PED", 0x18A3E9EE1297FD39 },
{  "SET_PED_FLEE_ATTRIBUTES", 0x70A2D1137C8ED7C9 },
{  "IS_ANY_PED_NEAR_POINT", 0x083961498679DC9F },
{  "IS_PED_HEADING_TOWARDS_POSITION", 0xFCF37A457CB96DC0 },
{  "REQUEST_PED_VISIBILITY_TRACKING", 0x7D7A2E43E74E2EB8 },
{  "REQUEST_PED_VEHICLE_VISIBILITY_TRACKING", 0x2BC338A7B21F4608 },
{  "IS_TRACKED_PED_VISIBLE", 0x91C8E617F64188AC },
{  "GET_PED_BONE_INDEX", 0x3F428D08BE5AAE31 },
{  "FORCE_PED_MOTION_STATE", 0xF28965D04F570DCA },
{  "SET_PED_MAX_MOVE_BLEND_RATIO", 0x433083750C5E064A },
{  "SET_PED_MIN_MOVE_BLEND_RATIO", 0x01A898D26E2333DD },
{  "SET_PED_MOVE_RATE_OVERRIDE", 0x085BF80FA50A39D1 },
{  "GET_PED_NEARBY_VEHICLES", 0xCFF869CBFA210D82 },
{  "GET_PED_NEARBY_PEDS", 0x23F8F5FC7E8C4A6B },
{  "IS_PED_USING_ACTION_MODE", 0x00E73468D085F745 },
{  "SET_PED_USING_ACTION_MODE", 0xD75ACCF5E0FB5367 },
{  "SET_PED_CAPSULE", 0x364DF566EC833DE2 },
{  "SET_IK_TARGET", 0xC32779C16FCEECD9 },
{  "SET_PED_LOD_MULTIPLIER", 0xDC2C5C242AAC342B },
// Category: _NAMESPACE48
// Category: _NAMESPACE49
// Category: PHYSICS
{  "ADD_ROPE", 0xE832D760399EB220 },
{  "DELETE_ROPE", 0x52B4829281364649 },
{  "DELETE_CHILD_ROPE", 0xAA5D6B1888E4DB20 },
{  "DOES_ROPE_EXIST", 0xFD5448BE3111ED96 },
{  "ROPE_DRAW_SHADOW_ENABLED", 0xF159A63806BB5BA8 },
{  "GET_ROPE_VERTEX_COUNT", 0x3655F544CD30F0B5 },
{  "ATTACH_ENTITIES_TO_ROPE", 0x3D95EC8B6D940AC3 },
{  "DETACH_ROPE_FROM_ENTITY", 0xBCF3026912A8647D },
{  "GET_ROPE_LAST_VERTEX_COORD", 0x21BB0FBD3E217C2D },
{  "GET_ROPE_VERTEX_COORD", 0xEA61CA8E80F09E4D },
{  "START_ROPE_WINDING", 0x1461C72C889E343E },
{  "STOP_ROPE_WINDING", 0xCB2D4AB84A19AA7C },
{  "START_ROPE_UNWINDING_FRONT", 0x538D1179EC1AA9A9 },
{  "STOP_ROPE_UNWINDING_FRONT", 0xFFF3A50779EFBBB3 },
{  "ROPE_FORCE_LENGTH", 0xD009F759A723DB1B },
{  "SET_DAMPING", 0xEEA3B200A6FEB65B },
{  "ACTIVATE_PHYSICS", 0x710311ADF0E20730 },
{  "BREAK_ENTITY_GLASS", 0x2E648D16F6E308F3 },
{  "SET_DISABLE_BREAKING", 0x5CEC1A84620E7D5B },
{  "SET_DISABLE_FRAG_DAMAGE", 0x01BA3AED21C16CFB },
// Category: PLAYER
{  "GET_PLAYER_PED", 0x275F255ED201B937 },
{  "GET_PLAYER_PED_SCRIPT_INDEX", 0x5C880F9056D784C8 },
{  "SET_PLAYER_MODEL", 0xED40380076A31506 },
{  "GET_PLAYER_TEAM", 0xB464EB6A40C7975B },
{  "SET_PLAYER_TEAM", 0xE8DD8536F01DE600 },
{  "GET_PLAYER_NAME", 0x7124FD9AC0E01BA0 },
{  "GET_WANTED_LEVEL_RADIUS", 0x80B00EB26D9521C7 },
{  "GET_WANTED_LEVEL_THRESHOLD", 0x1B1A3B358F7D8F07 },
{  "SET_PLAYER_WANTED_LEVEL", 0x384D4765395E006C },
{  "IS_PLAYER_WANTED_LEVEL_GREATER", 0xE1C0AD4C24324C36 },
{  "CLEAR_PLAYER_WANTED_LEVEL", 0x4E4B996C928C7AA6 },
{  "IS_PLAYER_DEAD", 0x2E9C3FCB6798F397 },
{  "SET_PLAYER_CONTROL", 0x4D51E59243281D80 },
{  "GET_PLAYER_WANTED_LEVEL", 0xABC532F9098BFD9D },
{  "SET_MAX_WANTED_LEVEL", 0x28A4BD2CEE236E19 },
{  "SET_POLICE_RADAR_BLIPS", 0x6FD7DD6B63F2820E },
{  "IS_PLAYER_PLAYING", 0xBFFB35986CAAE58C },
{  "SET_EVERYONE_IGNORE_PLAYER", 0x34630A768925B852 },
{  "SET_ALL_RANDOM_PEDS_FLEE", 0xE705309B8C6445A4 },
{  "SET_ALL_RANDOM_PEDS_FLEE_THIS_FRAME", 0xD5C198A62F1DEB0A },
{  "SET_WANTED_LEVEL_MULTIPLIER", 0xD7FA719CB54866C2 },
{  "RESET_WANTED_LEVEL_DIFFICULTY", 0x062D14F18E8B0CAE },
{  "CAN_PLAYER_START_MISSION", 0x2DF170B1185AF777 },
{  "IS_PLAYER_READY_FOR_CUTSCENE", 0xAA67BCB0097F2FA3 },
{  "IS_PLAYER_TARGETTING_ENTITY", 0x27F89FDC16688A7A },
{  "GET_PLAYER_TARGET_ENTITY", 0xAE663DDD99C8A670 },
{  "IS_PLAYER_FREE_AIMING", 0x936F967D4BE1CE9D },
{  "IS_PLAYER_FREE_AIMING_AT_ENTITY", 0x8C67C11C68713D25 },
{  "GET_ENTITY_PLAYER_IS_FREE_AIMING_AT", 0xA6817C110B830EAD },
{  "SET_PLAYER_LOCKON_RANGE_OVERRIDE", 0x3A3CD06597388322 },
{  "SET_PLAYER_CAN_BE_HASSLED_BY_GANGS", 0xC7FE774412046825 },
{  "SET_PLAYER_CAN_USE_COVER", 0x5EDA520F7A3BAF4E },
{  "GET_MAX_WANTED_LEVEL", 0xD04CFAD1E2B7984A },
{  "IS_PLAYER_TARGETTING_ANYTHING", 0x4605C66E0F935F83 },
{  "RESTORE_PLAYER_STAMINA", 0xC41F4B6E23FE6A4A },
{  "GET_PLAYER_GROUP", 0x9BAB31815159ABCF },
{  "IS_PLAYER_CONTROL_ON", 0x7964097FCE4C244B },
{  "IS_PLAYER_SCRIPT_CONTROL_ON", 0xB78350754157C00F },
{  "IS_PLAYER_CLIMBING", 0xB8A70C22FD48197A },
{  "IS_PLAYER_BEING_ARRESTED", 0xC8183AE963C58374 },
{  "RESET_PLAYER_ARREST_STATE", 0x12917931C31F1750 },
{  "GET_PLAYERS_LAST_VEHICLE", 0x2F96E7720B0B19EA },
{  "GET_PLAYER_INDEX", 0x47E385B0D957C8D4 },
{  "INT_TO_PLAYERINDEX", 0x748B3A65C2604C33 },
{  "INT_TO_PARTICIPANTINDEX", 0x58FF971FC8F2702C },
{  "PLAYER_ID", 0x217E9DC48139933D },
{  "PLAYER_PED_ID", 0x096275889B8E0EE0 },
{  "NETWORK_PLAYER_ID_TO_INT", 0x8A9386F0749A17FA },
{  "HAS_FORCE_CLEANUP_OCCURRED", 0xC11469DCA6FC3BB5 },
{  "FORCE_CLEANUP", 0x768C017FB878E4F4 },
{  "FORCE_CLEANUP_FOR_ALL_THREADS_WITH_THIS_NAME", 0xDAACAF8B687F2353 },
{  "FORCE_CLEANUP_FOR_THREAD_WITH_THIS_ID", 0xF4C9512A2F0A3031 },
{  "GET_CAUSE_OF_MOST_RECENT_FORCE_CLEANUP", 0x84E8E29EBD4A46D2 },
{  "SET_PLAYER_MAY_ONLY_ENTER_THIS_VEHICLE", 0xDA35A134038557EC },
{  "SET_PLAYER_MAY_NOT_ENTER_ANY_VEHICLE", 0xBEC463B3A11C909E },
{  "IS_SYSTEM_UI_BEING_DISPLAYED", 0x908258B6209E71F7 },
{  "SET_PLAYER_INVINCIBLE", 0xFEBEEBC9CBDF4B12 },
{  "GET_PLAYER_INVINCIBLE", 0x0CBBCB2CCFA7DC4E },
{  "SET_PLAYER_LOCKON", 0x462AA1973CBBA75E },
{  "SET_PLAYER_TARGETING_MODE", 0xD66A941F401E7302 },
{  "CLEAR_PLAYER_HAS_DAMAGED_AT_LEAST_ONE_PED", 0x270B63A641BE32F2 },
{  "HAS_PLAYER_DAMAGED_AT_LEAST_ONE_PED", 0xDA4A4B9B96E20092 },
{  "CLEAR_PLAYER_HAS_DAMAGED_AT_LEAST_ONE_NON_ANIMAL_PED", 0x0361096D6CE4372C },
{  "HAS_PLAYER_DAMAGED_AT_LEAST_ONE_NON_ANIMAL_PED", 0x16C8D205DD5A2E90 },
{  "SET_AIR_DRAG_MULTIPLIER_FOR_PLAYERS_VEHICLE", 0x5DA6500FE849DA16 },
{  "SET_SWIM_MULTIPLIER_FOR_PLAYER", 0xBFCEABDE34DA5085 },
{  "SET_PLAYER_FORCED_AIM", 0xD5FCC166AEB2FD0F },
{  "DISABLE_PLAYER_FIRING", 0x2970929FD5F9FC89 },
{  "START_PLAYER_TELEPORT", 0xDF8822C55EDDA65B },
{  "_HAS_PLAYER_TELEPORT_FINISHED", 0xC39DCE4672CBFBC1 },
{  "STOP_PLAYER_TELEPORT", 0x0858B86146601BE8 },
{  "IS_PLAYER_TELEPORT_ACTIVE", 0x085EEAEB8783FEB5 },
{  "GET_PLAYER_CURRENT_STEALTH_NOISE", 0xD7ECC25E176ECBA5 },
{  "SET_PLAYER_HEALTH_RECHARGE_MULTIPLIER", 0x8899C244EBCF70DE },
{  "SET_PLAYER_WEAPON_DAMAGE_MODIFIER", 0x94D529F7B73D7A85 },
{  "SET_PLAYER_WEAPON_DEFENSE_MODIFIER", 0xD15CC2D493160BE3 },
{  "SET_PLAYER_MELEE_WEAPON_DAMAGE_MODIFIER", 0xE4CB5A3F18170381 },
{  "SET_PLAYER_NOISE_MULTIPLIER", 0xB5EC6BDAEBCA454C },
{  "SET_PLAYER_SNEAKING_NOISE_MULTIPLIER", 0x4DE44FA389DCA565 },
{  "SIMULATE_PLAYER_INPUT_GAIT", 0xFA0C063C422C4355 },
{  "RESET_PLAYER_INPUT_GAIT", 0x61A2EECAB274829B },
{  "SET_PLAYER_SIMULATE_AIMING", 0xE0447DEF81CCDFD2 },
{  "SET_PLAYER_CLOTH_PIN_FRAMES", 0xD0D9317DFEEF9A66 },
{  "HAS_PLAYER_BEEN_SPOTTED_IN_STOLEN_VEHICLE", 0xC932F57F31EA9152 },
{  "IS_PLAYER_RIDING_TRAIN", 0x2FB0ACADA6A238DD },
// Category: _NAMESPACE52
// Category: _NAMESPACE53
// Category: _NAMESPACE54
// Category: QUEUE
{  "_EVENT_QUEUE_IS_EMPTY", 0x402B5D7D269FF796 },
{  "_EVENT_QUEUE_POP", 0xD87DF294B049211D },
// Category: _NAMESPACE56
{  "_STOP_RECORDING_THIS_FRAME", 0xA8C44C13419634F2 },
// Category: REPLAY
{  "_IS_INTERIOR_RENDERING_DISABLED", 0x0F838D47DE58EDB2 },
// Category: SCRIPTS
{  "REQUEST_SCRIPT", 0x46ED607DDD40D7FE },
{  "SET_SCRIPT_AS_NO_LONGER_NEEDED", 0x0086D3067E1CFD1C },
{  "HAS_SCRIPT_LOADED", 0xE97BD36574F8B0A6 },
{  "DOES_SCRIPT_EXIST", 0x552B171E3F69E5AE },
{  "REQUEST_SCRIPT_WITH_NAME_HASH", 0xF6B9CE3F8D5B9B74 },
{  "SET_SCRIPT_WITH_NAME_HASH_AS_NO_LONGER_NEEDED", 0x50723A1567C8361E },
{  "HAS_SCRIPT_WITH_NAME_HASH_LOADED", 0xA5D8E0C2F3C7EEBC },
{  "DOES_SCRIPT_WITH_NAME_HASH_EXIST", 0xA34E89749F628284 },
{  "TERMINATE_THREAD", 0x87ED52AE40EA1A52 },
{  "IS_THREAD_ACTIVE", 0x46E9AE36D8FA6417 },
{  "SCRIPT_THREAD_ITERATOR_RESET", 0x39382EB8DCD8684D },
{  "SCRIPT_THREAD_ITERATOR_GET_NEXT_THREAD_ID", 0x3CE3FB167E837D7C },
{  "GET_ID_OF_THIS_THREAD", 0x55525C346BEF6960 },
{  "TERMINATE_THIS_THREAD", 0x5E8B6D17FF91CD59 },
{  "_GET_NUMBER_OF_REFERENCES_OF_SCRIPT_WITH_NAME_HASH", 0x8E34C953364A76DD },
{  "GET_HASH_OF_THIS_SCRIPT_NAME", 0xBC2C927F5C264960 },
{  "GET_NUMBER_OF_EVENTS", 0x5CE8DE5909565748 },
{  "GET_EVENT_EXISTS", 0xC9F59C0A710ECD34 },
{  "GET_EVENT_AT_INDEX", 0xA85E614430EFF816 },
{  "GET_EVENT_DATA", 0x57EC5FA4D4D6AFCA },
{  "TRIGGER_SCRIPT_EVENT", 0x5AE99C571D5BBE5D },
{  "SHUTDOWN_LOADING_SCREEN", 0xFC179D7E8886DADF },
{  "SET_NO_LOADING_SCREEN", 0x5CB83156AA038F95 },
{  "GET_NO_LOADING_SCREEN", 0x323DAF00687E0F28 },
{  "BG_START_CONTEXT_HASH", 0x2EB67D564DCC09D5 },
{  "BG_END_CONTEXT_HASH", 0x6D1431744182CDE8 },
{  "BG_START_CONTEXT", 0x49BA5678BA040CA7 },
{  "BG_END_CONTEXT", 0x3ABF7BA1C3E2C8CF },
// Category: _NAMESPACE59
// Category: SHAPETEST
{  "START_SHAPE_TEST_LOS_PROBE", 0x7EE9F5D83DD4F90E },
{  "_START_SHAPE_TEST_RAY", 0x377906D8A31E5586 },
{  "START_SHAPE_TEST_BOX", 0xFE466162C4401D18 },
{  "START_SHAPE_TEST_CAPSULE", 0x28579D1B8F8AAC80 },
{  "START_SHAPE_TEST_SWEPT_SPHERE", 0xAA5B7C8309F73230 },
{  "_START_SHAPE_TEST_SURROUNDING_COORDS", 0x9839013D8B6014F1 },
{  "GET_SHAPE_TEST_RESULT", 0xEDE8AC7C5108FB1D },
// Category: SOCIALCLUB
{  "_GET_TOTAL_SC_INBOX_IDS", 0x8EF0F633280C0663 },
{  "_SC_INBOX_MESSAGE_INIT", 0xFF92537C4DDC1241 },
{  "_IS_SC_INBOX_VALID", 0x74CF39E030A382C4 },
{  "_SC_INBOX_MESSAGE_POP", 0x63CAC501FFF66DC4 },
{  "SC_INBOX_MESSAGE_GET_DATA_INT", 0x95BB39C4DA99F348 },
{  "SC_INBOX_MESSAGE_GET_DATA_STRING", 0x66F77FD58506FF6B },
{  "_SC_INBOX_MESSAGE_GET_STRING", 0x176D077685CD83E4 },
{  "SC_PROFANITY_CHECK_STRING", 0x9C74AC9D87B3FFF4 },
{  "SC_PROFANITY_GET_CHECK_IS_VALID", 0x08C8052AF40C4247 },
{  "SC_PROFANITY_GET_CHECK_IS_PENDING", 0x3A10BCD0C8AA0B82 },
{  "SC_PROFANITY_GET_STRING_PASSED", 0xF302973BB8BE70E6 },
{  "SC_PROFANITY_GET_STRING_STATUS", 0x0CF3BFB99EBBE5B1 },
// Category: SOCIALCLUBFEED
{  "_SC_FEED_SUBMIT_PRESET_MESSAGE", 0xEFB64240F6B17817 },
// Category: SPACTIONPROXY
{  "_START_MANAGER", 0x1F471B79ACC91BEE },
{  "_MANAGER_IS_READY", 0x1F471B79ACC91BED },
{  "_MANAGER_IS_FAILED", 0x1F471B79ACC91BEC },
// Category: _NAMESPACE64
// Category: STREAMING
{  "REQUEST_MODEL", 0xFA28FE3A6246FC30 },
{  "HAS_MODEL_LOADED", 0x1283B8B89DD5D1B6 },
{  "SET_MODEL_AS_NO_LONGER_NEEDED", 0x4AD96EF928BD4F9A },
{  "IS_MODEL_IN_CDIMAGE", 0xD6F3B6D7716CFF8E },
{  "IS_MODEL_VALID", 0x392C8D8E07B70EFC },
{  "IS_MODEL_A_VEHICLE", 0x354F62672DE7DB0A },
{  "REQUEST_COLLISION_AT_COORD", 0x0A3720F162A033C9 },
{  "REQUEST_COLLISION_FOR_MODEL", 0xF1767BE37F661551 },
{  "HAS_COLLISION_FOR_MODEL_LOADED", 0x210A79C9EC89778F },
{  "DOES_ANIM_DICT_EXIST", 0x537F44CB0D7F150D },
{  "REQUEST_ANIM_DICT", 0xA862A2AD321F94B4 },
{  "HAS_ANIM_DICT_LOADED", 0x27FF6FE8009B40CA },
{  "REMOVE_ANIM_DICT", 0x4763145053A33D46 },
{  "REQUEST_CLIP_SET", 0xEF7611B57A820126 },
{  "HAS_CLIP_SET_LOADED", 0x1F23D6B6DA1CC3B2 },
{  "REMOVE_CLIP_SET", 0x817FA1B1EE7CD6F0 },
{  "SET_GAME_PAUSES_FOR_STREAMING", 0xB3BC8250F4FE8B63 },
{  "GET_NUMBER_OF_STREAMING_REQUESTS", 0x30CCCC4D88E654CA },
{  "REQUEST_PTFX_ASSET", 0x001FF43843028E0C },
{  "HAS_PTFX_ASSET_LOADED", 0x13A3F30A9ED0BC31 },
{  "REMOVE_PTFX_ASSET", 0x042F9049EA419E86 },
{  "REQUEST_NAMED_PTFX_ASSET", 0xF2B2353BBC0D4E8F },
{  "HAS_NAMED_PTFX_ASSET_LOADED", 0x65BB72F29138F5D6 },
{  "REMOVE_NAMED_PTFX_ASSET", 0xF20866829E1C81A2 },
{  "CLEAR_FOCUS", 0x86CCAF7CE493EFBE },
{  "_SET_FOCUS_AREA", 0x25F6EF88664540E2 },
{  "SET_FOCUS_ENTITY", 0x955AEDD58F4BD309 },
{  "IS_ENTITY_FOCUS", 0xF87DE697E9A06EC6 },
{  "SET_MAPDATACULLBOX_ENABLED", 0x3CACC83F6FED837C },
{  "IS_PLAYER_SWITCH_IN_PROGRESS", 0xED20CB1F5297791D },
{  "PREFETCH_SRL", 0x354837E5A5BAA5AF },
{  "IS_SRL_LOADED", 0x5C2C88512CF6DAFB },
{  "BEGIN_SRL", 0x0360710033BE60D9 },
{  "END_SRL", 0x1CE71FB33CA079FE },
{  "SET_SRL_TIME", 0x18231AEF458BCFF2 },
{  "SET_HD_AREA", 0xB88B905AFA35CB4D },
{  "CLEAR_HD_AREA", 0xD83B22434E52728D },
// Category: TASK
{  "TASK_PAUSE", 0xE73A266DB0CA9042 },
{  "TASK_STAND_STILL", 0x919BE13EED931959 },
{  "TASK_JUMP", 0x0AE4086104E067B1 },
{  "TASK_COWER", 0x3EB1FE9E8E908E15 },
{  "TASK_HANDS_UP", 0xF2EAB31979A7F910 },
{  "UPDATE_TASK_HANDS_UP_DURATION", 0xA98FCAFD7893C834 },
{  "TASK_ENTER_VEHICLE", 0xC20E50AA46D09CA8 },
{  "TASK_LEAVE_VEHICLE", 0xD3DBCE61A490BE02 },
{  "TASK_VEHICLE_DRIVE_TO_COORD", 0xE2A2AA2F659D77A7 },
{  "TASK_VEHICLE_DRIVE_WANDER", 0x480142959D337D00 },
{  "TASK_FOLLOW_TO_OFFSET_OF_ENTITY", 0x304AE42E357B8C7E },
{  "TASK_GO_STRAIGHT_TO_COORD", 0xD76B57B44F1E6F8B },
{  "TASK_GO_STRAIGHT_TO_COORD_RELATIVE_TO_ENTITY", 0x61E360B7E040D12E },
{  "TASK_ACHIEVE_HEADING", 0x93B93A37987F1F3D },
{  "TASK_FLUSH_ROUTE", 0x841142A1376E9006 },
{  "TASK_EXTEND_ROUTE", 0x1E7889778264843A },
{  "TASK_GO_TO_ENTITY", 0x6A071245EB0D1882 },
{  "TASK_SMART_FLEE_COORD", 0x94587F17E9C365D5 },
{  "TASK_SMART_FLEE_PED", 0x22B0D0E37CCB840D },
{  "TASK_SHOCKING_EVENT_REACT", 0x452419CBD838065B },
{  "TASK_WANDER_IN_AREA", 0xE054346CA3A0F315 },
{  "TASK_WANDER_STANDARD", 0xBB9CE077274F6A1B },
{  "TASK_PLANT_BOMB", 0x965FEC691D55E9BF },
{  "TASK_FOLLOW_NAV_MESH_TO_COORD", 0x15D3A79D4E44B913 },
{  "TASK_FOLLOW_NAV_MESH_TO_COORD_ADVANCED", 0x17F58B88D085DBAC },
{  "SET_PED_PATH_CAN_USE_CLIMBOVERS", 0x8E06A6FE76C9EFF4 },
{  "SET_PED_PATH_CAN_USE_LADDERS", 0x77A5B103C87F476E },
{  "SET_PED_PATH_CAN_DROP_FROM_HEIGHT", 0xE361C5C71C431A4F },
{  "SET_PED_PATH_MAY_ENTER_WATER", 0xF35425A4204367EC },
{  "SET_PED_PATH_PREFER_TO_AVOID_WATER", 0x38FE1EC73743793C },
{  "SET_PED_PATH_AVOID_FIRE", 0x4455517B28441E60 },
{  "TASK_GO_TO_COORD_ANY_MEANS", 0x5BC448CB78FA3E88 },
{  "TASK_GO_TO_COORD_ANY_MEANS_EXTRA_PARAMS", 0x1DD45F9ECFDB1BC9 },
{  "TASK_GO_TO_COORD_ANY_MEANS_EXTRA_PARAMS_WITH_CRUISE_SPEED", 0xB8ECD61F531A7B02 },
{  "TASK_PLAY_ANIM", 0xEA47FE3719165B94 },
{  "TASK_PLAY_ANIM_ADVANCED", 0x83CDB10EA29B370B },
{  "STOP_ANIM_TASK", 0x97FF36A1D40EA00A },
{  "TASK_SCRIPTED_ANIMATION", 0x126EF75F1E17ABE5 },
{  "PLAY_ENTITY_SCRIPTED_ANIM", 0x77A1EEC547E7FCF1 },
{  "STOP_ANIM_PLAYBACK", 0xEE08C992D238C5D1 },
{  "SET_ANIM_RATE", 0x032D49C5E359C847 },
{  "TASK_LOOK_AT_COORD", 0x6FA46612594F7973 },
{  "TASK_LOOK_AT_ENTITY", 0x69F4BE8C8CC4796C },
{  "TASK_CLEAR_LOOK_AT", 0x0F804F1DB19B9689 },
{  "OPEN_SEQUENCE_TASK", 0xE8854A4326B9E12B },
{  "CLOSE_SEQUENCE_TASK", 0x39E72BC99E6360CB },
{  "TASK_PERFORM_SEQUENCE", 0x5ABA3986D90D8A3B },
{  "CLEAR_SEQUENCE_TASK", 0x3841422E9C488D8C },
{  "SET_SEQUENCE_TO_REPEAT", 0x58C70CF3A41E4AE7 },
{  "GET_SEQUENCE_PROGRESS", 0x00A9010CFE1E3533 },
{  "GET_IS_TASK_ACTIVE", 0xB0760331C7AA4155 },
{  "GET_SCRIPT_TASK_STATUS", 0x77F1BEB8863288D5 },
{  "GET_ACTIVE_VEHICLE_MISSION_TYPE", 0x534AEBA6E5ED4CAB },
{  "TASK_LEAVE_ANY_VEHICLE", 0x504D54DF3F6F2247 },
{  "TASK_AIM_GUN_AT_ENTITY", 0x9B53BB6E8943AF53 },
{  "TASK_TURN_PED_TO_FACE_ENTITY", 0x5AD23D40115353AC },
{  "TASK_AIM_GUN_AT_COORD", 0x6671F3EEC681BDA1 },
{  "TASK_SHOOT_AT_COORD", 0x46A6CC01E0826106 },
{  "TASK_SHUFFLE_TO_NEXT_VEHICLE_SEAT", 0x7AA80209BDA643EB },
{  "CLEAR_PED_TASKS", 0xE1EF3C1216AFF2CD },
{  "CLEAR_PED_SECONDARY_TASK", 0x176CECF6F920D707 },
{  "TASK_EVERYONE_LEAVE_VEHICLE", 0x7F93691AB4B92272 },
{  "TASK_GOTO_ENTITY_OFFSET", 0xE39B4FF4FDEBDE27 },
{  "TASK_GOTO_ENTITY_OFFSET_XY", 0x338E7EF52B6095A9 },
{  "TASK_TURN_PED_TO_FACE_COORD", 0x1DDA930A0AC38571 },
{  "TASK_VEHICLE_TEMP_ACTION", 0xC429DCEEB339E129 },
{  "TASK_VEHICLE_MISSION", 0x659427E0EF36BCDE },
{  "TASK_VEHICLE_MISSION_PED_TARGET", 0x9454528DF15D657A },
{  "TASK_VEHICLE_ESCORT", 0x0FA6E4B75F302400 },
{  "TASK_BOAT_MISSION", 0x15C86013127CE63F },
{  "TASK_DRIVE_BY", 0x2F8AF0E82773A171 },
{  "SET_DRIVEBY_TASK_TARGET", 0xE5B302114D8162EE },
{  "CLEAR_DRIVEBY_TASK_UNDERNEATH_DRIVING_TASK", 0xC35B5CDB2824CF69 },
{  "IS_DRIVEBY_TASK_UNDERNEATH_DRIVING_TASK", 0x8785E6E40C7A8818 },
{  "IS_MOUNTED_WEAPON_TASK_UNDERNEATH_DRIVING_TASK", 0xA320EF046186FA3B },
{  "TASK_WARP_PED_INTO_VEHICLE", 0x9A7D091411C5F684 },
{  "TASK_SHOOT_AT_ENTITY", 0x08DA95E8298AE772 },
{  "TASK_CLIMB", 0x89D9FCC2435112F1 },
{  "TASK_CLIMB_LADDER", 0xB6C987F9285A3814 },
{  "CLEAR_PED_TASKS_IMMEDIATELY", 0xAAA34F8A7CB32098 },
{  "TASK_PERFORM_SEQUENCE_FROM_PROGRESS", 0x89221B16730234F0 },
{  "SET_PED_DESIRED_MOVE_BLEND_RATIO", 0x1E982AC8716912C5 },
{  "GET_PED_DESIRED_MOVE_BLEND_RATIO", 0x8517D4A6CA8513ED },
{  "TASK_GOTO_ENTITY_AIMING", 0xA9DA48FAB8A76C12 },
{  "TASK_SET_SPHERE_DEFENSIVE_AREA", 0x933C06518B52A9A4 },
{  "TASK_CLEAR_DEFENSIVE_AREA", 0x95A6C46A31D1917D },
{  "TASK_PED_SLIDE_TO_COORD", 0xD04FE6765D990A06 },
{  "ADD_COVER_POINT", 0xD5C12A75C7B9497F },
{  "REMOVE_COVER_POINT", 0xAE287C923D891715 },
{  "DOES_SCRIPTED_COVER_POINT_EXIST_AT_COORDS", 0xA98B8E3C088E5A31 },
{  "GET_SCRIPTED_COVER_POINT_COORDS", 0x594A1028FC2A3E85 },
{  "TASK_COMBAT_PED", 0xF166E48407BAC484 },
{  "TASK_COMBAT_PED_TIMED", 0x944F30DCB7096BDE },
{  "TASK_SEEK_COVER_FROM_POS", 0x75AC2B60386D89F2 },
{  "TASK_SEEK_COVER_FROM_PED", 0x84D32B3BEC531324 },
{  "TASK_SEEK_COVER_TO_COVER_POINT", 0xD43D95C7A869447F },
{  "TASK_SEEK_COVER_TO_COORDS", 0x39246A6958EF072C },
{  "TASK_PUT_PED_DIRECTLY_INTO_COVER", 0x4172393E6BE1FECE },
{  "TASK_PUT_PED_DIRECTLY_INTO_MELEE", 0x1C6CD14A876FFE39 },
{  "TASK_GUARD_CURRENT_POSITION", 0x4A58A47A72E3FCB4 },
{  "TASK_GUARD_ASSIGNED_DEFENSIVE_AREA", 0xD2A207EEBDF9889B },
{  "TASK_STAND_GUARD", 0xAE032F8BBA959E90 },
{  "SET_DRIVE_TASK_CRUISE_SPEED", 0x5C9B84BD7D31D908 },
{  "SET_DRIVE_TASK_MAX_CRUISE_SPEED", 0x404A5AA9B9F0B746 },
{  "ADD_COVER_BLOCKING_AREA", 0x45C597097DD7CB81 },
{  "REMOVE_ALL_COVER_BLOCKING_AREAS", 0xDB6708C0B46F56D8 },
{  "TASK_USE_NEAREST_SCENARIO_TO_COORD_WARP", 0x58E2E0F23F6B76C3 },
{  "TASK_USE_NEAREST_SCENARIO_CHAIN_TO_COORD", 0x9FDA1B3D7E7028B3 },
{  "TASK_USE_NEAREST_SCENARIO_CHAIN_TO_COORD_WARP", 0x97A28E63F0BA5631 },
{  "DOES_SCENARIO_EXIST_IN_AREA", 0x5A59271FFADD33C1 },
{  "IS_SCENARIO_OCCUPIED", 0x788756D73AC2E07C },
{  "PED_HAS_USE_SCENARIO_TASK", 0x295E3CCEC879CCD7 },
{  "PLAY_ANIM_ON_RUNNING_SCENARIO", 0x748040460F8DF5DC },
{  "DOES_SCENARIO_GROUP_EXIST", 0xF9034C136C9E00D3 },
{  "IS_SCENARIO_GROUP_ENABLED", 0x367A09DED4E05B99 },
{  "SET_SCENARIO_GROUP_ENABLED", 0x02C8E5B49848664E },
{  "RESET_SCENARIO_GROUPS_ENABLED", 0xDD902D0349AFAD3A },
{  "IS_SCENARIO_TYPE_ENABLED", 0x3A815DB3EA088722 },
{  "SET_SCENARIO_TYPE_ENABLED", 0xEB47EC4E34FB7EE1 },
{  "RESET_SCENARIO_TYPES_ENABLED", 0x0D40EE2A7F2B2D6D },
{  "IS_PED_ACTIVE_IN_SCENARIO", 0xAA135F9482C82CC3 },
{  "TASK_COMBAT_HATED_TARGETS_IN_AREA", 0x4CF5F55DAC3280A0 },
{  "TASK_COMBAT_HATED_TARGETS_AROUND_PED", 0x7BF835BB9E2698C8 },
{  "TASK_COMBAT_HATED_TARGETS_AROUND_PED_TIMED", 0x2BBA30B854534A0C },
{  "TASK_THROW_PROJECTILE", 0x7285951DBF6B5A51 },
{  "TASK_SWAP_WEAPON", 0xA21C51255B205245 },
{  "TASK_RELOAD_WEAPON", 0x62D2916F56B9CD2D },
{  "IS_PED_GETTING_UP", 0x2A74E1D5F2F00EEC },
{  "IS_PED_IN_WRITHE", 0xDEB6D52126E7D640 },
{  "OPEN_PATROL_ROUTE", 0xA36BFB5EE89F3D82 },
{  "CLOSE_PATROL_ROUTE", 0xB043ECA801B8CBC1 },
{  "ADD_PATROL_ROUTE_NODE", 0x8EDF950167586B7C },
{  "ADD_PATROL_ROUTE_LINK", 0x23083260DEC3A551 },
{  "CREATE_PATROL_ROUTE", 0xAF8A443CCC8018DC },
{  "DELETE_PATROL_ROUTE", 0x7767DD9D65E91319 },
{  "TASK_PATROL", 0xBDA5DF49D080FE4E },
{  "TASK_STAY_IN_COVER", 0xE5DA8615A6180789 },
{  "TASK_VEHICLE_SHOOT_AT_PED", 0x10AB107B887214D8 },
{  "TASK_VEHICLE_AIM_AT_PED", 0xE41885592B08B097 },
{  "TASK_VEHICLE_SHOOT_AT_COORD", 0x5190796ED39C9B6D },
{  "TASK_VEHICLE_AIM_AT_COORD", 0x447C1E9EF844BC0F },
{  "TASK_VEHICLE_GOTO_NAVMESH", 0x195AEEB13CEFE2EE },
{  "TASK_GO_TO_COORD_WHILE_AIMING_AT_COORD", 0x11315AB3385B8AC0 },
{  "TASK_GO_TO_COORD_WHILE_AIMING_AT_ENTITY", 0xB2A16444EAD9AE47 },
{  "TASK_GO_TO_ENTITY_WHILE_AIMING_AT_ENTITY", 0x97465886D35210E9 },
{  "TASK_GO_TO_COORD_AND_AIM_AT_HATED_ENTITIES_NEAR_COORD", 0xA55547801EB331FC },
{  "SET_HIGH_FALL_TASK", 0x8C825BDC7741D37C },
{  "REQUEST_WAYPOINT_RECORDING", 0x9EEFB62EB27B5792 },
{  "GET_IS_WAYPOINT_RECORDING_LOADED", 0xCB4E8BE8A0063C5D },
{  "REMOVE_WAYPOINT_RECORDING", 0xFF1B8B4AA1C25DC8 },
{  "WAYPOINT_RECORDING_GET_NUM_POINTS", 0x5343532C01A07234 },
{  "WAYPOINT_RECORDING_GET_COORD", 0x2FB897405C90B361 },
{  "WAYPOINT_RECORDING_GET_SPEED_AT_POINT", 0x005622AEBC33ACA9 },
{  "WAYPOINT_RECORDING_GET_CLOSEST_WAYPOINT", 0xB629A298081F876F },
{  "TASK_FOLLOW_WAYPOINT_RECORDING", 0x0759591819534F7B },
{  "IS_WAYPOINT_PLAYBACK_GOING_ON_FOR_PED", 0xE03B3F2D3DC59B64 },
{  "GET_PED_WAYPOINT_PROGRESS", 0x2720AAA75001E094 },
{  "GET_PED_WAYPOINT_DISTANCE", 0xE6A877C64CAF1BC5 },
{  "SET_PED_WAYPOINT_ROUTE_OFFSET", 0xED98E10B0AFCE4B4 },
{  "GET_WAYPOINT_DISTANCE_ALONG_ROUTE", 0xA5B769058763E497 },
{  "WAYPOINT_PLAYBACK_GET_IS_PAUSED", 0x701375A7D43F01CB },
{  "WAYPOINT_PLAYBACK_PAUSE", 0x0F342546AA06FED5 },
{  "WAYPOINT_PLAYBACK_RESUME", 0x244F70C84C547D2D },
{  "WAYPOINT_PLAYBACK_OVERRIDE_SPEED", 0x7D7D2B47FA788E85 },
{  "WAYPOINT_PLAYBACK_USE_DEFAULT_SPEED", 0x6599D834B12D0800 },
{  "USE_WAYPOINT_RECORDING_AS_ASSISTED_MOVEMENT_ROUTE", 0x5A353B8E6B1095B5 },
{  "WAYPOINT_PLAYBACK_START_AIMING_AT_PED", 0x20E330937C399D29 },
{  "WAYPOINT_PLAYBACK_START_AIMING_AT_COORD", 0x8968400D900ED8B3 },
{  "WAYPOINT_PLAYBACK_START_SHOOTING_AT_PED", 0xE70BA7B90F8390DC },
{  "WAYPOINT_PLAYBACK_START_SHOOTING_AT_COORD", 0x057A25CFCC9DB671 },
{  "WAYPOINT_PLAYBACK_STOP_AIMING_OR_SHOOTING", 0x47EFA040EBB8E2EA },
{  "ASSISTED_MOVEMENT_REMOVE_ROUTE", 0x3548536485DD792B },
{  "ASSISTED_MOVEMENT_IS_ROUTE_LOADED", 0x60F9A4393A21F741 },
{  "ASSISTED_MOVEMENT_SET_ROUTE_PROPERTIES", 0xD5002D78B7162E1B },
{  "TASK_VEHICLE_FOLLOW_WAYPOINT_RECORDING", 0x3123FAA6DB1CF7ED },
{  "IS_WAYPOINT_PLAYBACK_GOING_ON_FOR_VEHICLE", 0xF5134943EA29868C },
{  "GET_VEHICLE_WAYPOINT_PROGRESS", 0x9824CFF8FC66E159 },
{  "GET_VEHICLE_WAYPOINT_TARGET_POINT", 0x416B62AC8B9E5BBD },
{  "VEHICLE_WAYPOINT_PLAYBACK_PAUSE", 0x8A4E6AC373666BC5 },
{  "VEHICLE_WAYPOINT_PLAYBACK_RESUME", 0xDC04FCAA7839D492 },
{  "VEHICLE_WAYPOINT_PLAYBACK_USE_DEFAULT_SPEED", 0x5CEB25A7D2848963 },
{  "VEHICLE_WAYPOINT_PLAYBACK_OVERRIDE_SPEED", 0x121F0593E0A431D7 },
{  "TASK_SET_BLOCKING_OF_NON_TEMPORARY_EVENTS", 0x90D2156198831D69 },
{  "TASK_FORCE_MOTION_STATE", 0x4F056E1AFFEF17AB },
{  "TASK_MOVE_NETWORK_BY_NAME", 0x2D537BA194896636 },
{  "IS_TASK_MOVE_NETWORK_ACTIVE", 0x921CE12C489C4C41 },
{  "IS_TASK_MOVE_NETWORK_READY_FOR_TRANSITION", 0x30ED88D5E0C56A37 },
{  "REQUEST_TASK_MOVE_NETWORK_STATE_TRANSITION", 0xD01015C7316AE176 },
{  "GET_TASK_MOVE_NETWORK_STATE", 0x717E4D1F2048376D },
{  "SET_TASK_MOVE_NETWORK_SIGNAL_FLOAT", 0xD5BB4025AE449A4E },
{  "SET_TASK_MOVE_NETWORK_SIGNAL_BOOL", 0xB0A6CFD2C69C1088 },
{  "GET_TASK_MOVE_NETWORK_EVENT", 0xB4F47213DF45A64C },
{  "IS_MOVE_BLEND_RATIO_STILL", 0x349CE7B56DAFD95C },
{  "IS_MOVE_BLEND_RATIO_WALKING", 0xF133BBBE91E1691F },
{  "IS_MOVE_BLEND_RATIO_RUNNING", 0xD4D8636C0199A939 },
{  "IS_MOVE_BLEND_RATIO_SPRINTING", 0x24A2AD74FA9814E2 },
{  "IS_PED_STILL", 0xAC29253EEF8F0180 },
{  "IS_PED_WALKING", 0xDE4C184B2B9B071A },
{  "IS_PED_RUNNING", 0xC5286FFC176F28A2 },
{  "IS_PED_SPRINTING", 0x57E457CD2C0FC168 },
{  "TASK_ARREST_PED", 0xF3B9A78A178572B1 },
{  "IS_PED_BEING_ARRESTED", 0x90A09F3A45FED688 },
{  "UNCUFF_PED", 0x67406F2C8F87FC4F },
{  "IS_PED_CUFFED", 0x74E559B3BC910685 },
// Category: _NAMESPACE67
// Category: TXD
{  "REQUEST_STREAMED_TEXTURE_DICT", 0xC1BA29DF5631B0F8 },
{  "HAS_STREAMED_TEXTURE_DICT_LOADED", 0x54D6900929CCF162 },
{  "SET_STREAMED_TEXTURE_DICT_AS_NO_LONGER_NEEDED", 0x4ACA10A91F66F1E2 },
// Category: UIAPPS
{  "_IS_APP_ACTIVE", 0x25B7A0206BDFAC76 },
{  "_IS_APP_RUNNING", 0x4E511D093A86AD49 },
{  "_IS_ANY_APP_RUNNING", 0xDB30BEC7A7A5CBD3 },
{  "_CLOSE_APP_BY_HASH", 0x2FF10C9C3F92277E },
{  "_CLOSE_APP_BY_HASH_IMMEDIATE", 0x04428420A248A354 },
{  "_CLOSE_APP", 0x3015635426D1B17C },
{  "_CLOSE_ALL_APPS", 0xAD7B70F7230C5A12 },
{  "_LAUNCH_APP_BY_HASH_WITH_ENTRY", 0xC1BCF31E975B3195 },
{  "_LAUNCH_APP_BY_HASH", 0xC8FC7F4E4CF4F581 },
{  "_LAUNCH_APP_WITH_ENTRY", 0x7B2027BAC5C8EC89 },
{  "_CAN_LAUNCH_APP_BY_HASH", 0xE555EC27D65EDE80 },
{  "_UIAPP_REQUEST_TRANSITION_BY_HASH", 0x7689CD255655BFD7 },
// Category: UIEVENTS
{  "_IS_EVENT_PENDING", 0x67ED5A7963F2F722 },
{  "_PEEK_EVENT", 0x90237103F27F7937 },
{  "_POP_EVENT", 0x8E8A2369F48EC839 },
// Category: _NAMESPACE71
// Category: UILOG
{  "_LOG_IS_ENTRY_REGISTERED", 0xB8188CCF52202475 },
{  "_LOG_ADD_ENTRY_HASH", 0x69D5479982355D8F },
{  "_LOG_SET_ENTRY_ICON_TEXTURE", 0x6965469934958D8F },
{  "_LOG_SET_ENTRY_BRIEF_TEXTURE", 0x69684D9936958D8F },
{  "_LOG_UPDATE_ENTRY_SUBHEADER", 0x80D6524190258C3E },
{  "_LOG_SET_ENTRY_PINNED", 0x72A5CD214B342568 },
{  "_MARK_MISSION_COMPLETED", 0xDE31D66D1E54C471 },
{  "_MARK_ENTRY_AVAILABILITY", 0x13E8D7DD08543482 },
{  "_MARK_ALL_ENTRIES_AVAILABILITY", 0x3920574CF0A2B7B6 },
{  "_REMOVE_LOG_ENTRY", 0xD594A19BE09A75C6 },
{  "_CLEAR_LOG_ENTRIES", 0xB95B4EA6B1EDF035 },
{  "_LOG_ADD_OR_UPDATE_OBJECTIVE", 0xB43163388484CC87 },
{  "_LOG_SET_CACHED_OBJECTIVE", 0xFA233F8FE190514C },
{  "_LOG_CLEAR_CACHED_OBJECTIVE", 0xDFF0D417277B41F8 },
{  "_LOG_HAS_DISPLAYED_CACHED_OBJECTIVE", 0xCC48FFBB45B54F71 },
{  "_LOG_CLEAR_HAS_DISPLAYED_CACHED_OBJECTIVE", 0xA3108D6981A5CADB },
{  "_LOG_SET_HAS_DISPLAYED_CACHED_OBJECTIVE", 0xA3108D6981A5CADC },
{  "_LOG_GET_CACHED_OBJECTIVE", 0x15A4461BEB788096 },
{  "_LOG_PRINT_CACHED_OBJECTIVE", 0xE9990552DEC71600 },
{  "_SET_PENDING_DETAILS_ID", 0x136A027CF37B0A4F },
{  "_ADD_TOTAL_TAKE_ENTRY", 0x60C59968E8E87E6B },
{  "_SET_TOTAL_TAKE_SUMMARY", 0xD106B211EF1B8F04 },
{  "_POST_NOTIFICATION", 0x49E58FE6EF40B987 },
// Category: _NAMESPACE73
// Category: UITUTORIAL
{  "_GET_IS_THREAT_INDICATOR_CAPABLE_RADAR_SHOWN", 0x2CC24A2A7A1489C4 },
{  "_GET_IS_THREAT_INDICATOR_ON", 0xFC2E0A5E9ED4E1B4 },
{  "_SET_RPG_ICON_VISIBILITY", 0xC116E6DF68DCE667 },
// Category: UISTATEMACHINE
{  "_REQUEST_FLOW_BLOCK", 0xC0081B34E395CE48 },
{  "_RELEASE_FLOW_BLOCK", 0xF320A77DD5F781DF },
{  "_IS_FLOW_BLOCK_LOADED", 0x10A93C057B6BD944 },
{  "_ENTER_FLOW_BLOCK", 0x3B7519720C9DCB45 },
{  "_DOES_STATE_MACHINE_EXIST", 0x5D15569C0FEBF757 },
{  "_STATE_MACHINE_CREATE", 0x4C6F2C4B7A03A266 },
{  "_STATE_MACHINE_REQUEST_TRANSITION", 0x7EA9C3547E80350E },
{  "_STATE_MACHINE_DESTROY", 0x4EB122210A90E2D8 },
// Category: _NAMESPACE76
// Category: UNLOCK
{  "_UNLOCK_IS_UNLOCKED", 0xC4B660C7B6040E75 },
{  "_UNLOCK_SET_UNLOCKED", 0x1B7C5ADA8A6910A0 },
{  "_UNLOCK_IS_VISIBLE", 0x8588A14B75AF096B },
{  "_UNLOCK_SET_VISIBLE", 0x46B901A8ECDB5A61 },
{  "_UNLOCK_IS_NEW", 0x644166BA7AA49DEA },
{  "_UNLOCK_SET_NEW", 0xA6D79C7AEF870A99 },
// Category: VEHICLE
{  "CREATE_VEHICLE", 0xAF35D0D2583051B0 },
{  "DELETE_VEHICLE", 0xE20A909D8C4A70F8 },
{  "SET_VEHICLE_ALLOW_NO_PASSENGERS_LOCKON", 0xECB9E9BC887E8060 },
{  "IS_VEHICLE_MODEL", 0x0045A54EC7A22455 },
{  "SET_ALL_VEHICLE_GENERATORS_ACTIVE_IN_AREA", 0xBBB134FB9D50C0CC },
{  "SET_ALL_VEHICLE_GENERATORS_ACTIVE", 0x3D596E6E88A02C24 },
{  "SET_VEHICLE_ON_GROUND_PROPERLY", 0x7263332501E07F52 },
{  "IS_VEHICLE_STOPPED", 0x78C3311A73135241 },
{  "GET_VEHICLE_NUMBER_OF_PASSENGERS", 0x59F3F16577CD79B2 },
{  "GET_VEHICLE_MAX_NUMBER_OF_PASSENGERS", 0xA9C55F1C15E62E06 },
{  "GET_VEHICLE_MODEL_NUMBER_OF_SEATS", 0x9A578736FF3A17C3 },
{  "IS_SEAT_WARP_ONLY", 0x7892685BF6D9775E },
{  "SET_VEHICLE_DENSITY_MULTIPLIER_THIS_FRAME", 0x606374EBFC27B133 },
{  "SET_RANDOM_VEHICLE_DENSITY_MULTIPLIER_THIS_FRAME", 0x1F91D44490E1EA0C },
{  "SET_PARKED_VEHICLE_DENSITY_MULTIPLIER_THIS_FRAME", 0xFEDFA97638D61D4A },
{  "SET_DISABLE_RANDOM_TRAINS_THIS_FRAME", 0xD4288603E8766FF7 },
{  "SET_VEHICLE_DOORS_LOCKED", 0x96F78A6A075D55D9 },
{  "_SET_VEHICLE_DOOR_DESTROY_TYPE", 0xA9F1D75195CC40F6 },
{  "SET_VEHICLE_DOORS_LOCKED_FOR_PLAYER", 0x359A8EA1FB8D6F0F },
{  "GET_VEHICLE_DOORS_LOCKED_FOR_PLAYER", 0xFA2CDDFEB8BC898B },
{  "SET_VEHICLE_DOORS_LOCKED_FOR_ALL_PLAYERS", 0x2381977DA948F8DC },
{  "SET_VEHICLE_DOORS_LOCKED_FOR_TEAM", 0xE712BC978770F105 },
{  "EXPLODE_VEHICLE", 0x75DCED9EEC5769D7 },
{  "SET_BOAT_ANCHOR", 0xAEAB044F05B92659 },
{  "_CAN_BOAT_BE_ANCHORED", 0xC075176CFB8B4128 },
{  "_SET_BOAT_FROZEN_WHEN_ANCHORED", 0x286771F3059A37A7 },
{  "_SET_BOAT_ANCHOR_BUOYANCY_COEFFICIENT", 0xE3261532550D6A9F },
{  "_SET_BOAT_EXPLODES_ON_WRECKED_ACTION", 0x62A6D317A011EA1D },
{  "SET_VEHICLE_STRONG", 0xAB315515C9F8803D },
{  "IS_VEHICLE_SEAT_FREE", 0xE052C1B1CAA4ECE4 },
{  "GET_PED_IN_VEHICLE_SEAT", 0xBB40DD2270B65366 },
{  "GET_LAST_PED_IN_VEHICLE_SEAT", 0x74583B19FEEAFDA7 },
{  "SET_VEHICLE_FORWARD_SPEED", 0xF9F92AF49F12F6E7 },
{  "BRING_VEHICLE_TO_HALT", 0x260BE8F09E326A20 },
{  "SET_VEHICLE_DOORS_SHUT", 0xA4FFCD645B11F25A },
{  "SET_VEHICLE_TYRES_CAN_BURST", 0xEBD0A4E935106FE5 },
{  "SET_VEHICLE_WHEELS_CAN_BREAK", 0x839137C40275FB77 },
{  "SET_VEHICLE_DOOR_OPEN", 0x550CE392A4672412 },
{  "REMOVE_VEHICLE_WINDOW", 0x745F15A215F2DDF1 },
{  "SET_VEHICLE_LIGHTS", 0x629F0A0E952CAE7D },
{  "SET_RANDOM_TRAINS", 0x1156C6EE7E82A98A },
{  "DELETE_ALL_TRAINS", 0xA3120A1385F17FF7 },
{  "SET_TRAIN_SPEED", 0xDFBA6BBFF7CCAFBB },
{  "SET_TRAIN_CRUISE_SPEED", 0x01021EB2E96B793C },
{  "SET_RANDOM_BOATS", 0xF44D446D4E36DB87 },
{  "REQUEST_VEHICLE_RECORDING", 0xC474CF16EDA45DC9 },
{  "HAS_VEHICLE_RECORDING_BEEN_LOADED", 0xBA9325BE372AB6EA },
{  "REMOVE_VEHICLE_RECORDING", 0x139E35755418F6AA },
{  "GET_POSITION_OF_VEHICLE_RECORDING_AT_TIME", 0x1A00961A1BE94E5E },
{  "GET_ROTATION_OF_VEHICLE_RECORDING_AT_TIME", 0x61787DD28B8CC0D5 },
{  "GET_TIME_POSITION_IN_RECORDING", 0x233B51C7913FA031 },
{  "START_PLAYBACK_RECORDED_VEHICLE", 0x4932B84E3276508E },
{  "STOP_PLAYBACK_RECORDED_VEHICLE", 0xBF9B4D6267E8C26D },
{  "IS_PLAYBACK_GOING_ON_FOR_VEHICLE", 0x02774B3A9034278F },
{  "IS_PLAYBACK_USING_AI_GOING_ON_FOR_VEHICLE", 0x5A7472606EC5B7C1 },
{  "SET_PLAYBACK_SPEED", 0xD78084EED4CD94C6 },
{  "SKIP_TIME_IN_PLAYBACK_RECORDED_VEHICLE", 0x5F5E6379C59EFC56 },
{  "GET_CLOSEST_VEHICLE", 0x52F45D033645181B },
{  "GET_TRAIN_CARRIAGE", 0xD0FB093A4CDB932C },
{  "DELETE_MISSION_TRAIN", 0x0D3630FB07E8B570 },
{  "SET_MISSION_TRAIN_AS_NO_LONGER_NEEDED", 0xBBE7648349B49BE8 },
{  "SET_MISSION_TRAIN_COORDS", 0x7632755962AB9922 },
{  "IS_THIS_MODEL_A_BOAT", 0x799CFC7C5B743B15 },
{  "IS_THIS_MODEL_A_TRAIN", 0xFC08C8F8C1EDF174 },
{  "SET_VEHICLE_CAN_BE_TARGETTED", 0x05254BA0B44ADC16 },
{  "SET_VEHICLE_CAN_BE_VISIBLY_DAMAGED", 0x4BF8131AE811541C },
{  "SET_VEHICLE_DIRT_LEVEL", 0x758C3460EE915D0A },
{  "IS_VEHICLE_DOOR_FULLY_OPEN", 0x7AE191143C7A9107 },
{  "SET_VEHICLE_ENGINE_ON", 0xB64CFA14CB9A2E78 },
{  "SET_VEHICLE_UNDRIVEABLE", 0x6E884BAB713A2A94 },
{  "SET_VEHICLE_PROVIDES_COVER", 0x652712478F1721F4 },
{  "SET_VEHICLE_DOOR_CONTROL", 0xD57F10EBBA814ECF },
{  "SET_VEHICLE_DOOR_LATCHED", 0x06F8A202EB312A3C },
{  "SET_VEHICLE_DOOR_SHUT", 0x6A3C24B91FD0EA09 },
{  "SET_VEHICLE_DOOR_BROKEN", 0x9666CF20A1C6D780 },
{  "SET_VEHICLE_CAN_BREAK", 0xC5ED9D59B4646611 },
{  "SET_VEHICLE_IS_CONSIDERED_BY_PLAYER", 0x54800D386C5825E5 },
{  "GET_VEHICLE_DOOR_LOCK_STATUS", 0xC867FD144F2469D3 },
{  "_SET_VEHICLE_DOOR_CAN_BREAK", 0x081FB9D6422F804C },
{  "IS_VEHICLE_ON_ALL_WHEELS", 0x0D5D119529654EE0 },
{  "SET_VEHICLE_FIXED", 0x79811282A9D1AE56 },
{  "SET_DISABLE_VEHICLE_PETROL_TANK_FIRES", 0xB70986AB19B04AFF },
{  "SET_DISABLE_VEHICLE_PETROL_TANK_DAMAGE", 0x5795FBE7A2001C14 },
{  "REMOVE_VEHICLES_FROM_GENERATORS_IN_AREA", 0xC619A44639BC0CB4 },
{  "SET_VEHICLE_STEER_BIAS", 0x84DAAE11E9EE4FC3 },
{  "IS_VEHICLE_EXTRA_TURNED_ON", 0xFA9A55D9C4351625 },
{  "SET_VEHICLE_EXTRA", 0xBB6F89150BC9D16B },
{  "DOES_EXTRA_EXIST", 0xAF5E7E9A7620FFB5 },
{  "SET_VEHICLE_DAMAGE", 0x1D7678F81452BB41 },
{  "GET_VEHICLE_ENGINE_HEALTH", 0x90DBFFAC43B22081 },
{  "SET_VEHICLE_ENGINE_HEALTH", 0x8BDC5B998B4654EF },
{  "GET_VEHICLE_PETROL_TANK_HEALTH", 0x1E5A9B356D5098BE },
{  "SET_VEHICLE_PETROL_TANK_HEALTH", 0x6AB2918EE3BEC94C },
{  "IS_VEHICLE_STUCK_TIMER_UP", 0x1ABA9753939503C5 },
{  "RESET_VEHICLE_STUCK_TIMER", 0x23298B468F7D88B6 },
{  "IS_VEHICLE_DRIVEABLE", 0xB86D29B10F627379 },
{  "SET_VEHICLE_HAS_BEEN_OWNED_BY_PLAYER", 0xBB5A3FA8ED3979C5 },
{  "START_VEHICLE_HORN", 0xB4E3BFC39CA16057 },
{  "SET_VEHICLE_HAS_STRONG_AXLES", 0x252253C8A45AA1FC },
{  "ARE_ANY_VEHICLE_SEATS_FREE", 0xA0A424505A1B6429 },
{  "SET_VEHICLE_EXPLODES_ON_HIGH_EXPLOSION_DAMAGE", 0xA402939C6761E1A3 },
{  "IS_ANY_VEHICLE_NEAR_POINT", 0x5698BA4FD04D39C4 },
{  "REQUEST_VEHICLE_HIGH_DETAIL_MODEL", 0x84B81EF78BD22357 },
{  "REQUEST_VEHICLE_ASSET", 0x81A15811460FAB3A },
{  "HAS_VEHICLE_ASSET_LOADED", 0xB935F3154BC913C8 },
{  "REMOVE_VEHICLE_ASSET", 0x888A4E675B38F5AD },
{  "SET_VEHICLE_AUTOMATICALLY_ATTACHES", 0x501354951CD942DE },
{  "IS_VEHICLE_IN_BURNOUT", 0x3F5029A8FC060C48 },
{  "SET_VEHICLE_HANDBRAKE", 0x91BE51AEC4E99710 },
{  "GET_VEHICLE_TRAILER_VEHICLE", 0xCF867A239EC30741 },
{  "GET_VEHICLE_ESTIMATED_MAX_SPEED", 0xFE52F34491529F0B },
{  "_ADD_SPEED_ZONE_FOR_COORD", 0x4C221BAC54D735C3 },
{  "_REMOVE_SPEED_ZONE", 0xFE9AB3354ACE6C9C },
{  "CAN_SHUFFLE_SEAT", 0xF8B2D32A2231FD24 },
{  "MODIFY_VEHICLE_TOP_SPEED", 0x35AD938C74CACD6A },
{  "GET_LAST_DRIVEN_VEHICLE", 0xA94F3E0AB9695E19 },
{  "CLEAR_LAST_DRIVEN_VEHICLE", 0x0EFC5DC62E67609B },
{  "SET_VEHICLE_LOD_MULTIPLIER", 0x5F5E2B1B9EAECC0F },
{  "COPY_VEHICLE_DAMAGES", 0xDBC28A8C683CD80B },
{  "SET_VEHICLE_SHOOT_AT_TARGET", 0xB79BE78C665B3E6D },
{  "SET_FORCE_HD_VEHICLE", 0x373CB1283308BD7B },
{  "TRACK_VEHICLE_VISIBILITY", 0x1F3969B140DEE157 },
{  "IS_VEHICLE_VISIBLE", 0x424910CD5DE8C246 },
{  "SET_VEHICLE_INACTIVE_DURING_PLAYBACK", 0x4EA71B4C9DB3C3F1 },
{  "SET_VEHICLE_ENGINE_CAN_DEGRADE", 0x48E4C137A71C2688 },
{  "SET_VEHICLE_IS_STOLEN", 0x6C32FC81DFF25C9A },
{  "DISABLE_VEHICLE_WEAPON", 0x94B1E71B144356A5 },
{  "SET_VEHICLE_CAN_BE_USED_BY_FLEEING_PEDS", 0xE42952510F84AFDB },
{  "_SET_VEHICLE_JET_ENGINE_ON", 0x1549BA7FE83A2383 },
{  "SET_VEHICLE_WHEELS_CAN_BREAK_OFF_WHEN_BLOW_UP", 0xC462C79379ABBCB1 },
{  "SET_VEHICLE_EXCLUSIVE_DRIVER", 0x0893DAFBFA67110E },
{  "_SET_VEHICLE_EXCLUSIVE_DRIVER_2", 0xC6B9BF123B9463B6 },
{  "GET_VEHICLE_BODY_HEALTH", 0x42113B857E33C16E },
{  "SET_VEHICLE_BODY_HEALTH", 0x55CCAAE4F28C67A0 },
// Category: _NAMESPACE79
// Category: WATER
// Category: _NAMESPACE81
{  "GET_WATER_HEIGHT", 0xFCA8B23F28813F69 },
{  "GET_WATER_HEIGHT_NO_WAVES", 0xDCF3690AA262C03F },
{  "TEST_PROBE_AGAINST_ALL_WATER", 0x8974647ED222EA5F },
{  "TEST_VERTICAL_PROBE_AGAINST_ALL_WATER", 0x2B3451FA1E3142E2 },
{  "_REMOVE_CURRENT_RISE", 0x4BEF8DD75AF6C71C },
// Category: WEAPON
{  "SET_PED_AMMO", 0x14E56BC5B5DB6A19 },
{  "SET_PED_INFINITE_AMMO", 0x3EDCB0505123623B },
{  "GET_AMMO_IN_PED_WEAPON", 0x015A522136D7F951 },
{  "GET_MAX_AMMO_IN_CLIP", 0xA38DCFFCEA8962FA },
{  "GET_AMMO_IN_CLIP", 0x2E1202248937775C },
{  "SET_AMMO_IN_CLIP", 0xDCD2A934D65CB497 },
{  "GET_MAX_AMMO", 0xDC16122C7A20C933 },
{  "SET_PED_AMMO_BY_TYPE", 0x5FD1E1F011E76D7E },
{  "GET_PED_AMMO_BY_TYPE", 0x39D22031557946C1 },
{  "SET_PED_AMMO_TO_DROP", 0xA4EFEF9440A5B0EF },
{  "GET_PED_AMMO_TYPE_FROM_WEAPON", 0x7FEAD38B326B9F74 },
{  "GET_WEAPONTYPE_GROUP", 0xEDCA14CA5199FF25 },
{  "SET_CURRENT_PED_WEAPON", 0xADF692B254977C0C },
{  "GET_CURRENT_PED_WEAPON", 0x3A87E44BB9A01D54 },
{  "GET_CURRENT_PED_WEAPON_ENTITY_INDEX", 0x3B390A939AF0B5FC },
{  "GET_BEST_PED_WEAPON", 0x8483E98E8B888AE2 },
{  "SET_CURRENT_PED_VEHICLE_WEAPON", 0x75C55983C2C39DAA },
{  "GET_CURRENT_PED_VEHICLE_WEAPON", 0x1017582BCD3832DC },
{  "IS_WEAPON_VALID", 0x937C71165CF334B3 },
{  "HAS_PED_GOT_WEAPON", 0x8DECB02F88F428BC },
{  "IS_PED_WEAPON_READY_TO_SHOOT", 0xB80CA294F2F26749 },
{  "GIVE_DELAYED_WEAPON_TO_PED", 0xB282DC6EBD803C75 },
{  "REMOVE_ALL_PED_WEAPONS", 0xF25DF915FA38C5F3 },
{  "REMOVE_WEAPON_FROM_PED", 0x4899CB088EDF59B8 },
{  "HIDE_PED_WEAPON_FOR_SCRIPTED_CUTSCENE", 0x6F6981D2253C208F },
{  "SET_PED_CURRENT_WEAPON_VISIBLE", 0x0725A4CCFDED9A70 },
{  "SET_PED_DROPS_WEAPONS_WHEN_DEAD", 0x476AE72C1D19D1A8 },
{  "GET_PED_LAST_WEAPON_IMPACT_COORD", 0x6C4D0409BA1A2BC2 },
{  "SET_PED_DROPS_INVENTORY_WEAPON", 0x208A1888007FC0E6 },
{  "REMOVE_WEAPON_COMPONENT_FROM_WEAPON_OBJECT", 0xF7D82B0D66777611 },
{  "HAS_WEAPON_GOT_WEAPON_COMPONENT", 0x76A18844E743BF91 },
{  "DRAW_TEXT_", 0xD79334A4BB99BAD1 },
{  "_GET_WAYPOINT_COORDS", 0x29B30D07C3F7873B },
{  "SET_PED_STAMINA", 0x675680D089BFA21F },
{  "SET_PED_VISIBLE", 0x283978A15512B2FE },
{  "_SET_TEXT_COLOR", 0x50A41AD966910F03 }

};
