/*
 * Adapted from The MIT License (MIT)
 *
 * Copyright (c) 2018-2020 DaPorkchop_
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software
 * is furnished to do so, subject to the following conditions:
 *
 * Any persons and/or organizations using this software must include the above copyright notice and this permission notice,
 * provide sufficient credit to the original authors of the project (IE: DaPorkchop_), as well as provide a link to the original project.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

package net.daporkchop.lib.http.util;

import lombok.Getter;
import lombok.experimental.Accessors;
import net.daporkchop.lib.common.misc.string.PStrings;
import net.daporkchop.lib.common.misc.string.PUnsafeStrings;
import net.daporkchop.lib.http.StatusCode;
import net.daporkchop.lib.http.util.exception.GenericHttpException;

/**
 * The official HTTP status codes.
 *
 * @author DaPorkchop_
 */
@Getter
@Accessors(fluent = true)
public enum StatusCodes implements StatusCode {
    // 1xx
    CONTINUE(100),
    SWITCHING_PROTOCOLS(101),
    PROCESSING(102),
    EARLY_HINTS(103),
    // 2xx
    OK(200),
    CREATED(201),
    ACCEPTED(202),
    NON_AUTHORITATIVE_INFORMATION(203, "Non-Authoritative Information"),
    NO_CONTENT(204),
    RESET_CONTENT(205),
    PARTIAL_CONTENT(206),
    MULTI_STATUS(207, "Multi-Status"),
    ALREADY_REPORTED(208),
    IM_USED(209),
    // 3xx
    MULTIPLE_CHOICES(300),
    MOVED_PERMANENTLY(301),
    MOVED_TEMPORARILY(302),
    SEE_OTHER(303),
    NOT_MODIFIED(304),
    TEMPORARY_REDIRECT(307),
    PERMANENT_REDIRECT(308),
    // 4xx
    BAD_REQUEST(400),
    UNAUTHORIZED(401),
    PAYMENT_REQUIRED(402),
    FORBIDDEN(403),
    NOT_FOUND(404, "The requested URL was not found on this server."),
    METHOD_NOT_ALLOWED(405),
    NOT_ACCEPTABLE(406),
    PROXY_AUTHENTICATION_REQUIRED(407),
    REQUEST_TIMEOUT(408),
    CONFLICT(409),
    GONE(410),
    LENGTH_REQUIRED(411),
    PRECONDITION_FAILED(412),
    PAYLOAD_TOO_LARGE(413),
    URI_TOO_LONG(414),
    UNSUPPORTED_MEDIA_TYPE(415),
    RANGE_NOT_SATISFIABLE(416),
    EXPECTATION_FAILED(417),
    IM_A_TEAPOT(418, "I'm a teapot", "See <a href=\"https://tools.ietf.org/html/rfc2324#section-2.3.2\">RFC2324, section 2.3.2</a>."),
    MISDIRECTED_REQUEST(421),
    UNPROCESSABLE_ENTITY(422),
    LOCKED(423),
    FAILED_DEPENDENCY(424),
    TOO_EARLY(425),
    UPGRADE_REQUIRED(426),
    PRECONDITION_REQUIRED(428),
    TOO_MANY_REQUESTS(429),
    REQUEST_HEADER_FIELDS_TOO_LARGE(431),
    UNAVAILABLE_FOR_LEGAL_REASONS(451),
    // 5xx
    INTERNAL_SERVER_ERROR(500),
    NOT_IMPLEMENTED(501),
    BAD_GATEWAY(502),
    SERVICE_UNAVAILABLE(503),
    GATEWAY_TIMEOUT(504),
    HTTP_VERSION_NOT_SUPPORTED(505),
    VARIANT_ALSO_NEGOTIATES(506),
    INSUFFICIENT_STORAGE(507),
    LOOP_DETECTED(508),
    NOT_EXTENDED(510),
    NETWORK_AUTHENTICATION_REQUIRED(511)
    ;

    private final String errorMessage;
    private final GenericHttpException exception;
    private final int code;

    StatusCodes(int code)    {
        this(code, null, null);
    }

    StatusCodes(int code, String errorMessage)    {
        this(code, null, errorMessage);
    }

    StatusCodes(int code, String name, String errorMessage)    {
        this.code = code;
        this.errorMessage = errorMessage;

        if (name == null)   {
            if (false) {
                char[] arr = PUnsafeStrings.unwrap(this.name()).clone();
                PUnsafeStrings.replace(arr, '_', ' ');
                PUnsafeStrings.titleFormat(arr);
                name = PUnsafeStrings.wrap(arr);
            } else {
                name = PStrings.split(this.name(), '_').titleFormat().join(' ');
            }
        }

        PUnsafeStrings.setEnumName(this, name);
        //this.msg = (name == null ? PStrings.split(this.name(), '_').titleFormat().join(' ') : name);

        this.exception = new GenericHttpException(this, false);
    }

    @Override
    public String toString() {
        return String.format("StatusCode(%d %s)", this.code, this.name());
    }
}
