#include "precomp.h"
#include "common_metapi.h"

/*!
 * @brief Allocates memory in the context of the supplied process.
 * @remark The 
 *     - TLV_TYPE_HANDLE          - The process handle to allocate memory within.
 *     - TLV_TYPE_LENGTH          - The amount of memory to allocate.
 *     - TLV_TYPE_ALLOCATION_TYPE - The type of memory to allocate.
 *     - TLV_TYPE_PROTECTION      - The protection flags to allocate the memory with.
 *     - TLV_TYPE_BASE_ADDRESS    - The address to allocate the memory at.
 */
DWORD request_sys_process_memory_allocate(Remote *remote, Packet *packet)
{
	Packet *response = met_api->packet.create_response(packet);
	HANDLE handle;
	LPVOID base;
	SIZE_T size;
	DWORD result = ERROR_SUCCESS;
	DWORD alloc, prot;

	// Snag the TLV values
	handle = (HANDLE)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_HANDLE);
	base = (LPVOID)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_BASE_ADDRESS);
	size = (SIZE_T)met_api->packet.get_tlv_value_uint(packet, TLV_TYPE_LENGTH);
	alloc = met_api->packet.get_tlv_value_uint(packet, TLV_TYPE_ALLOCATION_TYPE);
	prot = met_api->packet.get_tlv_value_uint(packet, TLV_TYPE_PROTECTION);

	// Allocate the memory
	if ((base = VirtualAllocEx(handle, base, size, alloc, prot)))
	{
		met_api->packet.add_tlv_qword(response, TLV_TYPE_BASE_ADDRESS, (QWORD)base);
	}
	else
	{
		result = GetLastError();
	}

	// Transmit the response
	met_api->packet.transmit_response(result, remote, response);

	return ERROR_SUCCESS;
}

/*
 * Free memory in the context of the supplied process
 *
 * req: TLV_TYPE_HANDLE       - The handle to free memory within.
 * req: TLV_TYPE_BASE_ADDRESS - The base address of the memory to free.
 * opt: TLV_TYPE_LENGTH       - The size, in bytes, to free.
 */
DWORD request_sys_process_memory_free(Remote *remote, Packet *packet)
{
	Packet *response = met_api->packet.create_response(packet);
	HANDLE handle;
	SIZE_T size;
	LPVOID base;
	DWORD result = ERROR_SUCCESS;

	handle = (HANDLE)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_HANDLE);
	base   = (LPVOID)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_BASE_ADDRESS);
	size   = met_api->packet.get_tlv_value_uint(packet, TLV_TYPE_LENGTH);

	// Free the memory
	if (!VirtualFreeEx(handle, base, size, MEM_RELEASE))
		result = GetLastError();

	// Transmit the response
	met_api->packet.transmit_response(result, remote, packet);

	return ERROR_SUCCESS;
}

/*
 * Read memory from the context of the supplied process at a given address for a
 * given length
 *
 * req: TLV_TYPE_HANDLE       - The handle of the process to read from.
 * req: TLV_TYPE_BASE_ADDRESS - The address to read from.
 * req: TLV_TYPE_LENGTH       - The number of bytes to read.
 */
DWORD request_sys_process_memory_read(Remote *remote, Packet *packet)
{
	Packet *response = met_api->packet.create_response(packet);
	LPVOID buffer = NULL;
	HANDLE handle;
	SIZE_T size;
	LPVOID base;
	SIZE_T bytesRead = 0;
	DWORD result = ERROR_SUCCESS;

	handle = (HANDLE)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_HANDLE);
	base   = (LPVOID)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_BASE_ADDRESS);
	size   = met_api->packet.get_tlv_value_uint(packet, TLV_TYPE_LENGTH);

	do
	{
		// No handle, base, or size supplied?
		if ((!handle) ||
		    (!base) ||
		    (!size))
		{
			result = ERROR_INVALID_PARAMETER;
			break;
		}

		// Allocate storage for to read into
		if (!(buffer = malloc(size)))
		{
			result = ERROR_NOT_ENOUGH_MEMORY;
			break;
		}

		// Read the memory from the process...break out on failure
		if ((!ReadProcessMemory(handle, base, buffer, size, &bytesRead)) &&
		    (GetLastError() != ERROR_PARTIAL_COPY))
		{
			result = GetLastError();
			break;
		}

		// Add the raw buffer to the response
		met_api->packet.add_tlv_raw(response, TLV_TYPE_PROCESS_MEMORY, buffer,
				(DWORD)bytesRead);

	} while (0);

	// Transmit the response
	met_api->packet.transmit_response(result, remote, response);

	// Free the temporary storage
	if (buffer)
		free(buffer);

	return ERROR_SUCCESS;
}

/*
 * Read memory from the context of the supplied process at a given address for a
 * given length
 *
 * req: TLV_TYPE_HANDLE         - The handle of the process to read from.
 * req: TLV_TYPE_BASE_ADDRESS   - The address to read from.
 * req: TLV_TYPE_PROCESS_MEMORY - The raw memory to write to the address.
 */
DWORD request_sys_process_memory_write(Remote *remote, Packet *packet)
{
	Packet *response = met_api->packet.create_response(packet);
	HANDLE handle;
	LPVOID base;
	DWORD result = ERROR_SUCCESS;
	size_t written = 0;
	Tlv data;

	handle = (HANDLE)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_HANDLE);
	base   = (LPVOID)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_BASE_ADDRESS);

	do
	{
		// Invalid handle, base, or data?
		if ((!handle) ||
		    (!base) ||
		    (met_api->packet.get_tlv(packet, TLV_TYPE_PROCESS_MEMORY, &data)) != ERROR_SUCCESS)
		{
			result = ERROR_INVALID_PARAMETER;
			break;
		}

		// Write the memory
		if ((!WriteProcessMemory(handle, base, data.buffer, data.header.length, 
				&written)) &&
		    (GetLastError() != ERROR_PARTIAL_COPY))
		{
			result = GetLastError();
			break;
		}

		// Set the number of bytes actually written on the response
		met_api->packet.add_tlv_uint(response, TLV_TYPE_LENGTH, (DWORD)written);

	} while (0);

	// Transmit the response
	met_api->packet.transmit_response(result, remote, response);

	return ERROR_SUCCESS;
}

/*
 * Queries an address region for its attributes, such as size and protection
 *
 * req: TLV_TYPE_HANDLE       - The process handle to operate on.
 * req: TLV_TYPE_BASE_ADDRESS - The address to query the attributes of.
 */
DWORD request_sys_process_memory_query(Remote *remote, Packet *packet)
{
	MEMORY_BASIC_INFORMATION info;
	Packet *response = met_api->packet.create_response(packet);
	HANDLE handle;
	LPVOID base;
	DWORD result = ERROR_SUCCESS;
	SIZE_T size = 0;

	handle = (HANDLE)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_HANDLE);
	base   = (LPVOID)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_BASE_ADDRESS);

	// Zero the info buffer
	memset(&info, 0, sizeof(info));

	do
	{
		// Validate parameters
		if (!handle)
		{
			result = ERROR_INVALID_PARAMETER;
			break;
		}

		// No bytes returned?  Suck.
		if (!(size = VirtualQueryEx(handle, base, &info, sizeof(info))))
		{
			result = GetLastError();
			break;
		}

		// Pass the parameters back to the requestor
		met_api->packet.add_tlv_qword(response, TLV_TYPE_BASE_ADDRESS,	(QWORD)info.BaseAddress);
		met_api->packet.add_tlv_qword(response, TLV_TYPE_ALLOC_BASE_ADDRESS, (QWORD)info.AllocationBase);
		met_api->packet.add_tlv_uint(response, TLV_TYPE_ALLOC_PROTECTION, info.AllocationProtect);
		met_api->packet.add_tlv_uint(response, TLV_TYPE_LENGTH, (DWORD)info.RegionSize);
		met_api->packet.add_tlv_uint(response, TLV_TYPE_MEMORY_STATE, (DWORD)info.State);
		met_api->packet.add_tlv_uint(response, TLV_TYPE_PROTECTION, info.Protect);
		met_api->packet.add_tlv_uint(response, TLV_TYPE_MEMORY_TYPE, info.Type);

	} while (0);

	// Transmit the response
	met_api->packet.transmit_response(result, remote, response);

	return ERROR_SUCCESS;
}

/*
 * Changes the protection flags on one or more pages
 *
 * req: TLV_TYPE_HANDLE       - The process handle to operate on
 * req: TLV_TYPE_BASE_ADDRESS - The base address to re-protect
 * req: TLV_TYPE_LENGTH       - The length of the region to re-protect
 * req: TLV_TYPE_PROTECTION   - The new protection mask
 */
DWORD request_sys_process_memory_protect(Remote *remote, Packet *packet)
{
	Packet *response = met_api->packet.create_response(packet);
	HANDLE handle;
	LPVOID base;
	SIZE_T size;
	DWORD prot, old;
	DWORD result = ERROR_SUCCESS;

	handle = (HANDLE)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_HANDLE);
	base   = (LPVOID)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_BASE_ADDRESS);
	size   = met_api->packet.get_tlv_value_uint(packet, TLV_TYPE_LENGTH);
	prot   = met_api->packet.get_tlv_value_uint(packet, TLV_TYPE_PROTECTION);

	do
	{
		// Validate parameters
		if ((!handle) ||
		    (!base) ||
		    (!size))
		{
			result = ERROR_INVALID_PARAMETER;
			break;
		}

		// Change the protection mask
		if (!VirtualProtectEx(handle, base, size, prot, &old))
		{
			result = GetLastError();
			break;
		}

		// Return the old protection mask to the requestor
		met_api->packet.add_tlv_uint(response, TLV_TYPE_PROTECTION, old);

	} while (0);

	// Transmit the response
	met_api->packet.transmit_response(result, remote, response);

	return ERROR_SUCCESS;
}

/*
 * Lock a region of memory in physical memory so that it cannot be swapped 
 * out.
 *
 * req: TLV_TYPE_BASE_ADDRESS - The base address to lock
 * req: TLV_TYPE_LENGTH       - The size of the region to lock
 */
DWORD request_sys_process_memory_lock(Remote *remote, Packet *packet)
{
	Packet *response = met_api->packet.create_response(packet);
	LPVOID base;
	SIZE_T size;
	DWORD result = ERROR_SUCCESS;

	base = (LPVOID)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_BASE_ADDRESS);
	size = met_api->packet.get_tlv_value_uint(packet, TLV_TYPE_LENGTH);

	if (!VirtualLock(base, size))
		result = GetLastError();

	// Transmit the response
	met_api->packet.transmit_response(result, remote, response);

	return ERROR_SUCCESS;
}

/*
 * Unlock a region so that it can be swapped to disk.
 *
 * req: TLV_TYPE_BASE_ADDRESS - The base address to lock
 * req: TLV_TYPE_LENGTH       - The size of the region to lock
 */
DWORD request_sys_process_memory_unlock(Remote *remote, Packet *packet)
{
	Packet *response = met_api->packet.create_response(packet);
	LPVOID base;
	SIZE_T size;
	DWORD result = ERROR_SUCCESS;

	base = (LPVOID)met_api->packet.get_tlv_value_qword(packet, TLV_TYPE_BASE_ADDRESS);
	size = met_api->packet.get_tlv_value_uint(packet, TLV_TYPE_LENGTH);

	if (!VirtualUnlock(base, size))
		result = GetLastError();

	// Transmit the response
	met_api->packet.transmit_response(result, remote, response);

	return ERROR_SUCCESS;
}
