/* MIT License
 * 
 * Copyright (c) [2020] [Ryan Wendland]
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "lv_conf.h"
#include "lvgl.h"
#include "menu.h"

static lv_indev_t *sdl_indev;
static lv_group_t *input_group;
static lv_obj_t *main_list;
static lv_obj_t *text_box;
static lv_obj_t *status_label;
static lv_obj_t *progress_bar;

static void main_list_cb(lv_obj_t *ta, lv_event_t e);

static const int XMARGIN = 15;
static const int YMARGIN = 15;

void create_menu(void)
{
    int screen_w = lv_obj_get_width(lv_scr_act());
    int screen_h = lv_obj_get_height(lv_scr_act());

    //Find previously registered input driver and create
    //an input group for it
    sdl_indev = lv_indev_get_next(NULL);
    if (sdl_indev == NULL)
        assert(0); //No input driver
    input_group = lv_group_create();
    lv_indev_set_group(sdl_indev, input_group);

    /* Create Xenium-Tools Title */
    LV_IMG_DECLARE(xenium_tools_header);
    lv_obj_t *header = lv_img_create(lv_scr_act(), NULL);
    lv_img_set_src(header, &xenium_tools_header);
    lv_obj_align(header, lv_scr_act(), LV_ALIGN_IN_TOP_MID, 0, YMARGIN);

    /* Create a List on the Active Screen for Menu Items */
    main_list = lv_list_create(lv_scr_act(), NULL);
    lv_obj_set_height(main_list, screen_h - 200);
    lv_obj_set_width(main_list, screen_w - 340);
    lv_obj_align(main_list, lv_scr_act(), LV_ALIGN_IN_TOP_LEFT, XMARGIN, YMARGIN + 65);
    lv_list_add_btn(main_list, LV_SYMBOL_REFRESH, "Rescan for Xenium");
    lv_list_add_btn(main_list, LV_SYMBOL_DOWNLOAD, "Dump Xenium flash memory");
    lv_list_add_btn(main_list, LV_SYMBOL_UPLOAD, "Write a raw 2Mb flash dump");
    lv_list_add_btn(main_list, LV_SYMBOL_UPLOAD, "Write XeniumOS update file");
    lv_list_add_btn(main_list, LV_SYMBOL_LOOP, "Cycle the Xenium RGB LED");
    lv_list_add_btn(main_list, LV_SYMBOL_POWER, "Reset");
    lv_list_add_btn(main_list, LV_SYMBOL_POWER, "Power Off");
    lv_obj_set_event_cb(main_list, main_list_cb);
    lv_group_add_obj(input_group, main_list);

    /* Create a text box widget for debug info*/
    text_box = lv_textarea_create(lv_scr_act(), NULL);
    lv_textarea_set_text(text_box, "");
    lv_textarea_set_cursor_hidden(text_box, true);
    lv_obj_set_height(text_box, lv_obj_get_height(main_list));
    lv_obj_set_width(text_box, screen_w - lv_obj_get_width(main_list) - 50);
    lv_obj_align(text_box, NULL, LV_ALIGN_IN_TOP_RIGHT, -XMARGIN, 80);

    /* Create a text label above progress bar for status */
    lv_obj_t *status_label = lv_label_create(lv_scr_act(), NULL);
    lv_label_set_align(status_label, LV_LABEL_ALIGN_CENTER); /*Center aligned lines*/
    lv_label_set_text(status_label, "Idle");
    lv_obj_set_width(status_label, 150);
    lv_obj_align(status_label, NULL, LV_ALIGN_IN_BOTTOM_LEFT, XMARGIN, -YMARGIN - 30);
    static lv_style_t status_label_style;
    lv_style_init(&status_label_style);
    lv_style_set_text_font(&status_label_style, LV_STATE_DEFAULT, &lv_font_montserrat_18);
    lv_obj_add_style(status_label, LV_LABEL_PART_MAIN, &status_label_style);

    /* Create a progress bar */
    progress_bar = lv_bar_create(lv_scr_act(), NULL);
    lv_obj_set_size(progress_bar, screen_w - 2 * XMARGIN, 20);
    lv_obj_align(progress_bar, NULL, LV_ALIGN_IN_BOTTOM_MID, 0, -YMARGIN);
    lv_bar_set_anim_time(progress_bar, 500);
    lv_bar_set_value(progress_bar, 50, LV_ANIM_ON);
}

static void main_list_cb(lv_obj_t *ta, lv_event_t e)
{
    char buf[64];
    if (e == LV_EVENT_PRESSED)
    {
        lv_obj_t *btn = lv_list_get_btn_selected(ta);
        sprintf(buf, "Pressed %s\n", lv_list_get_btn_text(btn));
        lv_textarea_add_text(text_box, buf);
    }
}
