#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "xenium.h"
#ifdef NXDK
#include <hal/io.h>
#endif

//See https://github.com/Ryzee119/OpenXenium/blob/master/Firmware/openxenium.vhd
//for Xenium CPLD details

static void _io_output(unsigned short address, unsigned char value)
{
    #ifdef NXDK
    IoOutputByte(address, value);
    #else
    printf("_io_output: 0x%04x, 0x%02x\n", address, value);
    #endif
}

static unsigned char _io_input(unsigned short address)
{
    #ifdef NXDK
    return IoInputByte(address);
    #else
    printf("_io_input: 0x%04x\n", address);
    #endif
}

//IO CONTROL INTERFACE
void xenium_set_bank(unsigned char bank)
{
    if (bank <= 10)
        _io_output(XENIUM_REGISTER_BANKING, bank);
}

unsigned char xenium_get_bank()
{
    return _io_input(XENIUM_REGISTER_BANKING) & 0x0F;
}

void xenium_set_led(unsigned char led)
{
    _io_output(XENIUM_REGISTER_LED, led);
}

//FLASH MEMORY INTERFACE
static void lpc_send_byte(unsigned int address, unsigned char data)
{
    volatile unsigned char *volatile lpc_mem_map = (unsigned char *)LPC_MEMORY_BASE;
    lpc_mem_map[address] = data;
}

static unsigned char xenium_flash_read_byte(unsigned int address)
{
    volatile unsigned char *volatile lpc_mem_map = (unsigned char *)LPC_MEMORY_BASE;
    return lpc_mem_map[address];
}

static void xenium_flash_read_stream(unsigned int address, unsigned char *data, unsigned int len)
{
    volatile unsigned char *volatile lpc_mem_map = (unsigned char *)LPC_MEMORY_BASE;
    memcpy(data, (void *)&lpc_mem_map[address], len);
}

static void xenium_flash_write_stream(unsigned int address, unsigned char *data, unsigned int len)
{
    volatile unsigned char *volatile lpc_mem_map = (unsigned char *)LPC_MEMORY_BASE;
    memcpy((void *)&lpc_mem_map[address], data, len);
}

static void xenium_flash_reset(void)
{
    lpc_send_byte(0x00000000, 0xF0);
}

static unsigned char xenium_flash_busy(void)
{
    return xenium_flash_read_byte(0) != xenium_flash_read_byte(0);
}

static unsigned int xenium_get_bank_size(unsigned char bank)
{
    unsigned int bank_size;
    switch (bank)
    {
    case XENIUM_BANK_1:
    case XENIUM_BANK_2:
    case XENIUM_BANK_3:
    case XENIUM_BANK_4:
        bank_size = 0x40000;
        break;
    case XENIUM_BANK_1_512:
    case XENIUM_BANK_2_512:
        bank_size = 0x80000;
        break;
    case XENIUM_BANK_1_1024:
        bank_size = 0x100000;
        break;
    //Let's not erase any other banks!
    default:
        bank_size = 0;
        break;
    }
    return bank_size;
}

static void xenium_sector_erase(unsigned int sector_address)
{
    lpc_send_byte(0xAAAA, 0xAA);
    lpc_send_byte(0x5555, 0x55);
    lpc_send_byte(0xAAAA, 0x80);
    lpc_send_byte(0xAAAA, 0xAA);
    lpc_send_byte(0x5555, 0x55);
    lpc_send_byte(sector_address, 0x30);
    while (xenium_flash_busy())
        ;
    xenium_flash_reset();
}

static void xenium_flash_program_byte(unsigned int address, unsigned char data)
{
    lpc_send_byte(0xAAAA, 0xAA);
    lpc_send_byte(0x5555, 0x55);
    lpc_send_byte(0xAAAA, 0xA0);
    lpc_send_byte(address, data);
    while (xenium_flash_busy())
        ;
}

unsigned char xenium_is_detected()
{
    xenium_flash_reset();
    lpc_send_byte(0xAAAA, 0xAA);
    lpc_send_byte(0x5555, 0x55);
    lpc_send_byte(0xAAAA, 0x90);
    unsigned char manuf = xenium_flash_read_byte(0x00);
    xenium_flash_reset();

    lpc_send_byte(0xAAAA, 0xAA);
    lpc_send_byte(0x5555, 0x55);
    lpc_send_byte(0xAAAA, 0x90);
    unsigned char devid = xenium_flash_read_byte(0x02);
    xenium_flash_reset();
    printf("manuf: %02x, devid: %02x\n", manuf, devid);
    if (manuf == XENIUM_MANUF_ID &&
        devid == XENIUM_DEVICE_ID)
    {
        printf("Xenium is detected\n");
        return 1;
    }
    printf("Xenium NOT detected\n");
    return 0;
}

void xenium_erase_bank(unsigned char bank)
{
    unsigned int bank_size = xenium_get_bank_size(bank);
    if (bank_size == 0)
        return;

    printf("Erasing Bank %u ", bank);
    unsigned char old_bank = xenium_get_bank();
    xenium_set_bank(bank);
    xenium_flash_reset();
    for (unsigned int i = 0; i < bank_size; i += XENIUM_FLASH_SECTOR_SIZE)
    {
        printf(" . ");
        xenium_sector_erase(i);
    }
    xenium_set_bank(old_bank);
    printf("\n", bank);
}

void xenium_write_bank(unsigned char bank, unsigned char *data)
{
    unsigned int bank_size = xenium_get_bank_size(bank);
    if (bank_size == 0)
        return;

    printf("Writing Bank %u ", bank);
    unsigned char old_bank = xenium_get_bank();
    xenium_set_bank(bank);
    xenium_flash_reset();
    for (unsigned int i = 0; i < bank_size; i++)
    {
        xenium_flash_program_byte(i, data[i]);
    }
    xenium_set_bank(old_bank);
}
