' VB.NET key generator example
' (c) 2004-2006 element 5
' (c) 2006 Digital River GmbH
' written by Stefan Weber
'
' SDK 3 File Revision 1

' define UNICODE_GEN to support UTF8 encoded in-/output files
#const UNICODE_GEN = True

' define BINARY_GEN if the returned key is a file of a specified MIME type
' Note that binary key generators can also return text files if the content
' type is specified as "text/plain". Therefore you should always provide
' binary key generators unless you need the CCKey feature of textual keys
' instead of XML notification.
#const BINARY_GEN = True

Imports System
Imports System.IO
Imports System.Text
Imports System.Collections
Imports System.Security.Cryptography

' This class implemens a key generator. 
Class MainClass
	' possible key generator exit codes - do not modify
	Public Enum KeyGenReturnCode As Integer 
		' success
		ERC_SUCCESS  		= 0
		ERC_SUCCESS_BIN  	= 1
		' failure
		ERC_ERROR 		= 10
		ERC_MEMORY 		= 11
		ERC_FILE_IO  		= 12
		ERC_BAD_ARGS  		= 13
		ERC_BAD_INPUT  		= 14
		ERC_EXPIRED  		= 15
		ERC_INTERNAL  		= 16
	End Enum 

	' key generator exception class
	Public Class KeyGenException
	Inherits System.Exception
		Public ERC As KeyGenReturnCode

		Public Sub New(ByVal message As String, ByVal e As KeyGenReturnCode)
			ERC = e
		End Sub
	End Class 
	
	' input/output file encoding
#if UNICODE_GEN	
	Private Shared fileEncoding As Encoding = New UTF8Encoding
#else	
	Private Shared fileEncoding As Encoding = Encoding.GetEncoding("ISO-8859-1")
#end if	
	
	' list of input values
	Private Shared Inputs As SortedList = New SortedList

	' generated key data
#if BINARY_GEN 
	Private Shared KeyMIMEType As String 		' the MIME type
	Private Shared KeyDisplayFileName As String 	' the displayed filename
	Private Shared KeyData As Byte() 		' the actual key data
#else
	Private Shared KeyResult1 As String 		' the key for the user
	Private Shared KeyResult2 As String 		' the cckey for the publisher
#end if
	
	' a simple example algorithm using MD5 message digests
	Public Shared Sub GenerateKey()
		' compose a string using a secret value (only known to the author of
		' the key generator and the application to be unlocked by the key)
		' and some of the input values

		' here the secret is simply "S.E.C.R.E.T" :-)
		' and we just use the registration name
		Dim digestInput As Byte() = fileEncoding.GetBytes("S.E.C.R.E.T" + Inputs("REG_NAME"))
		
		' this example uses the MD5 message disgest algorithm
		Dim md5 As MD5 = New MD5CryptoServiceProvider 
		
#if BINARY_GEN 
		' create a binary key
		KeyData = md5.ComputeHash(digestInput)
		KeyDisplayFileName = "key.bin"
		KeyMIMEType = "application/octet-stream"
		
		' note: there is no cckey generated for binary key generators since
		' the copy exectly matches the original. it is sent to the publisher
		' via XML order notification mails
#else	
		' create a textual key
		Dim HashCode As Byte() = md5.ComputeHash(digestInput)
		
		' result 1 - key for the customer
		KeyResult1 = Convert.ToBase64String(HashCode)
		
		' result 2 - cckey for the publisher
		KeyResult2 = Inputs("REG_NAME") + " " + KeyResult1
#end if
	End Sub

	' split a string at the first equals sign and add key/value to Inputs[]
	Public Shared Sub AddInputLine(ByVal line As String)
		Dim posEqual As Integer = line.IndexOf("="C)
		If posEqual > 0 Then
			Dim akey As String = line.Remove(posEqual, line.Length - posEqual)
			Dim avalue As String = line.Substring(posEqual + 1)
			If avalue.Length > 0 Then
				Inputs.Add(akey, avalue)
			End If
		End If
	End Sub

	' read the input file and parse its lines into the Inputs[] list
	Public Shared Sub ReadInput(ByVal pathname As String)
		Inputs.Clear

		' attempt to open the input file for read-only access
		Dim fsIn As FileStream = New FileStream (pathname, FileMode.Open, FileAccess.Read, FileShare.Read)
		Dim sr As StreamReader = New StreamReader (fsIn, fileEncoding, True)
		
		' process every line in the file
		Dim Line As String = sr.ReadLine
		While Not (Line Is Nothing)
			AddInputLine(Line.Trim)
			Line = sr.ReadLine
		End While
		
		' explicitly close the StreamReader to properly flush all buffers
		sr.Close ' this also closes the FileStream (fsIn)

		' check the input encoding
		Dim EncName As String = CType(Inputs("ENCODING"), String)
#if UNICODE_GEN
		If (EncName Is Nothing) OrElse (Not (EncName = "UTF8")) Then
			Throw New KeyGenException ("bad input encoding, expected UTF-8", KeyGenReturnCode.ERC_BAD_INPUT)
		End If
#else		
		If (Not (EncName Is Nothing)) AndAlso (Not (EncName = "ISO-8859-1")) Then
			Throw New KeyGenException ("bad input encoding, expected ISO-8859-1", KeyGenReturnCode.ERC_BAD_INPUT)
		End If
#end if

		' check for valid input
		Dim regName As String = Inputs("REG_NAME").ToString

		If regName.Length < 8 Then
			Throw New KeyGenException ("REG_NAME must have at least 8 characters", KeyGenReturnCode.ERC_BAD_INPUT)
		End If
	End Sub

	' write a string to an output file using the encoding specified in the input file
	Public Shared Sub WriteOutput(ByVal pathname As String, ByVal data As String)
		' Create an instance of StreamWriter to write text to a file.
		' The using statement also closes the StreamWriter.
		Dim fsOut As FileStream = New FileStream (pathname, FileMode.Create)
		Dim sw As StreamWriter = New StreamWriter (fsOut, fileEncoding)
		sw.Write(data)
		sw.Close
	End Sub

#if BINARY_GEN
	' write a binary byte array to an output file
	Public Shared Sub WriteOutputData(ByVal pathname As String, ByVal data As Byte())
		' Create an instance of StreamWriter to write text to a file.
		' The using statement also closes the StreamWriter.
		Dim fsOut As FileStream = New FileStream (pathname, FileMode.Create)
		Dim bw As BinaryWriter = New BinaryWriter (fsOut)
		bw.Write(data)
		bw.Close
	End Sub
#end if
	
	Public Shared Sub Main(ByVal args As String())
		Console.WriteLine("VB.NET example key generator")
		Try
			If args.Length = 3 Then
				Console.Write("> reading input file: ")
				Console.WriteLine(args(0))
				ReadInput(args(0))
				
				Console.WriteLine("> processing ... ")
				GenerateKey
				
				Console.Write("> writing output files: ")
#if BINARY_GEN				
				' write MIME type and display filename to output file #1
				WriteOutput(args(1), KeyMIMEType + ":" + KeyDisplayFileName)
				WriteOutputData(args(2), KeyData)
				
				' binary key generator must return ERC_SUCCESS_BIN on success
				Environment.ExitCode = CType(KeyGenReturnCode.ERC_SUCCESS_BIN, Integer)
#else				
				WriteOutput(args(1), KeyResult1)
				WriteOutput(args(2), KeyResult2)
				Environment.ExitCode = CType(KeyGenReturnCode.ERC_SUCCESS, Integer)
#end if				
			Else
				Console.Write("Usage: <input> <output1> <output2>")
				Environment.ExitCode = CType(KeyGenReturnCode.ERC_BAD_ARGS, Integer)
			End If

		Catch e As KeyGenException
			Console.Write("* KeyGen Exception: " + e.Message)
			
			' set the exit code to the ERC of the exception object
			Environment.ExitCode = CType(e.ERC, Integer)
			
			' and write the error message to output file #1
			Try
				WriteOutput(args(1), e.Message)
			Catch
			End Try

		Catch e As Exception
			' for general exceptions return ERC_ERROR
			Environment.ExitCode = CType(KeyGenReturnCode.ERC_ERROR, Integer)
			Console.Write("* CLR Exception: " + e.Message)
			
			' and write the error message to output file #1
			Try
				WriteOutput(args(1), e.Message)
			Catch
			End Try
		End Try

		Console.WriteLine("ExitCode: {0}", Environment.ExitCode)
	End Sub
End Class 