// m_fc_ntfs.c : implementation of ntfs master file table (MFT) file carver.
//
// REQUIRE: FORENSIC SUB-SYSTEM INIT: NTFS.
//
// The fs/ntfs module is responsible for displaying various file system files
// which are generated by carving the NTFS MFT.
//
// (c) Ulf Frisk, 2020
// Author: Ulf Frisk, pcileech@frizk.net
//

#include "fc.h"
#include "vmm.h"
#include "pluginmanager.h"
#include "util.h"

NTSTATUS M_FcNtfs_ReadInfo(_Out_ PBYTE pb, _In_ DWORD cb, _Out_ PDWORD pcbRead, _In_ QWORD cbOffset)
{
    NTSTATUS nt = VMMDLL_STATUS_FILE_INVALID;
    PFC_MAP_NTFSENTRY pe;
    PFCOB_MAP_NTFS pObNtfsMap = NULL;
    QWORD i, o, qwIdBase, qwIdTop, cId, cszuBuffer, cbOffsetBuffer;
    DWORD iDirDepth;
    LPSTR szuBuffer = NULL;
    CHAR szTimeCreate[24], szTimeModify[24];
    if(!FcNtfs_GetIdFromPosition(cbOffset, FALSE, &qwIdBase)) { goto fail; }
    if(!FcNtfs_GetIdFromPosition(cbOffset + cb, FALSE, &qwIdTop)) { goto fail; }
    cId = min(cb / M_NTFS_INFO_LINELENGTH_UTF8, qwIdTop - qwIdBase) + 1;
    if(!FcNtfsMap_GetFromIdRange(qwIdBase, cId, &pObNtfsMap) || !pObNtfsMap->cMap) { goto fail; }
    cbOffsetBuffer = pObNtfsMap->pMap[0].cszuOffset;
    if((cbOffsetBuffer > cbOffset) || (cbOffset - cbOffsetBuffer > 0x10000)) { goto fail; }
    cszuBuffer = 0x01000000;
    if(!(szuBuffer = LocalAlloc(0, cszuBuffer))) { goto fail; }
    for(i = 0, o = 0; (i < pObNtfsMap->cMap) && (o < cszuBuffer - 0x1000); i++) {
        pe = pObNtfsMap->pMap + i;
        Util_FileTime2String((PFILETIME)&pe->ftCreate, szTimeCreate);
        Util_FileTime2String((PFILETIME)&pe->ftModify, szTimeModify);
        iDirDepth = min(20, pe->dwDirDepth);
        o += snprintf(
            szuBuffer + o,
            cszuBuffer - o,
            "%6llx%12llx %8x %s : %s %12llx %3x %c ",
            pe->qwId,
            pe->pa,
            (pe->dwMftId & 0xf0000000 ? 0 : pe->dwMftId),
            szTimeCreate,
            szTimeModify,
            pe->qwFileSize,
            pe->dwFileSizeResident,
            pe->fDir ? 'D' : ' '
        );
        o += WideCharToMultiByte(CP_UTF8, 0, pe->wszText, -1, szuBuffer + o, (int)(cszuBuffer - o), NULL, NULL);
        if(o) {
            szuBuffer[o - 1] = '\n';
        }
    }
    nt = Util_VfsReadFile_FromPBYTE(szuBuffer, o, pb, cb, pcbRead, cbOffset - cbOffsetBuffer);
fail:
    LocalFree(szuBuffer);
    Ob_DECREF(pObNtfsMap);
    return nt;
}

NTSTATUS M_FcNtfs_Read(_In_ PVMMDLL_PLUGIN_CONTEXT ctx, _Out_ PBYTE pb, _In_ DWORD cb, _Out_ PDWORD pcbRead, _In_ QWORD cbOffset)
{
    NTSTATUS nt = VMMDLL_STATUS_FILE_INVALID;
    PFCOB_MAP_NTFS pObNtfsMap = NULL;
    PFC_MAP_NTFSENTRY peNtfs;
    QWORD qwHashPath;
    BYTE pbNtfsRecordMax[0x400];
    qwHashPath = Util_HashPathW_Registry(ctx->wszPath);
    if(!wcscmp(ctx->wszPath, L"ntfs_files.txt")) {
        nt = M_FcNtfs_ReadInfo(pb, cb, pcbRead, cbOffset);
    } else if(FcNtfsMap_GetFromHash(qwHashPath, &pObNtfsMap) && pObNtfsMap->cMap) {
        peNtfs = pObNtfsMap->pMap + 0;
        if(peNtfs->qwFileSize && (peNtfs->dwFileSizeResident == peNtfs->qwFileSize) && (peNtfs->qwFileSize < 0x400)) {
            if(FcNtfs_GetMftResidentData(peNtfs, pbNtfsRecordMax, peNtfs->dwFileSizeResident, NULL)) {
                nt = Util_VfsReadFile_FromPBYTE(pbNtfsRecordMax, peNtfs->dwFileSizeResident, pb, cb, pcbRead, cbOffset);
            }
        } else {
            nt = Util_VfsReadFile_FromZERO(peNtfs->qwFileSize, pb, cb, pcbRead, cbOffset);
        }
    }
    Ob_DECREF(pObNtfsMap);
    return nt;
}

VOID M_FcNtfs_ListDirectory(_In_ LPWSTR wszPath, _Inout_ PHANDLE pFileList)
{
    DWORD i;
    PFC_MAP_NTFSENTRY pe;
    PFCOB_MAP_NTFS pObNtfsMap = NULL;
    VMMDLL_VFS_FILELIST_EXINFO FileExInfo = { 0 };
    WCHAR wszNameFix[MAX_PATH];
    QWORD qwHashPath;
    FileExInfo.dwVersion = VMMDLL_VFS_FILELIST_EXINFO_VERSION;
    qwHashPath = Util_HashPathW_Registry(wszPath);
    if(!FcNtfsMap_GetFromHashParent(qwHashPath, &pObNtfsMap)) { return; }
    for(i = 0; i < pObNtfsMap->cMap; i++) {
        pe = pObNtfsMap->pMap + i;
        if(!Util_PathFileNameFix_Registry(wszNameFix, NULL, pe->wszTextName, 0, pe->dwTextSeq, FALSE)) { continue; }
        FileExInfo.qwCreationTime = pe->ftCreate;
        FileExInfo.qwLastWriteTime = pe->ftModify;
        FileExInfo.qwLastAccessTime = pe->ftRead;
        if(pe->fDir) {
            FileExInfo.fCompressed = FALSE;
            VMMDLL_VfsList_AddDirectory(pFileList, wszNameFix, &FileExInfo);
        } else {
            FileExInfo.fCompressed = !pe->qwFileSize || (pe->qwFileSize != pe->dwFileSizeResident);
            VMMDLL_VfsList_AddFile(pFileList, wszNameFix, pe->qwFileSize, &FileExInfo);
        }
    }
    Ob_DECREF(pObNtfsMap);
}

BOOL M_FcNtfs_List(_In_ PVMMDLL_PLUGIN_CONTEXT ctx, _Inout_ PHANDLE pFileList)
{
    QWORD cbFileSizeUTF8;
    M_FcNtfs_ListDirectory(ctx->wszPath, pFileList);
    if(!ctx->wszPath[0] && FcNtfs_GetFileSize(NULL, &cbFileSizeUTF8, NULL)) {
        VMMDLL_VfsList_AddFile(pFileList, L"ntfs_files.txt", cbFileSizeUTF8, NULL);
    }
    return TRUE;
}

VOID M_FcNtfs_Notify(_In_ DWORD fEvent, _In_opt_ PVOID pvEvent, _In_opt_ DWORD cbEvent)
{
    if((fEvent == VMMDLL_PLUGIN_EVENT_FORENSIC_INIT_COMPLETE) && ctxFc->fEnableNtfs) {
        PluginManager_SetVisibility(TRUE, L"\\forensic\\ntfs", TRUE);
    }
}

VOID M_FcNtfs_Initialize(_Inout_ PVMMDLL_PLUGIN_REGINFO pRI)
{
    if((pRI->magic != VMMDLL_PLUGIN_REGINFO_MAGIC) || (pRI->wVersion != VMMDLL_PLUGIN_REGINFO_VERSION)) { return; }
    if((pRI->tpSystem != VMM_SYSTEM_WINDOWS_X64) && (pRI->tpSystem != VMM_SYSTEM_WINDOWS_X86)) { return; }
    if(ctxMain->dev.fVolatile) { return; }
    wcscpy_s(pRI->reg_info.wszPathName, 128, L"\\forensic\\ntfs");              // module name
    pRI->reg_info.fRootModule = TRUE;                                           // module shows in root directory
    pRI->reg_info.fRootModuleHidden = TRUE;                                     // module hidden by default
    pRI->reg_fn.pfnList = M_FcNtfs_List;                                        // List function supported
    pRI->reg_fn.pfnRead = M_FcNtfs_Read;                                        // Read function supported
    pRI->reg_fn.pfnNotify = M_FcNtfs_Notify;                                    // Notify function supported
    pRI->pfnPluginManager_Register(pRI);
}
