/*  test_hmac_prng.c - TinyCrypt implementation of some HMAC-PRNG tests */

/*
 *  Copyright (C) 2017 by Intel Corporation, All Rights Reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *    - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *
 *    - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *    - Neither the name of Intel Corporation nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 *  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 *  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 *  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE.
 */

/*
  DESCRIPTION
  This module tests the following PRNG routines:

  Scenarios tested include:
  - HMAC-PRNG init
  - HMAC-PRNG reseed
  - HMAC-PRNG generate)
*/

#include <tinycrypt/hmac_prng.h>
#include <tinycrypt/constants.h>
#include <test_utils.h>

#include <stdio.h>

struct hmac_prng_test_vector {
	uint8_t entropyinputlen;
	uint8_t noncelen;
	uint8_t personalizationstringlen;
	uint8_t additionalinputlen;
	uint8_t returnedbitslen;
	uint8_t entropyinput[32];
	uint8_t nonce[16];
	uint8_t personalizationstring[32];
	uint8_t entropyinputreseed[32];
	uint8_t additionalinputreseed[32];
	uint8_t returnedbits[128];
};

unsigned int do_hmac_prng_pr_false_test(unsigned int testnum, const struct
					hmac_prng_test_vector *vec)
{
	struct tc_hmac_prng_struct h;
	uint8_t random[128];
	uint32_t seed_material_size;
	uint8_t seed_material[32 + 16 + 32]; /*entropyinput || nonce || personalizationstring */
	unsigned int result = TC_PASS;

	uint8_t *p = seed_material;

	memset(&h, 0x0, sizeof(h));

	if (vec->entropyinputlen > 0) {
		memcpy(p, vec->entropyinput, vec->entropyinputlen);
		p += vec->entropyinputlen;
	}

	if (vec->noncelen > 0) {
		memcpy(p, vec->nonce, vec->noncelen);
		p += vec->noncelen;
	}

	if (vec->personalizationstringlen > 0) {
		memcpy(p, vec->personalizationstring, vec->personalizationstringlen);
	}

	seed_material_size = vec->entropyinputlen + vec->noncelen + vec->personalizationstringlen;
	(void)tc_hmac_prng_init(&h, seed_material, seed_material_size);
	(void)tc_hmac_prng_reseed(&h, vec->entropyinputreseed, vec->entropyinputlen, 0, 0);
	(void)tc_hmac_prng_generate(random, vec->returnedbitslen, &h);
	(void)tc_hmac_prng_generate(random, vec->returnedbitslen, &h);
	result = check_result(testnum, vec->returnedbits, vec->returnedbitslen, random, vec->returnedbitslen);
	return result;
}

unsigned int test_1(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x06, 0x03, 0x2c, 0xd5, 0xee, 0xd3, 0x3f, 0x39, 0x26, 0x5f, 0x49, 0xec,
			0xb1, 0x42, 0xc5, 0x11, 0xda, 0x9a, 0xff, 0x2a, 0xf7, 0x12, 0x03, 0xbf,
			0xfa, 0xf3, 0x4a, 0x9c, 0xa5, 0xbd, 0x9c, 0x0d,
		},
		{
			0x0e, 0x66, 0xf7, 0x1e, 0xdc, 0x43, 0xe4, 0x2a, 0x45, 0xad, 0x3c, 0x6f,
			0xc6, 0xcd, 0xc4, 0xdf,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x01, 0x92, 0x0a, 0x4e, 0x66, 0x9e, 0xd3, 0xa8, 0x5a, 0xe8, 0xa3, 0x3b,
			0x35, 0xa7, 0x4a, 0xd7, 0xfb, 0x2a, 0x6b, 0xb4, 0xcf, 0x39, 0x5c, 0xe0,
			0x03, 0x34, 0xa9, 0xc9, 0xa5, 0xa5, 0xd5, 0x52,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x76, 0xfc, 0x79, 0xfe, 0x9b, 0x50, 0xbe, 0xcc, 0xc9, 0x91, 0xa1, 0x1b,
			0x56, 0x35, 0x78, 0x3a, 0x83, 0x53, 0x6a, 0xdd, 0x03, 0xc1, 0x57, 0xfb,
			0x30, 0x64, 0x5e, 0x61, 0x1c, 0x28, 0x98, 0xbb, 0x2b, 0x1b, 0xc2, 0x15,
			0x00, 0x02, 0x09, 0x20, 0x8c, 0xd5, 0x06, 0xcb, 0x28, 0xda, 0x2a, 0x51,
			0xbd, 0xb0, 0x38, 0x26, 0xaa, 0xf2, 0xbd, 0x23, 0x35, 0xd5, 0x76, 0xd5,
			0x19, 0x16, 0x08, 0x42, 0xe7, 0x15, 0x8a, 0xd0, 0x94, 0x9d, 0x1a, 0x9e,
			0xc3, 0xe6, 0x6e, 0xa1, 0xb1, 0xa0, 0x64, 0xb0, 0x05, 0xde, 0x91, 0x4e,
			0xac, 0x2e, 0x9d, 0x4f, 0x2d, 0x72, 0xa8, 0x61, 0x6a, 0x80, 0x22, 0x54,
			0x22, 0x91, 0x82, 0x50, 0xff, 0x66, 0xa4, 0x1b, 0xd2, 0xf8, 0x64, 0xa6,
			0xa3, 0x8c, 0xc5, 0xb6, 0x49, 0x9d, 0xc4, 0x3f, 0x7f, 0x2b, 0xd0, 0x9e,
			0x1e, 0x0f, 0x8f, 0x58, 0x85, 0x93, 0x51, 0x24,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(1, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_2(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xaa, 0xdc, 0xf3, 0x37, 0x78, 0x8b, 0xb8, 0xac, 0x01, 0x97, 0x66, 0x40,
			0x72, 0x6b, 0xc5, 0x16, 0x35, 0xd4, 0x17, 0x77, 0x7f, 0xe6, 0x93, 0x9e,
			0xde, 0xd9, 0xcc, 0xc8, 0xa3, 0x78, 0xc7, 0x6a,
		},
		{
			0x9c, 0xcc, 0x9d, 0x80, 0xc8, 0x9a, 0xc5, 0x5a, 0x8c, 0xfe, 0x0f, 0x99,
			0x94, 0x2f, 0x5a, 0x4d,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x03, 0xa5, 0x77, 0x92, 0x54, 0x7e, 0x0c, 0x98, 0xea, 0x17, 0x76, 0xe4,
			0xba, 0x80, 0xc0, 0x07, 0x34, 0x62, 0x96, 0xa5, 0x6a, 0x27, 0x0a, 0x35,
			0xfd, 0x9e, 0xa2, 0x84, 0x5c, 0x7e, 0x81, 0xe2,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x17, 0xd0, 0x9f, 0x40, 0xa4, 0x37, 0x71, 0xf4, 0xa2, 0xf0, 0xdb, 0x32,
			0x7d, 0xf6, 0x37, 0xde, 0xa9, 0x72, 0xbf, 0xff, 0x30, 0xc9, 0x8e, 0xbc,
			0x88, 0x42, 0xdc, 0x7a, 0x9e, 0x3d, 0x68, 0x1c, 0x61, 0x90, 0x2f, 0x71,
			0xbf, 0xfa, 0xf5, 0x09, 0x36, 0x07, 0xfb, 0xfb, 0xa9, 0x67, 0x4a, 0x70,
			0xd0, 0x48, 0xe5, 0x62, 0xee, 0x88, 0xf0, 0x27, 0xf6, 0x30, 0xa7, 0x85,
			0x22, 0xec, 0x6f, 0x70, 0x6b, 0xb4, 0x4a, 0xe1, 0x30, 0xe0, 0x5c, 0x8d,
			0x7e, 0xac, 0x66, 0x8b, 0xf6, 0x98, 0x0d, 0x99, 0xb4, 0xc0, 0x24, 0x29,
			0x46, 0x45, 0x23, 0x99, 0xcb, 0x03, 0x2c, 0xc6, 0xf9, 0xfd, 0x96, 0x28,
			0x47, 0x09, 0xbd, 0x2f, 0xa5, 0x65, 0xb9, 0xeb, 0x9f, 0x20, 0x04, 0xbe,
			0x6c, 0x9e, 0xa9, 0xff, 0x91, 0x28, 0xc3, 0xf9, 0x3b, 0x60, 0xdc, 0x30,
			0xc5, 0xfc, 0x85, 0x87, 0xa1, 0x0d, 0xe6, 0x8c,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(2, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_3(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x62, 0xcd, 0xa4, 0x41, 0xdd, 0x80, 0x2c, 0x76, 0x52, 0xc0, 0x0b, 0x99,
			0xca, 0xc3, 0x65, 0x2a, 0x64, 0xfc, 0x75, 0x38, 0x8d, 0xc9, 0xad, 0xcf,
			0x76, 0x35, 0x30, 0xac, 0x31, 0xdf, 0x92, 0x14,
		},
		{
			0x5f, 0xdc, 0x89, 0x7a, 0x0c, 0x1c, 0x48, 0x22, 0x04, 0xef, 0x07, 0xe0,
			0x80, 0x5c, 0x01, 0x4b,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xbd, 0x9b, 0xbf, 0x71, 0x74, 0x67, 0xbf, 0x4b, 0x5d, 0xb2, 0xaa, 0x34,
			0x4d, 0xd0, 0xd9, 0x09, 0x97, 0xc8, 0x20, 0x1b, 0x22, 0x65, 0xf4, 0x45,
			0x12, 0x70, 0x12, 0x8f, 0x5a, 0xc0, 0x5a, 0x1a,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x7e, 0x41, 0xf9, 0x64, 0x7a, 0x5e, 0x67, 0x50, 0xeb, 0x8a, 0xcf, 0x13,
			0xa0, 0x2f, 0x23, 0xf3, 0xbe, 0x77, 0x61, 0x1e, 0x51, 0x99, 0x2c, 0xed,
			0xb6, 0x60, 0x2c, 0x31, 0x45, 0x31, 0xaf, 0xf2, 0xa6, 0xe4, 0xc5, 0x57,
			0xda, 0x07, 0x77, 0xd4, 0xe8, 0x5f, 0xae, 0xfc, 0xb1, 0x43, 0xf1, 0xa9,
			0x2e, 0x0d, 0xba, 0xc8, 0xde, 0x8b, 0x88, 0x5c, 0xed, 0x62, 0xa1, 0x24,
			0xf0, 0xb1, 0x06, 0x20, 0xf1, 0x40, 0x9a, 0xe8, 0x7e, 0x22, 0x89, 0x94,
			0xb8, 0x30, 0xec, 0xa6, 0x38, 0xcc, 0xdc, 0xee, 0xdd, 0x3f, 0xcd, 0x07,
			0xd0, 0x24, 0xb6, 0x46, 0x70, 0x4f, 0x44, 0xd5, 0xd9, 0xc4, 0xc3, 0xa7,
			0xb7, 0x05, 0xf3, 0x71, 0x04, 0xb4, 0x5b, 0x9c, 0xfc, 0x2d, 0x93, 0x3a,
			0xe4, 0x3c, 0x12, 0xf5, 0x3e, 0x3e, 0x6f, 0x79, 0x8c, 0x51, 0xbe, 0x5f,
			0x64, 0x01, 0x15, 0xd4, 0x5c, 0xf9, 0x19, 0xa4,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(3, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_4(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x6b, 0xdc, 0x6c, 0xa8, 0xee, 0xf0, 0xe3, 0x53, 0x3a, 0xbd, 0x02, 0x58,
			0x0e, 0xbb, 0xc8, 0xa9, 0x2f, 0x38, 0x2c, 0x5b, 0x1c, 0x8e, 0x3e, 0xaa,
			0x12, 0x56, 0x6e, 0xcf, 0xb9, 0x03, 0x89, 0xa3,
		},
		{
			0x8f, 0x84, 0x81, 0xcc, 0x77, 0x35, 0x82, 0x74, 0x77, 0xe0, 0xe4, 0xac,
			0xb7, 0xf4, 0xa0, 0xfa,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x72, 0xec, 0xa6, 0xf1, 0x56, 0x07, 0x20, 0xe6, 0xbd, 0x1f, 0xf0, 0x15,
			0x2c, 0x12, 0xee, 0xff, 0x1f, 0x95, 0x94, 0x62, 0xfd, 0x62, 0xc7, 0x2b,
			0x7d, 0xde, 0x96, 0xab, 0xcb, 0x7f, 0x79, 0xfb,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xd5, 0xa2, 0xe2, 0xf2, 0x54, 0xb5, 0xae, 0x65, 0x59, 0x0d, 0x4f, 0xd1,
			0xff, 0x5c, 0x75, 0x8e, 0x42, 0x5b, 0xe4, 0xba, 0xcd, 0xee, 0xde, 0x79,
			0x89, 0x66, 0x9f, 0x0a, 0x22, 0xd3, 0x42, 0x74, 0xfd, 0xfc, 0x2b, 0xf8,
			0x71, 0x35, 0xe3, 0x0a, 0xbd, 0xae, 0x26, 0x91, 0x62, 0x9c, 0x2f, 0x6f,
			0x42, 0x5b, 0xd4, 0xe1, 0x19, 0x90, 0x4d, 0x47, 0x85, 0xec, 0xd9, 0x32,
			0x8f, 0x15, 0x25, 0x95, 0x63, 0xe5, 0xa7, 0x1f, 0x91, 0x5e, 0xc0, 0xc0,
			0x2b, 0x66, 0x65, 0x54, 0x71, 0x06, 0x7b, 0x01, 0x01, 0x6f, 0xdf, 0x93,
			0x4a, 0x47, 0xb0, 0x17, 0xe0, 0x7c, 0x21, 0x33, 0x26, 0x41, 0x40, 0x0b,
			0xbe, 0x57, 0x19, 0x05, 0x0d, 0xba, 0x22, 0xc0, 0x20, 0xb9, 0xb2, 0xd2,
			0xcd, 0xb9, 0x33, 0xdb, 0xc7, 0x0f, 0x76, 0xfe, 0xc4, 0xb1, 0xd8, 0x39,
			0x80, 0xfd, 0x1a, 0x13, 0xc4, 0x56, 0x58, 0x36,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(4, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_5(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x09, 0x6e, 0xf3, 0x72, 0x94, 0xd3, 0x69, 0xfa, 0xce, 0x1a, 0xdd, 0x3e,
			0xb8, 0xb4, 0x25, 0x89, 0x5e, 0x92, 0x16, 0x26, 0x49, 0x57, 0x05, 0xc5,
			0xa0, 0x3e, 0xe5, 0x66, 0xb3, 0x41, 0x58, 0xec,
		},
		{
			0x6e, 0x2e, 0x08, 0x25, 0x53, 0x4d, 0x29, 0x89, 0x71, 0x5c, 0xc8, 0x59,
			0x56, 0xe0, 0x14, 0x8d,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x1b, 0x4f, 0x71, 0x25, 0xf4, 0x72, 0xc2, 0x53, 0x83, 0x7f, 0xa7, 0x87,
			0xd5, 0xac, 0xf0, 0x38, 0x2a, 0x3b, 0x89, 0xc3, 0xf4, 0x1c, 0x21, 0x1d,
			0x26, 0x30, 0x52, 0x40, 0x2d, 0xcc, 0x62, 0xc5,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x45, 0x41, 0xf2, 0x4f, 0x75, 0x9b, 0x5f, 0x2a, 0xc2, 0xb5, 0x7b, 0x51,
			0x12, 0x50, 0x77, 0xcc, 0x74, 0x0b, 0x38, 0x59, 0xa7, 0x19, 0xa9, 0xba,
			0xb1, 0x19, 0x6e, 0x6c, 0x0c, 0xa2, 0xbd, 0x05, 0x7a, 0xf9, 0xd3, 0x89,
			0x23, 0x86, 0xa1, 0x81, 0x3f, 0xc8, 0x87, 0x5d, 0x8d, 0x36, 0x4f, 0x15,
			0xe7, 0xfd, 0x69, 0xd1, 0xcc, 0x66, 0x59, 0x47, 0x04, 0x15, 0x27, 0x81,
			0x64, 0xdf, 0x65, 0x62, 0x95, 0xba, 0x9c, 0xfc, 0xee, 0x79, 0xf6, 0xcb,
			0xe2, 0x6e, 0xe1, 0x36, 0xe6, 0xb4, 0x5e, 0xc2, 0x24, 0xad, 0x37, 0x9c,
			0x60, 0x79, 0xb1, 0x0a, 0x2e, 0x0c, 0xb5, 0xf7, 0xf7, 0x85, 0xef, 0x0a,
			0xb7, 0xa7, 0xc3, 0xfc, 0xd9, 0xcb, 0x65, 0x06, 0x05, 0x4d, 0x20, 0xe2,
			0xf3, 0xec, 0x61, 0x0c, 0xbb, 0xa9, 0xb0, 0x45, 0xa2, 0x48, 0xaf, 0x56,
			0xe4, 0xf6, 0xd3, 0xf0, 0xc8, 0xd9, 0x6a, 0x23,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(5, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_6(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xa7, 0xdc, 0xcd, 0xd4, 0x31, 0xae, 0x57, 0x26, 0xb8, 0x35, 0x85, 0xb5,
			0x4e, 0xae, 0x41, 0x08, 0xf7, 0xb7, 0xa2, 0x5c, 0x70, 0x18, 0x7c, 0x0a,
			0xcb, 0xb9, 0x4c, 0x96, 0xcc, 0x27, 0x7a, 0xa8,
		},
		{
			0x94, 0xc8, 0xf4, 0xb8, 0xe1, 0x95, 0xa4, 0x73, 0x56, 0xa8, 0x9a, 0x50,
			0xd1, 0x38, 0x9a, 0xb5,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x51, 0x73, 0x3e, 0xee, 0x2e, 0x92, 0x2f, 0x40, 0x55, 0xe5, 0x39, 0x39,
			0xe2, 0x22, 0xe7, 0x1f, 0xae, 0x73, 0x0e, 0xb0, 0x37, 0x44, 0x3d, 0xb2,
			0xc7, 0x67, 0x97, 0x08, 0xab, 0xb8, 0x6a, 0x65,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x99, 0xba, 0x26, 0x91, 0xa6, 0x22, 0xaf, 0xec, 0xc9, 0x47, 0x24, 0x18,
			0xe6, 0xa8, 0xf9, 0xf1, 0xcd, 0xc1, 0xe3, 0x58, 0x3c, 0x3b, 0xc7, 0xa2,
			0xa6, 0x50, 0xa1, 0xab, 0x79, 0xdc, 0xbc, 0xcb, 0xd6, 0x56, 0x63, 0x6c,
			0x57, 0x31, 0x79, 0x27, 0x6e, 0x78, 0x25, 0x69, 0x42, 0x0c, 0x97, 0x43,
			0x8c, 0x06, 0xbe, 0x89, 0x88, 0x67, 0xf6, 0x28, 0xb1, 0xc0, 0x1e, 0xb5,
			0x70, 0x26, 0x3d, 0x2c, 0x0f, 0x09, 0xc7, 0xaa, 0xb5, 0x36, 0xf6, 0xfb,
			0xa7, 0xdf, 0x6a, 0xad, 0x19, 0xe0, 0x5c, 0x23, 0x6b, 0x64, 0x56, 0x74,
			0x66, 0x7c, 0x03, 0xd1, 0xb6, 0xa0, 0x4d, 0x7f, 0xc1, 0x11, 0x77, 0xfe,
			0x78, 0x93, 0x3b, 0x30, 0x96, 0x79, 0xf5, 0xbf, 0x26, 0xa4, 0x63, 0x2b,
			0x9a, 0x13, 0xe3, 0x14, 0xc4, 0xbf, 0x45, 0x32, 0x42, 0x8d, 0x3d, 0x95,
			0xc6, 0x89, 0x00, 0x2b, 0x6d, 0xc1, 0xfb, 0xb1,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(6, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_7(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xc2, 0x86, 0x42, 0x5e, 0xcf, 0x54, 0x3a, 0x49, 0xbc, 0xc9, 0x19, 0x6b,
			0x0d, 0xb1, 0xa8, 0x0b, 0xc5, 0x4e, 0x49, 0x48, 0xad, 0xba, 0x6f, 0x41,
			0x71, 0x2a, 0x35, 0x0a, 0x02, 0x89, 0x1f, 0xa6,
		},
		{
			0x95, 0x7a, 0x65, 0x9a, 0x4e, 0xc2, 0xe0, 0xb7, 0xad, 0x18, 0x54, 0x83,
			0xc2, 0x20, 0xfd, 0x61,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x08, 0xc2, 0x12, 0x98, 0x13, 0xee, 0xa0, 0x77, 0x6f, 0xba, 0x72, 0x78,
			0x8f, 0xdf, 0x27, 0x18, 0x75, 0x9c, 0xc3, 0xc4, 0x20, 0x7f, 0xa2, 0x0a,
			0x5f, 0xe2, 0x3a, 0xc6, 0xe3, 0x2c, 0xc2, 0x8e,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x8e, 0x10, 0x20, 0xa4, 0xfd, 0x84, 0xc9, 0x9e, 0x0f, 0xc7, 0xe3, 0xf7,
			0xce, 0x48, 0xde, 0x5e, 0xd9, 0xec, 0x9a, 0x5c, 0x2c, 0xcd, 0x62, 0x4d,
			0xbe, 0x6f, 0x30, 0xe2, 0xf6, 0x88, 0xa3, 0x1d, 0xc5, 0x59, 0x57, 0x63,
			0x03, 0x57, 0xa5, 0xd4, 0x8c, 0xa2, 0xa4, 0x56, 0x24, 0x1a, 0x28, 0xbf,
			0xb1, 0x6d, 0x8b, 0xb0, 0x00, 0x87, 0x76, 0x97, 0xa7, 0xce, 0x24, 0xd9,
			0xad, 0x4d, 0x22, 0xb0, 0xc1, 0x51, 0x17, 0x99, 0x6f, 0x1f, 0x27, 0x0b,
			0x94, 0xf4, 0x6d, 0x7a, 0x9b, 0xdf, 0xa7, 0x60, 0x8f, 0xa1, 0xdd, 0x84,
			0x91, 0x77, 0xa9, 0xb8, 0x04, 0x9e, 0x51, 0xb6, 0xb7, 0xa2, 0x74, 0x26,
			0x23, 0x85, 0x4a, 0x1f, 0xdd, 0xb5, 0xef, 0xc4, 0x47, 0xee, 0xd1, 0xea,
			0x1a, 0xed, 0x6f, 0x02, 0xb4, 0xb2, 0x75, 0x4e, 0xcf, 0x71, 0xea, 0x05,
			0x09, 0xda, 0x2e, 0x54, 0xf5, 0x24, 0xa7, 0xe7,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(7, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_8(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x02, 0x81, 0x8b, 0xd7, 0xc1, 0xec, 0x45, 0x6a, 0xce, 0x55, 0xbe, 0xeb,
			0xa9, 0x9f, 0x64, 0x6a, 0x6d, 0x3a, 0xa0, 0xea, 0x78, 0x35, 0x6e, 0xa7,
			0x26, 0xb7, 0x63, 0xff, 0x0d, 0xd2, 0xd6, 0x56,
		},
		{
			0xc4, 0x82, 0x68, 0x7d, 0x50, 0x8c, 0x9b, 0x5c, 0x2a, 0x75, 0xf7, 0xce,
			0x39, 0x00, 0x14, 0xe8,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xcf, 0x31, 0x9b, 0xfa, 0x63, 0x98, 0x0e, 0x3c, 0xb9, 0x97, 0xfd, 0x28,
			0x77, 0x1b, 0xb5, 0x61, 0x4e, 0x3a, 0xcb, 0x11, 0x49, 0xba, 0x45, 0xc1,
			0x33, 0xff, 0xbb, 0xab, 0x17, 0x43, 0x31, 0x93,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x19, 0xa2, 0x31, 0xff, 0x26, 0xc1, 0x86, 0x5c, 0xe7, 0x5d, 0x7a, 0x71,
			0x85, 0xc3, 0x0d, 0xd0, 0xb3, 0x33, 0x12, 0x64, 0x33, 0xd0, 0xc8, 0xcb,
			0xf1, 0xbe, 0x0d, 0x2b, 0x38, 0x4d, 0x4e, 0xb3, 0xa8, 0xaf, 0xf0, 0x35,
			0x40, 0xfb, 0xfa, 0x5f, 0x54, 0x96, 0x52, 0x1a, 0x4e, 0x4a, 0x64, 0x07,
			0x1b, 0x44, 0xc7, 0x8b, 0xd0, 0xb7, 0xe6, 0x8f, 0xac, 0x9e, 0x56, 0x95,
			0xc5, 0xc1, 0x3f, 0xd3, 0xb9, 0xdb, 0xe7, 0xf7, 0x73, 0x97, 0x81, 0xa4,
			0xc8, 0xf0, 0xb9, 0x80, 0xf1, 0xb1, 0x7d, 0x99, 0xbc, 0xe1, 0x7c, 0xeb,
			0x52, 0xb5, 0x68, 0x66, 0xae, 0x02, 0x45, 0x6f, 0xfe, 0xf8, 0x33, 0x99,
			0xc8, 0xcf, 0x78, 0x26, 0xf3, 0xc4, 0x5c, 0x8a, 0x19, 0x31, 0x58, 0x90,
			0x91, 0x9d, 0x20, 0xf4, 0x0f, 0xc4, 0xe1, 0x8d, 0x07, 0xe9, 0xc8, 0xcc,
			0xd1, 0x6c, 0x33, 0x27, 0xb5, 0x98, 0x8f, 0x71,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(8, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_9(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x77, 0xa5, 0xc8, 0x6d, 0x99, 0xbe, 0x7b, 0xc2, 0x50, 0x28, 0x70, 0xf4,
			0x02, 0x5f, 0x9f, 0x75, 0x63, 0xe9, 0x17, 0x4e, 0xc6, 0x7c, 0x5f, 0x48,
			0x1f, 0x21, 0xfc, 0xf2, 0xb4, 0x1c, 0xae, 0x4b,
		},
		{
			0xed, 0x04, 0x4a, 0xd7, 0x2e, 0xe8, 0x22, 0x50, 0x6a, 0x6d, 0x0b, 0x12,
			0x11, 0x50, 0x29, 0x67,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x77, 0x81, 0x00, 0x74, 0x9f, 0x01, 0xa4, 0xd3, 0x5c, 0x3b, 0x4a, 0x95,
			0x8a, 0xaf, 0xe2, 0x96, 0x87, 0x7e, 0x0a, 0xca, 0xfd, 0x08, 0x9f, 0x50,
			0xbc, 0x77, 0x97, 0xa4, 0x2a, 0x33, 0xab, 0x71,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x83, 0x1a, 0x4d, 0xa5, 0x66, 0xf4, 0x62, 0x89, 0x90, 0x48, 0x93, 0xef,
			0x1c, 0xc1, 0xcd, 0x4a, 0xd1, 0x9e, 0xe4, 0x8f, 0x38, 0x57, 0xe2, 0xb6,
			0x9e, 0x93, 0x6d, 0x10, 0xaf, 0xbd, 0xc2, 0x98, 0x22, 0xe8, 0x5d, 0x02,
			0x66, 0x3d, 0x34, 0x6e, 0xf3, 0xe0, 0x9a, 0x84, 0x8b, 0x1d, 0x9c, 0xc0,
			0x4f, 0x4c, 0x4c, 0x6e, 0x3b, 0x3b, 0x0e, 0x56, 0xa0, 0x34, 0xe2, 0x33,
			0x4d, 0x34, 0xca, 0x08, 0xf8, 0x09, 0x7b, 0xe3, 0x07, 0xba, 0x41, 0xd0,
			0x20, 0xbc, 0x94, 0xf8, 0xc1, 0x93, 0x7f, 0xe8, 0x56, 0x44, 0xee, 0xb5,
			0x59, 0x2c, 0x2b, 0x5a, 0x21, 0x38, 0xf7, 0xde, 0xd9, 0xa5, 0xb4, 0x4b,
			0x20, 0x0c, 0x8b, 0x5b, 0xeb, 0x27, 0x59, 0x7c, 0x79, 0x0f, 0x94, 0xd6,
			0x60, 0xeb, 0x61, 0xe8, 0x24, 0x83, 0x91, 0xed, 0xc3, 0xae, 0x2d, 0x77,
			0x65, 0x6c, 0xbe, 0x83, 0x54, 0x27, 0x5b, 0x13,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(9, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_10(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x0e, 0xa4, 0x58, 0xcf, 0xf8, 0xbf, 0xd1, 0xdd, 0x8b, 0x1a, 0xdd, 0xcb,
			0xa9, 0xc0, 0x13, 0x17, 0xd5, 0x30, 0x39, 0xe5, 0x33, 0x10, 0x4e, 0x32,
			0xf9, 0x6e, 0x7d, 0x34, 0x2e, 0x6c, 0x7b, 0x9b,
		},
		{
			0x93, 0x5a, 0x4b, 0x66, 0xfc, 0x74, 0xc2, 0xa4, 0x87, 0x57, 0xa9, 0x9c,
			0x39, 0x9e, 0x64, 0xe3,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x6c, 0x5f, 0x37, 0x08, 0xe7, 0xb7, 0x14, 0xc4, 0xed, 0x13, 0x9b, 0x4f,
			0xa9, 0xe8, 0xc7, 0x63, 0xaf, 0x01, 0x77, 0x34, 0x84, 0x00, 0x51, 0x09,
			0xa8, 0x5e, 0x33, 0x65, 0x3b, 0xb0, 0xce, 0x98,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x37, 0x3a, 0x37, 0xaf, 0x84, 0xfd, 0xde, 0xc1, 0x36, 0x45, 0xa9, 0x76,
			0x8d, 0x6a, 0x78, 0x5a, 0xe5, 0xa2, 0x58, 0x9d, 0x64, 0xcd, 0x9b, 0x37,
			0x98, 0x0d, 0xde, 0x25, 0x41, 0x49, 0x92, 0x10, 0xc4, 0xf4, 0x08, 0x33,
			0x5d, 0xe1, 0xd5, 0x85, 0x34, 0x90, 0x64, 0xf3, 0xf5, 0x3a, 0x2b, 0x4c,
			0x5e, 0xc6, 0xdc, 0x2a, 0x09, 0x59, 0x1f, 0x99, 0xad, 0x9f, 0xad, 0x52,
			0x8a, 0xc8, 0x34, 0x74, 0x16, 0x4b, 0x45, 0x49, 0x7b, 0xf1, 0x67, 0xf8,
			0x1e, 0x66, 0xfa, 0x08, 0x46, 0x3f, 0xfe, 0xa9, 0x17, 0xf6, 0x89, 0x1e,
			0x48, 0xf1, 0x49, 0xfa, 0xfc, 0x20, 0x62, 0x2b, 0xb1, 0x17, 0x2f, 0x34,
			0x88, 0x6f, 0xeb, 0x45, 0xc2, 0x6f, 0xd4, 0x46, 0xa4, 0xa4, 0xe2, 0x89,
			0x1b, 0x4b, 0xc5, 0x94, 0x18, 0x68, 0x96, 0x14, 0x1a, 0xaa, 0xee, 0xb3,
			0x01, 0xb4, 0x9e, 0x7c, 0x1a, 0x26, 0xfe, 0xc7,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(10, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_11(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xbf, 0xb6, 0x8b, 0xe4, 0xce, 0x17, 0x56, 0xd2, 0x5b, 0xdf, 0xad, 0x5e,
			0x0c, 0x2f, 0x8b, 0xec, 0x29, 0x36, 0x09, 0x01, 0xcc, 0x4d, 0xa5, 0x1d,
			0x42, 0x3d, 0x15, 0x91, 0xcc, 0x57, 0xe1, 0xba,
		},
		{
			0x98, 0xaf, 0xe4, 0xbd, 0x19, 0x4c, 0x14, 0x3e, 0x09, 0x96, 0x80, 0xc5,
			0x04, 0xcc, 0xea, 0xab,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb9, 0x7c, 0xaf, 0x21, 0x0e, 0x82, 0x49, 0x8c, 0x34, 0x08, 0x79, 0x0d,
			0x41, 0xc3, 0x20, 0xdd, 0x4a, 0x72, 0x00, 0x77, 0x78, 0x38, 0x9b, 0x44,
			0xb7, 0xbc, 0x3c, 0x1c, 0x4b, 0x8c, 0x53, 0xf8,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x40, 0x9e, 0x0a, 0xa9, 0x49, 0xfb, 0x3b, 0x38, 0x23, 0x1b, 0xf8, 0x73,
			0x2e, 0x79, 0x59, 0xe9, 0x43, 0xa3, 0x38, 0xea, 0x39, 0x90, 0x26, 0xb7,
			0x44, 0xdf, 0x15, 0xcb, 0xfe, 0xff, 0x8d, 0x71, 0xb3, 0xda, 0x02, 0x3d,
			0xcc, 0xe0, 0x59, 0xa8, 0x8c, 0xf0, 0xd4, 0xb7, 0x47, 0x5f, 0x62, 0x8e,
			0x47, 0x64, 0xc8, 0xbe, 0xf1, 0x3c, 0x70, 0xcf, 0xbb, 0xbb, 0x6d, 0xa2,
			0xa1, 0x8a, 0xab, 0xca, 0xd9, 0x19, 0xdb, 0x09, 0xd0, 0x4f, 0xc5, 0x97,
			0x65, 0xed, 0xb1, 0x65, 0x14, 0x7c, 0x88, 0xdd, 0x47, 0x3a, 0x0f, 0x3c,
			0x5e, 0xe1, 0x92, 0x37, 0xca, 0x95, 0x56, 0x97, 0xe0, 0x01, 0xba, 0x65,
			0x4c, 0x5e, 0xe0, 0xbd, 0x26, 0x76, 0x1b, 0x49, 0x33, 0x31, 0x54, 0x42,
			0x6b, 0xc6, 0x32, 0x86, 0x29, 0x8a, 0x8b, 0xe6, 0x34, 0xfe, 0x0d, 0x72,
			0xcf, 0xde, 0xef, 0x0f, 0x3f, 0xc4, 0x8e, 0xca,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(11, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_12(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x4f, 0x68, 0x80, 0xa6, 0x46, 0x10, 0x00, 0x44, 0x63, 0x03, 0x1d, 0x67,
			0xd7, 0x92, 0x4f, 0xa4, 0x46, 0xc3, 0x91, 0x38, 0xd4, 0xd4, 0x10, 0x07,
			0xe8, 0xdf, 0x3d, 0x65, 0x69, 0x1a, 0x93, 0x67,
		},
		{
			0x6b, 0x33, 0xb2, 0xc1, 0x36, 0x00, 0xf4, 0xb1, 0xdf, 0x6c, 0xa3, 0xd1,
			0x96, 0x0e, 0x8d, 0xd4,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x57, 0xb8, 0x7b, 0x8c, 0x8f, 0x48, 0x31, 0x2b, 0x53, 0x33, 0xd4, 0x3b,
			0x36, 0x77, 0x30, 0xc0, 0xa5, 0xad, 0x47, 0x25, 0xa1, 0x67, 0x78, 0xfc,
			0xb5, 0x3f, 0xe1, 0x36, 0xd1, 0x36, 0xcb, 0xfd,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x73, 0xd0, 0xf3, 0x24, 0xed, 0x18, 0x6e, 0x2a, 0xd0, 0x6b, 0xd1, 0x80,
			0x0e, 0x26, 0x2b, 0xdb, 0xda, 0x79, 0xba, 0x54, 0xe6, 0x26, 0x76, 0x1b,
			0xd6, 0x0f, 0x74, 0xf4, 0x3e, 0x3b, 0xb6, 0x29, 0x58, 0xec, 0x1e, 0x2f,
			0x1d, 0x94, 0x0a, 0xf1, 0x63, 0xe1, 0xca, 0xdc, 0x12, 0x4e, 0x7e, 0xba,
			0xba, 0x2f, 0x72, 0xe6, 0x7e, 0xfd, 0x74, 0x6c, 0x7f, 0x6d, 0x0c, 0xad,
			0x53, 0xef, 0x03, 0xd8, 0x59, 0xd9, 0x3c, 0xff, 0x77, 0x8a, 0x32, 0xee,
			0x5b, 0xe1, 0x72, 0xfe, 0x7f, 0xdb, 0xdc, 0x23, 0x2d, 0xed, 0x36, 0x0d,
			0x70, 0x4a, 0x6f, 0xa0, 0xf7, 0x0b, 0xeb, 0xe9, 0x42, 0xe5, 0x64, 0x78,
			0x34, 0x54, 0x92, 0xf4, 0x9d, 0xc5, 0xc6, 0xfc, 0x34, 0x6b, 0x88, 0xa5,
			0x89, 0x47, 0xad, 0x25, 0x0e, 0x68, 0x8e, 0x8c, 0x62, 0x6f, 0xe1, 0xef,
			0xe7, 0x62, 0x46, 0x20, 0xe5, 0x71, 0x97, 0x6e,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(12, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_13(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xaa, 0xe3, 0x52, 0xe1, 0x11, 0x84, 0x32, 0x19, 0xca, 0xe8, 0xf7, 0x0e,
			0x7b, 0x8f, 0x6e, 0xb9, 0xbb, 0x53, 0xd2, 0x46, 0xcb, 0xec, 0x1e, 0x4f,
			0x07, 0xd4, 0x27, 0x57, 0x14, 0x32, 0x95, 0xb4,
		},
		{
			0xb8, 0x44, 0x85, 0xdc, 0xcd, 0x1b, 0xf9, 0x32, 0x10, 0xe3, 0x22, 0xea,
			0xfc, 0xbe, 0xbc, 0xd9,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xf9, 0x23, 0x7f, 0x00, 0xd7, 0x44, 0xd8, 0xfb, 0xff, 0x21, 0xb9, 0xd0,
			0x04, 0x3c, 0x25, 0x8e, 0x87, 0x31, 0x81, 0x7e, 0x6a, 0x5f, 0xb7, 0xb4,
			0xbf, 0x50, 0x11, 0x68, 0x0e, 0x5b, 0xc6, 0x42,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xcf, 0xb2, 0x8b, 0x93, 0x52, 0x2c, 0x7d, 0x61, 0xd8, 0xd3, 0xce, 0x3f,
			0x08, 0x0e, 0x43, 0x5e, 0x4c, 0x83, 0xc7, 0xe1, 0x3a, 0x9d, 0xab, 0x78,
			0x8d, 0xb8, 0xfe, 0xf0, 0x40, 0x72, 0x67, 0xa1, 0x4f, 0xbc, 0x93, 0x24,
			0xe0, 0x90, 0xe2, 0x4d, 0xf5, 0x49, 0x1f, 0xed, 0xfa, 0x81, 0x11, 0x68,
			0x69, 0x98, 0x39, 0x38, 0xd4, 0xd4, 0xd7, 0x32, 0x4a, 0x31, 0x0c, 0x3a,
			0xf3, 0x3a, 0x6f, 0x79, 0x38, 0xf6, 0x02, 0xc5, 0xe4, 0xe6, 0x3f, 0x17,
			0x71, 0xcd, 0xaa, 0xbd, 0xab, 0x07, 0x82, 0xb5, 0xaf, 0xfb, 0x54, 0xeb,
			0x53, 0x04, 0x7c, 0x10, 0x9a, 0x96, 0x06, 0x73, 0x9d, 0xd0, 0x06, 0x5b,
			0xd2, 0x1e, 0xca, 0x33, 0x13, 0x29, 0x86, 0x55, 0x48, 0x78, 0x35, 0x4f,
			0x5f, 0x9f, 0x85, 0x2e, 0x67, 0x4d, 0xd6, 0x90, 0x16, 0x3b, 0x0f, 0xf7,
			0x4c, 0x7a, 0x25, 0xe6, 0xba, 0xe8, 0xce, 0x39,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(13, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_14(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x58, 0x9e, 0x79, 0xe3, 0x39, 0xb7, 0xd2, 0xa1, 0xb8, 0x79, 0xf0, 0xb0,
			0xe1, 0xa7, 0xd1, 0xad, 0x24, 0x74, 0xea, 0xa8, 0x02, 0x5b, 0x07, 0x0f,
			0x1f, 0xfa, 0x87, 0x7b, 0x71, 0x24, 0xd4, 0xff,
		},
		{
			0x09, 0x61, 0xed, 0x64, 0xdb, 0xd6, 0x20, 0x65, 0xd9, 0x6e, 0x75, 0xde,
			0x6d, 0x2f, 0xf9, 0xd6,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xe9, 0x28, 0x38, 0x8d, 0x3a, 0xf4, 0x8c, 0x29, 0x68, 0x52, 0x7a, 0x4d,
			0x2f, 0x9c, 0x26, 0x26, 0xfb, 0xc3, 0xf3, 0xf5, 0xa5, 0xd8, 0x4e, 0x05,
			0x83, 0xab, 0x6f, 0x78, 0xe7, 0xf8, 0xb0, 0x81,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xfc, 0xe6, 0xce, 0xd1, 0xec, 0xf4, 0x74, 0xd1, 0x81, 0xab, 0x33, 0x1f,
			0x79, 0xc3, 0xd2, 0xcc, 0x8a, 0x76, 0x8e, 0xc2, 0x81, 0x8d, 0xe5, 0xb3,
			0xfc, 0x7c, 0xf4, 0x18, 0x32, 0x27, 0x16, 0xd6, 0xa6, 0x85, 0x37, 0x33,
			0x56, 0x1a, 0x49, 0x7c, 0x0c, 0x25, 0xcb, 0x28, 0x8d, 0x2c, 0x9f, 0xcf,
			0xbc, 0xa8, 0x91, 0xba, 0xfd, 0x5a, 0x83, 0x4c, 0x85, 0xf3, 0x60, 0x3f,
			0x40, 0x2a, 0xcf, 0x1a, 0x7b, 0x1e, 0xa9, 0x2d, 0xb8, 0x47, 0xed, 0x5c,
			0x25, 0x2a, 0x86, 0x2a, 0xd4, 0xab, 0x5e, 0x25, 0x97, 0x15, 0xf1, 0xfc,
			0x81, 0xda, 0x67, 0xf5, 0x23, 0x0b, 0xf8, 0xbe, 0x50, 0xee, 0x80, 0x69,
			0x75, 0x80, 0x95, 0xf7, 0xd0, 0xe5, 0x59, 0xe0, 0x3f, 0x2c, 0x60, 0x72,
			0x29, 0x0e, 0x61, 0x79, 0x44, 0x58, 0x43, 0x76, 0x09, 0xe4, 0x73, 0xeb,
			0x66, 0x58, 0x0c, 0xdd, 0xaa, 0xd1, 0x9b, 0x71,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(14, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_15(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x71, 0x42, 0x77, 0xd4, 0x08, 0xad, 0x87, 0xfd, 0xe3, 0x17, 0xf0, 0xa9,
			0x47, 0x32, 0xfc, 0xe6, 0x2f, 0x13, 0x52, 0xbd, 0xc9, 0x09, 0x36, 0x67,
			0x3b, 0x4f, 0x1d, 0xaa, 0x09, 0x25, 0xaa, 0x26,
		},
		{
			0xd1, 0x65, 0x82, 0xa9, 0x9f, 0x23, 0x01, 0x0b, 0x42, 0x48, 0xb8, 0x8d,
			0x86, 0x48, 0x54, 0x19,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xbd, 0x9f, 0xc7, 0xcb, 0x2f, 0xd5, 0x06, 0x3b, 0x2c, 0x3c, 0x0c, 0x4f,
			0x34, 0x6a, 0xd2, 0xe3, 0x87, 0x93, 0x71, 0xa9, 0xc8, 0x05, 0xe5, 0x9b,
			0x9f, 0x2c, 0xd2, 0xcc, 0x2a, 0x40, 0x89, 0x4f,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x62, 0xef, 0x7a, 0x43, 0x12, 0x88, 0x25, 0x2e, 0x0d, 0x73, 0x6c, 0x1d,
			0x4e, 0x36, 0xcc, 0x9a, 0xc3, 0x71, 0x07, 0xdc, 0xd0, 0xd0, 0xe9, 0x71,
			0xa2, 0x24, 0x44, 0xa4, 0xad, 0xae, 0x73, 0xa4, 0x1e, 0xff, 0x0b, 0x11,
			0xc8, 0x62, 0x5e, 0x11, 0x8d, 0xbc, 0x92, 0x26, 0x14, 0x2f, 0xd0, 0xa6,
			0xaa, 0x10, 0xac, 0x9b, 0x19, 0x09, 0x19, 0xbd, 0xa4, 0x4e, 0x72, 0x48,
			0xd6, 0xc8, 0x88, 0x74, 0x61, 0x2a, 0xbd, 0x77, 0xfb, 0x37, 0x16, 0xea,
			0x51, 0x5a, 0x2d, 0x56, 0x32, 0x37, 0xc4, 0x46, 0xe2, 0xa2, 0x82, 0xe7,
			0xc3, 0xb0, 0xa3, 0xae, 0xf2, 0x7d, 0x34, 0x27, 0xcc, 0x7d, 0x0a, 0x7d,
			0x38, 0x71, 0x46, 0x59, 0xc3, 0x40, 0x1d, 0xbc, 0x91, 0xd3, 0x59, 0x51,
			0x59, 0x31, 0x8e, 0xbc, 0xa0, 0x1a, 0xe7, 0xd7, 0xfd, 0x1c, 0x89, 0xf6,
			0xad, 0x6b, 0x60, 0x41, 0x73, 0xb0, 0xc7, 0x44,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(15, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_16(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xfa, 0x0e, 0xe1, 0xfe, 0x39, 0xc7, 0xc3, 0x90, 0xaa, 0x94, 0x15, 0x9d,
			0x0d, 0xe9, 0x75, 0x64, 0x34, 0x2b, 0x59, 0x17, 0x77, 0xf3, 0xe5, 0xf6,
			0xa4, 0xba, 0x2a, 0xea, 0x34, 0x2e, 0xc8, 0x40,
		},
		{
			0xdd, 0x08, 0x20, 0x65, 0x5c, 0xb2, 0xff, 0xdb, 0x0d, 0xa9, 0xe9, 0x31,
			0x0a, 0x67, 0xc9, 0xe5,
		},
		{
			0xf2, 0xe5, 0x8f, 0xe6, 0x0a, 0x3a, 0xfc, 0x59, 0xda, 0xd3, 0x75, 0x95,
			0x41, 0x5f, 0xfd, 0x31, 0x8c, 0xcf, 0x69, 0xd6, 0x77, 0x80, 0xf6, 0xfa,
			0x07, 0x97, 0xdc, 0x9a, 0xa4, 0x3e, 0x14, 0x4c,
		},
		{
			0xe0, 0x62, 0x9b, 0x6d, 0x79, 0x75, 0xdd, 0xfa, 0x96, 0xa3, 0x99, 0x64,
			0x87, 0x40, 0xe6, 0x0f, 0x1f, 0x95, 0x57, 0xdc, 0x58, 0xb3, 0xd7, 0x41,
			0x5f, 0x9b, 0xa9, 0xd4, 0xdb, 0xb5, 0x01, 0xf6,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xf9, 0x2d, 0x4c, 0xf9, 0x9a, 0x53, 0x5b, 0x20, 0x22, 0x2a, 0x52, 0xa6,
			0x8d, 0xb0, 0x4c, 0x5a, 0xf6, 0xf5, 0xff, 0xc7, 0xb6, 0x6a, 0x47, 0x3a,
			0x37, 0xa2, 0x56, 0xbd, 0x8d, 0x29, 0x8f, 0x9b, 0x4a, 0xa4, 0xaf, 0x7e,
			0x8d, 0x18, 0x1e, 0x02, 0x36, 0x79, 0x03, 0xf9, 0x3b, 0xdb, 0x74, 0x4c,
			0x6c, 0x2f, 0x3f, 0x34, 0x72, 0x62, 0x6b, 0x40, 0xce, 0x9b, 0xd6, 0xa7,
			0x0e, 0x7b, 0x8f, 0x93, 0x99, 0x2a, 0x16, 0xa7, 0x6f, 0xab, 0x6b, 0x5f,
			0x16, 0x25, 0x68, 0xe0, 0x8e, 0xe6, 0xc3, 0xe8, 0x04, 0xae, 0xfd, 0x95,
			0x2d, 0xdd, 0x3a, 0xcb, 0x79, 0x1c, 0x50, 0xf2, 0xad, 0x69, 0xe9, 0xa0,
			0x40, 0x28, 0xa0, 0x6a, 0x9c, 0x01, 0xd3, 0xa6, 0x2a, 0xca, 0x2a, 0xaf,
			0x6e, 0xfe, 0x69, 0xed, 0x97, 0xa0, 0x16, 0x21, 0x3a, 0x2d, 0xd6, 0x42,
			0xb4, 0x88, 0x67, 0x64, 0x07, 0x2d, 0x9c, 0xbe,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(16, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_17(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xcf, 0xf7, 0x2f, 0x34, 0x51, 0x15, 0x37, 0x6a, 0x57, 0xf4, 0xdb, 0x8a,
			0x5c, 0x9f, 0x64, 0x05, 0x3e, 0x73, 0x79, 0x17, 0x1a, 0x5a, 0x1e, 0x81,
			0xe8, 0x2a, 0xad, 0x34, 0x48, 0xd1, 0x7d, 0x44,
		},
		{
			0xd1, 0xe9, 0x71, 0xec, 0x79, 0x5d, 0x09, 0x8b, 0x3d, 0xae, 0x14, 0xff,
			0xcb, 0xee, 0xcf, 0xd9,
		},
		{
			0x6e, 0xc0, 0xc7, 0x98, 0xc2, 0x40, 0xf2, 0x27, 0x40, 0xca, 0xd7, 0xe2,
			0x7b, 0x41, 0xf5, 0xe4, 0x2d, 0xcc, 0xaf, 0x66, 0xde, 0xf3, 0xb7, 0xf3,
			0x41, 0xc4, 0xd8, 0x27, 0x29, 0x4f, 0x83, 0xc9,
		},
		{
			0x45, 0xec, 0x80, 0xf0, 0xc0, 0x0c, 0xad, 0x0f, 0xf0, 0xb7, 0x61, 0x6d,
			0x2a, 0x93, 0x0a, 0xf3, 0xf5, 0xcf, 0x23, 0xcd, 0x61, 0xbe, 0x7f, 0xbf,
			0x7c, 0x65, 0xbe, 0x00, 0x31, 0xe9, 0x3e, 0x38,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x17, 0xa7, 0x90, 0x1e, 0x25, 0x50, 0xde, 0x08, 0x8f, 0x47, 0x25, 0x18,
			0xd3, 0x77, 0xcc, 0x4c, 0xc6, 0x97, 0x9f, 0x4a, 0x64, 0xf4, 0x97, 0x5c,
			0x74, 0x34, 0x42, 0x15, 0xe4, 0x80, 0x7a, 0x12, 0x34, 0xee, 0xfe, 0xf9,
			0x9f, 0x64, 0xcb, 0x8a, 0xbc, 0x3f, 0xb8, 0x62, 0x09, 0xf6, 0xfc, 0x7d,
			0xdd, 0x03, 0xe9, 0x4f, 0x83, 0x74, 0x6c, 0x5a, 0xbe, 0x53, 0x60, 0xcd,
			0xde, 0x4f, 0x25, 0x25, 0xcc, 0xf7, 0x16, 0x7e, 0x6f, 0x0b, 0xef, 0xae,
			0x05, 0xb3, 0x8f, 0xd6, 0x08, 0x9a, 0x2a, 0xb8, 0x37, 0x19, 0x87, 0x4c,
			0xe8, 0xf6, 0x70, 0x48, 0x0d, 0x5f, 0x3e, 0xd9, 0xbf, 0x40, 0x53, 0x8a,
			0x15, 0xaa, 0xad, 0x11, 0x2d, 0xb1, 0x61, 0x8a, 0x58, 0xb1, 0x06, 0x87,
			0xb6, 0x88, 0x75, 0xf0, 0x0f, 0x13, 0x9a, 0x72, 0xbd, 0xf0, 0x43, 0xf7,
			0x36, 0xe4, 0xa3, 0x20, 0xc0, 0x6e, 0xfd, 0x2c,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(17, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_18(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xb7, 0x09, 0x9b, 0x06, 0xfc, 0x7a, 0x8a, 0x74, 0xc5, 0x82, 0x19, 0x72,
			0x9d, 0xb6, 0xb0, 0xf7, 0x80, 0xd7, 0xb4, 0xfa, 0x30, 0x7b, 0xc3, 0xd3,
			0xf9, 0xf2, 0x2b, 0xfb, 0x76, 0x35, 0x96, 0xa3,
		},
		{
			0xb8, 0x77, 0x20, 0x59, 0xa1, 0x35, 0xa6, 0xb6, 0x1d, 0xa7, 0x2f, 0x37,
			0x54, 0x11, 0xde, 0x26,
		},
		{
			0x2a, 0xc1, 0xbf, 0xb2, 0x4e, 0x0b, 0x8c, 0x6a, 0xc2, 0x80, 0x3e, 0x89,
			0x26, 0x18, 0x22, 0xb7, 0xf7, 0x2a, 0x03, 0x20, 0xdf, 0x2b, 0x19, 0x91,
			0x71, 0xb7, 0x9b, 0xcb, 0xdb, 0x40, 0xb7, 0x19,
		},
		{
			0x9a, 0xec, 0x4f, 0x56, 0xec, 0x5e, 0x96, 0xfb, 0xd9, 0x60, 0x48, 0xb9,
			0xa6, 0x3a, 0xc8, 0xd0, 0x47, 0xae, 0xdb, 0xbe, 0xea, 0x77, 0x12, 0xe2,
			0x41, 0x13, 0x3b, 0x1a, 0x35, 0x7e, 0xcf, 0xc4,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x0e, 0x1f, 0x2b, 0xfe, 0xf7, 0x78, 0xf5, 0xe5, 0xbe, 0x67, 0x1e, 0xcb,
			0x49, 0x71, 0x62, 0x4e, 0xc7, 0x84, 0xed, 0x27, 0x32, 0xab, 0xc4, 0xfb,
			0xb9, 0x8a, 0x8b, 0x48, 0x2f, 0xb6, 0x87, 0x37, 0xdf, 0x91, 0xfd, 0x15,
			0xac, 0xfa, 0xd2, 0x95, 0x14, 0x03, 0xac, 0x77, 0xc5, 0xca, 0x3e, 0xdf,
			0xfc, 0x1e, 0x03, 0x39, 0x8a, 0xe6, 0xcf, 0x6a, 0xc2, 0x4a, 0x91, 0x67,
			0x8d, 0xb5, 0xc7, 0x29, 0x0a, 0xbc, 0x3f, 0xa0, 0x01, 0xaa, 0x02, 0xd5,
			0x03, 0x99, 0x32, 0x6f, 0x85, 0xd2, 0xb8, 0x94, 0x21, 0x99, 0xa1, 0x57,
			0x5f, 0x67, 0x46, 0x36, 0x47, 0x40, 0xa5, 0x91, 0x05, 0x52, 0xc6, 0x39,
			0x80, 0x4d, 0x75, 0x30, 0xc0, 0xd4, 0x13, 0x39, 0x34, 0x5a, 0x58, 0xff,
			0x00, 0x80, 0xec, 0xcf, 0x17, 0x11, 0x89, 0x51, 0x92, 0xa3, 0x81, 0x7a,
			0x8d, 0xc3, 0xf0, 0x0f, 0x28, 0xcc, 0x10, 0xcc,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(18, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_19(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x7b, 0xa0, 0x2a, 0x73, 0x4c, 0x87, 0x44, 0xb1, 0x5e, 0xf8, 0xb4, 0x07,
			0x4f, 0xe6, 0x39, 0xb3, 0x2e, 0x44, 0x31, 0x76, 0x2a, 0xb5, 0xb7, 0xcd,
			0x4d, 0x5d, 0xf6, 0x75, 0xea, 0x90, 0x67, 0x2b,
		},
		{
			0x8a, 0x42, 0x4f, 0x32, 0x10, 0x86, 0x07, 0xc8, 0xf1, 0xf4, 0x5d, 0x97,
			0xf5, 0x00, 0xee, 0x12,
		},
		{
			0x3a, 0xd6, 0x27, 0x43, 0x3f, 0x46, 0x51, 0x87, 0xc4, 0x81, 0x41, 0xe3,
			0x0c, 0x26, 0x78, 0x10, 0x60, 0x91, 0xe7, 0xa6, 0x80, 0x22, 0x9a, 0x53,
			0x4b, 0x85, 0x1b, 0x8d, 0x46, 0xfe, 0xb9, 0x57,
		},
		{
			0xd8, 0xf0, 0x2b, 0x59, 0xb6, 0xa3, 0xdd, 0x27, 0x6b, 0xc6, 0x9c, 0xba,
			0x68, 0xef, 0xcf, 0x11, 0xab, 0x83, 0xea, 0xd1, 0x39, 0x7a, 0xfd, 0x98,
			0x41, 0x78, 0x6b, 0xd1, 0xbb, 0x5d, 0xa9, 0x7a,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x1f, 0xb9, 0x11, 0x86, 0xba, 0x4b, 0x44, 0x59, 0xd9, 0x94, 0xb4, 0xb9,
			0xf4, 0xca, 0x25, 0x2c, 0x7b, 0xe6, 0x29, 0x4d, 0x6c, 0xdb, 0x5f, 0xe5,
			0x6f, 0x8f, 0xf7, 0x84, 0xd4, 0xb1, 0x90, 0xa1, 0xc6, 0x45, 0x6e, 0x0a,
			0x41, 0x22, 0x3b, 0xbb, 0xdf, 0x83, 0xed, 0x8e, 0x7c, 0xfb, 0xfa, 0x76,
			0x5d, 0x9d, 0x8b, 0xc7, 0xea, 0x5f, 0x4d, 0x79, 0xea, 0x7e, 0xcc, 0xb4,
			0x92, 0x80, 0x81, 0xa2, 0x1d, 0xe4, 0xcc, 0xa3, 0x66, 0x20, 0xd6, 0x26,
			0x7f, 0x55, 0xd9, 0xa3, 0x52, 0xb7, 0x6f, 0xc0, 0xa5, 0x73, 0x75, 0x88,
			0x41, 0x12, 0xc3, 0x1f, 0x65, 0xff, 0x28, 0xe7, 0x6d, 0x31, 0x56, 0x98,
			0xc2, 0x9e, 0x6c, 0x4c, 0x05, 0xcb, 0x58, 0xb0, 0xa0, 0x7a, 0xe6, 0x61,
			0x43, 0xb4, 0xab, 0xc7, 0x8b, 0x9d, 0x25, 0xc7, 0x8b, 0x41, 0x21, 0xe1,
			0xe4, 0x5b, 0xef, 0x1a, 0x6c, 0x17, 0x93, 0xe2,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(19, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_20(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x9a, 0x88, 0x65, 0xdf, 0xe0, 0x53, 0xae, 0x77, 0xcb, 0x6a, 0x93, 0x65,
			0xb8, 0x8f, 0x34, 0xee, 0xc1, 0x7e, 0xa5, 0xcb, 0xfb, 0x0b, 0x1f, 0x04,
			0xd1, 0x45, 0x9e, 0x7f, 0xa9, 0xc4, 0xf3, 0xcb,
		},
		{
			0x18, 0x0c, 0x0a, 0x74, 0xda, 0x3e, 0xc4, 0x64, 0xdf, 0x11, 0xfa, 0xc1,
			0x72, 0xd1, 0xc6, 0x32,
		},
		{
			0x33, 0x63, 0x72, 0xec, 0x82, 0xd0, 0xd6, 0x8b, 0xef, 0xad, 0x83, 0x69,
			0x19, 0x66, 0xef, 0x6f, 0xfc, 0x65, 0x10, 0x53, 0x88, 0xeb, 0x2d, 0x6e,
			0xed, 0x82, 0x6c, 0x22, 0x85, 0x03, 0x7c, 0x77,
		},
		{
			0x75, 0xb9, 0x51, 0x08, 0xef, 0xf1, 0xfa, 0xbe, 0x83, 0x61, 0x3e, 0x1c,
			0x4d, 0xe5, 0x75, 0xe7, 0x2a, 0x5c, 0xdc, 0x4b, 0xb9, 0x31, 0x1d, 0xd0,
			0x06, 0xf9, 0x71, 0xa0, 0x52, 0x38, 0x66, 0x92,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x3c, 0x68, 0x3f, 0x6d, 0x4f, 0x8f, 0x5a, 0x40, 0x18, 0xd0, 0x16, 0x33,
			0xdf, 0xee, 0x74, 0x26, 0x6a, 0xaa, 0x68, 0xed, 0x6f, 0xc6, 0x49, 0xe8,
			0x1b, 0x64, 0xdf, 0xdf, 0x5f, 0x75, 0xe7, 0x5d, 0x5c, 0x05, 0x8d, 0x66,
			0xcf, 0x5f, 0xd0, 0x1a, 0x4f, 0x14, 0x3a, 0x6f, 0xf6, 0x95, 0x51, 0x7a,
			0x4a, 0x43, 0xbd, 0x3a, 0xdf, 0xd1, 0xfb, 0x2c, 0x28, 0xba, 0x9a, 0x41,
			0x06, 0x31, 0x40, 0xbe, 0xdb, 0xff, 0xdb, 0x4d, 0x21, 0xb1, 0xac, 0xe1,
			0x55, 0x0d, 0x59, 0x20, 0x9e, 0xc6, 0x1f, 0x1e, 0x2d, 0xba, 0xcb, 0x2a,
			0x91, 0x16, 0xa7, 0x9c, 0xb1, 0x41, 0x0b, 0xf2, 0xde, 0xca, 0x52, 0x18,
			0x08, 0x0a, 0xac, 0xd9, 0xc6, 0x8e, 0x1d, 0x65, 0x57, 0x72, 0x1a, 0x89,
			0x13, 0xe2, 0x3f, 0x61, 0x7e, 0x30, 0xf2, 0xe5, 0x94, 0xf6, 0x12, 0x67,
			0xd5, 0xed, 0x81, 0x46, 0x4e, 0xe7, 0x30, 0xb2,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(20, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_21(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x22, 0xc1, 0xaf, 0x2f, 0x2a, 0x4c, 0x88, 0x5f, 0x06, 0x98, 0x85, 0x67,
			0xda, 0x9f, 0xc9, 0x0f, 0x34, 0xf8, 0x0f, 0x6d, 0xd5, 0x10, 0x1c, 0x28,
			0x1b, 0xee, 0xf4, 0x97, 0xa6, 0xa1, 0xb2, 0xf8,
		},
		{
			0x3f, 0xaf, 0xde, 0xcf, 0x79, 0xa4, 0x17, 0x48, 0x01, 0xf1, 0x33, 0x13,
			0x16, 0x29, 0x03, 0x7b,
		},
		{
			0x80, 0x32, 0x7d, 0xac, 0x48, 0x61, 0x11, 0xb8, 0xa8, 0xb2, 0xc8, 0xe8,
			0x38, 0x1f, 0xb2, 0xd7, 0x13, 0xa6, 0x76, 0x95, 0xc2, 0xe6, 0x60, 0xb2,
			0xb0, 0xd4, 0xaf, 0x69, 0x6c, 0xc3, 0xe1, 0xde,
		},
		{
			0xf9, 0x5a, 0x0e, 0x4b, 0xd2, 0x4f, 0x0e, 0x2e, 0x9e, 0x44, 0x4f, 0x51,
			0x1b, 0x76, 0x32, 0x86, 0x8e, 0xad, 0x0d, 0x5b, 0xb3, 0x84, 0x67, 0x71,
			0x26, 0x4e, 0x03, 0xf8, 0xab, 0x8e, 0xd0, 0x74,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x77, 0xa7, 0xfe, 0xa2, 0xf3, 0x5a, 0x18, 0x8f, 0x6d, 0x1b, 0xfd, 0xd4,
			0x9b, 0x56, 0x9d, 0x8c, 0x45, 0xe2, 0xdd, 0x43, 0x1d, 0x35, 0xa1, 0x8c,
			0x6f, 0x43, 0x2c, 0x72, 0x4f, 0x1e, 0x33, 0xae, 0x92, 0xcb, 0x89, 0xa9,
			0xcf, 0x91, 0x51, 0x9e, 0x50, 0x70, 0x5a, 0x53, 0x19, 0x9f, 0x5b, 0x57,
			0x2d, 0xc8, 0x5c, 0x1a, 0xef, 0x8f, 0x28, 0xfb, 0x52, 0xdc, 0x79, 0x86,
			0x22, 0x8f, 0x66, 0x95, 0x4d, 0x54, 0xed, 0xa8, 0x4a, 0x86, 0x96, 0x2c,
			0xf2, 0x5c, 0xf7, 0x65, 0xbd, 0x99, 0x49, 0x87, 0x63, 0x49, 0x29, 0x1b,
			0x1a, 0xae, 0x5f, 0x88, 0xfc, 0xf4, 0xb3, 0x76, 0x91, 0x2d, 0x20, 0x5a,
			0xdd, 0x4f, 0x53, 0xb2, 0x77, 0x0c, 0x65, 0x79, 0x46, 0xc0, 0xd8, 0x24,
			0x28, 0x1f, 0x44, 0x15, 0x09, 0x15, 0x3f, 0x48, 0x35, 0x6d, 0x9d, 0x43,
			0xf8, 0xa9, 0x27, 0xe0, 0x69, 0x3d, 0xb8, 0xfc,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(21, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_22(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xd0, 0x84, 0x0e, 0x3a, 0x8d, 0x62, 0x9d, 0x5b, 0x88, 0x3d, 0x33, 0xe0,
			0x53, 0xa3, 0x41, 0xb2, 0x1c, 0x67, 0x4e, 0x67, 0xe1, 0x99, 0x9f, 0x06,
			0x8c, 0x49, 0x7e, 0xcf, 0xaa, 0xbf, 0xd6, 0xf6,
		},
		{
			0x07, 0x1d, 0xe7, 0x24, 0x4e, 0xcb, 0x2f, 0xdf, 0x7a, 0xb2, 0x7f, 0x2d,
			0x84, 0xaa, 0x7b, 0x7a,
		},
		{
			0x90, 0xd6, 0x09, 0x52, 0x7f, 0xad, 0x96, 0xff, 0xe6, 0x4a, 0xb1, 0x53,
			0x86, 0x03, 0x46, 0xf3, 0xd2, 0x37, 0xc8, 0x94, 0x05, 0x55, 0xae, 0x17,
			0xb4, 0x78, 0x42, 0xd8, 0x2d, 0x3b, 0x09, 0x43,
		},
		{
			0x1d, 0xd1, 0xa8, 0xb5, 0x98, 0x56, 0xc4, 0x9a, 0x38, 0x8f, 0x59, 0x4c,
			0x5f, 0x42, 0xcc, 0x2e, 0x4a, 0x56, 0xb3, 0xcc, 0xb8, 0xa6, 0x5e, 0x70,
			0x66, 0xe4, 0x4c, 0x12, 0xf4, 0x34, 0x4d, 0x50,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x7a, 0xb2, 0x8a, 0x9b, 0x2d, 0x3a, 0xe9, 0x99, 0x19, 0x55, 0x53, 0xe6,
			0x55, 0x0c, 0xce, 0xd4, 0xc2, 0xda, 0xcc, 0xbe, 0x7e, 0xc9, 0xdc, 0xbb,
			0x0d, 0x46, 0x7f, 0xab, 0xba, 0x18, 0x5b, 0x72, 0x7f, 0xbf, 0xd9, 0x83,
			0x02, 0x42, 0xcd, 0x09, 0x8f, 0x4d, 0xb3, 0xcf, 0x4a, 0x85, 0xe8, 0xbf,
			0x8e, 0x8d, 0x59, 0x74, 0xb6, 0x2b, 0x28, 0x55, 0x09, 0x22, 0xb3, 0x2e,
			0xd5, 0xbf, 0xc1, 0xa5, 0x22, 0xb6, 0x60, 0x5c, 0xf9, 0x3b, 0xf8, 0xd9,
			0x0b, 0xde, 0xc1, 0xc5, 0xb9, 0xe5, 0x9c, 0x6f, 0xc3, 0x7a, 0x81, 0x7d,
			0x43, 0x70, 0x68, 0xa8, 0x72, 0x54, 0xbe, 0x1f, 0x7c, 0x46, 0x18, 0xad,
			0xa4, 0x6f, 0xbc, 0x3a, 0x2e, 0xfb, 0x02, 0xe4, 0x45, 0x24, 0xe2, 0x1d,
			0x91, 0xbe, 0x75, 0x34, 0xcf, 0x05, 0xfb, 0xfd, 0x85, 0x83, 0x04, 0xb7,
			0x06, 0xd6, 0xa9, 0x1e, 0xa1, 0xcc, 0x6a, 0xd5,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(22, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_23(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x2e, 0x2d, 0xd5, 0x68, 0x69, 0x10, 0x44, 0x92, 0x76, 0x7a, 0x59, 0x77,
			0x86, 0x52, 0x83, 0x19, 0x19, 0xe1, 0xc8, 0xb9, 0x70, 0xf8, 0x4e, 0x82,
			0x4a, 0xe4, 0x11, 0x65, 0x97, 0xa0, 0xab, 0x7f,
		},
		{
			0x01, 0xc4, 0x2a, 0x7e, 0x98, 0x36, 0x41, 0xde, 0x46, 0xc8, 0x2f, 0xd0,
			0x9b, 0x4f, 0x2f, 0x76,
		},
		{
			0xbc, 0xd9, 0xe1, 0x50, 0x8f, 0xcc, 0x22, 0x82, 0x0a, 0x8b, 0xe0, 0x71,
			0x80, 0xfe, 0xa5, 0x04, 0x53, 0x67, 0x33, 0x3b, 0x56, 0x9e, 0x11, 0x1b,
			0x01, 0x1c, 0xd5, 0x7d, 0xc1, 0x85, 0x87, 0x65,
		},
		{
			0x73, 0x06, 0x50, 0x7c, 0xd3, 0xca, 0x7e, 0xec, 0x66, 0x7e, 0x64, 0x0d,
			0x27, 0x0c, 0xfb, 0xb0, 0x33, 0x06, 0x3d, 0x97, 0x52, 0x0b, 0x6b, 0x7e,
			0x38, 0xff, 0x3c, 0xea, 0x0e, 0x79, 0xd1, 0x2b,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb9, 0x15, 0x72, 0x6c, 0x7b, 0x8c, 0x5d, 0xc3, 0x97, 0x5f, 0x1a, 0x33,
			0x46, 0x84, 0xb9, 0x73, 0xab, 0xf6, 0xa9, 0x49, 0x5d, 0x93, 0x00, 0x88,
			0xcf, 0x5d, 0x07, 0x15, 0x48, 0xe4, 0xfd, 0x29, 0xa6, 0x7b, 0x55, 0xcc,
			0x56, 0x1e, 0xd6, 0x94, 0x9a, 0xd2, 0x81, 0x50, 0xa9, 0xfb, 0x43, 0x07,
			0xc1, 0xfa, 0x5f, 0x78, 0x3a, 0x7e, 0xa8, 0x72, 0xe8, 0xd7, 0xc7, 0xe6,
			0x7f, 0xf0, 0xc2, 0x90, 0x60, 0x81, 0xee, 0x91, 0x57, 0x37, 0xd8, 0x13,
			0xc2, 0x5b, 0xe5, 0xc3, 0x0b, 0x95, 0x2a, 0x36, 0xf3, 0x93, 0xe6, 0xba,
			0xa5, 0x6a, 0xb0, 0x1a, 0xdc, 0x2b, 0x47, 0x76, 0xad, 0x7b, 0x5d, 0x03,
			0x6a, 0x53, 0x65, 0x98, 0x77, 0xc7, 0xa4, 0xe5, 0x22, 0x0a, 0x89, 0x7d,
			0x6c, 0x07, 0x99, 0xaf, 0x37, 0xbe, 0xee, 0xd9, 0x11, 0x73, 0xfb, 0xe9,
			0xc6, 0x13, 0xc3, 0xb6, 0xb9, 0xbb, 0x28, 0xe5,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(23, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_24(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xd1, 0xaa, 0xb0, 0xf1, 0x6b, 0xd4, 0x7a, 0x5c, 0xcd, 0x67, 0xc2, 0x2e,
			0x09, 0x4d, 0xaa, 0x37, 0x35, 0xea, 0xe2, 0x1a, 0xa5, 0x7f, 0x0b, 0xcd,
			0x9e, 0x05, 0x3d, 0x9d, 0x0d, 0x54, 0x5c, 0xb8,
		},
		{
			0x19, 0x93, 0x10, 0xdf, 0xe1, 0xb0, 0x12, 0x65, 0xb8, 0xc0, 0xd2, 0xb4,
			0x6d, 0x6c, 0x7c, 0x9f,
		},
		{
			0x62, 0x5b, 0x4b, 0x8f, 0x4d, 0xe7, 0x2e, 0xa9, 0xcb, 0x6f, 0x70, 0x55,
			0x63, 0x22, 0xdc, 0x2a, 0x19, 0xd6, 0xb2, 0xb3, 0x2d, 0xe6, 0x23, 0xf5,
			0x57, 0xe4, 0x19, 0xa0, 0x84, 0xba, 0x60, 0xfd,
		},
		{
			0xf5, 0x0c, 0xab, 0xae, 0x4e, 0x06, 0x0f, 0x39, 0x71, 0x09, 0x6b, 0x78,
			0xe5, 0x50, 0xcd, 0xa2, 0x83, 0x7a, 0x26, 0xa6, 0x93, 0xd9, 0x05, 0xdb,
			0x2d, 0x99, 0x2d, 0x58, 0x9b, 0x26, 0x8f, 0x44,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x98, 0x7e, 0x1f, 0xdf, 0xe0, 0x04, 0xc6, 0x19, 0xcf, 0x1e, 0x90, 0x34,
			0x57, 0x67, 0x07, 0xec, 0xcd, 0x84, 0x94, 0x00, 0xe1, 0x9c, 0x87, 0xa1,
			0xfe, 0xf5, 0xb0, 0x17, 0x9e, 0xc5, 0x1c, 0x42, 0xa2, 0xf8, 0xc4, 0x5d,
			0x79, 0x42, 0xd0, 0x02, 0x3a, 0x02, 0x3c, 0x89, 0xf1, 0x88, 0xb2, 0x63,
			0x43, 0x62, 0x70, 0x39, 0x85, 0x69, 0x53, 0x69, 0x86, 0x33, 0x22, 0xf5,
			0x86, 0x19, 0xc5, 0x0a, 0x73, 0x85, 0xa2, 0xdc, 0x91, 0xfc, 0x78, 0xf9,
			0x4b, 0x59, 0xf0, 0x13, 0x1d, 0xc2, 0xb5, 0x6a, 0x0d, 0x7c, 0x69, 0x9d,
			0x42, 0x72, 0x85, 0xda, 0x1c, 0x10, 0x4b, 0x0a, 0xd1, 0x73, 0x9d, 0xa1,
			0x0d, 0x80, 0x71, 0xc2, 0x39, 0x93, 0x78, 0x70, 0x45, 0xdc, 0x21, 0xf0,
			0x07, 0x0e, 0x1e, 0x9a, 0xa1, 0x65, 0x8f, 0xc8, 0xe3, 0xad, 0xd7, 0x3d,
			0xac, 0x72, 0x62, 0xe8, 0x0e, 0x0a, 0xa2, 0xee,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(24, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_25(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x44, 0x94, 0x80, 0xea, 0xa1, 0x00, 0xaf, 0xf6, 0xf4, 0x8d, 0xc6, 0x28,
			0x6a, 0x5a, 0x81, 0xb9, 0x72, 0x8b, 0x08, 0x48, 0x64, 0xf7, 0x8a, 0x9d,
			0xa9, 0x8f, 0x60, 0x6a, 0x00, 0xa6, 0xa4, 0x1f,
		},
		{
			0xe5, 0x3c, 0x6c, 0x5a, 0xc3, 0xda, 0x9f, 0x47, 0x26, 0x38, 0x9a, 0x03,
			0xf9, 0x7b, 0xb6, 0x40,
		},
		{
			0x6b, 0x8f, 0xed, 0xc0, 0x84, 0xd8, 0xe2, 0x8d, 0x33, 0x3a, 0xef, 0x6d,
			0xb3, 0x70, 0x2b, 0x63, 0x51, 0xf0, 0xd2, 0x4e, 0x30, 0x90, 0x8c, 0xcc,
			0xb6, 0x37, 0x94, 0x28, 0x26, 0x55, 0x88, 0x6b,
		},
		{
			0x73, 0xa6, 0xd6, 0x4e, 0x19, 0x66, 0xae, 0x32, 0x43, 0x88, 0xdc, 0x12,
			0xc1, 0x45, 0x44, 0xe9, 0xdc, 0x5a, 0xe4, 0xfc, 0xb3, 0x31, 0xe9, 0x9d,
			0x35, 0x0c, 0x45, 0x6f, 0xf1, 0x6f, 0x9a, 0xa0,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xa0, 0x69, 0x12, 0xd3, 0x62, 0xda, 0x7e, 0xb2, 0x55, 0x98, 0x85, 0x7f,
			0x6d, 0x65, 0x34, 0x4c, 0x3e, 0x23, 0xec, 0x3d, 0xeb, 0x80, 0xc6, 0xe4,
			0x31, 0x58, 0x84, 0x5b, 0x95, 0xea, 0xec, 0xa2, 0x41, 0xc0, 0xbb, 0xbd,
			0x67, 0xac, 0x38, 0x5e, 0x24, 0x69, 0x34, 0x44, 0x45, 0x5c, 0xc1, 0xc2,
			0xc0, 0x8c, 0x11, 0x34, 0xd9, 0x56, 0xb8, 0xbc, 0x93, 0xb2, 0x8b, 0xe9,
			0xc2, 0xd3, 0x32, 0x2b, 0x3e, 0x09, 0x25, 0x29, 0x79, 0xdf, 0xb8, 0xd3,
			0x9d, 0x04, 0xc9, 0x4f, 0x81, 0xbe, 0xbd, 0xa5, 0xc7, 0x31, 0x10, 0x60,
			0x5a, 0x23, 0x7b, 0x56, 0x12, 0x16, 0xbd, 0xa9, 0xee, 0x9b, 0xde, 0xe1,
			0xcc, 0x0c, 0x77, 0x28, 0xbc, 0xc8, 0x30, 0x46, 0x82, 0x33, 0x4c, 0xa9,
			0x44, 0xe4, 0x67, 0xa2, 0x7a, 0x85, 0x31, 0x3f, 0xa5, 0x39, 0x5a, 0x9c,
			0x79, 0x0e, 0x35, 0xde, 0xfd, 0x2e, 0xdb, 0x12,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(25, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_26(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x9a, 0x61, 0x74, 0x16, 0x6e, 0x97, 0xaa, 0x49, 0x81, 0xdd, 0xf5, 0x80,
			0xbc, 0x01, 0xc9, 0x67, 0x54, 0xb9, 0xf0, 0xba, 0x04, 0x27, 0x50, 0xaa,
			0xbf, 0xda, 0x1c, 0xff, 0xe5, 0x6e, 0x85, 0x81,
		},
		{
			0xd7, 0x51, 0x2f, 0xf6, 0xb7, 0xdb, 0x7c, 0xe1, 0x41, 0xb2, 0xbb, 0x01,
			0xdc, 0xd0, 0x42, 0x5e,
		},
		{
			0xed, 0x75, 0x28, 0x8f, 0x23, 0x27, 0x5f, 0x94, 0x22, 0x44, 0x4d, 0xa5,
			0xd3, 0xb5, 0x3c, 0xcb, 0x3c, 0x4a, 0xc8, 0xac, 0xfb, 0x65, 0x9a, 0x1e,
			0x9b, 0x76, 0x55, 0xc2, 0xdb, 0x52, 0xf8, 0x79,
		},
		{
			0x68, 0x88, 0xb9, 0x27, 0x7e, 0x57, 0xdc, 0x57, 0x66, 0x3d, 0x40, 0x2e,
			0xba, 0x8d, 0x03, 0xcf, 0x56, 0xa0, 0x70, 0xdc, 0x86, 0x8e, 0x6a, 0x12,
			0x8b, 0x18, 0x04, 0x00, 0x02, 0xba, 0xf6, 0x90,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x03, 0x51, 0x9d, 0xfb, 0x2f, 0xf8, 0x8c, 0xc2, 0xb5, 0x3e, 0xec, 0xc4,
			0x8a, 0xe2, 0xa1, 0x8d, 0xdc, 0xf9, 0x1a, 0x5d, 0x69, 0xd5, 0xae, 0xfc,
			0xdd, 0xa8, 0x44, 0x4e, 0x6d, 0xf7, 0x90, 0xa5, 0x24, 0x0e, 0x67, 0xb2,
			0xa4, 0xde, 0x75, 0xb4, 0xbb, 0x8a, 0x31, 0xf0, 0xf8, 0xae, 0xb5, 0xe7,
			0x85, 0xff, 0xb7, 0xa1, 0x34, 0x1b, 0xb5, 0x2f, 0xe0, 0x0a, 0x05, 0xee,
			0x66, 0xfa, 0x2d, 0x44, 0xea, 0x99, 0x56, 0xe0, 0x55, 0xf9, 0xff, 0xa6,
			0x64, 0x7c, 0x3b, 0xfe, 0x85, 0x1a, 0xb3, 0x64, 0xad, 0xe7, 0x1a, 0x0d,
			0x35, 0x6d, 0xe7, 0x10, 0xdd, 0xaf, 0xb7, 0x62, 0x2b, 0x1d, 0xa1, 0xbc,
			0x53, 0xfd, 0x4d, 0x32, 0x10, 0x40, 0x72, 0x89, 0xc6, 0x8d, 0x8a, 0xeb,
			0x34, 0x6b, 0xf1, 0x58, 0x06, 0xdb, 0xe7, 0x87, 0xe7, 0x81, 0xb9, 0x4f,
			0x63, 0xda, 0x3e, 0x1f, 0x61, 0xb5, 0xac, 0x60,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(26, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_27(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x9c, 0x6a, 0xe1, 0x00, 0x2e, 0xe1, 0xb0, 0xad, 0xd0, 0xbe, 0x56, 0x3c,
			0xe5, 0x0f, 0x89, 0x9d, 0xa9, 0x36, 0xe1, 0x3e, 0xfa, 0x62, 0x0d, 0x08,
			0xc2, 0x68, 0x8c, 0x19, 0x25, 0x14, 0x76, 0x3a,
		},
		{
			0xfd, 0xe7, 0xdb, 0x51, 0x60, 0xc7, 0x30, 0x44, 0xbe, 0x73, 0xe9, 0xd4,
			0xc1, 0xb2, 0x2d, 0x86,
		},
		{
			0x8f, 0xda, 0xae, 0xff, 0xd6, 0x4e, 0x53, 0xf7, 0xb4, 0x37, 0x4d, 0x90,
			0x2d, 0x44, 0x12, 0x09, 0x96, 0x4e, 0x12, 0xb6, 0x5d, 0x29, 0xaf, 0xec,
			0x25, 0x8e, 0x65, 0xdb, 0x6d, 0xe1, 0x67, 0xca,
		},
		{
			0xbc, 0xc2, 0x8f, 0xd5, 0x8e, 0x39, 0x7f, 0x53, 0xf4, 0x94, 0xad, 0x81,
			0x32, 0xdf, 0x82, 0xc5, 0xd8, 0xc4, 0xc2, 0x2e, 0xa0, 0xb7, 0x13, 0x9b,
			0xd8, 0x1e, 0xeb, 0xa6, 0x56, 0x67, 0xbb, 0x69,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x02, 0x1d, 0x93, 0x8c, 0x9b, 0x4d, 0xb7, 0x80, 0xc7, 0xd8, 0x13, 0x4a,
			0xef, 0xf1, 0x05, 0x3e, 0x5b, 0x88, 0x43, 0x37, 0x0b, 0x8a, 0xe9, 0xa6,
			0x74, 0x9f, 0xca, 0x71, 0x99, 0xd8, 0x09, 0x81, 0x0f, 0x1b, 0xc8, 0xdf,
			0xa4, 0x94, 0x26, 0x47, 0x0c, 0x30, 0xc3, 0x61, 0x6f, 0x90, 0x3e, 0x35,
			0xfb, 0xac, 0xb2, 0x34, 0x20, 0xa3, 0x2f, 0x1b, 0xee, 0x56, 0x7c, 0xc3,
			0x23, 0x00, 0xf7, 0x04, 0x24, 0x6d, 0xdc, 0x02, 0x17, 0xf2, 0x36, 0xef,
			0x52, 0xc3, 0xec, 0x9e, 0x24, 0x33, 0xca, 0x66, 0xf0, 0x5c, 0x25, 0x72,
			0x1f, 0x76, 0x61, 0xc4, 0x3f, 0x22, 0xc1, 0xa1, 0x25, 0xed, 0x5d, 0xb5,
			0x31, 0xbd, 0x08, 0x36, 0xeb, 0x43, 0x5c, 0x27, 0xee, 0xfc, 0x74, 0x24,
			0xce, 0x9d, 0x84, 0x5e, 0x1d, 0x4c, 0xc4, 0xc5, 0x03, 0x09, 0x7b, 0x4f,
			0xfc, 0xa7, 0x88, 0xe6, 0x74, 0xa5, 0xcb, 0x53,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(27, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_28(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xfe, 0x96, 0xa8, 0x5b, 0x69, 0xd4, 0x6b, 0x54, 0x09, 0x18, 0x92, 0x7b,
			0xb6, 0x09, 0xdc, 0x57, 0x64, 0x2e, 0xea, 0xef, 0xd4, 0x6b, 0xb5, 0xda,
			0x21, 0x63, 0xa0, 0xbc, 0x60, 0x29, 0x4b, 0x58,
		},
		{
			0x22, 0x19, 0x5a, 0x41, 0x0d, 0x24, 0xdb, 0x45, 0x58, 0x94, 0x48, 0xdf,
			0xe9, 0x79, 0xd3, 0xfd,
		},
		{
			0x20, 0xf6, 0x98, 0x83, 0x3a, 0x44, 0x72, 0xfd, 0x7b, 0x78, 0xfb, 0x9b,
			0x0c, 0x4e, 0xb6, 0x86, 0x04, 0xf1, 0x66, 0xa2, 0x69, 0x4c, 0x4a, 0xf4,
			0x8d, 0xac, 0x2b, 0x23, 0x76, 0x79, 0x0e, 0x1e,
		},
		{
			0x09, 0xcb, 0x87, 0x08, 0x79, 0xd3, 0xf7, 0x34, 0x21, 0x4f, 0x6a, 0x4b,
			0xd2, 0xe0, 0x8c, 0x62, 0xa2, 0xa9, 0x54, 0xbe, 0xbe, 0x55, 0x94, 0x16,
			0xd8, 0xc3, 0x55, 0x1a, 0xaf, 0xe7, 0x1d, 0x6a,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xd3, 0xe9, 0x6d, 0xbe, 0x29, 0xe1, 0xfc, 0xb8, 0xed, 0x83, 0xb1, 0x9d,
			0xbf, 0xb2, 0x40, 0xe6, 0xf4, 0x16, 0x79, 0xfb, 0xe8, 0x38, 0x53, 0xaa,
			0x71, 0x44, 0x66, 0x17, 0xe6, 0x3e, 0x5a, 0xf7, 0x8c, 0xf9, 0x8b, 0x33,
			0x1d, 0x15, 0xbc, 0xcb, 0x8c, 0x67, 0x3c, 0x4e, 0x5d, 0x5d, 0xce, 0xc4,
			0x67, 0xa1, 0xfe, 0x26, 0xa6, 0xcd, 0x16, 0x96, 0xd0, 0xc9, 0xbc, 0x49,
			0xf7, 0x81, 0x39, 0xd0, 0x51, 0x28, 0x7d, 0xf7, 0xf3, 0xae, 0x0d, 0xbb,
			0x4b, 0xbf, 0x58, 0x1c, 0xb8, 0x21, 0x19, 0x31, 0x06, 0x3c, 0x3f, 0x46,
			0x12, 0xce, 0xd5, 0x3f, 0x59, 0xd1, 0xb4, 0xeb, 0xb8, 0x75, 0x72, 0x91,
			0x39, 0xf5, 0xd2, 0xa7, 0xd6, 0x06, 0x42, 0xe8, 0xf2, 0x83, 0x5e, 0xed,
			0x88, 0x8b, 0x7e, 0x3e, 0x49, 0xc0, 0xdf, 0xfd, 0x01, 0x2c, 0xd7, 0x46,
			0xab, 0xfa, 0x3e, 0x1c, 0x5c, 0x23, 0x08, 0xc6,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(28, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_29(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xa4, 0xfd, 0x69, 0x3f, 0xf0, 0xa8, 0xaf, 0x24, 0xbc, 0xec, 0x35, 0x2d,
			0x31, 0x96, 0x54, 0x9f, 0xd0, 0xda, 0x5e, 0xe5, 0xd9, 0x9c, 0xa5, 0x84,
			0x16, 0xca, 0x03, 0xce, 0x4c, 0x50, 0xf3, 0x8e,
		},
		{
			0x8c, 0xd6, 0x7f, 0x2b, 0xf7, 0x1d, 0x43, 0x66, 0xce, 0x61, 0x39, 0x66,
			0x42, 0x53, 0x1f, 0xf5,
		},
		{
			0x36, 0x89, 0x69, 0xc1, 0x5a, 0x48, 0x49, 0xd7, 0x59, 0x3b, 0xe8, 0xb1,
			0x62, 0x11, 0x3b, 0x92, 0x98, 0xa5, 0x35, 0xc1, 0x48, 0xff, 0x66, 0x8a,
			0x9e, 0x8b, 0x14, 0x7f, 0xb3, 0xaf, 0x4e, 0xba,
		},
		{
			0x83, 0xd2, 0xbe, 0x9a, 0x0d, 0x74, 0xe6, 0xa4, 0x21, 0x59, 0xae, 0x63,
			0x0a, 0xce, 0xbf, 0x4e, 0x15, 0x27, 0x1e, 0xf7, 0xf1, 0x4f, 0x3d, 0xe1,
			0x47, 0x52, 0xbe, 0x0e, 0x0e, 0x82, 0x2b, 0x11,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xe9, 0x18, 0x8f, 0xc0, 0xea, 0xec, 0x74, 0xb2, 0x60, 0x8e, 0x21, 0xe3,
			0xa4, 0x0b, 0xe9, 0x4a, 0xaf, 0x4a, 0xe0, 0x8e, 0xb6, 0x84, 0xde, 0x8f,
			0x8b, 0xba, 0x2d, 0x5f, 0xd3, 0xb0, 0x73, 0xaa, 0x55, 0x31, 0xc9, 0x38,
			0xc0, 0xfc, 0x62, 0x8d, 0xa6, 0x57, 0x25, 0xc5, 0x4b, 0x5c, 0x68, 0xbb,
			0x91, 0xd7, 0xd3, 0x26, 0x56, 0x5e, 0x96, 0x68, 0x5e, 0x0a, 0x4e, 0x7b,
			0x22, 0x0c, 0x50, 0xe0, 0xca, 0xf1, 0x62, 0x8e, 0xdb, 0xa5, 0xbd, 0x75,
			0x5b, 0x31, 0x89, 0x4f, 0x8c, 0xb9, 0x0a, 0xfa, 0x76, 0xe8, 0x8c, 0x5e,
			0xb9, 0xe6, 0x1b, 0x49, 0x32, 0x44, 0x4c, 0x13, 0x97, 0xde, 0xe3, 0xe3,
			0x22, 0x41, 0xa3, 0xfb, 0x70, 0xa3, 0x92, 0x9e, 0x49, 0xf6, 0xda, 0x02,
			0xee, 0xa5, 0x48, 0x12, 0xab, 0xb3, 0xd6, 0xb5, 0xce, 0xe1, 0x8f, 0x03,
			0xaf, 0x1e, 0x0b, 0x49, 0x58, 0x43, 0x0a, 0xb3,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(29, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_30(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x25, 0x4f, 0xf5, 0x68, 0x7a, 0x6d, 0xad, 0x3f, 0x1d, 0x23, 0x7d, 0xc7,
			0x62, 0xf5, 0x8d, 0x24, 0xef, 0x2e, 0x2c, 0x08, 0x4d, 0x0a, 0x48, 0xd2,
			0x6a, 0x3d, 0xc8, 0x1e, 0x54, 0x90, 0xcd, 0xa3,
		},
		{
			0xf2, 0xec, 0x39, 0x2a, 0xcc, 0xa4, 0x91, 0xe0, 0x3c, 0xe4, 0x7b, 0x95,
			0x96, 0x3a, 0x49, 0xfc,
		},
		{
			0xf8, 0x06, 0xb9, 0xb4, 0xa5, 0x66, 0x82, 0xc6, 0x1b, 0x55, 0xcb, 0x6a,
			0x33, 0x4c, 0xaf, 0x87, 0xff, 0xe1, 0x35, 0xad, 0xfe, 0xa6, 0xd0, 0xc3,
			0xfc, 0x22, 0xb3, 0x98, 0x98, 0xfb, 0xd0, 0x78,
		},
		{
			0xb8, 0x49, 0x4b, 0x1c, 0x1f, 0x17, 0x52, 0xfb, 0x6f, 0x80, 0xd7, 0x32,
			0xa8, 0x9b, 0x08, 0x11, 0x58, 0x57, 0xf7, 0xcc, 0x96, 0xe7, 0xdf, 0xf0,
			0x5e, 0xbb, 0x82, 0x27, 0x06, 0x88, 0x99, 0x17,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x0e, 0x52, 0x7e, 0x00, 0x49, 0x4d, 0x55, 0x56, 0x4f, 0x9d, 0x9b, 0x28,
			0xe7, 0x11, 0x0f, 0x9a, 0x61, 0xce, 0x36, 0xc8, 0x83, 0xb5, 0xbe, 0x2d,
			0xcb, 0x05, 0x54, 0x44, 0x16, 0x4c, 0xdd, 0xdd, 0x1a, 0x9f, 0x27, 0x31,
			0x71, 0x6f, 0x22, 0xd6, 0xff, 0x47, 0x6c, 0xe4, 0x13, 0xc7, 0x7a, 0xbf,
			0xc0, 0xe9, 0x46, 0x87, 0x1d, 0x54, 0x81, 0x34, 0x5c, 0x2e, 0x97, 0xb4,
			0xbf, 0xdd, 0x12, 0xac, 0x03, 0xdf, 0x60, 0x6f, 0xc5, 0x6b, 0xdb, 0x99,
			0xac, 0x7b, 0x71, 0xa6, 0x9b, 0x5b, 0x91, 0x60, 0x37, 0x3b, 0xbe, 0xc3,
			0xe9, 0xdd, 0xe4, 0x77, 0x18, 0x0a, 0xf4, 0x54, 0xe7, 0xac, 0xc6, 0xbc,
			0x58, 0xdc, 0x0a, 0xfb, 0x42, 0x81, 0xc0, 0xde, 0x43, 0x54, 0xc1, 0xbf,
			0x59, 0x90, 0x54, 0xe3, 0x80, 0x0c, 0x6d, 0x60, 0xd8, 0x92, 0x85, 0x88,
			0x65, 0xb5, 0x36, 0x1f, 0x50, 0xbf, 0xca, 0x9b,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(30, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_31(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xff, 0x0c, 0xdd, 0x55, 0x5c, 0x60, 0x46, 0x47, 0x60, 0xb2, 0x89, 0xb7,
			0xbc, 0x1f, 0x81, 0x1a, 0x41, 0xff, 0xf7, 0x2d, 0xe5, 0x90, 0x83, 0x85,
			0x8c, 0x02, 0x0a, 0x10, 0x53, 0xbd, 0xc7, 0x4a,
		},
		{
			0x7b, 0xc0, 0x99, 0x28, 0x5a, 0xd5, 0x62, 0x19, 0x93, 0xb6, 0x39, 0xc4,
			0xa9, 0x4c, 0x37, 0x6b,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x14, 0xfc, 0x6c, 0x9b, 0x17, 0x8d, 0xb6, 0x44, 0xa8, 0xcd, 0x71, 0x30,
			0xa4, 0xcf, 0x05, 0x16, 0x78, 0xc8, 0xf4, 0xfa, 0x8f, 0x24, 0xc2, 0x7b,
			0x0a, 0x53, 0x13, 0x38, 0xa5, 0xce, 0x85, 0x89,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x2f, 0x26, 0x20, 0x34, 0x7b, 0xdd, 0xca, 0xa2, 0x94, 0x36, 0x85, 0x34,
			0x6b, 0xbf, 0x31, 0xc4, 0x40, 0x81, 0xf8, 0x66, 0x5f, 0x3d, 0xdb, 0x2b,
			0x42, 0xae, 0x14, 0x16, 0xa7, 0x4c, 0x4b, 0x77, 0xfa, 0xb3, 0xfa, 0x19,
			0xae, 0xec, 0xc5, 0x47, 0xe7, 0x6c, 0x8c, 0xbe, 0x6a, 0xd1, 0xf1, 0x00,
			0xa3, 0xfc, 0x8b, 0x2c, 0xe2, 0xa1, 0xea, 0x3a, 0x3d, 0xd7, 0xcf, 0xad,
			0x46, 0xc1, 0xb2, 0x78, 0x30, 0xb9, 0x40, 0xba, 0x18, 0xd0, 0x9e, 0x9b,
			0x7f, 0xa9, 0x02, 0xbb, 0x76, 0x06, 0x69, 0xb1, 0x73, 0x5c, 0xc7, 0xb7,
			0xbd, 0x39, 0x05, 0x2d, 0xa7, 0xf2, 0x62, 0x6f, 0xa8, 0x70, 0x00, 0xcf,
			0xfa, 0xda, 0x41, 0x00, 0x19, 0xd0, 0x53, 0x38, 0x6a, 0xd8, 0x08, 0xbd,
			0x3c, 0x0c, 0xfc, 0xf5, 0x6b, 0x91, 0x87, 0x9e, 0xb8, 0xd3, 0xf9, 0x32,
			0xee, 0x2d, 0x18, 0x5e, 0x54, 0xf3, 0x1b, 0x74,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(31, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_32(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xf6, 0x1b, 0x9a, 0x29, 0x3d, 0x18, 0x95, 0x2c, 0x74, 0x65, 0xa5, 0x11,
			0x38, 0x2c, 0x03, 0x1b, 0x86, 0x0b, 0xcc, 0x85, 0x86, 0xa7, 0xec, 0x99,
			0xd7, 0x52, 0x3e, 0xad, 0x8a, 0xca, 0xd0, 0xcb,
		},
		{
			0xe3, 0x32, 0xa1, 0x0e, 0x88, 0xf0, 0x0d, 0x95, 0x1d, 0xdf, 0xd4, 0xc7,
			0xb0, 0xb5, 0x96, 0x58,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x10, 0xf2, 0x3a, 0xfe, 0xbe, 0x72, 0x60, 0x93, 0xde, 0x81, 0x70, 0x03,
			0x6a, 0x9f, 0x69, 0x4e, 0x80, 0x84, 0xc0, 0x9c, 0x6e, 0x62, 0x0e, 0x98,
			0xbb, 0x45, 0xf1, 0x6e, 0xe5, 0x60, 0x31, 0xce,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xd3, 0x43, 0x84, 0x89, 0x3e, 0xf4, 0xcb, 0x3e, 0x42, 0x38, 0x6a, 0x42,
			0xef, 0xbd, 0xce, 0x46, 0x7c, 0xd4, 0x11, 0xa7, 0x6e, 0x6a, 0x32, 0xe1,
			0xd6, 0x34, 0xa5, 0xa3, 0x67, 0x0a, 0x95, 0x97, 0x39, 0x85, 0x85, 0x1b,
			0x5f, 0x36, 0x9a, 0x1b, 0xe6, 0x5a, 0xdd, 0x3e, 0xd6, 0x1f, 0x7c, 0x6a,
			0x11, 0x98, 0x64, 0x85, 0xb7, 0xe1, 0x6b, 0x57, 0x26, 0x3e, 0xed, 0x71,
			0xff, 0xc5, 0xa8, 0x97, 0x56, 0xcb, 0x42, 0xb7, 0xbd, 0xfd, 0xf9, 0x2b,
			0x2e, 0x31, 0xc3, 0x6b, 0x4d, 0x7b, 0x7c, 0xbd, 0x2e, 0x65, 0x0c, 0xcb,
			0xd6, 0xc0, 0xb5, 0xdf, 0xc9, 0x5c, 0xd8, 0x93, 0x7e, 0xf1, 0xe8, 0xeb,
			0xc7, 0x1a, 0x04, 0xcf, 0x33, 0xc3, 0x98, 0xf7, 0xf9, 0x08, 0x6f, 0x0b,
			0xae, 0x57, 0xee, 0x9a, 0xf8, 0xf9, 0x3f, 0x3b, 0x8e, 0xc1, 0xc9, 0x81,
			0x69, 0x17, 0xf6, 0x99, 0xd8, 0xf4, 0x32, 0x72,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(32, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_33(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x1b, 0x6b, 0x52, 0xb7, 0xba, 0x19, 0xc2, 0x6e, 0x62, 0xbd, 0x0a, 0xc9,
			0xb2, 0x33, 0xbc, 0xf0, 0x04, 0xbb, 0xc9, 0xfd, 0x9e, 0x1e, 0x73, 0x7c,
			0xcf, 0x88, 0x4a, 0x76, 0x08, 0x29, 0xcd, 0x10,
		},
		{
			0xd3, 0x81, 0xc8, 0xbe, 0x20, 0x32, 0x74, 0xf1, 0x76, 0x69, 0x18, 0xb4,
			0x5e, 0x08, 0xdd, 0xc6,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x2f, 0x50, 0x4e, 0xb2, 0x9f, 0x8e, 0xc0, 0x70, 0x4f, 0x07, 0x0b, 0x59,
			0xc0, 0x20, 0x76, 0xf4, 0xa4, 0xe5, 0xc8, 0xe7, 0xb6, 0x45, 0x68, 0xb2,
			0x2c, 0x5f, 0xf5, 0xba, 0xe1, 0x96, 0xd5, 0xeb,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xa4, 0x15, 0x00, 0xfc, 0x54, 0xcc, 0xf5, 0x49, 0xfe, 0x21, 0xb3, 0x69,
			0x34, 0x28, 0xe0, 0x3c, 0xcd, 0xff, 0xbf, 0x42, 0xff, 0x79, 0x0d, 0x47,
			0xdf, 0xf3, 0x4f, 0xd8, 0x8f, 0xce, 0x01, 0xaf, 0xd4, 0x9a, 0x26, 0xd3,
			0x00, 0xad, 0x66, 0xa3, 0x4c, 0xd3, 0xd1, 0xae, 0xc7, 0x3d, 0xd0, 0x02,
			0xdc, 0x83, 0xc1, 0xb8, 0x6c, 0x43, 0x2d, 0x72, 0x17, 0x66, 0xfb, 0x90,
			0x36, 0x61, 0x30, 0xa3, 0xa2, 0x88, 0x88, 0xa6, 0x3a, 0x5d, 0x0b, 0x8b,
			0xef, 0xa8, 0x95, 0x61, 0x1c, 0x25, 0xbf, 0xa8, 0x81, 0x75, 0x36, 0xeb,
			0x0b, 0x20, 0xb9, 0x98, 0x2c, 0x7d, 0x97, 0x6f, 0x47, 0x07, 0x56, 0x3b,
			0x16, 0xb4, 0x9c, 0xcb, 0xd4, 0x35, 0xcf, 0x04, 0x9b, 0x6a, 0x6e, 0x11,
			0xdf, 0x12, 0xe5, 0xfa, 0xf0, 0xd2, 0x0f, 0xc9, 0xde, 0xab, 0x39, 0x5a,
			0xc6, 0x93, 0x03, 0x60, 0x1b, 0xa3, 0x1f, 0xe3,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(33, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_34(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x36, 0x36, 0x7f, 0xa9, 0x73, 0xaa, 0x4e, 0x2e, 0xc7, 0xbc, 0xef, 0x33,
			0x99, 0x00, 0xdd, 0x95, 0x94, 0x16, 0xbd, 0x63, 0xe7, 0x46, 0xaa, 0x3c,
			0x14, 0xa9, 0xb6, 0x6e, 0x83, 0x8f, 0xc7, 0x80,
		},
		{
			0xfb, 0xa0, 0x71, 0x24, 0xe6, 0xd5, 0x7f, 0x3c, 0x81, 0xba, 0x59, 0x6d,
			0x25, 0x6e, 0x43, 0x3c,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xf3, 0x0b, 0xd1, 0x1a, 0x1f, 0x2b, 0xc9, 0x7d, 0xd8, 0xd4, 0x82, 0x9f,
			0x75, 0xd5, 0x9d, 0x55, 0x77, 0xe7, 0x26, 0x82, 0x5e, 0x23, 0x64, 0xd2,
			0x04, 0xe9, 0x07, 0x7d, 0xf4, 0x8f, 0x65, 0xa9,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x6a, 0x5b, 0xdf, 0xdf, 0x37, 0x31, 0x9c, 0x2e, 0x6b, 0xd2, 0xbc, 0x44,
			0x93, 0x84, 0x55, 0xe0, 0x3a, 0x2e, 0x83, 0x0b, 0x23, 0xee, 0x8e, 0xbd,
			0x6f, 0x98, 0x2f, 0xf8, 0x38, 0xb4, 0x99, 0xd6, 0xd3, 0x7b, 0x44, 0xef,
			0x0a, 0x59, 0x41, 0xba, 0x42, 0xfe, 0x02, 0x45, 0xe6, 0x4b, 0x28, 0xee,
			0x1a, 0x2f, 0xa3, 0xe8, 0xd2, 0x24, 0x69, 0xce, 0xb4, 0xc6, 0xf6, 0x75,
			0x62, 0x9c, 0x3c, 0x6a, 0xbf, 0x61, 0xaa, 0xb6, 0xb1, 0xe8, 0x87, 0xc8,
			0xcf, 0x66, 0x44, 0x80, 0x3f, 0x5d, 0x51, 0x16, 0xbe, 0x11, 0xff, 0x42,
			0x69, 0x8d, 0x42, 0xac, 0x36, 0x86, 0x80, 0x70, 0x36, 0xf5, 0x1a, 0xf4,
			0xae, 0x34, 0x4a, 0x36, 0x5e, 0x8a, 0xd8, 0xeb, 0xc3, 0xa6, 0x23, 0xde,
			0x1b, 0xcc, 0x68, 0xe0, 0x64, 0xd9, 0xfa, 0x82, 0xb6, 0xc8, 0x0f, 0xb0,
			0x57, 0x00, 0x60, 0xbc, 0x58, 0xe8, 0x7e, 0xbd,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(34, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_35(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x24, 0x85, 0xcf, 0xc6, 0x15, 0xd7, 0xfa, 0xb6, 0x0f, 0x46, 0x57, 0xe4,
			0x70, 0xc8, 0x12, 0x1e, 0xdd, 0x6b, 0xc1, 0xfa, 0x22, 0x47, 0xb2, 0x05,
			0xce, 0x40, 0x40, 0xee, 0x13, 0xc3, 0x6f, 0x17,
		},
		{
			0x08, 0x7e, 0x6b, 0x4b, 0xf1, 0xe7, 0x75, 0xa4, 0xc9, 0xf1, 0x82, 0xdf,
			0x61, 0x80, 0x13, 0x43,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x9a, 0xee, 0x0e, 0xbe, 0xd8, 0xda, 0xf7, 0x03, 0x36, 0x0a, 0xa1, 0xfa,
			0x2e, 0x17, 0xfa, 0xc6, 0xb7, 0x94, 0x5a, 0xc4, 0x5c, 0xda, 0xaa, 0xcb,
			0xcc, 0x9f, 0xce, 0x22, 0xd2, 0x8c, 0x0d, 0x1d,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xca, 0x4e, 0x2c, 0x4b, 0x63, 0xbb, 0x7f, 0x08, 0x43, 0x8b, 0x82, 0x5c,
			0xde, 0xb6, 0xa8, 0x83, 0xa2, 0xca, 0xf3, 0x0d, 0xdb, 0xcc, 0xeb, 0x05,
			0xc9, 0x1d, 0x37, 0xc0, 0x56, 0x76, 0xf6, 0x3c, 0xc4, 0xde, 0x0b, 0xd6,
			0x80, 0xa8, 0x21, 0xa9, 0x95, 0x56, 0x4e, 0xcd, 0x12, 0xe5, 0x4f, 0xc0,
			0x56, 0xa3, 0x9e, 0x4e, 0xc5, 0xfe, 0x33, 0x73, 0xb4, 0xd6, 0x62, 0x04,
			0x02, 0xb6, 0xa5, 0x37, 0x76, 0x2f, 0x95, 0xa2, 0xd4, 0x90, 0x95, 0x81,
			0xc8, 0x98, 0x5d, 0xf8, 0x01, 0xc7, 0x2f, 0x32, 0x23, 0xbd, 0x94, 0x67,
			0x96, 0xac, 0x71, 0xdc, 0x8f, 0xc1, 0xb2, 0x83, 0x8e, 0xa5, 0xaa, 0x68,
			0x64, 0x49, 0x08, 0xcc, 0x74, 0xcd, 0x81, 0x84, 0x86, 0x1b, 0x88, 0x94,
			0xe0, 0x74, 0x92, 0x2e, 0x85, 0xa5, 0xe2, 0x58, 0xfb, 0x6e, 0xd3, 0xec,
			0x10, 0xe3, 0xff, 0xe2, 0x40, 0xb6, 0x73, 0x8c,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(35, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_36(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xd1, 0x09, 0xe7, 0x55, 0xa7, 0x25, 0xc9, 0x64, 0x8d, 0x33, 0x71, 0xff,
			0xbf, 0x2c, 0xe0, 0x80, 0x45, 0xd1, 0x71, 0x75, 0xbb, 0x1e, 0x77, 0x7a,
			0xaa, 0x73, 0xfc, 0x6f, 0x9e, 0xc2, 0xac, 0xf6,
		},
		{
			0xc1, 0x42, 0xf6, 0xe0, 0x4d, 0xc8, 0x87, 0xd0, 0x2d, 0x57, 0xb1, 0xc2,
			0x7c, 0x1c, 0x77, 0xae,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x09, 0x1f, 0xd9, 0xe5, 0xb5, 0xb7, 0x41, 0x5f, 0x87, 0x3f, 0x2c, 0xb1,
			0xe5, 0x54, 0xea, 0xbd, 0xd9, 0x1f, 0x96, 0xb6, 0x84, 0xc1, 0x0c, 0xd0,
			0x02, 0x4f, 0xb3, 0x2e, 0x48, 0x1d, 0x4c, 0xb0,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x1d, 0x6f, 0xd5, 0x0a, 0x84, 0x38, 0x69, 0xbb, 0x8a, 0x9b, 0x83, 0xcc,
			0xb4, 0xd1, 0x00, 0xaf, 0x18, 0xee, 0xfc, 0xbd, 0xcb, 0x98, 0xaf, 0xa2,
			0x73, 0x73, 0x7c, 0x68, 0x7b, 0x50, 0x40, 0x22, 0x73, 0xb9, 0x88, 0x6c,
			0xa1, 0x32, 0x0e, 0xca, 0x2c, 0x55, 0x8e, 0xd2, 0x64, 0xa4, 0x2e, 0x02,
			0xe1, 0xa7, 0xdc, 0xbc, 0xa2, 0x67, 0xa3, 0x85, 0xdc, 0x70, 0xd6, 0xe1,
			0x5a, 0xc4, 0x27, 0x6a, 0x6a, 0xd5, 0x91, 0x07, 0xd6, 0x3b, 0x2d, 0x07,
			0x73, 0x91, 0xee, 0xe7, 0x62, 0xd9, 0xfb, 0xe2, 0x29, 0x46, 0xc7, 0xd6,
			0xe3, 0x75, 0x78, 0xd4, 0xdd, 0xcc, 0xfa, 0xe9, 0x78, 0x96, 0x28, 0xa6,
			0x05, 0x37, 0xa3, 0x93, 0x3c, 0xf7, 0xcf, 0xcb, 0x12, 0x71, 0xa2, 0x92,
			0x15, 0x99, 0x1b, 0xd3, 0x41, 0xdf, 0xf3, 0x9b, 0x2c, 0x9b, 0x6c, 0x24,
			0x3a, 0x03, 0xdd, 0xde, 0x0b, 0x73, 0xff, 0xb1,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(36, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_37(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x8b, 0xc4, 0x7b, 0x94, 0xd0, 0x69, 0x19, 0xa1, 0x7b, 0x23, 0xfc, 0xcb,
			0xd4, 0xd2, 0x34, 0x94, 0xbe, 0x25, 0x87, 0xe0, 0xb5, 0xd2, 0xb8, 0x23,
			0x95, 0x59, 0x3d, 0xd4, 0x2a, 0x28, 0x37, 0xa5,
		},
		{
			0xbf, 0x4d, 0x19, 0xfd, 0xfa, 0xf2, 0xd5, 0xee, 0x48, 0x39, 0x11, 0x95,
			0x82, 0x8a, 0xcb, 0xed,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x08, 0x19, 0x0c, 0xb3, 0x1d, 0x8f, 0x20, 0xe8, 0xdf, 0x5e, 0x5a, 0x64,
			0x8e, 0xda, 0x89, 0x63, 0x68, 0xff, 0x3b, 0xae, 0x1b, 0x60, 0xf7, 0xca,
			0x1c, 0x93, 0xf8, 0xbc, 0xdc, 0xdf, 0xbd, 0xcb,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x61, 0x05, 0x19, 0x7b, 0x7f, 0xd3, 0xa3, 0x12, 0x17, 0x27, 0x89, 0x95,
			0x50, 0x14, 0x4f, 0x8b, 0x9f, 0x03, 0xcf, 0x9b, 0x1a, 0x57, 0x0e, 0x97,
			0x2a, 0xd2, 0x3f, 0x0e, 0xa7, 0x14, 0xdb, 0x42, 0x4a, 0x0b, 0xdc, 0xba,
			0x04, 0x5b, 0x6a, 0xeb, 0x12, 0x37, 0xc7, 0xab, 0xe9, 0x0b, 0xcf, 0x84,
			0x3d, 0x42, 0x53, 0xb3, 0x75, 0x72, 0x49, 0x53, 0xc3, 0xf7, 0x26, 0xa9,
			0xad, 0x8b, 0x25, 0xba, 0x91, 0x3e, 0x4d, 0xb3, 0x74, 0x18, 0x5a, 0xf9,
			0x36, 0x37, 0xf3, 0xdc, 0xdf, 0x97, 0x3b, 0x98, 0x67, 0x23, 0xf5, 0x90,
			0xee, 0x76, 0x69, 0x16, 0x05, 0x2c, 0x5e, 0xaa, 0x41, 0x91, 0x51, 0xf0,
			0x9d, 0x12, 0x41, 0x0e, 0x48, 0x28, 0x4a, 0xc8, 0xee, 0x87, 0xf2, 0x5e,
			0xd9, 0x67, 0x1b, 0x23, 0x5c, 0xd7, 0xac, 0xaa, 0x44, 0x0a, 0xd4, 0xb6,
			0xad, 0xa8, 0x64, 0xed, 0x62, 0xfb, 0x98, 0x23,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(37, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_38(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x8b, 0x95, 0xec, 0xfe, 0x9f, 0xd0, 0x90, 0xa0, 0x92, 0x10, 0x17, 0x2b,
			0x4c, 0xc1, 0x99, 0xef, 0x1b, 0xfa, 0x73, 0xed, 0xdd, 0xe5, 0x0d, 0x3d,
			0x41, 0x47, 0xc7, 0x5b, 0x6e, 0x0e, 0x76, 0xeb,
		},
		{
			0x13, 0x0c, 0x18, 0xae, 0xc9, 0x41, 0x5a, 0x2a, 0x54, 0xd6, 0xcf, 0xa4,
			0x33, 0x44, 0x08, 0x33,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x82, 0x5f, 0x0b, 0x1d, 0xd1, 0xa1, 0x57, 0x0f, 0xb7, 0x10, 0x23, 0x15,
			0x12, 0xcc, 0x28, 0x86, 0x81, 0xfc, 0xba, 0x15, 0xa0, 0xa4, 0x5c, 0xef,
			0x45, 0xaa, 0x1f, 0x26, 0x82, 0xcf, 0x37, 0xf8,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x5e, 0xec, 0xa5, 0x1c, 0x5a, 0x05, 0x8f, 0x08, 0xbc, 0xdc, 0xf8, 0x3c,
			0x1d, 0x90, 0x44, 0xb5, 0x20, 0x3b, 0x77, 0xb5, 0x36, 0xf8, 0x66, 0x23,
			0x15, 0x3e, 0x67, 0xf0, 0x5f, 0x3a, 0xdb, 0x28, 0xee, 0xb8, 0x92, 0x5b,
			0x61, 0x07, 0xd7, 0xa8, 0xac, 0xca, 0x77, 0x8b, 0xc4, 0x52, 0x6e, 0xf1,
			0xcf, 0xf5, 0xac, 0x5f, 0xdd, 0x02, 0x39, 0x52, 0x2f, 0xa1, 0x75, 0xd3,
			0xe0, 0x65, 0xd7, 0x8c, 0x19, 0xa6, 0xda, 0xba, 0x56, 0xe2, 0x0b, 0x28,
			0x4b, 0x43, 0x3e, 0xa8, 0x1c, 0xba, 0xe8, 0x13, 0x9c, 0xa7, 0x94, 0x45,
			0xb8, 0x87, 0x74, 0x99, 0x49, 0xda, 0xe8, 0xb9, 0xf0, 0x3e, 0x55, 0x9f,
			0xcf, 0x5f, 0x71, 0x07, 0x22, 0x83, 0x1e, 0xfa, 0x92, 0xad, 0xe0, 0xcd,
			0x0c, 0xab, 0x5c, 0x34, 0x3d, 0xfd, 0x4b, 0xcd, 0xd8, 0xd6, 0xbc, 0xef,
			0xba, 0xf0, 0x53, 0x1a, 0x8d, 0xed, 0x69, 0x3c,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(38, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_39(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xe8, 0x43, 0x6e, 0x86, 0x22, 0xea, 0x34, 0x5d, 0xcb, 0x94, 0x31, 0x64,
			0x0a, 0xff, 0x32, 0x8c, 0x21, 0xbe, 0x83, 0x83, 0x21, 0xda, 0xf0, 0x89,
			0x51, 0x0d, 0x27, 0xaa, 0x4d, 0x3d, 0x64, 0x0c,
		},
		{
			0xd6, 0x9a, 0xe9, 0xda, 0xe2, 0x16, 0xae, 0x6c, 0xf3, 0x6f, 0x48, 0xcf,
			0x3f, 0x97, 0xbd, 0x3a,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xd9, 0x12, 0xaa, 0xb4, 0x83, 0x68, 0xed, 0xe1, 0xc5, 0xf9, 0xc1, 0x11,
			0x6c, 0xb4, 0xe2, 0xe9, 0x91, 0xfb, 0xaf, 0x26, 0x24, 0xd9, 0xf8, 0x1a,
			0xf4, 0x7e, 0x8d, 0x74, 0x67, 0x46, 0x01, 0x05,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xc3, 0xcb, 0x77, 0x89, 0xf6, 0x9f, 0x6a, 0x99, 0xc3, 0x8c, 0xcd, 0x8d,
			0x73, 0xb6, 0x27, 0xc5, 0xa2, 0x4d, 0xa7, 0x11, 0x6f, 0x9d, 0x1b, 0x11,
			0xb4, 0xa9, 0xfe, 0x34, 0x48, 0xa6, 0x83, 0x7d, 0x21, 0x90, 0x99, 0xb3,
			0x39, 0xf0, 0x6f, 0x18, 0x5d, 0x9f, 0x4c, 0x3d, 0x1b, 0x9a, 0x47, 0x4b,
			0xbd, 0x2c, 0x63, 0x3a, 0xcc, 0x09, 0xfd, 0x54, 0x0a, 0xb8, 0xdf, 0x42,
			0x3b, 0x95, 0x15, 0xd5, 0x2c, 0x68, 0xc5, 0x39, 0x06, 0x8f, 0xcb, 0xa5,
			0x31, 0xb8, 0x85, 0xc2, 0x6c, 0xcc, 0x4f, 0x57, 0x6e, 0x34, 0x71, 0xf7,
			0xe0, 0x88, 0x6f, 0x39, 0x72, 0x51, 0x31, 0xc2, 0xf1, 0x25, 0x6b, 0x45,
			0x02, 0x20, 0x47, 0x3b, 0xcd, 0x07, 0xdd, 0xd9, 0x0a, 0xc8, 0x63, 0x05,
			0x9e, 0xb1, 0x8c, 0x69, 0x82, 0x70, 0x3c, 0xa4, 0x21, 0x9c, 0x74, 0x7d,
			0x84, 0x58, 0x00, 0x92, 0x09, 0x6a, 0x55, 0xc1,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(39, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_40(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x61, 0x30, 0x87, 0x33, 0xc0, 0x88, 0x36, 0x27, 0xae, 0x93, 0x19, 0xe1,
			0x86, 0x57, 0x07, 0x24, 0x40, 0xc7, 0x58, 0x8f, 0x58, 0x13, 0x1e, 0xda,
			0x02, 0x81, 0xa4, 0xe6, 0xc6, 0x3e, 0xc6, 0xf4,
		},
		{
			0x37, 0x38, 0x4d, 0x9f, 0x08, 0xc4, 0x25, 0x70, 0x35, 0xf7, 0x30, 0xdb,
			0x48, 0xa2, 0xc1, 0x53,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xec, 0x66, 0x81, 0xc5, 0x28, 0x13, 0x99, 0xbf, 0xb3, 0xad, 0x4f, 0x5c,
			0x6d, 0x73, 0xde, 0x97, 0x26, 0x35, 0x1e, 0x18, 0x9c, 0x46, 0xba, 0x94,
			0x30, 0x42, 0x50, 0xb7, 0x10, 0x52, 0xf9, 0x04,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb3, 0xf6, 0x42, 0xcd, 0x6f, 0xfb, 0x6d, 0xea, 0xa1, 0x1c, 0x07, 0x4e,
			0x76, 0x72, 0x99, 0xd9, 0x3d, 0x24, 0x11, 0xfe, 0x4d, 0xbd, 0xe2, 0xad,
			0xc0, 0x46, 0x30, 0x90, 0x4e, 0x3b, 0xd5, 0xe6, 0x1b, 0x53, 0x44, 0x0f,
			0x9d, 0x51, 0x39, 0x8a, 0xf3, 0xeb, 0x8e, 0x95, 0x12, 0x33, 0xe5, 0x85,
			0xd8, 0x89, 0x75, 0x73, 0x7e, 0x3f, 0x45, 0xe2, 0x3f, 0x5d, 0x63, 0xbb,
			0xd4, 0x2a, 0x28, 0x62, 0xe8, 0x27, 0xa8, 0xa4, 0x9c, 0xed, 0xd8, 0xec,
			0xf8, 0xfd, 0x82, 0x56, 0xa4, 0x75, 0x73, 0xd3, 0x5c, 0x26, 0x22, 0xd7,
			0xe8, 0x7d, 0x83, 0xe5, 0x35, 0x72, 0x05, 0xe0, 0xdc, 0xe5, 0x31, 0xf9,
			0x2e, 0x8f, 0xa3, 0x25, 0x77, 0x82, 0xf1, 0xdb, 0x47, 0xcf, 0x70, 0xdb,
			0xe9, 0x44, 0x01, 0xcd, 0x50, 0xe2, 0x7d, 0xcc, 0xe1, 0x83, 0x69, 0x3a,
			0xd5, 0x01, 0xdb, 0x7b, 0x0a, 0xeb, 0x55, 0xb2,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(40, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_41(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x7c, 0x6b, 0x15, 0xd4, 0x47, 0x7c, 0x5d, 0xdf, 0xf3, 0xea, 0xfa, 0xd5,
			0x56, 0x90, 0xe0, 0x21, 0xf8, 0xbf, 0x5d, 0x34, 0x3b, 0xc6, 0xdc, 0x2d,
			0x58, 0xd2, 0x9a, 0x97, 0x9a, 0xa9, 0x21, 0xc3,
		},
		{
			0x79, 0x38, 0x80, 0x4e, 0xeb, 0xf0, 0x61, 0x38, 0x8d, 0x3c, 0x34, 0x61,
			0x44, 0xf4, 0x92, 0x11,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x7d, 0x8b, 0x82, 0xd9, 0xee, 0x9a, 0x99, 0xaf, 0x59, 0x34, 0x7d, 0xa7,
			0x67, 0x78, 0x05, 0x14, 0x42, 0xbe, 0x40, 0x98, 0xa3, 0x00, 0xa4, 0xc6,
			0x06, 0xa3, 0x24, 0x73, 0x14, 0xea, 0x46, 0x18,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x62, 0xac, 0xe9, 0x9f, 0xe2, 0xdb, 0x12, 0xe9, 0x3c, 0x2c, 0x91, 0x55,
			0xd7, 0x0a, 0x58, 0x1b, 0x2b, 0xac, 0xb8, 0xf1, 0xed, 0x90, 0x9a, 0x24,
			0xde, 0x1b, 0x36, 0x5f, 0xde, 0xec, 0xc6, 0x71, 0x35, 0x60, 0x7a, 0x44,
			0x79, 0x2b, 0x00, 0x4e, 0x50, 0x7d, 0x07, 0xf5, 0xb3, 0x61, 0x48, 0x0c,
			0xe4, 0xb2, 0xf1, 0xf6, 0xd3, 0x2b, 0xb2, 0x53, 0x69, 0x33, 0xec, 0xe2,
			0x81, 0x6d, 0x47, 0xa9, 0x46, 0x77, 0xcd, 0x82, 0x3e, 0xf6, 0x4a, 0x17,
			0xba, 0xaa, 0x33, 0x59, 0x87, 0x72, 0x2b, 0x0e, 0x1c, 0x02, 0x39, 0x94,
			0x0b, 0x0f, 0x3d, 0x00, 0x52, 0x19, 0xd8, 0xd3, 0x78, 0x15, 0x7c, 0xcf,
			0x17, 0xc1, 0x41, 0x62, 0x16, 0x5a, 0x8e, 0xf3, 0x8f, 0xd4, 0x08, 0xa4,
			0xe9, 0xac, 0x85, 0xa7, 0x08, 0x04, 0xd5, 0x73, 0x53, 0xc7, 0x5f, 0xb1,
			0x87, 0xbe, 0x0d, 0x72, 0xcf, 0x69, 0x9c, 0xc8,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(41, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_42(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x71, 0x59, 0xc7, 0xae, 0xa9, 0x6d, 0xa8, 0xab, 0x86, 0x48, 0xf8, 0xf8,
			0x54, 0xf3, 0x51, 0xbe, 0x9b, 0x28, 0x45, 0x66, 0x90, 0xb7, 0x88, 0xc4,
			0x9c, 0x3d, 0xce, 0x74, 0xb2, 0x31, 0x1e, 0x63,
		},
		{
			0x6a, 0xed, 0x95, 0x1f, 0xc6, 0xe1, 0x55, 0x7f, 0x5f, 0x1c, 0xb6, 0xad,
			0xbe, 0xc5, 0x46, 0x86,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xce, 0x6d, 0xe3, 0xb5, 0x21, 0x17, 0x77, 0xf9, 0x05, 0x66, 0x69, 0xd0,
			0x46, 0x01, 0x84, 0x00, 0x3e, 0x87, 0x97, 0x84, 0xdf, 0xff, 0x86, 0xd7,
			0xeb, 0xf0, 0x8a, 0x23, 0x3e, 0x6a, 0x54, 0x82,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x13, 0x33, 0xd1, 0x21, 0x11, 0xb9, 0x59, 0xc5, 0x11, 0x74, 0xca, 0x21,
			0x88, 0xed, 0x84, 0xca, 0x7d, 0xfd, 0xa3, 0x0d, 0x5f, 0xf2, 0x53, 0x27,
			0xcd, 0x19, 0xf7, 0x37, 0x79, 0x59, 0x0b, 0x19, 0xc0, 0xb5, 0x24, 0x75,
			0x60, 0x33, 0x00, 0x8b, 0x64, 0x06, 0xab, 0xe6, 0x31, 0xcc, 0xa6, 0xb8,
			0x2e, 0xbd, 0xda, 0x5c, 0xe7, 0xb9, 0xe2, 0x96, 0x8a, 0x8a, 0xde, 0xe3,
			0xe4, 0xa6, 0xb3, 0xea, 0xa8, 0x07, 0x9a, 0x64, 0xe7, 0x62, 0x2f, 0x66,
			0x28, 0xc3, 0x62, 0xb5, 0xc6, 0xdc, 0xb6, 0x1b, 0x7c, 0xf6, 0x45, 0x67,
			0xe6, 0xec, 0x5a, 0xe3, 0x12, 0x04, 0x72, 0x50, 0x51, 0x3e, 0xc5, 0x65,
			0x91, 0x60, 0xeb, 0xa5, 0xc7, 0x58, 0x35, 0x2f, 0x74, 0x16, 0xa0, 0x39,
			0x86, 0x37, 0x90, 0x2a, 0x4d, 0x7e, 0x86, 0xad, 0xf9, 0x84, 0x50, 0xdf,
			0x11, 0x1e, 0xf4, 0x33, 0x11, 0x1f, 0xf2, 0xd1,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(42, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_43(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xeb, 0xcf, 0xe2, 0x5c, 0xea, 0x69, 0x66, 0x45, 0xc7, 0x9e, 0x77, 0x29,
			0x22, 0xdf, 0x04, 0xd4, 0x84, 0x17, 0xf9, 0xcf, 0x2e, 0xce, 0xf7, 0x27,
			0xcc, 0xcb, 0x3e, 0x89, 0x98, 0x89, 0x3c, 0x59,
		},
		{
			0x13, 0x6e, 0x0c, 0x8e, 0x8f, 0xad, 0x1c, 0x49, 0x43, 0x5e, 0x6e, 0xac,
			0x76, 0x3c, 0x77, 0x6f,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x38, 0xb4, 0x8d, 0xb1, 0x1f, 0xf3, 0x1c, 0xe5, 0xe0, 0xa0, 0xcb, 0x62,
			0x79, 0x86, 0xb8, 0x8b, 0xea, 0xd7, 0x4b, 0xe8, 0x34, 0xa4, 0xe3, 0x47,
			0xde, 0x17, 0x07, 0x4c, 0xe1, 0x17, 0x24, 0x4c,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xd9, 0x0c, 0xb0, 0x83, 0x5b, 0x0c, 0xa6, 0x4c, 0x8d, 0x1d, 0xec, 0x23,
			0xf2, 0x77, 0x3b, 0x6d, 0xd8, 0x35, 0xbe, 0xe7, 0xe2, 0x7c, 0x91, 0x82,
			0x2d, 0xbf, 0x08, 0x98, 0xd1, 0x81, 0xde, 0x5b, 0xe4, 0xe4, 0xf6, 0xa1,
			0xd5, 0x97, 0x12, 0xe3, 0xb4, 0x72, 0x00, 0xcb, 0x14, 0xe1, 0x22, 0x52,
			0xe9, 0xd7, 0xc5, 0x62, 0x4c, 0x84, 0x1e, 0x4c, 0x05, 0xc7, 0x37, 0x2a,
			0x5e, 0x70, 0xe8, 0x3a, 0x1d, 0x86, 0xd0, 0xc0, 0x69, 0xbf, 0xb2, 0x9a,
			0x21, 0xf5, 0x11, 0xc4, 0xcc, 0x8c, 0x4c, 0xd9, 0xf6, 0x6b, 0xa3, 0xa0,
			0x81, 0x02, 0x44, 0xdf, 0x25, 0x18, 0xbf, 0x37, 0x85, 0xb3, 0x03, 0xe7,
			0xae, 0xbe, 0x9f, 0x19, 0x06, 0xbc, 0xff, 0x55, 0xa4, 0x54, 0x5d, 0x53,
			0xd2, 0xfb, 0x8d, 0xd7, 0x2e, 0xee, 0x00, 0x3f, 0x10, 0x00, 0x9e, 0x9b,
			0x41, 0x82, 0x52, 0x11, 0xf2, 0x35, 0x2b, 0xe9,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(43, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_44(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xb0, 0xcc, 0xe5, 0xac, 0xf0, 0xd3, 0x30, 0x5c, 0xf5, 0xeb, 0xdb, 0x55,
			0x87, 0x0a, 0xf9, 0x31, 0x68, 0x20, 0x0d, 0xa5, 0x6f, 0xa2, 0xfa, 0x68,
			0x85, 0xbf, 0x1c, 0x2a, 0x93, 0x28, 0x95, 0x6f,
		},
		{
			0x6d, 0x79, 0x66, 0x19, 0x8e, 0xc2, 0x34, 0x18, 0x08, 0x97, 0x39, 0x00,
			0xa9, 0xfc, 0xf0, 0x44,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x2c, 0x0b, 0xb1, 0xbd, 0xa5, 0xa0, 0x12, 0x57, 0x57, 0xb6, 0x4d, 0x57,
			0x3a, 0xd0, 0x4c, 0xf3, 0x6d, 0x47, 0xee, 0x46, 0x15, 0x58, 0x3f, 0xa8,
			0xfe, 0x5a, 0x2d, 0xec, 0x8b, 0x80, 0x59, 0x21,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb4, 0x17, 0xb4, 0x47, 0x0e, 0xa9, 0x50, 0x77, 0x2c, 0xbe, 0xda, 0x48,
			0x5d, 0xe1, 0xe9, 0x45, 0xd2, 0x31, 0x9a, 0x6a, 0x35, 0x00, 0xd8, 0x8f,
			0xd5, 0x22, 0x08, 0x0a, 0x26, 0x44, 0xcf, 0x7c, 0x18, 0x61, 0xd4, 0x28,
			0x1d, 0xbe, 0xab, 0x81, 0x51, 0x8b, 0x59, 0x24, 0x50, 0x50, 0xb1, 0xb4,
			0xa5, 0xf8, 0x9e, 0x9a, 0x61, 0xbe, 0x4c, 0x5a, 0x43, 0x72, 0xb4, 0x97,
			0x2f, 0x1c, 0x17, 0xb9, 0x83, 0x35, 0x99, 0x07, 0xa4, 0x4d, 0x88, 0xb5,
			0xbb, 0x12, 0x84, 0x84, 0x28, 0xef, 0xea, 0x84, 0x40, 0x78, 0x93, 0x64,
			0x0f, 0xb0, 0x36, 0x3b, 0x23, 0x71, 0xe1, 0x48, 0x91, 0x24, 0x8a, 0xfd,
			0x61, 0xab, 0xb5, 0xac, 0x04, 0x74, 0xc5, 0x00, 0xe1, 0xc8, 0xef, 0x2c,
			0x11, 0x41, 0x84, 0xd7, 0xea, 0x47, 0x66, 0xd4, 0x6b, 0x78, 0xd2, 0x74,
			0x29, 0x86, 0xbb, 0x34, 0xe6, 0x2c, 0x49, 0xde,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(44, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_45(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x10, 0x3e, 0x04, 0xdc, 0x6c, 0x88, 0x5d, 0xe4, 0x22, 0x50, 0x67, 0xc8,
			0xdd, 0x67, 0xce, 0xc5, 0xa3, 0xdd, 0x7c, 0x45, 0x58, 0x6d, 0x2a, 0xdf,
			0x59, 0xb5, 0x4b, 0x54, 0x6c, 0xdc, 0x15, 0x91,
		},
		{
			0x5e, 0x37, 0xbf, 0x8f, 0x45, 0x5a, 0x28, 0x35, 0xcf, 0x44, 0x8d, 0xa6,
			0x83, 0xd6, 0x36, 0xb4,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x93, 0x6f, 0xfb, 0x8c, 0xb0, 0x83, 0x33, 0x24, 0xbe, 0x82, 0x3a, 0xe7,
			0x23, 0x92, 0x99, 0x96, 0xbc, 0x45, 0x0f, 0x81, 0x57, 0x6f, 0x2c, 0x43,
			0xb4, 0x82, 0x9e, 0xa8, 0x50, 0x44, 0xcb, 0x55,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x14, 0xef, 0xb3, 0xf1, 0x2f, 0xa0, 0x3b, 0xd2, 0x0b, 0x94, 0xb9, 0x79,
			0x4e, 0xee, 0xd7, 0xde, 0x9e, 0x7f, 0x7f, 0xc9, 0xf1, 0x06, 0xd6, 0x80,
			0x36, 0x63, 0x22, 0xe7, 0x10, 0x28, 0x5c, 0x1e, 0xb8, 0xc1, 0x43, 0x60,
			0x76, 0x41, 0xe0, 0xc4, 0x4f, 0xe9, 0xdb, 0x92, 0xc0, 0xab, 0xb5, 0xfd,
			0xa3, 0x80, 0x6b, 0xef, 0x59, 0x61, 0x28, 0x9e, 0x3e, 0xd7, 0x2b, 0x67,
			0xb1, 0x4f, 0x1e, 0x98, 0xdc, 0x3b, 0xd9, 0xb9, 0x38, 0x86, 0x3d, 0x5c,
			0xd9, 0xe4, 0x5f, 0xff, 0x3e, 0x7c, 0x5e, 0xb2, 0x56, 0xb4, 0x87, 0xf0,
			0x29, 0x64, 0x3a, 0x0a, 0x27, 0xc0, 0x9b, 0xcb, 0x6f, 0x5e, 0xec, 0x83,
			0x63, 0xc4, 0xde, 0xbe, 0xd6, 0x17, 0x52, 0xdf, 0x88, 0x40, 0xcd, 0x34,
			0xe6, 0xae, 0x7f, 0x5e, 0xe3, 0xfb, 0x53, 0xb7, 0x26, 0x76, 0x78, 0xd1,
			0x5e, 0x01, 0x72, 0xa6, 0x99, 0xfa, 0xbd, 0x8d,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(45, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_46(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x28, 0xba, 0x1a, 0x66, 0x16, 0x32, 0xef, 0xc8, 0xec, 0xce, 0xd5, 0xf5,
			0x1b, 0x79, 0x13, 0x00, 0xfb, 0x3b, 0x55, 0xb0, 0x5d, 0x04, 0x17, 0x08,
			0x63, 0x8d, 0xe4, 0xbe, 0xb7, 0x57, 0xa9, 0xe5,
		},
		{
			0x76, 0x82, 0x87, 0x96, 0xaf, 0xf0, 0x7f, 0x55, 0x79, 0x5c, 0xb5, 0x47,
			0x13, 0xc7, 0x7e, 0xd4,
		},
		{
			0x40, 0x93, 0x3f, 0xdc, 0xce, 0x41, 0x59, 0xb0, 0x95, 0x51, 0x11, 0xf8,
			0x44, 0x47, 0x1b, 0x0d, 0xb8, 0x5b, 0x73, 0xbd, 0xd2, 0xb7, 0x8c, 0x46,
			0x8d, 0xd3, 0x9e, 0x2a, 0x9b, 0x29, 0xae, 0xf2,
		},
		{
			0xa5, 0xf5, 0x42, 0xb0, 0x4a, 0xaa, 0x5d, 0xbc, 0x93, 0x1e, 0x47, 0x01,
			0x9f, 0xeb, 0x38, 0x96, 0x26, 0x16, 0xc5, 0x7a, 0xf0, 0x9b, 0x7c, 0x1d,
			0xf8, 0x3f, 0x2b, 0x86, 0x0f, 0xf7, 0x65, 0x86,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x65, 0xe5, 0xaa, 0x47, 0xb3, 0x85, 0xf1, 0xea, 0x42, 0xb2, 0x31, 0xb9,
			0xfe, 0x74, 0x42, 0x53, 0xb8, 0x59, 0x88, 0x59, 0xd7, 0x01, 0x1e, 0x52,
			0x5f, 0x5a, 0x2a, 0x1a, 0xd3, 0x2a, 0x97, 0x2a, 0x85, 0x08, 0x02, 0xc6,
			0x0a, 0x2b, 0xe1, 0x9b, 0xe2, 0x70, 0x06, 0x3a, 0x3c, 0xfb, 0xea, 0xae,
			0x95, 0x4f, 0x10, 0xb1, 0x22, 0x35, 0x2d, 0xe6, 0xa0, 0x8a, 0xc4, 0x10,
			0xe0, 0x99, 0x16, 0x53, 0xaa, 0xb2, 0x71, 0xb3, 0x60, 0xfe, 0x91, 0x91,
			0xcf, 0x5a, 0xdd, 0xcc, 0xcc, 0xed, 0x8c, 0x4a, 0xcf, 0xb6, 0x14, 0x57,
			0x04, 0x99, 0x92, 0x98, 0x8f, 0xd7, 0xa9, 0xac, 0xca, 0x1f, 0x1b, 0xca,
			0x35, 0xf1, 0x47, 0x58, 0x13, 0x69, 0x4a, 0x39, 0x98, 0x8e, 0x5f, 0xac,
			0x9f, 0x4a, 0xc0, 0x57, 0x22, 0x86, 0xbc, 0x46, 0x25, 0x82, 0xad, 0x0a,
			0xf7, 0x8a, 0xb3, 0xb8, 0x5e, 0xc1, 0x7a, 0x25,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(46, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_47(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xcf, 0xd5, 0x49, 0xc8, 0x08, 0x0a, 0x50, 0x2e, 0x3e, 0xdd, 0x79, 0x05,
			0x58, 0x4e, 0x5b, 0xe1, 0x4d, 0x7e, 0xf5, 0x59, 0x10, 0xd2, 0x82, 0xd8,
			0x1d, 0x34, 0x5c, 0x53, 0xca, 0x34, 0xe0, 0x7b,
		},
		{
			0xa5, 0x65, 0xe4, 0xbc, 0x2f, 0x46, 0x17, 0x69, 0xc0, 0xa8, 0xf2, 0x11,
			0xa6, 0x7d, 0xc9, 0xa7,
		},
		{
			0xd4, 0xad, 0xd6, 0x4c, 0x86, 0x03, 0x41, 0x56, 0x29, 0xda, 0x5c, 0xd5,
			0x8c, 0x51, 0x89, 0x58, 0xc8, 0x4d, 0x43, 0xad, 0xa7, 0x1b, 0xd1, 0xbd,
			0x57, 0x8f, 0xbe, 0x67, 0x74, 0x4b, 0x24, 0x76,
		},
		{
			0x27, 0x12, 0xa4, 0x16, 0x15, 0xf2, 0xb4, 0xaf, 0x37, 0xd6, 0x41, 0xc3,
			0xa7, 0x34, 0x07, 0xad, 0x62, 0xb4, 0x3e, 0xeb, 0xd0, 0x43, 0x9b, 0xc7,
			0x0f, 0xaf, 0x2b, 0x7f, 0x25, 0x25, 0xe2, 0xf1,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb5, 0x22, 0x5f, 0xa8, 0xb7, 0xb1, 0xff, 0x05, 0x8f, 0x3e, 0xd4, 0x92,
			0xcc, 0x7f, 0x66, 0xcb, 0xc3, 0x93, 0x47, 0x74, 0x03, 0x14, 0x47, 0x12,
			0x5a, 0xf8, 0x20, 0xcf, 0xe9, 0x42, 0x9c, 0x19, 0x5d, 0xf9, 0xb3, 0xcc,
			0x9a, 0x11, 0xb0, 0x2c, 0xf3, 0x80, 0x06, 0x13, 0xeb, 0x53, 0x4d, 0xc2,
			0xce, 0x13, 0x65, 0xbc, 0x40, 0x9b, 0x2b, 0x13, 0xd6, 0x5b, 0xed, 0xe5,
			0x4c, 0xd2, 0x13, 0xa5, 0x49, 0xf2, 0xef, 0x9e, 0x3d, 0x17, 0xc4, 0xfb,
			0x99, 0x55, 0x1e, 0xea, 0x53, 0x9b, 0x91, 0x25, 0x48, 0x83, 0xca, 0xd8,
			0x6f, 0x5c, 0x38, 0x53, 0xa8, 0x03, 0x56, 0x6a, 0xc1, 0xea, 0x86, 0x4a,
			0x08, 0xbf, 0x6e, 0x80, 0x29, 0x58, 0x8f, 0xf8, 0x58, 0x0f, 0x7d, 0x26,
			0x57, 0xdc, 0x8f, 0x38, 0x9f, 0x72, 0xca, 0xb4, 0x25, 0x1a, 0x19, 0xc4,
			0xfa, 0x2e, 0x53, 0x9a, 0xfa, 0xc7, 0xc1, 0xc1,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(47, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_48(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x5c, 0xc9, 0x13, 0xe1, 0x26, 0x29, 0x48, 0x12, 0x62, 0xad, 0x31, 0x7f,
			0xb8, 0xd3, 0x98, 0x70, 0xb4, 0xf7, 0x26, 0x0c, 0xca, 0xcd, 0xce, 0xbd,
			0x91, 0x31, 0x71, 0x0a, 0xe7, 0x1e, 0x0c, 0x98,
		},
		{
			0xd4, 0x6d, 0xe6, 0xc6, 0xf4, 0xa5, 0x43, 0xfe, 0x5f, 0x47, 0x3c, 0x2c,
			0x2d, 0x0e, 0x21, 0x4a,
		},
		{
			0xca, 0x7f, 0x2f, 0x65, 0x01, 0xe6, 0xde, 0xa1, 0xbf, 0x7f, 0x54, 0xe6,
			0xbd, 0x50, 0x30, 0x47, 0x97, 0xab, 0x80, 0xea, 0x02, 0x28, 0x81, 0x3e,
			0xb3, 0x84, 0x98, 0x42, 0xf2, 0x17, 0x27, 0x3f,
		},
		{
			0xe7, 0xf0, 0x08, 0xab, 0xd0, 0x0d, 0x52, 0x52, 0x97, 0xff, 0x6e, 0x2c,
			0x39, 0xa3, 0x58, 0xe6, 0xc7, 0x5c, 0x4a, 0x64, 0xf6, 0xac, 0x68, 0xde,
			0xfd, 0x27, 0x56, 0x23, 0xf4, 0x96, 0xca, 0x07,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x8c, 0xa7, 0xdd, 0x3d, 0x3b, 0x82, 0xf6, 0x40, 0xb3, 0x3c, 0xf1, 0x76,
			0xb2, 0x2d, 0xe7, 0x13, 0x0c, 0x1d, 0xc5, 0x4c, 0x68, 0x4d, 0xdf, 0x8b,
			0xc7, 0xb0, 0x8a, 0x18, 0x17, 0x53, 0xad, 0x7a, 0xf8, 0xc4, 0x63, 0x82,
			0x87, 0xf7, 0x24, 0xdf, 0x8f, 0xf9, 0x91, 0x43, 0x1f, 0x99, 0xa3, 0x10,
			0x51, 0x73, 0xb1, 0x07, 0xf0, 0x64, 0x54, 0x1d, 0xd9, 0x24, 0x40, 0xfd,
			0x4c, 0xee, 0x65, 0x9e, 0x93, 0x22, 0xb5, 0x15, 0x41, 0x68, 0xf4, 0x9d,
			0xb3, 0x6f, 0x32, 0x24, 0x5a, 0xe4, 0xd9, 0x83, 0x15, 0x59, 0xeb, 0x83,
			0x9b, 0x28, 0xb9, 0xa3, 0x88, 0xae, 0x49, 0x80, 0xce, 0xea, 0xdd, 0x99,
			0x03, 0xd4, 0xfe, 0xbc, 0x66, 0x2b, 0xb7, 0xd2, 0x46, 0x01, 0x74, 0x63,
			0xe8, 0x43, 0xe7, 0x72, 0x56, 0x72, 0x66, 0x22, 0xcd, 0x1f, 0x10, 0x25,
			0xc5, 0x1b, 0x66, 0x24, 0xce, 0x84, 0x77, 0x16,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(48, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_49(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xb8, 0xbc, 0x5e, 0x67, 0x20, 0xa4, 0x14, 0xb7, 0xf7, 0xa2, 0x13, 0xd4,
			0x55, 0xd4, 0x3e, 0xc5, 0x5c, 0x77, 0x3e, 0xdf, 0x24, 0x03, 0x31, 0x97,
			0x16, 0xde, 0xf9, 0xee, 0xc1, 0x29, 0x39, 0x75,
		},
		{
			0x2d, 0xa2, 0x86, 0x29, 0x23, 0x31, 0x05, 0x6d, 0xcf, 0x88, 0x4d, 0xe8,
			0x94, 0x4e, 0x02, 0x6b,
		},
		{
			0x51, 0x4e, 0xb0, 0xc6, 0x9b, 0xbe, 0xa5, 0xb9, 0x95, 0x32, 0x49, 0x76,
			0xca, 0x12, 0x33, 0x37, 0xe2, 0x6f, 0x63, 0x9b, 0x06, 0xa7, 0xa3, 0x62,
			0xff, 0x57, 0x67, 0x35, 0x20, 0xcb, 0x70, 0xdb,
		},
		{
			0x3c, 0x1b, 0xba, 0x68, 0x1b, 0x3e, 0x04, 0x08, 0xd0, 0x8b, 0x92, 0xfd,
			0xd3, 0x25, 0x2d, 0xda, 0x67, 0x68, 0x5d, 0x86, 0x5f, 0xe8, 0x28, 0x6d,
			0x6c, 0x4f, 0xd6, 0x69, 0xc3, 0x3c, 0x09, 0x20,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb9, 0x41, 0x60, 0x02, 0xc5, 0x44, 0xe0, 0x79, 0x99, 0x84, 0x64, 0xd5,
			0x18, 0x1a, 0xfe, 0x70, 0xc1, 0x06, 0xd4, 0x4b, 0x87, 0x6d, 0x16, 0xb3,
			0x9f, 0x93, 0x6a, 0xff, 0x36, 0xec, 0x86, 0xdd, 0xef, 0x5f, 0xb2, 0x68,
			0x8a, 0x0a, 0xad, 0xf6, 0x69, 0xc3, 0x9b, 0x6f, 0x67, 0x24, 0xff, 0x3e,
			0xea, 0xac, 0x90, 0xb7, 0x06, 0x74, 0xc1, 0x4d, 0x97, 0xe9, 0x94, 0xbd,
			0xec, 0x0f, 0xd5, 0x23, 0x73, 0xcf, 0x7a, 0x21, 0x25, 0x0c, 0x83, 0x5e,
			0x93, 0xc7, 0x3d, 0x7c, 0x49, 0x79, 0x0b, 0x59, 0x30, 0x0d, 0xb4, 0x85,
			0xf0, 0x21, 0xb1, 0x35, 0x93, 0xa3, 0x2d, 0x72, 0xb7, 0x05, 0x7d, 0xa0,
			0xd5, 0x5f, 0x63, 0xa6, 0xca, 0xe5, 0x18, 0x35, 0x40, 0xe8, 0x17, 0x29,
			0x0d, 0x8b, 0x91, 0xce, 0x55, 0xf4, 0x5f, 0xeb, 0x77, 0x30, 0x73, 0x2f,
			0xa9, 0x15, 0x6d, 0xa8, 0x27, 0x6d, 0x67, 0x32,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(49, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_50(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x28, 0xdc, 0x2f, 0x25, 0xb2, 0x63, 0x4c, 0x06, 0x72, 0xa0, 0xfa, 0x03,
			0x37, 0x85, 0x63, 0xde, 0x21, 0x45, 0x00, 0xfa, 0xa7, 0x7e, 0xe4, 0x07,
			0x64, 0x09, 0xaa, 0x16, 0xbc, 0x85, 0x12, 0xd2,
		},
		{
			0x85, 0xd5, 0x2e, 0xd3, 0x4e, 0x06, 0x29, 0x09, 0xbb, 0x5e, 0x61, 0x75,
			0xdf, 0xa1, 0x87, 0xb8,
		},
		{
			0x5b, 0xb0, 0xe4, 0x22, 0xce, 0xec, 0x47, 0xaa, 0xf4, 0x80, 0x82, 0x90,
			0x9f, 0x11, 0xc9, 0x98, 0xa5, 0xa7, 0x7d, 0x54, 0x32, 0x0b, 0x28, 0xea,
			0xea, 0xab, 0x71, 0x32, 0x21, 0xc9, 0x54, 0x19,
		},
		{
			0x43, 0x93, 0xb2, 0x0d, 0x7d, 0xd1, 0x5d, 0x67, 0xa2, 0xf7, 0x8e, 0x74,
			0x37, 0xe1, 0xe7, 0xa9, 0x51, 0x10, 0xdc, 0x2d, 0xf6, 0x46, 0xdb, 0x22,
			0x34, 0x2a, 0x71, 0x1a, 0xe1, 0x02, 0xd1, 0xd6,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x66, 0xdb, 0xf2, 0xb1, 0x21, 0xbf, 0xd4, 0xa7, 0x25, 0x84, 0xb5, 0x61,
			0xc6, 0x50, 0xf5, 0x09, 0x95, 0xd2, 0x2f, 0x0f, 0x37, 0x7d, 0xf2, 0xf6,
			0x0c, 0xd2, 0x6e, 0x03, 0x77, 0xb7, 0xd1, 0xae, 0x7b, 0x1a, 0xfc, 0x72,
			0xd2, 0x2f, 0x9a, 0xb5, 0xb5, 0x42, 0x76, 0x4a, 0x21, 0x9c, 0xbb, 0x66,
			0xf3, 0x25, 0x3e, 0x60, 0x2e, 0x93, 0x80, 0x7d, 0x3e, 0xc2, 0xbe, 0xe5,
			0xab, 0x13, 0x1f, 0x00, 0xfb, 0x4e, 0xd3, 0x28, 0x8b, 0xb1, 0xb6, 0xf5,
			0x2d, 0xd3, 0x9d, 0x79, 0xc8, 0x17, 0xab, 0x73, 0xd9, 0xb1, 0x1f, 0x81,
			0xe3, 0x01, 0x31, 0xd1, 0xf3, 0xca, 0xb9, 0x60, 0x44, 0x1d, 0x07, 0xec,
			0x23, 0xbe, 0xa0, 0xee, 0xd7, 0x71, 0x88, 0xd7, 0x14, 0x07, 0x87, 0x39,
			0xd8, 0x1c, 0x42, 0xcb, 0x22, 0x4b, 0xc5, 0xad, 0x0b, 0x65, 0xc8, 0xc9,
			0xc5, 0xa8, 0x08, 0xb6, 0x05, 0xf1, 0x92, 0x3c,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(50, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_51(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xaf, 0x84, 0x05, 0x9f, 0x96, 0xed, 0x19, 0xb2, 0xe5, 0x3f, 0x09, 0x12,
			0x28, 0xe7, 0x09, 0x67, 0x86, 0x47, 0x35, 0xdc, 0x83, 0x17, 0x8e, 0x15,
			0x9f, 0x57, 0x7c, 0x57, 0xbc, 0x12, 0x1c, 0xbf,
		},
		{
			0x84, 0xc6, 0x51, 0x5f, 0xcb, 0x88, 0x13, 0x6a, 0xdc, 0x23, 0x9f, 0x53,
			0x7f, 0xca, 0xbf, 0x4d,
		},
		{
			0x9f, 0xab, 0xe3, 0x0a, 0xdb, 0x15, 0xa1, 0x45, 0x2f, 0xba, 0xf2, 0xb6,
			0x27, 0xc0, 0xe4, 0x7a, 0xdc, 0x60, 0xce, 0x6e, 0x9b, 0x10, 0xa5, 0x6a,
			0x33, 0x8f, 0x08, 0xe8, 0x85, 0x8f, 0x82, 0xa1,
		},
		{
			0xf5, 0xfd, 0xb3, 0xb3, 0xaa, 0x2e, 0xde, 0xf8, 0xca, 0x8d, 0x31, 0xc9,
			0x91, 0x2e, 0xb4, 0x3a, 0x3a, 0xdc, 0x48, 0x25, 0xc8, 0x8d, 0x46, 0x32,
			0x30, 0xb1, 0x13, 0x91, 0x3f, 0x1d, 0x01, 0x17,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x18, 0x1c, 0x7a, 0x02, 0x26, 0x4e, 0x5a, 0x6a, 0x1f, 0xf4, 0xd9, 0x7d,
			0xbc, 0xe0, 0x74, 0x28, 0xe4, 0xb2, 0xb4, 0x49, 0xa2, 0xb8, 0x61, 0xeb,
			0x0a, 0x61, 0x72, 0x93, 0xfb, 0x84, 0x7e, 0x2a, 0xa1, 0x23, 0x4d, 0x21,
			0x4d, 0x50, 0x92, 0x07, 0x3f, 0x9d, 0xbb, 0x49, 0xfa, 0xe8, 0xb9, 0x63,
			0x56, 0x9b, 0xf4, 0xb7, 0xfe, 0x20, 0xe6, 0xaa, 0x35, 0x5e, 0x56, 0x5d,
			0xb4, 0xc7, 0x5d, 0x62, 0x3b, 0x47, 0xcb, 0x62, 0x7a, 0x7f, 0xde, 0x90,
			0x08, 0xd0, 0x29, 0xd2, 0x0c, 0x09, 0xbb, 0x50, 0xe8, 0x3a, 0xb0, 0x37,
			0xf3, 0x70, 0x8f, 0xc5, 0x18, 0x3f, 0xe0, 0xf2, 0x2a, 0x73, 0x2b, 0x89,
			0xf3, 0x9c, 0x3a, 0x45, 0xf1, 0xf1, 0x61, 0x68, 0x76, 0xf8, 0x72, 0x03,
			0xce, 0xef, 0x06, 0x05, 0xf0, 0x3c, 0x28, 0xb2, 0x65, 0x4d, 0xf0, 0x0a,
			0x4d, 0x55, 0xa0, 0xc0, 0x11, 0xb9, 0x39, 0x16,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(51, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_52(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x4a, 0xf2, 0xbb, 0xf3, 0x6d, 0x2a, 0x30, 0x25, 0x95, 0xde, 0x66, 0x7e,
			0x42, 0xe0, 0x8a, 0xbb, 0x5e, 0x44, 0xcb, 0xde, 0x9b, 0x1b, 0xe2, 0xee,
			0xd0, 0x33, 0x8a, 0xac, 0xb4, 0x86, 0x73, 0xbc,
		},
		{
			0xe0, 0xb9, 0x43, 0xf5, 0x5f, 0xc1, 0x34, 0x16, 0x86, 0x9a, 0x62, 0x4a,
			0x9a, 0x38, 0x3f, 0xcf,
		},
		{
			0xde, 0xa4, 0xc6, 0xb2, 0x49, 0x0f, 0x18, 0x72, 0xe6, 0x1a, 0x9b, 0xf7,
			0x7c, 0x3f, 0x3b, 0x67, 0xaf, 0xa8, 0xe8, 0x92, 0xbe, 0x0c, 0x3c, 0x1a,
			0xb8, 0x9e, 0x92, 0xa2, 0x70, 0x45, 0xe1, 0xa7,
		},
		{
			0xbb, 0x6a, 0x04, 0xce, 0x7f, 0x81, 0xef, 0x18, 0xbb, 0x34, 0x55, 0xc9,
			0x25, 0xb8, 0xd5, 0x76, 0xdf, 0x2b, 0xbf, 0x68, 0xc0, 0xfe, 0x4e, 0x65,
			0xc6, 0x80, 0x22, 0xaa, 0x69, 0xdc, 0x0e, 0xc2,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x31, 0x05, 0x2a, 0x14, 0xc9, 0x11, 0xb6, 0x92, 0x64, 0x57, 0xf5, 0x62,
			0xfc, 0x27, 0xd1, 0x9f, 0x22, 0xa3, 0xde, 0x7d, 0xd1, 0x45, 0xaa, 0x75,
			0xb9, 0xb6, 0x03, 0xa6, 0x10, 0x9f, 0xe1, 0xc1, 0xfe, 0xa3, 0xff, 0x2e,
			0x37, 0xe9, 0xd9, 0x84, 0x14, 0x50, 0x2e, 0x2d, 0xfe, 0xf0, 0x13, 0x77,
			0x88, 0x18, 0x7e, 0x42, 0xa7, 0x1c, 0xd6, 0x94, 0x75, 0xa8, 0xbb, 0x00,
			0x40, 0xac, 0x6d, 0xb2, 0x3a, 0xed, 0x06, 0x92, 0x74, 0x95, 0x95, 0x4a,
			0xaa, 0x23, 0x19, 0x3d, 0xb8, 0x32, 0x89, 0xf7, 0x5f, 0x96, 0x6e, 0xef,
			0x0a, 0x7c, 0xed, 0x68, 0xe8, 0x87, 0x1a, 0xb2, 0xa4, 0x7d, 0x38, 0x92,
			0xf5, 0x94, 0xa9, 0x8c, 0x90, 0xae, 0x51, 0xcf, 0x11, 0xbc, 0xf6, 0xed,
			0x87, 0x9d, 0x45, 0x8d, 0x17, 0x1f, 0x33, 0x4c, 0x04, 0x92, 0x34, 0xdc,
			0xe7, 0x94, 0x1f, 0x34, 0xb4, 0x83, 0x9f, 0xcb,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(52, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_53(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xd6, 0x66, 0xa2, 0xe7, 0x0a, 0xf2, 0x77, 0x12, 0x69, 0x42, 0x5c, 0x4d,
			0xc1, 0xec, 0xa8, 0x7f, 0xe1, 0x77, 0x31, 0x38, 0x43, 0xb8, 0xb6, 0xb9,
			0xc3, 0xd4, 0xda, 0x9d, 0x46, 0x54, 0x0c, 0xaf,
		},
		{
			0xd5, 0x48, 0x6a, 0x3b, 0x88, 0x1c, 0x35, 0xbe, 0x94, 0x4d, 0xd2, 0x35,
			0x9b, 0x88, 0xd6, 0xa2,
		},
		{
			0x5f, 0x0c, 0xcf, 0x34, 0xc2, 0x57, 0x60, 0x8e, 0xbf, 0x9e, 0x5f, 0xcb,
			0xb9, 0x79, 0xc1, 0xd8, 0xf6, 0x69, 0x04, 0x7e, 0x71, 0x8c, 0x76, 0x0d,
			0x0a, 0x9b, 0xe0, 0xbe, 0xe7, 0xc9, 0x4d, 0x79,
		},
		{
			0x1a, 0x6c, 0xf1, 0xd8, 0xf6, 0xa1, 0x7a, 0xd9, 0xdb, 0xa6, 0x89, 0x8f,
			0x7f, 0x82, 0xd3, 0x3d, 0xba, 0x59, 0x6e, 0xac, 0xcc, 0x20, 0x11, 0xd1,
			0x55, 0xac, 0x6e, 0x39, 0xd5, 0x9a, 0x75, 0x58,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x58, 0xfd, 0xe4, 0xf4, 0xe1, 0x08, 0xa5, 0x40, 0x22, 0x6f, 0x40, 0x84,
			0x6b, 0xce, 0x66, 0xda, 0xb6, 0x08, 0x2e, 0xfb, 0x63, 0x6e, 0x89, 0x87,
			0xe1, 0x85, 0x42, 0xdd, 0x6e, 0x28, 0x40, 0xa8, 0x8b, 0xe0, 0x45, 0x01,
			0x94, 0x02, 0xb2, 0x03, 0x71, 0xe0, 0x8e, 0x41, 0x52, 0x38, 0x92, 0x5a,
			0x44, 0x0a, 0x49, 0x3e, 0xd8, 0xe1, 0x03, 0x55, 0xfc, 0x16, 0x46, 0x85,
			0xe4, 0x99, 0x60, 0x0e, 0xe1, 0x92, 0xed, 0x0b, 0x95, 0x31, 0x59, 0x2d,
			0x4a, 0x52, 0xf4, 0xd7, 0x0a, 0xc0, 0xa0, 0xaa, 0xe1, 0x2c, 0x17, 0x05,
			0xe9, 0xc1, 0xec, 0xa2, 0x6b, 0x36, 0xc0, 0x90, 0x6e, 0x21, 0x6f, 0xda,
			0xe4, 0x79, 0xae, 0xf2, 0xdc, 0xa8, 0x46, 0x0d, 0x05, 0x58, 0xbd, 0xf7,
			0x68, 0xd7, 0xc0, 0x00, 0x2f, 0xc7, 0x1d, 0x6a, 0x5e, 0xb0, 0x6f, 0xd6,
			0x60, 0x56, 0x42, 0x57, 0xa1, 0x58, 0xdb, 0x53,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(53, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_54(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x20, 0x8b, 0x9e, 0xf5, 0x00, 0xb0, 0x49, 0x3b, 0x7f, 0x2d, 0xed, 0x5f,
			0x3a, 0xf6, 0xbd, 0x56, 0x8f, 0x5d, 0xff, 0x0e, 0xbe, 0xfd, 0x19, 0x98,
			0xd3, 0x48, 0x03, 0x7e, 0xc3, 0x63, 0xdf, 0x80,
		},
		{
			0x13, 0xeb, 0x96, 0x9c, 0xea, 0xc8, 0x94, 0x56, 0xab, 0x22, 0xd8, 0x69,
			0x33, 0xa1, 0x81, 0x81,
		},
		{
			0xcc, 0xae, 0xbe, 0xad, 0x9e, 0x2b, 0x91, 0x1e, 0xf9, 0x27, 0xf3, 0x06,
			0x9d, 0xa5, 0xe3, 0x58, 0x2a, 0x78, 0x86, 0xd9, 0x56, 0x57, 0x0d, 0x2e,
			0xdf, 0x2a, 0xf4, 0x34, 0xb9, 0x47, 0x3d, 0xba,
		},
		{
			0x06, 0x9b, 0x76, 0x7a, 0xc6, 0x12, 0x3f, 0x9c, 0x55, 0x1d, 0x3b, 0xbd,
			0x43, 0x22, 0x8d, 0xf5, 0x4f, 0x37, 0x9c, 0xa6, 0xb4, 0x18, 0x96, 0x77,
			0x22, 0x82, 0x5f, 0xad, 0x33, 0x34, 0x3f, 0x9c,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xec, 0xc7, 0x58, 0xfd, 0x31, 0x6a, 0x75, 0xe5, 0xf9, 0x7a, 0x14, 0x80,
			0x3a, 0xe5, 0x1f, 0x8c, 0x0a, 0x80, 0xec, 0xba, 0x95, 0x4b, 0x54, 0xec,
			0x8b, 0x9d, 0xa8, 0x08, 0x1f, 0x0b, 0xe9, 0x68, 0x0b, 0x11, 0xac, 0xdc,
			0xb9, 0xfc, 0x65, 0xc6, 0xef, 0x45, 0x87, 0x87, 0x54, 0x61, 0x37, 0xfa,
			0xdb, 0xbf, 0x07, 0x7a, 0x60, 0x02, 0x52, 0x88, 0x35, 0xca, 0x3c, 0x80,
			0x8f, 0xa5, 0x93, 0xd8, 0x20, 0xa0, 0x65, 0x01, 0x59, 0x2d, 0x85, 0x7a,
			0xb0, 0xa6, 0x7b, 0x80, 0x3e, 0xcb, 0x63, 0x15, 0xba, 0x1a, 0xa0, 0xc9,
			0x0c, 0xaf, 0xed, 0xe0, 0x24, 0x80, 0xb9, 0x8a, 0x07, 0xcf, 0x9b, 0xb8,
			0x67, 0xdc, 0xe3, 0xb1, 0x57, 0x5b, 0x6b, 0x46, 0xd4, 0x04, 0x6f, 0x44,
			0x08, 0xdd, 0xc7, 0x12, 0x82, 0xfa, 0x43, 0xac, 0xc7, 0x38, 0xa2, 0x76,
			0x65, 0xb3, 0x3a, 0xfc, 0xcf, 0x97, 0x40, 0x77,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(54, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_55(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x01, 0x46, 0x9d, 0xfd, 0xc9, 0x93, 0xba, 0x87, 0x0c, 0x34, 0x87, 0xf9,
			0x3b, 0xec, 0x20, 0x81, 0x97, 0x4d, 0xe5, 0xf4, 0x19, 0x3d, 0x10, 0x5b,
			0xd8, 0xaf, 0x41, 0xcf, 0x00, 0x1f, 0x69, 0x9c,
		},
		{
			0x76, 0x15, 0x4e, 0xbb, 0xfe, 0xe0, 0x89, 0x51, 0xcf, 0xfc, 0x61, 0xc1,
			0xc2, 0x2c, 0x96, 0x85,
		},
		{
			0x4e, 0x5f, 0xa3, 0x38, 0x43, 0x45, 0xb6, 0x6b, 0x7d, 0xd4, 0xfe, 0x9f,
			0xc7, 0x88, 0xa0, 0x3a, 0x38, 0x05, 0xc5, 0xe8, 0xac, 0xb4, 0x70, 0x7e,
			0x6d, 0x24, 0x41, 0x2c, 0x49, 0x27, 0xe8, 0xff,
		},
		{
			0x5e, 0x74, 0x35, 0xf0, 0x0f, 0x59, 0x38, 0x7b, 0xf0, 0x97, 0x7a, 0x6d,
			0x79, 0xc9, 0x5b, 0x83, 0x18, 0x84, 0xe8, 0xed, 0x26, 0x1b, 0xdd, 0x09,
			0x58, 0x87, 0x16, 0x91, 0xc7, 0x0a, 0x3f, 0xe2,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x26, 0xdf, 0xf7, 0x9a, 0xd2, 0xe7, 0x29, 0x32, 0xcc, 0xb0, 0x0a, 0xfe,
			0x66, 0x90, 0x5f, 0xb9, 0x66, 0xfc, 0x7a, 0x05, 0x7b, 0x49, 0x1c, 0x37,
			0x5a, 0x31, 0x54, 0x0a, 0x79, 0xe3, 0x0a, 0x47, 0x80, 0x70, 0x8a, 0x1e,
			0x92, 0x8d, 0x7c, 0x6e, 0xf9, 0x59, 0x43, 0x3b, 0x9a, 0x0d, 0xbd, 0x19,
			0xcc, 0x73, 0x08, 0x2c, 0xf5, 0xd3, 0x04, 0xff, 0x2e, 0x9a, 0xcb, 0xc5,
			0x62, 0x20, 0x02, 0x50, 0xe7, 0x37, 0x6d, 0x7d, 0x35, 0xf2, 0x2b, 0xfa,
			0x25, 0x7c, 0x86, 0xb0, 0xda, 0x74, 0x83, 0x79, 0x1f, 0x8f, 0x82, 0xe7,
			0x95, 0x59, 0xe5, 0xb4, 0xb9, 0xbf, 0xe0, 0xe1, 0xfe, 0xc5, 0x2a, 0xc4,
			0x09, 0x33, 0xb3, 0x93, 0x4b, 0x13, 0xdd, 0x65, 0x0e, 0xd5, 0xcc, 0x18,
			0x3e, 0x79, 0x6b, 0xe3, 0x5f, 0xc5, 0x9f, 0xaf, 0x51, 0x3c, 0x40, 0xf1,
			0x24, 0x9b, 0xc8, 0xfe, 0xc4, 0xf2, 0xf6, 0x11,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(55, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_56(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xf1, 0x9f, 0x4d, 0xd4, 0x01, 0xe7, 0xfb, 0x21, 0xb3, 0xf7, 0xf7, 0x3c,
			0x8b, 0x39, 0x0e, 0xb6, 0x76, 0xf8, 0x95, 0xda, 0xab, 0x86, 0x0b, 0x49,
			0x1c, 0xb2, 0xfa, 0xc8, 0x51, 0x97, 0xc1, 0x49,
		},
		{
			0x25, 0x38, 0xe6, 0xbe, 0x68, 0x44, 0x1f, 0x15, 0xac, 0x6f, 0xd0, 0xa3,
			0x4e, 0xf3, 0xb8, 0xb8,
		},
		{
			0x12, 0x35, 0x2e, 0x74, 0xb9, 0x12, 0xc5, 0xc6, 0x37, 0xaf, 0x35, 0xb1,
			0x90, 0x47, 0x77, 0x5f, 0x9c, 0xae, 0x26, 0xc1, 0xad, 0x49, 0xc5, 0xd5,
			0x6c, 0xd2, 0x87, 0xc7, 0xb4, 0x11, 0x76, 0x22,
		},
		{
			0x05, 0x1f, 0xfc, 0xc7, 0x64, 0x79, 0xa5, 0x67, 0x0c, 0xb0, 0xfb, 0xba,
			0x2b, 0xdc, 0x83, 0x79, 0x33, 0x1d, 0x8d, 0xba, 0xd0, 0x60, 0x69, 0xf9,
			0xa5, 0xe3, 0x6c, 0xb8, 0xf7, 0xaf, 0xb3, 0x98,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x8c, 0x44, 0xe3, 0xd5, 0xd0, 0xfe, 0xe7, 0x9b, 0x1a, 0x25, 0x8f, 0x3e,
			0x30, 0xe3, 0xcd, 0x9a, 0x0e, 0xbd, 0x12, 0xcf, 0x68, 0x97, 0x8c, 0xca,
			0x5e, 0xd0, 0x53, 0x44, 0x76, 0x57, 0xff, 0x41, 0x05, 0x77, 0x8b, 0x4d,
			0xd8, 0x72, 0xfd, 0xe9, 0x2f, 0x98, 0x80, 0x4a, 0x72, 0x42, 0xea, 0x57,
			0xe1, 0x3b, 0x0d, 0x6a, 0x8d, 0x49, 0x37, 0x37, 0x48, 0xc4, 0x95, 0xa7,
			0x7d, 0xe8, 0x0e, 0xfe, 0x24, 0xfa, 0xe7, 0x68, 0x60, 0x4d, 0x1c, 0xbb,
			0x94, 0xeb, 0xd0, 0x8f, 0x37, 0x91, 0x40, 0xea, 0x52, 0x08, 0xdb, 0x58,
			0x9d, 0x1d, 0x1d, 0xdc, 0xb6, 0x9e, 0xcc, 0x76, 0x4b, 0x51, 0xfc, 0x08,
			0x25, 0x08, 0xac, 0xd6, 0x93, 0xce, 0xdb, 0xc7, 0x07, 0x0f, 0x55, 0x2d,
			0x29, 0x4b, 0x6b, 0x2b, 0xab, 0x9c, 0x7b, 0x6c, 0x9c, 0x0e, 0x06, 0x88,
			0x8b, 0xab, 0xbd, 0x75, 0x68, 0x8c, 0xf5, 0x36,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(56, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_57(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xa0, 0x24, 0x5f, 0x99, 0x08, 0xf6, 0xc6, 0x64, 0xde, 0xa7, 0x68, 0x9e,
			0x97, 0x1a, 0x1f, 0xf2, 0xc1, 0x89, 0xee, 0xe4, 0x22, 0xea, 0x15, 0x7d,
			0x40, 0xa4, 0x44, 0xd8, 0x3c, 0x94, 0x19, 0x80,
		},
		{
			0xf5, 0x69, 0xf2, 0xaf, 0x4d, 0x5f, 0xbb, 0xb7, 0x81, 0x3f, 0x3e, 0x24,
			0x4e, 0x4d, 0x3d, 0xc4,
		},
		{
			0x88, 0x7e, 0x6c, 0x40, 0x26, 0x35, 0x97, 0xd7, 0x56, 0xc2, 0x5b, 0x40,
			0x15, 0xe7, 0x09, 0xa9, 0x66, 0x3e, 0x48, 0xd5, 0xa9, 0xfb, 0xb4, 0x81,
			0x3b, 0x85, 0xa5, 0x3b, 0x99, 0x7a, 0x64, 0xb6,
		},
		{
			0x2b, 0x85, 0x59, 0x8a, 0x4f, 0x92, 0xa0, 0x0f, 0x4c, 0xb4, 0xc1, 0xfa,
			0xd1, 0x82, 0x9e, 0xd6, 0xf8, 0xf0, 0x2a, 0x8a, 0x81, 0x02, 0x7d, 0x14,
			0xb6, 0x57, 0x4c, 0x91, 0x78, 0x3c, 0x1d, 0xc4,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb4, 0xc4, 0xf4, 0x13, 0x96, 0x70, 0x6c, 0xf3, 0x6d, 0x6c, 0xd9, 0xff,
			0xb6, 0xf8, 0xa9, 0x4c, 0x46, 0x82, 0xa8, 0x69, 0x2a, 0xe9, 0xd8, 0xea,
			0xd8, 0x0a, 0xe8, 0x9d, 0x96, 0x4a, 0x01, 0xa5, 0xde, 0xc1, 0x49, 0x3d,
			0x96, 0xe7, 0x64, 0xb9, 0x49, 0x35, 0xf8, 0x8d, 0xfd, 0x95, 0xe7, 0xb5,
			0xfd, 0x19, 0x6e, 0xed, 0xa8, 0xc2, 0xaf, 0xe9, 0x94, 0x6d, 0xca, 0x07,
			0x25, 0xff, 0x51, 0x7d, 0x7b, 0xa3, 0xd8, 0xf7, 0xe0, 0x17, 0x41, 0x19,
			0x23, 0xfa, 0xa9, 0x28, 0x9f, 0x68, 0xb7, 0xf7, 0xba, 0x6e, 0x29, 0xe3,
			0xde, 0x65, 0x2f, 0xa4, 0xd6, 0x0e, 0x8c, 0xac, 0xf5, 0x82, 0xef, 0xa2,
			0x15, 0x7a, 0x23, 0xe9, 0x53, 0xb5, 0x03, 0x98, 0x1c, 0x48, 0x5c, 0xf5,
			0x23, 0xbd, 0x14, 0x56, 0xbb, 0x7e, 0x8c, 0x32, 0x2f, 0x7b, 0xf0, 0x06,
			0x22, 0x3e, 0x37, 0x74, 0x69, 0x75, 0x00, 0xfd,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(57, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_58(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xee, 0x8f, 0x7a, 0x02, 0x9a, 0xa7, 0xd2, 0x20, 0xa5, 0xa1, 0x9d, 0x28,
			0x5f, 0xff, 0x41, 0x1f, 0x2e, 0xd5, 0x66, 0xa1, 0x2e, 0xc3, 0xc8, 0x13,
			0xf9, 0x55, 0x9d, 0x22, 0x1d, 0xa3, 0x8e, 0x81,
		},
		{
			0xb0, 0x28, 0xd4, 0xb9, 0x6d, 0x9a, 0x61, 0x80, 0x1d, 0x24, 0x53, 0xc4,
			0xb4, 0x2f, 0xaf, 0xdb,
		},
		{
			0x28, 0x2c, 0xf8, 0x31, 0x13, 0x75, 0x83, 0x7c, 0x25, 0x77, 0xd3, 0x1c,
			0x50, 0x3e, 0xc0, 0x60, 0x23, 0xf8, 0xe2, 0xa5, 0xa0, 0xa2, 0xe5, 0xcf,
			0x9c, 0xb7, 0xd9, 0x11, 0x89, 0x9f, 0x2b, 0x5a,
		},
		{
			0x4c, 0x7b, 0x59, 0xe4, 0xdf, 0xd6, 0x9c, 0x36, 0xfe, 0xd4, 0xa1, 0x28,
			0xd7, 0xe6, 0x3d, 0x5e, 0x76, 0x47, 0x14, 0x3f, 0x5d, 0x4a, 0xab, 0x55,
			0x51, 0x6e, 0xad, 0x95, 0x9c, 0x05, 0x94, 0x6a,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb6, 0xe8, 0x89, 0xd0, 0x2d, 0x5d, 0x4c, 0xd5, 0xa9, 0x37, 0xc7, 0x94,
			0x94, 0xe9, 0x92, 0x8c, 0x38, 0x3d, 0x5a, 0x3b, 0xb1, 0x1f, 0x6c, 0xea,
			0xa1, 0x80, 0xc7, 0x96, 0x47, 0x96, 0xac, 0x59, 0xfc, 0xc2, 0x46, 0x61,
			0x71, 0xd6, 0x1f, 0x53, 0x3c, 0xea, 0xee, 0x81, 0xb0, 0xdc, 0x76, 0x3f,
			0x0c, 0x62, 0xc4, 0x49, 0xb7, 0x79, 0xcd, 0x30, 0x3d, 0xdf, 0x56, 0xe1,
			0xce, 0x01, 0xdc, 0x12, 0x7b, 0xf1, 0xbb, 0x52, 0x67, 0x23, 0xfd, 0x7f,
			0x2f, 0x05, 0x46, 0x76, 0xd6, 0x51, 0xc9, 0x4d, 0xfb, 0x06, 0x23, 0xf9,
			0x72, 0x22, 0xe3, 0x42, 0x41, 0xb5, 0x65, 0x4d, 0x5d, 0x9f, 0x23, 0x1b,
			0x99, 0x41, 0x1c, 0x0f, 0x36, 0xd5, 0x7f, 0xcb, 0xd0, 0xed, 0x49, 0xe2,
			0xea, 0xc4, 0x7d, 0xb3, 0xf2, 0xa1, 0x22, 0x17, 0x8e, 0x11, 0x6d, 0x68,
			0xf0, 0x4f, 0xe0, 0x89, 0x88, 0x54, 0xe6, 0x95,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(58, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_59(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x10, 0xb8, 0x78, 0x9c, 0xdb, 0xd6, 0x77, 0x84, 0x42, 0xa4, 0x5e, 0xdf,
			0x22, 0x8b, 0x99, 0x23, 0xf4, 0x52, 0x63, 0x1a, 0xd0, 0xfe, 0x9e, 0x60,
			0x8d, 0x10, 0x82, 0x6b, 0xa7, 0x1d, 0xa7, 0xca,
		},
		{
			0x15, 0x9f, 0xc5, 0xd8, 0xe5, 0x0e, 0xb5, 0x6e, 0x22, 0x97, 0x47, 0x89,
			0xb1, 0xdc, 0x20, 0xd1,
		},
		{
			0x2d, 0xd5, 0x9e, 0x37, 0x76, 0x6c, 0x66, 0x75, 0x71, 0xb7, 0x79, 0xc0,
			0x6e, 0x12, 0xba, 0x21, 0x91, 0x88, 0x48, 0x97, 0x72, 0xf4, 0x86, 0x31,
			0xa6, 0x72, 0x8b, 0x5b, 0x86, 0x7e, 0x3c, 0xf4,
		},
		{
			0x96, 0x6d, 0x94, 0x20, 0x38, 0x03, 0x05, 0x09, 0xb2, 0x0e, 0x61, 0x00,
			0x62, 0x04, 0x2b, 0x6b, 0xf1, 0x04, 0x81, 0x28, 0x18, 0x89, 0x32, 0x92,
			0xa6, 0x8d, 0x57, 0xd1, 0xce, 0x86, 0x51, 0x51,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x3e, 0x6a, 0xcd, 0x8b, 0x4e, 0x85, 0xb4, 0xa0, 0xf7, 0x92, 0x8f, 0x6b,
			0xd4, 0x1a, 0x8e, 0x6b, 0x52, 0x52, 0x4f, 0xe7, 0x27, 0x23, 0xa0, 0x50,
			0x96, 0x37, 0xd3, 0x3f, 0x15, 0xaf, 0xe7, 0xd8, 0xda, 0x6a, 0x15, 0x20,
			0x9b, 0x9e, 0x41, 0x49, 0x57, 0x6f, 0xbb, 0x1f, 0xd8, 0x31, 0xf7, 0x84,
			0xc0, 0x44, 0x39, 0xab, 0xda, 0x46, 0x05, 0xd0, 0x65, 0x56, 0xdc, 0x30,
			0x02, 0x05, 0x5b, 0x58, 0x55, 0xfb, 0xa2, 0x01, 0xf6, 0xda, 0xef, 0x79,
			0xf7, 0x8d, 0x00, 0x1e, 0xd6, 0x9e, 0xca, 0x8a, 0x41, 0x85, 0x13, 0xd0,
			0x24, 0x64, 0xe8, 0xd7, 0x42, 0xc2, 0x79, 0x9c, 0xd6, 0x8e, 0xdf, 0xbe,
			0x88, 0xae, 0x9b, 0x35, 0xa0, 0xaa, 0x06, 0x5c, 0x42, 0xa4, 0x77, 0x00,
			0x58, 0xc4, 0xb0, 0x26, 0xd0, 0x35, 0x0a, 0x7a, 0xfa, 0x9c, 0x52, 0xc3,
			0xc7, 0xfa, 0x05, 0x4f, 0x8a, 0x96, 0xd8, 0x87,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(59, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_60(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x98, 0x29, 0xd2, 0xe5, 0x40, 0x96, 0x29, 0xd9, 0xc5, 0x3c, 0x7d, 0xbb,
			0x16, 0xc6, 0x19, 0xb6, 0x16, 0xb0, 0xd1, 0x73, 0x94, 0xa3, 0xe0, 0x55,
			0xbf, 0x6f, 0xae, 0x9f, 0xcb, 0xbb, 0xaf, 0xd4,
		},
		{
			0xe1, 0x5d, 0x27, 0x62, 0x63, 0x15, 0x16, 0x80, 0x7c, 0x1f, 0x07, 0x33,
			0x82, 0xe8, 0x52, 0x12,
		},
		{
			0x25, 0x56, 0xe3, 0xd7, 0x5e, 0xef, 0xe6, 0xb6, 0x17, 0x13, 0x28, 0xe6,
			0x60, 0xbb, 0x65, 0x58, 0xda, 0xdf, 0xfa, 0xa1, 0xe0, 0x32, 0xe7, 0xeb,
			0xb6, 0xb3, 0x26, 0x2a, 0xdd, 0xe0, 0xdd, 0xeb,
		},
		{
			0xa4, 0xf4, 0xe2, 0x8b, 0x33, 0x0b, 0x73, 0x38, 0x2f, 0xcc, 0x5c, 0x85,
			0x7e, 0xd6, 0x2e, 0x94, 0x5f, 0xd9, 0x71, 0x55, 0x40, 0x64, 0x54, 0x54,
			0x37, 0xa5, 0xd2, 0x2d, 0xdc, 0xbf, 0xf3, 0xee,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x74, 0x74, 0x02, 0xc5, 0x0d, 0xcc, 0xc3, 0xa6, 0x66, 0xd9, 0x4d, 0x2d,
			0x6f, 0x6f, 0x43, 0xfd, 0x2b, 0x50, 0x53, 0x84, 0x73, 0x3c, 0x97, 0x62,
			0xc2, 0xef, 0x84, 0x93, 0x6f, 0xb3, 0x29, 0xc3, 0x66, 0x54, 0x12, 0x88,
			0xfb, 0xc5, 0x78, 0xe9, 0x39, 0x7b, 0x5b, 0x11, 0x98, 0xf1, 0x66, 0x5f,
			0x8c, 0x37, 0x3d, 0xe8, 0x55, 0x6d, 0x3c, 0x08, 0xc6, 0x61, 0x9a, 0xa3,
			0xb3, 0x55, 0xdb, 0x20, 0x6f, 0x84, 0x0b, 0x6a, 0xa4, 0xcc, 0x72, 0x20,
			0x6d, 0x7f, 0xef, 0xb2, 0x43, 0x5d, 0x6b, 0x02, 0x4e, 0x4c, 0x15, 0x68,
			0xe9, 0xab, 0x31, 0xf7, 0x11, 0x0f, 0x43, 0x11, 0xb0, 0x5d, 0x79, 0x96,
			0xce, 0xba, 0x6e, 0x5e, 0x6b, 0xb9, 0x00, 0xf8, 0xe7, 0x21, 0xae, 0x10,
			0x6a, 0x95, 0x11, 0xf8, 0x92, 0x6d, 0xe2, 0xf8, 0x48, 0x71, 0x00, 0x69,
			0x8a, 0xa8, 0xfe, 0x99, 0x53, 0x88, 0xa0, 0x72,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(60, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_61(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x6a, 0xe8, 0x03, 0x03, 0x29, 0x23, 0x91, 0x33, 0x5b, 0xf9, 0xc9, 0x38,
			0x7f, 0xbd, 0x3b, 0xf6, 0x15, 0x75, 0x6c, 0x9c, 0x27, 0xc3, 0x47, 0x8c,
			0x87, 0xe2, 0x60, 0xcf, 0x97, 0xd4, 0x71, 0x10,
		},
		{
			0x01, 0xe1, 0x62, 0x47, 0xdd, 0x4c, 0xae, 0x64, 0x99, 0x33, 0x7d, 0x82,
			0x78, 0x4e, 0xa5, 0x7f,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x03, 0x57, 0x02, 0xef, 0x4e, 0x11, 0x2b, 0x17, 0x31, 0x12, 0xc5, 0x85,
			0x1d, 0x07, 0xb2, 0x79, 0x30, 0x98, 0x63, 0x74, 0x0d, 0x38, 0xd0, 0xd0,
			0x72, 0x02, 0x23, 0xe2, 0x40, 0x17, 0xbb, 0xc0,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xcf, 0x43, 0x15, 0x59, 0x8f, 0xcd, 0x6a, 0xf1, 0x31, 0x55, 0x18, 0xc4,
			0xbf, 0xba, 0xc0, 0x54, 0x0c, 0x58, 0x96, 0x35, 0x27, 0x35, 0x48, 0xa7,
			0xb5, 0x07, 0xe7, 0xd2, 0xe6, 0x85, 0xe5, 0x94, 0x7b, 0x87, 0xae, 0x25,
			0x7e, 0x58, 0xfa, 0xf2, 0x14, 0xf2, 0xb5, 0x8e, 0xd1, 0x0c, 0x3b, 0xd3,
			0x5f, 0x75, 0xf6, 0xc3, 0x5d, 0xd6, 0xd4, 0x41, 0xc9, 0x3b, 0xcd, 0x42,
			0xe7, 0x17, 0x20, 0x10, 0x26, 0x31, 0xb1, 0xa6, 0xa4, 0xba, 0x24, 0x7c,
			0x17, 0x5e, 0xd8, 0x00, 0xcf, 0xca, 0x6e, 0x1e, 0x83, 0x9b, 0x5a, 0xa9,
			0x07, 0x60, 0x4c, 0xcf, 0xe6, 0xf9, 0x84, 0xf6, 0x82, 0x2e, 0x00, 0x1a,
			0xb0, 0x2d, 0xd6, 0x63, 0x49, 0x64, 0xf7, 0x89, 0xcb, 0x10, 0x7a, 0x97,
			0x73, 0x46, 0x69, 0x3f, 0x32, 0x44, 0xc8, 0x95, 0xe8, 0x40, 0xdf, 0xa0,
			0xed, 0xf7, 0xf1, 0x4d, 0xc6, 0x1d, 0x79, 0x4f,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(61, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_62(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xda, 0x2c, 0x40, 0x6c, 0x1a, 0x40, 0xf1, 0xbe, 0x98, 0xfd, 0xb2, 0x86,
			0x07, 0x8a, 0x92, 0x7f, 0x60, 0x6a, 0xe8, 0xec, 0x91, 0xab, 0x3a, 0x66,
			0x3f, 0xe2, 0x13, 0xfd, 0x8f, 0x05, 0x23, 0xf9,
		},
		{
			0x4b, 0x49, 0x85, 0x7c, 0x62, 0x5f, 0x7a, 0x6c, 0xa2, 0x66, 0x9a, 0xba,
			0x21, 0x3d, 0x79, 0xcb,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x52, 0xc1, 0xc0, 0x06, 0xc5, 0x77, 0x96, 0x91, 0x79, 0xcc, 0x1f, 0xef,
			0xff, 0x1b, 0xb0, 0xdc, 0x42, 0x25, 0x5c, 0x35, 0x8b, 0x68, 0x6c, 0x73,
			0x21, 0x6d, 0xa1, 0x21, 0x6f, 0x82, 0xff, 0xfb,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x5b, 0x6e, 0x8e, 0xf0, 0x30, 0x07, 0x12, 0xaa, 0xeb, 0x44, 0xa5, 0x2f,
			0x91, 0xca, 0x2f, 0x3d, 0xee, 0xfc, 0xee, 0xa9, 0x5c, 0xe2, 0x8e, 0x8c,
			0x4d, 0x93, 0x10, 0x12, 0x5a, 0x7d, 0x25, 0x9c, 0x34, 0x23, 0xb2, 0xcd,
			0x66, 0x2b, 0x33, 0x37, 0xae, 0xdf, 0x7d, 0xf1, 0x27, 0x19, 0xaf, 0x7b,
			0xe6, 0xca, 0xcb, 0x3a, 0x65, 0x65, 0x07, 0x0f, 0xaf, 0xc9, 0x86, 0xc3,
			0x36, 0x83, 0x6a, 0xa2, 0xe1, 0x8c, 0x23, 0x7b, 0x23, 0xff, 0x19, 0x33,
			0x9b, 0x4b, 0x5e, 0xf7, 0x04, 0x63, 0x4f, 0x16, 0x26, 0xd6, 0x6b, 0xce,
			0x9e, 0x72, 0x55, 0x51, 0x83, 0x74, 0x2b, 0x59, 0x13, 0x35, 0x82, 0x1c,
			0xe9, 0xc3, 0xa9, 0x5f, 0x2a, 0x1b, 0xea, 0xb6, 0xb0, 0x65, 0x9f, 0xfe,
			0x6e, 0x94, 0x0b, 0x55, 0x9d, 0xfb, 0x8a, 0x8d, 0xf1, 0xa4, 0x73, 0x30,
			0x8c, 0x31, 0x54, 0x58, 0x4e, 0xbf, 0x96, 0x22,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(62, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_63(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x5b, 0x77, 0xe8, 0x85, 0x05, 0x6b, 0x4f, 0xaf, 0x9d, 0x8e, 0xf9, 0xc8,
			0x57, 0x30, 0xfd, 0x50, 0x75, 0xf0, 0xa0, 0x70, 0x15, 0xc6, 0xdb, 0xd7,
			0x32, 0x3c, 0xe6, 0x3e, 0xa5, 0xd7, 0x1e, 0x42,
		},
		{
			0x9a, 0xc3, 0xae, 0x89, 0xcf, 0xb2, 0x86, 0xb3, 0x11, 0x49, 0xe7, 0x8b,
			0xce, 0xd8, 0xcf, 0xf0,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x0d, 0xa5, 0xab, 0x25, 0x7b, 0x38, 0x96, 0x73, 0x85, 0x2c, 0xde, 0x48,
			0x00, 0xc0, 0x5a, 0xca, 0x59, 0x6e, 0xc5, 0x7c, 0x25, 0xd7, 0xda, 0x45,
			0x19, 0x11, 0x48, 0x7a, 0x5b, 0xa6, 0xbd, 0xaa,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x1d, 0x60, 0x4f, 0x4d, 0x5b, 0xb1, 0x0e, 0x11, 0x4a, 0xed, 0x67, 0x8a,
			0x51, 0x39, 0xbb, 0x46, 0xee, 0x69, 0x55, 0xd8, 0x6e, 0x06, 0x3d, 0x17,
			0xca, 0x5e, 0xf6, 0xdb, 0x38, 0x71, 0x51, 0x43, 0x0d, 0xb2, 0x3a, 0xa0,
			0xb4, 0x8a, 0x34, 0x3b, 0x2d, 0x35, 0x00, 0x3d, 0x29, 0x5f, 0xd0, 0x24,
			0xfc, 0x34, 0x4e, 0x7c, 0x01, 0x06, 0x49, 0x24, 0x84, 0x69, 0xda, 0x7d,
			0x9a, 0xb7, 0xe2, 0x50, 0x97, 0xbe, 0x35, 0xfe, 0xda, 0xd1, 0xba, 0x89,
			0xad, 0x66, 0xaf, 0x96, 0x80, 0x9c, 0x64, 0xba, 0x3b, 0x2e, 0xfe, 0x60,
			0x1c, 0x1d, 0x6c, 0x20, 0x1c, 0xa5, 0x5e, 0xa8, 0x2d, 0x90, 0x1a, 0xf1,
			0x46, 0xf2, 0x00, 0x29, 0x1b, 0x11, 0x3f, 0xbb, 0x05, 0x9a, 0x20, 0x43,
			0x28, 0x5a, 0x8d, 0x88, 0xee, 0x65, 0x5d, 0xa7, 0x6f, 0x83, 0x79, 0x08,
			0xec, 0xfc, 0x9b, 0xc0, 0xbb, 0xae, 0xc1, 0xc3,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(63, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_64(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x40, 0x30, 0xb2, 0xa9, 0x83, 0xa8, 0xe2, 0x14, 0xc9, 0xb6, 0x05, 0x3f,
			0x47, 0x4d, 0xeb, 0xb1, 0x35, 0x52, 0x2e, 0x2b, 0x0d, 0x50, 0xae, 0x98,
			0x44, 0x1b, 0xef, 0x99, 0xb9, 0x3e, 0x63, 0xb4,
		},
		{
			0xd6, 0x29, 0x2c, 0x76, 0x33, 0xac, 0xc7, 0xa1, 0x7e, 0x79, 0x7a, 0x14,
			0x99, 0x43, 0xe6, 0xe4,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xea, 0x9f, 0xf3, 0x34, 0x4c, 0xba, 0x86, 0x34, 0xe8, 0x05, 0x8f, 0x09,
			0x35, 0x10, 0x51, 0x9c, 0x4c, 0xb9, 0x3a, 0x1a, 0xb4, 0x7c, 0x31, 0x23,
			0x67, 0x42, 0x95, 0x28, 0x0a, 0x26, 0x30, 0x9c,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xe9, 0x8a, 0x2d, 0x67, 0xd2, 0xd5, 0x2d, 0xc6, 0x12, 0x43, 0x7c, 0x57,
			0x87, 0xad, 0xad, 0x2f, 0x5d, 0xaa, 0x82, 0x90, 0x42, 0x3c, 0x45, 0xe5,
			0x90, 0x94, 0x55, 0xe7, 0x55, 0xf2, 0xd4, 0xa7, 0x29, 0xa6, 0xe6, 0xcb,
			0x42, 0xf3, 0x6b, 0x0e, 0x40, 0x57, 0x1f, 0x1d, 0x77, 0x21, 0xb4, 0x0e,
			0x3d, 0x4f, 0x77, 0x3d, 0x84, 0x5e, 0x72, 0x6e, 0x8e, 0x1e, 0x5b, 0xd9,
			0x50, 0x4a, 0x94, 0x59, 0x43, 0xf3, 0x34, 0x60, 0x3f, 0x28, 0x2b, 0xfc,
			0x7c, 0x5d, 0x8b, 0x2a, 0x6a, 0x1c, 0x82, 0xcc, 0xa1, 0xee, 0x94, 0x3b,
			0xe6, 0xc4, 0x64, 0xbc, 0xf8, 0xa4, 0xdf, 0x47, 0x0d, 0x3d, 0xdc, 0xd7,
			0xea, 0xb1, 0xe8, 0xbb, 0xa0, 0x09, 0x0e, 0x53, 0x7c, 0x36, 0x70, 0xf4,
			0x24, 0x43, 0x8e, 0x3a, 0x0b, 0x19, 0x72, 0x97, 0xa7, 0xa1, 0x6d, 0x12,
			0xa2, 0x82, 0x55, 0x8d, 0x55, 0x6f, 0x3b, 0x5e,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(64, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_65(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x94, 0x7b, 0x3f, 0x93, 0xa6, 0x01, 0xda, 0x15, 0xaa, 0xec, 0x58, 0x07,
			0x8d, 0x19, 0x8c, 0x20, 0xc9, 0x13, 0x79, 0xd5, 0xbf, 0x1c, 0x0d, 0xea,
			0x32, 0x96, 0x91, 0xf4, 0x7b, 0x3c, 0xdb, 0x40,
		},
		{
			0x97, 0x69, 0x2e, 0xbc, 0x8c, 0x9b, 0x04, 0x5b, 0x73, 0xe8, 0x23, 0x50,
			0x0e, 0xba, 0x50, 0x77,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x59, 0x9f, 0xe8, 0x4d, 0x8a, 0x84, 0xc3, 0x06, 0xee, 0x33, 0x4a, 0x3b,
			0x44, 0x30, 0x05, 0xe3, 0x3b, 0xa0, 0xf8, 0xf5, 0x9b, 0x63, 0x6e, 0x67,
			0xd7, 0x4f, 0x8c, 0x59, 0x65, 0x58, 0xcf, 0x8e,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x38, 0x8d, 0xe4, 0xee, 0x5c, 0x19, 0xad, 0x97, 0x2c, 0x65, 0x0c, 0x86,
			0x5e, 0xc3, 0x4f, 0x7d, 0xb6, 0x69, 0x66, 0x9d, 0x32, 0x6f, 0x0c, 0x55,
			0x89, 0xc5, 0xab, 0x24, 0x10, 0xd6, 0x4f, 0x2d, 0x27, 0x14, 0x84, 0x3c,
			0xf5, 0xa2, 0x28, 0x8e, 0x37, 0x95, 0xfa, 0xe7, 0xf5, 0x41, 0xfd, 0xa4,
			0xbd, 0x4d, 0x48, 0x84, 0x01, 0x4c, 0xc7, 0x75, 0x8e, 0xc2, 0x97, 0x3b,
			0xce, 0x26, 0xa6, 0xb4, 0xc4, 0xda, 0x09, 0x11, 0xde, 0xba, 0x01, 0x64,
			0x24, 0x63, 0x0d, 0x30, 0xd5, 0xf3, 0x53, 0x6e, 0x4e, 0x79, 0x77, 0xdf,
			0xc7, 0x0f, 0x40, 0xb5, 0x30, 0x8f, 0x38, 0xef, 0x6e, 0x5a, 0x3d, 0xd7,
			0xa0, 0x60, 0xda, 0x5c, 0x68, 0xcc, 0xe2, 0x75, 0x05, 0x74, 0x5f, 0x57,
			0x68, 0x24, 0x5a, 0xb3, 0xc3, 0x1f, 0xdc, 0xc5, 0x40, 0xd3, 0x76, 0x7b,
			0x9f, 0xa6, 0x2d, 0x41, 0x71, 0xc7, 0x15, 0xc7,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(65, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_66(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xc8, 0x68, 0xe7, 0x22, 0x1f, 0xe5, 0x65, 0xc1, 0x3e, 0x46, 0xba, 0x78,
			0x14, 0x17, 0xda, 0x8c, 0xf4, 0x2b, 0xed, 0xc9, 0xbf, 0x5e, 0xca, 0xb8,
			0x98, 0xf6, 0x4f, 0x53, 0xa8, 0xe5, 0xbe, 0x2f,
		},
		{
			0x49, 0x24, 0x05, 0x87, 0xc2, 0xf9, 0xe2, 0xc9, 0xaf, 0xae, 0x3a, 0x7a,
			0xa7, 0x06, 0x20, 0x80,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xa6, 0x98, 0x20, 0x3c, 0x5f, 0x5e, 0x55, 0x9d, 0x5f, 0x08, 0xdf, 0x28,
			0x23, 0x0a, 0xb3, 0xb1, 0xff, 0x40, 0x59, 0xce, 0x8e, 0x8b, 0x2b, 0x96,
			0x76, 0x88, 0x03, 0x13, 0xd1, 0x34, 0xf7, 0xfd,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x64, 0xe6, 0x9b, 0x9c, 0xdd, 0x82, 0x94, 0x57, 0x3c, 0x67, 0xe0, 0x6c,
			0xd6, 0x62, 0x66, 0x2e, 0x98, 0xcb, 0xca, 0xb9, 0xb8, 0xfc, 0x8c, 0x9a,
			0xd7, 0xd3, 0x66, 0xc3, 0xf9, 0x78, 0xf1, 0x0a, 0xf1, 0xbb, 0xd9, 0x93,
			0x4a, 0xdb, 0x9a, 0x31, 0x22, 0x8d, 0x0c, 0x51, 0xa2, 0x68, 0xac, 0xb7,
			0x76, 0xe3, 0xdf, 0x0d, 0x9a, 0x5d, 0xc4, 0x13, 0xda, 0x62, 0x36, 0xe7,
			0x7c, 0x15, 0xf2, 0x2d, 0xff, 0x41, 0x09, 0x11, 0x5e, 0x2e, 0x7a, 0x09,
			0xd4, 0x5d, 0x43, 0xf5, 0xa2, 0x69, 0x10, 0x4a, 0x71, 0x7e, 0x10, 0x1f,
			0x2c, 0x84, 0x27, 0x04, 0x9a, 0xb6, 0x58, 0x8d, 0xaf, 0x89, 0x07, 0xe3,
			0x31, 0x68, 0xee, 0xbe, 0xde, 0xcc, 0x47, 0x01, 0x90, 0x65, 0x57, 0xf9,
			0x7f, 0xd3, 0x4b, 0x72, 0x46, 0x42, 0xf2, 0x85, 0xc0, 0xb0, 0x6a, 0xbc,
			0x3c, 0xa1, 0x5b, 0xe6, 0x87, 0xc9, 0xbe, 0x1d,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(66, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_67(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x5b, 0x75, 0xc3, 0xc8, 0x00, 0x0a, 0x81, 0x72, 0x18, 0x24, 0xe9, 0x67,
			0x66, 0x62, 0x85, 0x12, 0x57, 0xff, 0x31, 0x5b, 0x9f, 0x20, 0x65, 0x63,
			0x1e, 0x4b, 0x1c, 0x31, 0xa8, 0x24, 0xed, 0x0f,
		},
		{
			0x18, 0x71, 0xbf, 0x15, 0xff, 0x16, 0xe9, 0x4c, 0x9c, 0x8d, 0x93, 0xda,
			0x29, 0x51, 0x55, 0x66,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x99, 0x48, 0x67, 0xb4, 0x13, 0x45, 0x9c, 0x17, 0xa1, 0xe5, 0x07, 0x69,
			0xcb, 0x2d, 0xc3, 0x0a, 0x00, 0x12, 0xc8, 0xd5, 0x94, 0x74, 0x2e, 0xd4,
			0xea, 0x2c, 0x2a, 0xc4, 0x34, 0xbe, 0x33, 0x86,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x2b, 0x40, 0xdb, 0x08, 0xa5, 0xe1, 0x5e, 0xb1, 0x81, 0x7f, 0x05, 0x14,
			0x1c, 0xb0, 0xb5, 0x86, 0x46, 0x84, 0xea, 0x4e, 0x10, 0x04, 0xc2, 0x60,
			0x98, 0xb4, 0x83, 0x11, 0x9b, 0xde, 0x01, 0x75, 0x26, 0xdf, 0x8d, 0x39,
			0x3a, 0xbd, 0x6a, 0x26, 0x02, 0xf4, 0x2f, 0xe3, 0x52, 0x24, 0xea, 0x7d,
			0xee, 0xd2, 0xf6, 0x40, 0x3d, 0x92, 0xc4, 0x1d, 0xf8, 0x01, 0xab, 0x93,
			0x65, 0xce, 0x24, 0xa6, 0xa9, 0xae, 0xa9, 0x08, 0xdc, 0xf6, 0x11, 0x3e,
			0x02, 0xcb, 0x67, 0xae, 0x13, 0x01, 0x5b, 0x61, 0x74, 0xf2, 0x2c, 0xd9,
			0x43, 0x31, 0xa4, 0x79, 0xab, 0x03, 0xb7, 0xa1, 0x9d, 0xe2, 0x81, 0xe0,
			0x69, 0xa1, 0x7b, 0x60, 0x0c, 0x66, 0x49, 0xd7, 0xb8, 0x8e, 0x5b, 0xa3,
			0xf3, 0x40, 0xfb, 0xdb, 0x53, 0x7a, 0x9e, 0x45, 0x81, 0x49, 0x6c, 0x8e,
			0xaa, 0x40, 0xe5, 0xb8, 0x88, 0x62, 0xc4, 0xb0,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(67, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_68(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xbb, 0x44, 0x58, 0xcc, 0xeb, 0x0d, 0xce, 0x73, 0x32, 0xf3, 0x0c, 0xe9,
			0x0f, 0x69, 0xd8, 0x28, 0x25, 0xb5, 0x47, 0xbb, 0x51, 0x2a, 0xfb, 0xae,
			0x3e, 0x1c, 0xba, 0xaa, 0x4c, 0x17, 0x2b, 0x7b,
		},
		{
			0x60, 0xbd, 0x6a, 0x33, 0xbe, 0xe6, 0x1d, 0x1a, 0x9b, 0xbf, 0xdd, 0xfe,
			0xe6, 0x0f, 0x7a, 0x18,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xe8, 0x8a, 0x2a, 0xee, 0x17, 0xcc, 0x49, 0x16, 0x40, 0xf2, 0xe0, 0xa6,
			0x93, 0xf0, 0x3e, 0xbc, 0x48, 0xa9, 0x57, 0x90, 0x31, 0x86, 0x1d, 0xf3,
			0x1c, 0xdb, 0x42, 0x64, 0x99, 0x7f, 0x8a, 0x64,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x5b, 0xea, 0x97, 0xa6, 0xda, 0xae, 0xe5, 0x40, 0xab, 0x6d, 0x41, 0x7c,
			0xa4, 0x58, 0x7c, 0x19, 0x8f, 0x58, 0x9d, 0xb5, 0xac, 0x72, 0xb4, 0xf1,
			0x38, 0x4b, 0x1b, 0x1d, 0x8c, 0x03, 0xec, 0x73, 0xb2, 0x97, 0x93, 0x9f,
			0x5a, 0x8b, 0x82, 0x40, 0xfd, 0x10, 0x33, 0xca, 0xce, 0x0e, 0x97, 0x1e,
			0x3c, 0x3f, 0x49, 0x23, 0x9b, 0x16, 0x85, 0x76, 0xf8, 0x3d, 0xce, 0x89,
			0x53, 0x37, 0x87, 0xfc, 0xb7, 0x25, 0x37, 0xb3, 0x2f, 0x58, 0x5d, 0xf1,
			0x35, 0x50, 0x64, 0xf7, 0xd7, 0x37, 0xb6, 0xc2, 0x52, 0x59, 0x5f, 0x16,
			0xc4, 0x2d, 0x77, 0xb8, 0x37, 0xfb, 0x7d, 0x4d, 0x44, 0xbd, 0x84, 0xdb,
			0xe8, 0x4d, 0x0b, 0xb5, 0xa4, 0x22, 0x44, 0x87, 0xfa, 0xe3, 0xca, 0x26,
			0xe9, 0x30, 0xe5, 0x4c, 0x27, 0xe2, 0xeb, 0xdd, 0xe9, 0x92, 0xe7, 0x6d,
			0xc4, 0xc4, 0x07, 0x77, 0x7a, 0xa4, 0xbd, 0x11,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(68, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_69(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x7e, 0x70, 0x38, 0x1b, 0xfb, 0xa7, 0x80, 0x7b, 0xc7, 0x73, 0xe2, 0xcd,
			0x68, 0x3f, 0x90, 0xc7, 0xad, 0x60, 0x6d, 0xcf, 0x46, 0x27, 0x67, 0x98,
			0x8b, 0x3f, 0x6c, 0x75, 0x8f, 0x5c, 0x79, 0x5d,
		},
		{
			0x7e, 0x5a, 0xb4, 0x42, 0x70, 0x29, 0xe9, 0x9d, 0xf2, 0x20, 0x1d, 0x81,
			0xc3, 0xff, 0x28, 0x4e,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xe0, 0xdd, 0x5b, 0x4e, 0x7c, 0xa7, 0x2a, 0xfd, 0xe5, 0x3c, 0x16, 0xb8,
			0x22, 0x54, 0x03, 0xbb, 0x6e, 0x74, 0x97, 0x42, 0xf3, 0xe3, 0x88, 0x96,
			0xfb, 0xf8, 0xf1, 0x8b, 0x7c, 0xe6, 0xf7, 0x99,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x8a, 0xf2, 0x7c, 0x16, 0x3d, 0x25, 0xec, 0xf4, 0xf5, 0x39, 0x94, 0x4d,
			0x94, 0x50, 0xf5, 0x11, 0x6e, 0xa8, 0xaa, 0x7f, 0x97, 0x36, 0x1c, 0x69,
			0x6f, 0x8d, 0xc7, 0xd8, 0xb3, 0xaf, 0xe6, 0xff, 0xc0, 0x20, 0x14, 0x59,
			0xa1, 0x73, 0x3d, 0x9e, 0xe0, 0x9b, 0x2e, 0x45, 0xf3, 0xb6, 0xcd, 0x5f,
			0x45, 0xd9, 0xf9, 0xf4, 0x5f, 0xdc, 0x66, 0x2b, 0x56, 0x90, 0x59, 0x89,
			0x1f, 0x2e, 0x9e, 0xbc, 0x74, 0x0f, 0x78, 0x40, 0x4a, 0x0b, 0xb1, 0xc4,
			0x6a, 0x6b, 0xfe, 0xee, 0xed, 0x9c, 0x6b, 0x55, 0x2c, 0x5c, 0x46, 0x8a,
			0x49, 0x96, 0xad, 0x41, 0xcf, 0xff, 0x77, 0xae, 0x4d, 0x98, 0x96, 0xcb,
			0x80, 0xaf, 0x27, 0x63, 0x5a, 0x4e, 0x2a, 0x1b, 0x5c, 0x07, 0xc1, 0xb5,
			0x8b, 0x49, 0xc9, 0xa7, 0x68, 0x51, 0x14, 0x4e, 0x69, 0x03, 0x7e, 0x14,
			0xf7, 0xb8, 0x05, 0x16, 0x13, 0x29, 0xca, 0x15,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(69, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_70(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x2e, 0x54, 0x32, 0xac, 0xda, 0xb8, 0xca, 0x81, 0xee, 0x56, 0xf3, 0xe6,
			0x9a, 0x0c, 0x74, 0x8a, 0x72, 0xf8, 0x3e, 0x11, 0xd1, 0x8c, 0x2e, 0xf8,
			0xfc, 0x8a, 0xad, 0x56, 0x8e, 0x19, 0xe9, 0x1c,
		},
		{
			0xca, 0x5f, 0x6e, 0xb3, 0x25, 0xe4, 0xa2, 0xb2, 0x4c, 0x00, 0x91, 0x5b,
			0xc3, 0x7d, 0x99, 0x41,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xae, 0x3d, 0x7a, 0xc0, 0x62, 0x2f, 0x7d, 0xc0, 0x69, 0xe5, 0x2c, 0xbd,
			0x15, 0x84, 0x47, 0x29, 0x79, 0xe1, 0xc8, 0xa9, 0x6b, 0xf4, 0x61, 0xc8,
			0xd3, 0x97, 0x6d, 0x8d, 0x53, 0x5c, 0x10, 0xec,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xfd, 0xa5, 0xdb, 0x4b, 0x28, 0xbb, 0xc5, 0xd2, 0xf6, 0xbd, 0x33, 0xac,
			0x98, 0x39, 0x9b, 0xcf, 0xcf, 0xa4, 0x60, 0x37, 0x47, 0x8c, 0xf2, 0x69,
			0xbf, 0x35, 0x9d, 0xb2, 0x28, 0x28, 0x06, 0x62, 0x71, 0x8f, 0x6d, 0x4c,
			0xa4, 0x77, 0xbb, 0x1c, 0xac, 0xa5, 0x67, 0x30, 0x9a, 0xba, 0xab, 0x08,
			0x32, 0xe7, 0x88, 0x58, 0x84, 0xe8, 0x6c, 0x4e, 0x0b, 0x01, 0x7e, 0x86,
			0x77, 0x74, 0x9c, 0x5d, 0xb2, 0xc3, 0x3e, 0x65, 0xfa, 0x53, 0x8d, 0xf4,
			0xae, 0xc4, 0x46, 0xc7, 0xca, 0x72, 0x4a, 0xf1, 0x8e, 0x45, 0x98, 0x36,
			0x01, 0x78, 0x5f, 0x98, 0x00, 0xa0, 0xc5, 0xb2, 0x3f, 0xea, 0xfb, 0xbd,
			0x34, 0x9d, 0x20, 0x4a, 0xe5, 0x4b, 0x40, 0xa0, 0x19, 0xa5, 0x7a, 0x83,
			0x68, 0xf0, 0x7d, 0x70, 0x94, 0x5f, 0x22, 0xbe, 0x24, 0xaa, 0x99, 0x2f,
			0x93, 0x0f, 0x92, 0xa6, 0x26, 0xbb, 0x22, 0xb6,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(70, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_71(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xef, 0xfa, 0xaf, 0x06, 0xcd, 0x99, 0x7b, 0x99, 0x78, 0x33, 0x72, 0xc5,
			0x15, 0x67, 0x2e, 0xde, 0x1a, 0x74, 0x16, 0x22, 0xa2, 0x04, 0x39, 0x53,
			0x45, 0x1c, 0x33, 0x66, 0x8e, 0x8d, 0xf4, 0x65,
		},
		{
			0xfb, 0x92, 0x29, 0x10, 0xbc, 0x6c, 0x2a, 0xd0, 0x46, 0x1c, 0x98, 0x09,
			0xd1, 0x8a, 0x23, 0xf9,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x53, 0xce, 0x26, 0xba, 0xbc, 0x29, 0x4d, 0x2f, 0x1b, 0x48, 0xb5, 0x8a,
			0xec, 0xad, 0xab, 0x33, 0x08, 0xd8, 0x71, 0x18, 0x92, 0xcf, 0xa3, 0x93,
			0xf3, 0x58, 0x33, 0x5f, 0xc0, 0xf4, 0x67, 0x4e,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x8a, 0xe5, 0x35, 0xa3, 0xcc, 0x00, 0xda, 0x2c, 0x81, 0x7e, 0x7f, 0x38,
			0x9a, 0x9b, 0xc9, 0x97, 0xd8, 0xe7, 0x20, 0xe3, 0x94, 0xbc, 0xd3, 0xa6,
			0xc2, 0x3e, 0xc6, 0x78, 0xed, 0x9d, 0xa4, 0x8d, 0x3d, 0x83, 0xe3, 0x2f,
			0x73, 0xd8, 0x96, 0xda, 0x58, 0x2a, 0x5c, 0x2b, 0x0f, 0x44, 0x12, 0x7b,
			0x3f, 0x7e, 0xd3, 0x6d, 0x91, 0xfc, 0x87, 0x7b, 0xbe, 0x91, 0x33, 0x90,
			0x64, 0x89, 0x0a, 0x4c, 0x2d, 0xb0, 0x5f, 0x16, 0xb5, 0x53, 0x61, 0x1d,
			0x68, 0x72, 0xc8, 0xbf, 0x40, 0x84, 0x34, 0x4c, 0x5d, 0xae, 0x08, 0xca,
			0x48, 0x17, 0x2f, 0x98, 0x22, 0x24, 0x4d, 0x4f, 0xf3, 0x72, 0xe0, 0x4c,
			0xe9, 0x9a, 0xed, 0x28, 0xc3, 0x94, 0x4c, 0x47, 0xfa, 0x43, 0x4e, 0x20,
			0xe9, 0x97, 0x30, 0xba, 0xf2, 0xbe, 0x9e, 0xe8, 0x52, 0x47, 0xf1, 0xb9,
			0x4b, 0x12, 0x9c, 0x3f, 0x6c, 0x3e, 0x0f, 0xe0,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(71, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_72(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x4a, 0x35, 0x42, 0xce, 0xa1, 0xb5, 0xa7, 0xb8, 0x93, 0x85, 0x15, 0x32,
			0x54, 0x98, 0x84, 0xa9, 0xeb, 0xa9, 0x36, 0x95, 0xbe, 0x8d, 0x76, 0x76,
			0x1d, 0xb6, 0x98, 0x31, 0xbd, 0x10, 0x53, 0xee,
		},
		{
			0xff, 0x03, 0x20, 0xc4, 0x7e, 0x47, 0xb4, 0x40, 0x71, 0x8b, 0x80, 0xab,
			0x61, 0x6d, 0x7f, 0xff,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x86, 0x9a, 0xe8, 0xff, 0x94, 0x9a, 0x10, 0xc9, 0x3c, 0x05, 0xd6, 0x20,
			0xe4, 0xc4, 0xf1, 0x3d, 0x03, 0x76, 0xee, 0x1e, 0x00, 0xc7, 0x1b, 0xba,
			0x58, 0x83, 0xb3, 0xbc, 0x41, 0xf5, 0x24, 0x7b,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xad, 0xda, 0x83, 0xc6, 0x11, 0x88, 0x3e, 0x01, 0xf9, 0x58, 0xfe, 0xbf,
			0xf1, 0x05, 0x2b, 0x65, 0x8b, 0x0c, 0x84, 0xf8, 0x57, 0xad, 0x8d, 0x7d,
			0x18, 0xde, 0x89, 0x7d, 0xe2, 0xa8, 0xee, 0xaf, 0x39, 0x53, 0xd9, 0x40,
			0x7c, 0xbc, 0x4c, 0x05, 0x9c, 0x9f, 0xf9, 0xbf, 0xe6, 0x77, 0x44, 0xe1,
			0x34, 0x06, 0xcb, 0xfb, 0x09, 0xed, 0x31, 0xdc, 0x2f, 0xbc, 0xd9, 0x62,
			0x61, 0x29, 0xd0, 0xb6, 0xe4, 0xa8, 0x0a, 0xc3, 0xc1, 0xb2, 0x86, 0xd8,
			0x1f, 0x6f, 0x28, 0x18, 0x3a, 0x3e, 0x8c, 0xe2, 0x66, 0x68, 0x60, 0x05,
			0x60, 0x08, 0x4f, 0x3a, 0xde, 0xd9, 0x4e, 0xe5, 0x7d, 0xe1, 0x86, 0x57,
			0xf4, 0x43, 0x50, 0xf9, 0x75, 0x32, 0xa7, 0xc2, 0x5d, 0xcb, 0x99, 0xa8,
			0x68, 0x3a, 0xc3, 0xd6, 0x32, 0xa3, 0x76, 0x18, 0xeb, 0x9e, 0x2b, 0x18,
			0x54, 0x5c, 0x00, 0xb0, 0xc2, 0xcd, 0xc3, 0xc0,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(72, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_73(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xc5, 0x02, 0xa0, 0x09, 0xeb, 0x41, 0xc8, 0xf8, 0x22, 0x06, 0x66, 0xd4,
			0xe2, 0x33, 0x54, 0xb3, 0xd0, 0x86, 0xc8, 0xf6, 0xe0, 0x96, 0x28, 0xd2,
			0xa9, 0xbb, 0xe0, 0x72, 0x77, 0x0c, 0x99, 0xee,
		},
		{
			0xbb, 0xfc, 0xb4, 0x61, 0x3c, 0x45, 0x03, 0x26, 0x8b, 0xc9, 0x07, 0x45,
			0xdf, 0x3c, 0xc8, 0x90,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x6d, 0xa4, 0xc1, 0x2b, 0x0a, 0xa1, 0x28, 0xa5, 0x8f, 0xee, 0xb8, 0x7a,
			0xbb, 0xea, 0x04, 0xee, 0xcf, 0xb2, 0xb4, 0x3c, 0x62, 0x93, 0xb8, 0x86,
			0x3d, 0xe3, 0x16, 0xae, 0x6e, 0xd2, 0x22, 0x21,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xc2, 0xa8, 0x78, 0x4f, 0xbc, 0xe4, 0xfa, 0x6e, 0x15, 0x84, 0x4f, 0xdd,
			0x8a, 0x84, 0x41, 0xd6, 0xb5, 0x6e, 0xbe, 0x6c, 0x0d, 0xb5, 0x76, 0xd6,
			0x81, 0xe0, 0x2b, 0xdb, 0x45, 0x9e, 0x4b, 0xaa, 0xdd, 0xe1, 0x7b, 0xd0,
			0xb8, 0x7d, 0xfa, 0xfc, 0x9f, 0x92, 0x94, 0x1d, 0x7c, 0xa8, 0x99, 0xa1,
			0x3f, 0x7e, 0x19, 0xeb, 0xd4, 0x92, 0xb3, 0x47, 0xf7, 0x75, 0x0d, 0x7c,
			0x43, 0xe6, 0x70, 0x28, 0x65, 0x47, 0x48, 0x57, 0x2d, 0xc9, 0x1d, 0x20,
			0x02, 0xb1, 0x44, 0x27, 0x18, 0x7c, 0xf3, 0x3d, 0x42, 0x1a, 0x56, 0xef,
			0xef, 0xbd, 0x62, 0x1a, 0x84, 0xde, 0x1e, 0x56, 0x64, 0xdf, 0x1c, 0xb2,
			0xe7, 0xef, 0xc3, 0x6b, 0xce, 0xf0, 0x3a, 0xca, 0xe1, 0x3b, 0x05, 0xea,
			0x10, 0x59, 0x76, 0xf8, 0x40, 0xea, 0x75, 0x9d, 0x68, 0x00, 0x4b, 0x40,
			0x43, 0xe8, 0xca, 0x76, 0xd5, 0x2c, 0x73, 0x66,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(73, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_74(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x30, 0x4f, 0x12, 0x9a, 0xfb, 0x5b, 0xa1, 0xe0, 0x62, 0xbd, 0x6c, 0x0d,
			0x85, 0x9e, 0x71, 0x61, 0xe3, 0xb9, 0xea, 0x53, 0xef, 0x52, 0x6f, 0x00,
			0x32, 0x95, 0x33, 0xb6, 0xbe, 0x12, 0xfa, 0xaf,
		},
		{
			0x58, 0x56, 0x5e, 0x9d, 0x77, 0x6c, 0x89, 0xab, 0xf6, 0xcd, 0xa5, 0xe7,
			0x7c, 0xe7, 0xd2, 0x7b,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x19, 0x80, 0xa2, 0xd6, 0x5b, 0x26, 0xb7, 0xed, 0x3b, 0xfc, 0x2b, 0xa9,
			0x76, 0xf1, 0x4f, 0x14, 0x7d, 0xdb, 0x7f, 0xb1, 0x8c, 0x85, 0x43, 0xd0,
			0x46, 0x86, 0xe6, 0xad, 0xc0, 0x20, 0xcf, 0x72,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x8c, 0xda, 0x68, 0xca, 0xf5, 0x7c, 0x12, 0xdd, 0x0a, 0x2b, 0x62, 0x8f,
			0xe8, 0x75, 0xd5, 0xe0, 0xbb, 0xa5, 0x09, 0x81, 0x41, 0x5a, 0x4a, 0xe2,
			0xd5, 0x88, 0x06, 0xaf, 0x4c, 0x23, 0xf0, 0x59, 0x78, 0xe3, 0x9e, 0x56,
			0x62, 0x69, 0x07, 0x6c, 0xd4, 0x3e, 0x36, 0xea, 0xa1, 0x02, 0x5e, 0x35,
			0x13, 0x95, 0x51, 0x4a, 0xe5, 0x07, 0x11, 0xb1, 0x87, 0xed, 0xe5, 0xcf,
			0xd8, 0x45, 0x8a, 0x7b, 0x8a, 0x7d, 0x97, 0x88, 0xc2, 0x12, 0x0d, 0x1d,
			0x71, 0xf6, 0x1e, 0xa9, 0x5d, 0x84, 0xa7, 0xff, 0x1e, 0x04, 0xf5, 0x14,
			0xe8, 0x44, 0x8b, 0xa2, 0xea, 0x6b, 0x95, 0x6a, 0xea, 0xfd, 0xc3, 0xeb,
			0xb6, 0xc1, 0xcd, 0x0c, 0x7b, 0x7e, 0xc7, 0x6b, 0x95, 0x8e, 0xef, 0xb8,
			0x12, 0xec, 0xb3, 0xe8, 0x6b, 0x33, 0xa4, 0xd3, 0xa7, 0xb2, 0x97, 0x7f,
			0x66, 0xb2, 0x14, 0x08, 0x96, 0x62, 0xd1, 0x4a,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(74, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_75(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xc5, 0x58, 0x92, 0x6a, 0x4d, 0xbb, 0x9f, 0x41, 0xe7, 0x56, 0x89, 0xd9,
			0xc8, 0xae, 0xc8, 0x67, 0x97, 0xc2, 0x9f, 0x8c, 0x77, 0xe3, 0x56, 0xf0,
			0x33, 0x1b, 0xed, 0x78, 0x8b, 0x76, 0xe9, 0xb7,
		},
		{
			0xcf, 0x83, 0xe7, 0x71, 0x58, 0x26, 0xfd, 0x14, 0x89, 0xcc, 0x30, 0x54,
			0xd3, 0xcd, 0x57, 0x37,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xd1, 0x20, 0x7c, 0x7b, 0x91, 0x4f, 0x5e, 0x87, 0x3e, 0xa0, 0xcf, 0x6f,
			0x06, 0xdb, 0xd0, 0x2d, 0x59, 0x49, 0x4f, 0x6c, 0x8d, 0x4c, 0x9a, 0xef,
			0x5a, 0xd1, 0x80, 0xc9, 0xbb, 0xbe, 0xe2, 0x08,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xd3, 0xa7, 0xd7, 0x8c, 0xfe, 0x76, 0xff, 0x97, 0xbc, 0x54, 0xf7, 0xf6,
			0x94, 0xf7, 0x2f, 0x31, 0x24, 0x41, 0xd8, 0x60, 0x8d, 0x3c, 0x15, 0x13,
			0xfd, 0x28, 0xe9, 0x47, 0x42, 0xcb, 0x19, 0x68, 0x3a, 0x02, 0x70, 0x83,
			0x9b, 0x02, 0x6c, 0xee, 0x52, 0xe4, 0x92, 0x4d, 0xee, 0x99, 0x9e, 0xe8,
			0xcd, 0x7d, 0xd0, 0x1e, 0xa4, 0x45, 0xcf, 0x91, 0x00, 0x87, 0x98, 0xb8,
			0xed, 0x64, 0xc8, 0x45, 0x8d, 0x86, 0x78, 0xc3, 0x44, 0xd4, 0xe3, 0xfe,
			0x32, 0x01, 0xc1, 0xaf, 0x2f, 0xae, 0x3b, 0xea, 0x82, 0xea, 0x3a, 0xf0,
			0xcd, 0xac, 0x8e, 0x77, 0xb5, 0xe4, 0xd7, 0xc2, 0xb0, 0x84, 0xe4, 0x16,
			0xe5, 0x32, 0x23, 0x66, 0xad, 0x13, 0xe3, 0x6b, 0x45, 0x5f, 0x04, 0x7b,
			0xa4, 0x64, 0xcf, 0xf2, 0x87, 0xaf, 0x37, 0x22, 0x97, 0xc6, 0x8b, 0x51,
			0xaa, 0x47, 0x01, 0x90, 0x19, 0x43, 0xaa, 0xc8,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(75, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_76(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x63, 0x3d, 0x32, 0xe3, 0x00, 0x5f, 0x78, 0x11, 0x47, 0x23, 0xb3, 0xea,
			0x5a, 0xc1, 0x21, 0xba, 0x74, 0xaa, 0x00, 0xc5, 0x2d, 0x93, 0x96, 0x67,
			0xe3, 0x0c, 0x33, 0x51, 0xb3, 0x85, 0x49, 0xf7,
		},
		{
			0x37, 0xaf, 0xff, 0x50, 0x4a, 0x2d, 0x8a, 0xc1, 0x68, 0xc6, 0x8e, 0x24,
			0xd0, 0xfe, 0x66, 0xf6,
		},
		{
			0x9f, 0x16, 0x99, 0xc9, 0x9d, 0x60, 0xb0, 0x85, 0xbc, 0x61, 0xcb, 0x11,
			0x0e, 0xf8, 0xab, 0x59, 0x0d, 0x82, 0xa9, 0x70, 0x02, 0x1c, 0x3c, 0x6a,
			0x5d, 0x48, 0x02, 0x1c, 0x45, 0xde, 0x49, 0x56,
		},
		{
			0x3e, 0x33, 0x47, 0xc5, 0x47, 0xf1, 0x7f, 0x4d, 0x0b, 0x9f, 0x46, 0x40,
			0x5a, 0x54, 0xee, 0xdd, 0x7e, 0x98, 0x0d, 0x06, 0xa2, 0x15, 0xec, 0x15,
			0xe8, 0x93, 0x16, 0xab, 0x74, 0x3b, 0x75, 0x47,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x6e, 0x38, 0xe8, 0x29, 0x62, 0xd7, 0x07, 0xce, 0x9a, 0x6a, 0xc3, 0x83,
			0xa7, 0x38, 0xa7, 0x48, 0xf9, 0x75, 0xeb, 0x78, 0x56, 0x11, 0xfa, 0xd5,
			0xe3, 0xf5, 0xa4, 0xfe, 0x44, 0xd7, 0xb5, 0x9a, 0x98, 0x13, 0x7a, 0x2b,
			0xcd, 0xc3, 0x5f, 0x9e, 0xe9, 0xa1, 0xe2, 0x1b, 0xb1, 0x7d, 0xf1, 0x66,
			0x5c, 0xd1, 0x39, 0x76, 0x25, 0xa1, 0x77, 0x24, 0x7e, 0x2e, 0x32, 0x9a,
			0x66, 0x01, 0x40, 0x63, 0x61, 0x41, 0x56, 0x06, 0x10, 0xa3, 0x68, 0xbf,
			0xd4, 0x99, 0xc2, 0xe2, 0x5b, 0xe3, 0x18, 0xaa, 0x4d, 0xa9, 0xe7, 0xa3,
			0x52, 0xd1, 0x15, 0xdb, 0x82, 0x82, 0xed, 0x8d, 0x79, 0xec, 0xf9, 0xcd,
			0x82, 0x03, 0x60, 0xd3, 0xd2, 0xd1, 0xa5, 0x8a, 0x93, 0xe0, 0x40, 0xf5,
			0x55, 0x48, 0x87, 0xce, 0x6c, 0x98, 0x58, 0xbc, 0x2b, 0xb1, 0x02, 0x24,
			0x99, 0x80, 0xa8, 0x58, 0x49, 0x8a, 0xbc, 0xda,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(76, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_77(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xcb, 0x20, 0x15, 0x2e, 0x99, 0xd8, 0xca, 0x5e, 0x80, 0x95, 0xa1, 0x3e,
			0x72, 0xe0, 0xc1, 0xbb, 0x9e, 0x6c, 0xbe, 0xad, 0xbd, 0x0b, 0xf4, 0xe7,
			0x0f, 0xc6, 0xfa, 0x3c, 0x81, 0xec, 0x5d, 0x0f,
		},
		{
			0x0a, 0x55, 0xd4, 0x33, 0xd3, 0x74, 0x4a, 0x2f, 0x38, 0x19, 0x39, 0xbe,
			0x85, 0xc9, 0x7a, 0x98,
		},
		{
			0xb5, 0xbd, 0x74, 0x29, 0x59, 0x1e, 0x5e, 0x29, 0x14, 0x53, 0x2a, 0x7d,
			0xea, 0x75, 0xff, 0x76, 0xca, 0xa3, 0x9f, 0x15, 0x27, 0x9c, 0x73, 0x09,
			0x2c, 0x2c, 0xac, 0x3b, 0x0d, 0x28, 0xc1, 0x3b,
		},
		{
			0xd0, 0x8c, 0xa9, 0x6d, 0x74, 0x5f, 0x7e, 0x3a, 0x9f, 0x2a, 0xcb, 0xc1,
			0xe8, 0x00, 0xa9, 0x81, 0x95, 0x52, 0x2a, 0x4f, 0x7c, 0x60, 0x7a, 0xad,
			0x62, 0xf3, 0x8b, 0x1f, 0x63, 0x35, 0x4b, 0xd1,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x03, 0xe8, 0xcd, 0x60, 0x76, 0x82, 0x62, 0xcf, 0x4c, 0xc2, 0xf2, 0x95,
			0x50, 0x6c, 0x24, 0xf0, 0x6f, 0xb3, 0xfd, 0x3a, 0x88, 0x9c, 0x71, 0x52,
			0x14, 0x51, 0xdf, 0x6f, 0x2e, 0xfa, 0x07, 0x47, 0xff, 0xb8, 0x75, 0x78,
			0xb7, 0x1b, 0xcf, 0xc3, 0x7d, 0x3a, 0x43, 0xf8, 0x5a, 0xb4, 0xe5, 0xfc,
			0x02, 0xd2, 0x1a, 0x90, 0xe8, 0x5a, 0xf1, 0x8d, 0x6d, 0x75, 0xe9, 0xfa,
			0x95, 0xf7, 0xf9, 0x29, 0x42, 0xa2, 0x93, 0xda, 0x5d, 0x31, 0xa6, 0xb7,
			0x77, 0xf1, 0x10, 0xff, 0x86, 0x27, 0x91, 0x42, 0x8b, 0xa0, 0x1b, 0x3c,
			0x1b, 0x2f, 0xb4, 0xb2, 0x36, 0x63, 0x18, 0xd2, 0x44, 0xa9, 0x1e, 0xc1,
			0x30, 0xf4, 0xc0, 0xbd, 0xb8, 0xc4, 0x0c, 0xb7, 0xa6, 0x23, 0xe5, 0xed,
			0x02, 0x0b, 0xb3, 0x05, 0xa3, 0x8b, 0x13, 0x47, 0x12, 0x15, 0x1b, 0x48,
			0x91, 0xda, 0x7f, 0xfb, 0x06, 0x91, 0x2f, 0x4b,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(77, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_78(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xbd, 0xcb, 0xcf, 0x6d, 0x34, 0x75, 0xac, 0x89, 0x03, 0xdd, 0x32, 0xcb,
			0x12, 0x54, 0x36, 0xa6, 0xf4, 0xce, 0x74, 0x64, 0x49, 0x60, 0x43, 0xa1,
			0x80, 0xb2, 0x60, 0xd2, 0xa0, 0x0e, 0xed, 0xa1,
		},
		{
			0x4e, 0x63, 0xef, 0x1d, 0xd0, 0x4d, 0x3e, 0x6c, 0xa8, 0xa8, 0x09, 0x9d,
			0x7e, 0x3d, 0xa0, 0x95,
		},
		{
			0xe0, 0xb7, 0x81, 0x6c, 0xae, 0x86, 0x74, 0xb8, 0x73, 0x08, 0x54, 0x1f,
			0xdd, 0x65, 0xc0, 0xf8, 0x11, 0xb7, 0x80, 0x8f, 0x3f, 0x2f, 0xc1, 0x5f,
			0x57, 0x7d, 0x64, 0x74, 0x8e, 0xe3, 0x6d, 0x4b,
		},
		{
			0x04, 0x1d, 0x73, 0x1b, 0x34, 0x97, 0xb4, 0x19, 0x09, 0xde, 0xec, 0xaf,
			0xff, 0x8b, 0x71, 0x16, 0x08, 0x5d, 0x8e, 0xf2, 0x8c, 0x32, 0xf4, 0x23,
			0x3f, 0x6c, 0x66, 0x0e, 0x77, 0xa9, 0x9b, 0xb6,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xd1, 0x0f, 0x8e, 0x09, 0xba, 0x93, 0x52, 0x69, 0xd8, 0xc3, 0x87, 0x62,
			0xad, 0xca, 0xf2, 0x83, 0x08, 0xd2, 0x5d, 0xce, 0xcc, 0x19, 0xff, 0xdb,
			0xf7, 0x93, 0xa9, 0xc6, 0x75, 0x85, 0xcb, 0x79, 0x1d, 0xf6, 0x5a, 0x24,
			0xda, 0x8b, 0xa2, 0xb7, 0x79, 0x41, 0xf7, 0x01, 0xde, 0x86, 0x3b, 0xe3,
			0xd6, 0xf3, 0xc9, 0xd1, 0x3f, 0x44, 0x45, 0x97, 0x19, 0x38, 0x7b, 0x51,
			0x36, 0x53, 0x46, 0x63, 0x86, 0x2b, 0x70, 0xc7, 0xdc, 0xf9, 0x91, 0x31,
			0xde, 0x59, 0x25, 0x30, 0xb3, 0xce, 0xdc, 0x61, 0x68, 0x98, 0xd3, 0x65,
			0x6e, 0x20, 0x64, 0xd8, 0x79, 0x14, 0x9d, 0xb2, 0x54, 0xd2, 0x6e, 0x72,
			0xc5, 0x46, 0x6a, 0xd5, 0x5b, 0x1a, 0xc6, 0xfe, 0x52, 0x43, 0x86, 0xa0,
			0x99, 0xaa, 0x64, 0x2a, 0xd0, 0x96, 0xa0, 0x93, 0x3f, 0x92, 0xdb, 0x03,
			0xe9, 0x41, 0x70, 0xe0, 0x5d, 0xa5, 0x00, 0xe5,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(78, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_79(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x0c, 0xf0, 0x73, 0x1c, 0x47, 0xca, 0x73, 0xe8, 0xc4, 0x29, 0x42, 0xcc,
			0x61, 0xf9, 0x98, 0xf5, 0x34, 0x2e, 0x31, 0x8c, 0xb8, 0xed, 0x3b, 0x18,
			0x69, 0x02, 0x0e, 0x3d, 0xab, 0x1e, 0x12, 0x67,
		},
		{
			0x60, 0x4c, 0x64, 0x15, 0x4f, 0x91, 0x4e, 0xd0, 0xcf, 0x63, 0xc8, 0x61,
			0x65, 0x3b, 0x78, 0xd4,
		},
		{
			0x30, 0x50, 0xb2, 0xdf, 0x2f, 0x59, 0x76, 0x3e, 0x44, 0x3c, 0xaa, 0x92,
			0x5a, 0x54, 0xde, 0x1e, 0x5c, 0xed, 0x87, 0x59, 0xb5, 0x54, 0xb5, 0xf0,
			0xf6, 0x8c, 0xb0, 0x1b, 0x41, 0x50, 0xf9, 0x99,
		},
		{
			0xed, 0xa0, 0x82, 0xed, 0xfa, 0xf4, 0xf9, 0x76, 0x6d, 0xe2, 0x9a, 0xe2,
			0xcb, 0x93, 0x0f, 0xb1, 0xbc, 0x26, 0xbd, 0xe6, 0xd1, 0x8a, 0xb1, 0x6f,
			0x66, 0xfb, 0xf4, 0xb0, 0x6d, 0x3e, 0x74, 0x80,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x39, 0xb3, 0xf4, 0x7c, 0x02, 0x4b, 0xc8, 0x4a, 0xad, 0x80, 0x91, 0x13,
			0x6d, 0x34, 0x1e, 0xba, 0x3a, 0x89, 0xd0, 0x65, 0x77, 0xaa, 0x4f, 0xd3,
			0xd2, 0x4f, 0x8b, 0x7a, 0x5b, 0xe4, 0x28, 0x0f, 0xdf, 0x62, 0x35, 0x34,
			0x90, 0xd5, 0xf6, 0x6c, 0x1a, 0x7d, 0x57, 0xcf, 0x19, 0x8a, 0x63, 0x37,
			0x56, 0xc0, 0x90, 0x71, 0x86, 0xa7, 0x1a, 0x4f, 0xea, 0x41, 0xc3, 0xfa,
			0x42, 0xef, 0xee, 0xfb, 0x0a, 0xc2, 0x85, 0xea, 0x85, 0x87, 0xf1, 0xa8,
			0xd8, 0xfb, 0xb7, 0x83, 0xae, 0x62, 0x7c, 0x1e, 0x85, 0x8f, 0xfb, 0xe0,
			0x51, 0xd6, 0xb7, 0xa2, 0xa5, 0x18, 0xa2, 0x4e, 0x71, 0xd5, 0x54, 0xe8,
			0xa0, 0xda, 0x3d, 0x54, 0xbc, 0x69, 0xe4, 0xca, 0x94, 0x0b, 0x7c, 0xa3,
			0x20, 0xc4, 0xbc, 0x44, 0xdf, 0x1e, 0x47, 0x6e, 0x05, 0xe4, 0x89, 0xa2,
			0xc6, 0x68, 0x70, 0x2b, 0x0b, 0xb8, 0x91, 0x3d,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(79, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_80(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x97, 0x15, 0x5a, 0x62, 0x4e, 0xe4, 0x2b, 0x97, 0x4c, 0x4b, 0xc9, 0xe0,
			0xb9, 0xa7, 0x37, 0xeb, 0x79, 0x0c, 0xde, 0x2f, 0x0a, 0x93, 0xf8, 0xef,
			0x2c, 0x6f, 0xe7, 0x20, 0x15, 0xc5, 0x27, 0x6b,
		},
		{
			0x44, 0x4d, 0xf2, 0xbe, 0xae, 0x4a, 0xe6, 0xe9, 0xa6, 0x40, 0x41, 0x6f,
			0xc3, 0x94, 0x8d, 0xde,
		},
		{
			0x62, 0x5c, 0x7e, 0xf0, 0xfa, 0x30, 0xe0, 0xcb, 0x5c, 0x28, 0xff, 0x3a,
			0x23, 0x96, 0x63, 0x93, 0x24, 0x29, 0x2b, 0x20, 0x06, 0x9e, 0xcb, 0x1c,
			0x01, 0x63, 0xd9, 0x66, 0xf2, 0x52, 0x0e, 0x61,
		},
		{
			0x40, 0x1b, 0x91, 0xbb, 0xd7, 0x26, 0xeb, 0x24, 0xeb, 0x19, 0xb4, 0xf3,
			0x62, 0x88, 0x19, 0x2a, 0xe0, 0x71, 0x35, 0x91, 0x7d, 0xec, 0x75, 0xb7,
			0xf7, 0x75, 0x81, 0x77, 0x76, 0x72, 0x98, 0xb7,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x8b, 0x1c, 0x56, 0xcd, 0x41, 0x5d, 0xdf, 0x12, 0x3b, 0xc5, 0xbb, 0x32,
			0x7f, 0x0e, 0xc4, 0xcf, 0x53, 0xa6, 0x49, 0x83, 0x1d, 0x7c, 0x09, 0x20,
			0xe7, 0x42, 0x03, 0x3c, 0xe1, 0x83, 0xa4, 0x42, 0x3c, 0x4b, 0x16, 0x2e,
			0xac, 0xa3, 0x15, 0x7c, 0xb1, 0x94, 0xdc, 0xe0, 0x6a, 0x54, 0x3b, 0xf1,
			0x55, 0x16, 0x4c, 0x70, 0x41, 0x75, 0x0f, 0xb3, 0xaf, 0xc7, 0x78, 0x92,
			0xbd, 0x7b, 0x45, 0x2a, 0x20, 0xb6, 0x48, 0x90, 0x18, 0xe5, 0x36, 0x9f,
			0xce, 0x4e, 0x7e, 0xb7, 0x93, 0x54, 0xea, 0x8a, 0x03, 0x38, 0x80, 0xa6,
			0xdb, 0x0a, 0x79, 0xee, 0x0c, 0x1c, 0xa1, 0x5d, 0x80, 0x75, 0x12, 0x31,
			0x2c, 0x3f, 0x2d, 0x82, 0xbd, 0xf0, 0x06, 0x83, 0x64, 0xcc, 0x3e, 0xae,
			0x76, 0x3b, 0xe2, 0x4e, 0xdb, 0x39, 0x35, 0x6d, 0xc1, 0xa6, 0xd1, 0xc5,
			0xeb, 0x09, 0x3c, 0xde, 0x4b, 0xf9, 0xa5, 0x89,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(80, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_81(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x88, 0xe9, 0xbc, 0x0e, 0x56, 0xbb, 0xbf, 0xe8, 0xc0, 0x7f, 0x31, 0x1b,
			0x95, 0x5d, 0x2b, 0x87, 0x7f, 0xe7, 0x04, 0x7f, 0x3f, 0x85, 0xac, 0x83,
			0xf9, 0xda, 0x68, 0xac, 0x2a, 0xbe, 0x36, 0xe2,
		},
		{
			0x2c, 0xf7, 0x59, 0x3a, 0x98, 0x12, 0x25, 0x25, 0xc2, 0x81, 0xd3, 0xce,
			0xc1, 0xe2, 0xb9, 0x4c,
		},
		{
			0x0d, 0x28, 0x96, 0x1b, 0x6a, 0xba, 0xd3, 0x01, 0x91, 0xb2, 0x14, 0x70,
			0xba, 0x1e, 0xe0, 0x0d, 0xa1, 0x70, 0x37, 0x90, 0xdd, 0x8a, 0x15, 0x53,
			0x73, 0x97, 0xc7, 0x3e, 0x80, 0xeb, 0x7f, 0xbc,
		},
		{
			0xe3, 0x28, 0x52, 0xb8, 0x1c, 0x80, 0xf3, 0x7d, 0x79, 0xcf, 0xfd, 0x3c,
			0xae, 0x78, 0x59, 0x10, 0xe7, 0x92, 0x32, 0x81, 0x3d, 0x17, 0x4a, 0x2d,
			0x14, 0xf0, 0xbf, 0x27, 0x5f, 0x31, 0x29, 0x4b,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x4a, 0x6b, 0x13, 0xbd, 0xe6, 0xcb, 0xf8, 0x66, 0xd1, 0xb6, 0x09, 0x42,
			0x48, 0x8c, 0xef, 0x40, 0x86, 0x36, 0x7f, 0xd4, 0xe2, 0x44, 0x5f, 0x73,
			0xed, 0xef, 0x03, 0x28, 0xf6, 0x94, 0x1d, 0x40, 0x84, 0xa4, 0x46, 0x98,
			0x62, 0x81, 0x2f, 0xf6, 0xe4, 0xd9, 0xe4, 0x19, 0x2e, 0x29, 0xd5, 0x9d,
			0x56, 0x2e, 0x5d, 0xaf, 0xa0, 0xae, 0x10, 0x94, 0x4f, 0x8b, 0x62, 0xad,
			0x15, 0xa8, 0xf4, 0x83, 0xd6, 0xd1, 0x0e, 0xed, 0x6c, 0x4f, 0xe4, 0x26,
			0x45, 0x58, 0x7b, 0x61, 0x25, 0xd8, 0x63, 0x65, 0x37, 0x10, 0xbd, 0x6a,
			0x21, 0x27, 0x7a, 0xe4, 0x49, 0xb7, 0xd2, 0xa9, 0x79, 0x78, 0x29, 0x74,
			0xa7, 0x31, 0xda, 0x60, 0x1b, 0x29, 0xbc, 0x8e, 0x31, 0xfa, 0x12, 0x1b,
			0x8b, 0x7e, 0x95, 0xd1, 0xeb, 0x30, 0xa7, 0x5b, 0x3d, 0x4f, 0xb2, 0x13,
			0x1e, 0x17, 0xfd, 0xea, 0xa0, 0x39, 0x5e, 0xa6,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(81, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_82(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xdf, 0x81, 0x88, 0xfe, 0x2b, 0x7b, 0xc5, 0xed, 0x26, 0xf7, 0x47, 0xc9,
			0x3e, 0xd4, 0x80, 0x24, 0xe3, 0xed, 0x62, 0x75, 0x30, 0x10, 0x7d, 0x94,
			0x90, 0x6b, 0xb2, 0xde, 0xbd, 0x4f, 0xb9, 0x2b,
		},
		{
			0x0d, 0xd7, 0x50, 0xc7, 0x4f, 0x83, 0x63, 0xc1, 0x4c, 0x17, 0x2b, 0x40,
			0x98, 0xf5, 0x20, 0x58,
		},
		{
			0x8a, 0x9b, 0xa6, 0xbd, 0xb3, 0x05, 0x9e, 0xc8, 0x8b, 0xb2, 0x9e, 0x1c,
			0x51, 0x60, 0x51, 0xf8, 0xf9, 0x58, 0xb3, 0xc7, 0x75, 0x1f, 0xb3, 0x7a,
			0x61, 0x6e, 0x9d, 0x4a, 0x6c, 0xf1, 0x7a, 0x79,
		},
		{
			0x1b, 0x21, 0x45, 0xc0, 0xeb, 0x75, 0x77, 0x50, 0x30, 0xd1, 0x68, 0x1c,
			0x84, 0x8b, 0x92, 0x49, 0xda, 0xcb, 0x81, 0x37, 0xa3, 0xa0, 0xfe, 0x26,
			0x54, 0x4d, 0x17, 0xa2, 0x37, 0x4a, 0x51, 0x82,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb9, 0x25, 0xc2, 0x7c, 0x1b, 0x2a, 0x82, 0xe6, 0x96, 0x9a, 0x90, 0xfe,
			0x4e, 0x4b, 0x6f, 0x27, 0xc3, 0x69, 0xe8, 0x2b, 0x4b, 0x03, 0x47, 0x9a,
			0x9a, 0x15, 0x2a, 0xca, 0x12, 0x88, 0xb2, 0x62, 0x42, 0x0f, 0xaa, 0x6c,
			0x62, 0xda, 0x1f, 0x22, 0x5d, 0xc7, 0xf3, 0x12, 0x5d, 0x42, 0x88, 0x58,
			0x50, 0x78, 0x8b, 0x26, 0x44, 0x10, 0xc8, 0x14, 0x9e, 0xab, 0xf6, 0xba,
			0x16, 0x4c, 0x17, 0x32, 0xa2, 0xb6, 0x0b, 0x77, 0xa5, 0x25, 0xec, 0x74,
			0x5b, 0xac, 0x6e, 0xda, 0x51, 0xec, 0x7d, 0x88, 0xd1, 0x1b, 0x40, 0xc2,
			0xe1, 0x28, 0x54, 0xd3, 0x92, 0xe7, 0xc6, 0xaf, 0x2f, 0xff, 0x7a, 0xaf,
			0x15, 0x44, 0xab, 0x30, 0x6a, 0xc3, 0x1b, 0x6d, 0x97, 0x1b, 0x70, 0xa8,
			0x2a, 0x67, 0x2c, 0x81, 0xe9, 0x28, 0x20, 0x4d, 0x04, 0x96, 0x0e, 0xbf,
			0xf5, 0xbf, 0x5e, 0x38, 0xaa, 0x89, 0x7f, 0x23,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(82, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_83(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x6a, 0xeb, 0x84, 0xcf, 0x5f, 0x72, 0x98, 0x2b, 0xf7, 0xb1, 0xef, 0x4c,
			0xa1, 0x6f, 0x8a, 0x27, 0x96, 0x5c, 0x35, 0xb2, 0xa5, 0x55, 0x61, 0x94,
			0x6c, 0x6d, 0x2c, 0xf6, 0xa6, 0xf9, 0xe4, 0x1e,
		},
		{
			0x82, 0x76, 0x07, 0xb8, 0x06, 0xdf, 0x2c, 0x42, 0x37, 0x41, 0x20, 0x9b,
			0xed, 0x59, 0xb4, 0xeb,
		},
		{
			0xc2, 0x53, 0x73, 0xb6, 0x1d, 0x8d, 0x58, 0x2f, 0xc2, 0x68, 0xcb, 0xd6,
			0x49, 0x3a, 0x91, 0x61, 0xee, 0xbe, 0xde, 0x96, 0xc4, 0x3b, 0x02, 0x95,
			0x28, 0x1c, 0xc8, 0xac, 0x47, 0x38, 0x24, 0xe7,
		},
		{
			0x57, 0x54, 0x81, 0x26, 0x38, 0x1e, 0xd8, 0x62, 0x67, 0x05, 0xd3, 0x53,
			0x71, 0x22, 0x80, 0x0c, 0x4c, 0xd9, 0xef, 0x81, 0x32, 0x04, 0x4d, 0xfd,
			0x0f, 0x8d, 0x27, 0xd5, 0xeb, 0xce, 0x55, 0xd1,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb2, 0x0c, 0x3a, 0xe8, 0x13, 0x3b, 0x87, 0x20, 0x49, 0x8c, 0x17, 0xfc,
			0xa0, 0xb9, 0xa2, 0xf7, 0x04, 0x57, 0x26, 0xfd, 0xc1, 0x5e, 0x71, 0xa6,
			0xd3, 0x9e, 0x18, 0x95, 0x49, 0xb0, 0x2d, 0x39, 0x92, 0xa7, 0x4b, 0xfa,
			0x9d, 0x68, 0xec, 0xca, 0x5f, 0xf1, 0x71, 0x89, 0x0e, 0xe5, 0xba, 0xfc,
			0x4e, 0x59, 0xbf, 0xc6, 0x62, 0xfd, 0xb2, 0x45, 0x8e, 0xf2, 0x44, 0xe5,
			0x4e, 0x03, 0xfa, 0x43, 0xa0, 0x61, 0x67, 0x97, 0xbd, 0xca, 0x83, 0xb6,
			0xef, 0x8f, 0x36, 0xda, 0xd2, 0x80, 0xc9, 0xdf, 0xb1, 0xca, 0x56, 0xd3,
			0xaa, 0x5e, 0x2d, 0xa3, 0x2a, 0xaf, 0xe2, 0xf6, 0x89, 0xf5, 0x1a, 0x43,
			0xc7, 0x54, 0x7d, 0x24, 0xed, 0x94, 0xd0, 0x2a, 0x56, 0x7f, 0x78, 0x55,
			0xaa, 0xaf, 0xc8, 0xb6, 0x94, 0xea, 0x5f, 0x0b, 0xb5, 0x60, 0x2f, 0x72,
			0xda, 0xbf, 0x86, 0x56, 0xe7, 0x43, 0x58, 0xb2,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(83, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_84(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x49, 0xce, 0xab, 0x47, 0xef, 0x95, 0x58, 0x5a, 0xaa, 0xeb, 0x98, 0x06,
			0x4c, 0x61, 0xe8, 0xa2, 0x89, 0xee, 0xe5, 0x41, 0x2f, 0x4a, 0xf1, 0xfb,
			0x6e, 0x48, 0xa1, 0x23, 0x58, 0xf9, 0xcf, 0x67,
		},
		{
			0xd4, 0x8e, 0x4a, 0x8a, 0x61, 0x2c, 0xac, 0x57, 0xfe, 0x7b, 0xc1, 0x78,
			0x8c, 0x64, 0x83, 0x6b,
		},
		{
			0x62, 0x1e, 0x31, 0x3b, 0x13, 0x34, 0x80, 0xd5, 0xc4, 0x71, 0xba, 0xe8,
			0xda, 0x09, 0x7a, 0x03, 0x33, 0xb2, 0xb2, 0xea, 0x23, 0xf8, 0x3c, 0x1b,
			0x10, 0x28, 0x50, 0xd4, 0x11, 0xd1, 0x51, 0x5c,
		},
		{
			0x84, 0x40, 0x7b, 0x20, 0xd0, 0xad, 0xc2, 0x2c, 0x07, 0xbf, 0x24, 0x94,
			0x08, 0x0a, 0xd2, 0x86, 0x93, 0x8b, 0x21, 0x28, 0xf8, 0x24, 0xda, 0xe3,
			0xe4, 0x4d, 0x3c, 0x71, 0x81, 0x87, 0x02, 0xc0,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xdf, 0xb9, 0xcc, 0xd2, 0x74, 0x93, 0x63, 0xfd, 0x99, 0x3d, 0xf5, 0xdb,
			0x53, 0x82, 0x6d, 0xc3, 0x6a, 0xdf, 0x4e, 0x04, 0x08, 0x78, 0xf6, 0x85,
			0x15, 0x14, 0x46, 0x6d, 0xd7, 0x21, 0x4b, 0xa7, 0x0d, 0x35, 0xf4, 0x74,
			0xb2, 0x2d, 0x5d, 0x18, 0x0d, 0x71, 0xf8, 0xca, 0x00, 0x49, 0xfc, 0x90,
			0x15, 0x6d, 0x6d, 0x79, 0x67, 0x08, 0x28, 0x0c, 0xe1, 0xd2, 0x25, 0x18,
			0x4b, 0x06, 0xdb, 0xe3, 0xd7, 0x98, 0xb7, 0xb9, 0x02, 0x85, 0x62, 0xd2,
			0x22, 0xf1, 0x1e, 0xe8, 0xfe, 0xd9, 0xa9, 0x55, 0x87, 0xa6, 0x53, 0x6e,
			0x90, 0x7a, 0xde, 0xe3, 0x10, 0x2b, 0x2f, 0x3d, 0x92, 0xdb, 0x24, 0x67,
			0x37, 0xa9, 0xbc, 0x71, 0xaa, 0xaa, 0x4f, 0x40, 0xcd, 0xa3, 0x3d, 0x33,
			0xa3, 0x37, 0x9d, 0x2e, 0x8b, 0xfe, 0xd1, 0x76, 0x0e, 0xc1, 0x28, 0x01,
			0xe7, 0x2a, 0x38, 0x4b, 0x74, 0x8f, 0xfa, 0xf5,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(84, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_85(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xf3, 0x22, 0xb2, 0x58, 0xc6, 0x29, 0x51, 0xd3, 0x7b, 0xa6, 0x80, 0xab,
			0xc4, 0x9a, 0x61, 0x98, 0x3f, 0x3f, 0xd8, 0x99, 0x73, 0x94, 0x0c, 0x1d,
			0x4f, 0x07, 0x65, 0xc9, 0x5d, 0x0d, 0x70, 0x78,
		},
		{
			0x0b, 0x79, 0xd5, 0xbe, 0xfe, 0x42, 0x85, 0xa7, 0x3e, 0xa4, 0xb4, 0x8e,
			0x2b, 0xa1, 0x85, 0xc1,
		},
		{
			0x05, 0x32, 0xbc, 0x7e, 0x99, 0x70, 0x90, 0x06, 0xaa, 0x19, 0x1b, 0x37,
			0x9a, 0xce, 0xfa, 0x16, 0x45, 0xc6, 0x7d, 0x2f, 0x5b, 0x61, 0xf1, 0x3f,
			0x36, 0xf8, 0x68, 0xbc, 0xe4, 0x44, 0xa5, 0x29,
		},
		{
			0x2e, 0x4e, 0x16, 0x74, 0x8e, 0xbf, 0x36, 0x59, 0x02, 0xd9, 0x10, 0xa0,
			0x1d, 0x9c, 0xa2, 0x4a, 0xeb, 0x8e, 0x3e, 0x54, 0x58, 0xed, 0xfd, 0x88,
			0xf6, 0xb9, 0x09, 0xc1, 0xff, 0xc7, 0x46, 0x8c,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x85, 0x5e, 0xe8, 0xe0, 0x06, 0x0c, 0xb8, 0xb4, 0xfc, 0x56, 0x92, 0x07,
			0x3f, 0x62, 0xcc, 0x3e, 0x19, 0xa1, 0x9b, 0x89, 0xdf, 0x2d, 0xf9, 0x7c,
			0x2a, 0x21, 0x87, 0x9d, 0x74, 0x57, 0x1d, 0x33, 0x62, 0xec, 0xdd, 0x1a,
			0xd9, 0xe4, 0x64, 0x58, 0xe3, 0x5f, 0x5f, 0x35, 0x80, 0xe7, 0x66, 0x06,
			0x7e, 0x3d, 0x8f, 0x72, 0x58, 0xd8, 0xee, 0xcb, 0x17, 0xa0, 0x75, 0xf7,
			0xdc, 0x75, 0x23, 0xa4, 0x78, 0xb5, 0xd1, 0x77, 0xf6, 0x73, 0x65, 0x9e,
			0x38, 0x87, 0x67, 0x0e, 0xdf, 0x6e, 0x8f, 0x0f, 0xf7, 0x5e, 0x26, 0xd4,
			0x1a, 0xdf, 0xee, 0x6a, 0xe2, 0x8f, 0xbe, 0x5c, 0x2e, 0x24, 0x7e, 0x10,
			0x36, 0x24, 0x1e, 0x1f, 0x5b, 0xf3, 0x16, 0xfb, 0xd9, 0xad, 0xb3, 0xc1,
			0x59, 0x30, 0xe8, 0x42, 0x5a, 0xe7, 0x50, 0x24, 0xc2, 0x77, 0x02, 0x99,
			0x72, 0xf0, 0xd6, 0xc6, 0x45, 0xb4, 0xa1, 0x17,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(85, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_86(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x33, 0xa6, 0x8b, 0xad, 0x0f, 0xa3, 0x44, 0xfd, 0x6a, 0x46, 0xf7, 0x5f,
			0x04, 0x28, 0xcb, 0xdf, 0x27, 0x11, 0xd3, 0xf0, 0xc7, 0xef, 0x0f, 0x6a,
			0x39, 0x50, 0x73, 0xed, 0x70, 0x89, 0x7b, 0xaa,
		},
		{
			0x18, 0x92, 0x79, 0x9f, 0xeb, 0x41, 0x0b, 0x9d, 0xd6, 0xe6, 0x9b, 0xf4,
			0x40, 0x3e, 0xfc, 0x20,
		},
		{
			0x67, 0x7e, 0xcc, 0xd0, 0xc6, 0xe0, 0x1a, 0x26, 0x3f, 0xc7, 0xd2, 0xe2,
			0xc2, 0xd0, 0xc6, 0x29, 0x4a, 0x30, 0x7b, 0xf3, 0xee, 0x1b, 0xa3, 0x53,
			0x72, 0xc2, 0x16, 0x6d, 0x04, 0x04, 0x29, 0x72,
		},
		{
			0xa6, 0x44, 0x7d, 0xdb, 0xfa, 0xf7, 0xea, 0x07, 0xe1, 0x0a, 0x95, 0xdb,
			0x1f, 0xae, 0x3e, 0xa8, 0x6d, 0x10, 0x3c, 0xfc, 0xdd, 0xa5, 0x50, 0xf1,
			0x10, 0x1b, 0x02, 0xdd, 0xc5, 0xac, 0x46, 0x0c,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x22, 0x91, 0x43, 0x3d, 0xf6, 0xfa, 0x57, 0x25, 0x89, 0xc3, 0xa6, 0x51,
			0x3f, 0x38, 0x1a, 0xf4, 0x21, 0x27, 0x87, 0x49, 0x9b, 0xad, 0x08, 0x64,
			0xcb, 0x36, 0xd0, 0xeb, 0xee, 0x3b, 0x56, 0xe3, 0x40, 0xab, 0x7b, 0xdb,
			0x9f, 0x3b, 0x0e, 0x0d, 0x61, 0xdb, 0xa6, 0xb0, 0x14, 0x35, 0x77, 0x21,
			0x08, 0xbd, 0x89, 0x21, 0xcf, 0xc0, 0x0b, 0xeb, 0xd2, 0x07, 0x66, 0x94,
			0x41, 0x84, 0x04, 0x6a, 0x84, 0xf5, 0x6c, 0x43, 0x19, 0x69, 0x58, 0x80,
			0xd2, 0x43, 0x45, 0xac, 0xc6, 0x4f, 0x85, 0x79, 0xd3, 0xca, 0x30, 0xc1,
			0xf5, 0x4e, 0x41, 0x1c, 0xff, 0x51, 0x5e, 0xd7, 0x2a, 0x8f, 0xca, 0xa5,
			0xd2, 0xa1, 0x5f, 0xc2, 0x52, 0x6d, 0x5d, 0x11, 0x6f, 0xa1, 0x78, 0x3f,
			0xff, 0x6d, 0x70, 0xd1, 0x56, 0xb6, 0xe3, 0x3b, 0xf3, 0x1d, 0x21, 0x0d,
			0x3e, 0xe7, 0x9c, 0x44, 0xb7, 0x3e, 0xaa, 0x7e,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(86, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_87(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x31, 0x36, 0xf4, 0x42, 0xdd, 0x3c, 0xdc, 0x24, 0x95, 0x38, 0xbd, 0x5f,
			0x0e, 0xb3, 0x55, 0xc7, 0x67, 0x0a, 0xdd, 0x64, 0x27, 0xd1, 0xe1, 0x80,
			0x31, 0x7b, 0x7d, 0x64, 0x96, 0x78, 0x0b, 0x83,
		},
		{
			0x83, 0x65, 0x57, 0x29, 0x4d, 0x84, 0x4f, 0x21, 0x80, 0xd3, 0x7a, 0xc0,
			0x57, 0xcb, 0xb8, 0x69,
		},
		{
			0x48, 0x34, 0xc6, 0x8e, 0x72, 0xe2, 0x5a, 0xce, 0xc7, 0xce, 0x57, 0x7e,
			0xfd, 0x4d, 0xc9, 0x74, 0x6b, 0x8a, 0xbb, 0x89, 0xa4, 0x46, 0xad, 0x39,
			0xdb, 0x70, 0x42, 0xce, 0xb6, 0xda, 0x9c, 0xde,
		},
		{
			0xc6, 0x6a, 0x48, 0x65, 0xfb, 0x9b, 0x0f, 0x86, 0xf0, 0x69, 0x6d, 0x4a,
			0x2e, 0x1e, 0xa5, 0xf7, 0xa3, 0x93, 0x15, 0x7d, 0x09, 0x69, 0x29, 0xdf,
			0xf4, 0x5f, 0x64, 0xe7, 0x45, 0xd0, 0x6a, 0x3c,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xed, 0xb8, 0x11, 0x68, 0x13, 0x0e, 0xc4, 0x33, 0x01, 0x6e, 0x07, 0x42,
			0x93, 0x74, 0x07, 0x31, 0x37, 0xe0, 0xec, 0x1d, 0x14, 0x64, 0xe6, 0x3a,
			0xd2, 0xdd, 0x24, 0xbf, 0xe5, 0xcb, 0x76, 0x3a, 0xa8, 0x79, 0x3d, 0x8f,
			0xb3, 0x2e, 0xb2, 0xc7, 0x0e, 0xb6, 0x70, 0x53, 0x7f, 0xce, 0x9f, 0x7e,
			0x53, 0xc4, 0x2d, 0x4b, 0xd8, 0x5d, 0xbe, 0x0b, 0x2f, 0x50, 0xf4, 0x40,
			0xcb, 0x13, 0x2c, 0xe6, 0xcc, 0x46, 0xfa, 0xe1, 0x08, 0x69, 0xda, 0x2a,
			0xe6, 0xab, 0x8c, 0x10, 0xc3, 0xa1, 0xff, 0x2e, 0x45, 0x29, 0xe4, 0x6c,
			0x8f, 0x84, 0x3e, 0x0c, 0xf9, 0x31, 0xb7, 0x70, 0xec, 0xf0, 0xd0, 0x63,
			0x9f, 0x1a, 0x5d, 0x93, 0x94, 0x29, 0x96, 0x01, 0x71, 0x4c, 0xdd, 0xab,
			0xea, 0x3d, 0x00, 0x4d, 0x24, 0xda, 0x00, 0x56, 0x66, 0x51, 0x48, 0x48,
			0xd2, 0xfe, 0x0a, 0x1c, 0x07, 0xc7, 0x8f, 0x11,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(87, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_88(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x2e, 0x9b, 0xeb, 0x64, 0xfb, 0x0f, 0xd6, 0xfe, 0xa5, 0x9b, 0x58, 0x73,
			0xdc, 0x0a, 0x94, 0x6e, 0x8a, 0xf6, 0x8a, 0xfb, 0xf8, 0xf1, 0x63, 0x58,
			0x12, 0x4b, 0x4a, 0x56, 0x79, 0xa1, 0xa8, 0xdb,
		},
		{
			0x05, 0xfa, 0xc5, 0x86, 0xd8, 0x8f, 0x7d, 0xdd, 0x6f, 0xd8, 0x49, 0xe5,
			0xbd, 0x93, 0x32, 0xe1,
		},
		{
			0x81, 0xa6, 0x57, 0xd8, 0x96, 0xe2, 0x94, 0xdb, 0x49, 0x77, 0xb5, 0x26,
			0x53, 0xb8, 0xea, 0x29, 0xc8, 0x9e, 0xb2, 0xcd, 0xfd, 0xd9, 0x95, 0x67,
			0x6c, 0x81, 0x35, 0xba, 0x6e, 0x8d, 0xd3, 0x1f,
		},
		{
			0x0c, 0xc8, 0x1f, 0x75, 0x7d, 0x01, 0x42, 0x73, 0x66, 0xbb, 0x76, 0x4b,
			0x40, 0x37, 0x8c, 0x10, 0x24, 0xcf, 0x88, 0x13, 0x6b, 0xcc, 0x36, 0x24,
			0x3c, 0xdf, 0x11, 0x1c, 0x28, 0xd9, 0x76, 0xf8,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xde, 0x24, 0x72, 0x30, 0xbc, 0x5e, 0x80, 0x5a, 0x67, 0x88, 0xa4, 0xce,
			0x2b, 0xdd, 0x42, 0x87, 0x04, 0xc0, 0x76, 0x68, 0x63, 0x1f, 0x85, 0x91,
			0x78, 0x12, 0xcb, 0x04, 0xef, 0x3b, 0x76, 0x92, 0xe7, 0x26, 0x3a, 0x43,
			0x3a, 0x66, 0xcb, 0x78, 0x50, 0x4a, 0xdb, 0x04, 0xf5, 0xb6, 0x03, 0x95,
			0x68, 0x47, 0x83, 0xf4, 0x1a, 0x3c, 0x93, 0x7a, 0xd0, 0xe1, 0x0c, 0xb6,
			0x9c, 0x4e, 0x80, 0x95, 0x47, 0x7f, 0xf4, 0x37, 0x8b, 0x8a, 0x3a, 0xe0,
			0x6c, 0x9a, 0x03, 0xe9, 0x85, 0xbf, 0x14, 0xb9, 0x99, 0xd2, 0x76, 0xa4,
			0x5c, 0x27, 0x76, 0x2b, 0x4c, 0x09, 0xc5, 0x38, 0x72, 0xfe, 0xfc, 0xff,
			0xf0, 0x34, 0x96, 0xc1, 0xc8, 0x83, 0x3d, 0xaa, 0x4b, 0xb6, 0x09, 0x28,
			0xb4, 0x4a, 0x6c, 0x63, 0xaa, 0x44, 0x04, 0x03, 0x51, 0x53, 0x65, 0x90,
			0x7e, 0xd4, 0xd6, 0xcc, 0x87, 0x87, 0xb8, 0x92,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(88, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_89(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x51, 0x15, 0xf6, 0x8f, 0xf0, 0x85, 0x32, 0xab, 0xc3, 0x07, 0x7b, 0x0a,
			0x49, 0xf9, 0xeb, 0x9c, 0x29, 0x02, 0xfa, 0xb5, 0x23, 0x43, 0x9f, 0xb6,
			0x89, 0x15, 0x7f, 0xf9, 0xf0, 0x39, 0x97, 0xaf,
		},
		{
			0x41, 0xdf, 0x1a, 0xc0, 0x1f, 0x8c, 0xa0, 0x29, 0xff, 0xf6, 0x03, 0x4d,
			0x57, 0xd0, 0x93, 0xd0,
		},
		{
			0x1e, 0x92, 0xf8, 0xea, 0xb8, 0x16, 0x68, 0x25, 0x91, 0x5a, 0x56, 0xbb,
			0xb0, 0x0c, 0x4b, 0x25, 0x77, 0xcd, 0xc9, 0x9c, 0x36, 0x12, 0x62, 0x69,
			0xfc, 0xfe, 0xd6, 0x82, 0xb1, 0x2b, 0x06, 0xed,
		},
		{
			0x0b, 0x24, 0x5e, 0x6e, 0x6e, 0xd2, 0x3e, 0x17, 0x1a, 0x02, 0xd5, 0x2f,
			0x1f, 0x1a, 0x52, 0x0e, 0xd5, 0x0d, 0x67, 0xae, 0xdc, 0xf9, 0x99, 0xa9,
			0x06, 0x3e, 0xae, 0x51, 0xb3, 0xb3, 0x3e, 0x08,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x5b, 0x1e, 0xde, 0x46, 0xc5, 0x8a, 0x3d, 0x79, 0xcf, 0x63, 0x05, 0xae,
			0x59, 0xfc, 0xde, 0xdd, 0x1f, 0x88, 0xf1, 0xfe, 0x35, 0x14, 0xf6, 0xb8,
			0x16, 0x67, 0xeb, 0x65, 0x91, 0x72, 0xa1, 0x20, 0xb7, 0x0d, 0x12, 0x8d,
			0x23, 0xcd, 0xcf, 0x2a, 0xdd, 0xb9, 0xb9, 0x7c, 0xda, 0x13, 0x7a, 0xe3,
			0x02, 0xbc, 0x99, 0x3e, 0x9e, 0x40, 0x5b, 0x50, 0x85, 0xd8, 0xa0, 0xdf,
			0x2e, 0x4b, 0x9e, 0x0f, 0x4a, 0xcc, 0x64, 0x4a, 0x0b, 0xd4, 0x9c, 0xf6,
			0x0d, 0xd2, 0x17, 0x0c, 0x0e, 0xab, 0x8d, 0xf5, 0x37, 0x39, 0x78, 0xd8,
			0x9e, 0x41, 0x5f, 0xf6, 0x89, 0x10, 0x80, 0xae, 0xed, 0xed, 0xe6, 0x26,
			0x6e, 0xf1, 0x22, 0xd4, 0x48, 0xa0, 0xb2, 0x05, 0x74, 0xcc, 0x51, 0x1c,
			0x3b, 0xdf, 0x35, 0x38, 0x4f, 0x58, 0x87, 0x8a, 0x04, 0xeb, 0xcc, 0x86,
			0x2f, 0x2f, 0x1d, 0xd3, 0xc9, 0x3a, 0x10, 0x1a,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(89, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_90(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xdc, 0x59, 0x04, 0xbd, 0x75, 0x2c, 0x66, 0xe0, 0x25, 0x07, 0x26, 0x22,
			0x82, 0x4e, 0x4e, 0x78, 0x9d, 0xfa, 0xfe, 0xdc, 0xba, 0xad, 0x33, 0x56,
			0xa3, 0x44, 0x95, 0x3e, 0x00, 0x2f, 0xf8, 0x45,
		},
		{
			0x1a, 0xa5, 0x63, 0x88, 0x3b, 0xba, 0x44, 0x4c, 0x88, 0x50, 0x0e, 0xcf,
			0xcc, 0xa2, 0xaf, 0x94,
		},
		{
			0xab, 0x75, 0x4f, 0x51, 0xc0, 0x3f, 0x29, 0xeb, 0x77, 0x03, 0xca, 0x5e,
			0x4c, 0x8b, 0x10, 0xe1, 0x68, 0x49, 0x4a, 0x1f, 0x09, 0xb5, 0x44, 0x17,
			0x5b, 0xd8, 0x7b, 0x68, 0x7b, 0xf6, 0x9f, 0x7d,
		},
		{
			0xbb, 0xba, 0x28, 0x0c, 0x3e, 0x22, 0x07, 0x21, 0x2f, 0x18, 0xf7, 0x8d,
			0xba, 0x28, 0x62, 0x92, 0x0c, 0x2e, 0x1b, 0x69, 0xb8, 0xc9, 0x53, 0x1d,
			0xcf, 0x8a, 0x5c, 0x98, 0x1b, 0xf4, 0x38, 0x65,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x2f, 0x7d, 0x00, 0x53, 0xb5, 0x08, 0x3a, 0xe0, 0x49, 0x52, 0x4f, 0x18,
			0xa2, 0x5d, 0x07, 0x83, 0xdb, 0xff, 0xd2, 0xe5, 0x31, 0xf6, 0x73, 0xf8,
			0x7b, 0xbc, 0x28, 0x71, 0xc0, 0xc4, 0xbf, 0xcc, 0xbe, 0xb5, 0x44, 0xa0,
			0x9d, 0xc2, 0x08, 0xcf, 0xc7, 0x58, 0x2b, 0x48, 0x84, 0xc9, 0x94, 0xbf,
			0x84, 0xaa, 0xed, 0x1b, 0x55, 0xcd, 0x88, 0xe9, 0x56, 0x43, 0x02, 0x08,
			0x8c, 0xb5, 0x49, 0x92, 0xc4, 0x00, 0x26, 0xca, 0xcd, 0xd0, 0x28, 0x31,
			0x92, 0x6d, 0x11, 0xbf, 0x19, 0x94, 0xe0, 0x8e, 0xf2, 0xe9, 0x40, 0xf1,
			0xa7, 0xac, 0xc1, 0xcb, 0x3b, 0xdd, 0x4a, 0x3a, 0xfb, 0x2e, 0xe3, 0xa7,
			0xde, 0xfe, 0xfb, 0x6b, 0xa8, 0x7a, 0x5e, 0xf1, 0x82, 0x8c, 0x84, 0x02,
			0x86, 0x63, 0x1c, 0xf1, 0xc8, 0x84, 0x22, 0xba, 0x90, 0xe3, 0x95, 0x4b,
			0x42, 0x1b, 0x0a, 0x86, 0x9a, 0x9f, 0xde, 0xe1,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(90, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_91(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xea, 0x28, 0x92, 0x6c, 0xd5, 0xdf, 0x4f, 0xef, 0xd5, 0x72, 0xc9, 0x10,
			0x3d, 0x87, 0xff, 0xb0, 0x4f, 0x59, 0x9d, 0xa9, 0x5e, 0x1e, 0x6f, 0xec,
			0xb8, 0x4f, 0x53, 0xf7, 0x3f, 0xd0, 0x0d, 0x6c,
		},
		{
			0xcb, 0x40, 0xe1, 0x66, 0x55, 0xb9, 0xa2, 0xc7, 0x1e, 0x8e, 0x36, 0x77,
			0xb9, 0xea, 0x6c, 0x6f,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x3d, 0x2d, 0x1d, 0xb8, 0x8b, 0x84, 0x62, 0x78, 0x7a, 0x55, 0x76, 0xc9,
			0x5f, 0xd6, 0x60, 0x73, 0x4f, 0xb6, 0x81, 0xf8, 0x94, 0xe8, 0xef, 0xc4,
			0x7e, 0x3b, 0xe3, 0xbf, 0xc3, 0x09, 0x8e, 0x40,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xe9, 0x29, 0xc6, 0xe7, 0x49, 0xc5, 0x17, 0x50, 0x31, 0xdc, 0xc9, 0x26,
			0xbc, 0xe8, 0xd5, 0x29, 0x14, 0x7b, 0x5e, 0x94, 0x0f, 0x61, 0xd0, 0xba,
			0x1f, 0x02, 0x83, 0x1c, 0x80, 0xc2, 0x7a, 0x23, 0xcd, 0x4b, 0x5f, 0xfb,
			0x50, 0x7c, 0x7d, 0x09, 0xa7, 0x7e, 0x4c, 0x84, 0x27, 0xe2, 0x90, 0x10,
			0xcf, 0x1c, 0x80, 0x21, 0xa8, 0x0c, 0xa2, 0x95, 0x04, 0xca, 0xa3, 0x50,
			0xa2, 0x7d, 0x6c, 0xa4, 0x55, 0x4f, 0xe4, 0xd8, 0xb0, 0x23, 0x55, 0x54,
			0xf2, 0x51, 0xa5, 0x9e, 0xc6, 0x72, 0x9d, 0x80, 0x2b, 0x47, 0x30, 0x83,
			0xb0, 0xbd, 0x6c, 0xa8, 0x3f, 0x6d, 0x94, 0x5b, 0x3d, 0x1d, 0xe2, 0xb7,
			0x06, 0xbd, 0xcc, 0x3b, 0x50, 0xdd, 0xef, 0x57, 0x84, 0x7f, 0xff, 0x88,
			0xa4, 0x49, 0x85, 0x86, 0xca, 0x6a, 0xfe, 0x65, 0xe7, 0x6c, 0x2d, 0x97,
			0xf8, 0x7d, 0xde, 0xa6, 0x6f, 0x55, 0x63, 0xe3,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(91, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_92(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xbc, 0x3c, 0x34, 0x69, 0x28, 0x92, 0x91, 0x69, 0xc7, 0xa0, 0x0c, 0x6f,
			0x53, 0x2f, 0x7e, 0xb1, 0xc4, 0xa7, 0x7d, 0x42, 0x19, 0x3b, 0x65, 0x69,
			0x7b, 0xdd, 0xe5, 0xfe, 0x59, 0xae, 0x45, 0x20,
		},
		{
			0x1b, 0x2b, 0xa3, 0x3d, 0xa4, 0x9d, 0x3f, 0x6c, 0xf8, 0x40, 0x39, 0xcd,
			0x99, 0xcc, 0xe5, 0x0f,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb4, 0x25, 0x4a, 0xab, 0x62, 0xc4, 0x15, 0xd8, 0x32, 0x3d, 0x55, 0x6a,
			0xec, 0xa6, 0x14, 0x17, 0xb4, 0x1b, 0x09, 0x04, 0x65, 0x48, 0x0d, 0xb8,
			0xfb, 0x86, 0x4f, 0x57, 0x8f, 0x56, 0x7b, 0x3d,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x26, 0x02, 0xfc, 0xc0, 0xfc, 0xdb, 0x71, 0x04, 0x6d, 0xa4, 0x22, 0x6e,
			0x12, 0x77, 0xd5, 0xdf, 0x89, 0x22, 0x54, 0x73, 0xe4, 0x56, 0x4f, 0x04,
			0x7e, 0x60, 0x61, 0x0d, 0x98, 0x1a, 0x18, 0xe2, 0xb4, 0x4d, 0x79, 0xa9,
			0x3c, 0xd8, 0x64, 0xb8, 0x0f, 0xd3, 0x77, 0x68, 0x5f, 0xb4, 0x19, 0x8e,
			0xce, 0x5b, 0x22, 0x11, 0xae, 0xba, 0x57, 0xff, 0x49, 0xfd, 0xb6, 0x12,
			0x82, 0x62, 0xe2, 0x83, 0x6e, 0x7e, 0x81, 0x66, 0xf3, 0xc3, 0xd5, 0xa9,
			0x3c, 0x68, 0x6a, 0x9b, 0x70, 0xaa, 0x73, 0x9c, 0x66, 0xf8, 0x5f, 0x84,
			0xc4, 0x4c, 0x31, 0x46, 0x62, 0xd7, 0xaf, 0x17, 0xad, 0x62, 0x5e, 0x55,
			0xb6, 0xb4, 0x80, 0x1f, 0x0d, 0xda, 0xd8, 0xe6, 0x36, 0x85, 0x4d, 0x22,
			0x8f, 0x01, 0xba, 0x8f, 0x2f, 0x81, 0x20, 0x76, 0x20, 0x52, 0xd2, 0x89,
			0x48, 0x19, 0xa9, 0xfb, 0xc7, 0x6d, 0x6e, 0x6f,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(92, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_93(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x0b, 0xa0, 0x38, 0xbd, 0xb7, 0xad, 0x7b, 0xbe, 0x77, 0xa6, 0xc7, 0xda,
			0xa3, 0x3a, 0x2b, 0xdf, 0xd9, 0xa1, 0xa5, 0xf3, 0x1c, 0xd2, 0xe7, 0x53,
			0x77, 0xc7, 0x78, 0xf4, 0x92, 0xf5, 0xb3, 0xcb,
		},
		{
			0x17, 0xa1, 0x34, 0x23, 0x4a, 0x30, 0xe0, 0x91, 0x70, 0x89, 0x9c, 0x11,
			0x5b, 0x43, 0xca, 0x16,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xdb, 0x02, 0x0e, 0x00, 0xba, 0xce, 0x41, 0x7b, 0xa7, 0x68, 0x7e, 0x82,
			0x9c, 0x1f, 0xfc, 0x91, 0xd2, 0x2e, 0xf9, 0x3b, 0x27, 0xbf, 0xe1, 0x10,
			0x77, 0x32, 0xcf, 0xe0, 0x82, 0xa0, 0x5c, 0xf6,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x01, 0x79, 0x2f, 0x59, 0xd0, 0xa6, 0x3d, 0xef, 0xd8, 0x7d, 0x13, 0x18,
			0x8b, 0x1a, 0xec, 0xa9, 0x8c, 0x03, 0xb1, 0x1b, 0x7d, 0xd6, 0x69, 0x83,
			0x28, 0x17, 0x2d, 0xfb, 0x65, 0xcc, 0xb6, 0x95, 0xa1, 0xba, 0xa5, 0x8e,
			0x44, 0xfd, 0x75, 0x23, 0xfd, 0x17, 0x6d, 0x27, 0x06, 0x64, 0x33, 0x7e,
			0x59, 0x8a, 0xa4, 0x53, 0x64, 0x3c, 0xe3, 0xc6, 0xeb, 0xad, 0xed, 0xec,
			0x63, 0x11, 0xe2, 0xe1, 0xa6, 0xed, 0x22, 0x52, 0x47, 0x74, 0x01, 0x04,
			0x12, 0x53, 0xd3, 0xaf, 0xba, 0xda, 0xc0, 0x6e, 0x4d, 0x06, 0xf4, 0x6d,
			0xc3, 0xd7, 0xac, 0xe2, 0x87, 0xee, 0x3e, 0x23, 0x28, 0x6b, 0x28, 0x85,
			0x2e, 0xe0, 0xcf, 0x1e, 0x7a, 0xb3, 0xad, 0xb0, 0xff, 0x10, 0x07, 0xd0,
			0x77, 0x16, 0xb0, 0x9c, 0x22, 0xeb, 0x13, 0x80, 0xbb, 0xd0, 0xe8, 0xda,
			0xc4, 0x82, 0xdf, 0x48, 0x0d, 0x41, 0x58, 0x1d,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(93, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_94(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x2b, 0x37, 0xc7, 0x0c, 0x56, 0xa2, 0xa1, 0xc3, 0xd5, 0x15, 0x0e, 0xdc,
			0x4c, 0x31, 0x80, 0x7e, 0x17, 0x97, 0x3f, 0x4a, 0xfd, 0x85, 0x32, 0x18,
			0xe9, 0x94, 0x03, 0xd8, 0xdd, 0x9f, 0x19, 0x5f,
		},
		{
			0xa5, 0x11, 0x0c, 0x3a, 0x0f, 0x45, 0xba, 0x13, 0xb8, 0xea, 0x3f, 0x77,
			0xf3, 0x33, 0x58, 0x49,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x14, 0x50, 0xbb, 0xf9, 0xf6, 0xb6, 0x83, 0x30, 0xfa, 0x6c, 0x3e, 0xeb,
			0x20, 0x18, 0x5e, 0xcb, 0xb3, 0x0c, 0x4c, 0xcb, 0x4b, 0x7b, 0x5e, 0xe3,
			0x27, 0x12, 0x63, 0xa6, 0x43, 0xd4, 0xf3, 0xe9,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x57, 0x14, 0xe0, 0x8e, 0xcb, 0x34, 0xe5, 0x1c, 0x1b, 0x5b, 0x91, 0x40,
			0x5c, 0x81, 0x2e, 0x80, 0x10, 0x72, 0x9b, 0x69, 0x95, 0xb6, 0xcc, 0x90,
			0x14, 0x80, 0x9c, 0x47, 0xc3, 0xb6, 0xba, 0xf1, 0x10, 0x13, 0x45, 0x75,
			0x14, 0x6c, 0x6b, 0xb4, 0x3d, 0x20, 0xfe, 0x16, 0x38, 0x6d, 0x10, 0x05,
			0xb5, 0x64, 0x1b, 0xdc, 0x12, 0xb6, 0x45, 0x4f, 0x3b, 0xb7, 0x45, 0x88,
			0xb6, 0x16, 0xc6, 0x54, 0xbd, 0xca, 0x84, 0xf0, 0x71, 0x36, 0x4e, 0xfd,
			0x8d, 0x1e, 0xe6, 0x66, 0xcd, 0x60, 0xe4, 0xe2, 0xa8, 0xa3, 0xde, 0x99,
			0xe3, 0xf5, 0x6a, 0x72, 0x71, 0x4d, 0xb0, 0x4f, 0x15, 0xaa, 0x03, 0x76,
			0xac, 0xa8, 0xb1, 0x2d, 0x50, 0x31, 0x8c, 0x12, 0x8e, 0x1f, 0x4c, 0xe2,
			0x58, 0x40, 0x09, 0xed, 0x4d, 0x7e, 0x57, 0x54, 0x4a, 0xba, 0x9d, 0x63,
			0x12, 0x69, 0xef, 0xf9, 0x95, 0x31, 0x99, 0x23,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(94, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_95(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xab, 0x79, 0xf1, 0xff, 0x8a, 0x69, 0xc7, 0xf7, 0xeb, 0x33, 0xe5, 0x7f,
			0x03, 0xc5, 0xfd, 0x84, 0x58, 0xc0, 0xaf, 0xa9, 0xd2, 0x73, 0x88, 0xb7,
			0x84, 0x20, 0xa1, 0xac, 0x8d, 0x98, 0xbe, 0x68,
		},
		{
			0xb3, 0x6b, 0x72, 0xa0, 0x4b, 0xbd, 0xd2, 0x85, 0x96, 0x8d, 0x73, 0x05,
			0xd3, 0x56, 0xf8, 0xff,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x5c, 0xe8, 0x78, 0x3b, 0x45, 0x3b, 0xa8, 0x56, 0x7e, 0x7b, 0x6d, 0x7a,
			0x63, 0xac, 0xe5, 0xb4, 0xea, 0xf8, 0x75, 0x62, 0x42, 0x0e, 0x72, 0x9f,
			0xae, 0x94, 0x39, 0xce, 0x3a, 0x68, 0xdc, 0x86,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x9d, 0x10, 0x40, 0x4b, 0x95, 0x2e, 0x37, 0xc1, 0xd7, 0x00, 0x7b, 0xea,
			0xb7, 0xf8, 0xf1, 0xfa, 0xf2, 0x36, 0x17, 0x00, 0xf1, 0x38, 0x8b, 0xeb,
			0xfa, 0x83, 0xd8, 0x57, 0xac, 0xbd, 0x65, 0x20, 0x94, 0x7a, 0x4a, 0x06,
			0x9f, 0x80, 0x27, 0x8d, 0xae, 0x1d, 0x5a, 0x8e, 0xef, 0xf9, 0xd6, 0x9f,
			0x63, 0x30, 0x18, 0x31, 0x0e, 0x2a, 0x16, 0xac, 0xb9, 0x08, 0xb3, 0xb0,
			0x29, 0xb5, 0xbb, 0xef, 0x71, 0xf2, 0xa0, 0x59, 0x70, 0xf5, 0x44, 0xad,
			0xcf, 0xa6, 0xe7, 0x07, 0x79, 0x9f, 0xc1, 0xe6, 0xb7, 0x8b, 0x96, 0xea,
			0x1a, 0x57, 0x4f, 0x34, 0xd4, 0x49, 0x94, 0xe6, 0x8d, 0x1c, 0x40, 0xbc,
			0x2f, 0x39, 0x7a, 0xe1, 0xa4, 0x87, 0x28, 0xa9, 0x2b, 0x8b, 0x4d, 0x46,
			0xe3, 0x47, 0x22, 0xdd, 0xf2, 0x4d, 0xd3, 0xa0, 0x35, 0x26, 0x2e, 0x90,
			0xc5, 0xbf, 0xfe, 0x04, 0x89, 0x6b, 0x56, 0x58,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(95, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_96(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x05, 0xfa, 0xe5, 0xa2, 0x08, 0x03, 0xa1, 0x44, 0x21, 0x11, 0xac, 0x0b,
			0xd1, 0x81, 0x01, 0x86, 0x31, 0x9d, 0x72, 0x1d, 0x29, 0x46, 0xec, 0xa2,
			0x64, 0xfb, 0x31, 0xac, 0x8d, 0x8c, 0xbd, 0xa8,
		},
		{
			0x97, 0xc4, 0x1c, 0xa6, 0x69, 0xe2, 0x23, 0xee, 0xcf, 0xb2, 0x65, 0x99,
			0x41, 0xff, 0xa1, 0xbc,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xdf, 0x4f, 0x01, 0xea, 0x4a, 0x91, 0xaa, 0x2e, 0xb6, 0x51, 0x55, 0xf1,
			0x53, 0xdc, 0xa7, 0x0a, 0xfc, 0xe9, 0x8f, 0xf5, 0x6b, 0xff, 0x2a, 0x1c,
			0x94, 0x11, 0x78, 0xc2, 0xc2, 0xdb, 0xe9, 0x18,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xf3, 0xeb, 0x85, 0xc1, 0x68, 0x37, 0x79, 0x31, 0xa2, 0xf6, 0x77, 0xf1,
			0x06, 0x4c, 0x82, 0xde, 0x81, 0x5e, 0xff, 0x75, 0x70, 0x1b, 0x03, 0x16,
			0x06, 0x09, 0x87, 0x94, 0xee, 0xa7, 0x16, 0x96, 0x80, 0x54, 0x85, 0xf7,
			0x77, 0xad, 0xe2, 0xf2, 0x50, 0xe4, 0x8f, 0xd5, 0x8d, 0x23, 0x58, 0xaa,
			0x8e, 0xeb, 0xff, 0x59, 0x7d, 0x39, 0x6a, 0x7a, 0xfc, 0xf3, 0x8b, 0xe6,
			0x5e, 0x0e, 0x78, 0x40, 0x5d, 0x35, 0xee, 0x3d, 0x8d, 0xd2, 0x98, 0x5c,
			0xd6, 0x2c, 0x26, 0xca, 0xc1, 0xbd, 0x0d, 0x9c, 0x6e, 0x67, 0xac, 0xcf,
			0xbb, 0xe2, 0xbf, 0xda, 0x87, 0xd5, 0x0d, 0xe6, 0x41, 0x5a, 0x4a, 0x9b,
			0xaa, 0x27, 0x3f, 0x99, 0x7b, 0xdc, 0xe9, 0x03, 0x27, 0x1a, 0x05, 0x82,
			0xbe, 0xfc, 0x40, 0x2f, 0x33, 0xfc, 0xab, 0xb6, 0xdb, 0x6f, 0x14, 0x52,
			0x52, 0xd8, 0x70, 0x55, 0x4d, 0x78, 0x91, 0xd2,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(96, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_97(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xf0, 0x7a, 0xce, 0x29, 0x9e, 0xa3, 0xc1, 0x04, 0x9d, 0x88, 0x37, 0x8e,
			0x3a, 0x64, 0xd6, 0xfe, 0x78, 0x1b, 0x87, 0x72, 0xcb, 0xd1, 0x92, 0xc9,
			0x69, 0x2a, 0x4d, 0x8e, 0x7a, 0xec, 0x19, 0x87,
		},
		{
			0x87, 0xcd, 0x0d, 0x8f, 0xff, 0x8d, 0xad, 0xec, 0x5b, 0x8b, 0xcc, 0xd0,
			0x85, 0xf4, 0x1a, 0x1e,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x42, 0xfb, 0x6e, 0x92, 0xee, 0x26, 0x3b, 0x1c, 0x03, 0xa6, 0x3e, 0x5b,
			0xea, 0xf8, 0x06, 0x62, 0xad, 0x67, 0xb1, 0x95, 0xe1, 0x55, 0xd0, 0xac,
			0xb7, 0x47, 0x8a, 0x82, 0x93, 0x83, 0x1f, 0xc6,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x3e, 0x3e, 0xeb, 0x53, 0x80, 0x0b, 0x48, 0x74, 0xbd, 0xda, 0x33, 0x1c,
			0x1e, 0x9b, 0x98, 0xc5, 0x77, 0x00, 0xe7, 0xc2, 0x06, 0xf5, 0xf3, 0xd0,
			0xf4, 0x21, 0xe3, 0xc4, 0x90, 0x79, 0x70, 0x84, 0xc0, 0x9e, 0xd6, 0xb4,
			0xbd, 0x58, 0x85, 0xbc, 0xaa, 0x42, 0xe0, 0x36, 0x87, 0x0d, 0x63, 0x3d,
			0xf8, 0xb3, 0xbf, 0x1d, 0x1b, 0xaf, 0xf5, 0x88, 0xac, 0x9b, 0x24, 0xe0,
			0x82, 0x7c, 0x26, 0xed, 0x4f, 0x06, 0x18, 0xd9, 0x12, 0xd3, 0x90, 0x04,
			0xcc, 0xa9, 0x73, 0x9e, 0x0b, 0x99, 0x46, 0xf8, 0x92, 0x7b, 0x58, 0x0b,
			0xb2, 0xf3, 0x2c, 0x9f, 0x14, 0x2a, 0x7b, 0x1c, 0x09, 0xb3, 0x2a, 0x40,
			0x5e, 0x1e, 0xf5, 0x20, 0x9c, 0xae, 0x79, 0x0b, 0x07, 0x11, 0xd1, 0x49,
			0x93, 0x51, 0x09, 0x53, 0x01, 0x8f, 0x27, 0x9c, 0x8c, 0x8d, 0xfd, 0x06,
			0x64, 0x8c, 0xd1, 0xb1, 0xa5, 0x8b, 0xdf, 0x93,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(97, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_98(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x96, 0xc3, 0xe7, 0xe5, 0xfe, 0xcc, 0xc0, 0x61, 0x80, 0x59, 0x00, 0x6c,
			0x73, 0x5a, 0x6c, 0xe0, 0xb5, 0x11, 0x4d, 0x20, 0x8c, 0x79, 0x14, 0x3e,
			0x1c, 0x9f, 0x8e, 0xf5, 0x03, 0x6e, 0x33, 0x0d,
		},
		{
			0x6b, 0xbb, 0x09, 0xc4, 0x16, 0x51, 0xc1, 0xc7, 0x8c, 0x36, 0xc2, 0x5e,
			0x4c, 0xee, 0x8a, 0xa9,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xab, 0xa5, 0x10, 0x86, 0x57, 0x7c, 0x69, 0x48, 0xb8, 0x92, 0xcf, 0x52,
			0x54, 0x52, 0x0b, 0xf8, 0xd2, 0x40, 0xc2, 0xe3, 0xb0, 0x92, 0x4a, 0xfe,
			0x69, 0x65, 0xac, 0xfa, 0xbd, 0x7e, 0x0a, 0xbf,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x1d, 0x7e, 0xc4, 0x82, 0x34, 0xbb, 0x2a, 0xaa, 0xf5, 0x4f, 0xbd, 0x55,
			0xff, 0x06, 0x63, 0xf0, 0xfc, 0x82, 0x82, 0x25, 0x31, 0xf6, 0x66, 0xee,
			0x7f, 0x7f, 0x5e, 0x25, 0x70, 0x07, 0xf3, 0x04, 0x62, 0x42, 0x8a, 0x51,
			0x9d, 0x86, 0xa8, 0x76, 0x4b, 0xa0, 0xd6, 0xa2, 0x33, 0x1c, 0xb1, 0x29,
			0x23, 0x2d, 0x9a, 0x5c, 0x03, 0x18, 0x3b, 0x0f, 0x41, 0x9d, 0x81, 0x3f,
			0x4c, 0x0e, 0x00, 0xe4, 0xbe, 0x5d, 0xd1, 0x6a, 0x87, 0xa9, 0xfa, 0xec,
			0x89, 0x15, 0xa3, 0x9f, 0xd2, 0xd4, 0x06, 0xc5, 0x4e, 0xb8, 0xaa, 0xf2,
			0x28, 0xa0, 0x9a, 0x46, 0xdb, 0x55, 0xfd, 0xa8, 0xa9, 0x0b, 0x27, 0xef,
			0xc9, 0x6c, 0x46, 0x1e, 0x9e, 0x9b, 0xe0, 0x8e, 0x27, 0xaf, 0x86, 0xb0,
			0x17, 0x50, 0xdb, 0xbf, 0xb4, 0x02, 0x45, 0xe9, 0x98, 0xf0, 0x67, 0x96,
			0x86, 0xe1, 0x8d, 0x80, 0x17, 0x02, 0xff, 0xa2,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(98, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_99(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x21, 0x07, 0x25, 0xc6, 0x08, 0x39, 0x78, 0x2d, 0x36, 0x65, 0xa3, 0x00,
			0xad, 0xcf, 0x81, 0xc6, 0xd5, 0x14, 0x78, 0xb1, 0xf1, 0x2e, 0x5d, 0x53,
			0x10, 0x4b, 0xeb, 0x38, 0x5c, 0xe1, 0x50, 0x24,
		},
		{
			0x12, 0x9a, 0x70, 0x5c, 0x88, 0xe4, 0xfd, 0xb1, 0x67, 0xd3, 0x47, 0x94,
			0xf6, 0xf7, 0x39, 0xb4,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xe1, 0x2e, 0x9c, 0xc5, 0x23, 0xd6, 0x18, 0xae, 0x6c, 0xb6, 0x67, 0xec,
			0xa7, 0xed, 0xd3, 0x54, 0x58, 0x18, 0x7b, 0xd1, 0x9e, 0x90, 0x79, 0xb5,
			0xc5, 0xe1, 0xcb, 0x97, 0x1e, 0x38, 0x03, 0x09,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x34, 0x5b, 0x3b, 0x8c, 0x01, 0x1c, 0x78, 0x7e, 0x90, 0x63, 0xf8, 0x11,
			0xd5, 0x1f, 0x09, 0x6a, 0x31, 0xcc, 0x1e, 0xa9, 0xfa, 0xa0, 0xb8, 0xf2,
			0x9d, 0x91, 0x6b, 0xe1, 0xb9, 0x25, 0xa4, 0xb7, 0x1f, 0x8e, 0x6d, 0xb5,
			0x6e, 0x0e, 0x22, 0xab, 0xfd, 0xe3, 0xf0, 0x35, 0xf2, 0xd5, 0xa6, 0xfc,
			0xde, 0x02, 0xe5, 0xdf, 0x8c, 0x0b, 0xef, 0xe0, 0xc0, 0x04, 0x38, 0xfa,
			0xa4, 0x27, 0x7c, 0xa7, 0x90, 0xe6, 0x32, 0x31, 0x80, 0x61, 0xfb, 0xab,
			0xbb, 0x83, 0xdc, 0x83, 0xf7, 0x19, 0xf3, 0xce, 0xab, 0x43, 0x1e, 0x3f,
			0xe8, 0x06, 0x95, 0xbc, 0x44, 0xd7, 0x4d, 0x09, 0x35, 0xc8, 0x42, 0x25,
			0x21, 0xce, 0x78, 0x4a, 0x92, 0x3c, 0xac, 0xc3, 0x75, 0x17, 0xb9, 0xdd,
			0x93, 0xc7, 0xc0, 0x60, 0x8f, 0x95, 0x5e, 0x5b, 0x48, 0x68, 0x0b, 0x53,
			0x71, 0x42, 0x45, 0x9a, 0x6d, 0x7b, 0x79, 0xe8,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(99, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_100(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xf8, 0x7e, 0x04, 0x74, 0x38, 0xc6, 0xf5, 0xdc, 0x41, 0xa5, 0xe3, 0xbd,
			0x0d, 0xdb, 0x38, 0x6c, 0x99, 0x40, 0x7b, 0xf2, 0x76, 0xb5, 0x78, 0x24,
			0xd4, 0x36, 0x6e, 0x09, 0x73, 0xef, 0x65, 0xf2,
		},
		{
			0x73, 0x00, 0x2b, 0x5a, 0xc3, 0xcd, 0xb5, 0x61, 0x5a, 0x2e, 0x17, 0x92,
			0xb9, 0x63, 0x86, 0xe2,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xca, 0x22, 0xf8, 0x53, 0xf4, 0x37, 0x1e, 0xd3, 0x34, 0xab, 0x94, 0x44,
			0x4b, 0x58, 0xa4, 0xf9, 0xcb, 0xda, 0xcf, 0xda, 0x96, 0xe4, 0xa9, 0xe4,
			0x6c, 0x9c, 0xf7, 0x76, 0xa9, 0x03, 0x99, 0x0d,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x5d, 0x54, 0x82, 0x7f, 0xb3, 0x7f, 0xd8, 0x64, 0x41, 0xab, 0x4e, 0xb3,
			0x79, 0x60, 0x89, 0xde, 0x39, 0xf1, 0x13, 0x24, 0x41, 0x58, 0x23, 0x6e,
			0xc1, 0x62, 0x99, 0x69, 0xf9, 0x29, 0x5f, 0x37, 0x83, 0x6b, 0x2d, 0x7d,
			0x4d, 0x6e, 0x59, 0xe0, 0xbd, 0xce, 0xfd, 0x92, 0x3c, 0x2c, 0x44, 0x21,
			0x18, 0xc8, 0xca, 0x3b, 0x00, 0xa2, 0x58, 0xbe, 0xec, 0xad, 0x7d, 0x28,
			0xd7, 0xe1, 0x21, 0x62, 0xa4, 0x20, 0x21, 0x29, 0x18, 0x93, 0x7a, 0x4d,
			0x0d, 0xa5, 0xea, 0x7c, 0x0b, 0x96, 0x02, 0x81, 0x32, 0x66, 0x01, 0x5f,
			0x38, 0x5e, 0xd5, 0xc6, 0x9a, 0x48, 0x37, 0x5e, 0x9b, 0x37, 0x38, 0x81,
			0x4f, 0x16, 0xc0, 0x85, 0xd5, 0xcf, 0xd1, 0xd2, 0xe6, 0x26, 0x7e, 0x1c,
			0x5d, 0x7b, 0xad, 0x88, 0xf2, 0x0b, 0x3a, 0xb0, 0xab, 0x27, 0x9f, 0xcb,
			0x65, 0x10, 0x20, 0xec, 0xfc, 0x15, 0x07, 0x63,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(100, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_101(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x50, 0xeb, 0xb3, 0x84, 0x61, 0xd3, 0xf2, 0x6c, 0x5a, 0xc2, 0xec, 0xf5,
			0x3a, 0x03, 0x41, 0x78, 0x42, 0x52, 0x5b, 0xad, 0x57, 0x7f, 0x05, 0x7e,
			0xd9, 0x20, 0xc1, 0xc4, 0x97, 0x0e, 0x55, 0x29,
		},
		{
			0x99, 0x6c, 0x6b, 0x9d, 0xb4, 0xa1, 0xaa, 0xb9, 0xd3, 0x1c, 0x75, 0x4a,
			0xf4, 0x9a, 0x54, 0x45,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xea, 0x89, 0x85, 0x50, 0xce, 0xe3, 0x88, 0x88, 0xc0, 0x38, 0xf6, 0x7b,
			0x49, 0x12, 0xe4, 0x8a, 0xa9, 0xe3, 0x78, 0xff, 0x3a, 0xdc, 0x9c, 0xc6,
			0xc5, 0xba, 0xee, 0x8c, 0x37, 0x6f, 0xc7, 0x08,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x77, 0x37, 0x71, 0xf1, 0xf3, 0x90, 0x37, 0x63, 0xf4, 0xf9, 0xd6, 0xb2,
			0x16, 0x73, 0xd2, 0xa9, 0x3f, 0x6e, 0x4d, 0x03, 0x63, 0x6d, 0x38, 0x1b,
			0xc2, 0x68, 0x64, 0xb4, 0x64, 0x96, 0xe9, 0x58, 0x8a, 0x06, 0x4b, 0x00,
			0xe8, 0x46, 0xa8, 0xd9, 0x1e, 0x30, 0x2c, 0x47, 0xda, 0xcd, 0xab, 0x73,
			0x88, 0xf5, 0xcd, 0x00, 0x93, 0xd2, 0x17, 0x4e, 0x33, 0x32, 0x7f, 0x9d,
			0xf6, 0x53, 0x71, 0x24, 0xab, 0x91, 0xcf, 0x35, 0xcb, 0xee, 0xce, 0x38,
			0xf6, 0x37, 0x3a, 0x8b, 0x85, 0x1d, 0xa2, 0x64, 0x5d, 0xad, 0x5c, 0xf0,
			0x1a, 0x59, 0xf5, 0x31, 0x80, 0xa0, 0xfb, 0xfe, 0xb9, 0x28, 0xa7, 0x10,
			0x90, 0xd1, 0x99, 0x7b, 0x05, 0x5f, 0xa2, 0x0f, 0x14, 0xf4, 0x9c, 0x28,
			0xdc, 0x06, 0x5d, 0xdd, 0x84, 0xa1, 0x4d, 0x93, 0x23, 0x19, 0x2b, 0x24,
			0x28, 0xae, 0xc6, 0xf9, 0x15, 0xf9, 0x04, 0x91,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(101, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_102(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0xc1, 0x3d, 0x6c, 0xd6, 0x3b, 0xb7, 0x93, 0x11, 0x74, 0x69, 0x6f, 0x3e,
			0x04, 0xa0, 0xc4, 0x1c, 0xb0, 0xb2, 0x56, 0x11, 0x34, 0xe8, 0x47, 0xce,
			0x03, 0xe3, 0x63, 0x26, 0xb8, 0x03, 0xf8, 0xab,
		},
		{
			0x20, 0x84, 0xab, 0x32, 0x37, 0x43, 0x92, 0xea, 0x9f, 0x6e, 0x8a, 0x47,
			0x4f, 0x18, 0xe9, 0xd7,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xae, 0xc5, 0xa6, 0xa7, 0x23, 0x2a, 0x52, 0xb8, 0x1c, 0xe7, 0xe9, 0x81,
			0xa3, 0x59, 0xce, 0xf1, 0xbb, 0xd2, 0xf1, 0xef, 0xf8, 0x48, 0x83, 0x71,
			0x46, 0x8c, 0xd1, 0xf4, 0x14, 0x7a, 0x89, 0xc2,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xda, 0xea, 0x78, 0x88, 0x17, 0x37, 0xcb, 0x26, 0xd6, 0x0c, 0x36, 0xce,
			0xb9, 0xfe, 0xc3, 0xd2, 0x81, 0xc7, 0xae, 0xc5, 0x4b, 0x4b, 0x98, 0x50,
			0x93, 0x7b, 0x37, 0x3b, 0x2b, 0x26, 0x21, 0xfe, 0x07, 0x75, 0x85, 0xa1,
			0xfe, 0x88, 0x26, 0x5d, 0x84, 0xf2, 0x25, 0x55, 0x2e, 0x5c, 0x85, 0xcb,
			0xec, 0x8d, 0x00, 0x06, 0x96, 0x48, 0x06, 0x5a, 0xc1, 0x20, 0x73, 0xae,
			0xdc, 0xe8, 0xc9, 0x40, 0x46, 0x09, 0x49, 0xb5, 0x97, 0x66, 0x7e, 0xcf,
			0xce, 0xda, 0xbd, 0x7a, 0x86, 0xa9, 0x79, 0xb9, 0x04, 0xa2, 0x4d, 0x32,
			0xdb, 0x10, 0x22, 0x3e, 0xae, 0x5a, 0x98, 0xa0, 0xd1, 0xb6, 0x57, 0x1b,
			0x86, 0x43, 0xdf, 0x2c, 0x62, 0x65, 0xa5, 0xd6, 0x6c, 0xee, 0x9f, 0x4a,
			0xbf, 0xc5, 0x77, 0x81, 0x46, 0xd6, 0xfb, 0x2b, 0x85, 0x3d, 0x82, 0x63,
			0x6c, 0x13, 0x25, 0xb2, 0xd1, 0xef, 0x45, 0x76,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(102, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_103(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x67, 0xf9, 0x8d, 0x4d, 0x9f, 0x35, 0xa8, 0xda, 0xc0, 0x3d, 0x19, 0x92,
			0x22, 0xb0, 0x7c, 0x78, 0x5f, 0x11, 0xdc, 0x51, 0x7a, 0x8d, 0x88, 0x40,
			0xf5, 0xfe, 0xa4, 0xb6, 0xfd, 0x68, 0x44, 0x8e,
		},
		{
			0x1d, 0xcd, 0x3a, 0x1b, 0xbb, 0x4f, 0x94, 0xad, 0x78, 0x35, 0x58, 0x42,
			0x89, 0x3d, 0xe0, 0x35,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xf1, 0xf9, 0xca, 0xee, 0x31, 0x03, 0x6a, 0x2a, 0x20, 0x0a, 0x05, 0x2d,
			0x8a, 0x1e, 0x06, 0x59, 0x20, 0xb8, 0x36, 0x3f, 0xbb, 0xf1, 0xc3, 0x71,
			0xa2, 0x32, 0x44, 0xc6, 0x83, 0x51, 0x13, 0xd3,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x9f, 0x24, 0xa5, 0x6f, 0x98, 0x58, 0x51, 0x26, 0x2d, 0x0c, 0xfd, 0xf1,
			0x22, 0xb8, 0x9b, 0x3b, 0x93, 0xa1, 0xdf, 0xd9, 0x7a, 0x29, 0x7f, 0xb1,
			0xf6, 0xfc, 0x39, 0xd6, 0xc7, 0x19, 0x94, 0x2e, 0x28, 0x97, 0x7b, 0xde,
			0x91, 0xcb, 0x39, 0x41, 0xc0, 0x35, 0xe3, 0xec, 0x45, 0x4d, 0xbf, 0x9d,
			0xbb, 0x43, 0x57, 0x69, 0x20, 0xdc, 0x47, 0x27, 0x3d, 0x3f, 0x82, 0xcc,
			0xc8, 0xa8, 0xfa, 0xbc, 0x1b, 0x53, 0x2f, 0x9d, 0x1c, 0x48, 0x7d, 0xb3,
			0x1e, 0x7e, 0xaa, 0xc3, 0x32, 0x5e, 0xdd, 0x1f, 0xd8, 0x08, 0x70, 0xcf,
			0x30, 0xcc, 0xdc, 0x87, 0xa1, 0xfa, 0x4b, 0x88, 0xeb, 0x7a, 0x1c, 0x46,
			0x4d, 0x0b, 0x68, 0xef, 0xc3, 0xdd, 0x76, 0x0a, 0x32, 0x59, 0xad, 0x28,
			0x0d, 0x49, 0x64, 0x2b, 0xeb, 0xde, 0xbc, 0x1a, 0x58, 0x29, 0xf3, 0x2f,
			0x09, 0xfb, 0x77, 0x7a, 0xfb, 0x40, 0x9a, 0xee,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(103, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_104(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x0f, 0x4a, 0x78, 0xc8, 0x62, 0x3d, 0x30, 0x31, 0x54, 0x39, 0x8c, 0xb9,
			0x34, 0x08, 0x57, 0xba, 0x93, 0x73, 0x88, 0xf9, 0x5b, 0x01, 0x97, 0xec,
			0xba, 0x48, 0x7c, 0x4c, 0xf0, 0x21, 0xb3, 0xca,
		},
		{
			0x6c, 0x02, 0xe6, 0x93, 0x98, 0xab, 0x7b, 0x7d, 0xfc, 0x4a, 0xd8, 0x3e,
			0x7f, 0x3c, 0xa8, 0xff,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x03, 0x02, 0x0b, 0x45, 0x11, 0xb2, 0x52, 0x9b, 0x2a, 0x33, 0x48, 0x07,
			0x8a, 0xd0, 0x31, 0x8b, 0xa5, 0xf8, 0x25, 0xa1, 0xbb, 0x06, 0x72, 0xdc,
			0x89, 0x4c, 0xb4, 0xd1, 0xdd, 0x3e, 0x29, 0x4d,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xf5, 0xb9, 0xa5, 0x26, 0x29, 0x70, 0xb8, 0xcd, 0x94, 0x48, 0x2a, 0xc6,
			0xe4, 0xde, 0xdd, 0x8f, 0xfd, 0xca, 0xca, 0x0f, 0xc3, 0x7e, 0x6d, 0x0f,
			0xa7, 0x1a, 0xb1, 0x0e, 0x5a, 0xe0, 0x74, 0x87, 0xc6, 0x88, 0x01, 0x7d,
			0xc0, 0x15, 0x14, 0xf8, 0x57, 0x70, 0x2b, 0x28, 0xcd, 0x8e, 0x4f, 0x1b,
			0x08, 0x39, 0xb9, 0xec, 0x4d, 0x6f, 0x9e, 0x07, 0xe4, 0x55, 0x1e, 0x15,
			0x0c, 0x91, 0xcd, 0x4f, 0x4a, 0xa2, 0x2a, 0xa7, 0xe8, 0xa5, 0xe1, 0x07,
			0x78, 0x69, 0xb2, 0x45, 0x4b, 0xb8, 0xcd, 0xda, 0xee, 0x69, 0xa0, 0x67,
			0xc6, 0xb0, 0xdc, 0xc1, 0x78, 0xfe, 0x1d, 0xa6, 0xed, 0xe5, 0xd5, 0x32,
			0x69, 0x37, 0x25, 0x36, 0xf3, 0x4b, 0xed, 0x37, 0x85, 0x66, 0xcb, 0xd8,
			0x98, 0x6d, 0xc1, 0xae, 0x08, 0x48, 0xbe, 0x74, 0x90, 0x79, 0xdd, 0x57,
			0x2d, 0x4d, 0x62, 0xe5, 0x1f, 0xe6, 0x59, 0xca,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(104, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_105(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		0,
		0,
		128,
		{
			0x00, 0x1c, 0xf2, 0xbb, 0x87, 0xee, 0x4a, 0xa1, 0x45, 0x01, 0x19, 0x08,
			0xce, 0x79, 0x92, 0x07, 0x00, 0xff, 0x40, 0xce, 0x76, 0x50, 0x99, 0x29,
			0x2b, 0x7b, 0x7a, 0xec, 0x68, 0xee, 0xb0, 0x5e,
		},
		{
			0xb1, 0x01, 0x15, 0xcc, 0xfe, 0xfe, 0xc7, 0x48, 0x7c, 0x87, 0x24, 0x12,
			0x97, 0x70, 0x67, 0xa9,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x9b, 0x5d, 0xcf, 0xa4, 0x0b, 0x5c, 0x5a, 0x9d, 0xca, 0xe4, 0xe8, 0xcd,
			0xf1, 0x6e, 0x14, 0x58, 0xd4, 0x91, 0x64, 0xb3, 0x17, 0xce, 0xfc, 0xe6,
			0x3f, 0x28, 0xc1, 0x0d, 0x29, 0x22, 0x99, 0x74,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xe0, 0xf0, 0x73, 0xc0, 0xbb, 0x40, 0x24, 0xc4, 0x6c, 0xed, 0xd5, 0x6f,
			0x6d, 0xde, 0x44, 0x67, 0x7d, 0x8d, 0xcd, 0xaf, 0x8d, 0xf9, 0x39, 0xe2,
			0xcb, 0xfa, 0xaf, 0x0f, 0xc3, 0x81, 0x0f, 0x62, 0x1b, 0x29, 0xdd, 0xaa,
			0x78, 0x10, 0xee, 0x3c, 0xfc, 0x25, 0x12, 0x32, 0xc4, 0xf8, 0xb0, 0x1c,
			0x56, 0x64, 0xe4, 0xaf, 0x5e, 0xa0, 0xa7, 0xa2, 0x0c, 0x76, 0x8a, 0xae,
			0x83, 0x92, 0x18, 0x63, 0xd7, 0xe0, 0xff, 0x12, 0x77, 0x8e, 0xe1, 0xe6,
			0xe4, 0xef, 0xee, 0xa6, 0x32, 0x0f, 0x9c, 0x8d, 0xd4, 0x39, 0xe7, 0x98,
			0x7e, 0x62, 0x0a, 0xc2, 0xee, 0x40, 0x96, 0x52, 0xba, 0x7f, 0x07, 0x70,
			0x8c, 0xa1, 0x9a, 0x2e, 0x16, 0xf4, 0xd3, 0x24, 0xba, 0x1a, 0xcc, 0xa3,
			0xa4, 0xd2, 0x44, 0x30, 0x24, 0x38, 0x60, 0x8b, 0xa3, 0xb6, 0x4c, 0xae,
			0x05, 0x54, 0x5a, 0xd8, 0x67, 0x19, 0x49, 0x51,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(105, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_106(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x22, 0xd8, 0xc6, 0x2b, 0xcd, 0xdf, 0x5d, 0xa1, 0xdb, 0xdb, 0x09, 0x3d,
			0x6b, 0x1f, 0x66, 0x3d, 0xde, 0x33, 0x73, 0x43, 0xc5, 0x6c, 0xcf, 0xd4,
			0x0a, 0xcb, 0xe3, 0x30, 0x23, 0x09, 0xea, 0xe0,
		},
		{
			0x2a, 0x4b, 0x8e, 0x66, 0xde, 0xaa, 0xc3, 0x8b, 0x70, 0xd9, 0xff, 0xc2,
			0x0c, 0x58, 0x5d, 0xa5,
		},
		{
			0x0a, 0x33, 0x70, 0x38, 0xf4, 0xb4, 0x57, 0x3f, 0xf4, 0x3a, 0x43, 0x21,
			0xa5, 0x86, 0xca, 0x77, 0x7c, 0x30, 0x30, 0x12, 0x67, 0xd8, 0x2f, 0xdf,
			0x93, 0x71, 0x98, 0xac, 0x56, 0xc7, 0x06, 0x2c,
		},
		{
			0x0b, 0x6f, 0xd1, 0x79, 0xe4, 0x4f, 0x14, 0x9f, 0x06, 0x2d, 0xa4, 0xf6,
			0x6f, 0x82, 0x9c, 0x3c, 0x58, 0xc4, 0xa0, 0xa4, 0xf7, 0x5a, 0xc2, 0xa9,
			0xe0, 0x24, 0x0d, 0x43, 0xbe, 0xc3, 0x0e, 0x44,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x04, 0x6e, 0x19, 0xb4, 0xd8, 0xab, 0x38, 0xdd, 0x08, 0xde, 0xfd, 0xd0,
			0xd7, 0xc3, 0x0c, 0x8c, 0x75, 0xd5, 0x68, 0x9c, 0xa2, 0x6f, 0x2b, 0xc9,
			0x94, 0xac, 0x7f, 0xca, 0x4f, 0xdb, 0xee, 0x80, 0x67, 0x7d, 0xfb, 0xdd,
			0x85, 0x1e, 0x77, 0x22, 0x83, 0x58, 0x44, 0xdc, 0xa7, 0x9d, 0xec, 0x4a,
			0x3f, 0xa8, 0x2f, 0xeb, 0x88, 0x4d, 0xf7, 0xd4, 0x74, 0xbd, 0x97, 0x2e,
			0x12, 0x61, 0x9b, 0xd5, 0xd6, 0xcb, 0x1b, 0x95, 0x1e, 0xac, 0x47, 0xee,
			0xc2, 0x58, 0x11, 0x95, 0xb5, 0x31, 0x53, 0x4e, 0xde, 0x50, 0x7a, 0xf6,
			0xf7, 0x41, 0x7f, 0xca, 0x84, 0x53, 0x2b, 0xe7, 0xef, 0x5d, 0xa6, 0x73,
			0x8d, 0xbf, 0x7a, 0xad, 0xfc, 0xd7, 0xcb, 0x88, 0x88, 0x62, 0xb5, 0x2e,
			0xc7, 0x73, 0xcf, 0x3f, 0xd0, 0x0e, 0x6d, 0x4e, 0xfb, 0x30, 0x71, 0xaf,
			0x9b, 0x70, 0xa4, 0x99, 0x35, 0xa3, 0xba, 0x38,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(106, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_107(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x64, 0x11, 0x75, 0x6d, 0xc7, 0xaa, 0x3b, 0xe0, 0x96, 0xad, 0x41, 0x9c,
			0xe8, 0x38, 0x1c, 0x09, 0x10, 0xd2, 0xa2, 0xaf, 0x5e, 0x94, 0x4d, 0x8f,
			0xf2, 0x1f, 0x2f, 0xc5, 0x66, 0x6a, 0xcb, 0x87,
		},
		{
			0x73, 0xb6, 0x8f, 0x5c, 0x74, 0x9f, 0xac, 0x09, 0xf0, 0x1b, 0xa5, 0x65,
			0xa7, 0x57, 0x89, 0xf8,
		},
		{
			0x3c, 0xa3, 0xd9, 0x14, 0x6e, 0x2e, 0xbf, 0xf2, 0x1a, 0x28, 0x7d, 0x38,
			0x5c, 0x42, 0x24, 0xed, 0x23, 0x59, 0x6d, 0x74, 0x50, 0x17, 0xf3, 0xb8,
			0x11, 0x48, 0xc9, 0x2f, 0x3b, 0x9b, 0x27, 0xa1,
		},
		{
			0xd7, 0x70, 0x12, 0x8e, 0x71, 0x1a, 0x8d, 0xf1, 0x63, 0xf4, 0xe2, 0x89,
			0xe7, 0x75, 0x03, 0x2f, 0xdb, 0x47, 0x3a, 0xa3, 0x8f, 0x7e, 0x1d, 0x29,
			0x17, 0x26, 0x12, 0x13, 0x78, 0xe9, 0xa8, 0xde,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xcd, 0x72, 0xf8, 0x09, 0xd9, 0xcd, 0x1a, 0xe1, 0x3d, 0x99, 0x5d, 0x1c,
			0xac, 0x36, 0x8e, 0xdb, 0xff, 0xf4, 0x56, 0x97, 0xde, 0xb1, 0xd1, 0xb7,
			0x95, 0xff, 0xcf, 0x35, 0x49, 0x36, 0xf0, 0xbf, 0x61, 0xcf, 0x55, 0x0d,
			0x04, 0x3f, 0x9d, 0x2f, 0xa4, 0x62, 0x81, 0xa1, 0x39, 0x3e, 0xac, 0xe3,
			0x84, 0x7d, 0x5e, 0x4c, 0x63, 0x35, 0x18, 0x2f, 0x87, 0xf4, 0x65, 0x80,
			0x29, 0x89, 0xb2, 0x55, 0x70, 0x1f, 0x9d, 0xae, 0xe0, 0x3f, 0x59, 0x1b,
			0xa5, 0xc0, 0x80, 0xd1, 0xf0, 0x58, 0x4b, 0x02, 0x7a, 0xb3, 0xb1, 0xda,
			0x57, 0x35, 0xc9, 0x36, 0x11, 0xd9, 0xbe, 0x11, 0xa7, 0x9c, 0x0f, 0xde,
			0x3b, 0xd4, 0x20, 0x4e, 0xa3, 0x67, 0x40, 0x75, 0xea, 0x96, 0x7a, 0x24,
			0x4a, 0xb1, 0xdc, 0xac, 0xa6, 0xf0, 0x7b, 0x48, 0xbb, 0xb0, 0xb7, 0xe2,
			0x7d, 0x8a, 0x1a, 0x68, 0x0f, 0xa0, 0x79, 0x4c,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(107, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_108(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x68, 0xae, 0xa6, 0x3a, 0xa5, 0xd6, 0xa8, 0x3a, 0x5f, 0x71, 0xe3, 0x32,
			0x68, 0xd1, 0x62, 0xa5, 0x8d, 0x8e, 0xd1, 0x2f, 0xfb, 0x26, 0xc8, 0x25,
			0x70, 0x3e, 0xd0, 0xd1, 0x00, 0xdf, 0xd1, 0xeb,
		},
		{
			0xfa, 0x79, 0x9c, 0x67, 0x7c, 0x36, 0x1c, 0x3e, 0x3e, 0xb3, 0x21, 0xb0,
			0x73, 0x08, 0xf2, 0xfd,
		},
		{
			0xfd, 0x42, 0x73, 0xdf, 0xd5, 0xb3, 0xbc, 0x1c, 0x6a, 0x12, 0x46, 0xf5,
			0x43, 0x75, 0xe9, 0xa0, 0x02, 0x28, 0x64, 0xdc, 0x40, 0x08, 0xf1, 0x94,
			0x22, 0xec, 0x9c, 0x0f, 0xf8, 0xa9, 0x56, 0x44,
		},
		{
			0xc6, 0x73, 0x35, 0x99, 0x70, 0x4a, 0xb7, 0x20, 0x44, 0xfd, 0x28, 0x6e,
			0x3f, 0x32, 0x7b, 0x53, 0xe2, 0xf0, 0x4c, 0x33, 0xb1, 0x75, 0x44, 0xa4,
			0x2c, 0x39, 0x90, 0x59, 0x43, 0xce, 0x20, 0xe7,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x38, 0x92, 0x53, 0xf7, 0x3e, 0x2e, 0xd3, 0xf9, 0x10, 0x9e, 0x6f, 0x98,
			0xd8, 0x89, 0xb1, 0x55, 0x4f, 0x3b, 0xee, 0xdc, 0xf9, 0xd4, 0x12, 0xae,
			0xbc, 0x07, 0xba, 0x9a, 0xc8, 0x50, 0xaf, 0x07, 0x91, 0x3f, 0xe1, 0xc9,
			0x9c, 0xc8, 0x24, 0xe9, 0x4e, 0x55, 0xcb, 0xe9, 0x9e, 0xe2, 0x30, 0x66,
			0xe8, 0xe0, 0x18, 0x83, 0x90, 0xaf, 0xd8, 0x8d, 0xc3, 0xe1, 0x2d, 0x41,
			0xb5, 0xb2, 0x77, 0x62, 0xa9, 0x6b, 0x2d, 0xb3, 0x05, 0xbf, 0x7c, 0x12,
			0x06, 0x53, 0x09, 0xaa, 0xb5, 0x71, 0xad, 0xf3, 0x5b, 0x26, 0x9b, 0xfb,
			0x64, 0x79, 0x39, 0x79, 0x4c, 0x2e, 0x7d, 0x1d, 0xae, 0xf3, 0x5d, 0x85,
			0x74, 0x55, 0xde, 0xb0, 0xfe, 0x12, 0x45, 0x1c, 0x41, 0x7e, 0x95, 0x55,
			0x30, 0x11, 0x81, 0x2e, 0xad, 0x9b, 0x81, 0x38, 0xf7, 0xac, 0xeb, 0x75,
			0xa5, 0x5e, 0xdc, 0xbf, 0x96, 0x8e, 0x1f, 0x68,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(108, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_109(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xf9, 0xaa, 0xbb, 0x49, 0x60, 0x0f, 0xef, 0x8a, 0x93, 0x11, 0xc0, 0x28,
			0x80, 0xb3, 0xab, 0x2d, 0xf9, 0x0a, 0xbb, 0xe0, 0x7f, 0x51, 0x43, 0x63,
			0x2c, 0x6c, 0x35, 0xb5, 0x73, 0xff, 0xdb, 0x8d,
		},
		{
			0x7c, 0x64, 0xac, 0xa5, 0xa1, 0x21, 0x11, 0x4f, 0x6e, 0xbf, 0x5f, 0x62,
			0x9b, 0x08, 0x2b, 0x61,
		},
		{
			0xf5, 0x55, 0x69, 0xa6, 0xdf, 0xb8, 0x77, 0x69, 0xcf, 0x69, 0xb0, 0x2b,
			0x40, 0x43, 0xa0, 0xe9, 0x05, 0x5b, 0x5a, 0xec, 0x53, 0xa3, 0x1d, 0x24,
			0x59, 0x55, 0x66, 0xd5, 0x22, 0xf7, 0x6c, 0xd5,
		},
		{
			0x11, 0xdc, 0x58, 0x43, 0xc5, 0x5d, 0x03, 0x3b, 0x18, 0x01, 0x01, 0x6b,
			0xd0, 0xcf, 0x5d, 0x79, 0x9d, 0x76, 0x8c, 0xdc, 0x1e, 0xe3, 0xe2, 0x1d,
			0xa6, 0x1a, 0x02, 0x27, 0xb9, 0x83, 0x15, 0x8c,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x3b, 0x6a, 0xa6, 0x99, 0xd9, 0x6f, 0x4f, 0x1d, 0xd2, 0x9c, 0x19, 0xc5,
			0x58, 0xd0, 0x3d, 0xb3, 0x00, 0xdb, 0x1c, 0xc9, 0x2f, 0xb1, 0xa0, 0x56,
			0x52, 0x79, 0xdb, 0x1d, 0x05, 0x56, 0x67, 0xa7, 0x84, 0x49, 0x34, 0xf0,
			0x4d, 0xb0, 0x00, 0x59, 0x4c, 0xc0, 0x66, 0xb0, 0x4b, 0xe5, 0x68, 0xc0,
			0xde, 0xa8, 0xe4, 0xeb, 0xe5, 0x46, 0x7e, 0x1f, 0xf9, 0xaf, 0x2c, 0x62,
			0x72, 0x3b, 0x90, 0xd4, 0xf2, 0xc2, 0x2a, 0x6c, 0xba, 0x2a, 0xfe, 0x15,
			0xf5, 0x50, 0x45, 0x03, 0xc8, 0xe4, 0x54, 0xdb, 0xc0, 0xe8, 0x0a, 0x20,
			0x33, 0x15, 0xa8, 0xf4, 0x59, 0xf9, 0xea, 0xfb, 0x71, 0x11, 0x64, 0x53,
			0x9d, 0xef, 0xa4, 0x42, 0xb8, 0xf3, 0x52, 0x0e, 0xbe, 0xd4, 0xa5, 0xfc,
			0x4a, 0x84, 0x46, 0x10, 0x69, 0xee, 0xc4, 0x42, 0x5a, 0xa3, 0xeb, 0x88,
			0xae, 0x6d, 0x83, 0xad, 0x7a, 0x23, 0x98, 0x50,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(109, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_110(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x88, 0xa7, 0x6c, 0x16, 0xd3, 0x27, 0x0e, 0xd3, 0xfc, 0xd1, 0x76, 0xf9,
			0xd7, 0x93, 0xfa, 0x0c, 0x35, 0x51, 0x65, 0x74, 0xc1, 0xce, 0xf4, 0x25,
			0xb6, 0x00, 0x76, 0x28, 0xaf, 0xa3, 0x5e, 0x2b,
		},
		{
			0xff, 0x16, 0xcf, 0x7c, 0xb8, 0xe4, 0x9d, 0x48, 0x2c, 0xfd, 0x39, 0x94,
			0xab, 0xc5, 0xef, 0x8a,
		},
		{
			0x92, 0x13, 0xc5, 0x4e, 0x3d, 0x00, 0x2d, 0xf8, 0x74, 0x11, 0x63, 0xab,
			0x9d, 0x7e, 0x07, 0x57, 0xcd, 0x51, 0x2c, 0x69, 0x1a, 0xd6, 0x4b, 0xaf,
			0xef, 0x95, 0xcb, 0x72, 0x53, 0x9b, 0x0a, 0xc6,
		},
		{
			0x49, 0x3b, 0x64, 0x7f, 0xf0, 0xb3, 0xfa, 0xa2, 0x92, 0x1f, 0x12, 0xf8,
			0xf5, 0x7b, 0x91, 0x93, 0x29, 0xf2, 0xaf, 0x2f, 0xc1, 0xf1, 0x45, 0x76,
			0xd9, 0xdf, 0x2f, 0x8c, 0xc2, 0xad, 0xa7, 0xa6,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xf1, 0x33, 0x0a, 0x85, 0xf9, 0x00, 0x37, 0x87, 0x6b, 0x37, 0x49, 0x20,
			0x3e, 0x84, 0x92, 0x87, 0x44, 0x4a, 0x82, 0x7f, 0x0a, 0x58, 0xc2, 0x49,
			0xff, 0x86, 0x8f, 0xc1, 0xad, 0xba, 0x4d, 0xce, 0x28, 0x5e, 0x07, 0x6a,
			0x1f, 0x8a, 0xe1, 0xda, 0x8c, 0xf9, 0xfe, 0x14, 0x93, 0x1e, 0x81, 0x64,
			0x18, 0x6c, 0x97, 0xa8, 0xfe, 0xaf, 0x24, 0x58, 0x34, 0x51, 0xf1, 0x16,
			0xe6, 0x5f, 0x8e, 0x43, 0x2e, 0x7e, 0xd5, 0x5a, 0x36, 0x68, 0x31, 0x20,
			0x37, 0x7e, 0x23, 0x12, 0x8d, 0xca, 0x15, 0x40, 0xfe, 0xfb, 0xf3, 0xaf,
			0x1b, 0x56, 0xd5, 0xc7, 0x41, 0x87, 0xf5, 0x28, 0x6d, 0x0a, 0x95, 0xfb,
			0x55, 0x93, 0x17, 0x70, 0x54, 0x30, 0x60, 0xce, 0x8d, 0xf0, 0x8f, 0x3c,
			0x19, 0x59, 0xa1, 0xf4, 0xfc, 0x36, 0xb6, 0x46, 0x71, 0xe0, 0x65, 0x4f,
			0xff, 0xe7, 0x0d, 0x96, 0xd5, 0x21, 0xbe, 0x21,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(110, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_111(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xf8, 0xf2, 0xc3, 0x19, 0x4e, 0x8c, 0x4e, 0x90, 0x9e, 0x6f, 0x05, 0x27,
			0x95, 0x03, 0x07, 0x18, 0xcd, 0xd6, 0x23, 0x53, 0x03, 0xbd, 0xe5, 0x37,
			0xea, 0x3c, 0x70, 0x53, 0x05, 0x62, 0x26, 0xe1,
		},
		{
			0xd5, 0x0e, 0xf4, 0xd0, 0x6d, 0x00, 0x74, 0x54, 0x65, 0xce, 0x7b, 0x08,
			0xcd, 0x03, 0x51, 0x76,
		},
		{
			0xed, 0x89, 0x7e, 0xc3, 0x3a, 0xe3, 0xf4, 0xb0, 0xc2, 0x0b, 0x12, 0x8d,
			0x94, 0x61, 0x5f, 0xe3, 0x87, 0x27, 0x0f, 0x8f, 0x0d, 0x66, 0x5e, 0x11,
			0x9b, 0x9b, 0x5a, 0xd1, 0x0f, 0xd3, 0x65, 0x7a,
		},
		{
			0x91, 0x78, 0x49, 0x2a, 0x92, 0x51, 0x12, 0x80, 0x6b, 0x27, 0x80, 0x2e,
			0x0c, 0x9b, 0x0e, 0x70, 0xaf, 0x3f, 0xd5, 0xb6, 0xdb, 0x9a, 0xf8, 0x70,
			0x40, 0x28, 0x02, 0x56, 0x35, 0xc5, 0xed, 0x27,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x0e, 0x50, 0xd9, 0xdb, 0x9a, 0x5e, 0xb2, 0xd0, 0xb0, 0xd5, 0xc4, 0x53,
			0x01, 0x70, 0x7e, 0xe2, 0x81, 0x86, 0x68, 0x79, 0xe0, 0x72, 0xe5, 0x16,
			0x96, 0xe3, 0x28, 0x19, 0xb3, 0x3a, 0x44, 0xf0, 0xd0, 0x0b, 0x88, 0xab,
			0x5d, 0x6c, 0x31, 0xd9, 0x19, 0x88, 0xb1, 0x75, 0x7c, 0xee, 0xa8, 0x93,
			0x6e, 0xd5, 0xd0, 0xad, 0x67, 0x29, 0x90, 0xf4, 0x95, 0x1d, 0xfa, 0xea,
			0x99, 0x4d, 0xc0, 0x0f, 0x32, 0x65, 0x09, 0x5b, 0xdb, 0xef, 0xaf, 0x1d,
			0x42, 0x24, 0xda, 0x61, 0x68, 0x88, 0xa8, 0x06, 0x60, 0x9d, 0x28, 0x05,
			0xcf, 0x0f, 0x51, 0x83, 0xd5, 0xe9, 0x25, 0x26, 0x1e, 0x6b, 0x24, 0x90,
			0xdd, 0x85, 0x24, 0x2f, 0xc9, 0xfb, 0xd4, 0x26, 0xd7, 0x7a, 0xe8, 0x33,
			0x9a, 0xea, 0xc1, 0x83, 0x94, 0xbb, 0xf3, 0x51, 0xe0, 0x61, 0xe4, 0x95,
			0x61, 0x38, 0x36, 0x1d, 0x11, 0xb3, 0x57, 0x72,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(111, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_112(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x2a, 0x65, 0xb1, 0x29, 0x9d, 0x03, 0x4b, 0x95, 0x40, 0x07, 0xae, 0x77,
			0x34, 0x03, 0xf5, 0x09, 0xec, 0x8d, 0xf5, 0x2b, 0x25, 0xff, 0x57, 0x79,
			0x19, 0xd3, 0x3e, 0x97, 0x6f, 0xfd, 0x21, 0xa5,
		},
		{
			0x80, 0x80, 0xab, 0xa8, 0xc9, 0xf7, 0xa0, 0x4a, 0xd4, 0x1c, 0x41, 0xde,
			0xbc, 0x8d, 0x6b, 0x02,
		},
		{
			0x31, 0xa4, 0x66, 0x17, 0xe8, 0xee, 0xc9, 0x15, 0x70, 0x7b, 0x75, 0xf7,
			0xba, 0x78, 0x81, 0xf9, 0x63, 0x4e, 0x7b, 0x01, 0x5e, 0xf2, 0xa2, 0xfb,
			0x51, 0xf5, 0x92, 0xfd, 0xda, 0x27, 0xfb, 0x20,
		},
		{
			0xe3, 0xc8, 0x81, 0x51, 0x72, 0xec, 0xee, 0x13, 0xe6, 0x49, 0xff, 0x62,
			0x4c, 0x62, 0x43, 0xaf, 0xbb, 0x46, 0x60, 0x2b, 0xbf, 0xc3, 0xbd, 0x28,
			0xd0, 0x31, 0x7b, 0xad, 0x68, 0x56, 0xc6, 0x6f,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x5b, 0x1c, 0x01, 0xab, 0x1d, 0xf4, 0x42, 0x30, 0x5d, 0xaa, 0x89, 0x98,
			0x72, 0x8e, 0x1e, 0x99, 0x97, 0x5b, 0x32, 0x32, 0x7d, 0xa7, 0x27, 0xf1,
			0x62, 0xaf, 0xf5, 0xec, 0xbb, 0x01, 0xd4, 0xd6, 0xe8, 0xb8, 0x55, 0x1f,
			0x7a, 0xbe, 0x18, 0x92, 0x62, 0x50, 0x12, 0xc2, 0x48, 0x17, 0x8b, 0x2e,
			0xc1, 0x2d, 0x64, 0xd8, 0x40, 0x63, 0xf8, 0x7f, 0x12, 0x79, 0x28, 0xeb,
			0xab, 0x06, 0xcf, 0xc3, 0x77, 0xfd, 0x96, 0x55, 0xaa, 0xdb, 0xa3, 0x61,
			0xb9, 0x3f, 0x97, 0x6e, 0x51, 0x33, 0xb6, 0xfb, 0x87, 0x45, 0xf1, 0x30,
			0xd8, 0xc5, 0xea, 0xe6, 0xf0, 0xd3, 0x85, 0x7f, 0xe5, 0xc6, 0x89, 0x04,
			0xad, 0xea, 0xa2, 0x24, 0xcc, 0xb9, 0x3a, 0x96, 0x0e, 0xd7, 0x9e, 0xcf,
			0x55, 0xd6, 0xaf, 0x30, 0x7d, 0x28, 0x73, 0x66, 0xb8, 0xa2, 0xd4, 0x66,
			0xb9, 0x5e, 0xdb, 0xa1, 0x42, 0xe8, 0x6d, 0x12,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(112, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_113(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x85, 0xb6, 0x0f, 0xaa, 0x4e, 0xbf, 0xde, 0x82, 0x5f, 0x0a, 0x49, 0x70,
			0xc8, 0x0c, 0x78, 0xe3, 0x77, 0xa5, 0x1d, 0xce, 0xbb, 0x09, 0x45, 0x3a,
			0x03, 0xb6, 0x46, 0xcb, 0xdc, 0x1c, 0x2f, 0x0e,
		},
		{
			0xfb, 0x4d, 0xff, 0x7e, 0x1a, 0xd3, 0x7f, 0x1d, 0x00, 0xa1, 0x00, 0xde,
			0x0a, 0x9d, 0x5d, 0x2b,
		},
		{
			0x46, 0x10, 0xc5, 0x65, 0x63, 0x72, 0x6c, 0x7b, 0xaa, 0x3a, 0x21, 0x12,
			0x90, 0x4a, 0xd8, 0x03, 0x62, 0x55, 0xc9, 0x2f, 0xe7, 0x48, 0xd6, 0x5f,
			0x12, 0x33, 0x35, 0x48, 0x96, 0xfe, 0xe7, 0x2a,
		},
		{
			0x85, 0x40, 0x56, 0x92, 0x7b, 0x35, 0x05, 0x3b, 0x57, 0x44, 0x8a, 0x2e,
			0x0b, 0x6c, 0x7c, 0xf3, 0x71, 0xce, 0x10, 0x79, 0x7e, 0xab, 0x1c, 0x0a,
			0x00, 0x9d, 0x7a, 0xbd, 0x1a, 0x06, 0x7c, 0xcc,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x19, 0x42, 0xa4, 0xdc, 0x08, 0xfe, 0x8d, 0x92, 0xf5, 0xfa, 0x58, 0x17,
			0x56, 0xdd, 0xc3, 0xf7, 0xa3, 0x21, 0xcb, 0x96, 0x93, 0x3c, 0xc1, 0x93,
			0x71, 0x1a, 0x0f, 0x64, 0x40, 0xa7, 0x6b, 0x6c, 0x5a, 0x5d, 0xb8, 0x06,
			0x9c, 0x5c, 0x82, 0xfe, 0x3e, 0x1c, 0xdf, 0x3f, 0x35, 0x20, 0xde, 0xb1,
			0x65, 0x2a, 0x96, 0x98, 0xf9, 0x72, 0x6f, 0xd4, 0x04, 0x8b, 0xab, 0xaf,
			0x07, 0x05, 0x0e, 0x93, 0x7d, 0x56, 0x54, 0x28, 0x99, 0xee, 0xa5, 0xec,
			0x2c, 0x16, 0x91, 0x9a, 0x36, 0x0d, 0xe5, 0xcc, 0xf4, 0x9e, 0xee, 0xd1,
			0x15, 0xee, 0x02, 0xc0, 0x67, 0xb7, 0x25, 0x52, 0xbc, 0x4d, 0x4e, 0xef,
			0xb6, 0xaa, 0x91, 0xa2, 0x33, 0xdd, 0x73, 0x17, 0x95, 0x9a, 0x7e, 0x80,
			0xe1, 0x6b, 0x17, 0x37, 0x91, 0x84, 0x50, 0x3c, 0xbf, 0x39, 0x31, 0xa6,
			0x9e, 0x90, 0x2d, 0x97, 0xf8, 0x20, 0xf8, 0x03,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(113, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_114(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xbb, 0x71, 0x47, 0xf6, 0x74, 0x54, 0x51, 0x7b, 0x2f, 0x2b, 0x10, 0x37,
			0x90, 0xb4, 0x42, 0x47, 0xef, 0xb5, 0x28, 0xa7, 0x7a, 0x05, 0xaa, 0x22,
			0x0c, 0x8f, 0xe4, 0x0b, 0x32, 0x3a, 0x1f, 0x0a,
		},
		{
			0x33, 0xb4, 0x2b, 0x70, 0xeb, 0x65, 0xd9, 0x85, 0xf2, 0x47, 0x61, 0x36,
			0xc9, 0x49, 0xca, 0x18,
		},
		{
			0x7e, 0xaa, 0x52, 0xf6, 0xd4, 0x35, 0x29, 0x8f, 0xda, 0x70, 0xef, 0x9a,
			0xba, 0xe9, 0x0e, 0xb5, 0x49, 0x68, 0xbd, 0x2b, 0x4c, 0xe7, 0x49, 0x83,
			0x6c, 0xb2, 0xb1, 0x1e, 0x45, 0x2f, 0x67, 0x68,
		},
		{
			0x4c, 0x26, 0x86, 0xe2, 0xed, 0xc7, 0xa2, 0x5b, 0xc9, 0x27, 0x46, 0x42,
			0xb8, 0x86, 0xf1, 0x51, 0x0c, 0x4d, 0x0f, 0x00, 0x71, 0x5b, 0x75, 0x90,
			0x1a, 0x40, 0xf3, 0xd1, 0x53, 0x77, 0x88, 0xfc,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x50, 0xdc, 0x60, 0xbf, 0x8d, 0xf4, 0xb7, 0x3e, 0x0c, 0x95, 0xb3, 0xe1,
			0x73, 0x64, 0x11, 0x1d, 0xde, 0xff, 0x81, 0x44, 0x24, 0x60, 0xd6, 0xc9,
			0x35, 0xe1, 0xfc, 0x5d, 0x2e, 0xd2, 0x95, 0xc2, 0x54, 0xc5, 0xc9, 0xf3,
			0x9d, 0x18, 0xf6, 0x54, 0x0e, 0x69, 0x7a, 0x1a, 0x7f, 0xc5, 0x88, 0xb3,
			0x7e, 0x8f, 0xc4, 0x5f, 0x31, 0x4c, 0x06, 0xe9, 0xeb, 0x04, 0x45, 0x8f,
			0x3b, 0xd1, 0x8d, 0x23, 0xfb, 0x33, 0xdb, 0x2a, 0x31, 0x85, 0xa0, 0xcb,
			0x2b, 0x2d, 0x9b, 0xa6, 0x09, 0x15, 0x38, 0xe8, 0x5c, 0x2d, 0x6f, 0xba,
			0xc0, 0x20, 0x1f, 0xec, 0xd3, 0x37, 0xd2, 0x68, 0x34, 0x7a, 0xd8, 0xf3,
			0x04, 0x21, 0x3c, 0x3a, 0xf9, 0x45, 0xca, 0x31, 0xed, 0xf8, 0xbe, 0x84,
			0x7a, 0x7b, 0x56, 0xd7, 0xbd, 0xcd, 0x30, 0x83, 0x99, 0xb0, 0x65, 0x02,
			0x79, 0x44, 0x43, 0xf7, 0x38, 0xf6, 0xee, 0x55,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(114, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_115(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xd7, 0xdc, 0xdf, 0xaa, 0x81, 0xc3, 0x51, 0x19, 0xa4, 0x69, 0x51, 0xae,
			0xd8, 0xd8, 0x89, 0x95, 0x02, 0xfc, 0x21, 0xb9, 0x09, 0xc8, 0xde, 0x8c,
			0xf3, 0xa5, 0x67, 0xc1, 0xae, 0x81, 0x16, 0x1d,
		},
		{
			0x4c, 0x24, 0x49, 0x56, 0x7c, 0x08, 0x67, 0xc9, 0xf4, 0x3b, 0x75, 0x28,
			0x60, 0xe3, 0x4b, 0xde,
		},
		{
			0x09, 0x41, 0x7a, 0x19, 0x12, 0xef, 0xc0, 0x14, 0x65, 0xa8, 0x68, 0x2f,
			0xf7, 0x13, 0x78, 0x70, 0x65, 0x24, 0xb4, 0x6a, 0x0c, 0x27, 0x22, 0x0e,
			0x9e, 0x40, 0xec, 0x43, 0x07, 0x3c, 0x56, 0x3c,
		},
		{
			0x54, 0x8a, 0x97, 0xb1, 0x12, 0xab, 0x6c, 0xcb, 0x57, 0x8d, 0xd7, 0x75,
			0x99, 0xdf, 0x0c, 0xac, 0xde, 0x33, 0x2a, 0xc1, 0xa6, 0x00, 0x88, 0xdc,
			0x01, 0x8d, 0x18, 0x38, 0x75, 0xb6, 0x1c, 0x51,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb2, 0x00, 0xd6, 0x98, 0xfc, 0x72, 0xf0, 0x1e, 0x97, 0x09, 0xe5, 0x3a,
			0x0c, 0xa4, 0x82, 0xed, 0x3b, 0xe1, 0x0e, 0x27, 0x72, 0x96, 0x42, 0x4c,
			0xef, 0xd1, 0x10, 0x72, 0xc7, 0xb1, 0x9b, 0xdd, 0xf7, 0x0f, 0xf1, 0x37,
			0xc9, 0x56, 0xe4, 0xd1, 0x1d, 0x8e, 0xb1, 0xeb, 0x29, 0xbb, 0xe4, 0x97,
			0x9c, 0xba, 0x9e, 0x2e, 0xb9, 0x41, 0xf5, 0x29, 0xea, 0x96, 0x84, 0xce,
			0xc6, 0x61, 0x4a, 0x5f, 0x8c, 0x30, 0x95, 0x0d, 0x95, 0x44, 0x14, 0x8f,
			0xc4, 0xc9, 0x2b, 0xb8, 0xb7, 0xba, 0x89, 0x9e, 0x72, 0x17, 0x48, 0xa1,
			0xd6, 0xbf, 0x5e, 0x8b, 0xb6, 0x42, 0x9b, 0xa9, 0x02, 0x0e, 0x08, 0x9a,
			0x7c, 0x1e, 0x91, 0x0c, 0x8f, 0xc7, 0x08, 0xe0, 0x2b, 0xe7, 0x5e, 0xfb,
			0x89, 0xf0, 0x0d, 0x6f, 0x8b, 0x52, 0x05, 0x40, 0x62, 0xe4, 0x33, 0x43,
			0xa6, 0x0d, 0xee, 0xd3, 0x9c, 0xd8, 0x75, 0x6a,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(115, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_116(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x79, 0xad, 0xc5, 0xce, 0xae, 0xf7, 0x71, 0x10, 0x24, 0x01, 0xac, 0xc2,
			0xfc, 0xb3, 0xd0, 0x1d, 0x9b, 0xf7, 0x54, 0x1e, 0xd8, 0x6e, 0x8a, 0xa1,
			0x6c, 0x93, 0x8d, 0x95, 0x56, 0x82, 0x9a, 0x8e,
		},
		{
			0x37, 0x9d, 0x9a, 0xeb, 0xe0, 0xbd, 0xbc, 0xa0, 0x7c, 0x77, 0x4b, 0x5e,
			0xf1, 0xee, 0x73, 0x2d,
		},
		{
			0x8b, 0x13, 0x32, 0x6a, 0x52, 0x19, 0x57, 0xed, 0xe0, 0x04, 0x23, 0x5c,
			0xd1, 0x38, 0x76, 0x7b, 0x7f, 0x45, 0xf5, 0x50, 0xf2, 0xfc, 0x55, 0xb1,
			0xf6, 0x7c, 0xdc, 0xc7, 0x09, 0x23, 0x81, 0x1d,
		},
		{
			0xe2, 0x68, 0xce, 0xa9, 0xdd, 0xeb, 0x0c, 0xca, 0x53, 0x88, 0xe1, 0x24,
			0x17, 0xd9, 0x0c, 0x7f, 0xab, 0x62, 0x9b, 0x2b, 0x2c, 0x14, 0x4d, 0xdc,
			0x80, 0x54, 0xd0, 0x42, 0x10, 0x58, 0x67, 0x14,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0x00, 0x8f, 0xed, 0xe4, 0x41, 0xe5, 0x50, 0x95, 0x5d, 0x75, 0x9c, 0x89,
			0x22, 0x93, 0x59, 0xf6, 0xa0, 0x6e, 0xc2, 0xc0, 0x6c, 0xe2, 0xdc, 0xf4,
			0x2b, 0xb7, 0x2f, 0x25, 0x5c, 0x28, 0xed, 0x75, 0x8b, 0x38, 0xa5, 0xf7,
			0xf7, 0xea, 0x94, 0x51, 0xa1, 0xc9, 0xb7, 0xa6, 0x88, 0xd3, 0x79, 0xc9,
			0xd3, 0x6f, 0xa7, 0xc8, 0x6e, 0xb1, 0x1b, 0x67, 0xf4, 0xe3, 0xb9, 0x83,
			0xfe, 0x4b, 0x42, 0x15, 0x9f, 0xa6, 0x23, 0xf5, 0x5e, 0xea, 0x9f, 0xb1,
			0xff, 0x36, 0x76, 0xad, 0x54, 0x7d, 0xf4, 0x69, 0x60, 0x7a, 0x03, 0xe8,
			0xf8, 0x32, 0x26, 0x79, 0x68, 0xc8, 0x38, 0x8b, 0x2d, 0xb8, 0x49, 0x16,
			0x6a, 0x98, 0xa9, 0x8a, 0xef, 0x8d, 0x9e, 0x25, 0x88, 0x8b, 0x1f, 0x6a,
			0xec, 0x3d, 0x66, 0xa0, 0x26, 0x35, 0x32, 0x59, 0xbd, 0xa9, 0x02, 0xdf,
			0x45, 0x5e, 0x6d, 0x2d, 0x52, 0x2c, 0xde, 0xd4,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(116, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_117(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x37, 0x6d, 0xfb, 0x59, 0x42, 0x41, 0x45, 0x42, 0x0a, 0xc6, 0x59, 0x67,
			0x14, 0x5f, 0x6f, 0xa5, 0x1a, 0xac, 0xec, 0x27, 0xbe, 0xa1, 0xd3, 0xf9,
			0xf8, 0xb7, 0x61, 0x6f, 0xcb, 0x36, 0xf7, 0x3b,
		},
		{
			0x56, 0x4c, 0x1c, 0xb8, 0x2b, 0xef, 0x2b, 0xdc, 0x2c, 0x69, 0x99, 0x55,
			0xbd, 0x2b, 0x07, 0x77,
		},
		{
			0xa0, 0x83, 0x9d, 0xaa, 0xac, 0xe6, 0xff, 0x20, 0x9f, 0xba, 0x71, 0xc6,
			0xbb, 0xba, 0x05, 0x84, 0x49, 0x7c, 0xff, 0x77, 0x5d, 0x86, 0x50, 0x72,
			0x28, 0x71, 0x7b, 0x57, 0x6f, 0x3d, 0x5e, 0xd2,
		},
		{
			0x23, 0x0d, 0xa7, 0x2e, 0x74, 0x36, 0x7d, 0x92, 0x10, 0x77, 0xb7, 0xc3,
			0x8c, 0x11, 0xeb, 0x9e, 0x17, 0x13, 0x1f, 0x7e, 0xe4, 0xff, 0x58, 0x0e,
			0xd8, 0x83, 0x53, 0x0d, 0xac, 0xf3, 0x41, 0x99,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xa7, 0xfe, 0xff, 0x8c, 0x68, 0x59, 0xff, 0x3d, 0x69, 0x54, 0x3b, 0xe6,
			0x16, 0x57, 0xc8, 0xc1, 0x9b, 0xe1, 0x2b, 0x9f, 0x2b, 0x53, 0x48, 0xb9,
			0x4f, 0x23, 0x70, 0x04, 0x43, 0x7a, 0x20, 0xc6, 0xbe, 0x86, 0x19, 0xd5,
			0x31, 0x00, 0x5f, 0xc4, 0x7f, 0xcd, 0x6a, 0x1f, 0x07, 0x38, 0x0f, 0xca,
			0x90, 0x75, 0xfb, 0x11, 0x48, 0xf5, 0xe3, 0xe1, 0x57, 0x16, 0x99, 0x8c,
			0x12, 0x68, 0xba, 0xc9, 0x72, 0xf9, 0x4e, 0x1b, 0x3e, 0xdb, 0x89, 0xe2,
			0x86, 0xa7, 0xb3, 0x6a, 0x53, 0xc0, 0x21, 0xfc, 0xa6, 0x04, 0x53, 0xd7,
			0xfb, 0x03, 0xa4, 0xeb, 0x00, 0x96, 0x0d, 0xe3, 0x1c, 0x45, 0x8b, 0x37,
			0x79, 0x86, 0xaf, 0x33, 0xfc, 0xfc, 0xef, 0xa7, 0xf6, 0xc9, 0x38, 0xc7,
			0xfc, 0x61, 0x83, 0x37, 0xd4, 0xec, 0x8b, 0xd9, 0x42, 0xb2, 0x3a, 0xb0,
			0x1b, 0xb3, 0xb3, 0xde, 0x27, 0x0c, 0xc9, 0xa9,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(117, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_118(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0xfd, 0xbe, 0xe9, 0xba, 0xff, 0x0d, 0x02, 0x17, 0x18, 0xbc, 0x07, 0x32,
			0xce, 0x5b, 0x2e, 0x0b, 0xba, 0x13, 0x3b, 0xd3, 0x19, 0xb4, 0xc5, 0x37,
			0x5c, 0xbe, 0x32, 0xda, 0x82, 0x74, 0xef, 0x72,
		},
		{
			0x68, 0x48, 0xa7, 0x8d, 0xac, 0x81, 0x74, 0x3c, 0x0a, 0xc4, 0xc1, 0xd7,
			0xa0, 0x8b, 0x21, 0xf3,
		},
		{
			0xeb, 0xc1, 0xf9, 0x76, 0x83, 0x42, 0x63, 0x2f, 0x57, 0x6f, 0x7a, 0xf5,
			0x03, 0x69, 0x95, 0x98, 0xd1, 0x3b, 0xba, 0x5f, 0xcc, 0x8f, 0xfb, 0xe2,
			0x94, 0xbd, 0xd1, 0x6b, 0x8c, 0x86, 0x5f, 0xad,
		},
		{
			0xcc, 0x39, 0x27, 0x64, 0x5f, 0x65, 0xf5, 0x46, 0x60, 0xc1, 0x69, 0xfd,
			0x33, 0x94, 0xa3, 0x6a, 0x98, 0x9c, 0x76, 0x0a, 0x28, 0x7f, 0x23, 0xee,
			0x8b, 0xe2, 0x3d, 0x0b, 0xcd, 0xb2, 0xbf, 0x6c,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xb4, 0x58, 0xae, 0xd8, 0x8a, 0x53, 0xcb, 0x5a, 0x1a, 0x15, 0x71, 0x97,
			0xb3, 0x80, 0x79, 0xcd, 0xa4, 0x55, 0xbe, 0xdb, 0xc0, 0xe6, 0x9f, 0x7c,
			0x66, 0xc1, 0xca, 0x67, 0xad, 0xec, 0x05, 0xa4, 0x11, 0x71, 0x53, 0x09,
			0x58, 0x2c, 0x7a, 0x2d, 0x57, 0x09, 0xa4, 0xd6, 0xf2, 0x7c, 0x70, 0x68,
			0x66, 0x27, 0xad, 0xec, 0xc6, 0x8f, 0x3a, 0x20, 0x45, 0xb8, 0xe7, 0xce,
			0xd5, 0xb9, 0x2b, 0x5d, 0xfa, 0x4a, 0x36, 0xe5, 0x55, 0xfd, 0x42, 0x12,
			0xf1, 0xe0, 0x33, 0xd9, 0x4a, 0x52, 0x3b, 0x16, 0xf1, 0xa4, 0xfb, 0x15,
			0xb6, 0x95, 0xd5, 0x8d, 0xf4, 0xa5, 0xbe, 0x8e, 0xe7, 0xa2, 0xb9, 0xcb,
			0xa0, 0xaf, 0x05, 0xf2, 0x94, 0xcf, 0x0d, 0xaf, 0x54, 0xca, 0xba, 0x5b,
			0xfc, 0xd2, 0x89, 0xc6, 0xf7, 0xd9, 0x78, 0xef, 0xb4, 0x12, 0x63, 0xc1,
			0xba, 0x0d, 0xeb, 0xb0, 0x6b, 0x56, 0xf3, 0xdc,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(118, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_119(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x0d, 0x10, 0x15, 0xbc, 0xe0, 0xf7, 0xba, 0xc3, 0xc0, 0x74, 0x20, 0xb4,
			0xf3, 0xf9, 0xbe, 0xca, 0x05, 0x64, 0x87, 0x89, 0x7a, 0xb2, 0x6e, 0x8a,
			0xd5, 0xc3, 0x39, 0x03, 0x6c, 0x87, 0xa7, 0xce,
		},
		{
			0xad, 0x72, 0x17, 0x94, 0x44, 0x30, 0x6c, 0x67, 0x7b, 0xb2, 0x70, 0x9d,
			0x1d, 0x46, 0x4e, 0xfc,
		},
		{
			0xe8, 0xce, 0x89, 0xf4, 0xb7, 0xe0, 0xcb, 0xb3, 0xa2, 0xfb, 0x74, 0x79,
			0xd9, 0x5e, 0x80, 0xb3, 0x6e, 0xae, 0x5c, 0x62, 0xd7, 0x23, 0x2f, 0x26,
			0xc5, 0xa9, 0x10, 0x35, 0x99, 0xa4, 0x9b, 0xdd,
		},
		{
			0x15, 0x58, 0x98, 0x57, 0x01, 0x04, 0x68, 0xca, 0x37, 0xcb, 0x83, 0xad,
			0xd5, 0x9c, 0xfc, 0x73, 0x3f, 0x6e, 0x97, 0xf9, 0x53, 0xe2, 0xf8, 0xb8,
			0xb9, 0x8f, 0xab, 0xb8, 0x1f, 0x60, 0x8e, 0x47,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xcd, 0xb7, 0xa1, 0x15, 0x16, 0x1c, 0x33, 0x2a, 0x97, 0xca, 0xee, 0xa6,
			0xeb, 0x7a, 0x6b, 0xfd, 0x29, 0x9a, 0x76, 0x8a, 0x8f, 0x78, 0x27, 0xdf,
			0x16, 0x30, 0x48, 0xf4, 0xdb, 0xf1, 0x75, 0x0e, 0xe1, 0xe8, 0x4d, 0x52,
			0x1b, 0x80, 0x01, 0xad, 0x7d, 0xbf, 0x97, 0xa8, 0x67, 0xfb, 0x2b, 0x99,
			0xd2, 0x1f, 0x99, 0x52, 0x4b, 0x84, 0x11, 0x36, 0x36, 0x36, 0xe2, 0xd4,
			0x40, 0xe9, 0x4f, 0x26, 0xf2, 0xdd, 0x93, 0x51, 0x6e, 0x08, 0x96, 0x4d,
			0x29, 0x1f, 0x49, 0x96, 0x1a, 0xd2, 0xf3, 0xfa, 0x21, 0xf6, 0xac, 0x7a,
			0x00, 0xda, 0x08, 0xdb, 0xd8, 0x3c, 0x52, 0x02, 0x31, 0xa9, 0x7e, 0x48,
			0x29, 0xf6, 0x76, 0xfb, 0xc1, 0x87, 0x17, 0xab, 0xce, 0xdc, 0x4d, 0xee,
			0x2a, 0x41, 0x44, 0x0d, 0x57, 0x97, 0xab, 0x01, 0x45, 0x37, 0xa4, 0x2f,
			0x59, 0x53, 0xa9, 0x59, 0x53, 0x5b, 0x3c, 0x81,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(119, &vec);
	TC_END_RESULT(result);
	return result;
}

unsigned int test_120(void)
{
	unsigned int result = TC_PASS;
	const struct hmac_prng_test_vector vec = {
		32,
		16,
		32,
		0,
		128,
		{
			0x92, 0x84, 0xfd, 0x46, 0x60, 0xbe, 0xb7, 0xa1, 0x54, 0x34, 0x9d, 0x74,
			0xe2, 0xad, 0x84, 0x42, 0xa5, 0xac, 0x06, 0x67, 0xac, 0x4d, 0x28, 0xda,
			0xf0, 0x52, 0x9d, 0xac, 0x5c, 0x80, 0x5a, 0x21,
		},
		{
			0xaf, 0xe9, 0x6e, 0x33, 0xc6, 0xd5, 0xbe, 0xf1, 0x65, 0x28, 0x21, 0xde,
			0xdb, 0x96, 0x41, 0x83,
		},
		{
			0x1b, 0x34, 0x19, 0xbe, 0xc2, 0x51, 0x35, 0x31, 0xfa, 0x5f, 0xaf, 0xb4,
			0x27, 0x85, 0xeb, 0x07, 0xec, 0x01, 0xb6, 0x1a, 0x1e, 0x06, 0x45, 0x28,
			0x08, 0xb2, 0x27, 0xf5, 0xf5, 0xc8, 0x73, 0xcf,
		},
		{
			0xf4, 0x94, 0x2b, 0x20, 0xa4, 0x25, 0x89, 0xa2, 0x2e, 0x1b, 0x3b, 0x22,
			0x14, 0x33, 0x6d, 0x1b, 0xa3, 0xb1, 0x17, 0x53, 0x93, 0x7b, 0x93, 0xf1,
			0xd7, 0xc3, 0x63, 0x6b, 0x34, 0x6d, 0x70, 0xa1,
		},
		{
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		{
			0xf0, 0xb8, 0x7f, 0x8c, 0x3c, 0x67, 0xf4, 0xdf, 0x3c, 0xc6, 0x13, 0xc3,
			0xc6, 0x79, 0x99, 0x67, 0x59, 0x51, 0x17, 0x7c, 0x2d, 0x7d, 0x64, 0x6e,
			0xc0, 0x3e, 0x52, 0x22, 0x67, 0xea, 0x1d, 0x39, 0xd4, 0x20, 0x60, 0xd8,
			0xb1, 0x57, 0x77, 0x4a, 0xf7, 0xad, 0x7f, 0xd7, 0xf8, 0x51, 0xf9, 0xc1,
			0x3a, 0xaf, 0xa5, 0x6a, 0x64, 0xd7, 0xc0, 0xff, 0x86, 0xc8, 0xdd, 0x95,
			0x88, 0xf6, 0x9c, 0x6e, 0x7a, 0x96, 0x80, 0x75, 0x9b, 0xc1, 0xb6, 0x02,
			0x79, 0x8b, 0x23, 0x95, 0x9f, 0xe5, 0x4f, 0xf4, 0xbd, 0x2c, 0xe2, 0xd8,
			0xed, 0xe7, 0x29, 0x8f, 0x68, 0x80, 0x1a, 0xad, 0x61, 0x97, 0x74, 0x85,
			0xe0, 0xac, 0xba, 0xde, 0xec, 0xf4, 0x7b, 0x7e, 0xb3, 0x0e, 0x92, 0xf4,
			0x06, 0xd9, 0x73, 0xe3, 0x26, 0x51, 0x8d, 0xb3, 0xaf, 0x0e, 0xa0, 0x6e,
			0xa5, 0x74, 0xba, 0x7b, 0x01, 0x61, 0xab, 0xf2,
		}
	};

	TC_PRINT("HMAC-PRNG %s:\n", __func__);
	result = do_hmac_prng_pr_false_test(120, &vec);
	TC_END_RESULT(result);
	return result;
}


/*
 * Main task to test HMAC-PRNG
 */
int main(void)
{
	unsigned int result = TC_PASS;

	TC_START("Performing HMAC-PRNG tests:");

	result = test_1();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_2();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_3();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_4();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_5();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_6();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_7();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_8();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_9();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_10();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_11();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_12();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_13();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_14();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_15();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_16();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_17();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_18();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_19();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_20();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_21();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_22();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_23();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_24();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_25();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_26();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_27();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_28();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_29();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_30();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_31();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_32();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_33();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_34();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_35();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_36();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_37();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_38();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_39();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_40();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_41();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_42();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_43();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_44();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_45();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_46();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_47();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_48();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_49();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_50();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_51();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_52();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_53();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_54();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_55();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_56();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_57();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_58();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_59();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_60();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_61();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_62();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_63();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_64();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_65();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_66();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_67();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_68();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_69();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_70();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_71();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_72();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_73();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_74();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_75();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_76();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_77();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_78();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_79();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_80();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_81();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_82();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_83();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_84();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_85();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_86();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_87();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_88();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_89();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_90();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_91();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_92();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_93();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_94();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_95();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_96();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_97();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_98();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_99();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_100();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_101();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_102();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_103();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_104();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_105();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_106();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_107();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_108();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_109();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_110();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_111();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_112();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_113();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_114();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_115();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_116();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_117();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_118();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_119();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	result = test_120();
	if (result == TC_FAIL) {
		/* terminate test */
		TC_ERROR("HMAC test #{test_number} failed.\n");
		goto exitTest;
	}

	TC_PRINT("All HMAC-PRNG tests succeeded!\n");

exitTest:
	TC_END_RESULT(result);
	TC_END_REPORT(result);
}
