﻿//  Copyright 2020 Google Inc. All Rights Reserved.
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at
//
//  http://www.apache.org/licenses/LICENSE-2.0
//
//  Unless required by applicable law or agreed to in writing, software
//  distributed under the License is distributed on an "AS IS" BASIS,
//  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//  See the License for the specific language governing permissions and
//  limitations under the License.

using System;
using System.Collections.Generic;
using System.Linq;

namespace NtApiDotNet
{
    /// <summary>
    /// Get dictionary of well known WNF state names.
    /// </summary>
    /// <remarks>This was dumped from perf_nt_c.dll 10.0.18362.1 using https://github.com/ionescu007/wnfun.</remarks>
    internal static class NtWnfWellKnownNames
    {
        #region Private Members
        private static Lazy<Dictionary<ulong, string>> _state_name_to_name = new Lazy<Dictionary<ulong, string>>(GetStateNameToName);
        private static Lazy<Dictionary<string, ulong>> _name_to_state_name = new Lazy<Dictionary<string, ulong>>(GetNameToStateName);

        private static Dictionary<string, ulong> GetNameToStateName()
        {
            return _state_name_to_name.Value.ToDictionary(p => p.Value, p => p.Key, StringComparer.OrdinalIgnoreCase);
        }

        private static Dictionary<ulong, string> GetStateNameToName()
        {
            return new Dictionary<ulong, string>()
            {
            {0x41877c2ca3bc0875UL, "WNF_A2A_APPURIHANDLER_INSTALLED"},
            {0x41820f2ca3bc1075UL, "WNF_AAD_DEVICE_ATTRIBUTES_UPDATE"},
            {0x41820f2ca3bc0875UL, "WNF_AAD_DEVICE_REGISTRATION_STATUS_CHANGE"},
            {0x41c60f2ca3bc1075UL, "WNF_AA_CURATED_TILE_COLLECTION_STATUS"},
            {0x41c60f2ca3bc0875UL, "WNF_AA_LOCKDOWN_CHANGED"},
            {0x41c60f2ca3bc1875UL, "WNF_AA_MDM_STATUS_EVENT_LOGGED"},
            {0x41850d2ca3bc0835UL, "WNF_ACC_EC_ENABLED"},
            {0xa8e0d2ca3bc0875UL, "WNF_ACHK_SP_CORRUPTION_DETECTED"},
            {0x41920d2ca3bc0835UL, "WNF_ACT_DATA_UPDATED"},
            {0x4182082ca3bc0875UL, "WNF_AFD_IGNORE_ORDERLY_RELEASE_CHANGE"},
            {0x41c6072ca3bc1075UL, "WNF_AI_PACKAGEINSTALL"},
            {0x41c6072ca3bc2075UL, "WNF_AI_PACKAGEUNINSTALL"},
            {0x41c6072ca3bc1875UL, "WNF_AI_PACKAGEUPDATE"},
            {0x41c6072ca3bc0875UL, "WNF_AI_USERTILE"},
            {0x4191012ca3bc0875UL, "WNF_AOW_BOOT_PROGRESS"},
            {0x89e1e2ca3bc0875UL, "WNF_APXI_CRITICAL_PACKAGES_INSTALLED"},
            {0x41961a2ca3bc0875UL, "WNF_ATP_PUSH_NOTIFICATION_RECEIVED"},
            {0x2821b2ca3bc4075UL, "WNF_AUDC_CAPTURE"},
            {0x2821b2ca3bc6075UL, "WNF_AUDC_CHAT_APP_CONTEXT"},
            {0x2821b2ca3bc08b5UL, "WNF_AUDC_CPUSET_ID"},
            {0x2821b2ca3bc2875UL, "WNF_AUDC_CPUSET_ID_SYSTEM"},
            {0x2821b2ca3bc5875UL, "WNF_AUDC_DEFAULT_RENDER_ENDPOINT_PROPERTIES"},
            {0x2821b2ca3bc2075UL, "WNF_AUDC_HEALTH_PROBLEM"},
            {0x2821b2ca3bc1075UL, "WNF_AUDC_PHONECALL_ACTIVE"},
            {0x2821b2ca3bc3075UL, "WNF_AUDC_RENDER"},
            {0x2821b2ca3bc4875UL, "WNF_AUDC_RINGERVIBRATE_STATE_CHANGED"},
            {0x2821b2ca3bc5075UL, "WNF_AUDC_SPATIAL_STATUS"},
            {0x2821b2ca3bc1875UL, "WNF_AUDC_TUNER_DEVICE_AVAILABILITY"},
            {0x2821b2ca3bc6875UL, "WNF_AUDC_VAM_ACTIVE"},
            {0x2821b2ca3bc3875UL, "WNF_AUDC_VOLUME_CONTEXT"},
            {0x4187182ca3bc0875UL, "WNF_AVA_SOUNDDETECTOR_PATTERN_MATCH"},
            {0x28a182ca3bc0875UL, "WNF_AVLC_DRIVER_REQUEST"},
            {0x28a182ca3bc1875UL, "WNF_AVLC_SHOW_VOLUMELIMITWARNING"},
            {0x28a182ca3bc1075UL, "WNF_AVLC_VOLUME_WARNING_ACCEPTED"},
            {0x15950d2fa3bc0875UL, "WNF_BCST_APP_BROADCAST_STREAM_STATE"},
            {0x41c6072fa3bc3875UL, "WNF_BI_APPLICATION_SERVICING_START_CHANNEL"},
            {0x41c6072fa3bc4075UL, "WNF_BI_APPLICATION_SERVICING_STOP_CHANNEL"},
            {0x41c6072fa3bc3075UL, "WNF_BI_APPLICATION_UNINSTALL_CHANNEL"},
            {0x41c6072fa3bc6835UL, "WNF_BI_BI_READY"},
            {0x41c6072fa3bc0875UL, "WNF_BI_BROKER_WAKEUP_CHANNEL"},
            {0x41c6072fa3bc5075UL, "WNF_BI_EVENT_DELETION"},
            {0x41c6072fa3bc4875UL, "WNF_BI_LOCK_SCREEN_UPDATE_CHANNEL"},
            {0x41c6072fa3bc8075UL, "WNF_BI_NETWORK_LIMITED_CHANNEL"},
            {0x41c6072fa3bc7075UL, "WNF_BI_NOTIFY_NEW_SESSION"},
            {0x41c6072fa3bc5875UL, "WNF_BI_PSM_TEST_HOOK_CHANNEL"},
            {0x41c6072fa3bc7875UL, "WNF_BI_QUERY_APP_USAGE"},
            {0x41c6072fa3bc6075UL, "WNF_BI_QUIET_MODE_UPDATE_CHANNEL"},
            {0x41c6072fa3bc2075UL, "WNF_BI_SESSION_CONNECT_CHANNEL"},
            {0x41c6072fa3bc2875UL, "WNF_BI_SESSION_DISCONNECT_CHANNEL"},
            {0x41c6072fa3bc1875UL, "WNF_BI_USER_LOGOFF_CHANNEL"},
            {0x41c6072fa3bc1075UL, "WNF_BI_USER_LOGON_CHANNEL"},
            {0x992022fa3bc1075UL, "WNF_BLTH_BLUETOOTH_AUDIO_GATEWAY_STATUS"},
            {0x992022fa3bc4075UL, "WNF_BLTH_BLUETOOTH_AVRCP_VOLUME_CHANGED"},
            {0x992022fa3bc2075UL, "WNF_BLTH_BLUETOOTH_CONNECTION_STATE_CHANGE"},
            {0x992022fa3bc4875UL, "WNF_BLTH_BLUETOOTH_DEVICE_BATTERY_IS_LOW"},
            {0x992022fa3bc6075UL, "WNF_BLTH_BLUETOOTH_DEVICE_DOCK_STATUS"},
            {0x992022fa3bc3075UL, "WNF_BLTH_BLUETOOTH_GATT_CLIENT_LEGACY_INVALIDATE_TOKEN"},
            {0x992022fa3bc2875UL, "WNF_BLTH_BLUETOOTH_GATT_CLIENT_LEGACY_REQUEST"},
            {0x992022fa3bc6875UL, "WNF_BLTH_BLUETOOTH_HFP_HF_LINE_AVAILABLE"},
            {0x992022fa3bc5075UL, "WNF_BLTH_BLUETOOTH_LE_ADV_SCANNING_STATUS"},
            {0x992022fa3bc1875UL, "WNF_BLTH_BLUETOOTH_MAP_STATUS"},
            {0x992022fa3bc3875UL, "WNF_BLTH_BLUETOOTH_QUICKPAIR_STATUS_CHANGED"},
            {0x992022fa3bc5875UL, "WNF_BLTH_BLUETOOTH_SHOW_PBAP_CONSENT"},
            {0x992022fa3bc0875UL, "WNF_BLTH_BLUETOOTH_STATUS"},
            {0x4196032fa3bc1075UL, "WNF_BMP_BG_PLAYBACK_REVOKED"},
            {0x4196032fa3bc0875UL, "WNF_BMP_BG_PLAYSTATE_CHANGED"},
            {0x1589012fa3bc0875UL, "WNF_BOOT_DIRTY_SHUTDOWN"},
            {0x1589012fa3bc1075UL, "WNF_BOOT_INVALID_TIME_SOURCE"},
            {0x1589012fa3bc1875UL, "WNF_BOOT_MEMORY_PARTITIONS_RESTORE"},
            {0x418f1c2fa3bc0875UL, "WNF_BRI_ACTIVE_WINDOW"},
            {0x418b0f2ea3bcd875UL, "WNF_CAM_ACTIVITY_ACCESS_CHANGED"},
            {0x418b0f2ea3bda075UL, "WNF_CAM_ACTIVITY_USAGE_CHANGED"},
            {0x418b0f2ea3bcf875UL, "WNF_CAM_APPACTIVATION_WITHVOICEABOVELOCK_CHANGED"},
            {0x418b0f2ea3bcf075UL, "WNF_CAM_APPACTIVATION_WITHVOICE_CHANGED"},
            {0x418b0f2ea3bc0875UL, "WNF_CAM_APPDIAGNOSTICS_ACCESS_CHANGED"},
            {0x418b0f2ea3bd0075UL, "WNF_CAM_APPDIAGNOSTICS_USAGE_CHANGED"},
            {0x418b0f2ea3bc1075UL, "WNF_CAM_APPOINTMENTS_ACCESS_CHANGED"},
            {0x418b0f2ea3bd0875UL, "WNF_CAM_APPOINTMENTS_USAGE_CHANGED"},
            {0x418b0f2ea3bce075UL, "WNF_CAM_BLUETOOTHSYNC_ACCESS_CHANGED"},
            {0x418b0f2ea3bc1875UL, "WNF_CAM_BLUETOOTH_ACCESS_CHANGED"},
            {0x418b0f2ea3bcd075UL, "WNF_CAM_BROADFILESYSTEMACCESS_ACCESS_CHANGED"},
            {0x418b0f2ea3bd9875UL, "WNF_CAM_BROADFILESYSTEMACCESS_USAGE_CHANGED"},
            {0x418b0f2ea3bc2075UL, "WNF_CAM_CAMERA_ACCESS_CHANGED"},
            {0x418b0f2ea3bd1075UL, "WNF_CAM_CAMERA_USAGE_CHANGED"},
            {0x418b0f2ea3bc2875UL, "WNF_CAM_CELLULARDATA_ACCESS_CHANGED"},
            {0x418b0f2ea3bd1875UL, "WNF_CAM_CELLULARDATA_USAGE_CHANGED"},
            {0x418b0f2ea3bc3075UL, "WNF_CAM_CHAT_ACCESS_CHANGED"},
            {0x418b0f2ea3bd2075UL, "WNF_CAM_CHAT_USAGE_CHANGED"},
            {0x418b0f2ea3bc3875UL, "WNF_CAM_CONTACTS_ACCESS_CHANGED"},
            {0x418b0f2ea3bd2875UL, "WNF_CAM_CONTACTS_USAGE_CHANGED"},
            {0x418b0f2ea3bcb075UL, "WNF_CAM_DOCUMENTSLIBRARY_ACCESS_CHANGED"},
            {0x418b0f2ea3bd7875UL, "WNF_CAM_DOCUMENTSLIBRARY_USAGE_CHANGED"},
            {0x418b0f2ea3bc4075UL, "WNF_CAM_EMAIL_ACCESS_CHANGED"},
            {0x418b0f2ea3bd3075UL, "WNF_CAM_EMAIL_USAGE_CHANGED"},
            {0x418b0f2ea3bcc875UL, "WNF_CAM_GAZEINPUT_ACCESS_CHANGED"},
            {0x418b0f2ea3bd9075UL, "WNF_CAM_GAZEINPUT_USAGE_CHANGED"},
            {0x418b0f2ea3bc4875UL, "WNF_CAM_HID_ACCESS_CHANGED"},
            {0x418b0f2ea3bc5075UL, "WNF_CAM_LOCATION_ACCESS_CHANGED"},
            {0x418b0f2ea3bd3875UL, "WNF_CAM_LOCATION_USAGE_CHANGED"},
            {0x418b0f2ea3bc5875UL, "WNF_CAM_MICROPHONE_ACCESS_CHANGED"},
            {0x418b0f2ea3bd4075UL, "WNF_CAM_MICROPHONE_USAGE_CHANGED"},
            {0x418b0f2ea3bc6875UL, "WNF_CAM_PHONECALLHISTORY_ACCESS_CHANGED"},
            {0x418b0f2ea3bd5075UL, "WNF_CAM_PHONECALLHISTORY_USAGE_CHANGED"},
            {0x418b0f2ea3bc6075UL, "WNF_CAM_PHONECALL_ACCESS_CHANGED"},
            {0x418b0f2ea3bd4875UL, "WNF_CAM_PHONECALL_USAGE_CHANGED"},
            {0x418b0f2ea3bcb875UL, "WNF_CAM_PICTURESLIBRARY_ACCESS_CHANGED"},
            {0x418b0f2ea3bd8075UL, "WNF_CAM_PICTURESLIBRARY_USAGE_CHANGED"},
            {0x418b0f2ea3bc7075UL, "WNF_CAM_POS_ACCESS_CHANGED"},
            {0x418b0f2ea3bc7875UL, "WNF_CAM_RADIOS_ACCESS_CHANGED"},
            {0x418b0f2ea3bd5875UL, "WNF_CAM_RADIOS_USAGE_CHANGED"},
            {0x418b0f2ea3bc8075UL, "WNF_CAM_SENSORSCUSTOM_ACCESS_CHANGED"},
            {0x418b0f2ea3bc8875UL, "WNF_CAM_SERIAL_ACCESS_CHANGED"},
            {0x418b0f2ea3bc9075UL, "WNF_CAM_USB_ACCESS_CHANGED"},
            {0x418b0f2ea3bc9875UL, "WNF_CAM_USERACCOUNTINFO_ACCESS_CHANGED"},
            {0x418b0f2ea3bd6075UL, "WNF_CAM_USERACCOUNTINFO_USAGE_CHANGED"},
            {0x418b0f2ea3bca075UL, "WNF_CAM_USERDATATASKS_ACCESS_CHANGED"},
            {0x418b0f2ea3bd6875UL, "WNF_CAM_USERDATATASKS_USAGE_CHANGED"},
            {0x418b0f2ea3bca875UL, "WNF_CAM_USERNOTIFICATIONLISTENER_ACCESS_CHANGED"},
            {0x418b0f2ea3bd7075UL, "WNF_CAM_USERNOTIFICATIONLISTENER_USAGE_CHANGED"},
            {0x418b0f2ea3bcc075UL, "WNF_CAM_VIDEOSLIBRARY_ACCESS_CHANGED"},
            {0x418b0f2ea3bd8875UL, "WNF_CAM_VIDEOSLIBRARY_USAGE_CHANGED"},
            {0x418b0f2ea3bce875UL, "WNF_CAM_WIFIDIRECT_ACCESS_CHANGED"},
            {0x12960f2ea3bc0875UL, "WNF_CAPS_CENTRAL_ACCESS_POLICIES_CHANGED"},
            {0xd920d2ea3bc08b5UL, "WNF_CCTL_BUTTON_REQUESTS"},
            {0x41960a2ea3bc8075UL, "WNF_CDP_ALLOW_CLIPBOARDHISTORY_POLICY_CHANGE"},
            {0x41960a2ea3bc8875UL, "WNF_CDP_ALLOW_CROSSDEVICECLIPBOARD_POLICY_CHANGE"},
            {0x41960a2ea3bc0875UL, "WNF_CDP_CDPSVC_READY"},
            {0x41960a2ea3bc1075UL, "WNF_CDP_CDPSVC_STOPPING"},
            {0x41960a2ea3bc1835UL, "WNF_CDP_CDPUSERSVC_READY"},
            {0x41960a2ea3bc2035UL, "WNF_CDP_CDPUSERSVC_STOPPING"},
            {0x41960a2ea3bc3075UL, "WNF_CDP_CDP_ACTIVITIES_RECIEVED"},
            {0x41960a2ea3bc6875UL, "WNF_CDP_CDP_LOCAL_ACTIVITIES_RECIEVED"},
            {0x41960a2ea3bc2875UL, "WNF_CDP_CDP_MESSAGES_QUEUED"},
            {0x41960a2ea3bc4075UL, "WNF_CDP_CDP_NOTIFICATION_ACTION_FORWARD_FAILURE"},
            {0x41960a2ea3bc5875UL, "WNF_CDP_ENABLE_ACTIVITYFEED_POLICY_CHANGE"},
            {0x41960a2ea3bc6075UL, "WNF_CDP_PUBLISH_USER_ACTIVITIES_POLICY_CHANGE"},
            {0x41960a2ea3bc7875UL, "WNF_CDP_UPLOAD_USER_ACTIVITIES_POLICY_CHANGE"},
            {0x41960a2ea3bc3875UL, "WNF_CDP_USERAUTH_POLICY_CHANGE"},
            {0x41960a2ea3bc5035UL, "WNF_CDP_USER_NEAR_SHARE_SETTING_CHANGE"},
            {0x41960a2ea3bc7075UL, "WNF_CDP_USER_RESOURCE_INFO_CHANGED"},
            {0x41960a2ea3bc4835UL, "WNF_CDP_USER_ROME_SETTING_CHANGE"},
            {0xd8a0b2ea3bc3075UL, "WNF_CELL_AIRPLANEMODE"},
            {0xd8a0b2ea3bc9075UL, "WNF_CELL_AIRPLANEMODE_DETAILS"},
            {0xd8a0b2ea3bc5075UL, "WNF_CELL_AVAILABLE_OPERATORS_CAN0"},
            {0xd8a0b2ea3bd5875UL, "WNF_CELL_AVAILABLE_OPERATORS_CAN1"},
            {0xd8a0b2ea3bd0075UL, "WNF_CELL_CALLFORWARDING_STATUS_CAN0"},
            {0xd8a0b2ea3bde075UL, "WNF_CELL_CALLFORWARDING_STATUS_CAN1"},
            {0xd8a0b2ea3be5875UL, "WNF_CELL_CAN_CONFIGURATION_SET_COMPLETE_MODEM0"},
            {0xd8a0b2ea3bc8075UL, "WNF_CELL_CAN_STATE_CAN0"},
            {0xd8a0b2ea3bd9075UL, "WNF_CELL_CAN_STATE_CAN1"},
            {0xd8a0b2ea3bc4075UL, "WNF_CELL_CDMA_ACTIVATION_CAN0"},
            {0xd8a0b2ea3bd4875UL, "WNF_CELL_CDMA_ACTIVATION_CAN1"},
            {0xd8a0b2ea3bdf475UL, "WNF_CELL_CONFIGURED_LINES_CAN0"},
            {0xd8a0b2ea3bdfc75UL, "WNF_CELL_CONFIGURED_LINES_CAN1"},
            {0xd8a0b2ea3bcf875UL, "WNF_CELL_CSP_WWAN_PLUS_READYNESS"},
            {0xd8a0b2ea3bc6475UL, "WNF_CELL_DATA_ENABLED_BY_USER_MODEM0"},
            {0xd8a0b2ea3bc5875UL, "WNF_CELL_DEVICE_INFO_CAN0"},
            {0xd8a0b2ea3bd6075UL, "WNF_CELL_DEVICE_INFO_CAN1"},
            {0xd8a0b2ea3be6875UL, "WNF_CELL_EMERGENCY_CALLBACK_MODE_STATUS"},
            {0xd8a0b2ea3bcc075UL, "WNF_CELL_HOME_OPERATOR_CAN0"},
            {0xd8a0b2ea3bda875UL, "WNF_CELL_HOME_OPERATOR_CAN1"},
            {0xd8a0b2ea3bcc875UL, "WNF_CELL_HOME_PRL_ID_CAN0"},
            {0xd8a0b2ea3bdb075UL, "WNF_CELL_HOME_PRL_ID_CAN1"},
            {0xd8a0b2ea3be2075UL, "WNF_CELL_IMSI_CAN0"},
            {0xd8a0b2ea3be2875UL, "WNF_CELL_IMSI_CAN1"},
            {0xd8a0b2ea3be8075UL, "WNF_CELL_IMS_STATUS_CAN0"},
            {0xd8a0b2ea3be8875UL, "WNF_CELL_IMS_STATUS_CAN1"},
            {0xd8a0b2ea3be9075UL, "WNF_CELL_IWLAN_AVAILABILITY_CAN0"},
            {0xd8a0b2ea3be9875UL, "WNF_CELL_IWLAN_AVAILABILITY_CAN1"},
            {0xd8a0b2ea3be3075UL, "WNF_CELL_LEGACY_SETTINGS_MIGRATION"},
            {0xd8a0b2ea3bc4875UL, "WNF_CELL_NETWORK_TIME_CAN0"},
            {0xd8a0b2ea3bd5075UL, "WNF_CELL_NETWORK_TIME_CAN1"},
            {0xd8a0b2ea3bed075UL, "WNF_CELL_NITZ_INFO"},
            {0xd8a0b2ea3bc3875UL, "WNF_CELL_OPERATOR_NAME_CAN0"},
            {0xd8a0b2ea3bd4075UL, "WNF_CELL_OPERATOR_NAME_CAN1"},
            {0xd8a0b2ea3bcb875UL, "WNF_CELL_PERSO_STATUS_CAN0"},
            {0xd8a0b2ea3bde875UL, "WNF_CELL_PERSO_STATUS_CAN1"},
            {0xd8a0b2ea3bc6875UL, "WNF_CELL_PHONE_NUMBER_CAN0"},
            {0xd8a0b2ea3bd7075UL, "WNF_CELL_PHONE_NUMBER_CAN1"},
            {0xd8a0b2ea3bc9875UL, "WNF_CELL_POSSIBLE_DATA_ACTIVITY_CHANGE_MODEM0"},
            {0xd8a0b2ea3bc0875UL, "WNF_CELL_POWER_STATE_MODEM0"},
            {0xd8a0b2ea3be1075UL, "WNF_CELL_PREFERRED_LANGUAGES_SLOT0"},
            {0xd8a0b2ea3be1875UL, "WNF_CELL_PREFERRED_LANGUAGES_SLOT1"},
            {0xd8a0b2ea3bea475UL, "WNF_CELL_PS_MEDIA_PREFERENCES_CAN0"},
            {0xd8a0b2ea3beac75UL, "WNF_CELL_PS_MEDIA_PREFERENCES_CAN1"},
            {0xd8a0b2ea3bd0c75UL, "WNF_CELL_RADIO_TYPE_MODEM0"},
            {0xd8a0b2ea3be6075UL, "WNF_CELL_REGISTRATION_CHANGED_TRIGGER_MV"},
            {0xd8a0b2ea3bc7c75UL, "WNF_CELL_REGISTRATION_PREFERENCES_CAN0"},
            {0xd8a0b2ea3bd8c75UL, "WNF_CELL_REGISTRATION_PREFERENCES_CAN1"},
            {0xd8a0b2ea3bc2075UL, "WNF_CELL_REGISTRATION_STATUS_CAN0"},
            {0xd8a0b2ea3bd2075UL, "WNF_CELL_REGISTRATION_STATUS_CAN1"},
            {0xd8a0b2ea3bca875UL, "WNF_CELL_REGISTRATION_STATUS_DETAILS_CAN0"},
            {0xd8a0b2ea3bd9875UL, "WNF_CELL_REGISTRATION_STATUS_DETAILS_CAN1"},
            {0xd8a0b2ea3bc1075UL, "WNF_CELL_SIGNAL_STRENGTH_BARS_CAN0"},
            {0xd8a0b2ea3bd1075UL, "WNF_CELL_SIGNAL_STRENGTH_BARS_CAN1"},
            {0xd8a0b2ea3be7075UL, "WNF_CELL_SIGNAL_STRENGTH_DETAILS_CAN0"},
            {0xd8a0b2ea3be7875UL, "WNF_CELL_SIGNAL_STRENGTH_DETAILS_CAN1"},
            {0xd8a0b2ea3bcb075UL, "WNF_CELL_SUPPORTED_SYSTEM_TYPES_CAN0"},
            {0xd8a0b2ea3bda075UL, "WNF_CELL_SUPPORTED_SYSTEM_TYPES_CAN1"},
            {0xd8a0b2ea3bca475UL, "WNF_CELL_SYSTEM_CONFIG"},
            {0xd8a0b2ea3bc1875UL, "WNF_CELL_SYSTEM_TYPE_CAN0"},
            {0xd8a0b2ea3bd1875UL, "WNF_CELL_SYSTEM_TYPE_CAN1"},
            {0xd8a0b2ea3be3875UL, "WNF_CELL_UICC_ATR_SLOT0"},
            {0xd8a0b2ea3be4075UL, "WNF_CELL_UICC_ATR_SLOT1"},
            {0xd8a0b2ea3bec075UL, "WNF_CELL_UICC_PIN_STATE_SLOT0"},
            {0xd8a0b2ea3bec875UL, "WNF_CELL_UICC_PIN_STATE_SLOT1"},
            {0xd8a0b2ea3be4875UL, "WNF_CELL_UICC_SIMSEC_SLOT0"},
            {0xd8a0b2ea3be5075UL, "WNF_CELL_UICC_SIMSEC_SLOT1"},
            {0xd8a0b2ea3be0075UL, "WNF_CELL_UICC_STATUS_DETAILS_SLOT0"},
            {0xd8a0b2ea3be0875UL, "WNF_CELL_UICC_STATUS_DETAILS_SLOT1"},
            {0xd8a0b2ea3bc2875UL, "WNF_CELL_UICC_STATUS_SLOT0"},
            {0xd8a0b2ea3bd2875UL, "WNF_CELL_UICC_STATUS_SLOT1"},
            {0xd8a0b2ea3bc8c75UL, "WNF_CELL_USER_PREFERRED_POWER_STATE_MODEM0"},
            {0xd8a0b2ea3bcf075UL, "WNF_CELL_UTK_PROACTIVE_CMD"},
            {0xd8a0b2ea3bce875UL, "WNF_CELL_UTK_SETUP_MENU_SLOT0"},
            {0xd8a0b2ea3bdd075UL, "WNF_CELL_UTK_SETUP_MENU_SLOT1"},
            {0xd8a0b2ea3bc7075UL, "WNF_CELL_VOICEMAIL_NUMBER_CAN0"},
            {0xd8a0b2ea3beb075UL, "WNF_CELL_WIFI_CALL_SETTINGS_CAN0"},
            {0xd8a0b2ea3beb875UL, "WNF_CELL_WIFI_CALL_SETTINGS_CAN1"},
            {0x15940b2ea3bc1075UL, "WNF_CERT_FLUSH_CACHE_STATE"},
            {0x15940b2ea3bc0875UL, "WNF_CERT_FLUSH_CACHE_TRIGGER"},
            {0xd85082ea3bc1875UL, "WNF_CFCL_SC_CONFIGURATIONS_ADDED"},
            {0xd85082ea3bc0875UL, "WNF_CFCL_SC_CONFIGURATIONS_CHANGED"},
            {0xd85082ea3bc1075UL, "WNF_CFCL_SC_CONFIGURATIONS_DELETED"},
            {0x41c6072ea3bc0875UL, "WNF_CI_SMODE_CHANGE"},
            {0x118f022ea3bc2035UL, "WNF_CLIP_CLIPBOARD_HISTORY_ENABLED_CHANGED"},
            {0x118f022ea3bc2835UL, "WNF_CLIP_CLIPBOARD_USERSVC_READY"},
            {0x118f022ea3bc3035UL, "WNF_CLIP_CLIPBOARD_USERSVC_STOPPED"},
            {0x118f022ea3bc0875UL, "WNF_CLIP_CONTENT_CHANGED"},
            {0x118f022ea3bc1035UL, "WNF_CLIP_HISTORY_CHANGED"},
            {0x118f022ea3bc1835UL, "WNF_CLIP_ROAMING_CLIPBOARD_ENABLED_CHANGED"},
            {0x1583002ea3bc4875UL, "WNF_CNET_CELLULAR_CONNECTIONS_AVAILABLE"},
            {0x1583002ea3bc3075UL, "WNF_CNET_DPU_GLOBAL_STATE_NOT_TRACKED"},
            {0x1583002ea3bc1875UL, "WNF_CNET_DPU_GLOBAL_STATE_OFF_TRACK"},
            {0x1583002ea3bc2075UL, "WNF_CNET_DPU_GLOBAL_STATE_ON_TRACK"},
            {0x1583002ea3bc1075UL, "WNF_CNET_DPU_GLOBAL_STATE_OVER_LIMIT"},
            {0x1583002ea3bc2875UL, "WNF_CNET_DPU_GLOBAL_STATE_UNDER_TRACK"},
            {0x1583002ea3bc6875UL, "WNF_CNET_NON_CELLULAR_CONNECTED"},
            {0x1583002ea3bc7875UL, "WNF_CNET_RADIO_ACTIVITY"},
            {0x1583002ea3bc7075UL, "WNF_CNET_RADIO_ACTIVITY_OR_NON_CELLULAR_CONNECTED"},
            {0x1583002ea3bc8075UL, "WNF_CNET_WIFI_ACTIVITY"},
            {0x1588012ea3bc0875UL, "WNF_CONT_RESTORE_FROM_SNAPSHOT_COMPLETE"},
            {0x41851d2ea3bc0875UL, "WNF_CSC_SERVICE_START"},
            {0xd8e1d2ea3bc3835UL, "WNF_CSHL_COMPOSER_CONTEXT_CHANGED"},
            {0xd8e1d2ea3bc0835UL, "WNF_CSHL_COMPOSER_LAUNCH_READY"},
            {0xd8e1d2ea3bc3035UL, "WNF_CSHL_COMPOSER_TEARDOWN"},
            {0xd8e1d2ea3bc2835UL, "WNF_CSHL_PRODUCT_READY"},
            {0xd8e1d2ea3bc4035UL, "WNF_CSHL_SKIP_OOBE_CXH"},
            {0xd8e1d2ea3bc1035UL, "WNF_CSHL_UI_AUTOMATION"},
            {0xd8e1d2ea3bc2035UL, "WNF_CSHL_VIEWHOSTING_READY"},
            {0x418e1d2ea3bc08f5UL, "WNF_CSH_LAUNCH_EXPLORER_REQUESTED"},
            {0x418e162ea3bc1035UL, "WNF_CXH_APP_FINISHED"},
            {0x418e162ea3bc2035UL, "WNF_CXH_BACK"},
            {0x418e162ea3bc1835UL, "WNF_CXH_BACK_STATE"},
            {0x418e162ea3bc2875UL, "WNF_CXH_OOBE_APP_READY"},
            {0x418e162ea3bc0835UL, "WNF_CXH_WEBAPP_STATUS"},
            {0x41870c29a3bc0875UL, "WNF_DBA_DEVICE_ACCESS_CHANGED"},
            {0x11950d29a3bc0875UL, "WNF_DCSP_POLICIES_UPDATED"},
            {0x41960b29a3bc0c75UL, "WNF_DEP_OOBE_COMPLETE"},
            {0x41960b29a3bc1475UL, "WNF_DEP_UNINSTALL_DISABLED"},
            {0xc900b29a3bc1875UL, "WNF_DEVM_DMWAPPUSHSVC_READY"},
            {0xc900b29a3bc3075UL, "WNF_DEVM_MULTIVARIANT_PROVISIONING_SESSIONS"},
            {0xc900b29a3bc0875UL, "WNF_DEVM_PROVISIONING_COMPLETE"},
            {0x15850729a3bc0875UL, "WNF_DICT_CONTENT_ADDED"},
            {0x15850729a3bc1075UL, "WNF_DICT_PERSONALIZATION_FEEDBACK_SIGNAL"},
            {0xa950729a3bc0875UL, "WNF_DISK_SCRUB_REQUIRED"},
            {0x41800329a3bc1075UL, "WNF_DMF_MIGRATION_COMPLETE"},
            {0x41800329a3bc1875UL, "WNF_DMF_MIGRATION_PROGRESS"},
            {0x41800329a3bc0875UL, "WNF_DMF_MIGRATION_STARTED"},
            {0x41800329a3bc2075UL, "WNF_DMF_UX_COMPLETE"},
            {0x41950029a3bc1075UL, "WNF_DNS_ALL_SERVER_TIMEOUT"},
            {0x41c60129a3bc0875UL, "WNF_DO_MANAGER_ACTIVE"},
            {0x41c60129a3bc1075UL, "WNF_DO_POLICY_CHANGED"},
            {0x418b1d29a3bc0c75UL, "WNF_DSM_DSMAPPINSTALLED"},
            {0x418b1d29a3bc1475UL, "WNF_DSM_DSMAPPREMOVED"},
            {0xc951b29a3bc1075UL, "WNF_DUSM_IS_CELLULAR_BACKGROUND_RESTRICTED"},
            {0xc951b29a3bc0875UL, "WNF_DUSM_TASK_TOAST"},
            {0x418b1929a3bc2835UL, "WNF_DWM_COMPOSITIONCAPABILITIES"},
            {0x418b1929a3bc3035UL, "WNF_DWM_HOLOGRAPHIC_COMPOSITOR_EXCLUSIVE"},
            {0x418b1929a3bc1035UL, "WNF_DWM_HOLOGRAPHIC_COMPOSITOR_EXCLUSIVE_LOW_FRAMERATE"},
            {0x418b1929a3bc1835UL, "WNF_DWM_HOLOGRAPHIC_COMPOSITOR_HAS_PROTECTED_CONTENT"},
            {0x418b1929a3bc2035UL, "WNF_DWM_HOLOGRAPHIC_COMPOSITOR_LOW_FRAMERATE"},
            {0x418b1929a3bc0835UL, "WNF_DWM_RUNNING"},
            {0xa811629a3bc0875UL, "WNF_DXGK_ADAPTER_TDR_NOTIFICATION"},
            {0xa811629a3bc1075UL, "WNF_DXGK_PATH_FAILED_OR_INVALIDATED"},
            {0x41c61629a3bc8075UL, "WNF_DX_ADAPTER_START"},
            {0x41c61629a3bc8875UL, "WNF_DX_ADAPTER_STOP"},
            {0x41c61629a3bcb075UL, "WNF_DX_ADAPTIVE_COLOR_FORCED_STATE_CHANGE"},
            {0x41c61629a3bc9875UL, "WNF_DX_COLOR_OVERRIDE_STATE_CHANGE"},
            {0x41c61629a3bc7075UL, "WNF_DX_COLOR_PROFILE_CHANGE"},
            {0x41c61629a3bc60b5UL, "WNF_DX_DEVICE_REMOVAL"},
            {0x41c61629a3bca075UL, "WNF_DX_DISPLAY_COLORIMETRY_DATA_CHANGED"},
            {0x41c61629a3bc5835UL, "WNF_DX_DISPLAY_CONFIG_CHANGE_NOTIFICATION"},
            {0x41c61629a3bc7875UL, "WNF_DX_GPM_TARGET"},
            {0x41c61629a3bc4075UL, "WNF_DX_HARDWARE_CONTENT_PROTECTION_TILT_NOTIFICATION"},
            {0x41c61629a3bc4875UL, "WNF_DX_INTERNAL_PANEL_DIMENSIONS"},
            {0x41c61629a3bc5035UL, "WNF_DX_MODERN_OUTPUTDUPLICATION"},
            {0x41c61629a3bc6835UL, "WNF_DX_MODERN_OUTPUTDUPLICATION_CONTEXTS"},
            {0x41c61629a3bc1035UL, "WNF_DX_MODE_CHANGE_NOTIFICATION"},
            {0x41c61629a3bc2835UL, "WNF_DX_MONITOR_CHANGE_NOTIFICATION"},
            {0x41c61629a3bc2035UL, "WNF_DX_NETWORK_DISPLAY_STATE_CHANGE_NOTIFICATION"},
            {0x41c61629a3bc1835UL, "WNF_DX_OCCLUSION_CHANGE_NOTIFICATION"},
            {0x41c61629a3bc9035UL, "WNF_DX_SDR_WHITE_LEVEL_CHANGED"},
            {0x41c61629a3bc0c75UL, "WNF_DX_STEREO_CONFIG"},
            {0x41c61629a3bca8b5UL, "WNF_DX_VAIL_CHANGE_NOTIFICATION"},
            {0x41c61629a3bc3875UL, "WNF_DX_VIDMM_BUDGETCHANGE_NOTIFICATION"},
            {0x41c61629a3bc30b5UL, "WNF_DX_VIDMM_TRIM_NOTIFICATION"},
            {0x41960f28a3bc0875UL, "WNF_EAP_APPLICATION_HANDLE"},
            {0x4810a28a3bc18f5UL, "WNF_EDGE_EXTENSION_AVAILABLE"},
            {0x4810a28a3bc10f5UL, "WNF_EDGE_EXTENSION_INSTALLED"},
            {0x4810a28a3bc20f5UL, "WNF_EDGE_INPRIVATE_EXTENSION_AVAILABLE"},
            {0x4810a28a3bc08f5UL, "WNF_EDGE_LAST_NAVIGATED_HOST"},
            {0x41960a28a3bc3875UL, "WNF_EDP_AAD_REAUTH_REQUIRED"},
            {0x41960a28a3bc3035UL, "WNF_EDP_APP_UI_ENTERPRISE_CONTEXT_CHANGED"},
            {0x41960a28a3bc2035UL, "WNF_EDP_CLIPBOARD_METADATA_CHANGED"},
            {0x41960a28a3bc7075UL, "WNF_EDP_CREDENTIALS_UPDATING"},
            {0x41960a28a3bc2835UL, "WNF_EDP_DIALOG_CANCEL"},
            {0x41960a28a3bc5875UL, "WNF_EDP_DPL_KEYS_DROPPING"},
            {0x41960a28a3bc1875UL, "WNF_EDP_DPL_KEYS_STATE"},
            {0x41960a28a3bc4475UL, "WNF_EDP_ENTERPRISE_CONTEXTS_UPDATED"},
            {0x41960a28a3bc10f5UL, "WNF_EDP_IDENTITY_REVOKED"},
            {0x41960a28a3bc6075UL, "WNF_EDP_MISSING_CREDENTIALS"},
            {0x41960a28a3bc50b5UL, "WNF_EDP_PROCESS_TLS_INDEX"},
            {0x41960a28a3bc4875UL, "WNF_EDP_PROCESS_UI_ENFORCEMENT"},
            {0x41960a28a3bc6875UL, "WNF_EDP_PURGE_APP_LEARNING_EVT"},
            {0x41960a28a3bc0835UL, "WNF_EDP_TAGGED_APP_LAUNCHED"},
            {0x41930a28a3bc0875UL, "WNF_EDU_PRINTER_POLICY_CHANGED"},
            {0x41950828a3bc0875UL, "WNF_EFS_SERVICE_START"},
            {0x41950828a3bc1075UL, "WNF_EFS_SOFTWARE_HIVE_AVAILABLE"},
            {0x13920028a3bc7875UL, "WNF_ENTR_ABOVELOCK_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc3075UL, "WNF_ENTR_ACCOUNTS_POLICY_VALUE_CHANGED"},
            {0x13920028a3bcf875UL, "WNF_ENTR_ALLOWALLTRUSTEDAPPS_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc8075UL, "WNF_ENTR_ALLOWAPPLICATIONS_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd4875UL, "WNF_ENTR_ALLOWCELLULARDATAROAMING_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd5075UL, "WNF_ENTR_ALLOWCELLULARDATA_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd1875UL, "WNF_ENTR_ALLOWDEVELOPERUNLOCK_POLICY_VALUE_CHANGED"},
            {0x13920028a3bda075UL, "WNF_ENTR_ALLOWDEVICEHEALTHMONITORING_POLICY_VALUE_CHANGED"},
            {0x13920028a3bca875UL, "WNF_ENTR_ALLOWINPUTPANEL_POLICY_VALUE_CHANGED"},
            {0x13920028a3bdb875UL, "WNF_ENTR_ALLOWMANUALWIFICONFIGURATION_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd6875UL, "WNF_ENTR_ALLOWMESSAGESYNC_POLICY_VALUE_CHANGED"},
            {0x13920028a3bdd875UL, "WNF_ENTR_ALLOWMESSAGE_MMS_POLICY_VALUE_CHANGED"},
            {0x13920028a3bde075UL, "WNF_ENTR_ALLOWMESSAGE_RCS_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd3075UL, "WNF_ENTR_ALLOWNONMICROSOFTSIGNEDUPDATE_POLICY_VALUE_CHANGED"},
            {0x13920028a3be0075UL, "WNF_ENTR_ALLOWPROJECTIONFROMPC_POLICY_VALUE_CHANGED"},
            {0x13920028a3bdd075UL, "WNF_ENTR_ALLOWPROJECTIONTOPC_POLICY_VALUE_CHANGED"},
            {0x13920028a3bdf875UL, "WNF_ENTR_ALLOWSET24HOURCLOCK_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd0075UL, "WNF_ENTR_ALLOWSHAREDUSERDATA_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd2075UL, "WNF_ENTR_ALLOWUPDATESERVICE_POLICY_VALUE_CHANGED"},
            {0x13920028a3bdc875UL, "WNF_ENTR_ALLOWWIFIDIRECT_POLICY_VALUE_CHANGED"},
            {0x13920028a3bdb075UL, "WNF_ENTR_ALLOWWIFI_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd3875UL, "WNF_ENTR_ALLOW_WBA_EXECUTION_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd8475UL, "WNF_ENTR_APPHVSI_CACHED_POLICY_VALUE_CHANGED"},
            {0x13920028a3bdf075UL, "WNF_ENTR_APPHVSI_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc5875UL, "WNF_ENTR_APPLICATIONMANAGEMENT_POLICY_VALUE_CHANGED"},
            {0x13920028a3be1875UL, "WNF_ENTR_APPPRIVACY_POLICY_VALUE_CHANGED"},
            {0x13920028a3be6875UL, "WNF_ENTR_BITS_POLICY_VALUE_CHANGED"},
            {0x13920028a3bcd875UL, "WNF_ENTR_BLUETOOTH_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc4075UL, "WNF_ENTR_BROWSER_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc5075UL, "WNF_ENTR_CAMERA_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc2075UL, "WNF_ENTR_CONNECTIVITY_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc9875UL, "WNF_ENTR_CONTEXT_STATE_CHANGE"},
            {0x13920028a3bc0875UL, "WNF_ENTR_DEVICELOCK_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd7075UL, "WNF_ENTR_DISABLEADVERTISINGID_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd4075UL, "WNF_ENTR_DOMAIN_NAMES_FOR_EMAIL_SYNC_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd5c75UL, "WNF_ENTR_EDPENFORCEMENTLEVEL_CACHED_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc8875UL, "WNF_ENTR_EDPENFORCEMENTLEVEL_POLICY_VALUE_CHANGED"},
            {0x13920028a3bce075UL, "WNF_ENTR_EDPNETWORKING_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd9c75UL, "WNF_ENTR_EDPSHOWICONS_CACHED_POLICY_VALUE_CHANGED"},
            {0x13920028a3bde875UL, "WNF_ENTR_EDPSMB_POLICY_VALUE_CHANGED"},
            {0x13920028a3be5075UL, "WNF_ENTR_EMOJI_AVAILABILITY_POLICY_VALUE_CHANGED"},
            {0x13920028a3be2075UL, "WNF_ENTR_ENABLETOUCHKEYBOARDAUTOINVOKE_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd9075UL, "WNF_ENTR_EVALUATE_APPHVSI_CONFIGURATION_STATE"},
            {0x13920028a3bd7875UL, "WNF_ENTR_EVALUATE_EDP_CONFIGURATION_STATE"},
            {0x13920028a3bc2875UL, "WNF_ENTR_EXPERIENCE_POLICY_VALUE_CHANGED"},
            {0x13920028a3be0875UL, "WNF_ENTR_EXPLOITGUARD_POLICY_VALUE_CHANGED"},
            {0x13920028a3be5875UL, "WNF_ENTR_FORCEDOCKED_TOUCHKEYBOARD_POLICY_VALUE_CHANGED"},
            {0x13920028a3be2875UL, "WNF_ENTR_FULLLAYOUT_AVAILABILITY_POLICY_VALUE_CHANGED"},
            {0x13920028a3be4075UL, "WNF_ENTR_HANDWRITING_AVAILABILITY_POLICY_VALUE_CHANGED"},
            {0x13920028a3be3875UL, "WNF_ENTR_NARROWLAYOUT_AVAILABILITY_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd8875UL, "WNF_ENTR_NETWORKISOLATION_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd6475UL, "WNF_ENTR_PROTECTEDDOMAINNAMES_CACHED_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc6875UL, "WNF_ENTR_PUSH_NOTIFICATION_RECEIVED"},
            {0x13920028a3bca075UL, "WNF_ENTR_PUSH_RECEIVED"},
            {0x13920028a3be7075UL, "WNF_ENTR_REMOVABLEDISK_DENY_WRITE_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc6075UL, "WNF_ENTR_REQUIRE_DEVICE_ENCRYPTION_POLICY_VALUE_CHANGED"},
            {0x13920028a3bce875UL, "WNF_ENTR_REQUIRE_DPL_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd1075UL, "WNF_ENTR_RESTRICTAPPDATATOSYTEMVOLUME_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd0875UL, "WNF_ENTR_RESTRICTAPPTOSYTEMVOLUME_POLICY_VALUE_CHANGED"},
            {0x13920028a3bdc075UL, "WNF_ENTR_SEARCH_ALLOW_INDEXER"},
            {0x13920028a3bcd075UL, "WNF_ENTR_SEARCH_ALLOW_INDEXING_ENCRYPTED_STORES_OR_ITEMS"},
            {0x13920028a3bcb075UL, "WNF_ENTR_SEARCH_ALLOW_USING_DIACRITICS"},
            {0x13920028a3bcb875UL, "WNF_ENTR_SEARCH_ALWAYS_USE_AUTO_LANG_DETECTION"},
            {0x13920028a3bcc075UL, "WNF_ENTR_SEARCH_DISABLE_REMOVABLE_DRIVE_INDEXING"},
            {0x13920028a3bc7075UL, "WNF_ENTR_SEARCH_POLICY_VALUE_CHANGED"},
            {0x13920028a3bcc875UL, "WNF_ENTR_SEARCH_PREVENT_INDEXING_LOW_DISK_SPACE_MB"},
            {0x13920028a3bc3875UL, "WNF_ENTR_SECURITY_POLICY_VALUE_CHANGED"},
            {0x13920028a3be4875UL, "WNF_ENTR_SPLITLAYOUT_AVAILABILITY_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc1875UL, "WNF_ENTR_SYSTEM_POLICY_VALUE_CHANGED"},
            {0x13920028a3be6075UL, "WNF_ENTR_TOUCHKEYBOARDDICTATION_POLICY_VALUE_CHANGED"},
            {0x13920028a3bd2875UL, "WNF_ENTR_UPDATESERVICEURL_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc4875UL, "WNF_ENTR_UPDATE_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc9075UL, "WNF_ENTR_WAP_MESSAGE_FOR_DMWAPPUSHSVC_READY"},
            {0x13920028a3be3075UL, "WNF_ENTR_WIDELAYOUT_AVAILABILITY_POLICY_VALUE_CHANGED"},
            {0x13920028a3bc1075UL, "WNF_ENTR_WIFI_POLICY_VALUE_CHANGED"},
            {0x13920028a3be1075UL, "WNF_ENTR_WINDOWSDEFENDERSECURITYCENTER_POLICY_VALUE_CHANGED"},
            {0x13920028a3bcf075UL, "WNF_ENTR_WINDOWS_DEFENDER_POLICY_VALUE_CHANGED"},
            {0x41870128a3bc2035UL, "WNF_EOA_ATMANAGER_ATS_STARTED"},
            {0x41870128a3bc3035UL, "WNF_EOA_MAGNIFIER_RENDERING_INPUT"},
            {0x41870128a3bc08f5UL, "WNF_EOA_NARRATOR_FOCUS_CHANGE"},
            {0x41870128a3bc2875UL, "WNF_EOA_NARRATOR_KEYBOARD_REMAP"},
            {0x41870128a3bc1075UL, "WNF_EOA_NARRATOR_RUNNING"},
            {0x41870128a3bc1875UL, "WNF_EOA_UISETTINGS_CHANGED"},
            {0x41911a28a3bc0875UL, "WNF_ETW_SUBSYSTEM_INITIALIZED"},
            {0x2831628a3bc0875UL, "WNF_EXEC_OSTASKCOMPLETION_REVOKED"},
            {0x2831628a3bc1875UL, "WNF_EXEC_THERMAL_LIMITER_CLOSE_APPLICATION_VIEWS"},
            {0x2831628a3bc2875UL, "WNF_EXEC_THERMAL_LIMITER_DISPLAY_WARNING"},
            {0x2831628a3bc3875UL, "WNF_EXEC_THERMAL_LIMITER_MITIGATION_STATUS"},
            {0x2831628a3bc3075UL, "WNF_EXEC_THERMAL_LIMITER_STOP_MRC"},
            {0x2831628a3bc2075UL, "WNF_EXEC_THERMAL_LIMITER_TERMINATE_BACKGROUND_TASKS"},
            {0xa840a2ba3bc0875UL, "WNF_FDBK_QUESTION_NOTIFICATION"},
            {0xf92022ba3bc0875UL, "WNF_FLTN_WNF_ARRIVED"},
            {0x4192022ba3bc0875UL, "WNF_FLT_RUNDOWN_WAIT"},
            {0x159f022ba3bc0875UL, "WNF_FLYT_IDS_CHANGED"},
            {0x4182012ba3bc0875UL, "WNF_FOD_STATE_CHANGE"},
            {0xd941d2ba3bc1075UL, "WNF_FSRL_OPLOCK_BREAK"},
            {0xd941d2ba3bc0875UL, "WNF_FSRL_TIERED_VOLUME_DETECTED"},
            {0x4183182ba3bc3075UL, "WNF_FVE_BDESVC_TRIGGER_START"},
            {0x4183182ba3bc6875UL, "WNF_FVE_BITLOCKER_ENCRYPT_ALL_DRIVES"},
            {0x4183182ba3bc7075UL, "WNF_FVE_DETASK_SYNC_PROVISIONING_COMPLETE"},
            {0x4183182ba3bc6075UL, "WNF_FVE_DETASK_TRIGGER_START"},
            {0x4183182ba3bc1075UL, "WNF_FVE_DE_MANAGED_VOLUMES_COUNT"},
            {0x4183182ba3bc0875UL, "WNF_FVE_DE_SUPPORT"},
            {0x4183182ba3bc4075UL, "WNF_FVE_MDM_POLICY_REFRESH"},
            {0x4183182ba3bc7875UL, "WNF_FVE_PROCESS_PERSISTENT_REQUEST"},
            {0x4183182ba3bc4875UL, "WNF_FVE_REQUIRE_SDCARD_ENCRYPTION"},
            {0x4183182ba3bc5075UL, "WNF_FVE_SDCARD_ENCRYPTION_REQUEST"},
            {0x4183182ba3bc5875UL, "WNF_FVE_SDCARD_ENCRYPTION_STATUS"},
            {0x4183182ba3bc3875UL, "WNF_FVE_STATE_CHANGE"},
            {0x4183182ba3bc2875UL, "WNF_FVE_WIM_HASH_DELETION_TRIGGER"},
            {0x4183182ba3bc2075UL, "WNF_FVE_WIM_HASH_GENERATION_COMPLETION"},
            {0x4183182ba3bc1875UL, "WNF_FVE_WIM_HASH_GENERATION_TRIGGER"},
            {0x41c60d2aa3bc0875UL, "WNF_GC_INITIAL_PRESENT"},
            {0x4196072aa3bc0875UL, "WNF_GIP_ADAPTER_CHANGE"},
            {0x389022aa3bc0875UL, "WNF_GLOB_USERPROFILE_LANGLIST_CHANGED"},
            {0xd891e2aa3bc0875UL, "WNF_GPOL_SYSTEM_CHANGES"},
            {0xd891e2aa3bc10f5UL, "WNF_GPOL_USER_CHANGES"},
            {0x418b0f25a3bc0875UL, "WNF_HAM_SYSTEM_STATE_CHANGED"},
            {0x41950f25a3bc0875UL, "WNF_HAS_VERIFY_HEALTH_CERT"},
            {0xe8a0125a3bcc035UL, "WNF_HOLO_CAPTURE_STATE"},
            {0xe8a0125a3bc7835UL, "WNF_HOLO_DISPLAY_QUALITY_LEVEL"},
            {0xe8a0125a3bc5075UL, "WNF_HOLO_ENVIRONMENT_AUDIO_ASSET"},
            {0xe8a0125a3bc2835UL, "WNF_HOLO_FORCE_ROOM_BOUNDARY"},
            {0xe8a0125a3bc2075UL, "WNF_HOLO_INPUT_FOCUS_CHANGE"},
            {0xe8a0125a3bcb835UL, "WNF_HOLO_PROJECTION_REQUEST"},
            {0xe8a0125a3bc9035UL, "WNF_HOLO_REQUEST_HMD_USE_STATE"},
            {0xe8a0125a3bc9835UL, "WNF_HOLO_REQUEST_HOLOGRAPHIC_ACTIVATION_REALM"},
            {0xe8a0125a3bca035UL, "WNF_HOLO_RESET_IDLE_TIMER"},
            {0xe8a0125a3bc7035UL, "WNF_HOLO_RETAIL_DEMO_TIMER"},
            {0xe8a0125a3bc3835UL, "WNF_HOLO_ROOM_BOUNDARY_DATA_CHANGED"},
            {0xe8a0125a3bc4035UL, "WNF_HOLO_ROOM_BOUNDARY_VISIBILITY"},
            {0xe8a0125a3bc6835UL, "WNF_HOLO_SET_SHELL_SPAWN_POINT"},
            {0xe8a0125a3bcb035UL, "WNF_HOLO_SHARING_SESSION_CONTEXT"},
            {0xe8a0125a3bc4835UL, "WNF_HOLO_SHELL_INPUT_3DSWITCH_DISABLE"},
            {0xe8a0125a3bc1835UL, "WNF_HOLO_SHELL_STATE"},
            {0xe8a0125a3bca875UL, "WNF_HOLO_SHELL_STATE_INTERACTIVE_USER"},
            {0xe8a0125a3bcc875UL, "WNF_HOLO_SHELL_USER_PRESENCE_STATE"},
            {0xe8a0125a3bc3035UL, "WNF_HOLO_STREAMING_STATE"},
            {0xe8a0125a3bc8875UL, "WNF_HOLO_SYSTEM_DISPLAY_CONTEXT_CHANGE"},
            {0xe8a0125a3bc6075UL, "WNF_HOLO_UNINSTALL_COMPLETE"},
            {0xe8a0125a3bc5875UL, "WNF_HOLO_UNINSTALL_PREPARE"},
            {0xe8a0125a3bc0835UL, "WNF_HOLO_USER_DISPLAY_CONTEXT"},
            {0xe8a0125a3bc1035UL, "WNF_HOLO_USER_INPUT_CONTEXT"},
            {0x418a1825a3bc0875UL, "WNF_HVL_CPU_MGMT_PARTITION"},
            {0x17961725a3bc1075UL, "WNF_HYPV_HOST_WMI_EVENT_PROVIDER_STATE"},
            {0x17961725a3bc0875UL, "WNF_HYPV_HOST_WMI_OBJECT_PROVIDER_STATE"},
            {0x41830324a3bc1835UL, "WNF_IME_AUTOMATIC_PRIVATE_MODE"},
            {0x41830324a3bc1035UL, "WNF_IME_EXPLICIT_PRIVATE_MODE"},
            {0x41830324a3bc0875UL, "WNF_IME_INPUT_MODE_LABEL"},
            {0x41830324a3bc2035UL, "WNF_IME_INPUT_SWITCH_NOTIFY"},
            {0xf950324a3bc4835UL, "WNF_IMSN_GLOBALLIGHTSINVALIDATED"},
            {0xf950324a3bc1835UL, "WNF_IMSN_IMMERSIVEMONITORCHANGED"},
            {0xf950324a3bc3035UL, "WNF_IMSN_KILL_LOGICAL_FOCUS"},
            {0xf950324a3bc1035UL, "WNF_IMSN_LAUNCHERVISIBILITY"},
            {0xf950324a3bc0835UL, "WNF_IMSN_MONITORMODECHANGED"},
            {0xf950324a3bc3835UL, "WNF_IMSN_PROJECTIONDISPLAYAVAILABLE"},
            {0xf950324a3bc4035UL, "WNF_IMSN_TRANSPARENCYPOLICY"},
            {0x41950324a3bc0875UL, "WNF_IMS_PUSH_NOTIFICATION_RECEIVED"},
            {0x41920124a3bc0875UL, "WNF_IOT_EMBEDDED_MODE_POLICY_VALUE_CHANGED"},
            {0x41920124a3bc1075UL, "WNF_IOT_STARTUP_SETTINGS_CHANGED"},
            {0x418b1d24a3bc1835UL, "WNF_ISM_CURSOR_MANAGER_READY"},
            {0x418b1d24a3bc2075UL, "WNF_ISM_GAMECONTROLLER_ZEPHYRUS_FAULT"},
            {0x418b1d24a3bc1035UL, "WNF_ISM_INPUT_UPDATE_AFTER_TRACK_INTERVAL"},
            {0x418b1d24a3bc0835UL, "WNF_ISM_LAST_USER_ACTIVITY"},
            {0x128f1b24a3bc0835UL, "WNF_IUIS_SCALE_CHANGED"},
            {0x41901d26a3bc2875UL, "WNF_KSV_CAMERAPRIVACY"},
            {0x41901d26a3bc1075UL, "WNF_KSV_DEVICESTATE"},
            {0x41901d26a3bc1875UL, "WNF_KSV_FSSTREAMACTIVITY"},
            {0x41901d26a3bc2075UL, "WNF_KSV_KSSTREAMACTIVITY"},
            {0x41901d26a3bc0875UL, "WNF_KSV_STREAMSTATE"},
            {0x6880f21a3bc0875UL, "WNF_LANG_FOD_INSTALLATION_STARTED"},
            {0x41820b21a3bc0875UL, "WNF_LED_SETTINGSCHANGED"},
            {0x41950821a3bc4875UL, "WNF_LFS_ACTION_DIALOG_AVAILABLE"},
            {0x41950821a3bc3875UL, "WNF_LFS_CLIENT_RECALCULATE_PERMISSIONS"},
            {0x41950821a3bc2075UL, "WNF_LFS_GEOFENCETRACKING_STATE"},
            {0x41950821a3bc6075UL, "WNF_LFS_LOCATION_MDM_AREA_POLICY_CHANGED"},
            {0x41950821a3bc6875UL, "WNF_LFS_LOCATION_MDM_POLICY_ENABLELOCATION_CHANGED"},
            {0x41950821a3bc1875UL, "WNF_LFS_MASTERSWITCH_STATE"},
            {0x41950821a3bc4075UL, "WNF_LFS_PERMISSION_TO_SHOW_ICON_CHANGED"},
            {0x41950821a3bc3075UL, "WNF_LFS_POSITION_AVAILABLE"},
            {0x41950821a3bc2875UL, "WNF_LFS_RESERVED_WNF_EVENT_2"},
            {0x41950821a3bc1075UL, "WNF_LFS_RUNNING_STATE"},
            {0x41950821a3bc5075UL, "WNF_LFS_SIGNIFICANT_LOCATION_EVENT"},
            {0x41950821a3bc0875UL, "WNF_LFS_STATE"},
            {0x41950821a3bc5875UL, "WNF_LFS_VISITS_SIGNIFICANT_LOCATION_EVENT"},
            {0x41850721a3bc3075UL, "WNF_LIC_DEVICE_LICENSE_MISSING"},
            {0x41850721a3bc2875UL, "WNF_LIC_DEVICE_LICENSE_REMOVED"},
            {0x41850721a3bc2075UL, "WNF_LIC_DEVICE_LICENSE_UPDATED"},
            {0x41850721a3bc1875UL, "WNF_LIC_HARDWAREID_IN_DEVICE_LICENSE_IN_TOLERANCE"},
            {0x41850721a3bc1075UL, "WNF_LIC_HARDWAREID_IN_DEVICE_LICENSE_OUT_OF_TOLERANCE"},
            {0x41850721a3bc3875UL, "WNF_LIC_INT_DEVICE_LICENSE_EXPIRED"},
            {0x41850721a3bc4075UL, "WNF_LIC_LOCAL_MIGRATED_LICENSES_FOUND"},
            {0x41850721a3bc4875UL, "WNF_LIC_MANAGE_DEVICE_REGISTRATION_AND_REACTIVATION"},
            {0x41850721a3bc0875UL, "WNF_LIC_NO_APPLICABLE_LICENSES_FOUND"},
            {0x41c60321a3bc2875UL, "WNF_LM_APP_LICENSE_EVENT"},
            {0x41c60321a3bc1075UL, "WNF_LM_CONTENT_LICENSE_CHANGED"},
            {0x41c60321a3bc3875UL, "WNF_LM_LICENSE_REFRESHED"},
            {0x41c60321a3bc3075UL, "WNF_LM_OFFLINE_PC_CHANGED"},
            {0x41c60321a3bc2075UL, "WNF_LM_OPTIONAL_PACKAGE_SUSPEND_REQUIRED"},
            {0x41c60321a3bc0875UL, "WNF_LM_PACKAGE_SUSPEND_REQUIRED"},
            {0x41c60321a3bc1875UL, "WNF_LM_ROOT_LICENSE_CHANGED"},
            {0x41850121a3bc0875UL, "WNF_LOC_DEVICE_BROKER_ACCESS_CHANGED"},
            {0x41850121a3bc1075UL, "WNF_LOC_RESERVED_WNF_EVENT"},
            {0x41850121a3bc1875UL, "WNF_LOC_SHOW_SYSTRAY"},
            {0xf810121a3bc4075UL, "WNF_LOGN_BIO_ENROLLMENT_APP_INSTANCE_CHANGED"},
            {0xf810121a3bc3075UL, "WNF_LOGN_CREDENTIAL_TILE_SELECTION_CHANGED"},
            {0xf810121a3bc0835UL, "WNF_LOGN_EOA_FLYOUT_POSITION"},
            {0xf810121a3bc2875UL, "WNF_LOGN_LOCAL_SIGNON"},
            {0xf810121a3bc2035UL, "WNF_LOGN_PINPAD_VISIBLE"},
            {0xf810121a3bc1835UL, "WNF_LOGN_RETURN_TO_LOCK"},
            {0xf810121a3bc1035UL, "WNF_LOGN_SLIDE_TO_SHUTDOWN"},
            {0xf810121a3bc3835UL, "WNF_LOGN_SUPPRESS_FINGERPRINT_WAKE"},
            {0x12960f20a3bc2075UL, "WNF_MAPS_MAPLOADER_PACKAGE_CHANGE"},
            {0x12960f20a3bc1075UL, "WNF_MAPS_MAPLOADER_PROGRESS"},
            {0x12960f20a3bc1875UL, "WNF_MAPS_MAPLOADER_STATUS_CHANGE"},
            {0x41c60320a3bc0875UL, "WNF_MM_BAD_MEMORY_PENDING_REMOVAL"},
            {0x41c60320a3bc1075UL, "WNF_MM_PHYSICAL_MEMORY_CHANGE"},
            {0x41880120a3bc0875UL, "WNF_MON_THERMAL_CAP_CHANGED"},
            {0x41921c20a3bc2075UL, "WNF_MRT_MERGE_SYSTEM_PRI_FILES"},
            {0x41921c20a3bc1c75UL, "WNF_MRT_PERSISTENT_QUALIFIER_CHANGED"},
            {0x41921c20a3bc0875UL, "WNF_MRT_QUALIFIER_CONTRAST_CHANGED"},
            {0x41921c20a3bc1075UL, "WNF_MRT_QUALIFIER_THEME_CHANGED"},
            {0x41921c20a3bc2875UL, "WNF_MRT_SYSTEM_PRI_MERGE"},
            {0x41871d20a3bc0835UL, "WNF_MSA_ACCOUNTSTATECHANGE"},
            {0x41871d20a3bc2c75UL, "WNF_MSA_GLOBAL_DEVICE_IDENTITY_PROVISIONED"},
            {0x41871d20a3bc3475UL, "WNF_MSA_HARDWARE_DEVICE_IDENTITY_UPDATED"},
            {0x41871d20a3bc2075UL, "WNF_MSA_TPM_ATTESTATION_ATTEMPT_READY"},
            {0x41871d20a3bc1475UL, "WNF_MSA_TPM_AVAILABLE"},
            {0x41871d20a3bc1875UL, "WNF_MSA_TPM_SERVER_CLIENT_KEY_STATE_UPDATED"},
            {0x41941b20a3bc1075UL, "WNF_MUR_MEDIA_UI_REQUEST_LAN"},
            {0x41941b20a3bc0875UL, "WNF_MUR_MEDIA_UI_REQUEST_WLAN"},
            {0x17950f23a3bc2075UL, "WNF_NASV_DYNAMIC_LOCK_BLUETOOTH_STATUS"},
            {0x17950f23a3bc1075UL, "WNF_NASV_SERVICE_RUNNING"},
            {0x17950f23a3bc1835UL, "WNF_NASV_USER_AUTHENTICATION"},
            {0x17950f23a3bc0835UL, "WNF_NASV_USER_PRESENT"},
            {0x41840d23a3bc0875UL, "WNF_NCB_APP_AVAILABLE"},
            {0x128f0a23a3bc0875UL, "WNF_NDIS_ADAPTER_ARRIVAL"},
            {0x128f0a23a3bc1075UL, "WNF_NDIS_CORRUPTED_STORE"},
            {0x41850823a3bc0875UL, "WNF_NFC_SE_CARD_EMULATION_STATE_CHANGED"},
            {0x41850923a3bc1075UL, "WNF_NGC_AIKCERT_TRIGGER"},
            {0x41850923a3bc3875UL, "WNF_NGC_CREDENTIAL_REFRESH_REQUIRED"},
            {0x41850923a3bc5075UL, "WNF_NGC_CREDENTIAL_RESET_EXPERIENCE_ACTIVE"},
            {0x41850923a3bc3075UL, "WNF_NGC_CRYPTO_MDM_POLICY_CHANGED"},
            {0x41850923a3bc2875UL, "WNF_NGC_GESTURE_AUTHENTICATED"},
            {0x41850923a3bc6075UL, "WNF_NGC_LAUNCH_NTH_USER_SCENARIO"},
            {0x41850923a3bc4875UL, "WNF_NGC_LAUNCH_PIN_RESET_SCENARIO"},
            {0x41850923a3bc4035UL, "WNF_NGC_PIN_RESET_SCENARIO_STATE_CHANGE"},
            {0x41850923a3bc2075UL, "WNF_NGC_PREGEN_DELAY_TRIGGER"},
            {0x41850923a3bc6875UL, "WNF_NGC_PREGEN_NGCISOCTNR_TRIGGER"},
            {0x41850923a3bc0875UL, "WNF_NGC_PREGEN_TRIGGER"},
            {0x41850923a3bc1875UL, "WNF_NGC_PRO_CSP_POLICY_CHANGED"},
            {0x41870223a3bc0875UL, "WNF_NLA_CAPABILITY_CHANGE"},
            {0x41870223a3bc1875UL, "WNF_NLA_TASK_TRIGGER"},
            {0x418b0223a3bc1875UL, "WNF_NLM_HNS_HIDDEN_INTERFACE"},
            {0x418b0223a3bc1075UL, "WNF_NLM_INTERNET_PRESENT"},
            {0x418b0223a3bc0875UL, "WNF_NLM_VPN_RECONNECT_CHANGE"},
            {0x41950223a3bc2035UL, "WNF_NLS_GEOID_CHANGED"},
            {0x41950223a3bc1835UL, "WNF_NLS_LOCALE_INFO_CHANGED"},
            {0x41950223a3bc0835UL, "WNF_NLS_USER_DEFAULT_LOCALE_CHANGED"},
            {0x41950223a3bc1035UL, "WNF_NLS_USER_UILANG_CHANGED"},
            {0xc951e23a3bc0875UL, "WNF_NPSM_SERVICE_STARTED"},
            {0x418f1d23a3bc0875UL, "WNF_NSI_SERVICE_STATUS"},
            {0x28f0222a3bc5075UL, "WNF_OLIC_OS_EDITION_CHANGE"},
            {0x28f0222a3bc6875UL, "WNF_OLIC_OS_LICENSE_NON_GENUINE"},
            {0x28f0222a3bc5875UL, "WNF_OLIC_OS_LICENSE_POLICY_CHANGE"},
            {0x28f0222a3bc6075UL, "WNF_OLIC_OS_LICENSE_TERMS_ACCEPTED"},
            {0x4840122a3bc1035UL, "WNF_OOBE_SHL_MAGNIFIER_CONFIRM"},
            {0x4840122a3bc0835UL, "WNF_OOBE_SHL_MAGNIFIER_QUERY"},
            {0x4840122a3bc1875UL, "WNF_OOBE_SHL_MONITOR_STATE"},
            {0x4840122a3bc2035UL, "WNF_OOBE_SHL_SPEECH_CONTROLLER"},
            {0xf911d22a3bc8075UL, "WNF_OSWN_STORAGE_APP_PAIRING_CHANGE"},
            {0xf911d22a3bcb875UL, "WNF_OSWN_STORAGE_FINISHED_USAGE_CATEGORY_UPDATE"},
            {0xf911d22a3bc7075UL, "WNF_OSWN_STORAGE_FREE_SPACE_CHANGE"},
            {0xf911d22a3bc6075UL, "WNF_OSWN_STORAGE_PRESENCE_CHANGE"},
            {0xf911d22a3bcc075UL, "WNF_OSWN_STORAGE_SHELLHWD_EVENT"},
            {0xf911d22a3bc7875UL, "WNF_OSWN_STORAGE_TEMP_CLEANUP_CHANGE"},
            {0xf911d22a3bc6875UL, "WNF_OSWN_STORAGE_VOLUME_STATUS_CHANGE"},
            {0xf911d22a3bc5875UL, "WNF_OSWN_SYSTEM_CLOCK_CHANGED"},
            {0x41c61d22a3bc8075UL, "WNF_OS_IP_OVER_USB_AVAILABLE"},
            {0x41c61d22a3bc8875UL, "WNF_OS_IU_PROGRESS_REPORT"},
            {0x5941822a3bc0875UL, "WNF_OVRD_OVERRIDESCALEUPDATED"},
            {0x419f0f3da3bc0875UL, "WNF_PAY_CANMAKEPAYMENT_BROKER_READY"},
            {0x4181083da3bc1075UL, "WNF_PFG_PEN_FIRST_DRAG"},
            {0x4181083da3bc0875UL, "WNF_PFG_PEN_FIRST_TAP"},
            {0xd88063da3bc4075UL, "WNF_PHNL_LINE1_READY"},
            {0x1188063da3bc4875UL, "WNF_PHNP_ANNOTATION_ENDPOINT"},
            {0x1188063da3bc3875UL, "WNF_PHNP_SERVICE_INITIALIZED"},
            {0x1188063da3bc4075UL, "WNF_PHNP_SIMSEC_READY"},
            {0x4188063da3bc3075UL, "WNF_PHN_CALLFORWARDING_STATUS_LINE0"},
            {0x4188063da3bc2875UL, "WNF_PHN_CALL_STATUS"},
            {0xc83033da3bc0875UL, "WNF_PMEM_MEMORY_ERROR"},
            {0x96003da3bc0875UL, "WNF_PNPA_DEVNODES_CHANGED"},
            {0x96003da3bc1035UL, "WNF_PNPA_DEVNODES_CHANGED_SESSION"},
            {0x96003da3bc2875UL, "WNF_PNPA_HARDWAREPROFILES_CHANGED"},
            {0x96003da3bc3035UL, "WNF_PNPA_HARDWAREPROFILES_CHANGED_SESSION"},
            {0x96003da3bc3875UL, "WNF_PNPA_PORTS_CHANGED"},
            {0x96003da3bc4035UL, "WNF_PNPA_PORTS_CHANGED_SESSION"},
            {0x96003da3bc1875UL, "WNF_PNPA_VOLUMES_CHANGED"},
            {0x96003da3bc2035UL, "WNF_PNPA_VOLUMES_CHANGED_SESSION"},
            {0x396003da3bc0875UL, "WNF_PNPB_AWAITING_RESPONSE"},
            {0x296003da3bc1875UL, "WNF_PNPC_CONTAINER_CONFIG_REQUESTED"},
            {0x296003da3bc1075UL, "WNF_PNPC_DEVICE_INSTALL_REQUESTED"},
            {0x296003da3bc0875UL, "WNF_PNPC_REBOOT_REQUIRED"},
            {0x41c6013da3bcd075UL, "WNF_PO_BASIC_BRIGHTNESS_ENGINE_DISABLED"},
            {0x41c6013da3bc8075UL, "WNF_PO_BATTERY_CHARGE_LEVEL"},
            {0x41c6013da3bd3875UL, "WNF_PO_BATTERY_CHARGE_LIMITING_MODE"},
            {0x41c6013da3bc9875UL, "WNF_PO_BATTERY_DISCHARGING"},
            {0x41c6013da3bcd875UL, "WNF_PO_BRIGHTNESS_ALS_OFFSET"},
            {0x41c6013da3bcf075UL, "WNF_PO_CAD_STICKY_DISABLE_CHARGING"},
            {0x41c6013da3bc6075UL, "WNF_PO_CHARGE_ESTIMATE"},
            {0x41c6013da3bc1075UL, "WNF_PO_COMPOSITE_BATTERY"},
            {0x41c6013da3bc9075UL, "WNF_PO_DEPRECATED1"},
            {0x41c6013da3bc5075UL, "WNF_PO_DISCHARGE_ESTIMATE"},
            {0x41c6013da3bc5c75UL, "WNF_PO_DISCHARGE_START_FILETIME"},
            {0x41c6013da3bc7835UL, "WNF_PO_DISPLAY_REQUEST_ACTIVE"},
            {0x41c6013da3bcc875UL, "WNF_PO_DRIPS_DEVICE_CONSTRAINTS_REGISTERED"},
            {0x41c6013da3bd4075UL, "WNF_PO_DYNAMIC_BRIGHTNESS_SLIDER"},
            {0x41c6013da3bc3075UL, "WNF_PO_ENERGY_SAVER_OVERRIDE"},
            {0x41c6013da3bc2875UL, "WNF_PO_ENERGY_SAVER_SETTING"},
            {0x41c6013da3bc2075UL, "WNF_PO_ENERGY_SAVER_STATE"},
            {0x41c6013da3bd1875UL, "WNF_PO_INPUT_SUPPRESS_NOTIFICATION"},
            {0x41c6013da3bd3075UL, "WNF_PO_INPUT_SUPPRESS_NOTIFICATION_EX"},
            {0x41c6013da3bcb875UL, "WNF_PO_MODERN_STANDBY_EXIT_INITIATED"},
            {0x41c6013da3bd2875UL, "WNF_PO_OPPORTUNISTIC_CS"},
            {0x41c6013da3bce875UL, "WNF_PO_OVERLAY_POWER_SCHEME_UPDATE"},
            {0x41c6013da3bd4875UL, "WNF_PO_PDC_SHELL_PHASE_NOTIFICATION"},
            {0x41c6013da3bcf875UL, "WNF_PO_POWER_BUTTON_STATE"},
            {0x41c6013da3bc1875UL, "WNF_PO_POWER_STATE_CHANGE"},
            {0x41c6013da3bd1075UL, "WNF_PO_PRESLEEP_NOTIFICATION"},
            {0x41c6013da3bcb075UL, "WNF_PO_PREVIOUS_SHUTDOWN_STATE"},
            {0x41c6013da3bca875UL, "WNF_PO_PRIMARY_DISPLAY_LOGICAL_STATE"},
            {0x41c6013da3bca075UL, "WNF_PO_PRIMARY_DISPLAY_VISIBLE_STATE"},
            {0x41c6013da3bc0875UL, "WNF_PO_SCENARIO_CHANGE"},
            {0x41c6013da3bc8875UL, "WNF_PO_SLEEP_STUDY_USER_PRESENCE_CHANGED"},
            {0x41c6013da3bcc075UL, "WNF_PO_SW_HW_DRIPS_DIVERGENCE"},
            {0x41c6013da3bd0075UL, "WNF_PO_SYSTEM_TIME_CHANGED"},
            {0x41c6013da3bc4875UL, "WNF_PO_THERMAL_HIBERNATE_OCCURRED"},
            {0x41c6013da3bc6875UL, "WNF_PO_THERMAL_OVERTHROTTLE"},
            {0x41c6013da3bc4075UL, "WNF_PO_THERMAL_SHUTDOWN_OCCURRED"},
            {0x41c6013da3bc3875UL, "WNF_PO_THERMAL_STANDBY"},
            {0x41c6013da3bc7075UL, "WNF_PO_USER_AWAY_PREDICTION"},
            {0x41c6013da3bce075UL, "WNF_PO_VIDEO_INITIALIALIZED"},
            {0x41c6013da3bd2075UL, "WNF_PO_WAKE_ON_VOICE_STATE"},
            {0x41c6013da3bd0875UL, "WNF_PO_WEAK_CHARGER"},
            {0x17891c3da3bc2075UL, "WNF_PROV_AUTOPILOT_ASYNC_COMPLETE"},
            {0x17891c3da3bc1875UL, "WNF_PROV_AUTOPILOT_PROFILE_AVAILABLE"},
            {0x17891c3da3bc3475UL, "WNF_PROV_DEVICE_BOOTSTRAP_COMPLETE"},
            {0x17891c3da3bc1075UL, "WNF_PROV_TPM_ATTEST_COMPLETE"},
            {0x17891c3da3bc0875UL, "WNF_PROV_TURN_COMPLETE"},
            {0x41c61d3da3bc0875UL, "WNF_PS_WAKE_CHARGE_RESOURCE_POLICY"},
            {0x418f1a3da3bc0875UL, "WNF_PTI_WNS_RECEIVED"},
            {0x41940a3fa3bc0875UL, "WNF_RDR_SMB1_NOT_IN_USE_STATE_CHANGE"},
            {0x41c6033fa3bc1075UL, "WNF_RM_GAME_MODE_ACTIVE"},
            {0x41c6033fa3bc0875UL, "WNF_RM_MEMORY_MONITOR_USAGE_METRICS"},
            {0x41c6033fa3bc1875UL, "WNF_RM_QUIET_MODE"},
            {0x7851e3fa3bc0875UL, "WNF_RPCF_FWMAN_RUNNING"},
            {0x12821a3fa3bc1875UL, "WNF_RTDS_NAMED_PIPE_TRIGGER_CHANGED"},
            {0x12821a3fa3bc0875UL, "WNF_RTDS_RPC_INTERFACE_TRIGGER_CHANGED"},
            {0x2951a3fa3bc0875UL, "WNF_RTSC_PRIVACY_SETTINGS_CHANGED"},
            {0x41950c3ea3bc0875UL, "WNF_SBS_UPDATE_AVAILABLE"},
            {0x418b0d3ea3bc0875UL, "WNF_SCM_AUTOSTART_STATE"},
            {0x41890a3ea3bc0875UL, "WNF_SDO_ORIENTATION_CHANGE"},
            {0x41840b3ea3bd7075UL, "WNF_SEB_AIRPLANE_MODE_DISABLED_FOR_EMERGENCY_CALL"},
            {0x41840b3ea3bd1075UL, "WNF_SEB_APP_LAUNCH_PREFETCH"},
            {0x41840b3ea3bd2075UL, "WNF_SEB_APP_RESUME"},
            {0x41840b3ea3bdb075UL, "WNF_SEB_AUDIO_ACTIVITY"},
            {0x41840b3ea3bc8875UL, "WNF_SEB_BACKGROUND_WORK_COST_CHANGE"},
            {0x41840b3ea3bc9075UL, "WNF_SEB_BACKGROUND_WORK_COST_HIGH"},
            {0x41840b3ea3bc5075UL, "WNF_SEB_BATTERY_LEVEL"},
            {0x41840b3ea3bc6075UL, "WNF_SEB_BOOT"},
            {0x41840b3ea3bcc875UL, "WNF_SEB_CACHED_FILE_UPDATED"},
            {0x41840b3ea3bd6075UL, "WNF_SEB_CALL_HISTORY_CHANGED"},
            {0x41840b3ea3bd5075UL, "WNF_SEB_CALL_STATE_CHANGED"},
            {0x41840b3ea3bd9875UL, "WNF_SEB_DEFAULT_SIGN_IN_ACCOUNT_CHANGE"},
            {0x41840b3ea3bd1875UL, "WNF_SEB_DEPRECATED1"},
            {0x41840b3ea3bd2875UL, "WNF_SEB_DEPRECATED2"},
            {0x41840b3ea3bd3075UL, "WNF_SEB_DEPRECATED3"},
            {0x41840b3ea3bd3875UL, "WNF_SEB_DEPRECATED4"},
            {0x41840b3ea3bd4075UL, "WNF_SEB_DEPRECATED5"},
            {0x41840b3ea3bd4875UL, "WNF_SEB_DEPRECATED6"},
            {0x41840b3ea3bce075UL, "WNF_SEB_DEPRECATED7"},
            {0x41840b3ea3bce875UL, "WNF_SEB_DEPRECATED8"},
            {0x41840b3ea3bcb875UL, "WNF_SEB_DEV_MNF_CUSTOM_NOTIFICATION_RECEIVED"},
            {0x41840b3ea3bc5875UL, "WNF_SEB_DOMAIN_JOINED"},
            {0x41840b3ea3bc1075UL, "WNF_SEB_FREE_NETWORK_PRESENT"},
            {0x41840b3ea3bdb875UL, "WNF_SEB_FULL_SCREEN_HDR_VIDEO_PLAYBACK"},
            {0x41840b3ea3bd0075UL, "WNF_SEB_FULL_SCREEN_VIDEO_PLAYBACK"},
            {0x41840b3ea3bdd875UL, "WNF_SEB_GAME_MODE"},
            {0x41840b3ea3bcb075UL, "WNF_SEB_GEOLOCATION"},
            {0x41840b3ea3bde075UL, "WNF_SEB_INCOMING_CALL_DISMISSED"},
            {0x41840b3ea3bc0875UL, "WNF_SEB_INTERNET_PRESENT"},
            {0x41840b3ea3bc8075UL, "WNF_SEB_IP_ADDRESS_AVAILABLE"},
            {0x41840b3ea3bd6875UL, "WNF_SEB_LINE_CHANGED"},
            {0x41840b3ea3bcf075UL, "WNF_SEB_LOW_LATENCY_POWER_REQUEST"},
            {0x41840b3ea3bc2875UL, "WNF_SEB_MBAE_NOTIFICATION_RECEIVED"},
            {0x41840b3ea3bdd075UL, "WNF_SEB_MIXED_REALITY"},
            {0x41840b3ea3bd9075UL, "WNF_SEB_MOBILE_BROADBAND_DEVICE_SERVICE_NOTIFICATION"},
            {0x41840b3ea3bdc875UL, "WNF_SEB_MOBILE_BROADBAND_PCO_VALUE_CHANGE"},
            {0x41840b3ea3bd8875UL, "WNF_SEB_MOBILE_BROADBAND_PIN_LOCK_STATE_CHANGE"},
            {0x41840b3ea3bd8075UL, "WNF_SEB_MOBILE_BROADBAND_RADIO_STATE_CHANGE"},
            {0x41840b3ea3bd7875UL, "WNF_SEB_MOBILE_BROADBAND_REGISTRATION_STATE_CHANGE"},
            {0x41840b3ea3bcc075UL, "WNF_SEB_MOB_OPERATOR_CUSTOM_NOTIFICATION_RECEIVED"},
            {0x41840b3ea3bc7875UL, "WNF_SEB_MONITOR_ON"},
            {0x41840b3ea3bda075UL, "WNF_SEB_NETWORK_CONNECTIVITY_IN_STANDBY"},
            {0x41840b3ea3bc3075UL, "WNF_SEB_NETWORK_CONTROL_CHANNEL_TRIGGER_RESET"},
            {0x41840b3ea3bc2075UL, "WNF_SEB_NETWORK_STATE_CHANGES"},
            {0x41840b3ea3bd0875UL, "WNF_SEB_NFC_PERF_BOOST"},
            {0x41840b3ea3bc4075UL, "WNF_SEB_ONLINE_ID_CONNECTED_STATE_CHANGE"},
            {0x41840b3ea3bcf875UL, "WNF_SEB_RESILIENCY_NOTIFICATION_PHASE"},
            {0x41840b3ea3bcd075UL, "WNF_SEB_SMART_CARD_FIELD_INFO_NOTIFICATION"},
            {0x41840b3ea3bcd875UL, "WNF_SEB_SMART_CARD_HCE_APPLICATION_ACTIVATION_NOTIFICATION"},
            {0x41840b3ea3bca075UL, "WNF_SEB_SMART_CARD_TRANSACTION_NOTIFICATION"},
            {0x41840b3ea3bc1875UL, "WNF_SEB_SMS_RECEIVED"},
            {0x41840b3ea3bc7075UL, "WNF_SEB_SYSTEM_AC"},
            {0x41840b3ea3bc4875UL, "WNF_SEB_SYSTEM_IDLE"},
            {0x41840b3ea3bc9875UL, "WNF_SEB_SYSTEM_LPE"},
            {0x41840b3ea3bca875UL, "WNF_SEB_SYSTEM_MAINTENANCE"},
            {0x41840b3ea3bc3875UL, "WNF_SEB_TIME_ZONE_CHANGE"},
            {0x41840b3ea3bda875UL, "WNF_SEB_USER_PRESENCE_CHANGED"},
            {0x41840b3ea3bc6875UL, "WNF_SEB_USER_PRESENT"},
            {0x41840b3ea3bdc075UL, "WNF_SEB_UWP_APP_LAUNCH"},
            {0x41840b3ea3bd5875UL, "WNF_SEB_VOICEMAIL_CHANGED"},
            {0x4187083ea3bc0875UL, "WNF_SFA_AUTHENTICATION_STAGE_CHANGED"},
            {0xd83063ea3bd9835UL, "WNF_SHEL_ABOVE_LOCK_APP_ACTIVE"},
            {0xd83063ea3bda835UL, "WNF_SHEL_ABOVE_LOCK_BIO_ACTIVE"},
            {0xd83063ea3bf9835UL, "WNF_SHEL_ACTIONCENTER_READY"},
            {0xd83063ea3bed035UL, "WNF_SHEL_ACTIONCENTER_VIEWSTATE_CHANGED"},
            {0xd83063ea3bdd875UL, "WNF_SHEL_APPLICATION_SPATIAL_INFO_UPDATE"},
            {0xd83063ea3be0075UL, "WNF_SHEL_APPLICATION_STARTED"},
            {0xd83063ea3bc7075UL, "WNF_SHEL_APPLICATION_STATE_UPDATE"},
            {0xd83063ea3be0875UL, "WNF_SHEL_APPLICATION_TERMINATED"},
            {0xd83063ea3bee875UL, "WNF_SHEL_APPLIFECYCLE_INSTALL_STATE"},
            {0xd83063ea3bc5075UL, "WNF_SHEL_APPRESOLVER_SCAN"},
            {0xd83063ea3bf8875UL, "WNF_SHEL_ASSISTANT_STATE_CHANGE"},
            {0xd83063ea3bed875UL, "WNF_SHEL_CACHED_CLOUD_NETWORK_STATE"},
            {0xd83063ea3bdb875UL, "WNF_SHEL_CALM_DISPLAY_ACTIVE"},
            {0xd83063ea3bdf875UL, "WNF_SHEL_CDM_FEATURE_CONFIG_FIRST_USAGE"},
            {0xd83063ea3be9075UL, "WNF_SHEL_CDM_FEATURE_USAGE"},
            {0xd83063ea3be6835UL, "WNF_SHEL_CDM_REGISTRATION_COMPLETE"},
            {0xd83063ea3bea875UL, "WNF_SHEL_CLOUD_FILE_INDEXED_CHANGE"},
            {0xd83063ea3beb075UL, "WNF_SHEL_CLOUD_FILE_PROGRESS_CHANGE"},
            {0xd83063ea3be70f5UL, "WNF_SHEL_CONTENT_DELIVERY_MANAGER_MONITORING"},
            {0xd83063ea3be4875UL, "WNF_SHEL_CONTENT_DELIVERY_MANAGER_NEEDS_REMEDIATION"},
            {0xd83063ea3bc9875UL, "WNF_SHEL_CORTANA_APPINDEX_UPDATED"},
            {0xd83063ea3bde075UL, "WNF_SHEL_CORTANA_AUDIO_ACTIVE"},
            {0xd83063ea3bf1075UL, "WNF_SHEL_CORTANA_BEACON_STATE_CHANGED"},
            {0xd83063ea3bf7035UL, "WNF_SHEL_CORTANA_CAPABILTIES_CHANGED"},
            {0xd83063ea3bf0475UL, "WNF_SHEL_CORTANA_QUIET_MOMENT_AT_HOME"},
            {0xd83063ea3bdb035UL, "WNF_SHEL_CORTANA_SPEECH_CANCELHANDSFREE_REQUESTED"},
            {0xd83063ea3bf3075UL, "WNF_SHEL_CREATIVE_EVENT_BATTERY_SAVER_OVERRIDE_TRIGGERED"},
            {0xd83063ea3bcd875UL, "WNF_SHEL_CREATIVE_EVENT_TRIGGERED"},
            {0xd83063ea3bd2075UL, "WNF_SHEL_DDC_COMMAND_AVAILABLE"},
            {0xd83063ea3bd6075UL, "WNF_SHEL_DDC_CONNECTED_ACCOUNTS_CHANGED"},
            {0xd83063ea3bd3075UL, "WNF_SHEL_DDC_SMS_COMMAND"},
            {0xd83063ea3bd2875UL, "WNF_SHEL_DDC_WNS_COMMAND"},
            {0xd83063ea3be5075UL, "WNF_SHEL_DESKTOP_APPLICATION_STARTED"},
            {0xd83063ea3be5875UL, "WNF_SHEL_DESKTOP_APPLICATION_TERMINATED"},
            {0xd83063ea3bd3875UL, "WNF_SHEL_DEVICE_LOCKED"},
            {0xd83063ea3bf2875UL, "WNF_SHEL_DEVICE_OPEN"},
            {0xd83063ea3bcc075UL, "WNF_SHEL_DEVICE_UNLOCKED"},
            {0xd83063ea3bd1835UL, "WNF_SHEL_DICTATION_RUNNING"},
            {0xd83063ea3be8075UL, "WNF_SHEL_ENTERPRISE_HIDE_PEOPLE_BAR_POLICY_VALUE_CHANGED"},
            {0xd83063ea3bc9475UL, "WNF_SHEL_ENTERPRISE_START_LAYOUT_POLICY_VALUE_CHANGED"},
            {0xd83063ea3bec075UL, "WNF_SHEL_ENTERPRISE_START_PLACES_POLICY_VALUE_CHANGED"},
            {0xd83063ea3bc7875UL, "WNF_SHEL_FOCUS_CHANGE"},
            {0xd83063ea3bc8875UL, "WNF_SHEL_GAMECONTROLLER_FOCUS_INFO"},
            {0xd83063ea3bc8075UL, "WNF_SHEL_GAMECONTROLLER_LISTENER_INFO"},
            {0xd83063ea3bcf075UL, "WNF_SHEL_GAMECONTROLLER_NEXUS_INFO"},
            {0xd83063ea3be4075UL, "WNF_SHEL_HEALTH_STATE_CHANGED"},
            {0xd83063ea3bc0875UL, "WNF_SHEL_IMMERSIVE_SHELL_RUNNING"},
            {0xd83063ea3bdc075UL, "WNF_SHEL_INSTALL_PLACEHOLDER_TILES"},
            {0xd83063ea3bce075UL, "WNF_SHEL_JUMPLIST_CHANGED"},
            {0xd83063ea3bef875UL, "WNF_SHEL_LATEST_CONNECTED_AUTOPLAY_DEVICE"},
            {0xd83063ea3bf6835UL, "WNF_SHEL_LOCKAPPHOST_ACTIVE"},
            {0xd83063ea3bc5835UL, "WNF_SHEL_LOCKSCREEN_ACTIVE"},
            {0xd83063ea3bd5075UL, "WNF_SHEL_LOCKSCREEN_IMAGE_CHANGED"},
            {0xd83063ea3bde835UL, "WNF_SHEL_LOCKSCREEN_INFO_UPDATED"},
            {0xd83063ea3bdd075UL, "WNF_SHEL_LOCKSTATE"},
            {0xd83063ea3be3035UL, "WNF_SHEL_LOCK_APP_READY"},
            {0xd83063ea3be2835UL, "WNF_SHEL_LOCK_APP_RELOCK"},
            {0xd83063ea3bd7835UL, "WNF_SHEL_LOCK_APP_REQUESTING_UNLOCK"},
            {0xd83063ea3bd7035UL, "WNF_SHEL_LOCK_APP_SHOWN"},
            {0xd83063ea3bf2035UL, "WNF_SHEL_LOCK_ON_LOGON"},
            {0xd83063ea3bc1875UL, "WNF_SHEL_LOGON_COMPLETE"},
            {0xd83063ea3bf5875UL, "WNF_SHEL_NEXT_NOTIFICATION_SINK_SESSION_ID"},
            {0xd83063ea3bc1035UL, "WNF_SHEL_NOTIFICATIONS"},
            {0xd83063ea3bca835UL, "WNF_SHEL_NOTIFICATIONS_CRITICAL"},
            {0xd83063ea3bc3835UL, "WNF_SHEL_NOTIFICATION_SETTINGS_CHANGED"},
            {0xd83063ea3bd6835UL, "WNF_SHEL_OOBE_ENABLE_PROVISIONING"},
            {0xd83063ea3be9c75UL, "WNF_SHEL_OOBE_PROVISIONING_COMPLETE"},
            {0xd83063ea3bc2475UL, "WNF_SHEL_OOBE_USER_LOGON_COMPLETE"},
            {0xd83063ea3bfa835UL, "WNF_SHEL_PASTE_SUGGESTION_MESSAGE"},
            {0xd83063ea3be2035UL, "WNF_SHEL_PEOPLE_PANE_VIEW_CHANGED"},
            {0xd83063ea3bdc835UL, "WNF_SHEL_PEOPLE_PINNED_LIST_CHANGED"},
            {0xd83063ea3bcc875UL, "WNF_SHEL_PLACES_CHANGED"},
            {0xd83063ea3bf1c75UL, "WNF_SHEL_QUIETHOURS_ACTIVE_PROFILE_CHANGED"},
            {0xd83063ea3bf5075UL, "WNF_SHEL_QUIET_MOMENT_SHELL_MODE_CHANGED"},
            {0xd83063ea3bda035UL, "WNF_SHEL_RADIALCONTROLLER_EXPERIENCE_RESTART"},
            {0xd83063ea3bd1075UL, "WNF_SHEL_REQUEST_CORTANA_SETTINGSCONSTRAINTINDEX_BUILD"},
            {0xd83063ea3bef075UL, "WNF_SHEL_RESTORE_PAYLOAD_COMPLETE"},
            {0xd83063ea3bd5875UL, "WNF_SHEL_SCREEN_COVERED"},
            {0xd83063ea3be3835UL, "WNF_SHEL_SESSION_LOGON_COMPLETE"},
            {0xd83063ea3bcf875UL, "WNF_SHEL_SETTINGS_CHANGED"},
            {0xd83063ea3bf4875UL, "WNF_SHEL_SETTINGS_ENVIRONMENT_CHANGED"},
            {0xd83063ea3bfa075UL, "WNF_SHEL_SIGNALMANAGER_SIGNAL_REGISTRATION_UPDATED"},
            {0xd83063ea3be7835UL, "WNF_SHEL_SIGNAL_LOGONUI"},
            {0xd83063ea3bec875UL, "WNF_SHEL_SIGNAL_MANAGER_FEATURE_TRIGGERED"},
            {0xd83063ea3bea075UL, "WNF_SHEL_SIGNAL_MANAGER_SIGNAL_TRIGGERED"},
            {0xd83063ea3bee075UL, "WNF_SHEL_SIGNAL_MANAGER_TESTING"},
            {0xd83063ea3bd0835UL, "WNF_SHEL_SOFTLANDING_PUBLISHED"},
            {0xd83063ea3bca075UL, "WNF_SHEL_SOFTLANDING_RULES_UPDATED"},
            {0xd83063ea3bc4075UL, "WNF_SHEL_SOFTLANDING_RULE_TRIGGERED"},
            {0xd83063ea3bc6875UL, "WNF_SHEL_START_APPLIFECYCLE_DOWNLOAD_STARTED"},
            {0xd83063ea3bc6075UL, "WNF_SHEL_START_APPLIFECYCLE_INSTALL_FINISHED"},
            {0xd83063ea3bce875UL, "WNF_SHEL_START_APPLIFECYCLE_UNINSTALL_FINISHED"},
            {0xd83063ea3beb8f5UL, "WNF_SHEL_START_LAYOUT_MIGRATED"},
            {0xd83063ea3bc4875UL, "WNF_SHEL_START_LAYOUT_READY"},
            {0xd83063ea3bf3835UL, "WNF_SHEL_START_PROCESS_SUSPENDED_INTERNAL"},
            {0xd83063ea3bcb035UL, "WNF_SHEL_START_VISIBILITY_CHANGED"},
            {0xd83063ea3be60f5UL, "WNF_SHEL_SUGGESTED_APP_READY"},
            {0xd83063ea3bcd075UL, "WNF_SHEL_SUSPEND_APP_BACKGROUND_ACTIVITY"},
            {0xd83063ea3bf4035UL, "WNF_SHEL_SYSTEMDIALOG_PUBLISHED"},
            {0xd83063ea3bf6035UL, "WNF_SHEL_TAB_SHELL_INIT_COMPLETE"},
            {0xd83063ea3bd4075UL, "WNF_SHEL_TARGETED_CONTENT_SUBSCRIPTION_ACTIVATED"},
            {0xd83063ea3bd4875UL, "WNF_SHEL_TARGETED_CONTENT_SUBSCRIPTION_UPDATED"},
            {0xd83063ea3bf7875UL, "WNF_SHEL_TASKBAR_PINS_UPDATED"},
            {0xd83063ea3bc3075UL, "WNF_SHEL_TILECHANGE"},
            {0xd83063ea3bd8075UL, "WNF_SHEL_TILEINSTALL"},
            {0xd83063ea3bd9075UL, "WNF_SHEL_TILEUNINSTALL"},
            {0xd83063ea3bd8875UL, "WNF_SHEL_TILEUPDATE"},
            {0xd83063ea3bd0035UL, "WNF_SHEL_TOAST_PUBLISHED"},
            {0xd83063ea3bf9075UL, "WNF_SHEL_TOAST_PUBLISHED_SYSTEMSCOPE"},
            {0xd83063ea3bcb875UL, "WNF_SHEL_TRAY_SEARCHBOX_VISIBILITY_CHANGED"},
            {0xd83063ea3be1875UL, "WNF_SHEL_USER_IDLE"},
            {0xd83063ea3bc2875UL, "WNF_SHEL_VEEVENT_DISPATCHER_CLIENT_PIPE_CLOSED"},
            {0xd83063ea3bf8075UL, "WNF_SHEL_WCOS_SESSION_ID"},
            {0xd83063ea3be10f5UL, "WNF_SHEL_WINDOWSTIP_CONTENT_PUBLISHED"},
            {0xd83063ea3bfb035UL, "WNF_SHEL_WINDOW_ACTIVATED"},
            {0x4194063ea3bc1875UL, "WNF_SHR_DHCP_IPv4_FASTIP_ADDRS"},
            {0x4194063ea3bc1075UL, "WNF_SHR_DHCP_IPv4_LEASE_LIST"},
            {0x4194063ea3bc0835UL, "WNF_SHR_SHARING_CHANGED"},
            {0x4189073ea3bc1075UL, "WNF_SIO_BIO_ENROLLED"},
            {0x4189073ea3bc0875UL, "WNF_SIO_PIN_ENROLLED"},
            {0x59f053ea3bc0875UL, "WNF_SKYD_FILE_SYNC"},
            {0x59f053ea3bc1075UL, "WNF_SKYD_QUOTA_CHANGE"},
            {0x1395033ea3bc1875UL, "WNF_SMSR_NEW_MESSAGE_RECEIVED"},
            {0x1395033ea3bc0875UL, "WNF_SMSR_READY"},
            {0x1395033ea3bc1075UL, "WNF_SMSR_WWAN_READ_DONE"},
            {0x1295033ea3bc0875UL, "WNF_SMSS_MEMORY_COOLING_COMPATIBLE"},
            {0x4195033ea3bc0875UL, "WNF_SMS_CHECK_ACCESS"},
            {0x2871e3ea3bc0875UL, "WNF_SPAC_SPACEPORT_PROPERTY_CHANGED"},
            {0x2871e3ea3bc1075UL, "WNF_SPAC_SPACEPORT_WORK_REQUESTED"},
            {0x9851e3ea3bc2075UL, "WNF_SPCH_ALLOW_REMOTE_SPEECH_SERVICES"},
            {0x9851e3ea3bc1875UL, "WNF_SPCH_DISABLE_KWS_REQUEST"},
            {0x9851e3ea3bc0835UL, "WNF_SPCH_INPUT_STATE_UPDATE"},
            {0x9851e3ea3bc1075UL, "WNF_SPCH_REMOTE_SESSION_REQUEST"},
            {0x418f1e3ea3bc0835UL, "WNF_SPI_LOGICALDPIOVERRIDE"},
            {0x418f1e3ea3bc1035UL, "WNF_SPI_PRIMARY_MONITOR_DPI_CHANGED"},
            {0x41851c3ea3bc0875UL, "WNF_SRC_SYSTEM_RADIO_CHANGED"},
            {0x41921c3ea3bc0875UL, "WNF_SRT_WINRE_CONFIGURATION_CHANGE"},
            {0xc931c3ea3bc0875UL, "WNF_SRUM_SCREENONSTUDY_SESSION"},
            {0x41901c3ea3bc1075UL, "WNF_SRV_SMB1_NOT_IN_USE_STATE_CHANGE"},
            {0x41901c3ea3bc0875UL, "WNF_SRV_SRV2_STATE_CHANGE"},
            {0x13891a3ea3bc0875UL, "WNF_STOR_CONFIGURATION_DEVICE_INFO_UPDATED"},
            {0x13891a3ea3bc1075UL, "WNF_STOR_CONFIGURATION_MO_TASK_RUNNING"},
            {0x13891a3ea3bc1875UL, "WNF_STOR_CONFIGURATION_OEM_TASK_RUNNING"},
            {0x11961b3ea3bc0875UL, "WNF_SUPP_ENABLE_ERROR_DETAILS_CACHE"},
            {0x288173ea3bc0875UL, "WNF_SYNC_REQUEST_PROBE"},
            {0x4195173ea3bc0875UL, "WNF_SYS_SHUTDOWN_IN_PROGRESS"},
            {0x41c60c39a3bc0875UL, "WNF_TB_SYSTEM_TIME_CHANGED"},
            {0xc870b39a3bc0875UL, "WNF_TEAM_SHELL_HOTKEY_PRESSED"},
            {0x418a0b39a3be1075UL, "WNF_TEL_DAILY_UPLOAD_QUOTA"},
            {0x418a0b39a3be1875UL, "WNF_TEL_ONESETTINGS_UPDATED"},
            {0x418a0b39a3be2075UL, "WNF_TEL_SETTINGS_PUSH_NOTIFICATION_RECEIVED"},
            {0x418a0b39a3be0875UL, "WNF_TEL_STORAGE_CAPACITY"},
            {0x418a0b39a3be2875UL, "WNF_TEL_TIMER_RECONFIGURED"},
            {0x9920b39a3bc1075UL, "WNF_TETH_AUTOSTART_BLUETOOTH"},
            {0x9920b39a3bc0875UL, "WNF_TETH_TETHERING_STATE"},
            {0x48b0639a3bc0875UL, "WNF_THME_THEME_CHANGED"},
            {0xf840539a3bc4835UL, "WNF_TKBN_AUTOCOMPLETE"},
            {0xf840539a3bc7835UL, "WNF_TKBN_CANDIDATE_WINDOW_STATE"},
            {0xf840539a3bc4035UL, "WNF_TKBN_CARET_TRACKING"},
            {0xf840539a3bc9035UL, "WNF_TKBN_COMPOSITION_STATE"},
            {0xf840539a3bcb035UL, "WNF_TKBN_DESKTOP_MODE_AUTO_IHM"},
            {0xf840539a3bc3835UL, "WNF_TKBN_FOREGROUND_WINDOW"},
            {0xf840539a3bc1835UL, "WNF_TKBN_IMMERSIVE_FOCUS_TRACKING"},
            {0xf840539a3bca835UL, "WNF_TKBN_INPUT_PANE_DISPLAY_POLICY"},
            {0xf840539a3bc6835UL, "WNF_TKBN_KEYBOARD_GESTURE"},
            {0xf840539a3bc8035UL, "WNF_TKBN_KEYBOARD_LAYOUT_CHANGE"},
            {0xf840539a3bcb835UL, "WNF_TKBN_KEYBOARD_SET_VISIBLE"},
            {0xf840539a3bcc035UL, "WNF_TKBN_KEYBOARD_SET_VISIBLE_NOTIFICATION"},
            {0xf840539a3bc5835UL, "WNF_TKBN_KEYBOARD_VIEW_CHANGE"},
            {0xf840539a3bc0835UL, "WNF_TKBN_KEYBOARD_VISIBILITY"},
            {0xf840539a3bc3035UL, "WNF_TKBN_LANGUAGE"},
            {0xf840539a3bc5035UL, "WNF_TKBN_MODERN_KEYBOARD_FOCUS_TRACKING"},
            {0xf840539a3bc7035UL, "WNF_TKBN_RESTRICTED_KEYBOARD_GESTURE"},
            {0xf840539a3bc8835UL, "WNF_TKBN_RESTRICTED_KEYBOARD_LAYOUT_CHANGE"},
            {0xf840539a3bc6035UL, "WNF_TKBN_RESTRICTED_KEYBOARD_VIEW_CHANGE"},
            {0xf840539a3bc1035UL, "WNF_TKBN_RESTRICTED_KEYBOARD_VISIBILITY"},
            {0xf840539a3bc2835UL, "WNF_TKBN_RESTRICTED_TOUCH_EVENT"},
            {0xf840539a3bc9835UL, "WNF_TKBN_SYSTEM_IMMERSIVE_FOCUS_TRACKING"},
            {0xf840539a3bca035UL, "WNF_TKBN_SYSTEM_TOUCH_EVENT"},
            {0xf840539a3bc2035UL, "WNF_TKBN_TOUCH_EVENT"},
            {0x13840539a3bc1075UL, "WNF_TKBR_CHANGE_APP"},
            {0x13840539a3bc18f5UL, "WNF_TKBR_CHANGE_APP_INTERNAL"},
            {0x13840539a3bc08f5UL, "WNF_TKBR_CHANGE_SYSTEM"},
            {0xf850339a3bc0835UL, "WNF_TMCN_ISTABLETMODE"},
            {0x4960139a3bc0875UL, "WNF_TOPE_INP_POINTER_DEVICE_ACTIVITY"},
            {0x418b1e39a3bc2075UL, "WNF_TPM_CLEAR_PENDING"},
            {0x418b1e39a3bc2875UL, "WNF_TPM_CLEAR_RESULT"},
            {0x418b1e39a3bc1075UL, "WNF_TPM_DEVICEID_STATE"},
            {0x418b1e39a3bc3075UL, "WNF_TPM_DISABLE_DEACTIVATE_PENDING"},
            {0x418b1e39a3bc3875UL, "WNF_TPM_ENABLE_ACTIVATE_COMPLETED"},
            {0x418b1e39a3bc4075UL, "WNF_TPM_MAINTENANCE_TASK_STATUS"},
            {0x418b1e39a3bc0875UL, "WNF_TPM_OWNERSHIP_TAKEN"},
            {0x418b1e39a3bc1875UL, "WNF_TPM_PROVISION_TRIGGER"},
            {0x41c61439a3bc3075UL, "WNF_TZ_AUTOTIMEUPDATE_STATE_CHANGED"},
            {0x41c61439a3bc2075UL, "WNF_TZ_LAST_TIME_SYNC_INFO"},
            {0x41c61439a3bc0875UL, "WNF_TZ_LEGACY_STORE_CHANGED"},
            {0x41c61439a3bc2875UL, "WNF_TZ_NETWORK_TIME_SYNC_TRIGGER"},
            {0x41c61439a3bc1075UL, "WNF_TZ_STORE_CHANGED"},
            {0x41c61439a3bc1875UL, "WNF_TZ_TIMEZONE_CHANGED"},
            {0xc960c38a3bc1075UL, "WNF_UBPM_CONSOLE_MONITOR"},
            {0xc960c38a3bc1875UL, "WNF_UBPM_FRMU_ALLOWED"},
            {0xc960c38a3bc0875UL, "WNF_UBPM_POWER_SOURCE"},
            {0xc960c38a3bc2075UL, "WNF_UBPM_PRESHUTDOWN_PHASE"},
            {0x41870a38a3bc2835UL, "WNF_UDA_CONTACT_SORT_CHANGED"},
            {0x418b0a38a3bc0835UL, "WNF_UDM_SERVICE_INITIALIZED"},
            {0x7820338a3bc1075UL, "WNF_UMDF_DRVMGR_STATUS"},
            {0x7820338a3bc0875UL, "WNF_UMDF_WUDFSVC_START"},
            {0x13810338a3bc2875UL, "WNF_UMGR_SESSIONUSER_TOKEN_CHANGE"},
            {0x13810338a3bc3035UL, "WNF_UMGR_SESSION_ACTIVE_SHELL_USER_CHANGE"},
            {0x13810338a3bc0835UL, "WNF_UMGR_SIHOST_READY"},
            {0x13810338a3bc2075UL, "WNF_UMGR_SYSTEM_USER_CONTEXT_CHANGED"},
            {0x13810338a3bc1075UL, "WNF_UMGR_USER_LOGIN"},
            {0x13810338a3bc1875UL, "WNF_UMGR_USER_LOGOUT"},
            {0x13810338a3bc3875UL, "WNF_UMGR_USER_TILE_CHANGED"},
            {0x41841d38a3bc1075UL, "WNF_USB_BILLBOARD_CHANGE"},
            {0x41841d38a3bc2075UL, "WNF_USB_CHARGING_STATE"},
            {0x41841d38a3bc3075UL, "WNF_USB_ERROR_NOTIFICATION"},
            {0x41841d38a3bc2875UL, "WNF_USB_FUNCTION_CONTROLLER_STATE"},
            {0x41841d38a3bc1875UL, "WNF_USB_PEER_DEVICE_STATE"},
            {0x41841d38a3bc3875UL, "WNF_USB_POLICY_MANAGER_HUB_COLLECTION_STATE"},
            {0x41841d38a3bc0875UL, "WNF_USB_TYPE_C_PARTNER_STATE"},
            {0x41841d38a3bc4075UL, "WNF_USB_XHCI_AUDIO_OFFLOAD_STATE"},
            {0x41891d38a3bc7075UL, "WNF_USO_ACTIVEHOURS_STARTED"},
            {0x41891d38a3bc2875UL, "WNF_USO_ACTIVE_SESSION"},
            {0x41891d38a3bc4875UL, "WNF_USO_DOWNLOAD_STARTED"},
            {0x41891d38a3bc5075UL, "WNF_USO_INSTALL_STARTED"},
            {0x41891d38a3bc5875UL, "WNF_USO_INSTALL_STATE"},
            {0x41891d38a3bc4075UL, "WNF_USO_REBOOT_BLOCK_REQUESTED"},
            {0x41891d38a3bc2075UL, "WNF_USO_REBOOT_REQUIRED"},
            {0x41891d38a3bc6075UL, "WNF_USO_SERVICE_STOPPING"},
            {0x41891d38a3bc6875UL, "WNF_USO_SETTINGS_REFRESHED"},
            {0x41891d38a3bc1075UL, "WNF_USO_STATE_ATTENTION_REQUIRED"},
            {0x41891d38a3bc0875UL, "WNF_USO_STATE_CHANGE"},
            {0x41891d38a3bc1875UL, "WNF_USO_UPDATE_PROGRESS"},
            {0x41891d38a3bc3075UL, "WNF_USO_UPDATE_SUCCEEDED"},
            {0x41891d38a3bc3875UL, "WNF_USO_UPTODATE_STATUS_CHANGED"},
            {0x41951a38a3bc1475UL, "WNF_UTS_LOCKSCREEN_DISMISSAL_TRIGGERED"},
            {0x41951a38a3bc0c75UL, "WNF_UTS_USERS_ENROLLED"},
            {0x41801938a3bc1075UL, "WNF_UWF_OVERLAY_CRITICAL"},
            {0x41801938a3bc1875UL, "WNF_UWF_OVERLAY_NORMAL"},
            {0x41801938a3bc0875UL, "WNF_UWF_OVERLAY_WARNING"},
            {0x41880f3ba3bc0875UL, "WNF_VAN_VANUI_STATUS"},
            {0x41881e3ba3bc0875UL, "WNF_VPN_CLIENT_CONNECTIVITY_STATUS"},
            {0x17951a3ba3bc1075UL, "WNF_VTSV_ADD_CRED_NOTIFY"},
            {0x17951a3ba3bc0875UL, "WNF_VTSV_CDS_SYNC"},
            {0x12870f3aa3bc1075UL, "WNF_WAAS_FEATURE_IMPACT"},
            {0x12870f3aa3bc0875UL, "WNF_WAAS_QUALITY_IMPACT"},
            {0xe8f0c3aa3bc0875UL, "WNF_WBIO_ENROLLMENT_FINISHED"},
            {0x12820d3aa3bc0875UL, "WNF_WCDS_SYNC_WLAN"},
            {0x418b0d3aa3bc2875UL, "WNF_WCM_INTERFACE_CONNECTION_STATE"},
            {0x418b0d3aa3bc0875UL, "WNF_WCM_INTERFACE_LIST"},
            {0x418b0d3aa3bc1875UL, "WNF_WCM_MAPPING_POLICY_UPDATED"},
            {0x418b0d3aa3bc2075UL, "WNF_WCM_PROFILE_CONFIG_UPDATED"},
            {0x418b0d3aa3bc1075UL, "WNF_WCM_SERVICE_STATUS"},
            {0x6870a3aa3bc1075UL, "WNF_WDAG_SETTINGS_CHANGED_SYSTEM"},
            {0x6870a3aa3bc0875UL, "WNF_WDAG_SETTINGS_CHANGED_USER"},
            {0x2950a3aa3bc0875UL, "WNF_WDSC_ACCOUNT_PROTECTION_REFRESH"},
            {0x840b3aa3bc1075UL, "WNF_WEBA_CTAP_DEVICE_CHANGE_NOTIFY"},
            {0x840b3aa3bc0875UL, "WNF_WEBA_CTAP_DEVICE_STATE"},
            {0x41940b3aa3bc1875UL, "WNF_WER_CRASH_STATE"},
            {0x41940b3aa3bc1075UL, "WNF_WER_QUEUED_REPORTS"},
            {0x41940b3aa3bc0875UL, "WNF_WER_SERVICE_START"},
            {0x1287083aa3bc0875UL, "WNF_WFAS_FIREWALL_NETWORK_CHANGE_READY"},
            {0xf82083aa3bc1075UL, "WNF_WFDN_MOVEMENT_DETECTED"},
            {0xf82083aa3bc1875UL, "WNF_WFDN_STAY_CONNECTED_TRIGGER"},
            {0xf82083aa3bc0875UL, "WNF_WFDN_WFD_DISCONNECTION_PROPERTIES"},
            {0x4195083aa3bc1875UL, "WNF_WFS_FAMILYMEMBERLOGIN"},
            {0x4195083aa3bc0875UL, "WNF_WFS_SETTINGS"},
            {0x4195083aa3bc2075UL, "WNF_WFS_SETTINGSREFRESH"},
            {0x4195083aa3bc1075UL, "WNF_WFS_TIMEREMAININGALERTS"},
            {0x1192063aa3bc1075UL, "WNF_WHTP_WINHTTP_PROXY_AUTHENTICATION_REQUIRED"},
            {0x1192063aa3bc0875UL, "WNF_WHTP_WINHTTP_PROXY_DISCOVERED"},
            {0x880073aa3bc4875UL, "WNF_WIFI_AOAC_STATUS"},
            {0x880073aa3bc6875UL, "WNF_WIFI_AVERAGE_TRANSMIT"},
            {0x880073aa3bc5875UL, "WNF_WIFI_CONNECTION_SCORE"},
            {0x880073aa3bc0875UL, "WNF_WIFI_CONNECTION_STATUS"},
            {0x880073aa3bc1075UL, "WNF_WIFI_CPL_STATUS"},
            {0x880073aa3bc9075UL, "WNF_WIFI_HOTSPOT2_REGISTRATION_STATUS"},
            {0x880073aa3bc2875UL, "WNF_WIFI_HOTSPOT_HOST_READY"},
            {0x880073aa3bc8075UL, "WNF_WIFI_L3_AUTH_STATE"},
            {0x880073aa3bc7075UL, "WNF_WIFI_MEDIA_STREAMING_MODE"},
            {0x880073aa3bca075UL, "WNF_WIFI_MOVEMENT_DETECTED"},
            {0x880073aa3bc9875UL, "WNF_WIFI_PROTECTED_SCENARIO"},
            {0x880073aa3bc2075UL, "WNF_WIFI_SERVICE_NOTIFICATIONS"},
            {0x880073aa3bc7875UL, "WNF_WIFI_TASK_TRIGGER"},
            {0x880073aa3bc6075UL, "WNF_WIFI_TILE_UPDATE"},
            {0x880073aa3bc8875UL, "WNF_WIFI_WLANSVC_NOTIFICATION"},
            {0x418a073aa3bc1475UL, "WNF_WIL_BOOT_FEATURE_STORE"},
            {0x418a073aa3bc1c75UL, "WNF_WIL_FEATURE_DEVICE_USAGE_TRACKING_1"},
            {0x418a073aa3bc2475UL, "WNF_WIL_FEATURE_DEVICE_USAGE_TRACKING_2"},
            {0x418a073aa3bc2c75UL, "WNF_WIL_FEATURE_DEVICE_USAGE_TRACKING_3"},
            {0x418a073aa3bc4c75UL, "WNF_WIL_FEATURE_HEALTH_TRACKING_1"},
            {0x418a073aa3bc5475UL, "WNF_WIL_FEATURE_HEALTH_TRACKING_2"},
            {0x418a073aa3bc5c75UL, "WNF_WIL_FEATURE_HEALTH_TRACKING_3"},
            {0x418a073aa3bc6475UL, "WNF_WIL_FEATURE_HEALTH_TRACKING_4"},
            {0x418a073aa3bc6c75UL, "WNF_WIL_FEATURE_HEALTH_TRACKING_5"},
            {0x418a073aa3bc7475UL, "WNF_WIL_FEATURE_HEALTH_TRACKING_6"},
            {0x418a073aa3bc0c75UL, "WNF_WIL_FEATURE_STORE"},
            {0x418a073aa3bc9835UL, "WNF_WIL_FEATURE_USAGE_FOR_SRUM"},
            {0x418a073aa3bc3475UL, "WNF_WIL_FEATURE_USAGE_TRACKING_1"},
            {0x418a073aa3bc3c75UL, "WNF_WIL_FEATURE_USAGE_TRACKING_2"},
            {0x418a073aa3bc4475UL, "WNF_WIL_FEATURE_USAGE_TRACKING_3"},
            {0x418a073aa3bc7c75UL, "WNF_WIL_MACHINE_FEATURE_STORE"},
            {0x418a073aa3bc8075UL, "WNF_WIL_MACHINE_FEATURE_STORE_MODIFIED"},
            {0x418a073aa3bc88f5UL, "WNF_WIL_USER_FEATURE_STORE"},
            {0x418a073aa3bc90f5UL, "WNF_WIL_USER_FEATURE_STORE_MODIFIED"},
            {0x4195003aa3bc0875UL, "WNF_WNS_CONNECTIVITY_STATUS"},
            {0x4180013aa3bc0875UL, "WNF_WOF_OVERLAY_CONFIGURATION_CHANGE"},
            {0x295013aa3bc2075UL, "WNF_WOSC_DIRECTX_DATABASE_CHANGED"},
            {0x295013aa3bc1075UL, "WNF_WOSC_FEATURE_CONFIGURATION_CHANGED"},
            {0x295013aa3bc3075UL, "WNF_WOSC_FEATURE_CONFIGURATION_COMPLETED"},
            {0x295013aa3bc1875UL, "WNF_WOSC_MITIGATION_CONFIGURATION_CHANGED"},
            {0x295013aa3bc0875UL, "WNF_WOSC_ML_MODELS_CHANGED"},
            {0x295013aa3bc2875UL, "WNF_WOSC_MUSE_CONFIGURATION_CHANGED"},
            {0x41881e3aa3bc2875UL, "WNF_WPN_CHANNELS_REVOKED"},
            {0x41881e3aa3bc10f5UL, "WNF_WPN_PLATFORM_INITIALIZED"},
            {0x41881e3aa3bc1875UL, "WNF_WPN_SYSTEM_PLATFORM_READY"},
            {0x41881e3aa3bc2035UL, "WNF_WPN_USER_IN_SESSION_PLATFORM_READY"},
            {0x41881e3aa3bc08f5UL, "WNF_WPN_USER_PLATFORM_READY"},
            {0x41851d3aa3bc0875UL, "WNF_WSC_SECURITY_CENTER_USER_NOTIFICATION"},
            {0xc971d3aa3bc0875UL, "WNF_WSQM_IS_OPTED_IN"},
            {0x41871b3aa3bc1075UL, "WNF_WUA_AU_SCAN_COMPLETE"},
            {0x41871b3aa3bc1875UL, "WNF_WUA_CALL_HANG"},
            {0x41871b3aa3bc08f5UL, "WNF_WUA_NUM_PER_USER_UPDATES"},
            {0x41871b3aa3bc2075UL, "WNF_WUA_SERVICE_HANG"},
            {0x41871b3aa3bc2875UL, "WNF_WUA_STAGEUPDATE_DETAILS"},
            {0x41871b3aa3bc3075UL, "WNF_WUA_UPDATE_EXPIRING"},
            {0xf87193aa3bc1875UL, "WNF_WWAN_CELLULAR_STATE_SNAPSHOT_CHANGE"},
            {0xf87193aa3bc1075UL, "WNF_WWAN_EUICC_ARRIVAL"},
            {0xf87193aa3bc0875UL, "WNF_WWAN_OBJECT_LIST"},
            {0xf87193aa3bc2075UL, "WNF_WWAN_TASK_TRIGGER"},
            {0x19890c35a3be9075UL, "WNF_XBOX_ACCESSIBILITY_EXCLUSIVE_INPUT_MODE_CHANGED"},
            {0x19890c35a3bdf075UL, "WNF_XBOX_ACCESSIBILITY_NARRATOR_ENABLED"},
            {0x19890c35a3bc8075UL, "WNF_XBOX_ACHIEVEMENTS_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bea075UL, "WNF_XBOX_ACHIEVEMENT_TRACKER_STATE_CHANGED"},
            {0x19890c35a3be5875UL, "WNF_XBOX_ACTIVE_BACKGROUNDAUDIO_APPLICATION_CHANGED"},
            {0x19890c35a3be3075UL, "WNF_XBOX_ADJUST_SNAP_CPU_AFFINITY"},
            {0x19890c35a3bc1875UL, "WNF_XBOX_APPLICATION_ACTIVATING"},
            {0x19890c35a3bc2075UL, "WNF_XBOX_APPLICATION_COMPONENT_FOCUS"},
            {0x19890c35a3bcd875UL, "WNF_XBOX_APPLICATION_COM_RESILIENCY_STATUS_CHANGED"},
            {0x19890c35a3bc0875UL, "WNF_XBOX_APPLICATION_CONTEXT_CHANGED"},
            {0x19890c35a3be0075UL, "WNF_XBOX_APPLICATION_CURRENT_USER_CHANGED"},
            {0x19890c35a3bc6075UL, "WNF_XBOX_APPLICATION_ERROR"},
            {0x19890c35a3bc1075UL, "WNF_XBOX_APPLICATION_FOCUS_CHANGED"},
            {0x19890c35a3bc9075UL, "WNF_XBOX_APPLICATION_LAYOUT_CHANGED"},
            {0x19890c35a3bd0075UL, "WNF_XBOX_APPLICATION_LICENSE_CHANGED"},
            {0x19890c35a3bc5075UL, "WNF_XBOX_APPLICATION_NO_LONGER_RUNNING"},
            {0x19890c35a3bc5875UL, "WNF_XBOX_AUTOPLAY_CONTENT_DETECTED"},
            {0x19890c35a3bde075UL, "WNF_XBOX_AUTO_SIGNIN_IN_PROGRESS"},
            {0x19890c35a3bf2075UL, "WNF_XBOX_CLOUD_SETTINGS_UPDATED"},
            {0x19890c35a3bef075UL, "WNF_XBOX_CLUBCHAT_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bee875UL, "WNF_XBOX_CLUB_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bf6075UL, "WNF_XBOX_COMMANDSERVICE_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bfa075UL, "WNF_XBOX_CONNECTED_STANDBY_HISTORY"},
            {0x19890c35a3bf5075UL, "WNF_XBOX_COPYONLAN_UPLOAD_STATE_CHANGED"},
            {0x19890c35a3bdc875UL, "WNF_XBOX_CORTANAOVERLAY_VISIBILITY_CHANGED"},
            {0x19890c35a3be2075UL, "WNF_XBOX_CORTANA_SIGNEDIN_USERS_GRAMMAR_UPDATE_NOTIFICATION"},
            {0x19890c35a3be1875UL, "WNF_XBOX_CORTANA_TV_GRAMMAR_UPDATE_NOTIFICATION"},
            {0x19890c35a3be8875UL, "WNF_XBOX_CORTANA_USER_CHANGED_UPDATE_NOTIFICATION"},
            {0x19890c35a3bf5875UL, "WNF_XBOX_DASHBOARD_DIRECT_ACTIVATION"},
            {0x19890c35a3bf2875UL, "WNF_XBOX_ERA_FAST_ITERATION_STATUS_CHANGED"},
            {0x19890c35a3bf9075UL, "WNF_XBOX_ERA_INITIAL_PRESENT"},
            {0x19890c35a3bd5875UL, "WNF_XBOX_ERA_TITLE_LAUNCH_NOTIFICATION"},
            {0x19890c35a3be0875UL, "WNF_XBOX_ERA_VM_INSTANCE_CHANGED"},
            {0x19890c35a3be7075UL, "WNF_XBOX_ERA_VM_IOPRIORITY_CHANGED"},
            {0x19890c35a3bc8875UL, "WNF_XBOX_ERA_VM_STATUS_CHANGED"},
            {0x19890c35a3bcf875UL, "WNF_XBOX_EXIT_SILENT_BOOT_MODE"},
            {0x19890c35a3bf0875UL, "WNF_XBOX_EXPANDED_RESOURCES_INACTIVE"},
            {0x19890c35a3bdd875UL, "WNF_XBOX_EXTENDED_RESOURCE_MODE_CHANGED"},
            {0x19890c35a3bf8075UL, "WNF_XBOX_GAMECORE_TITLE_LAUNCH_NOTIFICATION"},
            {0x19890c35a3be6875UL, "WNF_XBOX_GAMER_ACCOUNT_CHANGED"},
            {0x19890c35a3bc4875UL, "WNF_XBOX_GLOBALIZATION_SETTING_CHANGED"},
            {0x19890c35a3bdf875UL, "WNF_XBOX_GLOBAL_SPEECH_INPUT_NOTIFICATION"},
            {0x19890c35a3bf7075UL, "WNF_XBOX_GUEST_VM_CRASH_DUMP_NOTIFICATION"},
            {0x19890c35a3bea875UL, "WNF_XBOX_GUIDE_DIRECT_ACTIVATION"},
            {0x19890c35a3bf9875UL, "WNF_XBOX_HAS_NETWORK_CONNECTIVITY"},
            {0x19890c35a3bcf075UL, "WNF_XBOX_HOST_STORAGE_CONFIGURATION_CHANGED"},
            {0x19890c35a3bf8875UL, "WNF_XBOX_HOST_XVC_CORRUPTION_DETECTED"},
            {0x19890c35a3bc4075UL, "WNF_XBOX_IDLE_DIMMER_CHANGED"},
            {0x19890c35a3be6075UL, "WNF_XBOX_KEYBOARD_LOCALE_CHANGED"},
            {0x19890c35a3be2875UL, "WNF_XBOX_KINECT_IS_REQUIRED"},
            {0x19890c35a3beb875UL, "WNF_XBOX_LIBRARY_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bed875UL, "WNF_XBOX_LIVETV_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bd9075UL, "WNF_XBOX_LIVETV_TUNER_COUNT_CHANGED"},
            {0x19890c35a3bc7075UL, "WNF_XBOX_LIVE_CONNECTIVITY_CHANGED"},
            {0x19890c35a3bf0075UL, "WNF_XBOX_MEDIA_IS_PLAYING_CHANGED"},
            {0x19890c35a3bec075UL, "WNF_XBOX_MESSAGING_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bd2075UL, "WNF_XBOX_MSA_ENVIRONMENT_CONFIGURED"},
            {0x19890c35a3bed075UL, "WNF_XBOX_MULTIPLAYER_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bf3875UL, "WNF_XBOX_NARRATOR_INPUT_LEARNING_MODE_CHANGED"},
            {0x19890c35a3bda875UL, "WNF_XBOX_NARRATOR_RECT_CHANGED"},
            {0x19890c35a3bf4875UL, "WNF_XBOX_NEON_SETTING_CHANGED"},
            {0x19890c35a3bf6875UL, "WNF_XBOX_NOTIFICATION_SETTING_CHANGED"},
            {0x19890c35a3bdd075UL, "WNF_XBOX_NOTIFICATION_UNREAD_COUNT"},
            {0x19890c35a3bf3075UL, "WNF_XBOX_NTM_CONSTRAINED_MODE_CHANGED"},
            {0x19890c35a3bd9875UL, "WNF_XBOX_PACKAGECACHE_CHANGED"},
            {0x19890c35a3bc3875UL, "WNF_XBOX_PACKAGE_INSTALL_STATE_CHANGED"},
            {0x19890c35a3bd7075UL, "WNF_XBOX_PACKAGE_STREAMING_STATE"},
            {0x19890c35a3bc3075UL, "WNF_XBOX_PACKAGE_UNMOUNTED_FROM_SYSTEM_FOR_LAUNCH"},
            {0x19890c35a3bdb075UL, "WNF_XBOX_PACKAGE_UNMOUNTED_FROM_SYSTEM_FOR_UNINSTALL"},
            {0x19890c35a3bf1075UL, "WNF_XBOX_PARENTAL_RESTRICTIONS_CHANGED"},
            {0x19890c35a3bf1875UL, "WNF_XBOX_PARTY_OVERLAY_STATE_CHANGED"},
            {0x19890c35a3bd1875UL, "WNF_XBOX_PASS3_UPDATE_NOTIFICATION"},
            {0x19890c35a3bec875UL, "WNF_XBOX_PEOPLE_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3be9875UL, "WNF_XBOX_PROACTIVE_NOTIFICATION_TRIGGERED"},
            {0x19890c35a3bd8075UL, "WNF_XBOX_QUERY_UPDATE_NOTIFICATION"},
            {0x19890c35a3bde875UL, "WNF_XBOX_REMOTE_SIGNOUT"},
            {0x19890c35a3bd8875UL, "WNF_XBOX_REPOSITORY_CHANGED"},
            {0x19890c35a3be1075UL, "WNF_XBOX_RESET_IDLE_TIMER"},
            {0x19890c35a3be5075UL, "WNF_XBOX_SAFEAREA_SETTING_CHANGED"},
            {0x19890c35a3bdb875UL, "WNF_XBOX_SEND_LTV_COMMAND_REQUESTED"},
            {0x19890c35a3bef875UL, "WNF_XBOX_SETTINGS_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bdc075UL, "WNF_XBOX_SHELL_DATACACHE_ENTITY_CHANGED"},
            {0x19890c35a3bd0875UL, "WNF_XBOX_SHELL_INITIALIZED"},
            {0x19890c35a3bc2875UL, "WNF_XBOX_SHELL_TOAST_NOTIFICATION"},
            {0x19890c35a3bd3875UL, "WNF_XBOX_SIP_FOCUS_TRANSFER_NOTIFICATION"},
            {0x19890c35a3bd2875UL, "WNF_XBOX_SIP_VISIBILITY_CHANGED"},
            {0x19890c35a3be7875UL, "WNF_XBOX_SPEECH_INPUT_DEVICE"},
            {0x19890c35a3bd6875UL, "WNF_XBOX_STORAGE_CHANGED"},
            {0x19890c35a3bc6875UL, "WNF_XBOX_STORAGE_ERROR"},
            {0x19890c35a3bd6075UL, "WNF_XBOX_STORAGE_STATUS"},
            {0x19890c35a3bd7875UL, "WNF_XBOX_STREAMING_QUEUE_CHANGED"},
            {0x19890c35a3bf4075UL, "WNF_XBOX_SUSPEND_SKELETAL_TRACKING_INITIALIZATION"},
            {0x19890c35a3bee075UL, "WNF_XBOX_SYSTEMUI_RAW_NOTIFICATION_RECEIVED"},
            {0x19890c35a3bca075UL, "WNF_XBOX_SYSTEM_CONSTRAINED_MODE_STATUS_CHANGED"},
            {0x19890c35a3bd3075UL, "WNF_XBOX_SYSTEM_GAME_STREAMING_STATE_CHANGED"},
            {0x19890c35a3bc9875UL, "WNF_XBOX_SYSTEM_IDLE_TIMEOUT_CHANGED"},
            {0x19890c35a3bd5075UL, "WNF_XBOX_SYSTEM_LOW_POWER_MAINTENANCE_WORK_ALLOWED"},
            {0x19890c35a3bc7875UL, "WNF_XBOX_SYSTEM_TITLE_AUTH_STATUS_CHANGED"},
            {0x19890c35a3bce075UL, "WNF_XBOX_SYSTEM_USER_CONTEXT_CHANGED"},
            {0x19890c35a3bf7875UL, "WNF_XBOX_TEST_NETWORK_CONNECTION_COMPLETE"},
            {0x19890c35a3beb075UL, "WNF_XBOX_TITLE_SPOP_VETO_RECEIVED"},
            {0x19890c35a3be3875UL, "WNF_XBOX_VIDEOPLAYER_ACTIVEPLAYER"},
            {0x19890c35a3be4875UL, "WNF_XBOX_VIDEOPLAYER_PLAYBACKPROGRESS"},
            {0x19890c35a3be4075UL, "WNF_XBOX_VIDEOPLAYER_PLAYERSTATE"},
            {0x19890c35a3bda075UL, "WNF_XBOX_WPN_PLATFORM_HOST_INITIALIZED"},
            {0x19890c35a3bd4075UL, "WNF_XBOX_XAM_SMB_SHARES_INIT_ALLOW_SYSTEM_READY"},
            {0x19890c35a3bd4875UL, "WNF_XBOX_XBBLACKBOX_SNAP_NOTIFICATION"}
        };
        }
        #endregion

        #region Static Members
        /// <summary>
        /// Get the state name to name mappings.
        /// </summary>
        public static Dictionary<ulong, string> StateNames => _state_name_to_name.Value;

        /// <summary>
        /// Get the name to state name mappings.
        /// </summary>
        public static Dictionary<string, ulong> Names => _name_to_state_name.Value;

        /// <summary>
        /// Get the name of a state name if known.
        /// </summary>
        /// <param name="state_name">The state name.</param>
        /// <returns>The name of the state name, or null if unknown.</returns>
        public static string GetName(ulong state_name)
        {
            if (StateNames.ContainsKey(state_name))
            {
                return StateNames[state_name];
            }
            return null;
        }
        #endregion
    }
}
