#!/usr/bin/env ruby
# -*- coding: utf-8 -*-
#
#--
# Copyright (C) 2009-2015 Thomas Leitner <t_leitner@gmx.at>
#
# This file is part of kramdown which is licensed under the MIT.
#++
#

require 'optparse'
require 'kramdown'
require 'uri'

module Kramdown
  module Converter
    class Html
      def convert_img(el, indent)
        el.attr["src"] = "https://forum.getmonero.org/get/image?link=" + URI.escape(el.attr["src"])
          "<img#{html_attributes(el.attr)} class=\"markdown-responsive\" />"
        end
      end
  end
end

options = {}
format = ['html']
OptionParser.new do |opts|
  opts.banner = "Usage: kramdown [options] [FILE FILE ...]"
  opts.summary_indent = ' '*4

  opts.separator ""
  opts.separator "Command line options:"
  opts.separator ""

  opts.on("-i", "--input ARG", "Specify the input format: kramdown (default), html, GFM or markdown") {|v| options[:input] = v}
  opts.on("-o", "--output ARG", Array, "Specify one or more output formats separated by commas: html (default), kramdown, latex, pdf or remove_html_tags") {|v| format = v}

  opts.on("-v", "--version", "Show the version of kramdown") do
    puts Kramdown::VERSION
    exit
  end
  opts.on("-h", "--help", "Show the help") do
    puts opts.summarize('', 5, 72)
    exit
  end

  opts.separator ""
  opts.separator "kramdown options:"
  opts.separator ""

  Kramdown::Options.definitions.each do |n, definition|
    no = n.to_s.tr('_', '-')
    if definition.type == Kramdown::Options::Boolean
      opts.on("--[no-]#{no}") {|v| options[n] = Kramdown::Options.parse(n, v)}
    else
      type = definition.type
      type = String if type == Symbol || type == Object
      opts.on("--#{no} ARG", type) {|v| options[n] = Kramdown::Options.parse(n, v)}
    end

    definition.desc.split(/\n/).each do |line|
      opts.separator opts.summary_indent + ' '*6 + line
    end
    opts.separator ''
  end

end.parse!

begin
  doc = Kramdown::Document.new(ARGF.read, options)
  result = ''
  format.each {|f| result = doc.send("to_#{f}")}
  puts result
  doc.warnings.each {|warn| $stderr.puts "Warning: #{warn}"}
rescue Kramdown::Error => e
  $stderr.puts "Error: #{e.message}"
  exit(1)
end
