package handlers

import (
	"encoding/json"
	"fmt"
	"log"
	"net/http"
	"strconv"

	DM "github.com/kycklingar/PBooru/DataManager"
	BM "github.com/kycklingar/PBooru/benchmark"
)

func APIHandler(w http.ResponseWriter, r *http.Request) {
	renderTemplate(w, "api", nil)
}

func APIv1Handler(w http.ResponseWriter, r *http.Request) {
	renderTemplate(w, "apiV1", nil)
}

type APIv1Post struct {
	ID          int
	Hash        string
	Sha256      string
	Md5         string
	ThumbHashes []DM.Thumb
	Mime        string
	Deleted     bool
	Tags        []APIv1TagI
	Dimension   *DM.Dimension
	Filesize    int64
}

type APIv1TagI interface {
	Parse(*DM.Tag)
}

type APIv1TagString string

func (t *APIv1TagString) Parse(tag *DM.Tag) {
	var str string
	if tag.QNamespace(DM.DB).QNamespace(DM.DB) == "none" {
		str = tag.QTag(DM.DB)
	} else {
		str = fmt.Sprintf("%s:%s", tag.QNamespace(DM.DB).QNamespace(DM.DB), tag.QTag(DM.DB))
	}

	*t = APIv1TagString(str)
}

type APIv1Tag struct {
	Tag       string
	Namespace string
}

func (t *APIv1Tag) Parse(tag *DM.Tag) {
	t.Tag = tag.QTag(DM.DB)
	t.Namespace = tag.QNamespace(DM.DB).QNamespace(DM.DB)
}

func jsonEncode(w http.ResponseWriter, v interface{}) error {
	enc := json.NewEncoder(w)
	enc.SetEscapeHTML(true)
	enc.SetIndent("", " ")
	err := enc.Encode(v)
	if err != nil {
		log.Print(err)
		APIError(w, ErrInternal, http.StatusInternalServerError)
		return err
	}
	return nil
}

func APIv1PostHandler(w http.ResponseWriter, r *http.Request) {
	p := DM.NewPost()

	firstNonEmpty := func(keys ...string) (string, string) {
		for _, key := range keys {
			if value := r.FormValue(key); value != "" {
				return key, value
			}
		}
		return "", ""
	}

	key, val := firstNonEmpty("id", "ipfs", "sha256", "md5")

	var err error

	switch key {
	case "id":
		var id int
		id, err = strconv.Atoi(val)
		if err != nil {
			break
		}
		err = p.SetID(DM.DB, id)
	case "ipfs":
		p.Hash = val
	case "sha256", "md5":
		p, err = DM.GetPostFromHash(key, val)
	default:
		APIError(w, "No Identifier", http.StatusBadRequest)
		return
	}

	if err != nil {
		APIError(w, err.Error(), http.StatusInternalServerError)
		return
	}

	if p.QID(DM.DB) == 0 {
		APIError(w, "Post Not Found", http.StatusNotFound)
		return
	}

	var combineTags bool
	if len(r.FormValue("combTagNamespace")) > 0 {
		combineTags = true
	}

	AP, err := DMToAPIPost(p, true, combineTags)
	if err != nil {
		log.Print(err)
		APIError(w, ErrInternal, http.StatusInternalServerError)
		return
	}

	jsonEncode(w, AP)
}

//func APIv1DuplicateHandler(w http.ResponseWriter, r *http.Request) {
//	p := DM.NewPost()
//	if postID := r.FormValue("id"); postID != "" {
//		id, err := strconv.Atoi(postID)
//		if err != nil {
//			APIError(w, "ID is not a number", http.StatusBadRequest)
//			return
//		}
//
//		err = p.SetID(DM.DB, id)
//		if err != nil {
//			log.Print(err)
//			APIError(w, ErrInternal, http.StatusInternalServerError)
//			return
//		}
//	} else if postHash := r.FormValue("hash"); postHash != "" {
//		p.Hash = postHash
//
//		if p.QID(DM.DB) == 0 {
//			//fmt.Fprint(w, "{}")
//			APIError(w, "Post Not Found", http.StatusNotFound)
//			return
//		}
//	} else {
//		APIError(w, "No Identifier", http.StatusBadRequest)
//		return
//	}
//
//	d, err := p.Duplicates()
//
//	type APIv1Duplicate struct {
//		ID    int
//		Level int
//	}
//
//	dp := APIv1Duplicate{d.QDupID(DM.DB), d.QLevel(DM.DB)}
//
//	jsonEncode(w, dp)
//}

func DMToAPIPost(p *DM.Post, includeTags, combineTagNamespace bool) (APIv1Post, error) {
	var AP APIv1Post

	p.QChecksums(DM.DB)
	p.QThumbnails(DM.DB)
	p.QDimensions(DM.DB)
	AP = APIv1Post{
		ID:          p.QID(DM.DB),
		Hash:        p.QHash(DM.DB),
		Sha256:      p.Checksums.Sha256,
		Md5:         p.Checksums.Md5,
		ThumbHashes: p.Thumbnails(),
		Mime:        p.QMime(DM.DB).Str(),
		Deleted:     p.QDeleted(DM.DB) == 1,
		Filesize:    p.QSize(DM.DB),
		Dimension:   p.Dimension,
	}

	tc := DM.TagCollector{}
	if includeTags {
		err := tc.GetPostTags(DM.DB, p)
		if err != nil {
			return AP, err
		}
	}

	for _, tag := range tc.Tags {
		var t APIv1TagI
		if combineTagNamespace {
			t = new(APIv1TagString)
		} else {
			t = &APIv1Tag{}
		}
		t.Parse(tag)
		AP.Tags = append(AP.Tags, t)
	}

	return AP, nil
}

type APIv1Posts struct {
	TotalPosts int
	Generated  float64
	Posts      []APIv1Post
}

func APIv1PostsHandler(w http.ResponseWriter, r *http.Request) {
	tagStr := r.FormValue("tags")
	filterStr := r.FormValue("filter")
	unlessStr := r.FormValue("unless")
	limitStr := r.FormValue("limit")
	order := r.FormValue("order")
	offsetStr := r.FormValue("offset")
	offset, err := strconv.Atoi(offsetStr)
	if err != nil {
		offset = 0
	}

	mimeGroups := r.Form["mime-type"]

	mimeIDs := DM.MimeIDsFromType(mimeGroups)

	mimes := r.Form["mime"]
	for _, mime := range mimes {
		id, err := strconv.Atoi(mime)
		if err != nil {
			log.Println(err)
			continue
		}
		contains := func(s []int, i int) bool {
			for _, x := range s {
				if x == i {
					return true
				}
			}

			return false
		}

		if !contains(mimeIDs, id) {
			mimeIDs = append(mimeIDs, id)
		}
	}

	var combineTags bool
	if len(r.FormValue("combTagNamespace")) > 0 {
		combineTags = true
	}

	var includeTags bool
	if len(r.FormValue("inclTags")) > 0 {
		includeTags = true
	}

	bm := BM.Begin()

	pc := &DM.PostCollector{}
	err = pc.Get(tagStr, filterStr, unlessStr, order, mimeIDs)
	if err != nil {
		log.Print(err)
		APIError(w, ErrInternal, http.StatusInternalServerError)
		return
	}

	pc = DM.CachedPostCollector(pc)

	limit, err := strconv.Atoi(limitStr)
	if err != nil {
		limit = 50
	} else {
		limit = DM.Larg(10, DM.Smal(100, limit))
	}

	var AP APIv1Posts

	for _, post := range pc.Search(limit, limit*offset) {
		APp, err := DMToAPIPost(post, includeTags, combineTags)
		if err != nil {
			log.Print(err)
			http.Error(w, ErrInternal, http.StatusInternalServerError)
			return
		}
		AP.Posts = append(AP.Posts, APp)
	}

	AP.TotalPosts = pc.TotalPosts

	AP.Generated = bm.End(false).Seconds()
	jsonEncode(w, AP)
}

func APIError(w http.ResponseWriter, err string, code int) {
	s := fmt.Sprintf("{\"Error\": \"%s\", \"Code\": %d}", err, code)
	http.Error(w, s, code)
	return
}

func APIv1ComicsHandler(w http.ResponseWriter, r *http.Request) {
	cc := DM.ComicCollector{}
	if err := cc.Search(r.FormValue("title"), r.FormValue("tags"), 10, 0); err != nil {
		log.Println(err)
		http.Error(w, ErrInternal, http.StatusInternalServerError)
	}

	// cm := tComics(5, cc.Comics)
	// cm[0].Chapters
}

func APIv1SuggestTagsHandler(w http.ResponseWriter, r *http.Request) {
	tagStr := r.FormValue("tags")
	timer := BM.Begin()
	var tc DM.TagCollector
	tc.Parse(tagStr)

	if len(r.FormValue("opensearch")) >= 1 {
		jsonEncode(w, openSearchSuggestions(tagStr, tc))
	} else {
		sugt := tc.SuggestedTags(DM.DB)

		var tags []APIv1Tag
		for _, t := range sugt.Tags {
			var nt APIv1Tag
			nt.Tag = t.QTag(DM.DB)
			nt.Namespace = t.QNamespace(DM.DB).QNamespace(DM.DB)

			tags = append(tags, nt)
		}
		jsonEncode(w, tags)
	}

	timer.End(performBenchmarks)
}

func openSearchSuggestions(query string, tc DM.TagCollector) []interface{} {
	var tags []string
	//var counts []string
	for _, t := range tc.SuggestedTags(DM.DB).Tags {
		var str string
		tag := t.QTag(DM.DB)
		namespace := t.QNamespace(DM.DB).QNamespace(DM.DB)
		if namespace == "none" {
			str = tag
		} else {
			str = namespace + ":" + tag
		}

		tags = append(tags, str)
		//counts = append(counts, fmt.Sprint(t.QCount(DM.DB), " results"))
	}

	return []interface{}{query, tags} //, counts}
}

func APIv1SimilarPostsHandler(w http.ResponseWriter, r *http.Request) {
	postIDstr := r.FormValue("id")
	postID, err := strconv.Atoi(postIDstr)
	if err != nil {
		APIError(w, "valid id required", http.StatusBadRequest)
		return
	}

	var combineTags bool
	if len(r.FormValue("combTagNamespace")) > 0 {
		combineTags = true
	}

	type page struct {
		Posts []APIv1Post
	}

	var p page

	post := DM.NewPost()
	if err = post.SetID(DM.DB, postID); err != nil {
		APIError(w, ErrInternal, http.StatusInternalServerError)
		return
	}
	var posts []*DM.Post
	if posts, err = post.FindSimilar(DM.DB, 5); err != nil {
		APIError(w, ErrInternal, http.StatusInternalServerError)
		return
	}

	for _, pst := range posts {
		var ps APIv1Post
		if ps, err = DMToAPIPost(pst, true, combineTags); err != nil {
			APIError(w, ErrInternal, http.StatusInternalServerError)
			return
		}
		p.Posts = append(p.Posts, ps)
	}
	jsonEncode(w, p)
}
