#ifndef _IN_TYPES_H
#define _IN_TYPES_H
#include <ultra64.h>
#include "constants.h"

#define bool u32

struct coord {
	f32 x;
	f32 y;
	f32 z;
};

struct bbox {
	f32 xmin;
	f32 xmax;
	f32 ymin;
	f32 ymax;
	f32 zmin;
	f32 zmax;
};

struct weaponobj;
struct prop;

/**
 * Most, if not all, entity types (chrs, objs, weapons etc) have a pointer to a
 * prop struct. The struct contains properties that most entities have in
 * common such as coords and a room number. You can think of it as generic
 * entity.
 *
 * The type field indicates what type it is (chr, obj etc) and the entity
 * pointer points back to the proper entity struct.
 */
struct prop {
	/*0x00*/ u8 type;
	/*0x01*/ u8 flags;
	/*0x02*/ s16 timetoregen;

	/*0x04*/
	union {
		struct chrdata *chr;
		struct defaultobj *obj;
		struct doorobj *door;
		struct weaponobj *weapon;
	};

	/*0x08*/ struct coord pos;
	/*0x14*/ f32 z;
	/*0x18*/ struct prop *parent;
	/*0x1c*/ struct prop *child;
	/*0x20*/ struct prop *next;
	/*0x24*/ struct prop *prev;
	/*0x28*/ s16 rooms[8];
	/*0x38*/ u16 unk38;
	/*0x3a*/ s16 unk3a;
	/*0x3c*/ u8 unk3c;
	/*0x3d*/ u8 propstateindex;
	/*0x3e*/ u8 unk3e;
	/*0x3f*/ u8 unk3f_00 : 1;
	/*0x3f*/ u8 unk3f_01 : 1;
	/*0x3f*/ u8 unk3f_02 : 1;
	/*0x3f*/ u8 unk3f_03 : 1;
	/*0x3f*/ u8 unk3f_04 : 1;
	/*0x3f*/ u8 unk3f_05 : 1;
	/*0x3f*/ u8 unk3f_06 : 1;
	/*0x3f*/ u8 unk3f_07 : 1;
	/*0x40*/ u32 unk40;
	/*0x44*/ u32 unk44;
};

struct pad {
	/*0x00*/ struct coord pos;
	/*0x0c*/ struct coord look;
	/*0x18*/ struct coord up;
	/*0x24*/ struct coord normal;
	/*0x30*/ struct bbox bbox;
	/*0x48*/ s32 room;
	/*0x4c*/ u32 flags;
	/*0x50*/ u8 liftnum; // 1-10, 0 indicates no lift
	/*0x52*/ s16 unk52;
};

struct anim {
	/*0x00*/ s16 animnum;
	/*0x02*/ s16 animnum2;
	/*0x04*/ u32 unk04;
	/*0x08*/ s8 flip;
	/*0x09*/ u8 flip2;
	/*0x0a*/ u8 looping;
	/*0x0b*/ u8 average;
	/*0x0c*/ f32 frame;
	/*0x10*/ u32 frac;
	/*0x14*/ u16 framea;
	/*0x16*/ u16 frameb;
	/*0x18*/ u32 endrame;
	/*0x1c*/ u32 speed;
	/*0x20*/ u32 newspeed;
	/*0x24*/ u32 oldspeed;
	/*0x28*/ u32 timespeed;
	/*0x2c*/ u32 elapsespeed;
	/*0x30*/ u32 frame2;
	/*0x34*/ u32 frac2;
	/*0x38*/ u16 frame2a;
	/*0x3a*/ u16 frame2b;
	/*0x3c*/ u32 endframe2;
	/*0x40*/ u32 speed2;
	/*0x44*/ u32 newspeed2;
	/*0x48*/ u32 oldspeed2;
	/*0x4c*/ u32 timespeed2;
	/*0x50*/ u32 elapsespeed2;
	/*0x54*/ u32 fracmerge;
	/*0x58*/ u32 timemerge;
	/*0x5c*/ u32 elapsemerge;
	/*0x60*/ u32 loopframe;
	/*0x64*/ u32 loopmerge;
	/*0x68*/ u32 unk68;
	/*0x6c*/ u32 unk6c;
	/*0x70*/ u32 unk70;
	/*0x74*/ f32 playspeed;
	/*0x78*/ u32 newplay;
	/*0x7c*/ u32 oldplay;
	/*0x80*/ u32 timeplay;
	/*0x84*/ u32 elapseplay;
	/*0x88*/ u32 animscale;
};

struct animdata {
	/*0x00*/ u32 unk00;
	/*0x04*/ u32 unk04;
	/*0x08*/ u32 unk08;
	/*0x0c*/ u32 unk0c;
	/*0x10*/ u32 unk10;
	/*0x14*/ f32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ struct anim *anim;
};

struct aibotammotype {
	s32 quantities[33];
};

struct aibot {
	/*0x000*/ u8 unk000;
	/*0x004*/ struct mpsim *simulant;
	/*0x008*/ u16 unk008;
	/*0x008*/ s16 followingplayernum;
	/*0x00c*/ s16 unk00c;
	/*0x010*/ struct prop *prop;
	/*0x014*/ u32 unk014;
	/*0x018*/ u32 unk018;
	/*0x01c*/ struct aibotammotype *ammotypes;
	/*0x020*/ u32 unk020;
	/*0x024*/ u32 unk024;
	/*0x028*/ u32 unk028;
	/*0x02c*/ u16 unk02c;
	/*0x02e*/ u16 unk02e;
	/*0x030*/ u32 unk030; // timer of some sort
	/*0x034*/ u32 unk034;
	/*0x038*/ u32 unk038;
	/*0x03c*/ u16 unk03c;
	/*0x03e*/ s16 unk03e;
	/*0x040*/ f32 unk040;
	/*0x044*/ u32 unk044;
	/*0x048*/ s16 unk048;
	/*0x04a*/ s16 unk04a;
	/*0x04c*/ u8 unk04c_00 : 1;
	/*0x04c*/ u8 unk04c_01 : 1;
	/*0x04c*/ u8 unk04c_02 : 1;
	/*0x04c*/ u8 unk04c_03 : 1;
	/*0x04c*/ u8 unk04c_04 : 1;
	/*0x04c*/ u8 unk04c_05 : 1;
	/*0x04c*/ u8 unk04c_06 : 1;
	/*0x04c*/ u8 unk04c_07 : 1;
	/*0x04d*/ u8 unk04d;
	/*0x04e*/ u8 unk04e;
	/*0x04f*/ u8 unk04f_00 : 1;
	/*0x04f*/ u8 unk04f_01 : 1;
	/*0x04f*/ u8 unk04f_02 : 1;
	/*0x04f*/ u8 unk04f_03 : 1;
	/*0x04f*/ u8 unk04f_04 : 1;
	/*0x04f*/ u8 unk04f_05 : 1;
	/*0x04f*/ u8 unk04f_06 : 1;
	/*0x04f*/ u8 unk04f_07 : 1;
	/*0x050*/ u32 unk050;
	/*0x054*/ u32 unk054;
	/*0x058*/ u8 unk058;
	/*0x059*/ u8 unk059;
	/*0x05c*/ u32 unk05c;
	/*0x060*/ u32 unk060;
	/*0x064*/ u16 unk064;
	/*0x068*/ u32 unk068;
	/*0x06c*/ f32 unk06c;
	/*0x070*/ f32 unk070;
	/*0x074*/ s8 unk074;
	/*0x075*/ s8 unk075;
	/*0x076*/ s8 unk076;
	/*0x077*/ s8 unk077;
	/*0x078*/ u8 unk078;
	/*0x079*/ u8 command;
	/*0x07a*/ s16 rooms[1];
	/*0x07c*/ u32 unk07c;
	/*0x080*/ u32 unk080;
	/*0x084*/ u32 unk084;
	/*0x088*/ u32 unk088;
	/*0x08c*/ struct coord defendholdpos;
	/*0x098*/ f32 unk098;
	/*0x09c*/ u8 unk09c_00 : 1;
	/*0x09c*/ u8 unk09c_01 : 1;
	/*0x09c*/ u8 unk09c_02 : 1;
	/*0x09c*/ u8 unk09c_03 : 1;
	/*0x09c*/ u8 unk09c_04 : 1;
	/*0x09c*/ u8 unk09c_05 : 1;
	/*0x09c*/ u8 unk09c_06 : 1;
	/*0x09c*/ u8 unk09c_07 : 1;
	/*0x09d*/ u8 unk09d;
	/*0x0a0*/ s32 unk0a0;
	/*0x0a4*/ f32 unk0a4;
	/*0x0a8*/ f32 unk0a8;
	/*0x0ac*/ f32 unk0ac;
	/*0x0b0*/ f32 unk0b0;
	/*0x0b4*/ f32 unk0b4;
	/*0x0b8*/ f32 unk0b8;
	/*0x0bc*/ s32 unk0bc;
	/*0x0c0*/ s32 attackpropnum;
	/*0x0c4*/ u32 unk0c4;
	/*0x0c8*/ u32 unk0c8;
	/*0x0cc*/ u32 unk0cc;
	/*0x0d0*/ u32 unk0d0;
	/*0x0d4*/ s32 followprotectpropnum;
	/*0x0d8*/ s32 unk0d8;
	/*0x0dc*/ u32 unk0dc;
	/*0x0e0*/ u16 unk0e0;
	/*0x0e2*/ u16 unk0e2;
	/*0x0e4*/ f32 unk0e4;
	/*0x0e8*/ f32 unk0e8;
	/*0x0ec*/ u32 unk0ec;
	/*0x0f0*/ u32 unk0f0;
	/*0x0f4*/ u32 unk0f4;
	/*0x0f8*/ u32 unk0f8;
	/*0x0fc*/ u32 unk0fc;
	/*0x100*/ u32 unk100;
	/*0x104*/ u32 unk104;
	/*0x108*/ f32 unk108;
	/*0x10c*/ f32 unk10c;
	/*0x110*/ f32 unk110;
	/*0x114*/ u32 unk114;
	/*0x118*/ u32 unk118;
	/*0x11c*/ u32 unk11c;
	/*0x120*/ s32 unk120;
	/*0x124*/ s32 unk124;
	/*0x128*/ u32 unk128;
	/*0x12c*/ u32 unk12c;
	/*0x130*/ s8 unk130[12];
	/*0x13c*/ f32 unk13c[12];
	/*0x16c*/ u8 unk16c[12];
	/*0x178*/ s32 unk178[12];
	/*0x1a8*/ s16 unk1a8[12];
	/*0x1c0*/ u32 unk1c0;
	/*0x1c4*/ u32 unk1c4;
	/*0x1c8*/ u32 unk1c8;
	/*0x1cc*/ u32 unk1cc;
	/*0x1d0*/ u32 unk1d0;
	/*0x1d4*/ u32 unk1d4;
	/*0x1d8*/ u32 unk1d8;
	/*0x1dc*/ u32 unk1dc;
	/*0x1e0*/ u32 unk1e0;
	/*0x1e4*/ u32 unk1e4;
	/*0x1e8*/ u32 unk1e8;
	/*0x1ec*/ u32 unk1ec;
	/*0x1f0*/ u32 unk1f0;
	/*0x1f4*/ u32 unk1f4;
	/*0x1f8*/ u32 unk1f8;
	/*0x1fc*/ u32 unk1fc;
	/*0x200*/ u32 unk200;
	/*0x204*/ u32 unk204;
	/*0x208*/ u32 unk208;
	/*0x20c*/ u32 unk20c;
	/*0x210*/ u32 unk210;
	/*0x214*/ u32 unk214;
	/*0x218*/ u32 unk218;
	/*0x21c*/ u32 unk21c;
	/*0x220*/ u32 unk220;
	/*0x224*/ u32 unk224;
	/*0x228*/ u32 unk228;
	/*0x22c*/ u32 unk22c;
	/*0x230*/ u32 unk230;
	/*0x234*/ u32 unk234;
	/*0x238*/ u32 unk238;
	/*0x23c*/ u32 unk23c;
	/*0x240*/ u32 unk240;
	/*0x244*/ u32 unk244;
	/*0x248*/ u32 unk248;
	/*0x24c*/ u32 unk24c;
	/*0x250*/ u32 unk250;
	/*0x254*/ u32 unk254;
	/*0x258*/ u32 unk258;
	/*0x25c*/ u32 unk25c;
	/*0x260*/ u32 unk260;
	/*0x264*/ u32 unk264;
	/*0x268*/ u32 unk268;
	/*0x26c*/ u32 unk26c;
	/*0x270*/ u32 unk270;
	/*0x274*/ u32 unk274;
	/*0x278*/ u32 unk278;
	/*0x27c*/ u32 unk27c;
	/*0x280*/ u32 unk280;
	/*0x284*/ u32 unk284;
	/*0x288*/ u32 unk288;
	/*0x28c*/ u32 unk28c;
	/*0x290*/ u32 unk290;
	/*0x294*/ u32 unk294;
	/*0x298*/ u32 unk298;
	/*0x29c*/ u32 unk29c;
	/*0x2a0*/ u32 unk2a0;
	/*0x2a4*/ u32 unk2a4;
	/*0x2a8*/ u32 unk2a8;
	/*0x2ac*/ u32 unk2ac;
	/*0x2b0*/ u32 unk2b0;
	/*0x2b4*/ u32 unk2b4;
	/*0x2b8*/ u32 unk2b8;
	/*0x2bc*/ u32 unk2bc;
	/*0x2c0*/ u32 unk2c0;
	/*0x2c4*/ f32 unk2c4;
	/*0x2c8*/ u32 unk2c8;
	/*0x2cc*/ u32 unk2cc;
	/*0x2d0*/ u32 unk2d0;
	/*0x2d4*/ f32 unk2d4;
};

struct act_stand {
	/*0x2c*/ s32 unk02c;
	/*0x30*/ u32 face_entitytype;
	/*0x34*/ u32 face_entityid;
	/*0x38*/ u32 unk038;
	/*0x3c*/ s32 unk03c;
	/*0x40*/ u32 unk040;
	/*0x44*/ u32 unk044;
	/*0x48*/ f32 unk048;
	/*0x4c*/ s8 face_target;
};

struct act_anim {
	/*0x2c*/ u32 unk02c;
	/*0x30*/ u32 unk030;
	/*0x34*/ u32 unk034;
	/*0x38*/ u32 unk038;
	/*0x3c*/ u32 unk03c;
	/*0x40*/ u8 unk040;
	/*0x41*/ u8 unk041;
	/*0x42*/ u16 unk042;
	/*0x44*/ u16 unk044;
	/*0x46*/ u16 unk046;
	/*0x48*/ s16 animnum;
	/*0x4a*/ u8 flip;
	/*0x4c*/ f32 startframe;
	/*0x50*/ f32 unk050;
	/*0x54*/ f32 unk054;
	/*0x58*/ f32 unk058;
};

struct act_die {
	/*0x2c*/ u32 unk02c;
	/*0x30*/ f32 unk030; // frame number
	/*0x34*/ f32 unk034; // frame number
	/*0x38*/ f32 unk038;
	/*0x3c*/ u32 unk03c;
	/*0x40*/ u32 unk040;
	/*0x44*/ u32 unk044;
	/*0x48*/ u32 unk048;
	/*0x4c*/ s16 unk04c;
};

struct act_dead {
	/*0x2c*/ u32 unk02c;
	/*0x30*/ u32 unk030;
	/*0x34*/ s32 unk034; // timer
	/*0x38*/ s32 unk038; // timer
	/*0x3c*/ u32 unk03c;
};

struct act_preargh {
	/*0x2c*/ struct coord pos;
	/*0x38*/ f32 unk038;
	/*0x3c*/ f32 unk03c;
	/*0x40*/ u32 unk040;
	/*0x44*/ u32 unk044;
};

struct act_attack {
	/*0x2c*/ u32 unk02c;
	/*0x30*/ u16 unk030;
	/*0x32*/ u8 unk032;
	/*0x33*/ s8 unk033;
	/*0x34*/ s8 unk034;
	/*0x38*/ u32 unk038;
	/*0x3c*/ u32 unk03c;
	/*0x40*/ u32 unk040;
	/*0x44*/ u32 unk044;
	/*0x48*/ u32 unk048;
	/*0x4c*/ u32 unk04c;
	/*0x50*/ u32 unk050;
	/*0x54*/ u32 unk054;
	/*0x58*/ u32 unk058;
};

struct act_sidestep {
	/*0x2c*/ u32 unk02c;
};

struct act_jumpout {
	/*0x2c*/ u32 unk02c;
};

struct act_runpos {
	/*0x2c*/ struct coord pos;
	/*0x38*/ f32 unk038;
	/*0x3c*/ s32 unk03c;
	/*0x40*/ f32 unk040;
};

struct act_gopos {
	/*0x02c*/ struct coord pos;
	/*0x038*/ s16 rooms[8];
	/*0x048*/ s32 numwaypoints;
	/*0x04c*/ s32 *waypoints[MAX_CHRWAYPOINTS];
	/*0x064*/ u8 nextwaypointindex;

	// This doesn't appear to be a proper bitfield, but is used as one
	// ....x... = on preset path
	// ......xx = speed
	/*0x065*/ u8 unk065;

	/*0x066*/ s16 unk066;
	/*0x068*/ s8 unk068;
	/*0x069*/ u8 unk069;
	/*0x06a*/ u8 unk06a;
	/*0x06b*/ u8 unk06b;
	/*0x06c*/ u16 unk06c;
	/*0x06e*/ u8 unk06e;
	/*0x070*/ u32 unk070;
	/*0x074*/ u32 unk074;
	/*0x078*/ u32 unk078;
	/*0x07c*/ u32 unk07c;
	/*0x080*/ u32 unk080;
	/*0x084*/ u32 unk084;
	/*0x088*/ u32 unk088;
	/*0x08c*/ u32 unk08c;
	/*0x090*/ u32 unk090;
	/*0x094*/ u32 unk094;
	/*0x098*/ u32 unk098;
	/*0x09c*/ u32 unk09c;
	/*0x0a0*/ u32 unk0a0;
	/*0x0a4*/ u32 unk0a4;
	/*0x0a8*/ s32 unk0a8;
	/*0x0ac*/ f32 unk0ac;
};

struct act_surprised {
	/*0x2c*/ u32 unk02c;
};

struct act_throwgrenade {
	/*0x2c*/ u32 entitytype;
	/*0x30*/ u32 entityid;
	/*0x34*/ u32 hand;
	/*0x38*/ bool needsequip;
};

struct act_druggeddrop {
	/*0x2c*/ u32 unk02c;
	/*0x30*/ f32 unk030;
	/*0x34*/ f32 unk034;
};

struct act_druggedko {
	/*0x2c*/ u32 unk02c;
	/*0x30*/ u32 unk030;
	/*0x34*/ s32 unk034;
	/*0x38*/ s32 unk038;
	/*0x3c*/ u32 unk03c;
};

struct act_druggedcomingup {
	/*0x2c*/ s16 unk02c; // timer
	/*0x30*/ f32 unk030;
	/*0x34*/ f32 unk034;
};

struct act_attackamount {
	/*0x2c*/ u32 unk02c;
	/*0x30*/ u16 unk030;
	/*0x32*/ u8 unk032;
	/*0x33*/ s8 unk033;
	/*0x34*/ s8 unk034;
};

struct act_robotattack {
	/*0x2c*/ u32 unk02c;
	/*0x30*/ u32 unk030;
	/*0x34*/ u32 unk034;
	/*0x38*/ u32 unk038;
	/*0x3c*/ u32 unk03c;
	/*0x40*/ u32 unk040;
	/*0x44*/ u32 unk044;
	/*0x48*/ u32 unk048;
	/*0x4c*/ u32 unk04c;
	/*0x50*/ u32 unk050;
	/*0x54*/ u32 unk054;
	/*0x58*/ u32 unk058;
	/*0x5c*/ u32 unk05c;
	/*0x60*/ u32 unk060;
	/*0x64*/ u32 unk064;
	/*0x68*/ u32 unk068;
	/*0x6c*/ u16 unk06c;
	/*0x6e*/ u8 unk06e;
};

struct act_skjump {
	/*0x2c*/ u8 state;
	/*0x2d*/ u8 needsnewanim;
	/*0x2e*/ u8 hit;
	/*0x30*/ f32 xspeed;
	/*0x34*/ f32 zspeed;
	/*0x38*/ f32 angle;
	/*0x3c*/ s32 unk03c;
	/*0x40*/ struct coord newpos;
	/*0x4c*/ s32 unk04c;
	/*0x50*/ f32 y;
};

struct geo {
	/*0x134*/ s8 type;
	/*0x135*/ u8 unk01;
	/*0x136*/ u16 unk136;
	/*0x138*/ f32 ymax;
	/*0x13c*/ f32 ymin;
	/*0x140*/ f32 x;
	/*0x144*/ f32 z;
	/*0x148*/ f32 width;
};

struct chrdata {
	/*0x000*/ s16 chrnum;
	/*0x002*/ s8 accuracyrating;
	/*0x003*/ s8 speedrating;
	/*0x004*/ u8 firecount[2];
	/*0x006*/ s8 headnum;
	/*0x007*/ s8 actiontype;
	/*0x008*/ s8 sleep;
	/*0x009*/ s8 invalidmove;
	/*0x00a*/ s8 numclosearghs;
	/*0x00b*/ s8 numarghs;
	/*0x00c*/ u8 fadealpha;
	/*0x00d*/ s8 arghrating;
	/*0x00e*/ u8 aimendcount;
	/*0x00f*/ u8 grenadeprob;
	/*0x010*/ s16 bodynum;
	/*0x012*/ s8 flinchcnt;
	/*0x013*/ s8 path;
	/*0x014*/ u32 hidden;
	/*0x018*/ u32 chrflags;
	/*0x01c*/ struct prop *prop;
	/*0x020*/ struct animdata *animdata;
	/*0x024*/ f32 chrwidth;
	/*0x028*/ f32 chrheight;

	union {
		struct act_stand act_stand;
		struct act_anim act_anim;
		struct act_die act_die;
		struct act_dead act_dead;
		struct act_preargh act_preargh;
		struct act_attack act_attack;
		struct act_sidestep act_sidestep;
		struct act_jumpout act_jumpout;
		struct act_runpos act_runpos;
		struct act_gopos act_gopos;
		struct act_surprised act_surprised;
		struct act_throwgrenade act_throwgrenade;
		struct act_druggeddrop act_druggeddrop;
		struct act_druggedko act_druggedko;
		struct act_druggedcomingup act_druggedcomingup;
		struct act_attackamount act_attackamount;
		struct act_robotattack act_robotattack;
		struct act_skjump act_skjump;
	};

	/*0x0b0*/ f32 sumground;
	/*0x0b4*/ f32 manground;
	/*0x0b8*/ f32 ground;
	/*0x0bc*/ struct coord fallspeed;
	/*0x0c8*/ struct coord prevpos;
	/*0x0d4*/ u32 lastwalk60;
	/*0x0d8*/ s32 lastmoveok60;
	/*0x0dc*/ f32 visionrange;
	/*0x0e0*/ s32 lastseetarget60;
	/*0x0e4*/ s32 lastvisibletarget60;
	/*0x0e8*/ void *unk0e8;
	/*0x0ec*/ s16 lastshooter;
	/*0x0ee*/ s16 timeshooter;
	/*0x0f0*/ f32 hearingscale;
	/*0x0f4*/ s32 lastheartarget60;
	/*0x0f8*/ u8 shadecol[4];
	/*0x0fc*/ u8 nextcol[4];
	/*0x100*/ f32 damage;
	/*0x104*/ f32 maxdamage;
	/*0x108*/ u8 *ailist;
	/*0x10c*/ u16 aioffset;
	/*0x10e*/ s16 aireturnlist;
	/*0x110*/ s16 aishotlist;
	/*0x112*/ u8 morale;
	/*0x113*/ u8 alertness;
	/*0x114*/ u32 flags;
	/*0x118*/ u32 flags2;
	/*0x11c*/ s32 timer60;
	/*0x120*/ s32 soundtimer;
	/*0x124*/ u8 random;
	/*0x125*/ u8 team;
	/*0x126*/ u8 soundgap;
	/*0x128*/ s16 padpreset1;
	/*0x12a*/ s16 chrpreset1;
	/*0x12c*/ s16 proppreset1;
	/*0x12e*/ s16 chrseeshot;
	/*0x130*/ s16 chrseedie;
	/*0x132*/ s16 chrdup;
	struct geo geo;
	/*0x14c*/ f32 shotbondsum;
	/*0x150*/ f32 aimuplshoulder;
	/*0x154*/ f32 aimuprshoulder;
	/*0x158*/ f32 aimupback;
	/*0x15c*/ f32 aimsideback;
	/*0x160*/ f32 aimendlshoulder;
	/*0x164*/ f32 aimendrshoulder;
	/*0x168*/ f32 aimendback;
	/*0x16c*/ f32 aimendsideback;
	/*0x170*/ struct prop *weapons_held[3];
	/*0x17c*/ s8 fireslot[2];
	/*0x17e*/ s16 target; // index into g_Vars.props
	/*0x180*/ f32 cshield;
	/*0x184*/ u8 cmnum;
	/*0x185*/ u8 cmnum2;
	/*0x186*/ u8 cmnum3;
	/*0x187*/ u8 cmnum4;
	/*0x188*/ u16 cmcount;
	/*0x18a*/ u16 floorcol;
	/*0x18c*/ f32 oldframe;
	/*0x190*/ s8 footstep;
	/*0x191*/ u8 floortype;
	/*0x192*/ u16 hidden2;
	/*0x194*/ f32 magicframe;
	/*0x198*/ f32 magicspeed;
	/*0x19c*/ s16 magicanim;
	/*0x19e*/ s16 goposforce;
	/*0x1a0*/ s32 bdlist[60];
	/*0x290*/ u8 bdstart;
	/*0x291*/ u8 goposhitcount;
	/*0x292*/ s16 cover;
	/*0x294*/ struct coord targetlastseenp;
	/*0x2a0*/ u8 myaction[1];
	/*0x2a1*/ u8 orders[1];
	/*0x2a2*/ u8 squadron;
	/*0x2a3*/ u8 listening[1];
	/*0x2a4*/ u32 convtalk;
	/*0x2a8*/ s32 talktimer;
	/*0x2ac*/ u8 question;
	/*0x2ad*/ u8 talkgap;
	/*0x2ae*/ u16 unk2ae;
	/*0x2b0*/ u8 tude;
	/*0x2b1*/ u8 voicebox;
	/*0x2b2*/ s16 floorroom;
	/*0x2b4*/ u32 unk2b4;
	/*0x2b8*/ s16 oldrooms[8];
	/*0x2c8*/ struct coord runfrompos;
	/*0x2d4*/ struct aibot *aibot;
	/*0x2d8*/ s16 blurdrugamount;
	/*0x2da*/ u16 cloakpause;
	/*0x2dc*/ f32 drugheadsway;
	/*0x2e0*/ u8 drugheadcount;
	/*0x2e1*/ u8 cloakfade_00 : 7;
	/*0x2e1*/ u8 cloakfade_06 : 1;
	/*0x2e2*/ u8 teamscandist;
	/*0x2e3*/ u8 naturalanim[1];
	/*0x2e4*/ s32 myspecial; // This is an object tag ID
	/*0x2e8*/ f32 timeextra;
	/*0x2ec*/ f32 elapseextra;
	/*0x2f0*/ struct coord extraspeed;
	/*0x2fc*/ u8 yvisang;
	/*0x2fd*/ u8 ivebeenhit;
	/*0x2fe*/ u8 race;
	/*0x2ff*/ u8 blurnumtimesdied;
	/*0x300*/ struct prop *gunprop;
	/*0x304*/ f32 pushspeed[2];
	/*0x30c*/ f32 gunroty[2];
	/*0x314*/ f32 gunrotx[2];
	/*0x31c*/ u32 onladder;
	/*0x320*/ struct coord laddernormal;

	/*0x32c*/
	u8 unk32c_00 : 8;

	u8 unk32c_08 : 1;
	u8 pouncebits : 3;
	u8 unk32c_12 : 2;
	u8 darkroomthing : 1;
	u8 playerdeadthing : 1;

	u8 p1p2 : 2;
	u8 unk32c_18 : 1;
	u8 unk32c_19 : 1;
	u8 unk32c_20 : 1;
	u8 unk32c_21 : 1;
	u8 unk32c_22 : 2;

	u8 specialdie : 8;

	/*0x330*/ u16 roomtosearch;
	/*0x332*/ u8 propsoundcount;
	/*0x333*/ s8 patrolnextstep;
	/*0x334*/ u8 bulletstaken;
	/*0x335*/ u8 woundedsplatsadded;
	/*0x336*/ u16 tickssincesplat;
	/*0x338*/ u8 splatsdroppedhe;
	/*0x339*/ u8 stdsplatsadded;
	/*0x33a*/ u8 deaddropsplatsadded;
	/*0x33b*/ s8 aimtesttimer60;
	/*0x33c*/ struct coord lastdroppos;
	/*0x348*/ u32 unk348;
	/*0x34c*/ u32 unk34c;
	/*0x350*/ u32 unk350;
	/*0x354*/ s16 aipunchdodgelist;
	/*0x356*/ s16 aishootingatmelist;
	/*0x358*/ u16 poisoncounter;
	/*0x35a*/ s16 aidarkroomlist;
	/*0x35c*/ s16 aiplayerdeadlist;
	/*0x35e*/ u8 dodgerating;
	/*0x35f*/ u8 maxdodgerating;
	/*0x360*/ u8 unarmeddodgerating;
	/*0x361*/ u8 lastfootsample;
	/*0x362*/ u8 drcarollimage_left : 4;
	/*0x362*/ u8 drcarollimage_right : 4;
	/*0x364*/ u32 unk364;
};

struct obj48 {
	/*0x00*/ u32 flags;
	/*0x04*/ struct coord unk04;
	/*0x10*/ u32 unk10;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ u32 unk20;
	/*0x24*/ u32 unk24;
	/*0x28*/ u32 unk28;
	/*0x2c*/ u32 unk2c;
	/*0x30*/ u32 unk30;
	/*0x34*/ u32 unk34;
	/*0x38*/ u32 unk38;
	/*0x3c*/ u32 unk3c;
	/*0x40*/ u32 unk40;
	/*0x44*/ u32 unk44;
	/*0x48*/ u32 unk48;
	/*0x4c*/ u32 unk4c;
	/*0x50*/ u32 unk50;
	/*0x54*/ u32 unk54;
	/*0x58*/ u32 unk58;
	/*0x5c*/ u32 unk5c;
	/*0x60*/ u32 unk60;
	/*0x64*/ u32 unk64;
	/*0x68*/ u32 unk68;
	/*0x6c*/ u32 unk6c;
	/*0x70*/ u32 unk70;
	/*0x74*/ u32 unk74;
	/*0x78*/ u32 unk78;
	/*0x7c*/ u32 unk7c;
	/*0x80*/ u32 unk80;
	/*0x84*/ u32 unk84;
	/*0x88*/ u32 unk88;
	/*0x8c*/ u32 unk8c;
	/*0x90*/ u32 unk90;
	/*0x94*/ u32 unk94;
	/*0x98*/ u32 unk98;
	/*0x9c*/ u32 unk9c;
	/*0xa0*/ u32 unka0;
	/*0xa4*/ u32 unka4;
	/*0xa8*/ u32 unka8;
	/*0xac*/ u32 unkac;
	/*0xb0*/ u32 unkb0;
	/*0xb4*/ u32 unkb4;
	/*0xb8*/ u32 unkb8;
	/*0xbc*/ u32 unkbc;
	/*0xc0*/ u32 unkc0;
	/*0xc4*/ u32 unkc4;
	/*0xc8*/ u32 unkc8;
	/*0xcc*/ u32 unkcc;
	/*0xd0*/ u32 unkd0;
	/*0xd4*/ u32 unkd4;
	/*0xd8*/ u32 unkd8;
	/*0xdc*/ f32 unkdc;
};

struct hov {
	/*0x00*/ u32 unk00;
	/*0x04*/ u32 unk04;
	/*0x08*/ u32 unk08;
	/*0x0c*/ u32 unk0c;
	/*0x10*/ f32 unk10;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ u32 unk20;
	/*0x24*/ u32 unk24;
	/*0x28*/ u32 unk28;
	/*0x2c*/ u32 unk2c;
	/*0x30*/ u32 unk30;
	/*0x34*/ u32 unk34;
	/*0x38*/ u32 unk38;
	/*0x3c*/ u32 unk3c;
};

struct defaultobj {
	/*0x00*/ u16 extrascale;
	/*0x02*/ u8 hidden2;
	/*0x03*/ u8 type;
	/*0x04*/ s16 obj;
	/*0x06*/ s16 pad;
	/*0x08*/ u32 flags;
	/*0x0c*/ u32 flags2;
	/*0x10*/ u32 flags3;
	/*0x14*/ struct prop *prop;
	/*0x18*/ struct animdata *animdata;
	/*0x1c*/ f32 realrot[9];
	/*0x40*/ u32 hidden;
	/*0x44*/ struct geo *geo;
	/*0x48*/ struct obj48 *unk48;
	/*0x4c*/ s16 damage;
	/*0x4e*/ s16 maxdamage;
	/*0x50*/ u32 shadecol;
	/*0x54*/ u32 nextcol;
	/*0x58*/ u16 floorcol;
	/*0x5a*/ u8 numtiles;
};

struct doorobj { // objtype 0x01
	struct defaultobj base;
	/*0x5c*/ f32 maxfrac;
	/*0x60*/ f32 perimfrac;
	/*0x64*/ f32 accel;
	/*0x68*/ f32 decel;
	/*0x6c*/ f32 maxspeed;
	/*0x70*/ u16 doorflags;
	/*0x72*/ u16 doortype;
	/*0x74*/ u32 keyflags;
	/*0x78*/ u32 autoclosetime;
	/*0x7c*/ f32 frac;
	/*0x80*/ f32 fracspeed;
	/*0x84*/ s8 mode;
	/*0x85*/ u8 glasshits;
	/*0x86*/ u16 fadealpha;
	/*0x88*/ u16 xludist;
	/*0x8a*/ u16 opadist;
	/*0x8c*/ struct coord startpos;
	/*0x98*/ u32 unk98;
	/*0x9c*/ u32 unk9c;
	/*0xa0*/ u32 unka0;
	/*0xa4*/ u32 unka4;
	/*0xa8*/ u32 unka8;
	/*0xac*/ u32 unkac;
	/*0xb0*/ u32 unkb0;
	/*0xb4*/ u32 unkb4;
	/*0xb8*/ u32 unkb8;
	/*0xbc*/ struct doorobj *sibling;
	/*0xc0*/ u32 lastopen60;
	/*0xc4*/ s16 portal;
	/*0xc6*/ s8 soundtype;
	/*0xc7*/ s8 fadetime60;
	/*0xc8*/ s32 lastcalc60;
	/*0xcc*/ u8 laserfade;
};

struct keyobj { // objtype 0x04
	struct defaultobj base;
	u32 keyflags;
};

struct image {
	/*0x00*/ u32 unk00;
	/*0x04*/ u32 unk04;
	/*0x08*/ u32 unk08;
	/*0x0c*/ u32 unk0c;
	/*0x10*/ u32 unk10;
	/*0x14*/ u32 unk14;
};

struct ammocrateobj { // objtype 0x07
	struct defaultobj base;
	/*0x5c*/ u32 ammotype;
};

struct weaponobj { // objtype 0x08
	struct defaultobj base;
	/*0x5c*/ u8 weapon_id;
	/*0x5d*/ s8 unk5d;
	/*0x5e*/ s8 unk5e;
	/*0x5f*/ u8 dragonthrown;
	/*0x60*/ s16 unk60;
	/*0x62*/ s16 team;
};

struct singlemonitorobj { // objtype 0x0a
	/*0x00*/ u16 extrascale;
	/*0x02*/ u8 hidden2;
	/*0x03*/ u8 type;
	/*0x04*/ u16 obj;
	/*0x06*/ u16 pad;
	/*0x08*/ u32 flags;
	/*0x0c*/ u32 flags2;
	/*0x10*/ u32 flags3;
	/*0x14*/ struct prop *prop;
	/*0x18*/ u32 unk18;
	/*0x1c*/ f32 realrot[9];
	/*0x40*/ u32 hidden;
	/*0x44*/ u32 unk44;
	/*0x48*/ u32 unk48;
	/*0x4c*/ s16 damage;
	/*0x4e*/ u16 maxdamage;
	/*0x50*/ u32 shadecol;
	/*0x54*/ u32 nextcol;
	/*0x58*/ u16 floorcol;
	/*0x5a*/ u8 numtiles;
	/*0x5c*/ struct image image;
};

struct multimonitorobj { // objtype 0b
	struct singlemonitorobj subobjs[4];
};

struct autogunobj { // objtype 0d
	struct defaultobj base;
	/*0x5c*/ u32 unk5c;
	/*0x60*/ u32 unk60;
	/*0x64*/ u32 unk64;
	/*0x68*/ u32 unk68;
	/*0x6c*/ u32 unk6c;
	/*0x70*/ u32 unk70;
	/*0x74*/ u32 unk74;
	/*0x78*/ u32 unk78;
	/*0x7c*/ u32 unk7c;
	/*0x80*/ u32 unk80;
	/*0x84*/ u32 unk84;
	/*0x88*/ u32 unk88;
	/*0x8c*/ u32 unk8c;
	/*0x90*/ u32 unk90;
	/*0x94*/ u32 unk94;
	/*0x98*/ u32 unk98;
	/*0x9c*/ u32 unk9c;
	/*0xa0*/ u32 unka0;
	/*0xa4*/ u32 unka4;
	/*0xa8*/ u8 autogun_type;
	/*0xa9*/ u8 ammoquantity;
};

struct multiammocrateobj { // objtype 14
	struct defaultobj base;
	/*0x5c*/ u32 quantities[19]; // indexed by ammotype minus 1
};

struct shieldobj { // objtype 15
	struct defaultobj base;
	/*0x5c*/ f32 unk5c;
	/*0x60*/ f32 amount;
};

struct truckobj {
	struct defaultobj base;
	/*0x5c*/ u8 *ailist;
	/*0x60*/ u16 aioffset;
	/*0x62*/ s16 aireturnlist;
	/*0x64*/ u32 unk64;
	/*0x68*/ u32 unk68;
	/*0x6c*/ u32 unk6c;
	/*0x70*/ f32 speedaim;
	/*0x74*/ f32 speedtime60;
	/*0x78*/ u32 unk78;
	/*0x7c*/ u32 unk7c;
	/*0x80*/ struct path *path;
	/*0x84*/ s32 nextstep;
};

struct heliobj {
	struct defaultobj base;
	/*0x5c*/ u8 *ailist;
	/*0x60*/ u16 aioffset;
	/*0x62*/ s16 aireturnlist;
	/*0x64*/ u32 unk64;
	/*0x68*/ f32 rotaryspeed;
	/*0x6c*/ f32 rotaryspeedaim;
	/*0x70*/ f32 rotaryspeedtime;
	/*0x74*/ f32 speed;
	/*0x78*/ f32 speedaim;
	/*0x7c*/ f32 speedtime60;
};

struct liftobj { // objtype 30
	struct defaultobj base;
	/*0x5c*/ s16 pads[4];
	/*0x64*/ struct doorobj *doors[4];
	/*0x74*/ f32 dist;
	/*0x78*/ f32 speed;
	/*0x7c*/ u32 accel;
	/*0x80*/ u32 maxspeed;
	/*0x84*/ u8 soundtype;
	/*0x85*/ s8 levelcur;
	/*0x86*/ s8 levelaim;
	/*0x88*/ struct coord prevpos;
};

struct hoverbikeobj { // objtype 33
	struct defaultobj base;
	struct hov hov;
	/*0x09c*/ f32 speed[2];
	/*0x0a4*/ f32 prevpos[2];
	/*0x0ac*/ f32 w;
	/*0x0b0*/ u32 rels;
	/*0x0b8*/ u32 exreal;
	/*0x0bc*/ u32 ezreal;
	/*0x0c0*/ u32 ezreal2;
	/*0x0c4*/ u32 leanspeed;
	/*0x0c8*/ u32 leandiff;
	/*0x0cc*/ u32 maxspeedtime240;
	/*0x0d0*/ u32 speedabs[2];
	/*0x0d8*/ u32 speedrel[2];
};

struct hoverpropobj { // objtype 35
	struct defaultobj base;
	struct hov hov;
};

struct hovercarobj {
	struct defaultobj base;
	/*0x5c*/ u8 *ailist;
	/*0x60*/ u16 aioffset;
	/*0x62*/ s16 aireturnlist;
};

struct chopperobj {
	struct defaultobj base;
	/*0x5c*/ u8 *ailist;
	/*0x60*/ u16 aioffset;
	/*0x62*/ s16 aireturnlist;
	/*0x64*/ u32 speed;
	/*0x68*/ f32 speedaim;
	/*0x6c*/ f32 speedtime60;
	/*0x70*/ u32 turnyspeed60;
	/*0x74*/ u32 turnxspeed60;
	/*0x78*/ f32 turnrot60;
	/*0x7c*/ f32 roty;
	/*0x80*/ f32 rotx;
	/*0x84*/ f32 rotz;
	/*0x88*/ struct path *path;
	/*0x8c*/ s32 nextstep;
	/*0x90*/ s16 weaponsarmed;
	/*0x92*/ u16 ontarget;
	/*0x94*/ s16 target;
	/*0x96*/ u8 attackmode;
	/*0x97*/ u8 cw;
	/*0x98*/ f32 vx;
	/*0x9c*/ f32 vy;
	/*0xa0*/ f32 vz;
	/*0xa4*/ f32 power;
	/*0xa8*/ f32 otx;
	/*0xac*/ f32 oty;
	/*0xb0*/ f32 otz;
	/*0xb4*/ f32 bob;
	/*0xb8*/ f32 bobstrength;
	/*0xbc*/ u32 targetvisible;
	/*0xc0*/ u32 timer60;
	/*0xc4*/ u32 patroltimer60;
	/*0xc8*/ u32 gunturnyspeed60;
	/*0xcc*/ u32 gunturnxspeed60;
	/*0xd0*/ f32 gunroty;
	/*0xd4*/ f32 gunrotx;
	/*0xd8*/ f32 barrelrotspeed;
	/*0xdc*/ f32 barrelrot;
};

struct tag {
	// These three words are possibly the same values as defined in the setup
	// file, but this just a guess.
	u32 unk00;
	u16 unk04;
	s16 unk06;
	u32 unk08;
	struct defaultobj *obj;
};

struct eyespy {
	/*0x00*/ struct prop *prop;
	/*0x04*/ struct coord look;
	/*0x10*/ struct coord up;
	/*0x1c*/ f32 theta;
	/*0x20*/ f32 costheta;
	/*0x24*/ f32 sintheta;
	/*0x28*/ f32 verta;
	/*0x2c*/ f32 cosverta;
	/*0x30*/ f32 sinverta;
	/*0x34*/ u8 init;
	/*0x35*/ u8 initialised;
	/*0x36*/ s8 startuptimer60;
	/*0x37*/ s8 active;
	/*0x38*/ u8 buttonheld;
	/*0x39*/ u8 camerabuttonheld;
	/*0x3a*/ s16 bobdir;
	/*0x3c*/ u8 bobtimer;
	/*0x3d*/ u8 bobactive;
	/*0x40*/ struct coord vel;
	/*0x4c*/ struct coord unk4c;
	/*0x58*/ f32 speed;
	/*0x5c*/ f32 oldground;
	/*0x60*/ f32 height; // height above ground - 80 to 160
	/*0x64*/ f32 gravity;
	/*0x68*/ s8 camerashuttertime;
	/*0x69*/ u8 hit;
	/*0x6a*/ u8 opendoor;
	/*0x6b*/ u8 mode;
	/*0x6c*/ s32 velf[2];
	/*0x74*/ s32 vels[2];
	/*0x7c*/ s32 pitch;
};

/**
 * List of guns that the player has equipped and the amount of time they've been
 * equipped for. It's used to determine the weapon of choice for the endscreen.
 * The time is increased for the equipped weapon on each tick.
 *
 * The list is only 10 items long. If an 11th item is used, the least used item
 * in the list will be removed.
 */
struct gunheld {
	s32 weapon1;
	s32 weapon2;
	s32 totaltime240_60;
};

struct playerbond {
	/*0x0338 0x036c*/ struct coord unk00;
	/*0x0344 0x0378*/ f32 width;
	/*0x0348 0x037c*/ struct coord unk10;
	/*0x0354 0x0388*/ struct coord unk1c;
	/*0x0360 0x0394*/ struct coord unk28;
};

struct player1630 {
	struct prop *prop;
	s16 unk04;
	s16 unk06;
	s16 unk08;
	s16 unk0a;
};

struct player638 {
	/*0x0638*/ u8 unk0638;
	/*0x0639*/ u8 unk0639;
	/*0x063a*/ u8 unk063a;
	/*0x063b*/ u8 unk063b;
	/*0x063c*/ u8 unk063c;
	/*0x0640*/ u32 unk0640;
	/*0x0644*/ u32 unk0644;
	/*0x0648*/ u32 unk0648;
	/*0x064c*/ u32 unk064c;
	/*0x0650*/ u32 unk0650;
	/*0x0654*/ u32 unk0654;
	/*0x0658*/ u32 unk0658;
	/*0x065c*/ u32 unk065c;
	/*0x0660*/ u32 unk0660;
	/*0x0664*/ u32 unk0664;
	/*0x0668*/ u32 unk0668;
	/*0x066c*/ u32 unk066c;
	/*0x0670*/ u32 unk0670;
	/*0x0674*/ u32 unk0674;
	/*0x0678*/ u32 unk0678;
	/*0x067c*/ u32 unk067c;
	/*0x0680*/ u32 unk0680;
	/*0x0684*/ u32 unk0684;
	/*0x0688*/ u32 unk0688;
	/*0x068c*/ u32 unk068c;
	/*0x0690*/ u32 unk0690;
	/*0x0694*/ u32 unk0694;
	/*0x0698*/ u32 unk0698;
	/*0x069c*/ u32 unk069c;
	/*0x06a0*/ u32 unk06a0;
	/*0x06a4*/ u32 unk06a4;
	/*0x06a8*/ u32 unk06a8;
	/*0x06ac*/ u32 unk06ac;
	/*0x06b0*/ u32 unk06b0;
	/*0x06b4*/ u32 unk06b4;
	/*0x06b8*/ u32 unk06b8;
	/*0x06bc*/ u32 unk06bc;
	/*0x06c0*/ u32 unk06c0;
	/*0x06c4*/ u32 unk06c4;
	/*0x06c8*/ u32 unk06c8;
	/*0x06cc*/ u32 unk06cc;
	/*0x06d0*/ u32 unk06d0;
	/*0x06d4*/ u32 unk06d4;
	/*0x06d8*/ u32 unk06d8;
	/*0x06dc*/ u32 unk06dc;
	/*0x06e0*/ u32 unk06e0;
	/*0x06e4*/ u32 unk06e4;
	/*0x06e8*/ u32 unk06e8;
	/*0x06ec*/ u32 unk06ec;
	/*0x06f0*/ u32 unk06f0;
	/*0x06f4*/ u32 unk06f4;
	/*0x06f8*/ u32 unk06f8;
	/*0x06fc*/ u32 unk06fc;
	/*0x0700*/ u32 unk0700;
	/*0x0704*/ u32 unk0704;
	/*0x0708*/ u32 unk0708;
	/*0x070c*/ u32 unk070c;
	/*0x0710*/ u32 unk0710;
	/*0x0714*/ u32 unk0714;
	/*0x0718*/ u32 unk0718;
	/*0x071c*/ u32 unk071c;
	/*0x0720*/ u32 unk0720;
	/*0x0724*/ u32 unk0724;
	/*0x0728*/ u32 unk0728;
	/*0x072c*/ u32 unk072c;
	/*0x0730*/ u32 unk0730;
	/*0x0734*/ u32 unk0734;
	/*0x0738*/ u32 unk0738;
	/*0x073c*/ u32 unk073c;
	/*0x0740*/ u32 unk0740;
	/*0x0744*/ u32 unk0744;
	/*0x0748*/ u32 unk0748;
	/*0x074c*/ u32 unk074c;
	/*0x0750*/ u32 unk0750;
	/*0x0754*/ u32 unk0754;
	/*0x0758*/ u32 unk0758;
	/*0x075c*/ u32 unk075c;
	/*0x0760*/ u32 unk0760;
	/*0x0764*/ u32 unk0764;
	/*0x0768*/ u32 unk0768;
	/*0x076c*/ u32 unk076c;
	/*0x0770*/ u32 unk0770;
	/*0x0774*/ u32 unk0774;
	/*0x0778*/ u32 unk0778;
	/*0x077c*/ u32 unk077c;
	/*0x0780*/ u32 unk0780;
	/*0x0784*/ u32 unk0784;
	/*0x0788*/ u32 unk0788;
	/*0x078c*/ u32 unk078c;
	/*0x0790*/ u32 unk0790;
	/*0x0794*/ u32 unk0794;
	/*0x0798*/ u32 unk0798;
	/*0x079c*/ u32 unk079c;
	/*0x07a0*/ u32 unk07a0;
	/*0x07a4*/ u32 unk07a4;
	/*0x07a8*/ u32 unk07a8;
	/*0x07ac*/ u32 unk07ac;
	/*0x07b0*/ u32 unk07b0;
	/*0x07b4*/ u32 unk07b4;
	/*0x07b8*/ u32 unk07b8;
	/*0x07bc*/ u32 unk07bc;
	/*0x07c0*/ u32 unk07c0;
	/*0x07c4*/ u32 unk07c4;
	/*0x07c8*/ u32 unk07c8;
	/*0x07cc*/ u32 unk07cc;
	/*0x07d0*/ u32 unk07d0;
	/*0x07d4*/ u32 unk07d4;
	/*0x07d8*/ u32 unk07d8;
	/*0x07dc*/ u32 unk07dc;
	/*0x07e0*/ u32 unk07e0;
	/*0x07e4*/ u32 unk07e4;
	/*0x07e8*/ u32 unk07e8;
	/*0x07ec*/ u32 unk07ec;
	/*0x07f0*/ u32 unk07f0;
	/*0x07f4*/ u32 unk07f4;
	/*0x07f8*/ struct coord unk07f8;
	/*0x0804*/ u32 unk0804;
	/*0x0808*/ u32 unk0808;
	/*0x080c*/ u32 unk080c;
	/*0x0810*/ u32 unk0810;
	/*0x0814*/ u32 unk0814;
	/*0x0818*/ u32 unk0818;
	/*0x081c*/ u32 unk081c;
	/*0x0820*/ u32 unk0820;
	/*0x0824*/ u32 unk0824;
	/*0x0828*/ u32 unk0828;
	/*0x082c*/ u32 unk082c;
	/*0x0830*/ u32 unk0830;
	/*0x0834*/ u32 unk0834;
	/*0x0838*/ u32 unk0838;
	/*0x083c*/ u32 unk083c;
	/*0x0840*/ u32 unk0840;
	/*0x0844*/ u32 unk0844;
	/*0x0848*/ u32 unk0848;
	/*0x084c*/ u32 unk084c;
	/*0x0850*/ u32 unk0850;
	/*0x0854*/ u32 unk0854;
	/*0x0858*/ u32 unk0858[2];
	/*0x0860*/ u32 unk0860;
	/*0x0864*/ u32 unk0864;
	/*0x0868*/ u32 unk0868;
	/*0x086c*/ u32 unk086c;
	/*0x0870*/ u32 unk0870;
	/*0x0874*/ u32 unk0874;
	/*0x0878*/ u32 unk0878;
	/*0x087c*/ u32 unk087c;
	/*0x0880*/ u32 unk0880;
	/*0x0884*/ u32 unk0884;
	/*0x0888*/ u32 unk0888;
	/*0x088c*/ u32 unk088c;
	/*0x0890*/ u32 unk0890;
	/*0x0894*/ u32 unk0894;
	/*0x0898*/ u32 unk0898;
	/*0x089c*/ u32 unk089c;
	/*0x08a0*/ u32 unk08a0;
	/*0x08a4*/ u32 unk08a4;
	/*0x08a8*/ u32 unk08a8;
	/*0x08ac*/ u32 unk08ac;
	/*0x08b0*/ u32 unk08b0;
	/*0x08b4*/ u32 unk08b4;
	/*0x08b8*/ u32 unk08b8;
	/*0x08bc*/ u32 unk08bc;
	/*0x08c0*/ u32 unk08c0;
	/*0x08c4*/ u32 unk08c4;
	/*0x08c8*/ u32 unk08c8;
	/*0x08cc*/ u32 unk08cc;
	/*0x08d0*/ u32 unk08d0;
	/*0x08d4*/ u32 unk08d4;
	/*0x08d8*/ u32 unk08d8;
	/*0x08dc*/ u32 unk08dc;
	/*0x08e0*/ u32 unk08e0;
	/*0x08e4*/ u32 unk08e4;
	/*0x08e8*/ u32 unk08e8;
	/*0x08ec*/ u32 unk08ec;
	/*0x08f0*/ u32 unk08f0;
	/*0x08f4*/ u32 unk08f4;
	/*0x08f8*/ u32 unk08f8;
	/*0x08fc*/ u32 unk08fc;
	/*0x0900*/ u32 unk0900;
	/*0x0904*/ u32 unk0904;
	/*0x0908*/ u32 unk0908;
	/*0x090c*/ u32 unk090c;
	/*0x0910*/ u32 unk0910;
	/*0x0914*/ u32 unk0914;
	/*0x0918*/ u32 unk0918;
	/*0x091c*/ u32 unk091c;
	/*0x0920*/ u32 unk0920;
	/*0x0924*/ u32 unk0924;
	/*0x0928*/ u32 unk0928;
	/*0x092c*/ u32 unk092c;
	/*0x0930*/ u32 unk0930;
	/*0x0934*/ u32 unk0934;
	/*0x0938*/ u32 unk0938;
	/*0x093c*/ u32 unk093c;
	/*0x0940*/ u32 unk0940;
	/*0x0944*/ u32 unk0944;
	/*0x0948*/ u32 unk0948;
	/*0x094c*/ u32 unk094c;
	/*0x0950*/ u32 unk0950;
	/*0x0954*/ u32 unk0954;
	/*0x0958*/ u32 unk0958;
	/*0x095c*/ u32 unk095c;
	/*0x0960*/ u32 unk0960;
	/*0x0964*/ u32 unk0964;
	/*0x0968*/ u32 unk0968;
	/*0x096c*/ u32 unk096c;
	/*0x0970*/ u32 unk0970;
	/*0x0974*/ u32 unk0974;
	/*0x0978*/ u32 unk0978;
	/*0x097c*/ u32 unk097c;
	/*0x0980*/ u32 unk0980;
	/*0x0984*/ u32 unk0984;
	/*0x0988*/ u32 unk0988;
	/*0x098c*/ u32 unk098c;
	/*0x0990*/ u32 unk0990;
	/*0x0994*/ u32 unk0994;
	/*0x0998*/ u32 unk0998;
	/*0x099c*/ u32 unk099c;
	/*0x09a0*/ u32 unk09a0;
	/*0x09a4*/ u32 unk09a4;
	/*0x09a8*/ u32 unk09a8;
	/*0x09ac*/ u32 unk09ac;
	/*0x09b0*/ u32 unk09b0;
	/*0x09b4*/ u32 unk09b4;
	/*0x09b8*/ u32 unk09b8;
	/*0x09bc*/ u32 unk09bc;
	/*0x09c0*/ u32 unk09c0;
	/*0x09c4*/ u32 unk09c4;
	/*0x09c8*/ u32 unk09c8;
	/*0x09cc*/ u32 unk09cc;
	/*0x09d0*/ u32 unk09d0;
	/*0x09d4*/ u32 unk09d4;
	/*0x09d8*/ u32 unk09d8;
	/*0x09dc*/ u32 unk09dc;
	/*0x09e0*/ u32 unk09e0;
	/*0x09e4*/ u32 unk09e4;
	/*0x09e8*/ u32 unk09e8;
	/*0x09ec*/ u32 unk09ec;
	/*0x09f0*/ u32 unk09f0;
	/*0x09f4*/ u32 unk09f4;
	/*0x09f8*/ u32 unk09f8;
	/*0x09fc*/ u32 unk09fc;
	/*0x0a00*/ u32 unk0a00;
	/*0x0a04*/ u32 unk0a04;
	/*0x0a08*/ u32 unk0a08;
	/*0x0a0c*/ u32 unk0a0c;
	/*0x0a10*/ u32 unk0a10;
	/*0x0a14*/ u32 unk0a14;
	/*0x0a18*/ u32 unk0a18;
	/*0x0a1c*/ u32 unk0a1c;
	/*0x0a20*/ u32 unk0a20;
	/*0x0a24*/ u32 unk0a24;
	/*0x0a28*/ u32 unk0a28;
	/*0x0a2c*/ u32 unk0a2c;
	/*0x0a30*/ u32 unk0a30;
	/*0x0a34*/ u32 unk0a34;
	/*0x0a38*/ u32 unk0a38;
	/*0x0a3c*/ u32 unk0a3c;
	/*0x0a40*/ u32 unk0a40;
	/*0x0a44*/ u32 unk0a44;
	/*0x0a48*/ u32 unk0a48;
	/*0x0a4c*/ u32 unk0a4c;
	/*0x0a50*/ u32 unk0a50;
	/*0x0a54*/ u32 unk0a54;
	/*0x0a58*/ u32 unk0a58;
	/*0x0a5c*/ u32 unk0a5c;
	/*0x0a60*/ u32 unk0a60;
	/*0x0a64*/ u32 unk0a64;
	/*0x0a68*/ u32 unk0a68;
	/*0x0a6c*/ u32 unk0a6c;
	/*0x0a70*/ u32 unk0a70;
	/*0x0a74*/ u32 unk0a74;
	/*0x0a78*/ u32 unk0a78;
	/*0x0a7c*/ u32 unk0a7c;
	/*0x0a80*/ u32 unk0a80;
	/*0x0a84*/ u32 unk0a84;
	/*0x0a88*/ u32 unk0a88;
	/*0x0a8c*/ u32 unk0a8c;
	/*0x0a90*/ u32 unk0a90;
	/*0x0a94*/ u32 unk0a94;
	/*0x0a98*/ u32 unk0a98;
	/*0x0a9c*/ u32 unk0a9c;
	/*0x0aa0*/ u32 unk0aa0;
	/*0x0aa4*/ u32 unk0aa4;
	/*0x0aa8*/ u32 unk0aa8;
	/*0x0aac*/ u32 unk0aac;
	/*0x0ab0*/ u32 unk0ab0;
	/*0x0ab4*/ u32 unk0ab4;
	/*0x0ab8*/ u32 unk0ab8;
	/*0x0abc*/ u32 unk0abc;
	/*0x0ac0*/ u32 unk0ac0;
	/*0x0ac4*/ u32 unk0ac4;
	/*0x0ac8*/ u32 unk0ac8;
	/*0x0acc*/ u32 unk0acc;
	/*0x0ad0*/ u32 unk0ad0;
	/*0x0ad4*/ u32 unk0ad4;
	/*0x0ad8*/ u32 unk0ad8;
	/*0x0adc*/ u32 unk0adc;
	/*0x0ae0*/ u32 unk0ae0;
	/*0x0ae4*/ u32 unk0ae4;
	/*0x0ae8*/ u32 unk0ae8;
	/*0x0aec*/ u32 unk0aec;
	/*0x0af0*/ u32 unk0af0;
	/*0x0af4*/ u32 unk0af4;
	/*0x0af8*/ u32 unk0af8;
	/*0x0afc*/ u32 unk0afc;
	/*0x0b00*/ u32 unk0b00;
	/*0x0b04*/ u32 unk0b04;
	/*0x0b08*/ u32 unk0b08;
	/*0x0b0c*/ u32 unk0b0c;
	/*0x0b10*/ u32 unk0b10;
	/*0x0b14*/ u32 unk0b14;
	/*0x0b18*/ u32 unk0b18;
	/*0x0b1c*/ u32 unk0b1c;
	/*0x0b20*/ u32 unk0b20;
	/*0x0b24*/ u32 unk0b24;
	/*0x0b28*/ u32 unk0b28;
	/*0x0b2c*/ u32 unk0b2c;
	/*0x0b30*/ u32 unk0b30;
	/*0x0b34*/ u32 unk0b34;
	/*0x0b38*/ u32 unk0b38;
	/*0x0b3c*/ u32 unk0b3c;
	/*0x0b40*/ u32 unk0b40;
	/*0x0b44*/ u32 unk0b44;
	/*0x0b48*/ u32 unk0b48;
	/*0x0b4c*/ u32 unk0b4c;
	/*0x0b50*/ u32 unk0b50;
	/*0x0b54*/ u32 unk0b54;
	/*0x0b58*/ u32 unk0b58;
	/*0x0b5c*/ u32 unk0b5c;
	/*0x0b60*/ u32 unk0b60;
	/*0x0b64*/ u32 unk0b64;
	/*0x0b68*/ u32 unk0b68;
	/*0x0b6c*/ u32 unk0b6c;
	/*0x0b70*/ u32 unk0b70;
	/*0x0b74*/ u32 unk0b74;
	/*0x0b78*/ u32 unk0b78;
	/*0x0b7c*/ u32 unk0b7c;
	/*0x0b80*/ u32 unk0b80;
	/*0x0b84*/ u32 unk0b84;
	/*0x0b88*/ u32 unk0b88;
	/*0x0b8c*/ u32 unk0b8c;
	/*0x0b90*/ u32 unk0b90;
	/*0x0b94*/ struct coord unk0b94;
	/*0x0ba0*/ u32 unk0ba0;
	/*0x0ba4*/ u32 unk0ba4;
	/*0x0ba8*/ u32 unk0ba8;
	/*0x0bac*/ u32 unk0bac;
	/*0x0bb0*/ u32 unk0bb0;
	/*0x0bb4*/ u32 unk0bb4;
	/*0x0bb8*/ u32 unk0bb8;
	/*0x0bbc*/ u32 unk0bbc;
	/*0x0bc0*/ u32 unk0bc0;
	/*0x0bc4*/ u32 unk0bc4;
	/*0x0bc8*/ u32 unk0bc8;
	/*0x0bcc*/ u32 unk0bcc;
	/*0x0bd0*/ u32 unk0bd0;
	/*0x0bd4*/ u32 unk0bd4;
	/*0x0bd8*/ u32 unk0bd8;
	/*0x0bdc*/ u32 unk0bdc;
	/*0x0be0*/ u32 unk0be0;
	/*0x0be4*/ u32 unk0be4;
	/*0x0be8*/ u32 unk0be8;
	/*0x0bec*/ u32 unk0bec;
	/*0x0bf0*/ u32 unk0bf0;
	/*0x0bf4*/ u32 unk0bf4;
	/*0x0bf8*/ u32 unk0bf8;
	/*0x0bfc*/ u32 unk0bfc;
	/*0x0c00*/ u32 unk0c00;
	/*0x0c04*/ u32 unk0c04;
	/*0x0c08*/ u32 unk0c08;
	/*0x0c0c*/ u32 unk0c0c;
	/*0x0c10*/ u32 unk0c10;
	/*0x0c14*/ u32 unk0c14;
	/*0x0c18*/ u32 unk0c18;
	/*0x0c1c*/ u32 unk0c1c;
	/*0x0c20*/ u32 unk0c20;
	/*0x0c24*/ u32 unk0c24;
	/*0x0c28*/ u32 unk0c28;
	/*0x0c2c*/ u32 unk0c2c;
	/*0x0c30*/ u32 unk0c30;
	/*0x0c34*/ u32 unk0c34;
	/*0x0c38*/ u32 unk0c38;
	/*0x0c3c*/ u32 unk0c3c;
	/*0x0c40*/ u32 unk0c40;
	/*0x0c44*/ u32 unk0c44;
	/*0x0c48*/ u32 unk0c48;
	/*0x0c4c*/ u32 unk0c4c;
	/*0x0c50*/ u32 unk0c50;
	/*0x0c54*/ u32 unk0c54;
	/*0x0c58*/ u32 unk0c58;
	/*0x0c5c*/ u32 unk0c5c;
	/*0x0c60*/ u32 unk0c60;
	/*0x0c64*/ u32 unk0c64;
	/*0x0c68*/ u32 unk0c68;
	/*0x0c6c*/ u32 unk0c6c;
	/*0x0c70*/ u32 unk0c70;
	/*0x0c74*/ u32 unk0c74;
	/*0x0c78*/ u32 unk0c78;
	/*0x0c7c*/ u32 unk0c7c;
	/*0x0c80*/ u32 unk0c80;
	/*0x0c84*/ u32 unk0c84;
	/*0x0c88*/ u32 unk0c88;
	/*0x0c8c*/ u32 unk0c8c;
	/*0x0c90*/ u32 unk0c90;
	/*0x0c94*/ u32 unk0c94;
	/*0x0c98*/ u32 unk0c98;
	/*0x0c9c*/ u32 unk0c9c;
	/*0x0ca0*/ u32 unk0ca0;
	/*0x0ca4*/ u32 unk0ca4;
	/*0x0ca8*/ u32 unk0ca8;
	/*0x0cac*/ u32 unk0cac;
	/*0x0cb0*/ u32 unk0cb0;
	/*0x0cb4*/ u32 unk0cb4;
	/*0x0cb8*/ u32 unk0cb8;
	/*0x0cbc*/ u32 unk0cbc;
	/*0x0cc0*/ u32 unk0cc0;
	/*0x0cc4*/ u32 unk0cc4;
	/*0x0cc8*/ u32 unk0cc8;
	/*0x0ccc*/ u32 unk0ccc;
	/*0x0cd0*/ u32 unk0cd0;
	/*0x0cd4*/ u32 unk0cd4;
	/*0x0cd8*/ u32 unk0cd8;
	/*0x0cdc*/ u32 unk0cdc;
	/*0x0ce0*/ u32 unk0ce0;
	/*0x0ce4*/ u32 unk0ce4;
	/*0x0ce8*/ u32 unk0ce8;
	/*0x0cec*/ bool unk0cec;
	/*0x0cf0*/ struct coord unk0cf0;
	/*0x0cfc*/ struct coord unk0cfc;
	/*0x0d08*/ u32 unk0d08;
	/*0x0d0c*/ u8 unk0d0c;
	/*0x0d0e*/ u8 unk0d0d;
	/*0x0d0d*/ u8 unk0d0e;
	/*0x0d0f*/ u8 unk0d0f_00 : 3;
	/*0x0d0f*/ u8 unk0d0f_03 : 1;
	/*0x0d0f*/ u8 unk0d0f_04 : 4;
	/*0x0d10*/ u32 unk0d10;
	/*0x0d14*/ u32 unk0d14;
	/*0x0d18*/ u32 unk0d18;
	/*0x0d1c*/ u32 unk0d1c;
	/*0x0d20*/ u32 unk0d20;
	/*0x0d24*/ u32 unk0d24;
	/*0x0d28*/ u32 unk0d28;
	/*0x0d2c*/ u32 unk0d2c;
	/*0x0d30*/ u32 unk0d30;
	/*0x0d34*/ u32 unk0d34;
	/*0x0d38*/ u32 unk0d38;
	/*0x0d3c*/ u32 unk0d3c;
	/*0x0d40*/ u32 unk0d40;
	/*0x0d44*/ u32 unk0d44;
	/*0x0d48*/ u32 unk0d48;
	/*0x0d4c*/ u32 unk0d4c;
	/*0x0d50*/ u32 unk0d50;
	/*0x0d54*/ u32 unk0d54;
	/*0x0d58*/ u32 unk0d58;
	/*0x0d5c*/ u32 unk0d5c;
	/*0x0d60*/ u32 unk0d60;
	/*0x0d64*/ u32 unk0d64;
	/*0x0d68*/ u32 unk0d68;
	/*0x0d6c*/ u32 unk0d6c;
	/*0x0d70*/ u32 unk0d70;
	/*0x0d74*/ u16 unk0d74[4];
	/*0x0d7c*/ u32 unk0d7c;
	/*0x0d80*/ u32 unk0d80;
	/*0x0d84*/ void *unk0d84;
	/*0x0d88*/ u32 unk0d88;
	/*0x0d8c*/ u32 unk0d8c;
	/*0x0d90*/ u32 unk0d90;
	/*0x0d94*/ u32 unk0d94;
	/*0x0d98*/ u32 unk0d98;
	/*0x0d9c*/ u32 unk0d9c;
	/*0x0da0*/ u32 unk0da0;
	/*0x0da4*/ u32 unk0da4;
	/*0x0da8*/ u32 unk0da8;
	/*0x0dac*/ u32 unk0dac;
	/*0x0db0*/ u32 unk0db0;
	/*0x0db4*/ u32 unk0db4;
	/*0x0db8*/ u32 unk0db8;
	/*0x0dbc*/ u32 unk0dbc;
	/*0x0dc0*/ u32 unk0dc0;
	/*0x0dc4*/ u32 unk0dc4;
	/*0x0dc8*/ u32 unk0dc8;
	/*0x0dcc*/ u32 unk0dcc;
	/*0x0dd0*/ u32 unk0dd0;
	/*0x0dd4*/ u32 unk0dd4;
	/*0x0dd8*/ u32 unk0dd8;
};

struct player {
	/*0x0000*/ s32 cameramode;
	/*0x0004*/ struct coord memcampos;
	/*0x0010*/ u16 visionmode;
	/*0x0014*/ s32 memcamroom;
	/*0x0018*/ u32 unk0018;
	/*0x001c*/ u32 unk001c;
	/*0x0020*/ u32 unk0020;
	/*0x0024*/ u32 unk0024;
	/*0x0028*/ u32 unk0028;
	/*0x002c*/ u32 unk002c;
	/*0x0030*/ bool isfalling;
	/*0x0034*/ s32 fallstart;
	/*0x0038*/ struct coord globaldrawworldoffset;
	/*0x0044*/ struct coord globaldrawcameraoffset;
	/*0x0050*/ struct coord globaldrawworldbgoffset;
	/*0x005c*/ f32 *matrix;
	/*0x0060*/ u32 unk0060;
	/*0x0064*/ u32 unk0064;
	/*0x0068*/ u32 unk0068;
	/*0x006c*/ u32 unk006c;
	/*0x0070*/ f32 sumground;
	/*0x0074*/ f32 vv_manground;
	/*0x0078*/ f32 vv_ground;
	/*0x007c*/ struct coord bdeltapos;
	/*0x0088*/ f32 sumcrouch;
	/*0x008c*/ f32 crouchheight;
	/*0x0090*/ s32 crouchtime240;
	/*0x0094*/ f32 crouchfall;
	/*0x0098*/ s32 swaypos;
	/*0x009c*/ f32 swayoffset;
	/*0x00a0*/ f32 swaytarget;
	/*0x00a4*/ f32 swayoffset0;
	/*0x00a8*/ f32 swayoffset2;
	/*0x00ac*/ s32 crouchpos;
	/*0x00b0*/ s32 autocrouchpos;
	/*0x00b4*/ f32 crouchoffset;
	/*0x00b8*/ f32 crouchspeed;
	/*0x00bc*/ struct prop *prop;
	/*0x00c0*/ u32 unk00c0;
	/*0x00c4*/ u32 unk00c4;
	/*0x00c8*/ u32 unk00c8;
	/*0x00cc*/ u32 unk00cc;
	/*0x00d0*/ u32 unk00d0;
	/*0x00d4*/ u32 unk00d4;
	/*0x00d8*/ bool isdead;
	/*0x00dc*/ f32 bondhealth;
	/*0x00e0*/ u32 unk00e0;
	/*0x00e4*/ f32 unk00e4;
	/*0x00e8*/ f32 unk00e8;
	/*0x00ec*/ f32 unk00ec;
	/*0x00f0*/ f32 unk00f0;
	/*0x00f4*/ u32 unk00f4;
	/*0x00f8*/ u32 unk00f8;
	/*0x00fc*/ u32 unk00fc;
	/*0x0100*/ u32 unk0100;
	/*0x0104*/ u32 unk0104;
	/*0x0108*/ u32 unk0108;
	/*0x010c*/ u32 unk010c;
	/*0x0110*/ u32 unk0110;
	/*0x0114*/ bool automovecentreenabled;
	/*0x0118*/ u32 unk0118;
	/*0x011c*/ u32 unk011c;
	/*0x0120*/ bool insightaimmode;

	/*0x0124*/ bool autoyaimenabled;
	/*0x0128*/ f32 autoaimy;
	/*0x012c*/ struct prop *autoyaimprop;
	/*0x0130*/ s32 autoyaimtime60;

	/*0x0134*/ bool autoxaimenabled;
	/*0x0138*/ f32 autoaimx;
	/*0x013c*/ struct prop *autoxaimprop;
	/*0x0140*/ s32 autoxaimtime60;

	/*0x0144*/ f32 vv_theta;   // turn angle
	/*0x0148*/ f32 speedtheta; // turn speed
	/*0x014c*/ f32 vv_costheta;
	/*0x0150*/ f32 vv_sintheta;

	/*0x0154*/ f32 vv_verta;   // look up/down angle. 0 = horizontal, 90 = up
	/*0x0158*/ f32 vv_verta360;
	/*0x015c*/ f32 speedverta; // look up/down speed
	/*0x0160*/ f32 vv_cosverta;
	/*0x0164*/ f32 vv_sinverta;

	/*0x0168*/ f32 speedsideways;
	/*0x016c*/ f32 speedstrafe;
	/*0x0170*/ f32 speedforwards;
	/*0x0174*/ f32 speedboost;
	/*0x0178*/ u32 speedmaxtime60;
	/*0x017c*/ f32 bondshotspeed[3];
	/*0x0188*/ f32 unk0188;
	/*0x018c*/ f32 unk018c;
	/*0x0190*/ f32 unk0190;
	/*0x0194*/ f32 unk0194;
	/*0x0198*/ f32 unk0198;
	/*0x019c*/ s32 activatetimelast;
	/*0x01a0*/ s32 activatetimethis;
	/*0x01a4*/ struct coord moveinitspeed;
	/*0x01b0*/ u32 bondmovemode;
	/*0x01b4*/ f32 gunextraaimx;
	/*0x01b8*/ f32 gunextraaimy;
	/*0x01bc*/ u32 unk01bc;
	/*0x01c0*/ u32 unk01c0;
	/*0x01c4*/ u32 unk01c4;
	/*0x01c8*/ u32 unk01c8;
	/*0x01cc*/ u32 unk01cc;
	/*0x01d0*/ u32 unk01d0;
	/*0x01d4*/ u32 unk01d4;
	/*0x01d8*/ u32 unk01d8;
	/*0x01dc*/ u32 unk01dc;
	/*0x01e0*/ u32 unk01e0;
	/*0x01e4*/ u32 unk01e4;
	/*0x01e8*/ u32 unk01e8;
	/*0x01ec*/ u32 unk01ec;
	/*0x01f0*/ u32 unk01f0;
	/*0x01f4*/ u32 unk01f4;
	/*0x01f8*/ u32 unk01f8;
	/*0x01fc*/ u32 unk01fc;
	/*0x0200*/ u32 unk0200;
	/*0x0204*/ u32 unk0204;
	/*0x0208*/ u32 unk0208;
	/*0x020c*/ u32 unk020c;
	/*0x0210*/ u32 unk0210;
	/*0x0214*/ u32 unk0214;
	/*0x0218*/ u32 unk0218;
	/*0x021c*/ u32 unk021c;
	/*0x0220*/ u32 unk0220;
	/*0x0224*/ u32 unk0224;
	/*0x0228*/ u32 unk0228;
	/*0x022c*/ u32 unk022c;
	/*0x0230*/ u32 unk0230;
	/*0x0234*/ u32 unk0234;
	/*0x0238*/ u32 unk0238;
	/*0x023c*/ u32 unk023c;
	/*0x0240*/ u32 unk0240;
	/*0x0244*/ u32 unk0244;
	/*0x0248*/ u32 unk0248;
	/*0x024c*/ s16 invdowntime;
	/*0x024e*/ s16 usedowntime;
	/*0x0250*/ u8 activemenumode;
	/*0x0254*/ u32 unk0254;
	/*0x0258*/ u32 unk0258;
	/*0x025c*/ u32 unk025c;
	/*0x0260*/ u32 unk0260;
	/*0x0264*/ u32 unk0264;
	/*0x0268*/ bool eyesshut;
	/*0x026c*/ f32 eyesshutfrac;
	/*0x0270*/ u32 unk0270;
	/*0x0274*/ u32 unk0274;
	/*0x0278*/ u32 unk0278;
	/*0x027c*/ u32 unk027c;
	/*0x0280*/ u32 unk0280;
	/*0x0284*/ s16 bondprevrooms[8];
	/*0x0294*/ f32 liftground;
	/*0x0298*/ struct prop *lift;
	/*0x029c*/ u32 unk029c;
	/*0x02a0*/ u32 unk02a0;
	/*0x02a4*/ u32 unk02a4;
	/*0x02a8*/ u32 unk02a8;
	/*0x02ac*/ bool onladder;
	/*0x02b0*/ bool inlift;
	/*0x02b4*/ struct coord posdie;
	/*0x02c0*/ struct coord bonddampeyesum;
	/*0x02cc*/ struct coord bonddampeye;
	/*0x02d8*/ u32 colourscreenred;
	/*0x02dc*/ u32 colourscreengreen;
	/*0x02e0*/ u32 colourscreenblue;
	/*0x02e4*/ f32 colourscreenfrac;
	/*0x02e8*/ f32 colourfadetime60;
	/*0x02ec*/ f32 colourfadetimemax60;
	/*0x02f0*/ u32 colourfaderedold;
	/*0x02f4*/ u32 colourfaderednew;
	/*0x02f8*/ u32 colourfadegreenold;
	/*0x02fc*/ u32 colourfadegreennew;
	/*0x0300*/ u32 colourfadeblueold;
	/*0x0304*/ u32 colourfadebluenew;
	/*0x0308*/ f32 colourfadefracold;
	/*0x030c*/ f32 colourfadefracnew;
	/*0x0310*/ struct coord bondprevpos;
	/*0x031c*/ f32 thetadie;
	/*0x0320*/ f32 vertadie;
	/*0x0324*/ u32 bondtype;
	/*0x0328*/ bool startnewbonddie;
	/*0x032c*/ bool redbloodfinished;
	/*0x0330*/ bool deathanimfinished;
	/*0x0334*/ u32 controldef;
	/*0x0338*/ struct playerbond bonddie;
	/*0x036c*/ struct playerbond bond2;
	/*0x03a0*/ bool resetheadpos;
	/*0x03a4*/ bool resetheadrot;
	/*0x03a8*/ u32 unk03a8;
	/*0x03ac*/ s32 unk03ac;
	/*0x03b0*/ f32 headamp;
	/*0x03b4*/ u32 unk03b4;
	/*0x03b8*/ u32 unk03b8;
	/*0x03bc*/ u32 unk03bc;
	/*0x03c0*/ struct coord headpos;
	/*0x03cc*/ struct coord headlook;
	/*0x03d8*/ struct coord headup;
	/*0x03e4*/ struct coord headpossum;
	/*0x03f0*/ struct coord headlooksum;
	/*0x03fc*/ struct coord headupsum;
	/*0x0408*/ struct coord headbodyoffset;
	/*0x0414*/ f32 standheight;
	/*0x0418*/ struct coord standbodyoffset;
	/*0x0424*/ f32 standfrac;
	/*0x0428*/ struct coord standlook[2];
	/*0x0440*/ struct coord standup[2];
	/*0x0458*/ s32 standcnt;
	/*0x045c*/ struct animdata animdata;
	/*0x0480*/ struct eyespy *eyespy;
	/*0x0484*/ u32 unk0484;
	/*0x0488*/ u32 unk0488;
	/*0x048c*/ bool aborted;
	/*0x0490*/ u32 unk0490;
	/*0x0494*/ u32 unk0494;
	/*0x0498*/ u32 unk0498;
	/*0x049c*/ u32 unk049c;
	/*0x04a0*/ u32 unk04a0;
	/*0x04a4*/ u32 unk04a4;
	/*0x04a8*/ u32 unk04a8;
	/*0x04ac*/ u32 unk04ac;
	/*0x04b0*/ u32 unk04b0;
	/*0x04b4*/ u32 unk04b4;
	/*0x04b8*/ u32 unk04b8;
	/*0x04bc*/ u32 unk04bc;
	/*0x04c0*/ u32 unk04c0;
	/*0x04c4*/ u32 unk04c4;
	/*0x04c8*/ u32 unk04c8;
	/*0x04cc*/ u32 unk04cc;
	/*0x04d0*/ u32 unk04d0;
	/*0x04d4*/ u32 unk04d4;
	/*0x04d8*/ u32 unk04d8;
	/*0x04dc*/ u32 unk04dc;
	/*0x04e0*/ u32 unk04e0;
	/*0x04e4*/ u32 unk04e4;
	/*0x04e8*/ u32 unk04e8;
	/*0x04ec*/ u32 unk04ec;
	/*0x04f0*/ u32 unk04f0;
	/*0x04f4*/ u32 unk04f4;
	/*0x04f8*/ u32 unk04f8;
	/*0x04fc*/ u32 unk04fc;
	/*0x0500*/ u32 unk0500;
	/*0x0504*/ u32 unk0504;
	/*0x0508*/ u32 unk0508;
	/*0x050c*/ u32 unk050c;
	/*0x0510*/ u32 unk0510;
	/*0x0514*/ u32 unk0514;
	/*0x0518*/ u32 unk0518;
	/*0x051c*/ u32 unk051c;
	/*0x0520*/ u32 unk0520;
	/*0x0524*/ u32 unk0524;
	/*0x0528*/ u32 unk0528;
	/*0x052c*/ u32 unk052c;
	/*0x0530*/ u32 unk0530;
	/*0x0534*/ u32 unk0534;
	/*0x0538*/ u32 unk0538;
	/*0x053c*/ u32 unk053c;
	/*0x0540*/ u32 unk0540;
	/*0x0544*/ u32 unk0544;
	/*0x0548*/ u32 unk0548;
	/*0x054c*/ u32 unk054c;
	/*0x0550*/ u32 unk0550;
	/*0x0554*/ u32 unk0554;
	/*0x0558*/ u32 unk0558;
	/*0x055c*/ u32 unk055c;
	/*0x0560*/ u32 unk0560;
	/*0x0564*/ u32 unk0564;
	/*0x0568*/ u32 unk0568;
	/*0x056c*/ u32 unk056c;
	/*0x0570*/ u32 unk0570;
	/*0x0574*/ u32 unk0574;
	/*0x0578*/ u32 unk0578;
	/*0x057c*/ u32 unk057c;
	/*0x0580*/ u32 unk0580;
	/*0x0584*/ u32 unk0584;
	/*0x0588*/ u32 unk0588;
	/*0x058c*/ u32 unk058c;
	/*0x0590*/ u32 unk0590;
	/*0x0594*/ u32 unk0594;
	/*0x0598*/ u32 unk0598;
	/*0x059c*/ u32 unk059c;
	/*0x05a0*/ u32 unk05a0;
	/*0x05a4*/ u32 unk05a4;
	/*0x05a8*/ u32 unk05a8;
	/*0x05ac*/ u32 unk05ac;
	/*0x05b0*/ u32 unk05b0;
	/*0x05b4*/ u32 unk05b4;
	/*0x05b8*/ u32 unk05b8;
	/*0x05bc*/ u32 unk05bc;
	/*0x05c0*/ u32 unk05c0;
	/*0x05c4*/ u32 unk05c4;
	/*0x05c8*/ u32 unk05c8;
	/*0x05cc*/ u32 unk05cc;
	/*0x05d0*/ u32 unk05d0;
	/*0x05d4*/ u32 unk05d4;
	/*0x05d8*/ u32 unk05d8;
	/*0x05dc*/ u32 unk05dc;
	/*0x05e0*/ u32 unk05e0;
	/*0x05e4*/ u32 unk05e4;
	/*0x05e8*/ u32 unk05e8;
	/*0x05ec*/ u32 unk05ec;
	/*0x05f0*/ u32 unk05f0;
	/*0x05f4*/ u32 unk05f4;
	/*0x05f8*/ u32 unk05f8;
	/*0x05fc*/ u32 unk05fc;
	/*0x0600*/ u32 unk0600;
	/*0x0604*/ u32 unk0604;
	/*0x0608*/ u32 unk0608;
	/*0x060c*/ u32 unk060c;
	/*0x0610*/ u32 unk0610;
	/*0x0614*/ u32 unk0614;
	/*0x0618*/ u32 unk0618;
	/*0x061c*/ u32 unk061c;
	/*0x0620*/ u32 unk0620;
	/*0x0624*/ u32 unk0624;
	/*0x0628*/ u32 unk0628;
	/*0x062c*/ u32 unk062c;
	/*0x0630*/ s16 viewx;    // 320 lo-res, 640 hi-res
	/*0x0632*/ s16 viewy;    // 220 regardless of res
	/*0x0634*/ s16 viewleft; // 0
	/*0x0636*/ s16 viewtop;  // 0
	/*0x0638*/ struct player638 unk0638[2];
	/*0x1580*/ s8 unk1580;
	/*0x1581*/ u8 unk1581;
	/*0x1582*/ u8 unk1582;
	/*0x1583*/ u8 unk1583_00 : 1;
	/*0x1583*/ u8 unk1583_01 : 1;
	/*0x1583*/ u8 invertgunfunc : 1;
	/*0x1583*/ u8 unk1583_03 : 1;
	/*0x1583*/ u8 unk1583_04 : 1;
	/*0x1583*/ u8 unk1583_05 : 1;
	/*0x1583*/ u8 unk1583_06 : 1;
	/*0x1583*/ u8 passivemode : 1;
	/*0x1584*/ u32 unk1584;
	/*0x1588*/ u32 unk1588;
	/*0x158c*/ u32 unk158c;
	/*0x1590*/ u32 unk1590;
	/*0x1594*/ u32 unk1594;
	/*0x1598*/ u32 unk1598;
	/*0x159c*/ u32 unk159c;
	/*0x15a0*/ u32 unk15a0;
	/*0x15a4*/ u32 unk15a4;
	/*0x15a8*/ u32 unk15a8;
	/*0x15ac*/ u32 unk15ac;
	/*0x15b0*/ u32 unk15b0;
	/*0x15b4*/ u32 unk15b4;
	/*0x15b8*/ u32 unk15b8;
	/*0x15bc*/ u32 unk15bc;
	/*0x15c0*/ u32 unk15c0;
	/*0x15c4*/ u32 unk15c4;
	/*0x15c8*/ u32 unk15c8;
	/*0x15cc*/ u32 unk15cc;
	/*0x15d0*/ u32 unk15d0;
	/*0x15d4*/ u32 unk15d4;
	/*0x15d8*/ u32 unk15d8;
	/*0x15dc*/ u32 unk15dc;
	/*0x15e0*/ u32 unk15e0;
	/*0x15e4*/ s8 unk15e4[2];
	/*0x15e8*/ u32 unk15e8;
	/*0x15ec*/ u32 unk15ec;
	/*0x15f0*/ u32 unk15f0;
	/*0x15f4*/ u32 unk15f4;
	/*0x15f8*/ u32 unk15f8;
	/*0x15fc*/ u32 unk15fc;
	/*0x1600*/ u32 unk1600;
	/*0x1604*/ u32 unk1604;
	/*0x1608*/ u32 unk1608;
	/*0x160c*/ u32 unk160c;
	/*0x1610*/ u32 unk1610;
	/*0x1614*/ u32 unk1614;
	/*0x1618*/ u16 resetshadecol;
	/*0x161a*/ u16 floorcol;
	/*0x161c*/ u16 floorflags;
	/*0x161e*/ u8 floortype;
	/*0x1620*/ u32 aimtype;
	/*0x1624*/ struct prop *lookingatprop;
	/*0x1628*/ u32 unk1628;
	/*0x162c*/ u32 unk162c;
	/*0x1630*/ struct player1630 unk1630[4];
	/*0x1660*/ u32 unk1660;
	/*0x1664*/ u32 unk1664;
	/*0x1668*/ u32 unk1668;
	/*0x166c*/ u32 unk166c;
	/*0x1670*/ u32 unk1670;
	/*0x1674*/ u32 unk1674;
	/*0x1678*/ u32 unk1678;
	/*0x167c*/ u32 unk167c;
	/*0x1680*/ u32 unk1680;
	/*0x1684*/ u32 unk1684;
	/*0x1688*/ u32 unk1688;
	/*0x168c*/ u32 unk168c;
	/*0x1690*/ u32 unk1690;
	/*0x1694*/ u32 unk1694;
	/*0x1698*/ u32 unk1698;
	/*0x169c*/ u32 unk169c;
	/*0x16a0*/ u32 unk16a0;
	/*0x16a4*/ u32 unk16a4;
	/*0x16a8*/ u32 unk16a8;
	/*0x16ac*/ u32 unk16ac;
	/*0x16b0*/ u32 unk16b0;
	/*0x16b4*/ u32 unk16b4;
	/*0x16b8*/ u32 unk16b8;
	/*0x16bc*/ u32 unk16bc;
	/*0x16c0*/ u32 unk16c0;
	/*0x16c4*/ u32 unk16c4;
	/*0x16c8*/ u32 unk16c8;
	/*0x16cc*/ u32 unk16cc;
	/*0x16d0*/ u32 unk16d0;
	/*0x16d4*/ u32 unk16d4;
	/*0x16d8*/ u32 unk16d8;
	/*0x16dc*/ u32 unk16dc;
	/*0x16e0*/ u32 unk16e0;
	/*0x16e4*/ u32 unk16e4;
	/*0x16e8*/ u32 unk16e8;
	/*0x16ec*/ u32 unk16ec;
	/*0x16f0*/ u32 unk16f0;
	/*0x16f4*/ f32 gunzoomfovs[3];
	/*0x1700*/ u32 lastroomforoffset;
	/*0x1704*/ f32 c_screenwidth;
	/*0x1708*/ f32 c_screenheight;
	/*0x170c*/ f32 c_screenleft;
	/*0x1710*/ f32 c_screentop;
	/*0x1714*/ f32 c_perspnear;
	/*0x1718*/ f32 c_perspfovy;
	/*0x171c*/ f32 c_perspaspect;
	/*0x1720*/ f32 c_halfwidth;
	/*0x1724*/ f32 c_halfheight;
	/*0x1728*/ f32 c_scalex;
	/*0x172c*/ f32 c_scaley;
	/*0x1730*/ f32 c_recipscalex;
	/*0x1734*/ f32 c_recipscaley;
	/*0x1738*/ u32 unk1738;
	/*0x173c*/ u32 unk173c;
	/*0x1740*/ f32 *unk1740;
	/*0x1744*/ u32 unk1744;
	/*0x1748*/ u32 unk1748;
	/*0x174c*/ u32 unk174c;
	/*0x1750*/ u32 unk1750;
	/*0x1754*/ u32 unk1754;
	/*0x1758*/ u32 unk1758;
	/*0x175c*/ u32 unk175c;
	/*0x1760*/ f32 *unk1760;
	/*0x1764*/ u32 unk1764;
	/*0x1768*/ u32 unk1768;
	/*0x176c*/ f32 c_scalelod60;
	/*0x1770*/ f32 c_scalelod;
	/*0x1774*/ f32 c_lodscalez;
	/*0x1778*/ u32 c_lodscalezu32;
	/*0x177c*/ struct coord c_cameratopnorm;
	/*0x1788*/ struct coord c_cameraleftnorm;
	/*0x1794*/ f32 screenxminf;
	/*0x1798*/ f32 screenyminf;
	/*0x179c*/ f32 screenxmaxf;
	/*0x17a0*/ f32 screenymaxf;
	/*0x17a4*/ u32 unk17a4;
	/*0x17a8*/ u32 unk17a8;
	/*0x17ac*/ u32 unk17ac;
	/*0x17b0*/ u32 unk17b0;
	/*0x17b4*/ u32 unk17b4;
	/*0x17b8*/ u32 unk17b8;
	/*0x17bc*/ u32 unk17bc;
	/*0x17c0*/ u32 unk17c0;
	/*0x17c4*/ u32 unk17c4;
	/*0x17c8*/ u32 unk17c8;
	/*0x17cc*/ u32 unk17cc;
	/*0x17d0*/ u32 unk17d0;
	/*0x17d4*/ u32 unk17d4;
	/*0x17d8*/ u32 unk17d8;
	/*0x17dc*/ u32 unk17dc;
	/*0x17e0*/ u32 unk17e0;
	/*0x17e4*/ u32 unk17e4;
	/*0x17e8*/ u32 unk17e8;
	/*0x17ec*/ u32 unk17ec;
	/*0x17f0*/ u32 unk17f0;
	/*0x17f4*/ u32 unk17f4;
	/*0x17f8*/ u32 unk17f8;
	/*0x17fc*/ u32 unk17fc;
	/*0x1800*/ u32 unk1800;
	/*0x1804*/ u32 unk1804;
	/*0x1808*/ u32 unk1808;
	/*0x180c*/ u32 unk180c;
	/*0x1810*/ u32 unk1810;
	/*0x1814*/ u32 unk1814;
	/*0x1818*/ u32 unk1818;
	/*0x181c*/ u32 unk181c;
	/*0x1820*/ u32 unk1820;
	/*0x1824*/ u32 unk1824;
	/*0x1828*/ u32 unk1828;
	/*0x182c*/ u32 unk182c;
	/*0x1830*/ u32 unk1830;
	/*0x1834*/ u32 unk1834;
	/*0x1838*/ u32 unk1838;
	/*0x183c*/ u32 unk183c;
	/*0x1840*/ f32 zoomintime;
	/*0x1844*/ f32 zoomintimemax;
	/*0x1848*/ f32 zoominfovy;
	/*0x184c*/ f32 zoominfovyold;
	/*0x1850*/ f32 zoominfovynew;
	/*0x1854*/ f32 fovy;
	/*0x1858*/ f32 aspect;
	/*0x185c*/ u32 flags;
	/*0x1860*/ u32 unk1860;
	/*0x1864*/ struct invitem *weapons; // circular linked list
	/*0x1868*/ struct invitem *equipment;
	/*0x186c*/ s32 equipmaxitems;
	/*0x1870*/ u32 equipallguns;
	/*0x1874*/ u32 equipcuritem;
	/*0x1878*/ struct gunheld gunheldarr[MAX_GUNHELD];
	/*0x18f0*/ u32 magnetattracttime;
	/*0x18f4*/ u32 angleoffset;
	/*0x18f8*/ u32 buthist[10];
	/*0x1920*/ u8 buthistindex;
	/*0x1921*/ u8 buthistlen;
	/*0x1922*/ u8 invincible;
	/*0x1924*/ u32 healthdamagetype;
	/*0x1928*/ f32 bondleandown;
	/*0x192c*/ bool mpmenuon;
	/*0x1930*/ u32 mpmenumode;
	/*0x1934*/ u32 mpquitconfirm;
	/*0x1938*/ u32 mpjoywascentre;
	/*0x193c*/ u32 damagetype;
	/*0x1940*/ u32 deathcount;
	/*0x1948*/ u32 oldcrosspos[2];
	/*0x194c*/ u32 lastkilltime60;
	/*0x1950*/ u32 lastkilltime60_2;
	/*0x1954*/ u32 lastkilltime60_3;
	/*0x1958*/ u32 lastkilltime60_4;
	/*0x195c*/ s32 lifestarttime60;
	/*0x1960*/ u32 killsthislife;
	/*0x1964*/ u32 healthdisplaytime60;
	/*0x1968*/ f32 guncloseroffset;
	/*0x196c*/ u32 shootrotx;
	/*0x1970*/ u32 shootroty;
	/*0x1974*/ char *award1;
	/*0x1978*/ char *award2;
	/*0x197c*/ u32 unk197c;
	/*0x1980*/ u32 unk1980;
	/*0x1984*/ u32 unk1984;
	/*0x1988*/ u16 unk1988;
	/*0x198c*/ u32 unk198c;
	/*0x1990*/ u32 unk1990;
	/*0x1994*/ u32 unk1994;
	/*0x1998*/ u32 unk1998;
	/*0x199c*/ u32 unk199c;
	/*0x19a0*/ u32 unk19a0;
	/*0x19a4*/ f32 speedgo;
	/*0x19a8*/ u32 unk19a8;
	/*0x19ac*/ s32 crouchoffsetreal;
	/*0x19b0*/ s16 floorroom;
	/*0x19b2*/ u8 unk19b2;
	/*0x19b3*/ u8 dostartnewlife;
	/*0x19b4*/ f32 crouchoffsetsmall;
	/*0x19b8*/ s32 crouchoffsetrealsmall; // 0 = standing, -90 = squatting, can be between during transition
	/*0x19bc*/ f32 vv_height;
	/*0x19c0*/ f32 vv_headheight;
	/*0x19c4*/ f32 vv_eyeheight;
	/*0x19c8*/ u32 unk19c8;
	/*0x19cc*/ u32 unk19cc;
	/*0x19d0*/ u32 unk19d0;
	/*0x19d4*/ u32 unk19d4;
	/*0x19d8*/ u32 unk19d8;
	/*0x19dc*/ u32 unk19dc;
	/*0x19e0*/ u32 unk19e0;
	/*0x19e4*/ u32 unk19e4;
	/*0x19e8*/ u32 unk19e8;
	/*0x19ec*/ u32 unk19ec;
	/*0x19f0*/ u32 unk19f0;
	/*0x19f4*/ u32 unk19f4;
	/*0x19f8*/ u32 unk19f8;
	/*0x19fc*/ f32 bondprevtheta;
	/*0x1a00*/ struct coord grabbedprevpos;
	/*0x1a0c*/ f32 grabbedrotoffset;
	/*0x1a10*/ struct coord grabbedposoffset;
	/*0x1a1c*/ s32 grabbeddoextra;
	/*0x1a20*/ f32 grabbedrotextra;
	/*0x1a24*/ u32 pausemode;
	/*0x1a28*/ u32 pausetime60;
	/*0x1a2c*/ struct coord grabbedposextra;
	/*0x1a38*/ f32 grabbedrotextrasum;
	/*0x1a3c*/ struct coord grabbedposextrasum;
	/*0x1a48*/ u32 unk1a48;
	/*0x1a4c*/ u32 unk1a4c;
	/*0x1a50*/ u32 unk1a50;
	/*0x1a54*/ u32 unk1a54;
	/*0x1a58*/ u32 unk1a58;
	/*0x1a5c*/ u32 unk1a5c;
	/*0x1a60*/ u32 unk1a60;
	/*0x1a64*/ u32 unk1a64;
	/*0x1a68*/ u32 unk1a68;
	/*0x1a6c*/ struct prop *hoverbike;
	/*0x1a70*/ f32 unk1a70;
	/*0x1a74*/ f32 unk1a74;
	/*0x1a78*/ f32 unk1a78;
	/*0x1a7c*/ u32 unk1a7c;
	/*0x1a80*/ f32 unk1a80;
	/*0x1a84*/ f32 unk1a84;
	/*0x1a88*/ u32 unk1a88;
	/*0x1a8c*/ u32 unk1a8c;
	/*0x1a90*/ struct coord bondenterpos;
	/*0x1a9c*/ f32 bondentermtx[16];
	/*0x1adc*/ f32 unk1adc;
	/*0x1ae0*/ u32 unk1ae0;
	/*0x1ae4*/ u32 unk1ae4;
	/*0x1ae8*/ f32 unk1ae8;
	/*0x1aec*/ u32 unk1aec;
	/*0x1af0*/ struct prop *unk1af0;
	/*0x1af4*/ u32 unk1af4;
	/*0x1af8*/ u32 walkinitmove;
	/*0x1afc*/ struct coord walkinitpos;
	/*0x1b08*/ f32 walkinitmtx[16];
	/*0x1b48*/ f32 walkinitt;
	/*0x1b4c*/ f32 walkinitt2;
	/*0x1b50*/ struct coord walkinitstart;
	/*0x1b5c*/ struct prop *grabbedprop;
	/*0x1b60*/ f32 unk1b60;
	/*0x1b64*/ s32 grabstarttime;
	/*0x1b68*/ u32 unk1b68;
	/*0x1b6c*/ struct coord bondforcespeed;
	/*0x1b78*/ bool bondtankexplode;
	/*0x1b7c*/ s32 bondviewlevtime60;
	/*0x1b80*/ f32 bondwatchtime60;
	/*0x1b84*/ bool tickdiefinished;
	/*0x1b88*/ s32 introanimnum;
	/*0x1b8c*/ u32 unk1b8c;
	/*0x1b90*/ u16 unk1b90[4];
	/*0x1b98*/ u32 unk1b98;
	/*0x1b9c*/ f32 speedthetacontrol;
	/*0x1ba0*/ s32 cam_room;
	/*0x1ba4*/ s16 autocontrol_aimpad;
	/*0x1ba6*/ s16 autocontrol_lookup;
	/*0x1ba8*/ s16 autocontrol_dist;
	/*0x1baa*/ s16 autocontrol_walkspeed;
	/*0x1bac*/ s32 autocontrol_turnspeed;
	/*0x1bb0*/ struct coord cam_pos;
	/*0x1bbc*/ struct coord cam_look;
	/*0x1bc8*/ struct coord cam_up;
	/*0x1bd4*/ u32 unk1bd4;
	/*0x1bd8*/ u32 unk1bd8;
	/*0x1bdc*/ u32 unk1bdc;
	/*0x1be0*/ u32 unk1be0;
	/*0x1be4*/ u16 unk1be4;
	/*0x1be6*/ u8 numaibuddies;
	/*0x1be7*/ u8 aibuddynums[MAX_SIMULANTS];
	/*0x1bf0*/ u32 bondexploding;
	/*0x1bf4*/ u32 bondnextexplode;
	/*0x1bf8*/ u32 bondcurexplode;
	/*0x1bfc*/ u8 teleportstate;
	/*0x1bfd*/ u8 teleporttime;
	/*0x1bfe*/ s16 teleportpad;
	/*0x1c00*/ u16 teleportcamerapad;
	/*0x1c04*/ u32 unk1c04;
	/*0x1c08*/ u32 training;
	/*0x1c0c*/ s32 deadtimer;
	/*0x1c10*/ bool coopcanrestart;
	/*0x1c14*/ s32 foot;
	/*0x1c18*/ f32 footstepdist;
	/*0x1c1c*/ u32 unk1c1c;
	/*0x1c20*/ u32 unk1c20;
	/*0x1c24*/ u32 unk1c24;
	/*0x1c28*/ u32 unk1c28;
	/*0x1c2c*/ u32 unk1c2c;
	/*0x1c30*/ u32 unk1c30;
	/*0x1c34*/ u32 unk1c34;
	/*0x1c38*/ u32 unk1c38;
	/*0x1c3c*/ u32 unk1c3c;
	/*0x1c40*/ u32 joybutinhibit;
	/*0x1c44*/ u32 unk1c44;
	/*0x1c48*/ u32 unk1c48;
	/*0x1c4c*/ u32 unk1c4c;
	/*0x1c50*/ u32 unk1c50;
	/*0x1c54*/ u32 unk1c54;
	/*0x1c58*/ f32 unk1c58;
	/*0x1c5c*/ f32 stealhealth;
};

struct stagesetup00 {
	/*0x000*/ u32 unk000;
	/*0x004*/ u32 unk004;
	/*0x008*/ u32 unk008;
	/*0x00c*/ u32 unk00c;
	/*0x010*/ u32 unk010;
	/*0x014*/ u32 unk014;
	/*0x018*/ u32 unk018;
	/*0x01c*/ u32 unk01c;
	/*0x020*/ u32 unk020;
	/*0x024*/ u32 unk024;
	/*0x028*/ u32 unk028;
	/*0x02c*/ u32 unk02c;
	/*0x030*/ u32 unk030;
	/*0x034*/ u32 unk034;
	/*0x038*/ u32 unk038;
	/*0x03c*/ u32 unk03c;
	/*0x040*/ u32 unk040;
	/*0x044*/ u32 unk044;
	/*0x048*/ u32 unk048;
	/*0x04c*/ u32 unk04c;
	/*0x050*/ u32 unk050;
	/*0x054*/ u32 unk054;
	/*0x058*/ u32 unk058;
	/*0x05c*/ u32 unk05c;
	/*0x060*/ u32 unk060;
	/*0x064*/ u32 unk064;
	/*0x068*/ u32 unk068;
	/*0x06c*/ u32 unk06c;
	/*0x070*/ u32 unk070;
	/*0x074*/ u32 unk074;
	/*0x078*/ u32 unk078;
	/*0x07c*/ u32 unk07c;
	/*0x080*/ u32 unk080;
	/*0x084*/ u32 unk084;
	/*0x088*/ u32 unk088;
	/*0x08c*/ u32 unk08c;
	/*0x090*/ u32 unk090;
	/*0x094*/ u32 unk094;
	/*0x098*/ u32 unk098;
	/*0x09c*/ u32 unk09c;
	/*0x0a0*/ u32 unk0a0;
	/*0x0a4*/ u32 unk0a4;
	/*0x0a8*/ u32 unk0a8;
	/*0x0ac*/ u32 unk0ac;
	/*0x0b0*/ u32 unk0b0;
	/*0x0b4*/ u32 unk0b4;
	/*0x0b8*/ u32 unk0b8;
	/*0x0bc*/ u32 unk0bc;
	/*0x0c0*/ u32 unk0c0;
	/*0x0c4*/ u32 unk0c4;
	/*0x0c8*/ u32 unk0c8;
	/*0x0cc*/ u32 unk0cc;
	/*0x0d0*/ u32 unk0d0;
	/*0x0d4*/ u32 unk0d4;
	/*0x0d8*/ u32 unk0d8;
	/*0x0dc*/ u32 unk0dc;
	/*0x0e0*/ u32 unk0e0;
	/*0x0e4*/ u32 unk0e4;
	/*0x0e8*/ u32 unk0e8;
	/*0x0ec*/ u32 unk0ec;
	/*0x0f0*/ u32 unk0f0;
	/*0x0f4*/ u32 unk0f4;
	/*0x0f8*/ u32 unk0f8;
	/*0x0fc*/ u32 unk0fc;
	/*0x100*/ u32 unk100;
	/*0x104*/ u32 unk104;
	/*0x108*/ u32 unk108;
	/*0x10c*/ u32 unk10c;
	/*0x110*/ u32 unk110;
	/*0x114*/ u32 unk114;
	/*0x118*/ u32 unk118;
	/*0x11c*/ u32 unk11c;
	/*0x120*/ u32 unk120;
	/*0x124*/ u32 unk124;
	/*0x128*/ u32 unk128;
	/*0x12c*/ u32 unk12c;
	/*0x130*/ u32 unk130;
	/*0x134*/ u32 unk134;
	/*0x138*/ u32 unk138;
	/*0x13c*/ u32 unk13c;
	/*0x140*/ u32 unk140;
	/*0x144*/ u32 unk144;
	/*0x148*/ u32 unk148;
	/*0x14c*/ u32 unk14c;
	/*0x150*/ u32 unk150;
	/*0x154*/ u32 unk154;
	/*0x158*/ u32 unk158;
	/*0x15c*/ u32 unk15c;
	/*0x160*/ u32 unk160;
	/*0x164*/ u32 unk164;
	/*0x168*/ u32 unk168;
	/*0x16c*/ u32 unk16c;
	/*0x170*/ u32 unk170;
	/*0x174*/ u32 unk174;
	/*0x178*/ u32 unk178;
	/*0x17c*/ u32 unk17c;
	/*0x180*/ u32 unk180;
	/*0x184*/ u32 unk184;
	/*0x188*/ u32 unk188;
	/*0x18c*/ u32 unk18c;
	/*0x190*/ u32 unk190;
	/*0x194*/ u32 unk194;
	/*0x198*/ u32 unk198;
	/*0x19c*/ u32 unk19c;
	/*0x1a0*/ u32 unk1a0;
	/*0x1a4*/ u32 unk1a4;
	/*0x1a8*/ u32 unk1a8;
	/*0x1ac*/ u32 unk1ac;
	/*0x1b0*/ u32 unk1b0;
	/*0x1b4*/ u32 unk1b4;
	/*0x1b8*/ u32 unk1b8;
	/*0x1bc*/ u32 unk1bc;
	/*0x1c0*/ u32 unk1c0;
	/*0x1c4*/ u32 unk1c4;
	/*0x1c8*/ u32 unk1c8;
	/*0x1cc*/ u32 unk1cc;
	/*0x1d0*/ u32 unk1d0;
	/*0x1d4*/ u32 unk1d4;
	/*0x1d8*/ u32 unk1d8;
	/*0x1dc*/ u32 unk1dc;
	/*0x1e0*/ u32 unk1e0;
	/*0x1e4*/ u32 unk1e4;
	/*0x1e8*/ u32 unk1e8;
	/*0x1ec*/ u32 unk1ec;
	/*0x1f0*/ u32 unk1f0;
	/*0x1f4*/ u32 unk1f4;
	/*0x1f8*/ u32 unk1f8;
	/*0x1fc*/ u32 unk1fc;
	/*0x200*/ u32 unk200;
	/*0x204*/ u32 unk204;
	/*0x208*/ u32 unk208;
	/*0x20c*/ u32 unk20c;
	/*0x210*/ u32 unk210;
	/*0x214*/ u32 unk214;
	/*0x218*/ u32 unk218;
	/*0x21c*/ u32 unk21c;
	/*0x220*/ u32 unk220;
	/*0x224*/ u32 unk224;
	/*0x228*/ u32 unk228;
	/*0x22c*/ u32 unk22c;
	/*0x230*/ u32 unk230;
	/*0x234*/ u32 unk234;
	/*0x238*/ u32 unk238;
	/*0x23c*/ u32 unk23c;
	/*0x240*/ u32 unk240;
	/*0x244*/ u32 unk244;
	/*0x248*/ u32 unk248;
	/*0x24c*/ u32 unk24c;
	/*0x250*/ u32 unk250;
	/*0x254*/ u32 unk254;
	/*0x258*/ u32 unk258;
	/*0x25c*/ u32 unk25c;
	/*0x260*/ u32 unk260;
	/*0x264*/ u32 unk264;
	/*0x268*/ u32 unk268;
	/*0x26c*/ u32 unk26c;
	/*0x270*/ u32 unk270;
	/*0x274*/ u32 unk274;
	/*0x278*/ u32 unk278;
	/*0x27c*/ u32 unk27c;
	/*0x280*/ u32 unk280;
	/*0x284*/ u32 unk284;
	/*0x288*/ u32 unk288;
	/*0x28c*/ u32 unk28c;
	/*0x290*/ u32 unk290;
	/*0x294*/ u32 unk294;
	/*0x298*/ u32 unk298;
	/*0x29c*/ u32 unk29c;
	/*0x2a0*/ u32 unk2a0;
	/*0x2a4*/ u32 unk2a4;
	/*0x2a8*/ u32 unk2a8;
	/*0x2ac*/ u32 unk2ac;
	/*0x2b0*/ u32 unk2b0;
	/*0x2b4*/ u32 unk2b4;
	/*0x2b8*/ u32 unk2b8;
	/*0x2bc*/ u32 unk2bc;
	/*0x2c0*/ u32 unk2c0;
	/*0x2c4*/ u32 unk2c4;
	/*0x2c8*/ u32 unk2c8;
	/*0x2cc*/ u32 unk2cc;
	/*0x2d0*/ u32 unk2d0;
	/*0x2d4*/ u32 unk2d4;
	/*0x2d8*/ u32 unk2d8;
	/*0x2dc*/ u32 unk2dc;
	/*0x2e0*/ u32 unk2e0;
	/*0x2e4*/ u32 unk2e4;
	/*0x2e8*/ u32 unk2e8;
	/*0x2ec*/ u32 unk2ec;
	/*0x2f0*/ u32 unk2f0;
	/*0x2f4*/ u32 unk2f4;
	/*0x2f8*/ u32 unk2f8;
	/*0x2fc*/ u32 unk2fc;
	/*0x300*/ u32 unk300;
	/*0x304*/ u32 unk304;
	/*0x308*/ u32 unk308;
	/*0x30c*/ u32 unk30c;
	/*0x310*/ u32 unk310;
};

struct ailist {
	u8 *list;
	s32 id;
};

struct path {
	s32 *pads;
	u8 id;
	u8 type;
};

struct coverdefinition {
	struct coord pos;
	struct coord look;
	u16 flags;
};

struct cover {
	/*0x00*/ struct coord *pos;
	/*0x04*/ struct coord *look;
	/*0x08*/ s16 room;
	/*0x0a*/ s16 unk0a;
	/*0x0c*/ u16 flags;
};

struct stagesetup {
	/*0x00*/ struct stagesetup00 *unk00;
	/*0x04*/ void *unk04;
	/*0x08*/ struct coverdefinition *cover;
	/*0x0c*/ s32 *intro;
	/*0x10*/ u32 *props;
	/*0x14*/ struct path *paths;
	/*0x18*/ struct ailist *ailists;
	/*0x1c*/ s8 *padfiledata;
};

struct inventory_menupos {
	f32 x;
	f32 y;
	f32 z;
	f32 rotation;
	f32 size;
};

struct inventory_typee {
	u32 unk00;
	u32 unk04;
	u32 unk08;
	u32 unk0c;
	u32 unk10;
	u32 unk14;
	u32 unk18;
	u32 unk1c;
};

struct inventory_typef {
	u32 unk00;
	u32 unk04;
	u32 unk08;
	u32 unk0c;
	u32 unk10;
};

struct weaponfunc {
	/*0x00*/ u32 type;
	/*0x04*/ u16 name;
	/*0x06*/ u8 unk06;
	/*0x07*/ s8 unk07;
	/*0x08*/ struct inventory_menupos *menupos;
	/*0x0c*/ u32 *fire_animation;
	/*0x10*/ u32 flags;
};

struct weaponfunc_shootsingle {
	struct weaponfunc base;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ u32 unk20;
	/*0x24*/ u32 unk24;
	/*0x28*/ u32 unk28;
	/*0x2c*/ u32 unk2c;
	/*0x30*/ u32 unk30;
	/*0x34*/ u32 unk34;
	/*0x38*/ u32 unk38;
	/*0x3c*/ u32 unk3c;
};

struct weaponfunc_shootauto {
	struct weaponfunc base;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ u32 unk20;
	/*0x24*/ u32 unk24;
	/*0x28*/ u32 unk28;
	/*0x2c*/ u32 unk2c;
	/*0x30*/ u32 unk30;
	/*0x34*/ u32 unk34;
	/*0x38*/ u32 unk38;
	/*0x3c*/ u32 unk3c;
	/*0x40*/ f32 initialfirerate; // rounds per minute
	/*0x44*/ f32 maxfirerate; // rounds per minute
	/*0x48*/ u32 unk48;
	/*0x4c*/ u32 unk4c;
	/*0x50*/ u32 unk50;
};

struct weaponfunc_shootprojectile {
	struct weaponfunc base;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ u32 unk20;
	/*0x24*/ u32 unk24;
	/*0x28*/ u32 unk28;
	/*0x2c*/ u32 unk2c;
	/*0x30*/ u32 unk30;
	/*0x34*/ u32 unk34;
	/*0x38*/ u32 unk38;
	/*0x3c*/ u32 unk3c;
	/*0x40*/ s32 projectilemodelnum;
	/*0x44*/ u32 unk44;
	/*0x48*/ u32 unk48;
	/*0x4c*/ u32 unk4c;
	/*0x50*/ u32 unk50;
	/*0x54*/ u32 unk54;
	/*0x58*/ u32 unk58;
	/*0x5c*/ u32 unk5c;
	/*0x60*/ u32 unk60;
};

struct weaponfunc_throw {
	struct weaponfunc base;
	/*0x14*/ s32 projectilemodelnum;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ u32 unk20;
};

struct weaponfunc_close {
	struct weaponfunc base;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ u32 unk20;
	/*0x24*/ u32 unk24;
	/*0x28*/ u32 unk28;
	/*0x2c*/ u32 unk2c;
	/*0x30*/ u32 unk30;
	/*0x34*/ u32 unk34;
	/*0x38*/ u32 unk38;
	/*0x3c*/ u32 unk3c;
	/*0x40*/ u32 unk40;
	/*0x44*/ u32 unk44;
	/*0x48*/ u32 unk48;
};

struct weaponfunc_special {
	struct weaponfunc base;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
};

struct weaponfunc_visual {
	struct weaponfunc base;
	/*0x14*/ u32 unk14;
};

struct inventory_ammo {
	u32 type;
	u32 casingeject;
	u16 clipsize;
	u32 *reload_animation;
	u32 style;
};

struct weapon {
	/*0x00*/ u16 lo_model;
	/*0x02*/ u16 hi_model;
	/*0x04*/ u32 *equip_animation;
	/*0x08*/ u32 *unequip_animation;
	/*0x0c*/ u32 *pritosec_animation;
	/*0x10*/ u32 *sectopri_animation;
	/*0x14*/ void *functions[2];
	/*0x1c*/ struct inventory_ammo *ammos[2];
	/*0x24*/ struct inventory_typee *eptr;
	/*0x28*/ f32 sway;
	/*0x2c*/ f32 leftright;
	/*0x30*/ f32 updown;
	/*0x34*/ f32 frontback;
	/*0x38*/ f32 unk38;
	/*0x3c*/ struct inventory_typef *fptr;
	/*0x40*/ s8 *gptr;
	/*0x44*/ u16 shortname;
	/*0x46*/ u16 name;
	/*0x48*/ u16 manufacturer;
	/*0x4a*/ u16 description;
	/*0x4c*/ u32 flags;
};

struct cutscene {
	u16 stage_id;
	u16 mission;
	u32 scene;
	u16 name;
};

struct cheat {
	u16 nametextid;
	u16 time;
	u8 stage_index;
	u8 difficulty;
	u8 flags;
};

struct body {
	u16 ismale : 1;
	u16 flags_01 : 15;
	u16 bodyfileid;
	f32 unk04;
	f32 unk08;
	void *unk0c; // likely a pointer to bodyfile data if loaded
	u16 handfileid;
};

struct stagetableentry {
	/*0x00*/ s16 id;
	/*0x02*/ u8 light_type;
	/*0x03*/ u8 light_alpha;
	/*0x04*/ u8 light_width;
	/*0x05*/ u8 light_height;
	/*0x06*/ u16 unk06;
	/*0x08*/ u16 bgfileid;
	/*0x0a*/ u16 tilefileid;
	/*0x0c*/ u16 padsfileid;
	/*0x0e*/ u16 setupfileid;
	/*0x10*/ u16 mpsetupfileid;
	/*0x14*/ f32 unk14;
	/*0x18*/ f32 unk18;
	/*0x1c*/ f32 unk1c;
	/*0x20*/ u16 unk20;
	/*0x22*/ u8 unk22;
	/*0x23*/ s8 unk23;
	/*0x24*/ u32 unk24;
	/*0x28*/ u32 unk28;
	/*0x2c*/ s16 unk2c;
	/*0x2e*/ u16 unk2e;
	/*0x30*/ u16 unk30;
	/*0x34*/ f32 unk34;
};

struct mpweaponset {
	/*0x00*/ u16 name;
	/*0x02*/ u8 slots[6];
	/*0x08*/ u8 unlocks[4];
	/*0x0c*/ u8 unk0c;
	/*0x0d*/ u8 unk0d;
	/*0x0e*/ u8 unk0e;
	/*0x0f*/ u8 unk0f;
	/*0x10*/ u8 unk10;
	/*0x11*/ u8 unk11;
};

struct mphead {
	s16 headid;
	u8 unlockvalue;
};

struct mpsimulant {
	u8 type;
	u8 skill;
	u16 name;
	u16 body;
	u8 unlockvalue;
};

struct mpbody {
	s16 bodyid;
	s16 name;
	s16 headid;
	u8 unk06;
};

struct mptrack {
	u16 audioid : 7;
	u16 duration : 9;
	s16 name;
	s16 unlockstage;
};

struct stageoverviewentry {
	u32 stage_id;
	u8 unk04;
	u16 name1; // "dataDyne Central"
	u16 name2; // " - Defection"
	u16 name3; // "dataDyne Defection"
};

struct stagemusic {
	u16 stage_id;
	s16 main_music;
	s16 sfx_music;
	s16 x_music;
};

struct mission {
	u32 first_stage_index;
	u16 name;
};

struct sun {
	u8 lens_flare;
	u8 red;
	u8 green;
	u8 blue;
	f32 x;
	f32 y;
	f32 z;
	u16 texture_size;
	u16 orb_size;
};

struct smallsky {
	// len 0x2c
	u16 stage;
	u16 blend_multiplier;
	u16 sky_far;
	u16 unk06;
	u16 unk08;
	u16 unk0a;
	u16 unk0c;
	u16 unk0e;
	u8 sky_r;
	u8 sky_g;
	u8 sky_b;
	u8 sky_enabled;
	struct sun *sun;
	u8 clouds_enabled;
	u8 unk19; // pad
	u16 unk1a;
	u8 unk1c;
	u8 clouds_r;
	u8 clouds_g;
	u8 clouds_b;
	u16 unk20;
	u8 unk22;
	u8 unk23;
	u8 water_enabled;
	u8 water_r;
	u8 water_g;
	u8 water_b;
	u8 unk28;
};

struct sky {
	s32 stage;
	u16 blend_multiplier;
	u16 sky_far;
	u32 unk08;
	u16 unk0c;
	u8 sky_r;
	u8 sky_g;
	u8 sky_b;
	u8 num_suns;
	struct sun *sunptr;
	u8 clouds_enabled;
	u8 clouds_r;
	u8 clouds_g;
	u8 clouds_b;
	f32 clouds_scale;
	u16 unk20;
	u8 water_enabled;
	u8 water_r;
	u8 water_g;
	u8 water_b;
	f32 water_scale;
	u16 water_type;
	u32 unk30;
	u32 unk34;
};

struct hoverprop {
	u32 fileid;
	s32 y_offset;
	u16 size;
};

struct menu_item {
	u8 type;
	u8 param;
	u32 param1;
	u32 param2;
	u32 param3;
	void *handler;
};

struct menu_dialog {
	u8 type;
	u32 title;
	struct menu_item *items;
	void *unk0c;
	u32 unk10;
	struct menu_dialog *nextsibling;
};

struct stagethinglist {
	u16 stage;
	u16 numthings;
	u16 *things;
};

struct twowords {
	u32 unk00;
	u32 unk04;
};

struct something {
	u32 *unk00;
	u32 *unk04;
	u32 unk08;
};

struct audiodefinition {
	u16 unk00;
	u16 audioconfig_index;
};

struct audioconfig {
	f32 unk00;
	f32 unk04;
	f32 unk08;
	f32 unk0c;
	u32 unk10;
	s32 unk14;
	u32 unk18;
	u32 flags; // 0x10 = contains swearing
};

struct bootbufferthing {
	u8 unk00[2400];
};

struct playerstats {
	/*0x00*/ u32 shotcount[7];
	/*0x1c*/ u32 killcount;
	/*0x20*/ u32 ggkillcount;
	/*0x24*/ u32 kills[4];
	/*0x34*/ u32 drawplayercount;
	/*0x38*/ f32 distance;
	/*0x3c*/ u32 backshotcount;
	/*0x40*/ f32 armourcount;
	/*0x44*/ s32 fastest2kills;
	/*0x48*/ s32 slowest2kills;
	/*0x4c*/ s32 longestlife;
	/*0x50*/ s32 shortestlife;
	/*0x54*/ u32 maxkills;
	/*0x58*/ u32 maxsimulkills;
	/*0x5c*/ f32 damagescale;
	/*0x60*/ s32 tokenheldtime;
	/*0x64*/ u32 unk64;
	/*0x68*/ u32 cloaktime;
	/*0x6c*/ u32 speedpillcount;
	/*0x70*/ s32 mpindex;
	/*0x74*/ f32 scale_bg2gfx;
	/*0x78*/ f32 damreceived;
	/*0x7c*/ f32 damtransmitted;
};

struct credit {
	u8 unk00;
	u8 type;
	u16 text1;
	u16 text2;
};

struct gvars800a3fc8 {
	u32 unk00;
	u32 unk04;
	u32 unk08;
	u32 unk0c;
	u32 unk10;
	u32 unk14;
	u32 unk18;
	u32 unk1c;
	u32 unk20;
};

struct roombitfield {
	u8 a : 4;
	u8 b : 4;
};

struct room {
	/*0x00*/ u16 flags;
	/*0x02*/ s16 unk02;
	/*0x04*/ u8 unk04;
	/*0x05*/ u8 unk05;
	/*0x06*/ u8 unk06;
	/*0x07*/ u8 unk07;
	/*0x08*/ u32 unk08;
	/*0x0c*/ u32 unk0c;
	/*0x10*/ u32 unk10;
	/*0x14*/ u32 unk14;
	/*0x18*/ struct coord unk18;
	/*0x24*/ struct coord unk24;
	/*0x30*/ u32 unk30;
	/*0x34*/ u32 unk34;
	/*0x38*/ u32 unk38;
	/*0x3c*/ u32 unk3c;
	/*0x40*/ u32 unk40;
	/*0x44*/ u32 unk44;
	/*0x48*/ u32 unk48;
	/*0x4c*/ u8 unk4c;
	/*0x4d*/ u8 unk4d;
	/*0x4e*/ struct roombitfield bitfield;
	/*0x50*/ u32 unk50;
	/*0x54*/ u32 unk54;
	/*0x58*/ u32 unk58;
	/*0x5c*/ u32 unk5c;
	/*0x60*/ u32 unk60;
	/*0x64*/ u32 unk64;
	/*0x68*/ u32 unk68;
	/*0x6c*/ u32 unk6c;
	/*0x70*/ u32 unk70;
	/*0x74*/ u32 unk74;
	/*0x78*/ u32 unk78;
	/*0x7c*/ u32 unk7c;
	/*0x80*/ u32 unk80;
	/*0x84*/ u32 unk84;
	/*0x88*/ u32 unk88;
};

struct fireslot {
	/*0x00*/ s32 unk00;
	/*0x04*/ u32 unk04;
	/*0x08*/ u32 unk08;
	/*0x0c*/ u32 unk0c;
	/*0x10*/ u32 unk10;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ u32 unk20;
	/*0x24*/ u32 unk24;
	/*0x28*/ u32 unk28;
	/*0x2c*/ u32 unk2c;
};

struct menulayer {
	struct menuframe *siblings[5];
	s8 numsiblings;
	s8 cursibling;
};

struct menuframe {
	struct menu_dialog *dialog;
	u32 unk04;
	struct menu_item *item;
	/*0x0c*/ u32 unk0c;
	/*0x10*/ u32 unk10;
	/*0x14*/ s32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ s32 unk1c;
	/*0x20*/ s32 unk20;
	/*0x24*/ u32 unk24;
	/*0x28*/ u32 unk28;
	/*0x2c*/ s32 unk2c;
	/*0x30*/ u32 unk30;
	/*0x34*/ u32 unk34;
	/*0x38*/ u32 unk38;
	/*0x3c*/ u8 unk3c;
	/*0x40*/ u32 unk40;
	/*0x44*/ u32 unk44;
	/*0x48*/ u32 unk48;
	/*0x4c*/ u32 unk4c;
	/*0x50*/ u32 unk50;
	/*0x54*/ u32 unk54;
	/*0x58*/ u32 unk58;
	/*0x5c*/ u32 unk5c;
	/*0x60*/ u32 unk60;
	/*0x64*/ u32 unk64;
	/*0x68*/ u32 unk68;
	/*0x6c*/ u8 unk6c;
	/*0x6d*/ s8 unk6d;
};

struct menustackdfc {
	u32 unk00;
	f32 unk04;
};

struct menustackitem {
	struct menuframe frames[10];
	/*0x460*/ s16 numframes;
	/*0x464*/ struct menulayer layers[6];
	/*0x4f4*/ s16 depth; // index into layers. 1-indexed?
	/*0x4f8*/ struct menuframe *curframe;
	/*0x4fc*/ u32 unk4fc;
	/*0x500*/ u32 unk500;
	/*0x504*/ u32 unk504;
	/*0x508*/ u32 unk508;
	/*0x50c*/ u32 unk50c;
	/*0x510*/ u32 unk510;
	/*0x514*/ u32 unk514;
	/*0x518*/ u32 unk518;
	/*0x51c*/ u32 unk51c;
	/*0x520*/ u32 unk520;
	/*0x524*/ u32 unk524;
	/*0x528*/ u32 unk528;
	/*0x52c*/ u32 unk52c;
	/*0x530*/ u32 unk530;
	/*0x534*/ u32 unk534;
	/*0x538*/ u32 unk538;
	/*0x53c*/ u32 unk53c;
	/*0x540*/ u32 unk540;
	/*0x544*/ u32 unk544;
	/*0x548*/ u32 unk548;
	/*0x54c*/ u32 unk54c;
	/*0x550*/ u32 unk550;
	/*0x554*/ u32 unk554;
	/*0x558*/ u32 unk558;
	/*0x55c*/ u32 unk55c;
	/*0x560*/ u32 unk560;
	/*0x564*/ u32 unk564;
	/*0x568*/ u32 unk568;
	/*0x56c*/ u32 unk56c;
	/*0x570*/ u32 unk570;
	/*0x574*/ u32 unk574;
	/*0x578*/ u32 unk578;
	/*0x57c*/ u32 unk57c;
	/*0x580*/ u32 unk580;
	/*0x584*/ u32 unk584;
	/*0x588*/ u32 unk588;
	/*0x58c*/ u32 unk58c;
	/*0x590*/ u32 unk590;
	/*0x594*/ u32 unk594;
	/*0x598*/ u32 unk598;
	/*0x59c*/ u32 unk59c;
	/*0x5a0*/ u32 unk5a0;
	/*0x5a4*/ u32 unk5a4;
	/*0x5a8*/ u32 unk5a8;
	/*0x5ac*/ u32 unk5ac;
	/*0x5b0*/ u32 unk5b0;
	/*0x5b4*/ u32 unk5b4;
	/*0x5b8*/ u32 unk5b8;
	/*0x5bc*/ u32 unk5bc;
	/*0x5c0*/ u32 unk5c0;
	/*0x5c4*/ u32 unk5c4;
	/*0x5c8*/ u32 unk5c8;
	/*0x5cc*/ u32 unk5cc;
	/*0x5d0*/ u32 unk5d0;
	/*0x5d4*/ u32 unk5d4;
	/*0x5d8*/ u32 unk5d8;
	/*0x5dc*/ u32 unk5dc;
	/*0x5e0*/ u32 unk5e0;
	/*0x5e4*/ u32 unk5e4;
	/*0x5e8*/ u32 unk5e8;
	/*0x5ec*/ u32 unk5ec;
	/*0x5f0*/ u32 unk5f0;
	/*0x5f4*/ u32 unk5f4;
	/*0x5f8*/ u32 unk5f8;
	/*0x5fc*/ u32 unk5fc;
	/*0x600*/ u32 unk600;
	/*0x604*/ u32 unk604;
	/*0x608*/ u32 unk608;
	/*0x60c*/ u32 unk60c;
	/*0x610*/ u32 unk610;
	/*0x614*/ u32 unk614;
	/*0x618*/ u32 unk618;
	/*0x61c*/ u32 unk61c;
	/*0x620*/ u32 unk620;
	/*0x624*/ u32 unk624;
	/*0x628*/ u32 unk628;
	/*0x62c*/ u32 unk62c;
	/*0x630*/ u32 unk630;
	/*0x634*/ u32 unk634;
	/*0x638*/ u32 unk638;
	/*0x63c*/ u32 unk63c;
	/*0x640*/ u32 unk640;
	/*0x644*/ u32 unk644;
	/*0x648*/ u32 unk648;
	/*0x64c*/ u32 unk64c;
	/*0x650*/ u32 unk650;
	/*0x654*/ u32 unk654;
	/*0x658*/ u32 unk658;
	/*0x65c*/ u32 unk65c;
	/*0x660*/ u32 unk660;
	/*0x664*/ u32 unk664;
	/*0x668*/ u32 unk668;
	/*0x66c*/ u32 unk66c;
	/*0x670*/ u32 unk670;
	/*0x674*/ u32 unk674;
	/*0x678*/ u32 unk678;
	/*0x67c*/ u32 unk67c;
	/*0x680*/ u32 unk680;
	/*0x684*/ u32 unk684;
	/*0x688*/ u32 unk688;
	/*0x68c*/ u32 unk68c;
	/*0x690*/ u32 unk690;
	/*0x694*/ u32 unk694;
	/*0x698*/ u32 unk698;
	/*0x69c*/ u32 unk69c;
	/*0x6a0*/ u32 unk6a0;
	/*0x6a4*/ u32 unk6a4;
	/*0x6a8*/ u32 unk6a8;
	/*0x6ac*/ u32 unk6ac;
	/*0x6b0*/ u32 unk6b0;
	/*0x6b4*/ u32 unk6b4;
	/*0x6b8*/ u32 unk6b8;
	/*0x6bc*/ u32 unk6bc;
	/*0x6c0*/ u32 unk6c0;
	/*0x6c4*/ u32 unk6c4;
	/*0x6c8*/ u32 unk6c8;
	/*0x6cc*/ u32 unk6cc;
	/*0x6d0*/ u32 unk6d0;
	/*0x6d4*/ u32 unk6d4;
	/*0x6d8*/ u32 unk6d8;
	/*0x6dc*/ u32 unk6dc;
	/*0x6e0*/ u32 unk6e0;
	/*0x6e4*/ u32 unk6e4;
	/*0x6e8*/ u32 unk6e8;
	/*0x6ec*/ u32 unk6ec;
	/*0x6f0*/ u32 unk6f0;
	/*0x6f4*/ u32 unk6f4;
	/*0x6f8*/ u32 unk6f8;
	/*0x6fc*/ u32 unk6fc;
	/*0x700*/ u32 unk700;
	/*0x704*/ u32 unk704;
	/*0x708*/ u32 unk708;
	/*0x70c*/ u32 unk70c;
	/*0x710*/ u32 unk710;
	/*0x714*/ u32 unk714;
	/*0x718*/ u32 unk718;
	/*0x71c*/ u32 unk71c;
	/*0x720*/ u32 unk720;
	/*0x724*/ u32 unk724;
	/*0x728*/ u32 unk728;
	/*0x72c*/ u32 unk72c;
	/*0x730*/ u32 unk730;
	/*0x734*/ u32 unk734;
	/*0x738*/ u32 unk738;
	/*0x73c*/ u32 unk73c;
	/*0x740*/ u32 unk740;
	/*0x744*/ u32 unk744;
	/*0x748*/ u32 unk748;
	/*0x74c*/ u32 unk74c;
	/*0x750*/ u32 unk750;
	/*0x754*/ u32 unk754;
	/*0x758*/ u32 unk758;
	/*0x75c*/ u32 unk75c;
	/*0x760*/ u32 unk760;
	/*0x764*/ u32 unk764;
	/*0x768*/ u32 unk768;
	/*0x76c*/ u32 unk76c;
	/*0x770*/ u32 unk770;
	/*0x774*/ u32 unk774;
	/*0x778*/ u32 unk778;
	/*0x77c*/ u32 unk77c;
	/*0x780*/ u32 unk780;
	/*0x784*/ u32 unk784;
	/*0x788*/ u32 unk788;
	/*0x78c*/ u32 unk78c;
	/*0x790*/ u32 unk790;
	/*0x794*/ u32 unk794;
	/*0x798*/ u32 unk798;
	/*0x79c*/ u32 unk79c;
	/*0x7a0*/ u32 unk7a0;
	/*0x7a4*/ u32 unk7a4;
	/*0x7a8*/ u32 unk7a8;
	/*0x7ac*/ u32 unk7ac;
	/*0x7b0*/ u32 unk7b0;
	/*0x7b4*/ u32 unk7b4;
	/*0x7b8*/ u32 unk7b8;
	/*0x7bc*/ u32 unk7bc;
	/*0x7c0*/ u32 unk7c0;
	/*0x7c4*/ u32 unk7c4;
	/*0x7c8*/ u32 unk7c8;
	/*0x7cc*/ u32 unk7cc;
	/*0x7d0*/ u32 unk7d0;
	/*0x7d4*/ u32 unk7d4;
	/*0x7d8*/ u32 unk7d8;
	/*0x7dc*/ u32 unk7dc;
	/*0x7e0*/ u32 unk7e0;
	/*0x7e4*/ u32 unk7e4;
	/*0x7e8*/ u32 unk7e8;
	/*0x7ec*/ u32 unk7ec;
	/*0x7f0*/ u32 unk7f0;
	/*0x7f4*/ u32 unk7f4;
	/*0x7f8*/ u32 unk7f8;
	/*0x7fc*/ u32 unk7fc;
	/*0x800*/ u32 unk800;
	/*0x804*/ u32 unk804;
	/*0x808*/ u32 unk808;
	/*0x80c*/ u32 unk80c;
	/*0x810*/ u32 unk810;
	/*0x814*/ u32 unk814;
	/*0x818*/ u32 unk818;
	/*0x81c*/ u32 unk81c;
	/*0x820*/ u32 unk820;
	/*0x824*/ u32 unk824;
	/*0x828*/ u32 unk828;
	/*0x82c*/ u32 unk82c;
	/*0x830*/ u32 unk830;
	/*0x834*/ u32 unk834;
	/*0x838*/ u16 unk838;
	/*0x83a*/ u8 unk83a;
	/*0x83b*/ u8 playernum;
	/*0x83c*/ u8 unk83c;
	/*0x83d*/ u8 unk83d;
	/*0x83e*/ u8 unk83e;
	/*0x83f*/ u8 unk83f;
	/*0x840*/ u8 unk840;
	/*0x844*/ u32 unk844;
	/*0x848*/ u32 unk848;
	/*0x84c*/ u32 unk84c;
	/*0x850*/ u32 unk850;
	/*0x854*/ u32 unk854;
	/*0x858*/ u32 unk858;
	/*0x85c*/ u32 unk85c;
	/*0x860*/ u32 unk860;
	/*0x864*/ u32 unk864;
	/*0x868*/ u32 unk868;
	/*0x86c*/ u32 unk86c;
	/*0x870*/ u32 unk870;
	/*0x874*/ u32 unk874;
	/*0x878*/ u32 unk878;
	/*0x87c*/ u32 unk87c;
	/*0x880*/ u32 unk880;
	/*0x884*/ u32 unk884;
	/*0x888*/ u32 unk888;
	/*0x88c*/ u32 unk88c;
	/*0x890*/ u32 unk890;
	/*0x894*/ u32 unk894;
	/*0x898*/ u32 unk898;
	/*0x89c*/ u32 unk89c;
	/*0x8a0*/ u32 unk8a0;
	/*0x8a4*/ u32 unk8a4;
	/*0x8a8*/ u32 unk8a8;
	/*0x8ac*/ u32 unk8ac;
	/*0x8b0*/ u32 unk8b0;
	/*0x8b4*/ u32 unk8b4;
	/*0x8b8*/ u32 unk8b8;
	/*0x8bc*/ u32 unk8bc;
	/*0x8c0*/ u32 unk8c0;
	/*0x8c4*/ u32 unk8c4;
	/*0x8c8*/ u32 unk8c8;
	/*0x8cc*/ u32 unk8cc;
	/*0x8d0*/ u32 unk8d0;
	/*0x8d4*/ u32 unk8d4;
	/*0x8d8*/ u32 unk8d8;
	/*0x8dc*/ u32 unk8dc;
	/*0x8e0*/ u32 unk8e0;
	/*0x8e4*/ u32 unk8e4;
	/*0x8e8*/ u32 unk8e8;
	/*0x8ec*/ u32 unk8ec;
	/*0x8f0*/ u32 unk8f0;
	/*0x8f4*/ u32 unk8f4;
	/*0x8f8*/ u32 unk8f8;
	/*0x8fc*/ u32 unk8fc;
	/*0x900*/ u32 unk900;
	/*0x904*/ u32 unk904;
	/*0x908*/ u32 unk908;
	/*0x90c*/ u32 unk90c;
	/*0x910*/ u32 unk910;
	/*0x914*/ u32 unk914;
	/*0x918*/ u32 unk918;
	/*0x91c*/ u32 unk91c;
	/*0x920*/ u32 unk920;
	/*0x924*/ u32 unk924;
	/*0x928*/ u32 unk928;
	/*0x92c*/ u32 unk92c;
	/*0x930*/ u32 unk930;
	/*0x934*/ u32 unk934;
	/*0x938*/ u32 unk938;
	/*0x93c*/ u32 unk93c;
	/*0x940*/ u32 unk940;
	/*0x944*/ u32 unk944;
	/*0x948*/ u32 unk948;
	/*0x94c*/ u32 unk94c;
	/*0x950*/ u32 unk950;
	/*0x954*/ u32 unk954;
	/*0x958*/ u32 unk958;
	/*0x95c*/ u32 unk95c;
	/*0x960*/ u32 unk960;
	/*0x964*/ u32 unk964;
	/*0x968*/ u32 unk968;
	/*0x96c*/ u32 unk96c;
	/*0x970*/ u32 unk970;
	/*0x974*/ u32 unk974;
	/*0x978*/ u32 unk978;
	/*0x97c*/ u32 unk97c;
	/*0x980*/ u32 unk980;
	/*0x984*/ u32 unk984;
	/*0x988*/ u32 unk988;
	/*0x98c*/ u32 unk98c;
	/*0x990*/ u32 unk990;
	/*0x994*/ u32 unk994;
	/*0x998*/ u32 unk998;
	/*0x99c*/ u32 unk99c;
	/*0x9a0*/ u32 unk9a0;
	/*0x9a4*/ u32 unk9a4;
	/*0x9a8*/ u32 unk9a8;
	/*0x9ac*/ u32 unk9ac;
	/*0x9b0*/ u32 unk9b0;
	/*0x9b4*/ u32 unk9b4;
	/*0x9b8*/ u32 unk9b8;
	/*0x9bc*/ u32 unk9bc;
	/*0x9c0*/ u32 unk9c0;
	/*0x9c4*/ u32 unk9c4;
	/*0x9c8*/ u32 unk9c8;
	/*0x9cc*/ u32 unk9cc;
	/*0x9d0*/ u32 unk9d0;
	/*0x9d4*/ u32 unk9d4;
	/*0x9d8*/ u32 unk9d8;
	/*0x9dc*/ u32 unk9dc;
	/*0x9e0*/ u32 unk9e0;
	/*0x9e4*/ u32 unk9e4;
	/*0x9e8*/ u32 unk9e8;
	/*0x9ec*/ u32 unk9ec;
	/*0x9f0*/ u32 unk9f0;
	/*0x9f4*/ u32 unk9f4;
	/*0x9f8*/ u32 unk9f8;
	/*0x9fc*/ u32 unk9fc;
	/*0xa00*/ u32 unka00;
	/*0xa04*/ u32 unka04;
	/*0xa08*/ u32 unka08;
	/*0xa0c*/ u32 unka0c;
	/*0xa10*/ u32 unka10;
	/*0xa14*/ u32 unka14;
	/*0xa18*/ u32 unka18;
	/*0xa1c*/ u32 unka1c;
	/*0xa20*/ u32 unka20;
	/*0xa24*/ u32 unka24;
	/*0xa28*/ u32 unka28;
	/*0xa2c*/ u32 unka2c;
	/*0xa30*/ u32 unka30;
	/*0xa34*/ u32 unka34;
	/*0xa38*/ u32 unka38;
	/*0xa3c*/ u32 unka3c;
	/*0xa40*/ u32 unka40;
	/*0xa44*/ u32 unka44;
	/*0xa48*/ u32 unka48;
	/*0xa4c*/ u32 unka4c;
	/*0xa50*/ u32 unka50;
	/*0xa54*/ u32 unka54;
	/*0xa58*/ u32 unka58;
	/*0xa5c*/ u32 unka5c;
	/*0xa60*/ u32 unka60;
	/*0xa64*/ u32 unka64;
	/*0xa68*/ u32 unka68;
	/*0xa6c*/ u32 unka6c;
	/*0xa70*/ u32 unka70;
	/*0xa74*/ u32 unka74;
	/*0xa78*/ u32 unka78;
	/*0xa7c*/ u32 unka7c;
	/*0xa80*/ u32 unka80;
	/*0xa84*/ u32 unka84;
	/*0xa88*/ u32 unka88;
	/*0xa8c*/ u32 unka8c;
	/*0xa90*/ u32 unka90;
	/*0xa94*/ u32 unka94;
	/*0xa98*/ u32 unka98;
	/*0xa9c*/ u32 unka9c;
	/*0xaa0*/ u32 unkaa0;
	/*0xaa4*/ u32 unkaa4;
	/*0xaa8*/ u32 unkaa8;
	/*0xaac*/ u32 unkaac;
	/*0xab0*/ u32 unkab0;
	/*0xab4*/ u32 unkab4;
	/*0xab8*/ u32 unkab8;
	/*0xabc*/ u32 unkabc;
	/*0xac0*/ u32 unkac0;
	/*0xac4*/ u32 unkac4;
	/*0xac8*/ u32 unkac8;
	/*0xacc*/ u32 unkacc;
	/*0xad0*/ u32 unkad0;
	/*0xad4*/ u32 unkad4;
	/*0xad8*/ u32 unkad8;
	/*0xadc*/ u32 unkadc;
	/*0xae0*/ u32 unkae0;
	/*0xae4*/ u32 unkae4;
	/*0xae8*/ u32 unkae8;
	/*0xaec*/ u32 unkaec;
	/*0xaf0*/ u32 unkaf0;
	/*0xaf4*/ u32 unkaf4;
	/*0xaf8*/ u32 unkaf8;
	/*0xafc*/ u32 unkafc;
	/*0xb00*/ u32 unkb00;
	/*0xb04*/ u32 unkb04;
	/*0xb08*/ u32 unkb08;
	/*0xb0c*/ u32 unkb0c;
	/*0xb10*/ u32 unkb10;
	/*0xb14*/ u32 unkb14;
	/*0xb18*/ u32 unkb18;
	/*0xb1c*/ u32 unkb1c;
	/*0xb20*/ u32 unkb20;
	/*0xb24*/ u32 unkb24;
	/*0xb28*/ u32 unkb28;
	/*0xb2c*/ u32 unkb2c;
	/*0xb30*/ u32 unkb30;
	/*0xb34*/ u32 unkb34;
	/*0xb38*/ u32 unkb38;
	/*0xb3c*/ u32 unkb3c;
	/*0xb40*/ u32 unkb40;
	/*0xb44*/ u32 unkb44;
	/*0xb48*/ u32 unkb48;
	/*0xb4c*/ u32 unkb4c;
	/*0xb50*/ u32 unkb50;
	/*0xb54*/ u32 unkb54;
	/*0xb58*/ u32 unkb58;
	/*0xb5c*/ u32 unkb5c;
	/*0xb60*/ u32 unkb60;
	/*0xb64*/ u32 unkb64;
	/*0xb68*/ u32 unkb68;
	/*0xb6c*/ u32 unkb6c;
	/*0xb70*/ u32 unkb70;
	/*0xb74*/ u32 unkb74;
	/*0xb78*/ u32 unkb78;
	/*0xb7c*/ u32 unkb7c;
	/*0xb80*/ u32 unkb80;
	/*0xb84*/ u32 unkb84;
	/*0xb88*/ u32 unkb88;
	/*0xb8c*/ u32 unkb8c;
	/*0xb90*/ u32 unkb90;
	/*0xb94*/ u32 unkb94;
	/*0xb98*/ u32 unkb98;
	/*0xb9c*/ u32 unkb9c;
	/*0xba0*/ u32 unkba0;
	/*0xba4*/ u32 unkba4;
	/*0xba8*/ u32 unkba8;
	/*0xbac*/ u32 unkbac;
	/*0xbb0*/ u32 unkbb0;
	/*0xbb4*/ u32 unkbb4;
	/*0xbb8*/ u32 unkbb8;
	/*0xbbc*/ u32 unkbbc;
	/*0xbc0*/ u32 unkbc0;
	/*0xbc4*/ u32 unkbc4;
	/*0xbc8*/ u32 unkbc8;
	/*0xbcc*/ u32 unkbcc;
	/*0xbd0*/ u32 unkbd0;
	/*0xbd4*/ u32 unkbd4;
	/*0xbd8*/ u32 unkbd8;
	/*0xbdc*/ u32 unkbdc;
	/*0xbe0*/ u32 unkbe0;
	/*0xbe4*/ u32 unkbe4;
	/*0xbe8*/ u32 unkbe8;
	/*0xbec*/ u32 unkbec;
	/*0xbf0*/ u32 unkbf0;
	/*0xbf4*/ u32 unkbf4;
	/*0xbf8*/ u32 unkbf8;
	/*0xbfc*/ u32 unkbfc;
	/*0xc00*/ u32 unkc00;
	/*0xc04*/ u32 unkc04;
	/*0xc08*/ u32 unkc08;
	/*0xc0c*/ u32 unkc0c;
	/*0xc10*/ u32 unkc10;
	/*0xc14*/ u32 unkc14;
	/*0xc18*/ u32 unkc18;
	/*0xc1c*/ u32 unkc1c;
	/*0xc20*/ u32 unkc20;
	/*0xc24*/ u32 unkc24;
	/*0xc28*/ u32 unkc28;
	/*0xc2c*/ u32 unkc2c;
	/*0xc30*/ u32 unkc30;
	/*0xc34*/ u32 unkc34;
	/*0xc38*/ u32 unkc38;
	/*0xc3c*/ u32 unkc3c;
	/*0xc40*/ u32 unkc40;
	/*0xc44*/ u32 unkc44;
	/*0xc48*/ u32 unkc48;
	/*0xc4c*/ u32 unkc4c;
	/*0xc50*/ u32 unkc50;
	/*0xc54*/ u32 unkc54;
	/*0xc58*/ u32 unkc58;
	/*0xc5c*/ u32 unkc5c;
	/*0xc60*/ u32 unkc60;
	/*0xc64*/ u32 unkc64;
	/*0xc68*/ u32 unkc68;
	/*0xc6c*/ u32 unkc6c;
	/*0xc70*/ u32 unkc70;
	/*0xc74*/ u32 unkc74;
	/*0xc78*/ u32 unkc78;
	/*0xc7c*/ u32 unkc7c;
	/*0xc80*/ u32 unkc80;
	/*0xc84*/ u32 unkc84;
	/*0xc88*/ u32 unkc88;
	/*0xc8c*/ u32 unkc8c;
	/*0xc90*/ u32 unkc90;
	/*0xc94*/ u32 unkc94;
	/*0xc98*/ u32 unkc98;
	/*0xc9c*/ u32 unkc9c;
	/*0xca0*/ u32 unkca0;
	/*0xca4*/ u32 unkca4;
	/*0xca8*/ u32 unkca8;
	/*0xcac*/ u32 unkcac;
	/*0xcb0*/ u32 unkcb0;
	/*0xcb4*/ u32 unkcb4;
	/*0xcb8*/ u32 unkcb8;
	/*0xcbc*/ u32 unkcbc;
	/*0xcc0*/ u32 unkcc0;
	/*0xcc4*/ u32 unkcc4;
	/*0xcc8*/ u32 unkcc8;
	/*0xccc*/ u32 unkccc;
	/*0xcd0*/ u32 unkcd0;
	/*0xcd4*/ u32 unkcd4;
	/*0xcd8*/ u32 unkcd8;
	/*0xcdc*/ u32 unkcdc;
	/*0xce0*/ u32 unkce0;
	/*0xce4*/ u32 unkce4;
	/*0xce8*/ u32 unkce8;
	/*0xcec*/ u32 unkcec;
	/*0xcf0*/ u32 unkcf0;
	/*0xcf4*/ u32 unkcf4;
	/*0xcf8*/ u32 unkcf8;
	/*0xcfc*/ u32 unkcfc;
	/*0xd00*/ u32 unkd00;
	/*0xd04*/ u32 unkd04;
	/*0xd08*/ u32 unkd08;
	/*0xd0c*/ u32 unkd0c;
	/*0xd10*/ u32 unkd10;
	/*0xd14*/ u32 unkd14;
	/*0xd18*/ u32 unkd18;
	/*0xd1c*/ u32 unkd1c;
	/*0xd20*/ u32 unkd20;
	/*0xd24*/ u32 unkd24;
	/*0xd28*/ u32 unkd28;
	/*0xd2c*/ u32 unkd2c;
	/*0xd30*/ u32 unkd30;
	/*0xd34*/ u32 unkd34;
	/*0xd38*/ u32 unkd38;
	/*0xd3c*/ u32 unkd3c;
	/*0xd40*/ u32 unkd40;
	/*0xd44*/ u32 unkd44;
	/*0xd48*/ u32 unkd48;
	/*0xd4c*/ u32 unkd4c;
	/*0xd50*/ f32 unkd50;
	/*0xd54*/ f32 unkd54;
	/*0xd58*/ f32 unkd58;
	/*0xd5c*/ f32 unkd5c;
	/*0xd60*/ f32 unkd60;
	/*0xd64*/ f32 unkd64;
	/*0xd68*/ f32 unkd68;
	/*0xd6c*/ f32 unkd6c;
	/*0xd70*/ f32 unkd70;
	/*0xd74*/ f32 unkd74;
	/*0xd78*/ f32 unkd78;
	/*0xd7c*/ f32 unkd7c;
	/*0xd80*/ f32 unkd80;
	/*0xd84*/ u32 unkd84;
	/*0xd88*/ f32 unkd88;
	/*0xd8c*/ f32 unkd8c;
	/*0xd90*/ f32 unkd90;
	/*0xd94*/ f32 unkd94;
	/*0xd98*/ u32 unkd98;
	/*0xd9c*/ u32 unkd9c;
	/*0xda0*/ u32 unkda0;
	/*0xda4*/ u32 unkda4;
	/*0xda8*/ u32 unkda8;
	/*0xdac*/ u32 unkdac;
	/*0xdb0*/ u32 unkdb0;
	/*0xdb4*/ u32 unkdb4;
	/*0xdb8*/ u32 unkdb8;
	/*0xdbc*/ u32 unkdbc;
	/*0xdc0*/ u32 unkdc0;
	/*0xdc4*/ u32 unkdc4;
	/*0xdc8*/ u32 unkdc8;
	/*0xdcc*/ u32 unkdcc;
	/*0xdd0*/ u32 unkdd0;
	/*0xdd4*/ u32 unkdd4;
	/*0xdd8*/ u32 unkdd8;
	/*0xddc*/ u32 unkddc;
	/*0xde0*/ u32 unkde0;
	/*0xde4*/ u32 unkde4;
	/*0xde8*/ u32 unkde8;
	/*0xdec*/ u32 unkdec;
	/*0xdf0*/ u32 unkdf0;
	/*0xdf4*/ s8 *unkdf4;
	/*0xdf8*/ u8 unkdf8;
	/*0xdfc*/ struct menustackdfc unkdfc[4];
	/*0xe1c*/ s32 slotindex;
	/*0xe20*/ u32 slotcount;
	/*0xe24*/ u32 unke24;
	/*0xe28*/ u32 unke28;
	/*0xe2c*/ u32 unke2c;
	/*0xe30*/ u32 unke30;
	/*0xe34*/ u16 unke34; // I/O error code
	/*0xe38*/ u32 unke38;
	/*0xe3c*/ u8 unke3c;
	/*0xe3d*/ u8 unke3d;
	/*0xe3e*/ u8 unke3e;
	/*0xe3f*/ u8 unke3f;
	/*0xe40*/ u8 unke40;
	/*0xe41*/ u8 unke41;
	/*0xe42*/ u8 unke42;
	/*0xe44*/ u32 unke44;
	/*0xe48*/ u32 unke48;
	/*0xe4c*/ u32 unke4c;
	/*0xe50*/ u16 unke50;
	/*0xe52*/ u8 unke52;
	/*0xe53*/ char unke53[1];
	/*0xe54*/ u32 unke54;
	/*0xe58*/ u32 unke58;
	/*0xe5c*/ u32 unke5c;
	/*0xe60*/ u32 unke60;
	/*0xe64*/ u32 unke64;
	/*0xe68*/ u32 unke68;
	/*0xe6c*/ s8 savedevice; // 0-3 = controller pak 1-4, 4 = game pak
};

struct savefile_solo {
	/**
	 * This is a little weird. The name entry is clearly 10 characters plus a
	 * null byte, but there is other code that uses unk08 for other purposes.
	 */
	/*0x00*/ char name[11];
	/*0x0b*/ u8 unk0b_0 : 5;
	/*0x0b*/ u8 unk0b_5 : 3;
	/*0x0c*/ u8 unk0c;
	/*0x10*/ u32 unk10;
	/*0x14*/ u32 flags;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u16 unk1c;
	/*0x1e*/ u16 unk1e;
	/*0x20*/ u16 besttimes[NUM_SOLOSTAGES][3];
	/*0xa0*/ s32 coopcompletions[3];
	/*0xac*/ u8 firingrangescores[9];
	/*0xb5*/ u8 weaponsfound[6];
};

struct propstate {
	/*0x00*/ u16 propcount;
	/*0x02*/ u16 chrpropcount;
	/*0x04*/ u16 foregroundpropcount;
	/*0x06*/ u16 foregroundchrpropcount;
	/*0x08*/ u32 updatetime;
	/*0x0c*/ u32 chrupdatetime;
	/*0x10*/ u32 slotupdate240;
	/*0x14*/ u32 slotupdate60error;
	/*0x18*/ u32 lastupdateframe;
};

struct mpchr {
	/*0x00*/ char name[1]; // len unknown
	/*0x04*/ u32 unk04;
	/*0x08*/ u32 unk08;
	/*0x0c*/ u8 unk0c;
	/*0x0d*/ u8 unk0d;
	/*0x0e*/ u8 unk0e;
	/*0x0f*/ u8 headnum;
	/*0x10*/ u8 bodynum;
	/*0x11*/ u8 team;
	/*0x14*/ u32 displayoptions;
	/*0x18*/ u16 unk18;
	/*0x1a*/ u16 unk1a;
	/*0x1c*/ u16 unk1c;
	/*0x1e*/ s8 placement;
	/*0x20*/ u32 unk20;
	/*0x24*/ s16 unk24[12];
	/*0x3c*/ s16 unk3c;
	/*0x3e*/ s16 unk3e;
	/*0x40*/ u32 unk40;
	/*0x44*/ u8 controlmode;
	/*0x45*/ s8 unk45;
	/*0x46*/ s8 unk46;
	/*0x47*/ u8 simtype;
};

struct mpplayer {
	/*0x00*/ struct mpchr base;
	/*0x48*/ u16 options;
	/*0x4c*/ bool saved;
	/*0x50*/ u16 unk50;
	/*0x54*/ u32 kills;
	/*0x58*/ u32 deaths;
	/*0x5c*/ u32 gamesplayed;
	/*0x60*/ u32 gameswon;
	/*0x64*/ u32 gameslost;
	/*0x68*/ u32 time;
	/*0x6c*/ u32 distance;
	/*0x70*/ f32 accuracy;
	/*0x74*/ u32 damagedealt;
	/*0x78*/ f32 painreceived;
	/*0x7c*/ u32 headshots;
	/*0x80*/ u32 ammoused;
	/*0x84*/ u32 accuracymedals;
	/*0x88*/ u32 headshotmedals;
	/*0x8c*/ u32 killmastermedals;
	/*0x90*/ u32 survivormedals;
	/*0x94*/ u8 unk94;
	/*0x95*/ u8 title;
	/*0x96*/ u8 unk96;
	/*0x97*/ u8 gunfuncs[5];
	/*0x9c*/ u8 unk9c;
	/*0x9d*/ u8 handicap;
};

struct mpsim {
	/*0x00*/ struct mpchr base;
	/*0x48*/ u8 unk48;
};

struct missionconfig {
	u8 difficulty : 7;
	u8 pdmode : 1;

	/*0x01*/ u8 stagenum;
	/*0x02*/ u8 stageindex;

	u8 iscoop : 1;
	u8 isanti : 1;

	/*0x04*/ u8 pdmodereaction;
	/*0x05*/ u8 pdmodehealth;
	/*0x06*/ u8 pdmodedamage;
	/*0x07*/ u8 pdmodeaccuracy;
	/*0x08*/ f32 pdmodereactionf;
	/*0x0c*/ f32 pdmodehealthf;
	/*0x10*/ f32 pdmodedamagef;
	/*0x14*/ f32 pdmodeaccuracyf;
};

struct mpsetup {
	/*0x800acb88*/ char name[12];
	/*0x800acb94*/ u32 options;
	/*0x800acb98*/ u8 scenario;
	/*0x800acb99*/ u8 stagenum;
	/*0x800acb9a*/ u8 timelimit;
	/*0x800acb9b*/ u8 scorelimit;
	/*0x800acb9c*/ u16 teamscorelimit;

	/**
	 * Each bit signifies that a player or sim is participating.
	 *
	 * Bits 0x000f are for players
	 * Bits 0x0ff0 are for sims
	 * Bits 0xf000 are probably not used
	 */
	/*0x800acb9e*/ u16 chrslots;
	/*0x800acba0*/ u32 unk18;
	/*0x800acba4*/ u16 unk1c;
	/*0x800acba6*/ u8 paused;
	/*0x800acba8*/ bool saved;
	/*0x800acbac*/ u16 unk24;
	/*0x800acbae*/ u16 unk26;
};

struct savefile_setup {
	/*0x00*/ char teamnames[8][12];
	/*0x60*/ u8 locktype;
	/*0x61*/ u8 unk89;
	/*0x62*/ u8 usingmultipletunes;
	/*0x63*/ u8 unk8b;
	/*0x64*/ s8 tracknum; // -1 = random
	/*0x8d*/ u8 multipletracknums[5];
};

struct mpscenario {
	struct menu_dialog *optionsdialog;
	void (*initfunc)(void);
	s32 (*unk08)(void);
	void (*resetfunc)(void);
	void (*unk10)(void);
	void (*unk14)(struct chrdata *chr);
	void *unk18;
	void (*killfunc)(struct mpchr *mpchr, s32 arg1, s32 *score, s32 *arg3);
	Gfx *(*radarfunc)(Gfx *gdl);
	bool (*radar2func)(Gfx **gdl, struct prop *prop);
	bool (*highlightfunc)(struct prop *prop, u32 *colour);
	bool (*unk2c)(f32 arg0, struct coord *pos, s16 *arg2, struct prop *prop, f32 *arg4);
	s32 (*maxteamsfunc)(void);
	bool (*isroomhighlightedfunc)(s16 room);
	void (*unk38)(s16 arg0, s32 *arg1, s32 *arg2, s32 *arg3);
	void *unk3c;
	void (*unk40)(s32 *arg0);
	void (*unk44)(s32 *arg0);
};

struct mpscenariooverview {
	u16 name;
	u16 shortname;
	u8 unk04;
	u8 unk05;
};

struct mparena {
	s16 stagenum;
	u8 unlock;
	u16 name;
};

// Throwaway struct for passing two values to menu handlers
struct numandtext {
	s32 num;
	char *text;
};

struct savelocation {
	/*0x000*/ u32 unk000;
	/*0x004*/ u32 unk004;
	/*0x008*/ u32 unk008;
	/*0x00c*/ u32 unk00c;
	/*0x010*/ u32 unk010;
	/*0x014*/ u32 unk014;
	/*0x018*/ u32 unk018;
	/*0x01c*/ u32 unk01c;
	/*0x020*/ u32 unk020;
	/*0x024*/ u32 unk024;
	/*0x028*/ u32 unk028;
	/*0x02c*/ u32 unk02c;
	/*0x030*/ u32 unk030;
	/*0x034*/ u32 unk034;
	/*0x038*/ u32 unk038;
	/*0x03c*/ u32 unk03c;
	/*0x040*/ u32 unk040;
	/*0x044*/ u32 unk044;
	/*0x048*/ u32 unk048;
	/*0x04c*/ u32 unk04c;
	/*0x050*/ u32 unk050;
	/*0x054*/ u32 unk054;
	/*0x058*/ u32 unk058;
	/*0x05c*/ u32 unk05c;
	/*0x060*/ u32 unk060;
	/*0x064*/ u32 unk064;
	/*0x068*/ u32 unk068;
	/*0x06c*/ u32 unk06c;
	/*0x070*/ u32 unk070;
	/*0x074*/ u32 unk074;
	/*0x078*/ u32 unk078;
	/*0x07c*/ u32 unk07c;
	/*0x080*/ u32 unk080;
	/*0x084*/ u32 unk084;
	/*0x088*/ u32 unk088;
	/*0x08c*/ u32 unk08c;
	/*0x090*/ u32 unk090;
	/*0x094*/ u32 unk094;
	/*0x098*/ u32 unk098;
	/*0x09c*/ u32 unk09c;
	/*0x0a0*/ u32 unk0a0;
	/*0x0a4*/ u32 unk0a4;
	/*0x0a8*/ u32 unk0a8;
	/*0x0ac*/ u32 unk0ac;
	/*0x0b0*/ u32 unk0b0;
	/*0x0b4*/ u32 unk0b4;
	/*0x0b8*/ u32 unk0b8;
	/*0x0bc*/ u32 unk0bc;
	/*0x0c0*/ u32 unk0c0;
	/*0x0c4*/ u32 unk0c4;
	/*0x0c8*/ u32 unk0c8;
	/*0x0cc*/ u32 unk0cc;
	/*0x0d0*/ u32 unk0d0;
	/*0x0d4*/ u32 unk0d4;
	/*0x0d8*/ u32 unk0d8;
	/*0x0dc*/ u32 unk0dc;
	/*0x0e0*/ u32 unk0e0;
	/*0x0e4*/ u32 unk0e4;
	/*0x0e8*/ u32 unk0e8;
	/*0x0ec*/ u32 unk0ec;
	/*0x0f0*/ u32 unk0f0;
	/*0x0f4*/ u32 unk0f4;
	/*0x0f8*/ u32 unk0f8;
	/*0x0fc*/ u32 unk0fc;
	/*0x100*/ u32 unk100;
	/*0x104*/ u32 unk104;
	/*0x108*/ u32 unk108;
	/*0x10c*/ u32 unk10c;
	/*0x110*/ u32 unk110;
	/*0x114*/ u32 unk114;
	/*0x118*/ u32 unk118;
	/*0x11c*/ u32 unk11c;
	/*0x120*/ u32 unk120;
	/*0x124*/ u32 unk124;
	/*0x128*/ u32 unk128;
	/*0x12c*/ u32 unk12c;
	/*0x130*/ u32 unk130;
	/*0x134*/ u32 unk134;
	/*0x138*/ u32 unk138;
	/*0x13c*/ u32 unk13c;
	/*0x140*/ u32 unk140;
	/*0x144*/ u32 unk144;
	/*0x148*/ u32 unk148;
	/*0x14c*/ u32 unk14c;
	/*0x150*/ u32 unk150;
	/*0x154*/ u32 unk154;
	/*0x158*/ u32 unk158;
	/*0x15c*/ u32 unk15c;
	/*0x160*/ u32 unk160;
	/*0x164*/ u32 unk164;
	/*0x168*/ u32 unk168;
	/*0x16c*/ u32 unk16c;
	/*0x170*/ u32 unk170;
	/*0x174*/ u32 unk174;
	/*0x178*/ u32 unk178;
	/*0x17c*/ u32 unk17c;
	/*0x180*/ u32 unk180;
	/*0x184*/ u32 unk184;
	/*0x188*/ u32 unk188;
	/*0x18c*/ u32 unk18c;
	/*0x190*/ u32 unk190;
	/*0x194*/ u32 unk194;
	/*0x198*/ u32 unk198;
	/*0x19c*/ u32 unk19c;
	/*0x1a0*/ u32 unk1a0;
	/*0x1a4*/ u32 unk1a4;
	/*0x1a8*/ u32 unk1a8;
	/*0x1ac*/ u32 unk1ac;
	/*0x1b0*/ u32 unk1b0;
	/*0x1b4*/ u32 unk1b4;
	/*0x1b8*/ u32 unk1b8;
	/*0x1bc*/ u32 unk1bc;
	/*0x1c0*/ u32 unk1c0;
	/*0x1c4*/ u32 unk1c4;
	/*0x1c8*/ u32 unk1c8;
	/*0x1cc*/ u32 unk1cc;
	/*0x1d0*/ u32 unk1d0;
	/*0x1d4*/ u32 unk1d4;
	/*0x1d8*/ u32 unk1d8;
	/*0x1dc*/ u32 unk1dc;
	/*0x1e0*/ u32 unk1e0;
	/*0x1e4*/ u32 unk1e4;
	/*0x1e8*/ u32 unk1e8;
	/*0x1ec*/ u32 unk1ec;
	/*0x1f0*/ u32 unk1f0;
	/*0x1f4*/ u32 unk1f4;
	/*0x1f8*/ u32 unk1f8;
	/*0x1fc*/ u32 unk1fc;
	/*0x200*/ u32 unk200;
	/*0x204*/ u32 unk204;
	/*0x208*/ u32 unk208;
	/*0x20c*/ u32 unk20c;
	/*0x210*/ u32 unk210;
	/*0x214*/ u32 unk214;
	/*0x218*/ u32 unk218;
	/*0x21c*/ u32 unk21c;
	/*0x220*/ u32 unk220;
	/*0x224*/ u32 unk224;
	/*0x228*/ u32 unk228;
	/*0x22c*/ u32 unk22c;
	/*0x230*/ u32 unk230;
	/*0x234*/ u32 unk234;
	/*0x238*/ u32 unk238;
	/*0x23c*/ u32 unk23c;
	/*0x240*/ u32 unk240;
	/*0x244*/ u32 unk244;
	/*0x248*/ u32 unk248;
	/*0x24c*/ u32 unk24c;
	/*0x250*/ u32 unk250;
	/*0x254*/ u32 unk254;
	/*0x258*/ u32 unk258;
	/*0x25c*/ u32 unk25c;
	/*0x260*/ u32 unk260;
	/*0x264*/ u32 unk264;
	/*0x268*/ u32 unk268;
	/*0x26c*/ u32 unk26c;
	/*0x270*/ u32 unk270;
	/*0x274*/ u32 unk274;
	/*0x278*/ u32 unk278;
	/*0x27c*/ u32 unk27c;
	/*0x280*/ u32 unk280;
	/*0x284*/ u32 unk284;
	/*0x288*/ u32 unk288;
	/*0x28c*/ u32 unk28c;
	/*0x290*/ u32 unk290;
	/*0x294*/ u32 unk294;
	/*0x298*/ u32 unk298;
	/*0x29c*/ u32 unk29c;
	/*0x2a0*/ u32 unk2a0;
	/*0x2a4*/ u32 unk2a4;
	/*0x2a8*/ u32 unk2a8;
	/*0x2ac*/ u32 unk2ac;
	/*0x2b0*/ u32 unk2b0;
	/*0x2b4*/ u32 unk2b4;
	/*0x2b8*/ u32 unk2b8;
	/*0x2bc*/ u32 unk2bc;
	/*0x2c0*/ u32 unk2c0;
	/*0x2c4*/ u32 unk2c4;
	/*0x2c8*/ u32 unk2c8;
	/*0x2cc*/ u32 unk2cc;
	/*0x2d0*/ u16 unk2d0;
	/*0x2d2*/ s8 unk2d2[4];
};

struct challenge {
	/*0x00*/ u16 name;
	/*0x02*/ s16 confignum;

	// One byte for each number of players,
	// and 4 bits in each to mark which players completed it
	/*0x04*/ u8 completions[4];

	/*0x08*/ u16 unk08;
	/*0x0a*/ u16 unk0a;
	/*0x0c*/ u16 unk0c;
	/*0x0e*/ u16 unk0e;
	/*0x10*/ u16 unk10;
	/*0x12*/ u16 unk12;
	/*0x14*/ u16 unk14;
	/*0x16*/ u16 unk16;
	/*0x18*/ u16 unk18;
};

struct scenariodata_cbt {
	u32 unk00;
	u32 unk04;
	u32 unk08;
	u16 unk0c;
	s16 unk0e[1]; // possibly for a different scenario - see mpGetNumTeammatesDefendingHill
};

struct scenariodata_htb {
	u32 unk00;
	struct prop *token; // Briefcase or chr
	struct coord pos;
};

struct htmthing {
	struct prop *prop;
	u32 unk04;
	u32 unk08;
};

struct scenariodata_htm {
	/*0x800ac110*/ u32 unk000;
	/*0x800ac114*/ u32 unk004;
	/*0x800ac118*/ u32 unk008;
	/*0x800ac11c*/ u32 unk00c;
	/*0x800ac120*/ u32 unk010;
	/*0x800ac124*/ u32 unk014;
	/*0x800ac128*/ u32 unk018;
	/*0x800ac12c*/ u32 unk01c;
	/*0x800ac130*/ u32 unk020;
	/*0x800ac134*/ u32 unk024;
	/*0x800ac138*/ u32 unk028;
	/*0x800ac13c*/ u32 unk02c;
	/*0x800ac140*/ u32 unk030;
	/*0x800ac144*/ u32 unk034;
	/*0x800ac148*/ u32 unk038;
	/*0x800ac14c*/ u32 unk03c;
	/*0x800ac150*/ u32 unk040;
	/*0x800ac154*/ u32 unk044;
	/*0x800ac158*/ u32 unk048;
	/*0x800ac15c*/ u32 unk04c;
	/*0x800ac160*/ u32 unk050;
	/*0x800ac164*/ u32 unk054;
	/*0x800ac168*/ s32 unk058;
	/*0x800ac16c*/ s32 unk05c;
	/*0x800ac170*/ s32 unk060;
	/*0x800ac174*/ s32 unk064;
	/*0x800ac178*/ u32 unk068;
	/*0x800ac17c*/ u32 unk06c;
	/*0x800ac180*/ u32 unk070;
	/*0x800ac184*/ u32 unk074;
	/*0x800ac188*/ u32 unk078;
	/*0x800ac18c*/ u32 unk07c;
	/*0x800ac190*/ struct htmthing unk080[1];
	/*0x800ac19c*/ u32 unk08c;
	/*0x800ac1a0*/ u32 unk090;
	/*0x800ac1a4*/ u32 unk094;
	/*0x800ac1a8*/ u32 unk098;
	/*0x800ac1ac*/ u32 unk09c;
	/*0x800ac1b0*/ u32 unk0a0;
	/*0x800ac1b4*/ u32 unk0a4;
	/*0x800ac1b8*/ u32 unk0a8;
	/*0x800ac1bc*/ u32 unk0ac;
	/*0x800ac1c0*/ u32 unk0b0;
	/*0x800ac1c4*/ u32 unk0b4;
	/*0x800ac1c8*/ u32 unk0b8;
	/*0x800ac1cc*/ u32 unk0bc;
	/*0x800ac1d0*/ u32 unk0c0;
	/*0x800ac1d4*/ u32 unk0c4;
	/*0x800ac1d8*/ u32 unk0c8;
	/*0x800ac1dc*/ u32 unk0cc;
	/*0x800ac1e0*/ u32 unk0d0;
	/*0x800ac1e4*/ u32 unk0d4;
	/*0x800ac1e8*/ u32 unk0d8[1];
	/*0x800ac1ec*/ u32 unk0dc;
	/*0x800ac1f0*/ u32 unk0e0;
	/*0x800ac1f4*/ u32 unk0e4;
	/*0x800ac1f8*/ u32 unk0e8;
	/*0x800ac1fc*/ u32 unk0ec;
	/*0x800ac200*/ u32 unk0f0;
	/*0x800ac204*/ u32 unk0f4;
	/*0x800ac208*/ u32 unk0f8;
	/*0x800ac20c*/ u32 unk0fc;
	/*0x800ac210*/ u32 unk100;
	/*0x800ac214*/ u32 unk104;
	/*0x800ac218*/ u32 unk108;
	/*0x800ac21c*/ u32 unk10c;
	/*0x800ac220*/ u32 unk110;
	/*0x800ac224*/ u32 unk114;
	/*0x800ac228*/ u32 unk118;
	/*0x800ac22c*/ u32 unk11c;
	/*0x800ac230*/ u32 unk120;
	/*0x800ac234*/ u32 unk124;
	/*0x800ac238*/ u32 unk128;
	/*0x800ac23c*/ u32 unk12c;
	/*0x800ac240*/ u32 unk130;
	/*0x800ac244*/ u32 unk134;
	/*0x800ac248*/ u32 unk138;
	/*0x800ac24c*/ struct prop *uplink;
};

struct scenariodata_pac {
	s16 unk00;
	u16 age240;
	s32 victimindex;
	s16 victims[12]; // shuffled list of player numbers
	s16 unk20[12];
	s16 wincounts[12]; // indexed by player num
};

struct scenariodata_koh {
	/*0x800ac110*/ u32 unk00;
	/*0x800ac114*/ s16 occupiedteam;
	/*0x800ac116*/ s16 unk06;
	/*0x800ac118*/ s16 unk08;
	/*0x800ac11a*/ s16 hillindex;
	/*0x800ac11c*/ s16 hillcount;
	/*0x800ac11e*/ s16 hillroom;
	/*0x800ac120*/ s16 unk10;
	/*0x800ac122*/ s16 hillpads[9];
	/*0x800ac134*/ struct coord hillpos;
	/*0x800ac140*/ f32 unk30;
	/*0x800ac144*/ f32 unk34;
	/*0x800ac148*/ f32 unk38;
};

struct ctcthing {
	s16 unk00;
	s16 unk02;
	s16 unk04[6];
};

struct scenariodata_ctc {
	/*0x00*/ s16 unk00[4];
	/*0x08*/ s16 unk08[4];
	/*0x10*/ s16 baserooms[4];
	/*0x18*/ struct ctcthing unk18[4];
	/*0x58*/ struct prop *tokens[4];
};

struct scenariodata {
	union {
		struct scenariodata_cbt cbt;
		struct scenariodata_htb htb;
		struct scenariodata_htm htm;
		struct scenariodata_pac pac;
		struct scenariodata_koh koh;
		struct scenariodata_ctc ctc;
	};
};

struct portal {
	u16 unk00;
	s16 unk02;
	s16 unk04;
	u8 flags;
};

struct portalthing {
	u16 unk00;
	u16 unk02;
	u16 unk04;
	s16 unk06;
	u32 unk08;
	u32 unk0c;
};

struct trainingdata {
	u8 intraining : 1;
	u8 failed : 1;
	u8 completed : 1;
	u8 finished : 1;
	u8 holographedpc : 1;
	s8 timeleft;
	s32 timetaken;
	struct defaultobj *obj;
	u32 unk0c;
};

struct activemenuthing {
	/*0x00*/ s8 screenindex;
	/*0x02*/ u16 unk02;
	/*0x04*/ s16 unk04;
	/*0x06*/ u16 unk06;
	/*0x08*/ u16 unk08;
	/*0x0a*/ s16 unk0a;
	/*0x0c*/ u16 unk0c;
	/*0x0e*/ u8 slotnum;
	/*0x10*/ u32 unk10;
	/*0x14*/ u32 unk14;
	/*0x18*/ f32 unk18; // seems like a fade duration or current alpha level
	/*0x1c*/ f32 unk1c;
	/*0x20*/ u8 weaponnums[8];
	/*0x28*/ u32 unk28;
	/*0x2c*/ u32 unk2c;
	/*0x30*/ u8 unk30;
	/*0x31*/ u8 unk31;
	/*0x32*/ u8 allbots; // when player holds R on the bot command screen
	/*0x33*/ u8 unk33;
	/*0x34*/ u8 unk34;
};

struct objective {
	/*0x00*/ u32 unk00;
	/*0x04*/ u32 unk04;
	/*0x08*/ u32 text;
	/*0x0c*/ u16 unk0c;
	/*0x0e*/ u8 unk0e;
	/*0x0f*/ s8 difficulties;
};

struct briefing {
	u16 objectivenames[6];
	u16 objectivedifficulties[6];
};

struct criteria_roomentered {
	u32 unk00;
	u32 pad;
	u32 status;
	struct criteria_roomentered *next;
};

struct criteria_multiroomentered {
	u32 unk00;
	u32 unk04;
	u32 pad;
	u32 status;
	struct criteria_multiroomentered *next;
};

struct criteria_holograph {
	u32 unk00;
	u32 obj;
	u32 status;
	struct criteria_holograph *next;
};

struct objectivething {
	u32 unk00;
	u32 unk04;
	u32 name;
	struct objectivething *next;
};

struct mppreset {
	u16 name;
	u32 confignum;
	u8 challenges[16];
};

struct explosiontype {
	f32 rangeh;
	f32 rangev;
	f32 changerateh;
	f32 changeratev;
	f32 innersize;
	f32 blastradius;
	f32 damageradius;
	u16 duration;
	u16 propagationrate;
	f32 flarespeed;
	u8 smoketype;
	u16 sound;
	f32 damage;
};

struct smoketype {
	u16 duration;
	u16 fadespeed;
	u16 spreadspeed;
	u16 size;
	f32 bgrotatespeed;
	u32 colour;
	f32 fgrotatespeed;
	u16 numclouds;
	f32 unk18;
	f32 unk1c;
	f32 unk20;
};

struct textoverride {
	/*0x00*/ u32 unk00;
	/*0x04*/ u32 unk04;
	/*0x08*/ s32 weapon;
	/*0x0c*/ u32 unk0c;
	/*0x10*/ u32 unk10;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 activatetextid;
	/*0x20*/ struct textoverride *next;
	/*0x24*/ struct defaultobj *obj;
};

struct var800a2380 {
	/*0x000*/ u32 unk000;
	/*0x004*/ u32 unk004;
	/*0x008*/ u32 unk008;
	/*0x00c*/ u32 unk00c;
	/*0x010*/ u32 unk010;
	/*0x014*/ u32 unk014;
	/*0x018*/ u32 unk018;
	/*0x01c*/ u32 unk01c;
	/*0x020*/ u32 unk020;
	/*0x024*/ u32 unk024;
	/*0x028*/ u32 unk028;
	/*0x02c*/ u32 unk02c;
	/*0x030*/ u32 unk030;
	/*0x034*/ u32 unk034;
	/*0x038*/ u32 unk038;
	/*0x03c*/ u32 unk03c;
	/*0x040*/ u32 unk040;
	/*0x044*/ u32 unk044;
	/*0x048*/ u32 unk048;
	/*0x04c*/ u32 unk04c;
	/*0x050*/ u32 unk050;
	/*0x054*/ u32 unk054;
	/*0x058*/ u32 unk058;
	/*0x05c*/ u32 unk05c;
	/*0x060*/ u32 unk060;
	/*0x064*/ u32 unk064;
	/*0x068*/ u32 unk068;
	/*0x06c*/ u32 unk06c;
	/*0x070*/ u32 unk070;
	/*0x074*/ u32 unk074;
	/*0x078*/ u32 unk078;
	/*0x07c*/ u32 unk07c;
	/*0x080*/ u32 unk080;
	/*0x084*/ u32 unk084;
	/*0x088*/ u32 unk088;
	/*0x08c*/ u32 unk08c;
	/*0x090*/ u32 unk090;
	/*0x094*/ u32 unk094;
	/*0x098*/ u32 unk098;
	/*0x09c*/ u32 unk09c;
	/*0x0a0*/ u32 unk0a0;
	/*0x0a4*/ u32 unk0a4;
	/*0x0a8*/ u32 unk0a8;
	/*0x0ac*/ u32 unk0ac;
	/*0x0b0*/ u32 unk0b0;
	/*0x0b4*/ u32 unk0b4;
	/*0x0b8*/ u32 unk0b8;
	/*0x0bc*/ u32 unk0bc;
	/*0x0c0*/ u32 unk0c0;
	/*0x0c4*/ u32 unk0c4;
	/*0x0c8*/ u32 unk0c8;
	/*0x0cc*/ u32 unk0cc;
	/*0x0d0*/ u32 unk0d0;
	/*0x0d4*/ u32 unk0d4;
	/*0x0d8*/ u32 unk0d8;
	/*0x0dc*/ u32 unk0dc;
	/*0x0e0*/ u32 unk0e0;
	/*0x0e4*/ u32 unk0e4;
	/*0x0e8*/ u32 unk0e8;
	/*0x0ec*/ u32 unk0ec;
	/*0x0f0*/ u32 unk0f0;
	/*0x0f4*/ u32 unk0f4;
	/*0x0f8*/ u32 unk0f8;
	/*0x0fc*/ u32 unk0fc;
	/*0x100*/ u32 unk100;
	/*0x104*/ u32 unk104;
	/*0x108*/ u32 unk108;
	/*0x10c*/ u32 unk10c;
	/*0x110*/ u32 unk110;
	/*0x114*/ u32 unk114;
	/*0x118*/ u32 unk118;
	/*0x11c*/ u32 unk11c;
	/*0x120*/ u32 unk120;
	/*0x124*/ u32 unk124;
	/*0x128*/ u32 unk128;
	/*0x12c*/ u32 unk12c;
	/*0x130*/ u32 unk130;
	/*0x134*/ u32 unk134;
	/*0x138*/ u32 unk138;
	/*0x13c*/ u32 unk13c;
	/*0x140*/ u32 unk140;
	/*0x144*/ u32 unk144;
	/*0x148*/ u32 unk148;
	/*0x14c*/ u32 unk14c;
	/*0x150*/ u32 unk150;
	/*0x154*/ u32 unk154;
	/*0x158*/ u32 unk158;
	/*0x15c*/ u32 unk15c;
	/*0x160*/ u32 unk160;
	/*0x164*/ u32 unk164;
	/*0x168*/ u32 unk168;
	/*0x16c*/ u32 unk16c;
	/*0x170*/ u32 unk170;
	/*0x174*/ u32 unk174;
	/*0x178*/ u32 unk178;
	/*0x17c*/ u32 unk17c;
	/*0x180*/ u32 unk180;
	/*0x184*/ u32 unk184;
	/*0x188*/ u32 unk188;
	/*0x18c*/ u32 unk18c;
	/*0x190*/ u32 unk190;
	/*0x194*/ u32 unk194;
	/*0x198*/ u32 unk198;
	/*0x19c*/ u32 unk19c;
	/*0x1a0*/ u32 unk1a0;
	/*0x1a4*/ u32 unk1a4;
	/*0x1a8*/ u32 unk1a8;
	/*0x1ac*/ u32 unk1ac;
	/*0x1b0*/ u32 unk1b0;
	/*0x1b4*/ u32 unk1b4;
	/*0x1b8*/ u32 unk1b8;
	/*0x1bc*/ u32 unk1bc;
	/*0x1c0*/ u32 unk1c0;
	/*0x1c4*/ u32 unk1c4;
	/*0x1c8*/ u32 unk1c8;
	/*0x1cc*/ u32 unk1cc;
	/*0x1d0*/ u32 unk1d0;
	/*0x1d4*/ u32 unk1d4;
	/*0x1d8*/ u32 unk1d8;
	/*0x1dc*/ u32 unk1dc;
	/*0x1e0*/ u32 unk1e0;
	/*0x1e4*/ u32 unk1e4;
	/*0x1e8*/ u32 unk1e8;
	/*0x1ec*/ u32 unk1ec;
	/*0x1f0*/ u32 unk1f0;
	/*0x1f4*/ u32 unk1f4;
	/*0x1f8*/ u32 unk1f8;
	/*0x1fc*/ u32 unk1fc;
	/*0x200*/ u32 unk200;
	/*0x204*/ u32 unk204;
	/*0x208*/ u32 unk208;
	/*0x20c*/ u32 unk20c;
	/*0x210*/ u32 unk210;
	/*0x214*/ u32 unk214;
	/*0x218*/ u32 unk218;
	/*0x21c*/ u32 unk21c;
	/*0x220*/ u32 unk220;
	/*0x224*/ u32 unk224;
	/*0x228*/ u32 unk228;
	/*0x22c*/ u32 unk22c;
	/*0x230*/ u32 unk230;
	/*0x234*/ u32 unk234;
	/*0x238*/ u32 unk238;
	/*0x23c*/ u32 unk23c;
	/*0x240*/ u32 unk240;
	/*0x244*/ u32 unk244;
	/*0x248*/ u32 unk248;
	/*0x24c*/ u32 unk24c;
	/*0x250*/ u32 unk250;
	/*0x254*/ u32 unk254;
	/*0x258*/ u32 unk258;
	/*0x25c*/ u32 unk25c;
	/*0x260*/ u32 unk260;
	/*0x264*/ u32 unk264;
	/*0x268*/ u32 unk268;
	/*0x26c*/ u32 unk26c;
	/*0x270*/ u32 unk270;
	/*0x274*/ u32 unk274;
	/*0x278*/ u32 unk278;
	/*0x27c*/ u32 unk27c;
	/*0x280*/ u32 unk280;
	/*0x284*/ u32 unk284;
	/*0x288*/ u32 unk288;
	/*0x28c*/ u32 unk28c;
	/*0x290*/ u32 unk290;
	/*0x294*/ u32 unk294;
	/*0x298*/ u32 unk298;
	/*0x29c*/ u32 unk29c;
	/*0x2a0*/ u32 unk2a0;
	/*0x2a4*/ u32 unk2a4;
	/*0x2a8*/ u32 unk2a8;
	/*0x2ac*/ u32 unk2ac;
	/*0x2b0*/ u32 unk2b0;
	/*0x2b4*/ u32 unk2b4;
	/*0x2b8*/ u32 unk2b8;
	/*0x2bc*/ u32 unk2bc;
	/*0x2c0*/ u32 unk2c0;
	/*0x2c4*/ u32 unk2c4;
	/*0x2c8*/ u32 unk2c8;
};

struct gecreditsdata {
	/*0x00*/ u16 text1;
	/*0x02*/ u16 text2;
	/*0x04*/ u16 posoffset1;
	/*0x06*/ u16 alignoffset1;
	/*0x08*/ u16 posoffset2;
	/*0x0a*/ u16 alignoffset2;
};

struct memorypool {
	/*0x00*/ u32 unk00;
	/*0x04*/ void *start;
	/*0x08*/ void *end;
	/*0x0c*/ u32 unk0c;
	/*0x10*/ u32 unk10;
};

struct invitem_weap {
	s16 weapon1;
	s16 pickuppad;
};

struct invitem_prop {
	struct prop *prop;
};

struct invitem_dual {
	s32 weapon1;
	s32 weapon2;
};

struct invitem {
	/*0x00*/ s32 type;

	union {
		struct invitem_weap type_weap;
		struct invitem_prop type_prop;
		struct invitem_dual type_dual;
	};

	/*0x0c*/ struct invitem *next;
	/*0x10*/ struct invitem *prev;
};

struct chrnumaction {
	s16 chrnum;
	u8 myaction;
};

struct propdefinition {
	void *filedata;
	u16 fileid;
	u16 scale;
};

struct var80087d14 {
	u8 unk00;
	f32 unk04;
	f32 unk08;
	u16 unk0c;
	f32 unk10;
	f32 unk14;
	f32 unk18;
	s32 blurdrugamount;
};

struct somedruggedracethingdeep {
	s16 unk00;
	u32 unk04;
	f32 unk08;
	f32 unk0c;
	u32 unk10;
	f32 unk14;
	f32 unk18;
};

struct somedruggedracething {
	s32 index;
	struct somedruggedracethingdeep *unk04;
	struct somedruggedracethingdeep *unk08;
	s32 unk0c;
	u32 unk10;
};

struct var80075c00 {
	u16 unk00;
	f32 unk04;
	f32 unk08;
	f32 unk0c;
	f32 unk10;
	f32 unk14;
};

struct var800705c0 {
	u16 unk00;
	s16 unk02;
	u16 unk04;
	s16 unk06;
	u32 unk08;
	u32 unk0c;
	u32 unk10;
	u32 unk14;
	u32 unk18;
	u32 unk1c;
	u32 unk20;
	u32 unk24;
	u32 unk28;
};

struct miscbio {
	u32 name;
	u32 description;
};

struct hangarbio {
	u32 name;
	u32 description;
	u32 unk08;
	u32 unk0c;
};

struct chrbio {
	u32 name;
	u32 race;
	u32 age;
	u32 description;
};

struct mpteaminfo {
	u32 unk00;
	u32 unk04;
	u32 unk08;
	u32 unk0c;
	s32 score;
};

struct hudmessagething {
	s32 unk00;
};

struct hudmessageconfig {
	/*0x00*/ u8 unk00;
	/*0x01*/ u8 unk01;
	/*0x02*/ u8 unk02;
	/*0x04*/ struct hudmessagething *unk04;
	/*0x08*/ struct hudmessagething *unk08;
	/*0x0c*/ u32 colour;
	/*0x10*/ u32 unk10;
	/*0x14*/ u8 alignh;
	/*0x15*/ u8 alignv;
	/*0x16*/ s16 unk16;
	/*0x18*/ s16 unk18;
	/*0x1c*/ s32 duration;
};

struct hudmessage {
	/*0x000*/ u8 state;
	/*0x001*/ u8 unk001;
	/*0x002*/ u8 unk002;
	/*0x003*/ u8 unk003;
	/*0x004*/ u16 unk004;
	/*0x006*/ u16 unk006;
	/*0x008*/ u32 unk008;
	/*0x00c*/ u32 unk00c;
	/*0x010*/ u32 textcolour;
	/*0x014*/ u32 shadowcolour;
	/*0x018*/ u32 unk018;
	/*0x01c*/ u16 unk01c;
	/*0x01e*/ u16 unk01e;
	/*0x020*/ char text[400];
	/*0x1b0*/ s32 unk1b0;
	/*0x1b4*/ u32 type;
	/*0x1b8*/ s32 id;
	/*0x1bc*/ u32 unk1bc; // duration
	/*0x1c0*/ s32 playernum;
	/*0x1c4*/ u32 flags;
	/*0x1c8*/ u8 alignh;
	/*0x1c9*/ u8 alignv;
	/*0x1cc*/ u32 unk1cc;
	/*0x1d0*/ u32 unk1d0;
	/*0x1d4*/ u32 unk1d4;
	/*0x1d8*/ u32 hash;
};

struct frtarget {
	/*0x00*/ u32 unk00_01 : 1;
	/*0x00*/ u32 unk00_02 : 1;
	/*0x00*/ u32 unk00_03 : 1;
	/*0x00*/ u32 unk00_04 : 1;
	/*0x00*/ u32 unk00_05 : 1;
	/*0x00*/ u32 unk00_06 : 1;
	/*0x00*/ u32 unk00_07 : 1;
	/*0x00*/ u32 unk00_08 : 1;
	/*0x04*/ struct prop *prop;
	/*0x08*/ struct coord unk08;
	/*0x14*/ u32 unk14;
	/*0x18*/ u32 unk18;
	/*0x1c*/ f32 unk1c;
	/*0x20*/ u8 unk20;
	/*0x21*/ u8 unk21;
	/*0x24*/ f32 unk24;
	/*0x28*/ f32 unk28;
	/*0x2c*/ f32 unk2c;
	/*0x30*/ u8 unk30;
	/*0x31*/ u8 unk31;
	/*0x32*/ u8 unk32;
	/*0x33*/ u8 unk33;
	/*0x34*/ s8 unk34;
	/*0x38*/ u32 unk38;
};

struct frdata {
	/*0x000*/ u8 unk000;
	/*0x001*/ u8 unk001;
	/*0x002*/ u16 goalscore;
	/*0x004*/ u8 goaltargets;
	/*0x005*/ u8 timelimit;
	/*0x006*/ u8 ammolimit;
	/*0x007*/ u8 sdgrenadelimit;
	/*0x008*/ u8 goalaccuracy;
	/*0x00c*/ f32 unk00c;
	/*0x010*/ struct frtarget targets[18];
	/*0x448*/ u8 difficulty;
	/*0x44c*/ s32 timetaken;
	/*0x450*/ u32 score;
	/*0x454*/ u8 unk454;
	/*0x455*/ u8 targetsdestroyed;
	/*0x456*/ u16 slot;
	/*0x458*/ u16 numshots;
	/*0x45a*/ u8 unk45a;
	/*0x45b*/ u8 failreason;
	/*0x45c*/ u16 numhitstype1;
	/*0x45e*/ u16 numhitstype2;
	/*0x460*/ u16 numhitstype3;
	/*0x462*/ u16 numhitstype4;
	/*0x464*/ s8 unk464;
	/*0x465*/ u8 unk465_00 : 3;
	/*0x465*/ u8 unk465_03 : 1;
	/*0x465*/ u8 unk465_04 : 1;
	/*0x465*/ u8 unk465_05 : 1;
	/*0x465*/ u8 unk465_06 : 2;
	/*0x466*/ u8 unk466;
	/*0x467*/ u8 unk467;
	/*0x468*/ u8 unk468;
	/*0x46c*/ u32 unk46c;
	/*0x470*/ u8 unk470;
	/*0x471*/ u8 feedbackzone;
	/*0x472*/ s8 feedbackttl;
	/*0x474*/ u16 unk474;
	/*0x476*/ s16 unk476;
	/*0x478*/ u32 unk478;
	/*0x47c*/ u32 unk47c;
};

struct menudata {
	/*0x000*/ u32 count;
	/*0x004*/ u32 root;
	/*0x008*/ u32 unk008;
	/*0x00c*/ u32 unk00c;
	/*0x010*/ u32 unk010;
	/*0x014*/ u8 unk014;
	/*0x015*/ u8 unk015;
	/*0x016*/ u8 unk016;
	/*0x017*/ u8 unk017[4];
	/*0x01b*/ u8 unk01b;
	/*0x01c*/ u32 unk01c;
	/*0x020*/ u32 unk020;
	/*0x024*/ u32 unk024;
	/*0x028*/ u32 unk028;
	/*0x02c*/ u32 unk02c;
	/*0x030*/ u32 unk030;
	/*0x034*/ u32 unk034;
	/*0x038*/ u32 unk038;
	/*0x03c*/ u32 unk03c;
	/*0x040*/ u32 unk040;
	/*0x044*/ u32 unk044;
	/*0x048*/ u32 unk048;
	/*0x04c*/ u32 unk04c;
	/*0x050*/ u32 unk050;
	/*0x054*/ u32 unk054;
	/*0x058*/ u32 unk058;
	/*0x05c*/ u32 unk05c;
	/*0x060*/ u32 unk060;
	/*0x064*/ u32 unk064;
	/*0x068*/ u32 unk068;
	/*0x06c*/ u32 unk06c;
	/*0x070*/ u32 unk070;
	/*0x074*/ u32 unk074;
	/*0x078*/ u32 unk078;
	/*0x07c*/ u32 unk07c;
	/*0x080*/ u32 unk080;
	/*0x084*/ u32 unk084;
	/*0x088*/ u32 unk088;
	/*0x08c*/ u32 unk08c;
	/*0x090*/ u32 unk090;
	/*0x094*/ u32 unk094;
	/*0x098*/ u32 unk098;
	/*0x09c*/ u32 unk09c;
	/*0x0a0*/ u32 unk0a0;
	/*0x0a4*/ u32 unk0a4;
	/*0x0a8*/ u32 unk0a8;
	/*0x0ac*/ u32 unk0ac;
	/*0x0b0*/ u32 unk0b0;
	/*0x0b4*/ u32 unk0b4;
	/*0x0b8*/ u32 unk0b8;
	/*0x0bc*/ u32 unk0bc;
	/*0x0c0*/ u32 unk0c0;
	/*0x0c4*/ u32 unk0c4;
	/*0x0c8*/ u32 unk0c8;
	/*0x0cc*/ u32 unk0cc;
	/*0x0d0*/ u32 unk0d0;
	/*0x0d4*/ u32 unk0d4;
	/*0x0d8*/ u32 unk0d8;
	/*0x0dc*/ u32 unk0dc;
	/*0x0e0*/ u32 unk0e0;
	/*0x0e4*/ u32 unk0e4;
	/*0x0e8*/ u32 unk0e8;
	/*0x0ec*/ u32 unk0ec;
	/*0x0f0*/ u32 unk0f0;
	/*0x0f4*/ u32 unk0f4;
	/*0x0f8*/ u32 unk0f8;
	/*0x0fc*/ u32 unk0fc;
	/*0x100*/ u32 unk100;
	/*0x104*/ u32 unk104;
	/*0x108*/ u32 unk108;
	/*0x10c*/ u32 unk10c;
	/*0x110*/ u32 unk110;
	/*0x114*/ u32 unk114;
	/*0x118*/ u32 unk118;
	/*0x11c*/ u32 unk11c;
	/*0x120*/ u32 unk120;
	/*0x124*/ u32 unk124;
	/*0x128*/ u32 unk128;
	/*0x12c*/ u32 unk12c;
	/*0x130*/ u32 unk130;
	/*0x134*/ u32 unk134;
	/*0x138*/ u32 unk138;
	/*0x13c*/ u32 unk13c;
	/*0x140*/ u32 unk140;
	/*0x144*/ u32 unk144;
	/*0x148*/ u32 unk148;
	/*0x14c*/ u32 unk14c;
	/*0x150*/ u32 unk150;
	/*0x154*/ u32 unk154;
	/*0x158*/ u32 unk158;
	/*0x15c*/ u32 unk15c;
	/*0x160*/ u32 unk160;
	/*0x164*/ u32 unk164;
	/*0x168*/ u32 unk168;
	/*0x16c*/ u32 unk16c;
	/*0x170*/ u32 unk170;
	/*0x174*/ u32 unk174;
	/*0x178*/ u32 unk178;
	/*0x17c*/ u32 unk17c;
	/*0x180*/ u32 unk180;
	/*0x184*/ u32 unk184;
	/*0x188*/ u32 unk188;
	/*0x18c*/ u32 unk18c;
	/*0x190*/ u32 unk190;
	/*0x194*/ u32 unk194;
	/*0x198*/ u32 unk198;
	/*0x19c*/ u32 unk19c;
	/*0x1a0*/ u32 unk1a0;
	/*0x1a4*/ u32 unk1a4;
	/*0x1a8*/ u32 unk1a8;
	/*0x1ac*/ u32 unk1ac;
	/*0x1b0*/ u32 unk1b0;
	/*0x1b4*/ u32 unk1b4;
	/*0x1b8*/ u32 unk1b8;
	/*0x1bc*/ u32 unk1bc;
	/*0x1c0*/ u32 unk1c0;
	/*0x1c4*/ u32 unk1c4;
	/*0x1c8*/ u32 unk1c8;
	/*0x1cc*/ u32 unk1cc;
	/*0x1d0*/ u32 unk1d0;
	/*0x1d4*/ u32 unk1d4;
	/*0x1d8*/ u32 unk1d8;
	/*0x1dc*/ u32 unk1dc;
	/*0x1e0*/ u32 unk1e0;
	/*0x1e4*/ u32 unk1e4;
	/*0x1e8*/ u32 unk1e8;
	/*0x1ec*/ u32 unk1ec;
	/*0x1f0*/ u32 unk1f0;
	/*0x1f4*/ u32 unk1f4;
	/*0x1f8*/ u32 unk1f8;
	/*0x1fc*/ u32 unk1fc;
	/*0x200*/ u32 unk200;
	/*0x204*/ u32 unk204;
	/*0x208*/ u32 unk208;
	/*0x20c*/ u32 unk20c;
	/*0x210*/ u32 unk210;
	/*0x214*/ u32 unk214;
	/*0x218*/ u32 unk218;
	/*0x21c*/ u32 unk21c;
	/*0x220*/ u32 unk220;
	/*0x224*/ u32 unk224;
	/*0x228*/ u32 unk228;
	/*0x22c*/ u32 unk22c;
	/*0x230*/ u32 unk230;
	/*0x234*/ u32 unk234;
	/*0x238*/ u32 unk238;
	/*0x23c*/ u32 unk23c;
	/*0x240*/ u32 unk240;
	/*0x244*/ u32 unk244;
	/*0x248*/ u32 unk248;
	/*0x24c*/ u32 unk24c;
	/*0x250*/ u32 unk250;
	/*0x254*/ u32 unk254;
	/*0x258*/ u32 unk258;
	/*0x25c*/ u32 unk25c;
	/*0x260*/ u32 unk260;
	/*0x264*/ u32 unk264;
	/*0x268*/ u32 unk268;
	/*0x26c*/ u32 unk26c;
	/*0x270*/ u32 unk270;
	/*0x274*/ u32 unk274;
	/*0x278*/ u32 unk278;
	/*0x27c*/ u32 unk27c;
	/*0x280*/ u32 unk280;
	/*0x284*/ u32 unk284;
	/*0x288*/ u32 unk288;
	/*0x28c*/ u32 unk28c;
	/*0x290*/ u32 unk290;
	/*0x294*/ u32 unk294;
	/*0x298*/ u32 unk298;
	/*0x29c*/ u32 unk29c;
	/*0x2a0*/ u32 unk2a0;
	/*0x2a4*/ u32 unk2a4;
	/*0x2a8*/ u32 unk2a8;
	/*0x2ac*/ u32 unk2ac;
	/*0x2b0*/ u32 unk2b0;
	/*0x2b4*/ u32 unk2b4;
	/*0x2b8*/ u32 unk2b8;
	/*0x2bc*/ u32 unk2bc;
	/*0x2c0*/ u32 unk2c0;
	/*0x2c4*/ u32 unk2c4;
	/*0x2c8*/ u32 unk2c8;
	/*0x2cc*/ u32 unk2cc;
	/*0x2d0*/ u32 unk2d0;
	/*0x2d4*/ u32 unk2d4;
	/*0x2d8*/ u32 unk2d8;
	/*0x2dc*/ u32 unk2dc;
	/*0x2e0*/ u32 unk2e0;
	/*0x2e4*/ u32 unk2e4;
	/*0x2e8*/ u32 unk2e8;
	/*0x2ec*/ u32 unk2ec;
	/*0x2f0*/ u32 unk2f0;
	/*0x2f4*/ u32 unk2f4;
	/*0x2f8*/ u32 unk2f8;
	/*0x2fc*/ u32 unk2fc;
	/*0x300*/ u32 unk300;
	/*0x304*/ u32 unk304;
	/*0x308*/ u32 unk308;
	/*0x30c*/ u32 unk30c;
	/*0x310*/ u32 unk310;
	/*0x314*/ u32 unk314;
	/*0x318*/ u32 unk318;
	/*0x31c*/ u32 unk31c;
	/*0x320*/ u32 unk320;
	/*0x324*/ u32 unk324;
	/*0x328*/ u32 unk328;
	/*0x32c*/ u32 unk32c;
	/*0x330*/ u32 unk330;
	/*0x334*/ u32 unk334;
	/*0x338*/ u32 unk338;
	/*0x33c*/ u32 unk33c;
	/*0x340*/ u32 unk340;
	/*0x344*/ u32 unk344;
	/*0x348*/ u32 unk348;
	/*0x34c*/ u32 unk34c;
	/*0x350*/ u32 unk350;
	/*0x354*/ u32 unk354;
	/*0x358*/ u32 unk358;
	/*0x35c*/ u32 unk35c;
	/*0x360*/ u32 unk360;
	/*0x364*/ u32 unk364;
	/*0x368*/ u32 unk368;
	/*0x36c*/ u32 unk36c;
	/*0x370*/ u32 unk370;
	/*0x374*/ u32 unk374;
	/*0x378*/ u32 unk378;
	/*0x37c*/ u32 unk37c;
	/*0x380*/ u32 unk380;
	/*0x384*/ u32 unk384;
	/*0x388*/ u32 unk388;
	/*0x38c*/ u32 unk38c;
	/*0x390*/ u32 unk390;
	/*0x394*/ u32 unk394;
	/*0x398*/ u32 unk398;
	/*0x39c*/ u32 unk39c;
	/*0x3a0*/ u32 unk3a0;
	/*0x3a4*/ u32 unk3a4;
	/*0x3a8*/ u32 unk3a8;
	/*0x3ac*/ u32 unk3ac;
	/*0x3b0*/ u32 unk3b0;
	/*0x3b4*/ u32 unk3b4;
	/*0x3b8*/ u32 unk3b8;
	/*0x3bc*/ u32 unk3bc;
	/*0x3c0*/ u32 unk3c0;
	/*0x3c4*/ u32 unk3c4;
	/*0x3c8*/ u32 unk3c8;
	/*0x3cc*/ u32 unk3cc;
	/*0x3d0*/ u32 unk3d0;
	/*0x3d4*/ u32 unk3d4;
	/*0x3d8*/ u32 unk3d8;
	/*0x3dc*/ u32 unk3dc;
	/*0x3e0*/ u32 unk3e0;
	/*0x3e4*/ u32 unk3e4;
	/*0x3e8*/ u32 unk3e8;
	/*0x3ec*/ u32 unk3ec;
	/*0x3f0*/ u32 unk3f0;
	/*0x3f4*/ u32 unk3f4;
	/*0x3f8*/ u32 unk3f8;
	/*0x3fc*/ u32 unk3fc;
	/*0x400*/ u32 unk400;
	/*0x404*/ u32 unk404;
	/*0x408*/ u32 unk408;
	/*0x40c*/ u32 unk40c;
	/*0x410*/ u32 unk410;
	/*0x414*/ u32 unk414;
	/*0x418*/ u32 unk418;
	/*0x41c*/ u32 unk41c;
	/*0x420*/ u32 unk420;
	/*0x424*/ u32 unk424;
	/*0x428*/ u32 unk428;
	/*0x42c*/ u32 unk42c;
	/*0x430*/ u32 unk430;
	/*0x434*/ u32 unk434;
	/*0x438*/ u32 unk438;
	/*0x43c*/ u32 unk43c;
	/*0x440*/ u32 unk440;
	/*0x444*/ u32 unk444;
	/*0x448*/ u32 unk448;
	/*0x44c*/ u32 unk44c;
	/*0x450*/ u32 unk450;
	/*0x454*/ u32 unk454;
	/*0x458*/ u32 unk458;
	/*0x45c*/ u32 unk45c;
	/*0x460*/ u32 unk460;
	/*0x464*/ u32 unk464;
	/*0x468*/ u32 unk468;
	/*0x46c*/ u32 unk46c;
	/*0x470*/ u32 unk470;
	/*0x474*/ u32 unk474;
	/*0x478*/ u32 unk478;
	/*0x47c*/ u32 unk47c;
	/*0x480*/ u32 unk480;
	/*0x484*/ u32 unk484;
	/*0x488*/ u32 unk488;
	/*0x48c*/ u32 unk48c;
	/*0x490*/ u32 unk490;
	/*0x494*/ u32 unk494;
	/*0x498*/ u32 unk498;
	/*0x49c*/ u32 unk49c;
	/*0x4a0*/ u32 unk4a0;
	/*0x4a4*/ u32 unk4a4;
	/*0x4a8*/ u32 unk4a8;
	/*0x4ac*/ u32 unk4ac;
	/*0x4b0*/ u32 unk4b0;
	/*0x4b4*/ u32 unk4b4;
	/*0x4b8*/ u32 unk4b8;
	/*0x4bc*/ u32 unk4bc;
	/*0x4c0*/ u32 unk4c0;
	/*0x4c4*/ u32 unk4c4;
	/*0x4c8*/ u32 unk4c8;
	/*0x4cc*/ u32 unk4cc;
	/*0x4d0*/ u32 unk4d0;
	/*0x4d4*/ u32 unk4d4;
	/*0x4d8*/ u32 unk4d8;
	/*0x4dc*/ u32 unk4dc;
	/*0x4e0*/ u32 unk4e0;
	/*0x4e4*/ u32 unk4e4;
	/*0x4e8*/ u32 unk4e8;
	/*0x4ec*/ u32 unk4ec;
	/*0x4f0*/ u32 unk4f0;
	/*0x4f4*/ u32 unk4f4;
	/*0x4f8*/ u32 unk4f8;
	/*0x4fc*/ u32 unk4fc;
	/*0x500*/ u32 unk500;
	/*0x504*/ u32 unk504;
	/*0x508*/ u32 unk508;
	/*0x50c*/ u32 unk50c;
	/*0x510*/ u32 unk510;
	/*0x514*/ u32 unk514;
	/*0x518*/ u32 unk518;
	/*0x51c*/ u32 unk51c;
	/*0x520*/ u32 unk520;
	/*0x524*/ u32 unk524;
	/*0x528*/ u32 unk528;
	/*0x52c*/ u32 unk52c;
	/*0x530*/ u32 unk530;
	/*0x534*/ u32 unk534;
	/*0x538*/ u32 unk538;
	/*0x53c*/ u32 unk53c;
	/*0x540*/ u32 unk540;
	/*0x544*/ u32 unk544;
	/*0x548*/ u32 unk548;
	/*0x54c*/ u32 unk54c;
	/*0x550*/ u32 unk550;
	/*0x554*/ u32 unk554;
	/*0x558*/ u32 unk558;
	/*0x55c*/ u32 unk55c;
	/*0x560*/ u32 unk560;
	/*0x564*/ u32 unk564;
	/*0x568*/ u32 unk568;
	/*0x56c*/ u32 unk56c;
	/*0x570*/ u32 unk570;
	/*0x574*/ u32 unk574;
	/*0x578*/ u32 unk578;
	/*0x57c*/ u32 unk57c;
	/*0x580*/ u32 unk580;
	/*0x584*/ u32 unk584;
	/*0x588*/ u32 unk588;
	/*0x58c*/ u32 unk58c;
	/*0x590*/ u32 unk590;
	/*0x594*/ u32 unk594;
	/*0x598*/ u32 unk598;
	/*0x59c*/ u32 unk59c;
	/*0x5a0*/ u32 unk5a0;
	/*0x5a4*/ u32 unk5a4;
	/*0x5a8*/ u32 unk5a8;
	/*0x5ac*/ u32 unk5ac;
	/*0x5b0*/ u32 unk5b0;
	/*0x5b4*/ u32 unk5b4;
	/*0x5b8*/ u32 unk5b8;
	/*0x5bc*/ u32 unk5bc;
	/*0x5c0*/ u32 unk5c0;
	/*0x5c4*/ u32 unk5c4;
	/*0x5c8*/ u32 unk5c8;
	/*0x5cc*/ u32 unk5cc;
	/*0x5d0*/ u32 unk5d0;
	/*0x5d4*/ u8 unk5d4;
	/*0x5d8*/ u32 unk5d8;
	/*0x5dc*/ u32 unk5dc;
	/*0x5e0*/ u32 unk5e0;
	/*0x5e4*/ u32 unk5e4;
	/*0x5e8*/ u32 unk5e8;
	/*0x5ec*/ u32 unk5ec;
	/*0x5f0*/ u32 unk5f0;
	/*0x5f4*/ u32 unk5f4;
	/*0x5f8*/ u32 unk5f8;
	/*0x5fc*/ u32 unk5fc;
	/*0x600*/ u32 unk600;
	/*0x604*/ u32 unk604;
	/*0x608*/ u32 unk608;
	/*0x60c*/ u32 unk60c;
	/*0x610*/ u32 unk610;
	/*0x614*/ u32 unk614;
	/*0x618*/ u32 unk618;
	/*0x61c*/ u32 unk61c;
	/*0x620*/ u32 unk620;
	/*0x624*/ u32 unk624;
	/*0x628*/ u32 unk628;
	/*0x62c*/ u32 unk62c;
	/*0x630*/ u32 unk630;
	/*0x634*/ u32 unk634;
	/*0x638*/ u32 unk638;
	/*0x63c*/ u32 unk63c;
	/*0x640*/ u32 unk640;
	/*0x644*/ u32 unk644;
	/*0x648*/ u32 unk648;
	/*0x64c*/ u32 unk64c;
	/*0x650*/ u32 unk650;
	/*0x654*/ u32 unk654;
	/*0x658*/ u32 unk658;
	/*0x65c*/ u32 unk65c;
	/*0x660*/ u32 unk660;
	/*0x664*/ u32 unk664;
	/*0x668*/ u8 unk668;
	/*0x669*/ u8 unk669[4];
	/*0x66d*/ u8 unk66d;
	/*0x66e*/ s8 unk66e; // index into 669
	/*0x66f*/ u8 unk66f;
};

struct ammotype {
	u32 capacity;
	u32 unk04;
	f32 unk08;
};

struct weather58 {
	f32 unk00;
	f32 unk04;
	s32 unk08;
};

struct weatherdata {
	/*0x00*/ f32 windspeedx;
	/*0x04*/ f32 windspeedz;
	/*0x08*/ f32 windanglerad;
	/*0x0c*/ f32 unk0c;
	/*0x10*/ s32 unk10;
	/*0x14*/ f32 windspeed;
	/*0x18*/ u32 unk18;
	/*0x1c*/ u32 unk1c;
	/*0x20*/ s32 type;
	/*0x24*/ struct weatherparticledata *particledata;
	/*0x28*/ u32 unk28;
	/*0x2c*/ u32 unk2c;
	/*0x30*/ u32 unk30;
	/*0x34*/ void *unk34[4];
	/*0x44*/ s32 unk44;
	/*0x48*/ s32 unk48;
	/*0x4c*/ s32 unk4c;
	/*0x50*/ s32 unk50;
	/*0x54*/ s32 unk54;
	/*0x58*/ struct weather58 unk58[4];
	/*0x88*/ f32 unk88;
	/*0x8c*/ f32 unk8c;
	/*0x90*/ s32 unk90;
	/*0x94*/ s32 unk94;
	/*0x98*/ u32 unk98;
	/*0x9c*/ u32 unk9c;
	/*0xa0*/ u32 unka0;
	/*0xa4*/ u32 unka4;
	/*0xa8*/ u32 unka8;
	/*0xac*/ u32 unkac;
	/*0xb0*/ u32 unkb0;
	/*0xb4*/ u32 unkb4;
	/*0xb8*/ f32 unkb8;
	/*0xbc*/ f32 unkbc;
	/*0xc0*/ s32 unkc0;
	/*0xc4*/ f32 unkc4;
	/*0xc8*/ f32 unkc8;
	/*0xcc*/ s32 intensity;
	/*0xd0*/ s32 unkd0;
	/*0xd4*/ s32 unkd4;
	/*0xd8*/ u32 unkd8;
	/*0xdc*/ u32 unkdc;
	/*0xe0*/ u32 unke0;
	/*0xe4*/ u32 unke4;
	/*0xe8*/ u32 unke8;
	/*0xec*/ u32 unkec;
	/*0xf0*/ u32 unkf0;
	/*0xf4*/ u32 unkf4;
	/*0xf8*/ s16 unkf8;
	/*0xfc*/ u32 unkfc;
};

struct weatherparticle {
	struct coord pos;
	s32 active;
	struct coord inc;
	f32 unk1c;
};

struct weatherparticledata {
	struct weatherparticle particles[500];
	/*0x3e80*/ f32 unk3e80;
	/*0x3e84*/ f32 unk3e84;
	/*0x3e88*/ f32 unk3e88;
	/*0x3e8c*/ struct coord boundarymax;
	/*0x3e98*/ struct coord boundarymin;
	/*0x3ea4*/ struct coord boundaryrange;
	/*0x3eb0*/ u32 unk3eb0;
	/*0x3eb4*/ u32 unk3eb4;
	/*0x3eb8*/ u32 unk3eb8;
	/*0x3ebc*/ u32 unk3ebc;
	/*0x3ec0*/ u32 unk3ec0;
	/*0x3ec4*/ u32 unk3ec4;
	/*0x3ec8*/ f32 unk3ec8[8];
	/*0x3ee8*/ u32 unk3ee8;
	/*0x3eec*/ u32 unk3eec;
	/*0x3ef0*/ u32 unk3ef0;
	/*0x3ef4*/ u32 unk3ef4;
	/*0x3ef8*/ u32 unk3ef8;
	/*0x3efc*/ u32 unk3efc;
};

struct texture {
	u8 unk00;
	u8 unk01;
	u16 dataoffset;
	u32 unk08;
};

struct var800aabb8 {
	u16 unk00_00 : 2;
	u16 unk00_02 : 14;
};

struct fileinfo {
	u32 unk00;
	u32 unk04;
};

struct portalcmd {
	u8 type;
	u8 len;
	s32 param;
};

struct screenbox {
	s16 xmin;
	s16 ymin;
	s16 xmax;
	s16 ymax;
};

struct var800a4640 {
	/*0x000*/ u32 unk000;
	/*0x004*/ u32 unk004;
	/*0x008*/ u32 unk008;
	/*0x00c*/ u32 unk00c;
	/*0x010*/ u32 unk010;
	/*0x014*/ u32 unk014;
	/*0x018*/ u32 unk018;
	/*0x01c*/ u32 unk01c;
	/*0x020*/ u32 unk020;
	/*0x024*/ u32 unk024;
	/*0x028*/ u32 unk028;
	/*0x02c*/ u32 unk02c;
	/*0x030*/ u32 unk030;
	/*0x034*/ u32 unk034;
	/*0x038*/ u32 unk038;
	/*0x03c*/ u32 unk03c;
	/*0x040*/ u32 unk040;
	/*0x044*/ u32 unk044;
	/*0x048*/ u32 unk048;
	/*0x04c*/ u32 unk04c;
	/*0x050*/ u32 unk050;
	/*0x054*/ u32 unk054;
	/*0x058*/ u32 unk058;
	/*0x05c*/ u32 unk05c;
	/*0x060*/ u32 unk060;
	/*0x064*/ u32 unk064;
	/*0x068*/ u32 unk068;
	/*0x06c*/ u32 unk06c;
	/*0x070*/ u32 unk070;
	/*0x074*/ u32 unk074;
	/*0x078*/ u32 unk078;
	/*0x07c*/ u32 unk07c;
	/*0x080*/ u32 unk080;
	/*0x084*/ u32 unk084;
	/*0x088*/ u32 unk088;
	/*0x08c*/ u32 unk08c;
	/*0x090*/ u32 unk090;
	/*0x094*/ u32 unk094;
	/*0x098*/ u32 unk098;
	/*0x09c*/ u32 unk09c;
	/*0x0a0*/ u32 unk0a0;
	/*0x0a4*/ u32 unk0a4;
	/*0x0a8*/ u32 unk0a8;
	/*0x0ac*/ u32 unk0ac;
	/*0x0b0*/ u32 unk0b0;
	/*0x0b4*/ u32 unk0b4;
	/*0x0b8*/ u32 unk0b8;
	/*0x0bc*/ u32 unk0bc;
	/*0x0c0*/ u32 unk0c0;
	/*0x0c4*/ u32 unk0c4;
	/*0x0c8*/ u32 unk0c8;
	/*0x0cc*/ u32 unk0cc;
	/*0x0d0*/ u32 unk0d0;
	/*0x0d4*/ u32 unk0d4;
	/*0x0d8*/ u32 unk0d8;
	/*0x0dc*/ u32 unk0dc;
	/*0x0e0*/ u32 unk0e0;
	/*0x0e4*/ u32 unk0e4;
	/*0x0e8*/ u32 unk0e8;
	/*0x0ec*/ u32 unk0ec;
	/*0x0f0*/ u32 unk0f0;
	/*0x0f4*/ u32 unk0f4;
	/*0x0f8*/ u32 unk0f8;
	/*0x0fc*/ u32 unk0fc;
	/*0x100*/ u32 unk100;
	/*0x104*/ u32 unk104;
	/*0x108*/ u32 unk108;
	/*0x10c*/ u32 unk10c;
	/*0x110*/ u32 unk110;
	/*0x114*/ u32 unk114;
	/*0x118*/ u32 unk118;
	/*0x11c*/ u32 unk11c;
	/*0x120*/ u32 unk120;
	/*0x124*/ u32 unk124;
	/*0x128*/ u32 unk128;
	/*0x12c*/ u32 unk12c;
	/*0x130*/ u32 unk130;
	/*0x134*/ u32 unk134;
	/*0x138*/ u32 unk138;
	/*0x13c*/ u32 unk13c;
	/*0x140*/ u32 unk140;
	/*0x144*/ u32 unk144;
	/*0x148*/ u32 unk148;
	/*0x14c*/ u32 unk14c;
	/*0x150*/ u32 unk150;
	/*0x154*/ u32 unk154;
	/*0x158*/ u32 unk158;
	/*0x15c*/ u32 unk15c;
	/*0x160*/ u32 unk160;
	/*0x164*/ u32 unk164;
	/*0x168*/ u32 unk168;
	/*0x16c*/ u32 unk16c;
	/*0x170*/ u32 unk170;
	/*0x174*/ u32 unk174;
	/*0x178*/ u32 unk178;
	/*0x17c*/ u32 unk17c;
	/*0x180*/ u32 unk180;
	/*0x184*/ u32 unk184;
	/*0x188*/ u32 unk188;
	/*0x18c*/ u32 unk18c;
	/*0x190*/ u32 unk190;
	/*0x194*/ u32 unk194;
	/*0x198*/ u32 unk198;
	/*0x19c*/ u32 unk19c;
	/*0x1a0*/ u32 unk1a0;
	/*0x1a4*/ u32 unk1a4;
	/*0x1a8*/ u32 unk1a8;
	/*0x1ac*/ u32 unk1ac;
	/*0x1b0*/ u32 unk1b0;
	/*0x1b4*/ u32 unk1b4;
	/*0x1b8*/ u32 unk1b8;
	/*0x1bc*/ u32 unk1bc;
	/*0x1c0*/ u32 unk1c0;
	/*0x1c4*/ u32 unk1c4;
	/*0x1c8*/ u32 unk1c8;
	/*0x1cc*/ u32 unk1cc;
	/*0x1d0*/ u32 unk1d0;
	/*0x1d4*/ u32 unk1d4;
	/*0x1d8*/ u32 unk1d8;
	/*0x1dc*/ u32 unk1dc;
	/*0x1e0*/ u32 unk1e0;
	/*0x1e4*/ u32 unk1e4;
	/*0x1e8*/ u32 unk1e8;
	/*0x1ec*/ u32 unk1ec;
	/*0x1f0*/ u32 unk1f0;
	/*0x1f4*/ u32 unk1f4;
	/*0x1f8*/ u32 unk1f8;
	/*0x1fc*/ u32 unk1fc;
	/*0x200*/ u32 unk200;
	/*0x204*/ u32 unk204;
	/*0x208*/ u32 unk208;
	/*0x20c*/ u32 unk20c;
	/*0x210*/ u32 unk210;
	/*0x214*/ u32 unk214;
	/*0x218*/ u32 unk218;
	/*0x21c*/ u32 unk21c;
	/*0x220*/ u32 unk220;
	/*0x224*/ u32 unk224;
	/*0x228*/ u32 unk228;
	/*0x22c*/ u32 unk22c;
	/*0x230*/ u32 unk230;
	/*0x234*/ u32 unk234;
	/*0x238*/ u32 unk238;
	/*0x23c*/ u32 unk23c;
	/*0x240*/ u32 unk240;
	/*0x244*/ u32 unk244;
	/*0x248*/ u32 unk248;
	/*0x24c*/ u32 unk24c;
	/*0x250*/ u32 unk250;
	/*0x254*/ u32 unk254;
	/*0x258*/ u32 unk258;
	/*0x25c*/ u32 unk25c;
	/*0x260*/ u32 unk260;
	/*0x264*/ u32 unk264;
	/*0x268*/ u32 unk268;
	/*0x26c*/ u32 unk26c;
	/*0x270*/ u32 unk270;
	/*0x274*/ u32 unk274;
	/*0x278*/ u32 unk278;
	/*0x27c*/ u32 unk27c;
	/*0x280*/ u32 unk280;
	/*0x284*/ u32 unk284;
	/*0x288*/ u32 unk288;
	/*0x28c*/ u32 unk28c;
	/*0x290*/ u32 unk290;
	/*0x294*/ u32 unk294;
	/*0x298*/ u32 unk298;
	/*0x29c*/ u32 unk29c;
	/*0x2a0*/ u32 unk2a0;
	/*0x2a4*/ u32 unk2a4;
	/*0x2a8*/ u32 unk2a8;
	/*0x2ac*/ u32 unk2ac;
	/*0x2b0*/ u32 unk2b0;
	/*0x2b4*/ u32 unk2b4;
	/*0x2b8*/ u32 unk2b8;
	/*0x2bc*/ u32 unk2bc;
	/*0x2c0*/ u32 unk2c0;
	/*0x2c4*/ u32 unk2c4;
	/*0x2c8*/ u32 unk2c8;
	/*0x2cc*/ u32 unk2cc;
	/*0x2d0*/ s16 unk2d0;
	/*0x2d2*/ u8 unk2d2;
	/*0x2d3*/ u8 unk2d3;
	struct screenbox unk2d4;
};

struct var800a4cf0 {
	u16 unk00;
	u16 unk02;
	u16 unk04;
	f32 unk08;
	f32 unk0c;
};

struct menuthing {
	u8 unk00;
	u8 unk01;
	u8 unk02;
	u8 unk03;
	u32 unk04;
	u8 unk08;
	u8 unk09;
	s8 unk0a;
};

struct mpconfigsim {
	u16 mphead;
	u8 mpbody;
	u8 team;
	u8 difficulties[4]; // per player count
};

struct mpconfig {
	/*0x00*/ char name[12];
	/*0x0c*/ u32 options;
	/*0x10*/ u8 scenario;
	/*0x11*/ u8 stage;
	/*0x12*/ u8 timelimit;       // minus 1, 60 = no limit
	/*0x13*/ u8 scorelimit;      // minus 1, 100 = no limit
	/*0x14*/ u16 teamscorelimit; // minus 1, 400 = no limit
	/*0x16*/ u16 unk16;
	/*0x18*/ u8 weapons[16];
	struct mpconfigsim simulants[8];
};

struct mpweapon {
	u8 weaponnum;
	u8 unk01;
	u8 unk02;
	u8 unk03;
	u16 unk04;
	u16 model;
	u8 unk08;
	u8 unk09;
};

struct mpstrings {
	char description[200];
	char aibotnames[8][15];
};

#endif
