"""
efiguids.py

This is a giant list of protocol GUIDs I grepped out of the TianoCore source code. It should be relatively complete,
but won't contain any of Apple's proprietary GUIDs. I'll add those as I come across them.

See the following URL for more info and the latest version:
https://github.com/snarez/ida-efiutils
"""

GUIDs = {
    'ACPI_TABLE_GUID': [0xeb9d2d30, 0x2d88, 0x11d3, 0x9a, 0x16, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'APPLE_REMOVABLE_MEDIA_PROTOCOL_GUID': [0x2ea9743a, 0x23d9, 0x425e, 0x87, 0x2c, 0xf6, 0x15, 0xaa, 0x19, 0x57, 0x88],
    'ARM_GLOBAL_VARIABLE_PPI_GUID': [0xab1c1816, 0xd542, 0x4e6f, 0x9b, 0x1e, 0x8e, 0xcd, 0x92, 0x53, 0xe2, 0xe7],
    'ARM_HOB_GLOBAL_VARIABLE_GUID': [0xc3253c90, 0xa24f, 0x4599, 0xa6, 0x64, 0x1f, 0x88, 0x13, 0x77, 0x8f, 0xc9],
    'ARM_MP_CORE_INFO_GUID': [0xa4ee0728, 0xe5d7, 0x4ac5, 0xb2, 0x1e, 0x65, 0x8e, 0xd8, 0x57, 0xe8, 0x34],
    'ARM_MP_CORE_INFO_PPI_GUID': [0x6847cc74, 0xe9ec, 0x4f8f, 0xa2, 0x9d, 0xab, 0x44, 0xe7, 0x54, 0xa8, 0xfc],
    'BDS_LIB_STRING_PACKAGE_GUID': [0x3b4d9b23, 0x95ac, 0x44f6, 0x9f, 0xcd, 0xe, 0x95, 0x94, 0x58, 0x6c, 0x72],
    'BLOCKIO_VENDOR_GUID': [0xcf31fac5, 0xc24e, 0x11d2, 0x85, 0xf3, 0x0, 0xa0, 0xc9, 0x3e, 0xc9, 0x3b],
    'BLOCK_MMIO_PROTOCOL_GUID': [0x6b558ce3, 0x69e5, 0x4c67, 0xa6, 0x34, 0xf7, 0xfe, 0x72, 0xad, 0xbe, 0x84],
    'BOOT_MAINT_FORMSET_GUID': [0x642237c7, 0x35d4, 0x472d, 0x83, 0x65, 0x12, 0xe0, 0xcc, 0xf2, 0x7a, 0x22],
    'BOOT_MANAGER_FORMSET_GUID': [0x847bc3fe, 0xb974, 0x446d, 0x94, 0x49, 0x5a, 0xd5, 0x41, 0x2e, 0x99, 0x3b],
    'CONNECT_CONIN_EVENT_GUID': [0xdb4e8151, 0x57ed, 0x4bed, 0x88, 0x33, 0x67, 0x51, 0xb5, 0xd1, 0xa8, 0xd7],
    'DEVICE_MANAGER_FORMSET_GUID': [0x3ebfa8e6, 0x511d, 0x4b5b, 0xa9, 0x5f, 0xfb, 0x38, 0x26, 0xf, 0x1c, 0x27],
    'DP_HII_GUID': [0xeb832fd9, 0x9089, 0x4898, 0x83, 0xc9, 0x41, 0x61, 0x8f, 0x5c, 0x48, 0xb9],
    'DRIVER_HEALTH_FORMSET_GUID': [0xf76e0a70, 0xb5ed, 0x4c38, 0xac, 0x9a, 0xe5, 0xf5, 0x4b, 0xf1, 0x6e, 0x34],
    'DRIVER_SAMPLE_FORMSET_GUID': [0xA04A27f4, 0xDF00, 0x4D42, 0xB5, 0x52, 0x39, 0x51, 0x13, 0x02, 0x11, 0x3D],
    'DRIVER_SAMPLE_INVENTORY_GUID': [0xb3f56470, 0x6141, 0x4621, 0x8f, 0x19, 0x70, 0x4e, 0x57, 0x7a, 0xa9, 0xe8],
    'DUET_CONSOLEOUT_CONFIG_GUID': [0xED150714, 0xDF30, 0x407D, 0xB2, 0x4A, 0x4B, 0x74, 0x2F, 0xD5, 0xCE, 0xA2],
    'DXE_CORE_FILE_NAME_GUID': [0xD6A2CB7F, 0x6A18, 0x4e2f, 0xB4, 0x3B, 0x99, 0x20, 0xA7, 0x33, 0x70, 0x0A],
    'DXE_SERVICES_TABLE_GUID': [0x5ad34ba, 0x6f02, 0x4214, 0x95, 0x2e, 0x4d, 0xa0, 0x39, 0x8e, 0x2b, 0xb9],
    'EBL_ADD_COMMAND_PROTOCOL_GUID': [0xaeda2428, 0x9a22, 0x4637, 0x9b, 0x21, 0x54, 0x5e, 0x28, 0xfb, 0xb8, 0x29],
    'ECP_PEI_PCI_CFG_PPI_GUID': [0xb0ee53d4, 0xa049, 0x4a79, 0xb2, 0xff, 0x19, 0xd9, 0xfa, 0xef, 0xaa, 0x94],
    'EFI_ABSOLUTE_POINTER_PROTOCOL_GUID': [0x8D59D32B, 0xC655, 0x4AE9, 0x9B, 0x15, 0xF2, 0x59, 0x04, 0x99, 0x2A, 0x43],
    'EFI_ACPI_20_TABLE_GUID': [0x8868e871, 0xe4f1, 0x11d3, 0xbc, 0x22, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_ACPI_S3_CONTEXT_GUID': [0xef98d3a, 0x3e33, 0x497a, 0xa4, 0x1, 0x77, 0xbe, 0x3e, 0xb7, 0x4f, 0x38],
    'EFI_ACPI_S3_SAVE_GUID': [0x125f2de1, 0xfb85, 0x440c, 0xa5, 0x4c, 0x4d, 0x99, 0x35, 0x8a, 0x8d, 0x38],
    'EFI_ACPI_SDT_PROTOCOL_GUID': [0xeb97088e, 0xcfdf, 0x49c6, 0xbe, 0x4b, 0xd9, 0x6, 0xa5, 0xb2, 0xe, 0x86],
    'EFI_ACPI_SUPPORT_GUID': [0xdbff9d55, 0x89b7, 0x46da, 0xbd, 0xdf, 0x67, 0x7d, 0x3d, 0xc0, 0x24, 0x1d],
    'EFI_ACPI_TABLE_GUID': [0x8868e871, 0xe4f1, 0x11d3, 0xbc, 0x22, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_ACPI_TABLE_GUID': [0xeb9d2d30, 0x2d88, 0x11d3, 0x9a, 0x16, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_ACPI_TABLE_PROTOCOL_GUID': [0xffe06bdd, 0x6107, 0x46a6, 0x7b, 0xb2, 0x5a, 0x9c, 0x7e, 0xc5, 0x27, 0x5c],
    'EFI_ACPI_TABLE_STORAGE_GUID': [0x7e374e25, 0x8e01, 0x4fee, 0x87, 0xf2, 0x39, 0xc, 0x23, 0xc6, 0x6, 0xcd],
    'EFI_ACPI_VARIABLE_COMPATIBILITY_GUID': [0xc020489e, 0x6db2, 0x4ef2, 0x9a, 0xa5, 0xca, 0x6, 0xfc, 0x11, 0xd3, 0x6a],
    'EFI_ALTERNATE_FV_BLOCK_GUID': [0xf496922d, 0x172f, 0x4bbc, 0xa1, 0xeb, 0xe, 0xeb, 0x94, 0x9c, 0x34, 0x86],
    'EFI_APRIORI_GUID': [0xfc510ee7, 0xffdc, 0x11d4, 0xbd, 0x41, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_ARP_PROTOCOL_GUID': [0xf4b427bb, 0xba21, 0x4f16, 0xbc, 0x4e, 0x43, 0xe4, 0x16, 0xab, 0x61, 0x9c],
    'EFI_ARP_SERVICE_BINDING_PROTOCOL_GUID': [0xf44c00ee, 0x1f2c, 0x4a00, 0xaa, 0x9, 0x1c, 0x9f, 0x3e, 0x8, 0x0, 0xa3],
    'EFI_ATA_PASS_THRU_PROTOCOL_GUID': [0x1d3de7f0, 0x807, 0x424f, 0xaa, 0x69, 0x11, 0xa5, 0x4e, 0x19, 0xa4, 0x6f],
    'EFI_AUTHENTICATED_VARIABLE_GUID': [0xaaf32c78, 0x947b, 0x439a, 0xa1, 0x80, 0x2e, 0x14, 0x4e, 0xc3, 0x77, 0x92],
    'EFI_AUTHENTICATION_CHAP_LOCAL_GUID': [0xc280c73e, 0x15ca, 0x11da, 0xb0, 0xca, 0x00, 0x10, 0x83, 0xff, 0xca, 0x4d],
    'EFI_AUTHENTICATION_CHAP_RADIUS_GUID': [0xd6062b50, 0x15ca, 0x11da, 0x92, 0x19, 0x00, 0x10, 0x83, 0xff, 0xca, 0x4d],
    'EFI_AUTHENTICATION_INFO_PROTOCOL_GUID': [0x7671d9d0, 0x53db, 0x4173, 0xaa, 0x69, 0x23, 0x27, 0xf2, 0x1f, 0x0b, 0xc7],
    'EFI_BDS_ARCH_PROTOCOL_GUID': [0x665E3FF6, 0x46CC, 0x11d4, 0x9A, 0x38, 0x00, 0x90, 0x27, 0x3F, 0xC1, 0x4D],
    'EFI_BIS_PROTOCOL_GUID': [0x0b64aab0, 0x5429, 0x11d4, 0x98, 0x16, 0x00, 0xa0, 0xc9, 0x1f, 0xad, 0xcf],
    'EFI_BLOCK_IO2_PROTOCOL_GUID': [0xa77b2472, 0xe282, 0x4e9f, 0xa2, 0x45, 0xc2, 0xc0, 0xe2, 0x7b, 0xbc, 0xc1],
    'EFI_BLOCK_IO_PROTOCOL_GUID': [0x964e5b21, 0x6459, 0x11d2, 0x8e, 0x39, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_BOOT_LOGO_PROTOCOL_GUID': [0xcdea2bd3, 0xfc25, 0x4c1c, 0xb9, 0x7c, 0xb3, 0x11, 0x86, 0x6, 0x49, 0x90],
    'EFI_BOOT_SCRIPT_EXECUTOR_CONTEXT_GUID': [0x79cb58c4, 0xac51, 0x442f, 0xaf, 0xd7, 0x98, 0xe4, 0x7d, 0x2e, 0x99, 0x8],
    'EFI_BOOT_SCRIPT_EXECUTOR_VARIABLE_GUID': [0x3079818c, 0x46d4, 0x4a73, 0xae, 0xf3, 0xe3, 0xe4, 0x6c, 0xf1, 0xee, 0xdb],
    'EFI_BOOT_SCRIPT_SAVE_PROTOCOL_GUID': [0x470e1529, 0xb79e, 0x4e32, 0xa0, 0xfe, 0x6a, 0x15, 0x6d, 0x29, 0xf9, 0xb2],
    'EFI_BUS_SPECIFIC_DRIVER_OVERRIDE_PROTOCOL_GUID': [0x3bc1b285, 0x8a15, 0x4a82, 0xaa, 0xbf, 0x4d, 0x7d, 0x13, 0xfb, 0x32, 0x65],
    'EFI_CACHE_SUBCLASS_GUID': [0x7f0013a7, 0xdc79, 0x4b22, 0x80, 0x99, 0x11, 0xf7, 0x5f, 0xdc, 0x82, 0x9d],
    'EFI_CAPSULE_ARCH_PROTOCOL_GUID': [0x5053697e, 0x2cbc, 0x4819, 0x90, 0xd9, 0x05, 0x80, 0xde, 0xee, 0x57, 0x54],
    'EFI_CAPSULE_ARCH_PROTOCOL_GUID': [0x5053697e, 0x2cbc, 0x4819, 0x90, 0xd9, 0x5, 0x80, 0xde, 0xee, 0x57, 0x54],
    'EFI_CAPSULE_GUID': [0x3B6686BD, 0x0D76, 0x4030, 0xB7, 0x0E, 0xB5, 0x51, 0x9E, 0x2F, 0xC5, 0xA0],
    'EFI_CAPSULE_INFO_GUID': [0x8B34EAC7, 0x2690, 0x460B, 0x8B, 0xA5, 0xD5, 0xCF, 0x32, 0x83, 0x17, 0x35],
    'EFI_CAPSULE_VENDOR_GUID': [0x711C703F, 0xC285, 0x4B10, 0xA3, 0xB0, 0x36, 0xEC, 0xBD, 0x3C, 0x8B, 0xE2],
    'EFI_CERT_RSA2048_GUID': [0x3c5766e8, 0x269c, 0x4e34, 0xaa, 0x14, 0xed, 0x77, 0x6e, 0x85, 0xb3, 0xb6],
    'EFI_CERT_RSA2048_SHA1_GUID': [0x67f8444f, 0x8743, 0x48f1, 0xa3, 0x28, 0x1e, 0xaa, 0xb8, 0x73, 0x60, 0x80],
    'EFI_CERT_RSA2048_SHA256_GUID': [0xe2b36190, 0x879b, 0x4a3d, 0xad, 0x8d, 0xf2, 0xe7, 0xbb, 0xa3, 0x27, 0x84],
    'EFI_CERT_SHA1_GUID': [0x826ca512, 0xcf10, 0x4ac9, 0xb1, 0x87, 0xbe, 0x1, 0x49, 0x66, 0x31, 0xbd],
    'EFI_CERT_SHA224_GUID': [0xb6e5233, 0xa65c, 0x44c9, 0x94, 0x7, 0xd9, 0xab, 0x83, 0xbf, 0xc8, 0xbd],
    'EFI_CERT_SHA256_GUID': [0xc1c41626, 0x504c, 0x4092, 0xac, 0xa9, 0x41, 0xf9, 0x36, 0x93, 0x43, 0x28],
    'EFI_CERT_SHA384_GUID': [0xff3e5307, 0x9fd0, 0x48c9, 0x85, 0xf1, 0x8a, 0xd5, 0x6c, 0x70, 0x1e, 0x1],
    'EFI_CERT_SHA512_GUID': [0x93e0fae, 0xa6c4, 0x4f50, 0x9f, 0x1b, 0xd4, 0x1e, 0x2b, 0x89, 0xc1, 0x9a],
    'EFI_CERT_TYPE_PKCS7_GUID': [0x4aafd29d, 0x68df, 0x49ee, 0x8a, 0xa9, 0x34, 0x7d, 0x37, 0x56, 0x65, 0xa7],
    'EFI_CERT_TYPE_RSA2048_SHA256_GUID': [0xa7717414, 0xc616, 0x4977, 0x94, 0x20, 0x84, 0x47, 0x12, 0xa7, 0x35, 0xbf],
    'EFI_CERT_X509_GUID': [0xa5c059a1, 0x94e4, 0x4aa7, 0x87, 0xb5, 0xab, 0x15, 0x5c, 0x2b, 0xf0, 0x72],
    'EFI_COMPATIBLE_MEMORY_TESTED_PROTOCOL_GUID': [0x64c475ef, 0x344b, 0x492c, 0x93, 0xad, 0xab, 0x9e, 0xb4, 0x39, 0x50, 0x4],
    'EFI_COMPONENT_NAME2_PROTOCOL_GUID': [0x6a7a5cff, 0xe8d9, 0x4f70, 0xba, 0xda, 0x75, 0xab, 0x30, 0x25, 0xce, 0x14],
    'EFI_COMPONENT_NAME_PROTOCOL_GUID': [0x107a772c, 0xd5e1, 0x11d4, 0x9a, 0x46, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_CONFIG_FILE_NAME_GUID': [0x98B8D59B, 0xE8BA, 0x48EE, 0x98, 0xDD, 0xC2, 0x95, 0x39, 0x2F, 0x1E, 0xDB],
    'EFI_CONSOLE_CONTROL_PROTOCOL_GUID': [0xf42f7782, 0x12e, 0x4c12, 0x99, 0x56, 0x49, 0xf9, 0x43, 0x4, 0xf7, 0x21],
    'EFI_CONSOLE_IN_DEVICE_GUID': [0xd3b36f2b, 0xd551, 0x11d4, 0x9a, 0x46, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_CONSOLE_OUT_DEVICE_GUID': [0xd3b36f2c, 0xd551, 0x11d4, 0x9a, 0x46, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_CPU_ARCH_PROTOCOL_GUID': [0x26baccb1, 0x6f42, 0x11d4, 0xbc, 0xe7, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_CPU_IO2_PROTOCOL_GUID': [0xad61f191, 0xae5f, 0x4c0e, 0xb9, 0xfa, 0xe8, 0x69, 0xd2, 0x88, 0xc6, 0x4f],
    'EFI_CPU_IO_PROTOCOL_GUID': [0xB0732526, 0x38C8, 0x4b40, 0x88, 0x77, 0x61, 0xC7, 0xB0, 0x6A, 0xAC, 0x45],
    'EFI_CRC32_GUIDED_SECTION_EXTRACTION_GUID': [0xFC1BCDB0, 0x7D31, 0x49aa, 0x93, 0x6A, 0xA4, 0x60, 0x0D, 0x9D, 0xD0, 0x83],
    'EFI_CRC32_GUIDED_SECTION_EXTRACTION_PROTOCOL_GUID': [0xFC1BCDB0, 0x7D31, 0x49aa, 0x93, 0x6A, 0xA4, 0x60, 0x0D, 0x9D, 0xD0, 0x83],
    'EFI_CUSTOMIZED_DECOMPRESS_PROTOCOL_GUID': [0x9a44198e, 0xa4a2, 0x44e6, 0x8a, 0x1f, 0x39, 0xbe, 0xfd, 0xac, 0x89, 0x6f],
    'EFI_DATA_HUB_PROTOCOL_GUID': [0xae80d021, 0x618e, 0x11d4, 0xbc, 0xd7, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_DATA_HUB_STATUS_CODE_RECORD_GUID': [0xd083e94c, 0x6560, 0x42e4, 0xb6, 0xd4, 0x2d, 0xf7, 0x5a, 0xdf, 0x6a, 0x2a],
    'EFI_DEBUGPORT_PROTOCOL_GUID': [0xEBA4E8D2, 0x3858, 0x41EC, 0xA2, 0x81, 0x26, 0x47, 0xBA, 0x96, 0x60, 0xD0],
    'EFI_DEBUG_AGENT_GUID': [0x865a5a9b, 0xb85d, 0x474c, 0x84, 0x55, 0x65, 0xd1, 0xbe, 0x84, 0x4b, 0xe2],
    'EFI_DEBUG_ASSERT_PROTOCOL_GUID': [0xbe499c92, 0x7d4b, 0x11d4, 0xbc, 0xee, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_DEBUG_IMAGE_INFO_TABLE_GUID': [0x49152e77, 0x1ada, 0x4764, 0xb7, 0xa2, 0x7a, 0xfe, 0xfe, 0xd9, 0x5e, 0x8b],
    'EFI_DEBUG_MASK_PROTOCOL_GUID': [0x4c8a2451, 0xc207, 0x405b, 0x96, 0x94, 0x99, 0xea, 0x13, 0x25, 0x13, 0x41],
    'EFI_DEBUG_SERIAL_IO_PROTOCOL_GUID': [0xe683dc4f, 0x9ed, 0x4f22, 0x86, 0x6b, 0x8e, 0x40, 0x46, 0x94, 0x7c, 0x6c],
    'EFI_DEBUG_SUPPORT_PERIODIC_CALLBACK_PROTOCOL_GUID': [0x9546e07c, 0x2cbb, 0x4c88, 0x98, 0x6c, 0xcd, 0x34, 0x10, 0x86, 0xf0, 0x44],
    'EFI_DEBUG_SUPPORT_PROTOCOL_GUID': [0x2755590C, 0x6F3C, 0x42FA, 0x9E, 0xA4, 0xA3, 0xBA, 0x54, 0x3C, 0xDA, 0x25],
    'EFI_DECOMPRESS_PROTOCOL_GUID': [0xd8117cfe, 0x94a6, 0x11d4, 0x9a, 0x3a, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_DEFAULT_BMP_LOGO_GUID': [0x7BB28B99, 0x61BB, 0x11d5, 0x9A, 0x5D, 0x00, 0x90, 0x27, 0x3F, 0xC1, 0x4D],
    'EFI_DEFERRED_IMAGE_LOAD_PROTOCOL_GUID': [0x15853d7c, 0x3ddf, 0x43e0, 0xa1, 0xcb, 0xeb, 0xf8, 0x5b, 0x8f, 0x87, 0x2c],
    'EFI_DEVICE_IO_PROTOCOL_GUID': [0xaf6ac311, 0x84c3, 0x11d2, 0x8e, 0x3c, 0x00, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_DEVICE_PATH_FROM_TEXT_PROTOCOL_GUID': [0x5c99a21, 0xc70f, 0x4ad2, 0x8a, 0x5f, 0x35, 0xdf, 0x33, 0x43, 0xf5, 0x1e],
    'EFI_DEVICE_PATH_PROTOCOL_GUID': [0x9576e91, 0x6d3f, 0x11d2, 0x8e, 0x39, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_DEVICE_PATH_TO_TEXT_PROTOCOL_GUID': [0x8b843e20, 0x8132, 0x4852, 0x90, 0xcc, 0x55, 0x1a, 0x4e, 0x4a, 0x7f, 0x1c],
    'EFI_DEVICE_PATH_UTILITIES_PROTOCOL_GUID': [0x379be4e, 0xd706, 0x437d, 0xb0, 0x37, 0xed, 0xb8, 0x2f, 0xb7, 0x72, 0xa4],
    'EFI_DHCP4_PROTOCOL_GUID': [0x8a219718, 0x4ef5, 0x4761, 0x91, 0xc8, 0xc0, 0xf0, 0x4b, 0xda, 0x9e, 0x56],
    'EFI_DHCP4_SERVICE_BINDING_PROTOCOL_GUID': [0x9d9a39d8, 0xbd42, 0x4a73, 0xa4, 0xd5, 0x8e, 0xe9, 0x4b, 0xe1, 0x13, 0x80],
    'EFI_DHCP6_PROTOCOL_GUID': [0x87c8bad7, 0x595, 0x4053, 0x82, 0x97, 0xde, 0xde, 0x39, 0x5f, 0x5d, 0x5b],
    'EFI_DHCP6_SERVICE_BINDING_PROTOCOL_GUID': [0x9fb9a8a1, 0x2f4a, 0x43a6, 0x88, 0x9c, 0xd0, 0xf7, 0xb6, 0xc4, 0x7a, 0xd5],
    'EFI_DISK_INFO_AHCI_INTERFACE_GUID': [0x9e498932, 0x4abc, 0x45af, 0xa3, 0x4d, 0x2, 0x47, 0x78, 0x7b, 0xe7, 0xc6],
    'EFI_DISK_INFO_IDE_INTERFACE_GUID': [0x5e948fe3, 0x26d3, 0x42b5, 0xaf, 0x17, 0x61, 0x2, 0x87, 0x18, 0x8d, 0xec],
    'EFI_DISK_INFO_PROTOCOL_GUID': [0xd432a67f, 0x14dc, 0x484b, 0xb3, 0xbb, 0x3f, 0x2, 0x91, 0x84, 0x93, 0x27],
    'EFI_DISK_INFO_SCSI_INTERFACE_GUID': [0x8f74baa, 0xea36, 0x41d9, 0x95, 0x21, 0x21, 0xa7, 0xf, 0x87, 0x80, 0xbc],
    'EFI_DISK_INFO_USB_INTERFACE_GUID': [0xcb871572, 0xc11a, 0x47b5, 0xb4, 0x92, 0x67, 0x5e, 0xaf, 0xa7, 0x77, 0x27],
    'EFI_DISK_IO_PROTOCOL_GUID': [0xce345171, 0xba0b, 0x11d2, 0x8e, 0x4f, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_DPC_PROTOCOL_GUID': [0x480f8ae9, 0xc46, 0x4aa9, 0xbc, 0x89, 0xdb, 0x9f, 0xba, 0x61, 0x98, 0x6],
    'EFI_DRIVER_BINDING_PROTOCOL_GUID': [0x18a031ab, 0xb443, 0x4d1a, 0xa5, 0xc0, 0xc, 0x9, 0x26, 0x1e, 0x9f, 0x71],
    'EFI_DRIVER_CONFIGURATION2_PROTOCOL_GUID': [0xbfd7dc1d, 0x24f1, 0x40d9, 0x82, 0xe7, 0x2e, 0x09, 0xbb, 0x6b, 0x4e, 0xbe],
    'EFI_DRIVER_CONFIGURATION_PROTOCOL_GUID': [0x107a772b, 0xd5e1, 0x11d4, 0x9a, 0x46, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_DRIVER_DIAGNOSTICS2_PROTOCOL_GUID': [0x4d330321, 0x025f, 0x4aac, 0x90, 0xd8, 0x5e, 0xd9, 0x0, 0x17, 0x3b, 0x63],
    'EFI_DRIVER_DIAGNOSTICS2_PROTOCOL_GUID': [0x4d330321, 0x025f, 0x4aac, 0x90, 0xd8, 0x5e, 0xd9, 0x00, 0x17, 0x3b, 0x63],
    'EFI_DRIVER_DIAGNOSTICS_PROTOCOL_GUID': [0x0784924f, 0xe296, 0x11d4, 0x9a, 0x49, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_DRIVER_FAMILY_OVERRIDE_PROTOCOL_GUID': [0xb1ee129e, 0xda36, 0x4181, 0x91, 0xf8, 0x4, 0xa4, 0x92, 0x37, 0x66, 0xa7],
    'EFI_DRIVER_HEALTH_PROTOCOL_GUID': [0x2a534210, 0x9280, 0x41d8, 0xae, 0x79, 0xca, 0xda, 0x1, 0xa2, 0xb1, 0x27],
    'EFI_DRIVER_SUPPORTED_EFI_VERSION_PROTOCOL_GUID': [0x5c198761, 0x16a8, 0x4e69, 0x97, 0x2c, 0x89, 0xd6, 0x79, 0x54, 0xf8, 0x1d],
    'EFI_DXE_IPL_PPI_GUID': [0xae8ce5d, 0xe448, 0x4437, 0xa8, 0xd7, 0xeb, 0xf5, 0xf1, 0x94, 0xf7, 0x31],
    'EFI_DXE_SERVICES_TABLE_GUID': [0x5ad34ba, 0x6f02, 0x4214, 0x95, 0x2e, 0x4d, 0xa0, 0x39, 0x8e, 0x2b, 0xb9],
    'EFI_DXE_SMM_READY_TO_LOCK_PROTOCOL_GUID': [0x60ff8964, 0xe906, 0x41d0, 0xaf, 0xed, 0xf2, 0x41, 0xe9, 0x74, 0xe0, 0x8e],
    'EFI_EAP_MANAGEMENT_PROTOCOL_GUID': [0xbb62e663, 0x625d, 0x40b2, 0xa0, 0x88, 0xbb, 0xe8, 0x36, 0x23, 0xa2, 0x45],
    'EFI_EAP_PROTOCOL_GUID': [0x5d9f96db, 0xe731, 0x4caa, 0xa0, 0xd, 0x72, 0xe1, 0x87, 0xcd, 0x77, 0x62],
    'EFI_EBC_INTERPRETER_PROTOCOL_GUID': [0x13AC6DD1, 0x73D0, 0x11D4, 0xB0, 0x6B, 0x00, 0xAA, 0x00, 0xBD, 0x6D, 0xE7],
    'EFI_EBC_SIMPLE_DEBUGGER_PROTOCOL_GUID': [0x2a72d11e, 0x7376, 0x40f6, 0x9c, 0x68, 0x23, 0xfa, 0x2f, 0xe3, 0x63, 0xf1],
    'EFI_EBC_VM_TEST_PROTOCOL_GUID': [0xAAEACCFD, 0xF27B, 0x4C17, 0xB6, 0x10, 0x75, 0xCA, 0x1F, 0x2D, 0xFB, 0x52],
    'EFI_EBC_VM_TEST_PROTOCOL_GUID': [0xAAEACCFD, 0xF27B, 0x4C17, 0xB6, 0x10, 0x75, 0xCA, 0x1F, 0x2D, 0xFB, 0x52],
    'EFI_EDID_ACTIVE_PROTOCOL_GUID': [0xbd8c1056, 0x9f36, 0x44ec, 0x92, 0xa8, 0xa6, 0x33, 0x7f, 0x81, 0x79, 0x86],
    'EFI_EDID_DISCOVERED_PROTOCOL_GUID': [0x1c0c34f6, 0xd380, 0x41fa, 0xa0, 0x49, 0x8a, 0xd0, 0x6c, 0x1a, 0x66, 0xaa],
    'EFI_EDID_DISCOVERED_PROTOCOL_GUID': [0x1c0c34f6, 0xd380, 0x41fa, 0xa0, 0x49, 0x8a, 0xd0, 0x6c, 0x1a, 0x66, 0xaa],
    'EFI_EDID_OVERRIDE_PROTOCOL_GUID': [0x48ecb431, 0xfb72, 0x45c0, 0xa9, 0x22, 0xf4, 0x58, 0xfe, 0x4, 0xb, 0xd5],
    'EFI_EMU_PHYSICAL_DISK_GUID': [0xf2ba331a, 0x8985, 0x11db, 0xa4, 0x06, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_EMU_SYSTEM_CONFIG_GUID': [0x9C4FB516, 0x3A1E, 0xD847, 0xA1, 0xA1, 0x70, 0x58, 0xB6, 0x98, 0x67, 0x32],
    'EFI_EMU_VIRTUAL_DISK_GUID': [0xf2ba331a, 0x8985, 0x11db, 0xa4, 0x06, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_END_OF_DXE_EVENT_GROUP_GUID': [0x2ce967a, 0xdd7e, 0x4ffc, 0x9e, 0xe7, 0x81, 0x0c, 0xf0, 0x47, 0x8, 0x80],
    'EFI_END_OF_DXE_EVENT_GROUP_GUID': [0x2ce967a, 0xdd7e, 0x4ffc, 0x9e, 0xe7, 0x81, 0xc, 0xf0, 0x47, 0x8, 0x80],
    'EFI_ERROR_SECTION_DIRECTED_IO_DMAR_GUID': [0x71761d37, 0x32b2, 0x45cd, 0xa7, 0xd0, 0xb0, 0xfe, 0xdd, 0x93, 0xe8, 0xcf],
    'EFI_ERROR_SECTION_DMAR_GENERIC_GUID': [0x5b51fef7, 0xc79d, 0x4434, 0x8f, 0x1b, 0xaa, 0x62, 0xde, 0x3e, 0x2c, 0x64],
    'EFI_ERROR_SECTION_FW_ERROR_RECORD_GUID': [0x81212a96, 0x09ed, 0x4996, 0x94, 0x71, 0x8d, 0x72, 0x9c, 0x8e, 0x69, 0xed],
    'EFI_ERROR_SECTION_IOMMU_DMAR_GUID': [0x036f84e1, 0x7f37, 0x428c, 0xa7, 0x9e, 0x57, 0x5f, 0xdf, 0xaa, 0x84, 0xec],
    'EFI_ERROR_SECTION_PCIE_GUID': [0xd995e954, 0xbbc1, 0x430f, 0xad, 0x91, 0xb4, 0x4d, 0xcb, 0x3c, 0x6f, 0x35],
    'EFI_ERROR_SECTION_PCI_DEVICE_GUID': [0xeb5e4685, 0xca66, 0x4769, 0xb6, 0xa2, 0x26, 0x06, 0x8b, 0x00, 0x13, 0x26],
    'EFI_ERROR_SECTION_PCI_PCIX_BUS_GUID': [0xc5753963, 0x3b84, 0x4095, 0xbf, 0x78, 0xed, 0xda, 0xd3, 0xf9, 0xc9, 0xdd],
    'EFI_ERROR_SECTION_PLATFORM_MEMORY_GUID': [0xa5bc1114, 0x6f64, 0x4ede, 0xb8, 0x63, 0x3e, 0x83, 0xed, 0x7c, 0x83, 0xb1],
    'EFI_ERROR_SECTION_PROCESSOR_GENERIC_GUID': [0x9876ccad, 0x47b4, 0x4bdb, 0xb6, 0x5e, 0x16, 0xf1, 0x93, 0xc4, 0xf3, 0xdb],
    'EFI_ERROR_SECTION_PROCESSOR_SPECIFIC_GUID': [0xdc3ea0b0, 0xa144, 0x4797, 0xb9, 0x5b, 0x53, 0xfa, 0x24, 0x2b, 0x6e, 0x1d],
    'EFI_EVENT_GROUP_DXE_DISPATCH_GUID': [0x7081e22f, 0xcac6, 0x4053, 0x94, 0x68, 0x67, 0x57, 0x82, 0xcf, 0x88, 0xe5],
    'EFI_EVENT_LEGACY_BOOT_GUID': [0x2a571201, 0x4966, 0x47f6, 0x8b, 0x86, 0xf3, 0x1e, 0x41, 0xf3, 0x2f, 0x10],
    'EFI_EVENT_NOTIFICATION_TYEP_BOOT_GUID': [0x3D61A466, 0xAB40, 0x409a, 0xA6, 0x98, 0xF3, 0x62, 0xD4, 0x64, 0xB3, 0x8F],
    'EFI_EVENT_NOTIFICATION_TYEP_CMC_GUID': [0x2DCE8BB1, 0xBDD7, 0x450e, 0xB9, 0xAD, 0x9C, 0xF4, 0xEB, 0xD4, 0xF8, 0x90],
    'EFI_EVENT_NOTIFICATION_TYEP_CPE_GUID': [0x4E292F96, 0xD843, 0x4a55, 0xA8, 0xC2, 0xD4, 0x81, 0xF2, 0x7E, 0xBE, 0xEE],
    'EFI_EVENT_NOTIFICATION_TYEP_DMAR_GUID': [0x667DD791, 0xC6B3, 0x4c27, 0x8A, 0x6B, 0x0F, 0x8E, 0x72, 0x2D, 0xEB, 0x41],
    'EFI_EVENT_NOTIFICATION_TYEP_INIT_GUID': [0xCC5263E8, 0x9308, 0x454a, 0x89, 0xD0, 0x34, 0x0B, 0xD3, 0x9B, 0xC9, 0x8E],
    'EFI_EVENT_NOTIFICATION_TYEP_MCE_GUID': [0xE8F56FFE, 0x919C, 0x4cc5, 0xBA, 0x88, 0x65, 0xAB, 0xE1, 0x49, 0x13, 0xBB],
    'EFI_EVENT_NOTIFICATION_TYEP_NMI_GUID': [0x5BAD89FF, 0xB7E6, 0x42c9, 0x81, 0x4A, 0xCF, 0x24, 0x85, 0xD6, 0xE9, 0x8A],
    'EFI_EVENT_NOTIFICATION_TYEP_PCIE_GUID': [0xCF93C01F, 0x1A16, 0x4dfc, 0xB8, 0xBC, 0x9C, 0x4D, 0xAF, 0x67, 0xC1, 0x04],
    'EFI_EXTENDED_SAL_BASE_IO_SERVICES_PROTOCOL_GUID': [0x5aea42b5, 0x31e1, 0x4515, 0xbc, 0x31, 0xb8, 0xd5, 0x25, 0x75, 0x65, 0xa6],
    'EFI_EXTENDED_SAL_BASE_SERVICES_PROTOCOL_GUID': [0xd9e9fa06, 0x0fe0, 0x41c3, 0x96, 0xfb, 0x83, 0x42, 0x5a, 0x33, 0x94, 0xf8],
    'EFI_EXTENDED_SAL_CACHE_SERVICES_PROTOCOL_GUID': [0xedc9494, 0x2743, 0x4ba5, 0x88, 0x18, 0x0a, 0xef, 0x52, 0x13, 0xf1, 0x88],
    'EFI_EXTENDED_SAL_ELOG_SERVICES_PROTOCOL_GUID': [0xd5e4ee5f, 0x3e0a, 0x453c, 0xa7, 0x25, 0xb6, 0x92, 0xbb, 0x6, 0x36, 0x5a],
    'EFI_EXTENDED_SAL_FV_BLOCK_SERVICES_PROTOCOL_GUID': [0xa2271df1, 0xbcbb, 0x4f1d, 0x98, 0xa9, 0x06, 0xbc, 0x17, 0x2f, 0x07, 0x1a],
    'EFI_EXTENDED_SAL_LOCK_SERVICES_PROTOCOL_GUID': [0x76b75c23, 0xfe4f, 0x4e17, 0xa2, 0xad, 0x1a, 0x65, 0x3d, 0xbb, 0x49, 0x4a],
    'EFI_EXTENDED_SAL_MCA_LOG_SERVICES_PROTOCOL_GUID': [0xcb3fd86e, 0x38a3, 0x4c03, 0x9a, 0x5c, 0x90, 0xcf, 0xa3, 0xa2, 0xab, 0x7a],
    'EFI_EXTENDED_SAL_MCA_SERVICES_PROTOCOL_GUID': [0x2a591128, 0x6cc7, 0x42b1, 0x8a, 0xf0, 0x58, 0x93, 0x3b, 0x68, 0x2d, 0xbb],
    'EFI_EXTENDED_SAL_MP_SERVICES_PROTOCOL_GUID': [0x697d81a2, 0xcf18, 0x4dc0, 0x9e, 0x0d, 0x06, 0x11, 0x3b, 0x61, 0x8a, 0x3f],
    'EFI_EXTENDED_SAL_MTC_SERVICES_PROTOCOL_GUID': [0x899afd18, 0x75e8, 0x408b, 0xa4, 0x1a, 0x6e, 0x2e, 0x7e, 0xcd, 0xf4, 0x54],
    'EFI_EXTENDED_SAL_PAL_SERVICES_PROTOCOL_GUID': [0xe1cd9d21, 0x0fc2, 0x438d, 0x97, 0x03, 0x04, 0xe6, 0x6d, 0x96, 0x1e, 0x57],
    'EFI_EXTENDED_SAL_PCI_SERVICES_PROTOCOL_GUID': [0xa46b1a31, 0xad66, 0x4905, 0x92, 0xf6, 0x2b, 0x46, 0x59, 0xdc, 0x30, 0x63],
    'EFI_EXTENDED_SAL_RESET_SERVICES_PROTOCOL_GUID': [0x7d019990, 0x8ce1, 0x46f5, 0xa7, 0x76, 0x3c, 0x51, 0x98, 0x67, 0x6a, 0xa0],
    'EFI_EXTENDED_SAL_RTC_SERVICES_PROTOCOL_GUID': [0x7e97a470, 0xefdb, 0x4d02, 0x8f, 0xce, 0x61, 0x90, 0xd2, 0x7b, 0xa2, 0x96],
    'EFI_EXTENDED_SAL_SENSOR_SERVICES_PROTOCOL_GUID': [0x4a153b6e, 0x85a1, 0x4982, 0x98, 0xf4, 0x6a, 0x8c, 0xfc, 0xa4, 0xab, 0xa1],
    'EFI_EXTENDED_SAL_SM_COM_LAYER_SERVICES_PROTOCOL_GUID': [0x4356799, 0x81b7, 0x4e08, 0xa3, 0x8d, 0xd9, 0x78, 0xfa, 0x47, 0xba, 0x42],
    'EFI_EXTENDED_SAL_SST_GUID': [0x38802700, 0x868a, 0x4b4e, 0x81, 0xd4, 0x4f, 0x1b, 0xdc, 0xcf, 0xb4, 0x6f],
    'EFI_EXTENDED_SAL_STALL_SERVICES_PROTOCOL_GUID': [0x53a58d06, 0xac27, 0x4d8c, 0xb5, 0xe9, 0xf0, 0x8a, 0x80, 0x65, 0x41, 0x70],
    'EFI_EXTENDED_SAL_STATUS_CODE_SERVICES_PROTOCOL_GUID': [0xdbd91d, 0x55e9, 0x420f, 0x96, 0x39, 0x5e, 0x9f, 0x84, 0x37, 0xb4, 0x4f],
    'EFI_EXTENDED_SAL_VARIABLE_SERVICES_PROTOCOL_GUID': [0x4ecb6c53, 0xc641, 0x4370, 0x8c, 0xb2, 0x3b, 0x0e, 0x49, 0x6e, 0x83, 0x78],
    'EFI_EXTENDED_SAL_VIRTUAL_SERVICES_PROTOCOL_GUID': [0xc1a74056, 0x260e, 0x4871, 0xa0, 0x31, 0xe6, 0x45, 0xa6, 0x5b, 0x6e, 0x11],
    'EFI_EXT_SCSI_PASS_THRU_PROTOCOL_GUID': [0x143b7632, 0xb81b, 0x4cb7, 0xab, 0xd3, 0xb6, 0x25, 0xa5, 0xb9, 0xbf, 0xfe],
    'EFI_FAULT_TOLERANT_WRITE_PROTOCOL_GUID': [0x3ebd9e82, 0x2c78, 0x4de6, 0x97, 0x86, 0x8d, 0x4b, 0xfc, 0xb7, 0xc8, 0x81],
    'EFI_FFS_VOLUME_TOP_FILE_GUID': [0x1BA0062E, 0xC779, 0x4582, 0x85, 0x66, 0x33, 0x6A, 0xE8, 0xF7, 0x8F, 0x09],
    'EFI_FILE_SYSTEM_INFO_ID_GUID': [0x9576e93, 0x6d3f, 0x11d2, 0x8e, 0x39, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_FILE_SYSTEM_VOLUME_LABEL_INFO_ID_GUID': [0xDB47D7D3, 0xFE81, 0x11d3, 0x9A, 0x35, 0x00, 0x90, 0x27, 0x3F, 0xC1, 0x4D],
    'EFI_FIND_FV_PPI_GUID': [0x36164812, 0xa023, 0x44e5, 0xbd, 0x85, 0x5, 0xbf, 0x3c, 0x77, 0x0, 0xaa],
    'EFI_FIRMWARE_CONTENTS_SIGNED_GUID': [0xf9d89e8, 0x9259, 0x4f76, 0xa5, 0xaf, 0xc, 0x89, 0xe3, 0x40, 0x23, 0xdf],
    'EFI_FIRMWARE_FILE_SYSTEM2_GUID': [0x8c8ce578, 0x8a3d, 0x4f1c, 0x99, 0x35, 0x89, 0x61, 0x85, 0xc3, 0x2d, 0xd3],
    'EFI_FIRMWARE_FILE_SYSTEM3_GUID': [0x5473c07a, 0x3dcb, 0x4dca, 0xbd, 0x6f, 0x1e, 0x96, 0x89, 0xe7, 0x34, 0x9a],
    'EFI_FIRMWARE_FILE_SYSTEM_GUID': [0x7A9354D9, 0x0468, 0x444a, 0x81, 0xCE, 0x0B, 0xF6, 0x17, 0xD8, 0x90, 0xDF],
    'EFI_FIRMWARE_MANAGEMENT_PROTOCOL_GUID': [0x86c77a67, 0xb97, 0x4633, 0xa1, 0x87, 0x49, 0x10, 0x4d, 0x6, 0x85, 0xc7],
    'EFI_FIRMWARE_PERFORMANCE_GUID': [0xc095791a, 0x3001, 0x47b2, 0x80, 0xc9, 0xea, 0xc7, 0x31, 0x9f, 0x2f, 0xa4],
    'EFI_FIRMWARE_VOLUME2_PROTOCOL_GUID': [0x220e73b6, 0x6bdb, 0x4413, 0x84, 0x5, 0xb9, 0x74, 0xb1, 0x8, 0x61, 0x9a],
    'EFI_FIRMWARE_VOLUME_BLOCK2_PROTOCOL_GUID': [0x8f644fa9, 0xe850, 0x4db1, 0x9c, 0xe2, 0xb, 0x44, 0x69, 0x8e, 0x8d, 0xa4],
    'EFI_FIRMWARE_VOLUME_BLOCK_PROTOCOL_GUID': [0x8f644fa9, 0xe850, 0x4db1, 0x9c, 0xe2, 0xb, 0x44, 0x69, 0x8e, 0x8d, 0xa4],
    'EFI_FIRMWARE_VOLUME_DISPATCH_PROTOCOL_GUID': [0x7aa35a69, 0x506c, 0x444f, 0xa7, 0xaf, 0x69, 0x4b, 0xf5, 0x6f, 0x71, 0xc8],
    'EFI_FIRMWARE_VOLUME_PROTOCOL_GUID': [0x389F751F, 0x1838, 0x4388, 0x83, 0x90, 0xCD, 0x81, 0x54, 0xBD, 0x27, 0xF8],
    'EFI_FORM_BROWSER2_PROTOCOL_GUID': [0xb9d4c360, 0xbcfb, 0x4f9b, 0x92, 0x98, 0x53, 0xc1, 0x36, 0x98, 0x22, 0x58],
    'EFI_FORM_BROWSER_COMPATIBILITY_PROTOCOL_GUID': [0xfb7c852, 0xadca, 0x4853, 0x8d, 0xf, 0xfb, 0xa7, 0x1b, 0x1c, 0xe1, 0x1a],
    'EFI_FORM_BROWSER_PROTOCOL_GUID': [0xe5a1333e, 0xe1b4, 0x4d55, 0xce, 0xeb, 0x35, 0xc3, 0xef, 0x13, 0x34, 0x43],
    'EFI_FORM_BROWSER_PROTOCOL_GUID': [0xfb7c852, 0xadca, 0x4853, 0x8d, 0xf, 0xfb, 0xa7, 0x1b, 0x1c, 0xe1, 0x1a],
    'EFI_FORM_CALLBACK_PROTOCOL_GUID': [0xf3e4543d, 0xcf35, 0x6cef, 0x35, 0xc4, 0x4f, 0xe6, 0x34, 0x4d, 0xfc, 0x54],
    'EFI_FRAMEWORK_DEVICE_PATH_GUID': [0xb7084e63, 0x46b7, 0x4d1a, 0x86, 0x77, 0xe3, 0x0b, 0x53, 0xdb, 0xf0, 0x50],
    'EFI_FTP4_PROTOCOL_GUID': [0xeb338826, 0x681b, 0x4295, 0xb3, 0x56, 0x2b, 0x36, 0x4c, 0x75, 0x7b, 0x9],
    'EFI_FTP4_SERVICE_BINDING_PROTOCOL_GUID': [0xfaaecb1, 0x226e, 0x4782, 0xaa, 0xce, 0x7d, 0xb9, 0xbc, 0xbf, 0x4d, 0xaf],
    'EFI_FTW_LITE_PROTOCOL_GUID': [0x3f557189, 0x8dae, 0x45ae, 0xa0, 0xb3, 0x2b, 0x99, 0xca, 0x7a, 0xa7, 0xa0],
    'EFI_FVB_EXTENSION_PROTOCOL_GUID': [0x53a4c71b, 0xb581, 0x4170, 0x91, 0xb3, 0x8d, 0xb8, 0x7a, 0x4b, 0x5c, 0x46],
    'EFI_GENERIC_MEMORY_TEST_PROTOCOL_GUID': [0x309de7f1, 0x7f5e, 0x4ace, 0xb4, 0x9c, 0x53, 0x1b, 0xe5, 0xaa, 0x95, 0xef],
    'EFI_GENERIC_VARIABLE_GUID': [0x59d1c24f, 0x50f1, 0x401a, 0xb1, 0x01, 0xf3, 0x3e, 0x0d, 0xae, 0xd4, 0x43],
    'EFI_GLOBAL_VARIABLE_GUID': [0x8BE4DF61, 0x93CA, 0x11d2, 0xAA, 0x0D, 0x00, 0xE0, 0x98, 0x03, 0x2B, 0x8C],
    'EFI_GRAPHICS_OUTPUT_PROTOCOL_GUID': [0x9042a9de, 0x23dc, 0x4a38, 0x96, 0xfb, 0x7a, 0xde, 0xd0, 0x80, 0x51, 0x6a],
    'EFI_HARDWARE_ERROR_VARIABLE_GUID': [0x414E6BDD, 0xE47B, 0x47cc, 0xB2, 0x44, 0xBB, 0x61, 0x02, 0x0C, 0xF5, 0x16],
    'EFI_HASH_ALGORITHM_SHA1_GUID': [0x2ae9d80f, 0x3fb2, 0x4095, 0xb7, 0xb1, 0xe9, 0x31, 0x57, 0xb9, 0x46, 0xb6],
    'EFI_HASH_ALGORITHM_SHA1_NOPAD_GUID': [0x24c5dc2f, 0x53e2, 0x40ca, 0x9e, 0xd6, 0xa5, 0xd9, 0xa4, 0x9f, 0x46, 0x3b],
    'EFI_HASH_ALGORITHM_SHA224_GUID': [0x8df01a06, 0x9bd5, 0x4bf7, 0xb0, 0x21, 0xdb, 0x4f, 0xd9, 0xcc, 0xf4, 0x5b],
    'EFI_HASH_ALGORITHM_SHA256_GUID': [0x51aa59de, 0xfdf2, 0x4ea3, 0xbc, 0x63, 0x87, 0x5f, 0xb7, 0x84, 0x2e, 0xe9],
    'EFI_HASH_ALGORITHM_SHA256_NOPAD_GUID': [0x8628752a, 0x6cb7, 0x4814, 0x96, 0xfc, 0x24, 0xa8, 0x15, 0xac, 0x22, 0x26],
    'EFI_HASH_ALGORITHM_SHA384_GUID': [0xefa96432, 0xde33, 0x4dd2, 0xae, 0xe6, 0x32, 0x8c, 0x33, 0xdf, 0x77, 0x7a],
    'EFI_HASH_ALGORITHM_SHA512_GUID': [0xcaa4381e, 0x750c, 0x4770, 0xb8, 0x70, 0x7a, 0x23, 0xb4, 0xe4, 0x21, 0x30],
    'EFI_HASH_ALGORTIHM_MD5_GUID': [0xaf7c79c, 0x65b5, 0x4319, 0xb0, 0xae, 0x44, 0xec, 0x48, 0x4e, 0x4a, 0xd7],
    'EFI_HASH_PROTOCOL_GUID': [0xc5184932, 0xdba5, 0x46db, 0xa5, 0xba, 0xcc, 0x0b, 0xda, 0x9c, 0x14, 0x35],
    'EFI_HASH_SERVICE_BINDING_PROTOCOL_GUID': [0x42881c98, 0xa4f3, 0x44b0, 0xa3, 0x9d, 0xdf, 0xa1, 0x86, 0x67, 0xd8, 0xcd],
    'EFI_HII_COMPATIBILITY_PROTOCOL_GUID': [0x5542cce1, 0xdf5c, 0x4d1b, 0xab, 0xca, 0x36, 0x4f, 0x77, 0xd3, 0x99, 0xfb],
    'EFI_HII_CONFIG_ACCESS_PROTOCOL_GUID': [0x330d4706, 0xf2a0, 0x4e4f, 0xa3, 0x69, 0xb6, 0x6f, 0xa8, 0xd5, 0x43, 0x85],
    'EFI_HII_CONFIG_ROUTING_PROTOCOL_GUID': [0x587e72d7, 0xcc50, 0x4f79, 0x82, 0x09, 0xca, 0x29, 0x1f, 0xc1, 0xa1, 0x0f],
    'EFI_HII_DATABASE_PROTOCOL_GUID': [0xef9fc172, 0xa1b2, 0x4693, 0xb3, 0x27, 0x6d, 0x32, 0xfc, 0x41, 0x60, 0x42],
    'EFI_HII_DRIVER_HEALTH_FORMSET_GUID': [0xf22fc20c, 0x8cf4, 0x45eb, 0x8e, 0x6, 0xad, 0x4e, 0x50, 0xb9, 0x5d, 0xd3],
    'EFI_HII_FONT_PROTOCOL_GUID': [0xe9ca4775, 0x8657, 0x47fc, 0x97, 0xe7, 0x7e, 0xd6, 0x5a, 0x8, 0x43, 0x24],
    'EFI_HII_FRONT_PAGE_CLASS_GUID': [0x94d411b7, 0x7669, 0x45c3, 0xba, 0x3b, 0xf3, 0xa5, 0x8a, 0x71, 0x56, 0x81],
    'EFI_HII_IMAGE_PROTOCOL_GUID': [0x31a6406a, 0x6bdf, 0x4e46, 0xb2, 0xa2, 0xeb, 0xaa, 0x89, 0xc4, 0x9, 0x20],
    'EFI_HII_PACKAGE_LIST_PROTOCOL_GUID': [0x6a1ee763, 0xd47a, 0x43b4, 0xaa, 0xbe, 0xef, 0x1d, 0xe2, 0xab, 0x56, 0xfc],
    'EFI_HII_PLATFORM_SETUP_FORMSET_GUID': [0x93039971, 0x8545, 0x4b04, 0xb4, 0x5e, 0x32, 0xeb, 0x83, 0x26, 0x4, 0xe],
    'EFI_HII_PROTOCOL_GUID': [0x5542cce1, 0xdf5c, 0x4d1b, 0xab, 0xca, 0x36, 0x4f, 0x77, 0xd3, 0x99, 0xfb],
    'EFI_HII_PROTOCOL_GUID': [0xd7ad636e, 0xb997, 0x459b, 0xbf, 0x3f, 0x88, 0x46, 0x89, 0x79, 0x80, 0xe1],
    'EFI_HII_SET_KEYBOARD_LAYOUT_EVENT_GUID': [0x14982a4f, 0xb0ed, 0x45b8, 0xa8, 0x11, 0x5a, 0x7a, 0x9b, 0xc2, 0x32, 0xdf],
    'EFI_HII_STANDARD_FORM_GUID': [0x3bd2f4ec, 0xe524, 0x46e4, 0xa9, 0xd8, 0x51, 0x1, 0x17, 0x42, 0x55, 0x62],
    'EFI_HII_STRING_PROTOCOL_GUID': [0xfd96974, 0x23aa, 0x4cdc, 0xb9, 0xcb, 0x98, 0xd1, 0x77, 0x50, 0x32, 0x2a],
    'EFI_HII_USER_CREDENTIAL_FORMSET_GUID': [0x337f4407, 0x5aee, 0x4b83, 0xb2, 0xa7, 0x4e, 0xad, 0xca, 0x30, 0x88, 0xcd],
    'EFI_HOB_LIST_GUID': [0x7739f24c, 0x93d7, 0x11d4, 0x9a, 0x3a, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_HOB_MEMORY_ALLOC_BSP_STORE_GUID': [0x564b33cd, 0xc92a, 0x4593, 0x90, 0xbf, 0x24, 0x73, 0xe4, 0x3c, 0x63, 0x22],
    'EFI_HOB_MEMORY_ALLOC_STACK_GUID': [0x4ed4bf27, 0x4092, 0x42e9, 0x80, 0x7d, 0x52, 0x7b, 0x1d, 0x0, 0xc9, 0xbd],
    'EFI_IA32_X64_ERROR_TYPE_BUS_CHECK_GUID': [0x1CF3F8B3, 0xC5B1, 0x49a2, 0xAA, 0x59, 0x5E, 0xEF, 0x92, 0xFF, 0xA6, 0x3C],
    'EFI_IA32_X64_ERROR_TYPE_CACHE_CHECK_GUID': [0xA55701F5, 0xE3EF, 0x43de, 0xAC, 0x72, 0x24, 0x9B, 0x57, 0x3F, 0xAD, 0x2C],
    'EFI_IA32_X64_ERROR_TYPE_MS_CHECK_GUID': [0x48AB7F57, 0xDC34, 0x4f6c, 0xA7, 0xD3, 0xB0, 0xB5, 0xB0, 0xA7, 0x43, 0x14],
    'EFI_IA32_X64_ERROR_TYPE_TLB_CHECK_GUID': [0xFC06B535, 0x5E1F, 0x4562, 0x9F, 0x25, 0x0A, 0x3B, 0x9A, 0xDB, 0x63, 0xC3],
    'EFI_IDE_CONTROLLER_INIT_PROTOCOL_GUID': [0xa1e37052, 0x80d9, 0x4e65, 0xa3, 0x17, 0x3e, 0x9a, 0x55, 0xc4, 0x3e, 0xc9],
    'EFI_IFR_FRAMEWORK_GUID': [0x31ca5d1a, 0xd511, 0x4931, 0xb7, 0x82, 0xae, 0x6b, 0x2b, 0x17, 0x8c, 0xd7],
    'EFI_IFR_REFRESH_ID_OP_GUID': [0xF5E655D9, 0x02A6, 0x46f2, 0x9E, 0x76, 0xB8, 0xBE, 0x8E, 0x60, 0xAB, 0x22],
    'EFI_IFR_TIANO_GUID': [0xf0b1735, 0x87a0, 0x4193, 0xb2, 0x66, 0x53, 0x8c, 0x38, 0xaf, 0x48, 0xce],
    'EFI_IMAGE_SECURITY_DATABASE_GUID': [0xd719b2cb, 0x3d3a, 0x4596, 0xa3, 0xbc, 0xda, 0xd0, 0xe, 0x67, 0x65, 0x6f],
    'EFI_INCOMPATIBLE_PCI_DEVICE_SUPPORT_PROTOCOL_GUID': [0xeb23f55a, 0x7863, 0x4ac2, 0x8d, 0x3d, 0x95, 0x65, 0x35, 0xde, 0x03, 0x75],
    'EFI_IOBASE_HOB_GUID': [0xd4a28a3e, 0xdcf2, 0x43cf, 0xa2, 0xb7, 0xf3, 0x57, 0x2a, 0x7c, 0xab, 0x9],
    'EFI_IP4_CONFIG_PROTOCOL_GUID': [0x3b95aa31, 0x3793, 0x434b, 0x86, 0x67, 0xc8, 0x07, 0x08, 0x92, 0xe0, 0x5e],
    'EFI_IP4_PROTOCOL_GUID': [0x41d94cd2, 0x35b6, 0x455a, 0x82, 0x58, 0xd4, 0xe5, 0x13, 0x34, 0xaa, 0xdd],
    'EFI_IP4_SERVICE_BINDING_PROTOCOL_GUID': [0xc51711e7, 0xb4bf, 0x404a, 0xbf, 0xb8, 0x0a, 0x04, 0x8e, 0xf1, 0xff, 0xe4],
    'EFI_IP6_CONFIG_PROTOCOL_GUID': [0x937fe521, 0x95ae, 0x4d1a, 0x89, 0x29, 0x48, 0xbc, 0xd9, 0x0a, 0xd3, 0x1a],
    'EFI_IP6_PROTOCOL_GUID': [0x2c8759d5, 0x5c2d, 0x66ef, 0x92, 0x5f, 0xb6, 0x6c, 0x10, 0x19, 0x57, 0xe2],
    'EFI_IP6_SERVICE_BINDING_PROTOCOL_GUID': [0xec835dd3, 0xfe0f, 0x617b, 0xa6, 0x21, 0xb3, 0x50, 0xc3, 0xe1, 0x33, 0x88],
    'EFI_IPSEC2_PROTOCOL_GUID': [0xa3979e64, 0xace8, 0x4ddc, 0xbc, 0x7, 0x4d, 0x66, 0xb8, 0xfd, 0x9, 0x77],
    'EFI_IPSEC_CONFIG_PROTOCOL_GUID': [0xce5e5929, 0xc7a3, 0x4602, 0xad, 0x9e, 0xc9, 0xda, 0xf9, 0x4e, 0xbf, 0xcf],
    'EFI_IPSEC_PROTOCOL_GUID': [0xdfb386f7, 0xe100, 0x43ad, 0x9c, 0x9a, 0xed, 0x90, 0xd0, 0x8a, 0x5e, 0x12],
    'EFI_ISA_ACPI_PROTOCOL_GUID': [0x64a892dc, 0x5561, 0x4536, 0x92, 0xc7, 0x79, 0x9b, 0xfc, 0x18, 0x33, 0x55],
    'EFI_ISA_IO_PROTOCOL_GUID': [0x7ee2bd44, 0x3da0, 0x11d4, 0x9a, 0x38, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_ISCSI_INITIATOR_NAME_PROTOCOL_GUID': [0x59324945, 0xec44, 0x4c0d, 0xb1, 0xcd, 0x9d, 0xb1, 0x39, 0xdf, 0x7, 0xc],
    'EFI_KMS_FORMAT_AESCBC_128_GUID': [0xa0e8ee6a, 0x0e92, 0x44d4, 0x86, 0x1b, 0x0e, 0xaa, 0x4a, 0xca, 0x44, 0xa2],
    'EFI_KMS_FORMAT_AESCBC_256_GUID': [0xd7e69789, 0x1f68, 0x45e8, 0x96, 0xef, 0x3b, 0x64, 0x07, 0xa5, 0xb2, 0xdc],
    'EFI_KMS_FORMAT_AESXTS_128_GUID': [0x4776e33f, 0xdb47, 0x479a, 0xa2, 0x5f, 0xa1, 0xcd, 0x0a, 0xfa, 0xb3, 0x8b],
    'EFI_KMS_FORMAT_AESXTS_256_GUID': [0xdc7e8613, 0xc4bb, 0x4db0, 0x84, 0x62, 0x13, 0x51, 0x13, 0x57, 0xab, 0xe2],
    'EFI_KMS_FORMAT_GENERIC_1024_GUID': [0x43be0b44, 0x874b, 0x4ead, 0xb0, 0x9c, 0x24, 0x1a, 0x4f, 0xbd, 0x7e, 0xb3],
    'EFI_KMS_FORMAT_GENERIC_128_GUID': [0xec8a3d69, 0x6ddf, 0x4108, 0x94, 0x76, 0x73, 0x37, 0xfc, 0x52, 0x21, 0x36],
    'EFI_KMS_FORMAT_GENERIC_160_GUID': [0xa3b3e6f8, 0xefca, 0x4bc1, 0x88, 0xfb, 0xcb, 0x87, 0x33, 0x9b, 0x25, 0x79],
    'EFI_KMS_FORMAT_GENERIC_2048_GUID': [0x40093f23, 0x630c, 0x4626, 0x9c, 0x48, 0x40, 0x37, 0x3b, 0x19, 0xcb, 0xbe],
    'EFI_KMS_FORMAT_GENERIC_256_GUID': [0x70f64793, 0xc323, 0x4261, 0xac, 0x2c, 0xd8, 0x76, 0xf2, 0x7c, 0x53, 0x45],
    'EFI_KMS_FORMAT_GENERIC_3072_GUID': [0xb9237513, 0x6c44, 0x4411, 0xa9, 0x90, 0x21, 0xe5, 0x56, 0xe0, 0x5a, 0xde],
    'EFI_KMS_FORMAT_GENERIC_512_GUID': [0x978fe043, 0xd7af, 0x422e, 0x8a, 0x92, 0x2b, 0x48, 0xe4, 0x63, 0xbd, 0xe6],
    'EFI_KMS_FORMAT_MD2_128_GUID': [0x78be11c4, 0xee44, 0x4a22, 0x9f, 0x05, 0x03, 0x85, 0x2e, 0xc5, 0xc9, 0x78],
    'EFI_KMS_FORMAT_MD4_128_GUID': [0xd1c17aa1, 0xcac5, 0x400f, 0xbe, 0x17, 0xe2, 0xa2, 0xae, 0x06, 0x67, 0x7c],
    'EFI_KMS_FORMAT_MD5SHA_128_GUID': [0x1c178237, 0x6897, 0x459e, 0x9d, 0x36, 0x67, 0xce, 0x8e, 0xf9, 0x4f, 0x76],
    'EFI_KMS_FORMAT_MD5_128_GUID': [0xdcbc3662, 0x9cda, 0x4b52, 0xa0, 0x4c, 0x82, 0xeb, 0x1d, 0x23, 0x48, 0xc7],
    'EFI_KMS_FORMAT_MDC2_128_GUID': [0xf7ad60f8, 0xefa8, 0x44a3, 0x91, 0x13, 0x23, 0x1f, 0x39, 0x9e, 0xb4, 0xc7],
    'EFI_KMS_FORMAT_MDC4_128_GUID': [0x3fa4f847, 0xd8eb, 0x4df4, 0xbd, 0x49, 0x10, 0x3a, 0x0a, 0x84, 0x7b, 0xbc],
    'EFI_KMS_FORMAT_RSASHA1_1024_GUID': [0x56417bed, 0x6bbe, 0x4882, 0x86, 0xa0, 0x3a, 0xe8, 0xbb, 0x17, 0xf8, 0xf9],
    'EFI_KMS_FORMAT_RSASHA1_2048_GUID': [0xf66447d4, 0x75a6, 0x463e, 0xa8, 0x19, 0x07, 0x7f, 0x2d, 0xda, 0x05, 0xe9],
    'EFI_KMS_FORMAT_RSASHA256_2048_GUID': [0xa477af13, 0x877d, 0x4060, 0xba, 0xa1, 0x25, 0xd1, 0xbe, 0xa0, 0x8a, 0xd3],
    'EFI_KMS_FORMAT_SHA1_160_GUID': [0x453c5e5a, 0x482d, 0x43f0, 0x87, 0xc9, 0x59, 0x41, 0xf3, 0xa3, 0x8a, 0xc2],
    'EFI_KMS_FORMAT_SHA256_256_GUID': [0x6bb4f5cd, 0x8022, 0x448d, 0xbc, 0x6d, 0x77, 0x1b, 0xae, 0x93, 0x5f, 0xc6],
    'EFI_KMS_FORMAT_SHA512_512_GUID': [0x2f240e12, 0xe14d, 0x475c, 0x83, 0xb0, 0xef, 0xff, 0x22, 0xd7, 0x7b, 0xe7],
    'EFI_KMS_PROTOCOL_GUID': [0xEC3A978D, 0x7C4E, 0x48FA, 0x9A, 0xBE, 0x6A, 0xD9, 0x1C, 0xC8, 0xF8, 0x11],
    'EFI_LEGACY_8259_PROTOCOL_GUID': [0x38321dba, 0x4fe0, 0x4e17, 0x8a, 0xec, 0x41, 0x30, 0x55, 0xea, 0xed, 0xc1],
    'EFI_LEGACY_BIOS_GUID': [0x2e3044ac, 0x879f, 0x490f, 0x97, 0x60, 0xbb, 0xdf, 0xaf, 0x69, 0x5f, 0x50],
    'EFI_LEGACY_BIOS_PLATFORM_PROTOCOL_GUID': [0x783658a3, 0x4172, 0x4421, 0xa2, 0x99, 0xe0, 0x9, 0x7, 0x9c, 0xc, 0xb4],
    'EFI_LEGACY_BIOS_PROTOCOL_GUID': [0xdb9a1e3d, 0x45cb, 0x4abb, 0x85, 0x3b, 0xe5, 0x38, 0x7f, 0xdb, 0x2e, 0x2d],
    'EFI_LEGACY_BIOS_THUNK_PROTOCOL_GUID': [0x4c51a7ba, 0x7195, 0x442d, 0x87, 0x92, 0xbe, 0xea, 0x6e, 0x2f, 0xf6, 0xec],
    'EFI_LEGACY_DEV_ORDER_VARIABLE_GUID': [0xa56074db, 0x65fe, 0x45f7, 0xbd, 0x21, 0x2d, 0x2b, 0xdd, 0x8e, 0x96, 0x52],
    'EFI_LEGACY_INTERRUPT_PROTOCOL_GUID': [0x31ce593d, 0x108a, 0x485d, 0xad, 0xb2, 0x78, 0xf2, 0x1f, 0x29, 0x66, 0xbe],
    'EFI_LEGACY_REGION2_PROTOCOL_GUID': [0x70101eaf, 0x85, 0x440c, 0xb3, 0x56, 0x8e, 0xe3, 0x6f, 0xef, 0x24, 0xf0],
    'EFI_LEGACY_REGION_PROTOCOL_GUID': [0xfc9013a, 0x568, 0x4ba9, 0x9b, 0x7e, 0xc9, 0xc3, 0x90, 0xa6, 0x60, 0x9b],
    'EFI_LOADED_IMAGE_DEVICE_PATH_PROTOCOL_GUID': [0xbc62157e, 0x3e33, 0x4fec, 0x99, 0x20, 0x2d, 0x3b, 0x36, 0xd7, 0x50, 0xdf],
    'EFI_LOADED_IMAGE_PROTOCOL_GUID': [0x5B1B31A1, 0x9562, 0x11d2, 0x8E, 0x3F, 0x00, 0xA0, 0xC9, 0x69, 0x72, 0x3B],
    'EFI_LOAD_FILE2_PROTOCOL_GUID': [0x4006c0c1, 0xfcb3, 0x403e, 0x99, 0x6d, 0x4a, 0x6c, 0x87, 0x24, 0xe0, 0x6d],
    'EFI_LOAD_FILE_PROTOCOL_GUID': [0x56EC3091, 0x954C, 0x11d2, 0x8E, 0x3F, 0x00, 0xA0, 0xC9, 0x69, 0x72, 0x3B],
    'EFI_LOAD_FIXED_ADDRESS_CONFIGURATION_TABLE_GUID': [0x2CA88B53, 0xD296, 0x4080, 0xA4, 0xA5, 0xCA, 0xD9, 0xBA, 0xE2, 0x4B, 0x9],
    'EFI_LOCK_BOX_PROTOCOL_GUID': [0xbd445d79, 0xb7ad, 0x4f04, 0x9a, 0xd8, 0x29, 0xbd, 0x20, 0x40, 0xeb, 0x3c],
    'EFI_MANAGED_NETWORK_PROTOCOL_GUID': [0x7ab33a91, 0xace5, 0x4326, 0xb5, 0x72, 0xe7, 0xee, 0x33, 0xd3, 0x9f, 0x16],
    'EFI_MANAGED_NETWORK_SERVICE_BINDING_PROTOCOL_GUID': [0xf36ff770, 0xa7e1, 0x42cf, 0x9e, 0xd2, 0x56, 0xf0, 0xf2, 0x71, 0xf4, 0x4c],
    'EFI_MEASURED_FV_HOB_GUID': [0xb2360b42, 0x7173, 0x420a, 0x86, 0x96, 0x46, 0xca, 0x6b, 0xab, 0x10, 0x60],
    'EFI_MEMORY_PRODUCER_GUID': [0x1d7add6e, 0xb2da, 0x4b0b, 0xb2, 0x9f, 0x49, 0xcb, 0x42, 0xf4, 0x63, 0x56],
    'EFI_MEMORY_SUBCLASS_GUID': [0x4E8F4EBB, 0x64B9, 0x4e05, 0x9B, 0x18, 0x4C, 0xFE, 0x49, 0x23, 0x50, 0x97],
    'EFI_MEMORY_TYPE_INFORMATION_GUID': [0x4c19049f, 0x4137, 0x4dd3, 0x9c, 0x10, 0x8b, 0x97, 0xa8, 0x3f, 0xfd, 0xfa],
    'EFI_METRONOME_ARCH_PROTOCOL_GUID': [0x26baccb2, 0x6f42, 0x11d4, 0xbc, 0xe7, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_MINI_SHELL_FILE_GUID': [0x86ad232b, 0xd33a, 0x465c, 0xbf, 0x5f, 0x41, 0x37, 0xb, 0xa9, 0x2f, 0xe2],
    'EFI_MISC_PRODUCER_GUID': [0x62512c92, 0x63c4, 0x4d80, 0x82, 0xb1, 0xc1, 0xa4, 0xdc, 0x44, 0x80, 0xe5],
    'EFI_MISC_SUBCLASS_GUID': [0x772484B2, 0x7482, 0x4b91, 0x9F, 0x9A, 0xAD, 0x43, 0xF8, 0x1C, 0x58, 0x81],
    'EFI_MMC_HOST_PROTOCOL_GUID': [0x3e591c00, 0x9e4a, 0x11df, 0x92, 0x44, 0x00, 0x02, 0xA5, 0xD5, 0xC5, 0x1B],
    'EFI_MONOTONIC_COUNTER_ARCH_PROTOCOL_GUID': [0x1da97072, 0xbddc, 0x4b30, 0x99, 0xf1, 0x72, 0xa0, 0xb5, 0x6f, 0xff, 0x2a],
    'EFI_MONTONIC_COUNTER_ARCH_PROTOCOL_GUID': [0x1da97072, 0xbddc, 0x4b30, 0x99, 0xf1, 0x72, 0xa0, 0xb5, 0x6f, 0xff, 0x2a],
    'EFI_MPS_TABLE_GUID': [0xeb9d2d2f, 0x2d88, 0x11d3, 0x9a, 0x16, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_MP_SERVICES_PROTOCOL_GUID': [0x3fdda605, 0xa76e, 0x4f46, 0xad, 0x29, 0x12, 0xf4, 0x53, 0x1b, 0x3d, 0x08],
    'EFI_MTFTP4_PROTOCOL_GUID': [0x78247c57, 0x63db, 0x4708, 0x99, 0xc2, 0xa8, 0xb4, 0xa9, 0xa6, 0x1f, 0x6b],
    'EFI_MTFTP4_SERVICE_BINDING_PROTOCOL_GUID': [0x2FE800BE, 0x8F01, 0x4aa6, 0x94, 0x6B, 0xD7, 0x13, 0x88, 0xE1, 0x83, 0x3F],
    'EFI_MTFTP6_PROTOCOL_GUID': [0xbf0a78ba, 0xec29, 0x49cf, 0xa1, 0xc9, 0x7a, 0xe5, 0x4e, 0xab, 0x6a, 0x51],
    'EFI_MTFTP6_SERVICE_BINDING_PROTOCOL_GUID': [0xd9760ff3, 0x3cca, 0x4267, 0x80, 0xf9, 0x75, 0x27, 0xfa, 0xfa, 0x42, 0x23],
    'EFI_NETWORK_INTERFACE_IDENTIFIER_PROTOCOL_GUID': [0x1ACED566, 0x76ED, 0x4218, 0xBC, 0x81, 0x76, 0x7F, 0x1F, 0x97, 0x7A, 0x89],
    'EFI_NETWORK_INTERFACE_IDENTIFIER_PROTOCOL_GUID': [0xE18541CD, 0xF755, 0x4f73, 0x92, 0x8D, 0x64, 0x3C, 0x8A, 0x79, 0xB2, 0x29],
    'EFI_NIC_IP4_CONFIG_NVDATA_GUID': [0x9d5b53f, 0xf4b0, 0x4f59, 0xa0, 0xb1, 0x7b, 0x57, 0xd3, 0x5c, 0xe, 0x5],
    'EFI_NIC_IP4_CONFIG_PROTOCOL_GUID': [0xdca3d4d, 0x12da, 0x4728, 0xbf, 0x7e, 0x86, 0xce, 0xb9, 0x28, 0xd0, 0x67],
    'EFI_NIC_IP4_CONFIG_VARIABLE_GUID': [0xd8944553, 0xc4dd, 0x41f4, 0x9b, 0x30, 0xe1, 0x39, 0x7c, 0xfb, 0x26, 0x7b],
    'EFI_NT_LOAD_AS_DLL_PPI_GUID': [0xccc53f6b, 0xa03a, 0x4ed8, 0x83, 0x9a, 0x3, 0xd9, 0x9c, 0x2, 0xb4, 0xe3],
    'EFI_OEM_BADGING_PROTOCOL_GUID': [0x170e13c0, 0xbf1b, 0x4218, 0x87, 0x1d, 0x2a, 0xbd, 0xc6, 0xf8, 0x87, 0xbc],
    'EFI_PART_TYPE_EFI_SYSTEM_PART_GUID': [0xc12a7328, 0xf81f, 0x11d2, 0xba, 0x4b, 0x00, 0xa0, 0xc9, 0x3e, 0xc9, 0x3b],
    'EFI_PART_TYPE_LEGACY_MBR_GUID': [0x024dee41, 0x33e7, 0x11d3, 0x9d, 0x69, 0x00, 0x08, 0xc7, 0x81, 0xf3, 0x9f],
    'EFI_PATH_FILE_NAME_GUID': [0x7644C181, 0xFA6E, 0x46DA, 0x80, 0xCB, 0x04, 0xB9, 0x90, 0x40, 0x62, 0xE8],
    'EFI_PCD_PROTOCOL_GUID': [0x13a3f0f6, 0x264a, 0x3ef0, 0xf2, 0xe0, 0xde, 0xc5, 0x12, 0x34, 0x2f, 0x34],
    'EFI_PCI_ENUMERATION_COMPLETE_GUID': [0x30cfe3e7, 0x3de1, 0x4586, 0xbe, 0x20, 0xde, 0xab, 0xa1, 0xb3, 0xb7, 0x93],
    'EFI_PCI_EXPRESS_BASE_ADDRESS_GUID': [0x3677d529, 0x326f, 0x4603, 0xa9, 0x26, 0xea, 0xac, 0xe0, 0x1d, 0xcb, 0xb0],
    'EFI_PCI_HOST_BRIDGE_RESOURCE_ALLOCATION_PROTOCOL_GUID': [0xCF8034BE, 0x6768, 0x4d8b, 0xB7, 0x39, 0x7C, 0xCE, 0x68, 0x3A, 0x9F, 0xBE],
    'EFI_PCI_HOTPLUG_DEVICE_GUID': [0x0b280816, 0x52e7, 0x4e51, 0xaa, 0x57, 0x11, 0xbd, 0x41, 0xcb, 0xef, 0xc3],
    'EFI_PCI_HOTPLUG_REQUEST_PROTOCOL_GUID': [0x19cb87ab, 0x2cb9, 0x4665, 0x83, 0x60, 0xdd, 0xcf, 0x60, 0x54, 0xf7, 0x9d],
    'EFI_PCI_HOT_PLUG_INIT_PROTOCOL_GUID': [0xaa0e8bc1, 0xdabc, 0x46b0, 0xa8, 0x44, 0x37, 0xb8, 0x16, 0x9b, 0x2b, 0xea],
    'EFI_PCI_IO_PROTOCOL_GUID': [0x4cf5b200, 0x68b8, 0x4ca5, 0x9e, 0xec, 0xb2, 0x3e, 0x3f, 0x50, 0x2, 0x9a],
    'EFI_PCI_OPTION_ROM_TABLE_GUID': [0x7462660f, 0x1cbd, 0x48da, 0xad, 0x11, 0x91, 0x71, 0x79, 0x13, 0x83, 0x1c],
    'EFI_PCI_OVERRIDE_GUID': [0xb5b35764, 0x460c, 0x4a06, 0x99, 0xfc, 0x77, 0xa1, 0x7c, 0x1b, 0x5c, 0xeb],
    'EFI_PCI_PLATFORM_PROTOCOL_GUID': [0x7d75280, 0x27d4, 0x4d69, 0x90, 0xd0, 0x56, 0x43, 0xe2, 0x38, 0xb3, 0x41],
    'EFI_PCI_ROOT_BRIDGE_IO_PROTOCOL_GUID': [0x2f707ebb, 0x4a1a, 0x11d4, 0x9a, 0x38, 0x00, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_PC_ANSI_GUID': [0xe0c14753, 0xf9be, 0x11d2, 0x9a, 0x0c, 0x00, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_PEI_APRIORI_FILE_NAME_GUID': [0x1b45cc0a, 0x156a, 0x428a, 0xaf, 0x62, 0x49, 0x86, 0x4d, 0xa0, 0xe6, 0xe6],
    'EFI_PEI_BOOT_SCRIPT_EXECUTER_PPI_GUID': [0xabd42895, 0x78cf, 0x4872, 0x84, 0x44, 0x1b, 0x5c, 0x18, 0x0b, 0xfb, 0xff],
    'EFI_PEI_CPU_IO_PPI_INSTALLED_GUID': [0xe6af1f7b, 0xfc3f, 0x46da, 0xa8, 0x28, 0xa3, 0xb4, 0x57, 0xa4, 0x42, 0x82],
    'EFI_PEI_DECOMPRESS_PPI_GUID': [0x1a36e4e7, 0xfab6, 0x476a, 0x8e, 0x75, 0x69, 0x5a, 0x5, 0x76, 0xfd, 0xd7],
    'EFI_PEI_DEVICE_RECOVERY_MODULE_PPI_GUID': [0x0DE2CE25, 0x446A, 0x45a7, 0xBF, 0xC9, 0x37, 0xDA, 0x26, 0x34, 0x4B, 0x37],
    'EFI_PEI_END_OF_PEI_PHASE_PPI_GUID': [0x605EA650, 0xC65C, 0x42e1, 0xBA, 0x80, 0x91, 0xA5, 0x2A, 0xB6, 0x18, 0xC6],
    'EFI_PEI_FIND_FV_PPI_GUID': [0x36164812, 0xa023, 0x44e5, 0xbd, 0x85, 0x5, 0xbf, 0x3c, 0x77, 0x0, 0xaa],
    'EFI_PEI_FIRMWARE_VOLUME_INFO_PPI_GUID': [0x49edb1c1, 0xbf21, 0x4761, 0xbb, 0x12, 0xeb, 0x0, 0x31, 0xaa, 0xbb, 0x39],
    'EFI_PEI_FLUSH_INSTRUCTION_CACHE_GUID': [0xd8117cfc, 0x94a6, 0x11d4, 0x9a, 0x3a, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_PEI_LOADED_IMAGE_PPI_GUID': [0xc1fcd448, 0x6300, 0x4458, 0xb8, 0x64, 0x28, 0xdf, 0x01, 0x53, 0x64, 0xbc],
    'EFI_PEI_LOAD_FILE_GUID': [0xb9e0abfe, 0x5979, 0x4914, 0x97, 0x7f, 0x6d, 0xee, 0x78, 0xc2, 0x78, 0xa6],
    'EFI_PEI_LOAD_FILE_PPI_GUID': [0xb9e0abfe, 0x5979, 0x4914, 0x97, 0x7f, 0x6d, 0xee, 0x78, 0xc2, 0x78, 0xa6],
    'EFI_PEI_PCD_PPI_GUID': [0x1f34d25, 0x4de2, 0x23ad, 0x3f, 0xf3, 0x36, 0x35, 0x3f, 0xf3, 0x23, 0xf1],
    'EFI_PEI_PCI_CFG2_PPI_GUID': [0x57a449a, 0x1fdc, 0x4c06, 0xbf, 0xc9, 0xf5, 0x3f, 0x6a, 0x99, 0xbb, 0x92],
    'EFI_PEI_PCI_CFG_PPI_INSTALLED_GUID': [0xe1f2eba0, 0xf7b9, 0x4a26, 0x86, 0x20, 0x13, 0x12, 0x21, 0x64, 0x2a, 0x90],
    'EFI_PEI_PERFORMANCE_HOB_GUID': [0x10f432de, 0xdeec, 0x4631, 0x80, 0xcd, 0x47, 0xf6, 0x5d, 0x8f, 0x80, 0xbb],
    'EFI_PEI_PERMANENT_MEMORY_INSTALLED_PPI_GUID': [0xf894643d, 0xc449, 0x42d1, 0x8e, 0xa8, 0x85, 0xbd, 0xd8, 0xc6, 0x5b, 0xde],
    'EFI_PEI_PE_COFF_LOADER_GUID': [0xd8117cff, 0x94a6, 0x11d4, 0x9a, 0x3a, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_PEI_READ_ONLY_VARIABLE2_PPI_GUID': [0x2ab86ef5, 0xecb5, 0x4134, 0xb5, 0x56, 0x38, 0x54, 0xca, 0x1f, 0xe1, 0xb4],
    'EFI_PEI_READ_ONLY_VARIABLE_ACCESS_PPI_GUID': [0x3cdc90c6, 0x13fb, 0x4a75, 0x9e, 0x79, 0x59, 0xe9, 0xdd, 0x78, 0xb9, 0xfa],
    'EFI_PEI_RECOVERY_BLOCK_IO_PPI_GUID': [0x695d8aa1, 0x42ee, 0x4c46, 0x80, 0x5c, 0x6e, 0xa6, 0xbc, 0xe7, 0x99, 0xe3],
    'EFI_PEI_RECOVERY_MODULE_PPI_GUID': [0xFB6D9542, 0x612D, 0x4f45, 0x87, 0x2F, 0x5C, 0xFF, 0x52, 0xE9, 0x3D, 0xCF],
    'EFI_PEI_REPORT_PROGRESS_CODE_PPI_GUID': [0x229832d3, 0x7a30, 0x4b36, 0xb8, 0x27, 0xf4, 0xc, 0xb7, 0xd4, 0x54, 0x36],
    'EFI_PEI_RESET_PPI_GUID': [0xef398d58, 0x9dfd, 0x4103, 0xbf, 0x94, 0x78, 0xc6, 0xf4, 0xfe, 0x71, 0x2f],
    'EFI_PEI_RSC_HANDLER_PPI_GUID': [0x65d394, 0x9951, 0x4144, 0x82, 0xa3, 0xa, 0xfc, 0x85, 0x79, 0xc2, 0x51],
    'EFI_PEI_S3_RESUME2_PPI_GUID': [0x6D582DBC, 0xDB85, 0x4514, 0x8F, 0xCC, 0x5A, 0xDF, 0x62, 0x27, 0xB1, 0x47],
    'EFI_PEI_S3_RESUME_PPI_GUID': [0x4426CCB2, 0xE684, 0x4a8a, 0xAE, 0x40, 0x20, 0xD4, 0xB0, 0x25, 0xB7, 0x10],
    'EFI_PEI_SECTION_EXTRACTION_PPI_GUID': [0x4F89E208, 0xE144, 0x4804, 0x9E, 0xC8, 0x0F, 0x89, 0x4F, 0x7E, 0x36, 0xD7],
    'EFI_PEI_SECURITY2_PPI_GUID': [0xdcd0be23, 0x9586, 0x40f4, 0xb6, 0x43, 0x06, 0x52, 0x2c, 0xed, 0x4e, 0xde],
    'EFI_PEI_SECURITY_PPI_GUID': [0x1388066e, 0x3a57, 0x4efa, 0x98, 0xf3, 0xc1, 0x2f, 0x3a, 0x95, 0x8a, 0x29],
    'EFI_PEI_SMBUS2_PPI_GUID': [0x9ca93627, 0xb65b, 0x4324, 0xa2, 0x2, 0xc0, 0xb4, 0x61, 0x76, 0x45, 0x43],
    'EFI_PEI_SMBUS_PPI_GUID': [0xabd42895, 0x78cf, 0x4872, 0x84, 0x44, 0x1b, 0x5c, 0x18, 0xb, 0xfb, 0xda],
    'EFI_PEI_SMM_COMMUNICATION_PPI_GUID': [0xae933e1c, 0xcc47, 0x4e38, 0x8f, 0xe, 0xe2, 0xf6, 0x1d, 0x26, 0x5, 0xdf],
    'EFI_PEI_STALL_PPI_GUID': [0x1f4c6f90, 0xb06b, 0x48d8, 0xa2, 0x01, 0xba, 0xe5, 0xf1, 0xcd, 0x7d, 0x56],
    'EFI_PEI_TEMPORARY_RAM_DONE_PPI_GUID': [0xceab683c, 0xec56, 0x4a2d, 0xa9, 0x06, 0x40, 0x53, 0xfa, 0x4e, 0x9c, 0x16],
    'EFI_PEI_TEMPORARY_RAM_SUPPORT_PPI_GUID': [0xdbe23aa9, 0xa345, 0x4b97, 0x85, 0xb6, 0xb2, 0x26, 0xf1, 0x61, 0x73, 0x89],
    'EFI_PEI_TRANSFER_CONTROL_GUID': [0xd8117d02, 0x94a6, 0x11d4, 0x9a, 0x3a, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_PEI_VECTOR_HANDOFF_INFO_PPI_GUID': [0x3cd652b4, 0x6d33, 0x4dce, 0x89, 0xdb, 0x83, 0xdf, 0x97, 0x66, 0xfc, 0xca],
    'EFI_PERFORMANCE_PROTOCOL_GUID': [0xFFECFFFF, 0x923C, 0x14d2, 0x9E, 0x3F, 0x22, 0xA0, 0xC9, 0x69, 0x56, 0x3B],
    'EFI_PHYSICAL_PRESENCE_DATA_GUID': [0xf6499b1, 0xe9ad, 0x493d, 0xb9, 0xc2, 0x2f, 0x90, 0x81, 0x5c, 0x6c, 0xbc],
    'EFI_PLATFORM_DRIVER_OVERRIDE_PROTOCOL_GUID': [0x6b30c738, 0xa391, 0x11d4, 0x9a, 0x3b, 0x00, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_PLATFORM_MEMTEST_PROTOCOL_GUID': [0x859ba18, 0x7dd7, 0x4ed7, 0xa8, 0x8e, 0x10, 0x9c, 0x63, 0x91, 0x7b, 0xdd],
    'EFI_PLATFORM_TO_DRIVER_CONFIGURATION_CLP_GUID': [0x345ecc0e, 0x0cb6, 0x4b75, 0xbb, 0x57, 0x1b, 0x12, 0x9c, 0x47, 0x33, 0x3e],
    'EFI_PLATFORM_TO_DRIVER_CONFIGURATION_PROTOCOL_GUID': [0x642cd590, 0x8059, 0x4c0a, 0xa9, 0x58, 0xc5, 0xec, 0x07, 0xd2, 0x3c, 0x4b],
    'EFI_PRIMARY_CONSOLE_IN_DEVICE_GUID': [0xe451dcbe, 0x96a1, 0x4729, 0xa5, 0xcf, 0x6b, 0x9c, 0x2c, 0xff, 0x47, 0xfd],
    'EFI_PRIMARY_CONSOLE_OUT_DEVICE_GUID': [0x62bdf38a, 0xe3d5, 0x492c, 0x95, 0xc, 0x23, 0xa7, 0xf6, 0x6e, 0x67, 0x2e],
    'EFI_PRIMARY_STANDARD_ERROR_DEVICE_GUID': [0x5a68191b, 0x9b97, 0x4752, 0x99, 0x46, 0xe3, 0x6a, 0x5d, 0xa9, 0x42, 0xb1],
    'EFI_PRINT2_PROTOCOL_GUID': [0xf05976ef, 0x83f1, 0x4f3d, 0x86, 0x19, 0xf7, 0x59, 0x5d, 0x41, 0xe5, 0x38],
    'EFI_PRINT_PROTOCOL_GUID': [0xdf2d868e, 0x32fc, 0x4cf0, 0x8e, 0x6b, 0xff, 0xd9, 0x5d, 0x13, 0x43, 0xd0],
    'EFI_PROCESSOR_PRODUCER_GUID': [0x1bf06aea, 0x5bec, 0x4a8d, 0x95, 0x76, 0x74, 0x9b, 0x09, 0x56, 0x2d, 0x30],
    'EFI_PROCESSOR_SUBCLASS_GUID': [0x26fdeb7e, 0xb8af, 0x4ccf, 0xaa, 0x97, 0x02, 0x63, 0x3c, 0xe4, 0x8c, 0xa7],
    'EFI_PS2_POLICY_PROTOCOL_GUID': [0x4df19259, 0xdc71, 0x4d46, 0xbe, 0xf1, 0x35, 0x7b, 0xb5, 0x78, 0xc4, 0x18],
    'EFI_PXE_BASE_CODE_CALLBACK_PROTOCOL_GUID': [0x245dca21, 0xfb7b, 0x11d3, 0x8f, 0x01, 0x00, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_PXE_BASE_CODE_PROTOCOL_GUID': [0x03c4e603, 0xac28, 0x11d3, 0x9a, 0x2d, 0x00, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_PXE_DHCP4_CALLBACK_PROTOCOL_GUID': [0xc1544c01, 0x92a4, 0x4198, 0x8a, 0x84, 0x77, 0x85, 0x83, 0xc2, 0x36, 0x21],
    'EFI_PXE_DHCP4_PROTOCOL_GUID': [0x03c4e624, 0xac28, 0x11d3, 0x9a, 0x2d, 0x00, 0x90, 0x29, 0x3f, 0xc1, 0x4d],
    'EFI_REAL_TIME_CLOCK_ARCH_PROTOCOL_GUID': [0x27CFAC87, 0x46CC, 0x11d4, 0x9A, 0x38, 0x00, 0x90, 0x27, 0x3F, 0xC1, 0x4D],
    'EFI_RESET_ARCH_PROTOCOL_GUID': [0x27CFAC88, 0x46CC, 0x11d4, 0x9A, 0x38, 0x00, 0x90, 0x27, 0x3F, 0xC1, 0x4D],
    'EFI_RSC_HANDLER_PROTOCOL_GUID': [0x86212936, 0xe76, 0x41c8, 0xa0, 0x3a, 0x2a, 0xf2, 0xfc, 0x1c, 0x39, 0xe2],
    'EFI_RUNTIME_ARCH_PROTOCOL_GUID': [0xb7dfb4e1, 0x52f, 0x449f, 0x87, 0xbe, 0x98, 0x18, 0xfc, 0x91, 0xb7, 0x33],
    'EFI_RUNTIME_CRYPT_PROTOCOL_GUID': [0xe1475e0c, 0x1746, 0x4802, 0x86, 0x2e, 0x1, 0x1c, 0x2c, 0x2d, 0x9d, 0x86],
    'EFI_S3_SAVE_STATE_PROTOCOL_GUID': [0xe857caf6, 0xc046, 0x45dc, 0xbe, 0x3f, 0xee, 0x7, 0x65, 0xfb, 0xa8, 0x87],
    'EFI_S3_SMM_SAVE_STATE_PROTOCOL_GUID': [0x320afe62, 0xe593, 0x49cb, 0xa9, 0xf1, 0xd4, 0xc2, 0xf4, 0xaf, 0x1, 0x4c],
    'EFI_SAL_MCA_INIT_PMI_PROTOCOL_GUID': [0xb60dc6e8, 0x3b6f, 0x11d5, 0xaf, 0x9, 0x0, 0xa0, 0xc9, 0x44, 0xa0, 0x5b],
    'EFI_SAL_SYSTEM_TABLE_GUID': [0xeb9d2d32, 0x2d88, 0x11d3, 0x9a, 0x16, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_SAS_DEVICE_PATH_GUID': [0xd487ddb4, 0x008b, 0x11d9, 0xaf, 0xdc, 0x00, 0x10, 0x83, 0xff, 0xca, 0x4d],
    'EFI_SCSI_BUS_PROTOCOL_GUID': [0x5261213D, 0x3A3D, 0x441E, 0xB3, 0xAF, 0x21, 0xD3, 0xF7, 0xA4, 0xCA, 0x17],
    'EFI_SCSI_IO_PROTOCOL_GUID': [0x932f47e6, 0x2362, 0x4002, 0x80, 0x3e, 0x3c, 0xd5, 0x4b, 0x13, 0x8f, 0x85],
    'EFI_SCSI_PASS_THRU_PROTOCOL_GUID': [0xa59e8fcf, 0xbda0, 0x43bb, 0x90, 0xb1, 0xd3, 0x73, 0x2e, 0xca, 0xa8, 0x77],
    'EFI_SECTION_EXTRACTION_PROTOCOL_GUID': [0x448F5DA4, 0x6DD7, 0x4FE1, 0x93, 0x07, 0x69, 0x22, 0x41, 0x92, 0x21, 0x5D],
    'EFI_SECURITY2_ARCH_PROTOCOL_GUID': [0x94ab2f58, 0x1438, 0x4ef1, 0x91, 0x52, 0x18, 0x94, 0x1a, 0x3a, 0x0e, 0x68],
    'EFI_SECURITY_ARCH_PROTOCOL_GUID': [0xA46423E3, 0x4617, 0x49f1, 0xB9, 0xFF, 0xD1, 0xBF, 0xA9, 0x11, 0x58, 0x39],
    'EFI_SECURITY_POLICY_PROTOCOL_GUID': [0x78E4D245, 0xCD4D, 0x4a05, 0xA2, 0xBA, 0x47, 0x43, 0xE8, 0x6C, 0xFC, 0xAB],
    'EFI_SEC_PLATFORM_INFORMATION_GUID': [0x6f8c2b35, 0xfef4, 0x448d, 0x82, 0x56, 0xe1, 0x1b, 0x19, 0xd6, 0x10, 0x77],
    'EFI_SERIAL_IO_PROTOCOL_GUID': [0xBB25CF6F, 0xF1D4, 0x11D2, 0x9A, 0x0C, 0x00, 0x90, 0x27, 0x3F, 0xC1, 0xFD],
    'EFI_SE_EXT_SIGNATURE_GUID': [0xd2c18636, 0x40e5, 0x4eb5, 0xa3, 0x1b, 0x36, 0x69, 0x5f, 0xd4, 0x2c, 0x87],
    'EFI_SHELLPKG_TOKEN_SPACE_GUID': [0x171e9188, 0x31d3, 0x40f5, 0xb1, 0xc, 0x53, 0x9b, 0x2d, 0xb9, 0x40, 0xcd],
    'EFI_SHELL_FILE_GUID': [0xc57ad6b7, 0x0515, 0x40a8, 0x9d, 0x21, 0x55, 0x16, 0x52, 0x85, 0x4e, 0x37],
    'EFI_SHELL_PARAMETERS_PROTOCOL_GUID': [0x752f3136, 0x4e16, 0x4fdc, 0xa2, 0x2a, 0xe5, 0xf4, 0x68, 0x12, 0xf4, 0xca],
    'EFI_SHELL_PROTOCOL_GUID': [0x6302d008, 0x7f9b, 0x4f30, 0x87, 0xac, 0x60, 0xc9, 0xfe, 0xf5, 0xda, 0x4e],
    'EFI_SIMPLE_FILE_SYSTEM_PROTOCOL_GUID': [0x964e5b22, 0x6459, 0x11d2, 0x8e, 0x39, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_SIMPLE_NETWORK_PROTOCOL_GUID': [0xA19832B9, 0xAC25, 0x11D3, 0x9A, 0x2D, 0x00, 0x90, 0x27, 0x3F, 0xC1, 0x4D],
    'EFI_SIMPLE_POINTER_PROTOCOL_GUID': [0x31878c87, 0xb75, 0x11d5, 0x9a, 0x4f, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_SIMPLE_TEXT_INPUT_EX_PROTOCOL_GUID': [0xdd9e7534, 0x7762, 0x4698, 0x8c, 0x14, 0xf5, 0x85, 0x17, 0xa6, 0x25, 0xaa],
    'EFI_SIMPLE_TEXT_INPUT_PROTOCOL_GUID': [0x387477c1, 0x69c7, 0x11d2, 0x8e, 0x39, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_SIMPLE_TEXT_IN_PROTOCOL_GUID': [0x387477c1, 0x69c7, 0x11d2, 0x8e, 0x39, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_SIMPLE_TEXT_OUTPUT_PROTOCOL_GUID': [0x387477c2, 0x69c7, 0x11d2, 0x8e, 0x39, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_SIMPLE_TEXT_OUT_PROTOCOL_GUID': [0x387477c2, 0x69c7, 0x11d2, 0x8e, 0x39, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_SIO_PROTOCOL_GUID': [0x215fdd18, 0xbd50, 0x4feb, 0x89, 0xb, 0x58, 0xca, 0xb, 0x47, 0x39, 0xe9],
    'EFI_SMBIOS_PROTOCOL_GUID': [0x3583ff6, 0xcb36, 0x4940, 0x94, 0x7e, 0xb9, 0xb3, 0x9f, 0x4a, 0xfa, 0xf7],
    'EFI_SMBIOS_TABLE_GUID': [0xeb9d2d31, 0x2d88, 0x11d3, 0x9a, 0x16, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_SMBUS_HC_PROTOCOL_GUID': [0xe49d33ed, 0x513d, 0x4634, 0xb6, 0x98, 0x6f, 0x55, 0xaa, 0x75, 0x1c, 0x1b],
    'EFI_SMM_ACCESS2_PROTOCOL_GUID': [0xc2702b74, 0x800c, 0x4131, 0x87, 0x46, 0x8f, 0xb5, 0xb8, 0x9c, 0xe4, 0xac],
    'EFI_SMM_ACCESS_PROTOCOL_GUID': [0x3792095a, 0xe309, 0x4c1e, 0xaa, 0x01, 0x85, 0xf5, 0x65, 0x5a, 0x17, 0xf1],
    'EFI_SMM_BASE2_PROTOCOL_GUID': [0xf4ccbfb7, 0xf6e0, 0x47fd, 0x9d, 0xd4, 0x10, 0xa8, 0xf1, 0x50, 0xc1, 0x91],
    'EFI_SMM_BASE_HELPER_READY_PROTOCOL_GUID': [0x910dca07, 0x1f94, 0x4ee7, 0xaf, 0x2f, 0xff, 0x72, 0xf3, 0x15, 0x43, 0x53],
    'EFI_SMM_BASE_PROTOCOL_GUID': [0x1390954D, 0xda95, 0x4227, 0x93, 0x28, 0x72, 0x82, 0xc2, 0x17, 0xda, 0xa8],
    'EFI_SMM_COMMUNICATION_PROTOCOL_GUID': [0xc68ed8e2, 0x9dc6, 0x4cbd, 0x9d, 0x94, 0xdb, 0x65, 0xac, 0xc5, 0xc3, 0x32],
    'EFI_SMM_CONFIGURATION_PROTOCOL_GUID': [0x26eeb3de, 0xb689, 0x492e, 0x80, 0xf0, 0xbe, 0x8b, 0xd7, 0xda, 0x4b, 0xa7],
    'EFI_SMM_CONTROL2_PROTOCOL_GUID': [0x843dc720, 0xab1e, 0x42cb, 0x93, 0x57, 0x8a, 0x0, 0x78, 0xf3, 0x56, 0x1b],
    'EFI_SMM_CONTROL_PROTOCOL_GUID': [0x8d12e231, 0xc667, 0x4fd1, 0x98, 0xf2, 0x24, 0x49, 0xa7, 0xe7, 0xb2, 0xe5],
    'EFI_SMM_CPU_IO2_PROTOCOL_GUID': [0x3242A9D8, 0xCE70, 0x4AA0, 0x95, 0x5D, 0x5E, 0x7B, 0x14, 0x0D, 0xE4, 0xD2],
    'EFI_SMM_CPU_IO_GUID': [0x5f439a0b, 0x45d8, 0x4682, 0xa4, 0xf4, 0xf0, 0x57, 0x6b, 0x51, 0x34, 0x41],
    'EFI_SMM_CPU_PROTOCOL_GUID': [0xeb346b97, 0x975f, 0x4a9f, 0x8b, 0x22, 0xf8, 0xe9, 0x2b, 0xb3, 0xd5, 0x69],
    'EFI_SMM_CPU_SAVE_STATE_PROTOCOL_GUID': [0x21f302ad, 0x6e94, 0x471b, 0x84, 0xbc, 0xb1, 0x48, 0x0, 0x40, 0x3a, 0x1d],
    'EFI_SMM_END_OF_DXE_PROTOCOL_GUID': [0x24e70042, 0xd5c5, 0x4260, 0x8c, 0x39, 0xa, 0xd3, 0xaa, 0x32, 0xe9, 0x3d],
    'EFI_SMM_FAULT_TOLERANT_WRITE_PROTOCOL_GUID': [0x3868fc3b, 0x7e45, 0x43a7, 0x90, 0x6c, 0x4b, 0xa4, 0x7d, 0xe1, 0x75, 0x4d],
    'EFI_SMM_FIRMWARE_VOLUME_BLOCK_PROTOCOL_GUID': [0xd326d041, 0xbd31, 0x4c01, 0xb5, 0xa8, 0x62, 0x8b, 0xe8, 0x7f, 0x6, 0x53],
    'EFI_SMM_GPI_DISPATCH2_PROTOCOL_GUID': [0x25566b03, 0xb577, 0x4cbf, 0x95, 0x8c, 0xed, 0x66, 0x3e, 0xa2, 0x43, 0x80],
    'EFI_SMM_GPI_DISPATCH_PROTOCOL_GUID': [0xe0744b81, 0x9513, 0x49cd, 0x8c, 0xea, 0xe9, 0x24, 0x5e, 0x70, 0x39, 0xda],
    'EFI_SMM_ICHN_DISPATCH_PROTOCOL_GUID': [0xc50b323e, 0x9075, 0x4f2a, 0xac, 0x8e, 0xd2, 0x59, 0x6a, 0x10, 0x85, 0xcc],
    'EFI_SMM_IO_TRAP_DISPATCH2_PROTOCOL_GUID': [0x58dc368d, 0x7bfa, 0x4e77, 0xab, 0xbc, 0xe, 0x29, 0x41, 0x8d, 0xf9, 0x30],
    'EFI_SMM_LOCK_BOX_COMMUNICATION_GUID': [0x2a3cfebd, 0x27e8, 0x4d0a, 0x8b, 0x79, 0xd6, 0x88, 0xc2, 0xa3, 0xe1, 0xc0],
    'EFI_SMM_PCI_ROOT_BRIDGE_IO_PROTOCOL_GUID': [0x8bc1714d, 0xffcb, 0x41c3, 0x89, 0xdc, 0x6c, 0x74, 0xd0, 0x6d, 0x98, 0xea],
    'EFI_SMM_PERIODIC_TIMER_DISPATCH2_PROTOCOL_GUID': [0x4cec368e, 0x8e8e, 0x4d71, 0x8b, 0xe1, 0x95, 0x8c, 0x45, 0xfc, 0x8a, 0x53],
    'EFI_SMM_PERIODIC_TIMER_DISPATCH_PROTOCOL_GUID': [0x9cca03fc, 0x4c9e, 0x4a19, 0x9b, 0x6, 0xed, 0x7b, 0x47, 0x9b, 0xde, 0x55],
    'EFI_SMM_POWER_BUTTON_DISPATCH2_PROTOCOL_GUID': [0x1b1183fa, 0x1823, 0x46a7, 0x88, 0x72, 0x9c, 0x57, 0x87, 0x55, 0x40, 0x9d],
    'EFI_SMM_POWER_BUTTON_DISPATCH_PROTOCOL_GUID': [0xb709efa0, 0x47a6, 0x4b41, 0xb9, 0x31, 0x12, 0xec, 0xe7, 0xa8, 0xee, 0x56],
    'EFI_SMM_READY_TO_LOCK_PROTOCOL_GUID': [0x47b7fa8c, 0xf4bd, 0x4af6, 0x82, 0x00, 0x33, 0x30, 0x86, 0xf0, 0xd2, 0xc8],
    'EFI_SMM_RSC_HANDLER_PROTOCOL_GUID': [0x2ff29fa7, 0x5e80, 0x4ed9, 0xb3, 0x80, 0x1, 0x7d, 0x3c, 0x55, 0x4f, 0xf4],
    'EFI_SMM_STANDBY_BUTTON_DISPATCH2_PROTOCOL_GUID': [0x7300c4a1, 0x43f2, 0x4017, 0xa5, 0x1b, 0xc8, 0x1a, 0x7f, 0x40, 0x58, 0x5b],
    'EFI_SMM_STANDBY_BUTTON_DISPATCH_PROTOCOL_GUID': [0x78965b98, 0xb0bf, 0x449e, 0x8b, 0x22, 0xd2, 0x91, 0x4e, 0x49, 0x8a, 0x98],
    'EFI_SMM_STATUS_CODE_PROTOCOL_GUID': [0x6afd2b77, 0x98c1, 0x4acd, 0xa6, 0xf9, 0x8a, 0x94, 0x39, 0xde, 0xf, 0xb1],
    'EFI_SMM_SWAP_ADDRESS_RANGE_PROTOCOL_GUID': [0x67c4f112, 0x3385, 0x4e55, 0x9c, 0x5b, 0xc0, 0x5b, 0x71, 0x7c, 0x42, 0x28],
    'EFI_SMM_SW_DISPATCH2_PROTOCOL_GUID': [0x18a3c6dc, 0x5eea, 0x48c8, 0xa1, 0xc1, 0xb5, 0x33, 0x89, 0xf9, 0x89, 0x99],
    'EFI_SMM_SW_DISPATCH_PROTOCOL_GUID': [0xe541b773, 0xdd11, 0x420c, 0xb0, 0x26, 0xdf, 0x99, 0x36, 0x53, 0xf8, 0xbf],
    'EFI_SMM_SX_DISPATCH2_PROTOCOL_GUID': [0x456d2859, 0xa84b, 0x4e47, 0xa2, 0xee, 0x32, 0x76, 0xd8, 0x86, 0x99, 0x7d],
    'EFI_SMM_SX_DISPATCH_PROTOCOL_GUID': [0x14fc52be, 0x1dc, 0x426c, 0x91, 0xae, 0xa2, 0x3c, 0x3e, 0x22, 0xa, 0xe8],
    'EFI_SMM_USB_DISPATCH2_PROTOCOL_GUID': [0xee9b8d90, 0xc5a6, 0x40a2, 0xbd, 0xe2, 0x52, 0x55, 0x8d, 0x33, 0xcc, 0xa1],
    'EFI_SMM_USB_DISPATCH_PROTOCOL_GUID': [0xa05b6ffd, 0x87af, 0x4e42, 0x95, 0xc9, 0x62, 0x28, 0xb6, 0x3c, 0xf3, 0xf3],
    'EFI_SMM_VARIABLE_PROTOCOL_GUID': [0xed32d533, 0x99e6, 0x4209, 0x9c, 0xc0, 0x2d, 0x72, 0xcd, 0xd9, 0x98, 0xa7],
    'EFI_SMM_VARIABLE_WRITE_GUID': [0x93ba1826, 0xdffb, 0x45dd, 0x82, 0xa7, 0xe7, 0xdc, 0xaa, 0x3b, 0xbd, 0xf3],
    'EFI_STANDARD_CALLER_ID_GUID': [0xC9DCF469, 0xA7C4, 0x11D5, 0x87, 0xDA, 0x00, 0x06, 0x29, 0x45, 0xC3, 0xB9],
    'EFI_STANDARD_ERROR_DEVICE_GUID': [0xd3b36f2d, 0xd551, 0x11d4, 0x9a, 0x46, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_STATUS_CODE_DATA_TYPE_ASSERT_GUID': [0xDA571595, 0x4D99, 0x487C, 0x82, 0x7C, 0x26, 0x22, 0x67, 0x7D, 0x33, 0x07],
    'EFI_STATUS_CODE_DATA_TYPE_DEBUG_GUID': [0x9A4E9246, 0xD553, 0x11D5, 0x87, 0xE2, 0x00, 0x06, 0x29, 0x45, 0xC3, 0xb9],
    'EFI_STATUS_CODE_DATA_TYPE_ERROR_GUID': [0xAB359CE3, 0x99B3, 0xAE18, 0xC8, 0x9D, 0x95, 0xD3, 0xB0, 0x72, 0xE1, 0x9B],
    'EFI_STATUS_CODE_DATA_TYPE_EXCEPTION_HANDLER_GUID': [0x3BC2BD12, 0xAD2E, 0x11D5, 0x87, 0xDD, 0x00, 0x06, 0x29, 0x45, 0xC3, 0xB9],
    'EFI_STATUS_CODE_DATA_TYPE_PROGRESS_CODE_GUID': [0xA356AB39, 0x35C4, 0x35DA, 0xB3, 0x7A, 0xF8, 0xEA, 0x9E, 0x8B, 0x36, 0xA3],
    'EFI_STATUS_CODE_DATA_TYPE_STRING_GUID': [0x92D11080, 0x496F, 0x4D95, 0xBE, 0x7E, 0x03, 0x74, 0x88, 0x38, 0x2B, 0x0A],
    'EFI_STATUS_CODE_GUID': [0xd083e94c, 0x6560, 0x42e4, 0xb6, 0xd4, 0x2d, 0xf7, 0x5a, 0xdf, 0x6a, 0x2a],
    'EFI_STATUS_CODE_RUNTIME_PROTOCOL_GUID': [0xd2b2b828, 0x826, 0x48a7, 0xb3, 0xdf, 0x98, 0x3c, 0x0, 0x60, 0x24, 0xf0],
    'EFI_STATUS_CODE_SPECIFIC_DATA_GUID': [0x335984bd, 0xe805, 0x409a, 0xb8, 0xf8, 0xd2, 0x7e, 0xce, 0x5f, 0xf7, 0xa6],
    'EFI_STORAGE_SECURITY_COMMAND_PROTOCOL_GUID': [0xC88B0B6D, 0x0DFC, 0x49A7, 0x9C, 0xB4, 0x49, 0x07, 0x4B, 0x4C, 0x3A, 0x78],
    'EFI_SWAP_ADDRESS_RANGE_PROTOCOL_GUID': [0x1259f60d, 0xb754, 0x468e, 0xa7, 0x89, 0x4d, 0xb8, 0x5d, 0x55, 0xe8, 0x7e],
    'EFI_SYSTEM_NV_DATA_FV_GUID': [0xfff12b8d, 0x7696, 0x4c8b, 0xa9, 0x85, 0x27, 0x47, 0x7, 0x5b, 0x4f, 0x50],
    'EFI_SYSTEM_NV_DATA_HOB_GUID': [0xd6e5092d, 0xc7b2, 0x4872, 0xaf, 0x66, 0xfd, 0xc0, 0xe6, 0xf9, 0x5e, 0x78],
    'EFI_TAPE_IO_PROTOCOL_GUID': [0x1e93e633, 0xd65a, 0x459e, 0xab, 0x84, 0x93, 0xd9, 0xec, 0x26, 0x6d, 0x18],
    'EFI_TCG_EVENT_HOB_GUID': [0x2e3044ac, 0x879f, 0x490f, 0x97, 0x60, 0xbb, 0xdf, 0xaf, 0x69, 0x5f, 0x50],
    'EFI_TCG_PLATFORM_PROTOCOL_GUID': [0x8c4c9a41, 0xbf56, 0x4627, 0x9e, 0xa, 0xc8, 0x38, 0x6d, 0x66, 0x11, 0x5c],
    'EFI_TCG_PROTOCOL_GUID': [0xf541796d, 0xa62e, 0x4954, 0xa7, 0x75, 0x95, 0x84, 0xf6, 0x1b, 0x9c, 0xdd],
    'EFI_TCP4_PROTOCOL_GUID': [0x65530BC7, 0xA359, 0x410f, 0xB0, 0x10, 0x5A, 0xAD, 0xC7, 0xEC, 0x2B, 0x62],
    'EFI_TCP4_SERVICE_BINDING_PROTOCOL_GUID': [0x00720665, 0x67EB, 0x4a99, 0xBA, 0xF7, 0xD3, 0xC3, 0x3A, 0x1C, 0x7C, 0xC9],
    'EFI_TCP6_PROTOCOL_GUID': [0x46e44855, 0xbd60, 0x4ab7, 0xab, 0x0d, 0xa6, 0x79, 0xb9, 0x44, 0x7d, 0x77],
    'EFI_TCP6_SERVICE_BINDING_PROTOCOL_GUID': [0xec20eb79, 0x6c1a, 0x4664, 0x9a, 0x0d, 0xd2, 0xe4, 0xcc, 0x16, 0xd6, 0x64],
    'EFI_TCP_PROTOCOL_GUID': [0x02b3d5f2, 0xac28, 0x11d3, 0x9a, 0x2d, 0x00, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_TIANO_DECOMPRESS_PROTOCOL_GUID': [0xe84cf29c, 0x191f, 0x4eae, 0x96, 0xe1, 0xf4, 0x6a, 0xec, 0xea, 0xea, 0x0b],
    'EFI_TIMER_ARCH_PROTOCOL_GUID': [0x26baccb3, 0x6f42, 0x11d4, 0xbc, 0xe7, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_TSC_FREQUENCY_GUID': [0xdba6a7e3, 0xbb57, 0x4be7, 0x8a, 0xf8, 0xd5, 0x78, 0xdb, 0x7e, 0x56, 0x87],
    'EFI_UART_DEVICE_PATH_GUID': [0x37499a9d, 0x542f, 0x4c89, 0xa0, 0x26, 0x35, 0xda, 0x14, 0x20, 0x94, 0xe4],
    'EFI_UDP4_PROTOCOL_GUID': [0x3ad9df29, 0x4501, 0x478d, 0xb1, 0xf8, 0x7f, 0x7f, 0xe7, 0x0e, 0x50, 0xf3],
    'EFI_UDP4_SERVICE_BINDING_PROTOCOL_GUID': [0x83f01464, 0x99bd, 0x45e5, 0xb3, 0x83, 0xaf, 0x63, 0x05, 0xd8, 0xe9, 0xe6],
    'EFI_UDP6_PROTOCOL_GUID': [0x4f948815, 0xb4b9, 0x43cb, 0x8a, 0x33, 0x90, 0xe0, 0x60, 0xb3, 0x49, 0x55],
    'EFI_UDP6_SERVICE_BINDING_PROTOCOL_GUID': [0x66ed4721, 0x3c98, 0x4d3e, 0x81, 0xe3, 0xd0, 0x3d, 0xd3, 0x9a, 0x72, 0x54],
    'EFI_UGA_DRAW_PROTOCOL_GUID': [0x982c298b, 0xf4fa, 0x41cb, 0xb8, 0x38, 0x77, 0xaa, 0x68, 0x8f, 0xb8, 0x39],
    'EFI_UGA_IO_PROTOCOL_GUID': [0x61a4d49e, 0x6f68, 0x4f1b, 0xb9, 0x22, 0xa8, 0x6e, 0xed, 0xb, 0x7, 0xa2],
    'EFI_UGA_SPLASH_PROTOCOL_GUID': [0xa45b3a0d, 0x2e55, 0x4c03, 0xad, 0x9c, 0x27, 0xd4, 0x82, 0xb, 0x50, 0x7e],
    'EFI_UNICODE_COLLATION2_PROTOCOL_GUID': [0xa4c751fc, 0x23ae, 0x4c3e, 0x92, 0xe9, 0x49, 0x64, 0xcf, 0x63, 0xf3, 0x49],
    'EFI_UNICODE_COLLATION_PROTOCOL2_GUID': [0xa4c751fc, 0x23ae, 0x4c3e, 0x92, 0xe9, 0x49, 0x64, 0xcf, 0x63, 0xf3, 0x49],
    'EFI_UNICODE_COLLATION_PROTOCOL_GUID': [0x1d85cd7f, 0xf43d, 0x11d2, 0x9a, 0xc, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_UNIX_CONSOLE_GUID': [0xf2cc5d06, 0x8985, 0x11db, 0xbb, 0x19, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_CPU_MODEL_GUID': [0xf2d3b330, 0x8985, 0x11db, 0x8a, 0xa3, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_CPU_SPEED_GUID': [0xf2d74e5a, 0x8985, 0x11db, 0x97, 0x05, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_FILE_SYSTEM_GUID': [0xf2c16b9e, 0x8985, 0x11db, 0x92, 0xc8, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_GOP_GUID': [0xbace07c2, 0x8987, 0x11db, 0xa5, 0x9a, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_IO_PROTOCOL_GUID': [0xf2e23f54, 0x8985, 0x11db, 0xac, 0x79, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_MEMORY_GUID': [0xf2d006cc, 0x8985, 0x11db, 0xa4, 0x72, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_NETWORK_GUID': [0x081603B4, 0x0F1D, 0x4022, 0xB6, 0xFD, 0x4C, 0xE3, 0x5E, 0x09, 0xA1, 0xA6],
    'EFI_UNIX_PHYSICAL_DISKS_GUID': [0xf2bdcc96, 0x8985, 0x11db, 0x87, 0x19, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_SERIAL_PORT_GUID': [0x6d3a727d, 0x66c8, 0x4d19, 0x87, 0xe6, 0x2, 0x15, 0x86, 0x14, 0x90, 0xf3],
    'EFI_UNIX_THUNK_PROTOCOL_GUID': [0xf2e98868, 0x8985, 0x11db, 0x9a, 0x59, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_UGA_GUID': [0xf2c8b80e, 0x8985, 0x11db, 0x93, 0xf1, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_UGA_IO_PROTOCOL_GUID': [0xf2e5e2c6, 0x8985, 0x11db, 0xa1, 0x91, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UNIX_VIRTUAL_DISKS_GUID': [0xf2ba331a, 0x8985, 0x11db, 0xa4, 0x06, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'EFI_UPDATE_DATA_FILE_GUID': [0x283fa2ee, 0x532c, 0x484d, 0x93, 0x83, 0x9f, 0x93, 0xb3, 0x6f, 0xb, 0x7e],
    'EFI_USB2_HC_PROTOCOL_GUID': [0x3e745226, 0x9818, 0x45b6, 0xa2, 0xac, 0xd7, 0xcd, 0xe, 0x8b, 0xa2, 0xbc],
    'EFI_USB_ATAPI_PROTOCOL_GUID': [0x2B2F68DA, 0x0CD2, 0x44cf, 0x8E, 0x8B, 0xBB, 0xA2, 0x0B, 0x1B, 0x5B, 0x75],
    'EFI_USB_BUS_PROTOCOL_GUID': [0x2B2F68CC, 0x0CD2, 0x44cf, 0x8E, 0x8B, 0xBB, 0xA2, 0x0B, 0x1B, 0x5B, 0x75],
    'EFI_USB_HC_PROTOCOL_GUID': [0xf5089266, 0x1aa0, 0x4953, 0x97, 0xd8, 0x56, 0x2f, 0x8a, 0x73, 0xb5, 0x19],
    'EFI_USB_IO_PROTOCOL_GUID': [0x2B2F68D6, 0x0CD2, 0x44cf, 0x8E, 0x8B, 0xBB, 0xA2, 0x0B, 0x1B, 0x5B, 0x75],
    'EFI_USER_CREDENTIAL2_PROTOCOL_GUID': [0xe98adb03, 0xb8b9, 0x4af8, 0xba, 0x20, 0x26, 0xe9, 0x11, 0x4c, 0xbc, 0xe5],
    'EFI_USER_CREDENTIAL_PROTOCOL_GUID': [0x71ee5e94, 0x65b9, 0x45d5, 0x82, 0x1a, 0x3a, 0x4d, 0x86, 0xcf, 0xe6, 0xbe],
    'EFI_USER_INFO_ACCESS_SETUP_ADMIN_GUID': [0x85b75607, 0xf7ce, 0x471e, 0xb7, 0xe4, 0x2a, 0xea, 0x5f, 0x72, 0x32, 0xee],
    'EFI_USER_INFO_ACCESS_SETUP_NORMAL_GUID': [0x1db29ae0, 0x9dcb, 0x43bc, 0x8d, 0x87, 0x5d, 0xa1, 0x49, 0x64, 0xdd, 0xe2],
    'EFI_USER_INFO_ACCESS_SETUP_RESTRICTED_GUID': [0xbdb38125, 0x4d63, 0x49f4, 0x82, 0x12, 0x61, 0xcf, 0x5a, 0x19, 0xa, 0xf8],
    'EFI_USER_MANAGER_PROTOCOL_GUID': [0x6fd5b00c, 0xd426, 0x4283, 0x98, 0x87, 0x6c, 0xf5, 0xcf, 0x1c, 0xb1, 0xfe],
    'EFI_UXIX_SYSTEM_CONFIG_GUID': [0x375ea976, 0x3ccd, 0x4e74, 0xa8, 0x45, 0x26, 0xb9, 0xb3, 0x24, 0xb1, 0x3c],
    'EFI_VARIABLE_ARCH_PROTOCOL_GUID': [0x1e5668e2, 0x8481, 0x11d4, 0xbc, 0xf1, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_VARIABLE_GUID': [0xddcf3616, 0x3275, 0x4164, 0x98, 0xb6, 0xfe, 0x85, 0x70, 0x7f, 0xfe, 0x7d],
    'EFI_VARIABLE_INDEX_TABLE_GUID': [0x8cfdb8c8, 0xd6b2, 0x40f3, 0x8e, 0x97, 0x02, 0x30, 0x7c, 0xc9, 0x8b, 0x7c],
    'EFI_VARIABLE_STORE_PROTOCOL_GUID': [0xf088cd91, 0xa046, 0x11d2, 0x8e, 0x42, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'EFI_VARIABLE_WRITE_ARCH_PROTOCOL_GUID': [0x6441f818, 0x6362, 0x4e44, 0xb5, 0x70, 0x7d, 0xba, 0x31, 0xdd, 0x24, 0x53],
    'EFI_VGA_MINI_PORT_PROTOCOL_GUID': [0xc7735a2f, 0x88f5, 0x4882, 0xae, 0x63, 0xfa, 0xac, 0x8c, 0x8b, 0x86, 0xb3],
    'EFI_VIRTUAL_MEMORY_ACCESS_PROTOCOL_GUID': [0x745d377a, 0xb988, 0x47b2, 0xb1, 0x8f, 0xbb, 0xc8, 0xd, 0xc5, 0x66, 0x98],
    'EFI_VLAN_CONFIG_PROTOCOL_GUID': [0x9e23d768, 0xd2f3, 0x4366, 0x9f, 0xc3, 0x3a, 0x7a, 0xba, 0x86, 0x43, 0x74],
    'EFI_VT_100_GUID': [0xdfa66065, 0xb419, 0x11d3, 0x9a, 0x2d, 0x00, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'EFI_VT_100_PLUS_GUID': [0x7baec70b, 0x57e0, 0x4c76, 0x8e, 0x87, 0x2f, 0x9e, 0x28, 0x08, 0x83, 0x43],
    'EFI_VT_UTF8_GUID': [0xad15a0d6, 0x8bec, 0x4acf, 0xa0, 0x73, 0xd0, 0x1d, 0xe7, 0x7e, 0x2d, 0x88],
    'EFI_WATCHDOG_TIMER_ARCH_PROTOCOL_GUID': [0x665E3FF5, 0x46CC, 0x11d4, 0x9A, 0x38, 0x00, 0x90, 0x27, 0x3F, 0xC1, 0x4D],
    'EFI_WIN_NT_CONSOLE_GUID': [0xba73672c, 0xa5d3, 0x11d4, 0xbd, 0x0, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_CPU_MODEL_GUID': [0xbee9b6ce, 0x2f8a, 0x11d4, 0xbd, 0xd, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_CPU_SPEED_GUID': [0xd4f29055, 0xe1fb, 0x11d4, 0xbd, 0xd, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_FILE_SYSTEM_GUID': [0xc95a935, 0xa006, 0x11d4, 0xbc, 0xfa, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_GOP_GUID': [0x4e11e955, 0xccca, 0x11d4, 0xbd, 0xd, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_IO_PROTOCOL_GUID': [0x96eb4ad6, 0xa32a, 0x11d4, 0xbc, 0xfd, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_MEMORY_GUID': [0x99042912, 0x122a, 0x11d4, 0xbd, 0xd, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_PASS_THROUGH_GUID': [0xcc664eb8, 0x3c24, 0x4086, 0xb6, 0xf6, 0x34, 0xe8, 0x56, 0xbc, 0xe3, 0x6e],
    'EFI_WIN_NT_PHYSICAL_DISKS_GUID': [0xc95a92f, 0xa006, 0x11d4, 0xbc, 0xfa, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_SERIAL_PORT_GUID': [0xc95a93d, 0xa006, 0x11d4, 0xbc, 0xfa, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_SYSTEM_CONFIG_GUID': [0xb347f047, 0xaf8c, 0x490e, 0xac, 0x07, 0x0a, 0xa9, 0xb7, 0xe5, 0x38, 0x58],
    'EFI_WIN_NT_THUNK_PROTOCOL_GUID': [0x58c518b1, 0x76f3, 0x11d4, 0xbc, 0xea, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_UGA_GUID': [0xab248e99, 0xabe1, 0x11d4, 0xbd, 0xd, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_WIN_NT_VIRTUAL_DISKS_GUID': [0xc95a928, 0xa006, 0x11d4, 0xbc, 0xfa, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'EFI_XEN_INFO_GUID': [0xd3b46f3b, 0xd441, 0x1244, 0x9a, 0x12, 0x0, 0x12, 0x27, 0x3f, 0xc1, 0x4d],
    'EMBEDDED_DEVICE_PROTOCOL_GUID': [0xbf4b9d10, 0x13ec, 0x43dd, 0x88, 0x80, 0xe9, 0xb, 0x71, 0x8f, 0x27, 0xde],
    'EMBEDDED_EXTERNAL_DEVICE_PROTOCOL_GUID': [0x735F8C64, 0xD696, 0x44D0, 0xBD, 0xF2, 0x44, 0x7F, 0xD0, 0x5A, 0x54, 0x06],
    'EMU_BLOCK_IO_PROTOCOL_GUID': [0x6888A4AE, 0xAFCE, 0xE84B, 0x91, 0x02, 0xF7, 0xB9, 0xDA, 0xE6, 0xA0, 0x30],
    'EMU_GRAPHICS_WINDOW_PROTOCOL_GUID': [0x30FD316A, 0x6728, 0x2E41, 0xA6, 0x90, 0x0D, 0x13, 0x33, 0xD8, 0xCA, 0xC1],
    'EMU_IO_THUNK_PROTOCO_GUID': [0x453368F6, 0x7C85, 0x434A, 0xA9, 0x8A, 0x72, 0xD1, 0xB7, 0xFF, 0xA9, 0x26],
    'EMU_SNP_PROTOCOL_GUID': [0xFD5FBE54, 0x8C35, 0xB345, 0x8A, 0x0F, 0x7A, 0xC8, 0xA5, 0xFD, 0x05, 0x21],
    'EMU_THUNK_PPI_GUID': [0xB958B78C, 0x1D3E, 0xEE40, 0x8B, 0xF4, 0xF0, 0x63, 0x2D, 0x06, 0x39, 0x16],
    'EMU_THUNK_PROTOCOL_GUID': [0x5CF32E0B, 0x8EDF, 0x2E44, 0x9C, 0xDA, 0x93, 0x20, 0x5E, 0x99, 0xEC, 0x1C],
    'EXTENDED_SAL_BOOT_SERVICE_PROTOCOL_GUID': [0xde0ee9a4, 0x3c7a, 0x44f2, 0xb7, 0x8b, 0xe3, 0xcc, 0xd6, 0x9c, 0x3a, 0xf7],
    'EXTENDED_SAL_BOOT_SERVICE_PROTOCOL_GUID': [0xde0ee9a4, 0x3c7a, 0x44f2, 0xb7, 0x8b, 0xe3, 0xcc, 0xd6, 0x9c, 0x3a, 0xf7],
    'FFS_GUID': [0xac05bf33, 0x995a, 0x4ed4, 0xaa, 0xb8, 0xef, 0x7a, 0xe8, 0xf, 0x5c, 0xb0],
    'FILE_EXPLORE_FORMSET_GUID': [0x1f2d63e1, 0xfebd, 0x4dc7, 0x9c, 0xc5, 0xba, 0x2b, 0x1c, 0xef, 0x9c, 0x5b],
    'FILE_GUID': [0xcbd2e4d5, 0x7068, 0x4ff5, 0xb4, 0x62, 0x98, 0x22, 0xb4, 0xad, 0x8d, 0x60],
    'FORM_BROWSER_EXTENSION_PROTOCOL_GUID': [0x1f73b18d, 0x4630, 0x43c1, 0xa1, 0xde, 0x6f, 0x80, 0x85, 0x5d, 0x7d, 0xa4],
    'FRAMEWORK_BDS_FRONTPAGE_FORMSET_GUID': [0x9e0c30bc, 0x3f06, 0x4ba6, 0x82, 0x88, 0x9, 0x17, 0x9b, 0x85, 0x5d, 0xbe],
    'FRAMEWORK_EFI_FIRMWARE_VOLUME_BLOCK_PROTOCOL_GUID': [0xDE28BC59, 0x6228, 0x41BD, 0xBD, 0xF6, 0xA3, 0xB9, 0xAD, 0xB5, 0x8D, 0xA1],
    'FRAMEWORK_EFI_MP_SERVICES_PROTOCOL_GUID': [0xf33261e7, 0x23cb, 0x11d5, 0xbd, 0x5c, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'FRONT_PAGE_FORMSET_GUID': [0x9e0c30bc, 0x3f06, 0x4ba6, 0x82, 0x88, 0x9, 0x17, 0x9b, 0x85, 0x5d, 0xbe],
    'HANDLE_PARSING_HII_GUID': [0xb8969637, 0x81de, 0x43af, 0xbc, 0x9a, 0x24, 0xd9, 0x89, 0x13, 0xf2, 0xf6],
    'HD_BOOT_DEVICE_PATH_VARIABLE_GUID': [0xfab7e9e1, 0x39dd, 0x4f2b, 0x84, 0x8, 0xe2, 0xe, 0x90, 0x6c, 0xb6, 0xde],
    'HII_RESOURCE_SAMPLE_FORM_SET_GUID': [0x4f4ef7f0, 0xaa29, 0x4ce9, 0xba, 0x41, 0x64, 0x3e, 0x1, 0x23, 0xa9, 0x9f],
    'HOB_LIST_GUID': [0x7739f24c, 0x93d7, 0x11d4, 0x9a, 0x3a, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'HOT_PLUG_DEVICE_GUID': [0x220ac432, 0x1d43, 0x49e5, 0xa7, 0x4f, 0x4c, 0x9d, 0xa6, 0x7a, 0xd2, 0x3b],
    'IDLE_LOOP_EVENT_GUID': [0x3c8d294c, 0x5fc3, 0x4451, 0xbb, 0x31, 0xc4, 0xc0, 0x32, 0x29, 0x5e, 0x6c],
    'INTEL_FRAMEWORK_MODULEPKG_TOKEN_SPACE_GUID': [0xD3705011, 0xBC19, 0x4af7, 0xBE, 0x16, 0xF6, 0x80, 0x30, 0x37, 0x8C, 0x15],
    'IP4_ISCSI_CONFIG_GUID': [0x6456ed61, 0x3579, 0x41c9, 0x8a, 0x26, 0x0a, 0x0b, 0xd6, 0x2b, 0x78, 0xfc],
    'IP6_CONFIG_NVDATA_GUID': [0x2eea107, 0x98db, 0x400e, 0x98, 0x30, 0x46, 0xa, 0x15, 0x42, 0xd7, 0x99],
    'ISCSI_CHAP_AUTH_INFO_GUID': [0x786ec0ac, 0x65ae, 0x4d1b, 0xb1, 0x37, 0xd, 0x11, 0xa, 0x48, 0x37, 0x97],
    'ISCSI_CONFIG_GUID': [0x4b47d616, 0xa8d6, 0x4552, 0x9d, 0x44, 0xcc, 0xad, 0x2e, 0xf, 0x4c, 0xf9],
    'ISCSI_V4_PRIVATE_GUID': [0xfa3cde4c, 0x87c2, 0x427d, 0xae, 0xde, 0x7d, 0xd0, 0x96, 0xc8, 0x8c, 0x58],
    'ISCSI_V6_PRIVATE_GUID': [0x28be27e5, 0x66cc, 0x4a31, 0xa3, 0x15, 0xdb, 0x14, 0xc3, 0x74, 0x4d, 0x85],
    'LAST_ENUM_LANGUAGE_GUID': [0xe8c545b, 0xa2ee, 0x470d, 0x8e, 0x26, 0xbd, 0xa1, 0xa1, 0x3c, 0xa, 0xa3],
    'LDR_MEMORY_DESCRIPTOR_GUID': [0x7701d7e5, 0x7d1d, 0x4432, 0xa4, 0x68, 0x67, 0x3d, 0xab, 0x8a, 0xde, 0x60],
    'LOAD_FILE_PROTOCOL_GUID': [0x56EC3091, 0x954C, 0x11d2, 0x8E, 0x3F, 0x00, 0xA0, 0xC9, 0x69, 0x72, 0x3B],
    'LOCAL_EFI_WIN_NT_BUS_DRIVER_IO_PROTOCOL_GUID': [0x96eb4ad6, 0xa32a, 0x11d4, 0xbc, 0xfd, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'LOCAL_EFI_WIN_NT_SERIAL_PORT_GUID': [0xc95a93d, 0xa006, 0x11d4, 0xbc, 0xfa, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'LOCAL_EFI_WIN_NT_THUNK_PROTOCOL_GUID': [0x58c518b1, 0x76f3, 0x11d4, 0xbc, 0xea, 0x0, 0x80, 0xc7, 0x3c, 0x88, 0x81],
    'LZMAF86_CUSTOM_DECOMPRESS_GUID': [0xD42AE6BD, 0x1352, 0x4bfb, 0x90, 0x9A, 0xCA, 0x72, 0xA6, 0xEA, 0xE8, 0x89],
    'LZMA_CUSTOM_DECOMPRESS_GUID': [0xEE4E5898, 0x3914, 0x4259, 0x9D, 0x6E, 0xDC, 0x7B, 0xD7, 0x94, 0x03, 0xCF],
    'MDEMODULEPKG_TOKEN_SPACE_GUID': [0xA1AFF049, 0xFDEB, 0x442a, 0xB3, 0x20, 0x13, 0xAB, 0x4C, 0xB7, 0x2B, 0xBC],
    'MDEPKG_TOKEN_SPACE_GUID': [0x914AEBE7, 0x4635, 0x459b, 0xAA, 0x1C, 0x11, 0xE2, 0x19, 0xB0, 0x3A, 0x10],
    'MEMORY_ONLY_RESET_CONTROL_GUID': [0xe20939be, 0x32d4, 0x41be, 0xa1, 0x50, 0x89, 0x7f, 0x85, 0xd4, 0x98, 0x29],
    'MEMORY_STATUS_CODE_RECORD_GUID': [0x60cc026, 0x4c0d, 0x4dda, 0x8f, 0x41, 0x59, 0x5f, 0xef, 0x0, 0xa5, 0x2],
    'MTC_VENDOR_GUID': [0xeb704011, 0x1402, 0x11d3, 0x8e, 0x77, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'MY_GUID': [0x12345678, 0xAABB, 0xCCDD, 0xEE, 0xFF, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66],
    'NT_FWH_PPI_GUID': [0x4e76928f, 0x50ad, 0x4334, 0xb0, 0x6b, 0xa8, 0x42, 0x13, 0x10, 0x8a, 0x57],
    'PCATCHIPSET_TOKEN_SPACE_GUID': [0x326ae723, 0xae32, 0x4589, 0x98, 0xb8, 0xca, 0xc2, 0x3c, 0xdc, 0xc1, 0xb1],
    'PCD_DATABASE_HOB_GUID': [0xEA296D92, 0x0B69, 0x423C, 0x8C, 0x28, 0x33, 0xB4, 0xE0, 0xA9, 0x12, 0x68],
    'PCD_PPI_GUID': [0x6e81c58, 0x4ad7, 0x44bc, 0x83, 0x90, 0xf1, 0x2, 0x65, 0xf7, 0x24, 0x80],
    'PCD_PROTOCOL_GUID': [0x11b34006, 0xd85b, 0x4d0a, 0xa2, 0x90, 0xd5, 0xa5, 0x71, 0x31, 0xe, 0xf7],
    'PE32_IMAGE_PROTOCOL_GUID': [0x5cb5c776, 0x60d5, 0x45ee, 0x88, 0x3c, 0x45, 0x27, 0x8, 0xcd, 0x74, 0x3f],
    'PEI_ATA_CONTROLLER_PPI_GUID': [0xa45e60d1, 0xc719, 0x44aa, 0xb0, 0x7a, 0xaa, 0x77, 0x7f, 0x85, 0x90, 0x6d],
    'PEI_BASE_MEMORY_TEST_GUID': [0xb6ec423c, 0x21d2, 0x490d, 0x85, 0xc6, 0xdd, 0x58, 0x64, 0xea, 0xa6, 0x74],
    'PEI_BLOCK_IO_PPI_GUID': [0x695d8aa1, 0x42ee, 0x4c46, 0x80, 0x5c, 0x6e, 0xa6, 0xbc, 0xe7, 0x99, 0xe3],
    'PEI_BOOT_SCRIPT_EXECUTER_PPI_GUID': [0xabd42895, 0x78cf, 0x4872, 0x84, 0x44, 0x1b, 0x5c, 0x18, 0x0b, 0xfb, 0xff],
    'PEI_CAPSULE_PPI_GUID': [0x3acf33ee, 0xd892, 0x40f4, 0xa2, 0xfc, 0x38, 0x54, 0xd2, 0xe1, 0x32, 0x3d],
    'PEI_CPU_IO_PPI_GUID': [0xe6af1f7b, 0xfc3f, 0x46da, 0xa8, 0x28, 0xa3, 0xb4, 0x57, 0xa4, 0x42, 0x82],
    'PEI_END_OF_PEI_PHASE_PPI_GUID': [0x605EA650, 0xC65C, 0x42e1, 0xBA, 0x80, 0x91, 0xA5, 0x2A, 0xB6, 0x18, 0xC6],
    'PEI_FLASH_MAP_PPI_GUID': [0xf34c2fa0, 0xde88, 0x4270, 0x84, 0x14, 0x96, 0x12, 0x22, 0xf4, 0x52, 0x1c],
    'PEI_IN_MEMORY_GUID': [0x643b8786, 0xb417, 0x48d2, 0x8f, 0x5e, 0x78, 0x19, 0x93, 0x1c, 0xae, 0xd8],
    'PEI_LOCK_PHYSICAL_PRESENCE_PPI_GUID': [0xef9aefe5, 0x2bd3, 0x4031, 0xaf, 0x7d, 0x5e, 0xfe, 0x5a, 0xbb, 0x9a, 0xd],
    'PEI_NT_THUNK_GUID': [0x98c281e5, 0xf906, 0x43dd, 0xa9, 0x2b, 0xb0, 0x3, 0xbf, 0x27, 0x65, 0xda],
    'PEI_NT_THUNK_PPI_GUID': [0x98c281e5, 0xf906, 0x43dd, 0xa9, 0x2b, 0xb0, 0x3, 0xbf, 0x27, 0x65, 0xda],
    'PEI_OPERATOR_PRESENCE_PPI_GUID': [0x20a7378c, 0xaa83, 0x4ce1, 0x82, 0x1f, 0x47, 0x40, 0xee, 0x1b, 0x3f, 0x9f],
    'PEI_PCI_CFG_PPI_GUID': [0xe1f2eba0, 0xf7b9, 0x4a26, 0x86, 0x20, 0x13, 0x12, 0x21, 0x64, 0x2a, 0x90],
    'PEI_PERMANENT_MEMORY_INSTALLED_PPI_GUID': [0xf894643d, 0xc449, 0x42d1, 0x8e, 0xa8, 0x85, 0xbd, 0xd8, 0xc6, 0x5b, 0xde],
    'PEI_READ_ONLY_VARIABLE_ACCESS_PPI_GUID': [0x3cdc90c6, 0x13fb, 0x4a75, 0x9e, 0x79, 0x59, 0xe9, 0xdd, 0x78, 0xb9, 0xfa],
    'PEI_RESET_PPI_GUID': [0xef398d58, 0x9dfd, 0x4103, 0xbf, 0x94, 0x78, 0xc6, 0xf4, 0xfe, 0x71, 0x2f],
    'PEI_S3_RESUME_PPI_GUID': [0x4426CCB2, 0xE684, 0x4a8a, 0xAE, 0x40, 0x20, 0xD4, 0xB0, 0x25, 0xB7, 0x10],
    'PEI_SECURITY_PPI_GUID': [0x1388066e, 0x3a57, 0x4efa, 0x98, 0xf3, 0xc1, 0x2f, 0x3a, 0x95, 0x8a, 0x29],
    'PEI_SEC_PERFORMANCE_PPI_GUID': [0x0ecc666b, 0x4662, 0x47f9, 0x9d, 0xd5, 0xd0, 0x96, 0xff, 0x7d, 0xa4, 0x9e],
    'PEI_SMBUS2_PPI_GUID': [0x9ca93627, 0xb65b, 0x4324, 0xa2, 0x2, 0xc0, 0xb4, 0x61, 0x76, 0x45, 0x43],
    'PEI_SMBUS_PPI_GUID': [0xabd42895, 0x78cf, 0x4872, 0x84, 0x44, 0x1b, 0x5c, 0x18, 0xb, 0xfb, 0xda],
    'PEI_SMM_ACCESS_PPI_GUID': [0x268f33a9, 0xcccd, 0x48be, 0x88, 0x17, 0x86, 0x5, 0x3a, 0xc3, 0x2e, 0xd6],
    'PEI_SMM_CONTROL_PPI_GUID': [0x61c68702, 0x4d7e, 0x4f43, 0x8d, 0xef, 0xa7, 0x43, 0x5, 0xce, 0x74, 0xc5],
    'PEI_STALL_PPI_GUID': [0x1f4c6f90, 0xb06b, 0x48d8, 0xa2, 0x01, 0xba, 0xe5, 0xf1, 0xcd, 0x7d, 0x56],
    'PEI_STATUS_CODE_MEMORY_PPI_GUID': [0x26f8ab01, 0xd3cd, 0x489c, 0x98, 0x4f, 0xdf, 0xde, 0xf7, 0x68, 0x39, 0x5b],
    'PEI_STATUS_CODE_PPI_GUID': [0x229832d3, 0x7a30, 0x4b36, 0xb8, 0x27, 0xf4, 0xc, 0xb7, 0xd4, 0x54, 0x36],
    'PEI_TPM_INITIALIZED_PPI_GUID': [0xe9db0d58, 0xd48d, 0x47f6, 0x9c, 0x6e, 0x6f, 0x40, 0xe8, 0x6c, 0x7b, 0x41],
    'PEI_UNIX_AUTOSCAN_PPI_GUID': [0xf2ed3d14, 0x8985, 0x11db, 0xb0, 0x57, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'PEI_UNIX_THUNK_PPI_GUID': [0xf2f830f2, 0x8985, 0x11db, 0x80, 0x6b, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'PEI_USB2_HOST_CONTROLLER_PPI_GUID': [0xa7d09fe1, 0x74d4, 0x4ba5, 0x84, 0x7c, 0x12, 0xed, 0x5b, 0x19, 0xad, 0xe4],
    'PEI_USB_CONTROLLER_PPI_GUID': [0x3bc1f6de, 0x693e, 0x4547, 0xa3, 0x0, 0x21, 0x82, 0x3c, 0xa4, 0x20, 0xb2],
    'PEI_USB_HOST_CONTROLLER_PPI_GUID': [0x652b38a9, 0x77f4, 0x453f, 0x89, 0xd5, 0xe7, 0xbd, 0xc3, 0x52, 0xfc, 0x53],
    'PEI_USB_IO_PPI_GUID': [0x7c29785c, 0x66b9, 0x49fc, 0xb7, 0x97, 0x1c, 0xa5, 0x55, 0xe, 0xf2, 0x83],
    'PERFORMANCEPKG_TOKEN_SPACE_GUID': [0x669346ef, 0xFDad, 0x4aeb, 0x08, 0xa6, 0x21, 0x46, 0x2d, 0x3f, 0xef, 0x7d],
    'PERFORMANCE_EX_PROTOCOL_GUID': [0x1ea81bec, 0xf01a, 0x4d98, 0xa2, 0x1, 0x4a, 0x61, 0xce, 0x2f, 0xc0, 0x22],
    'PERFORMANCE_PROTOCOL_GUID': [0x76b6bdfa, 0x2acd, 0x4462, 0x9E, 0x3F, 0xcb, 0x58, 0xC9, 0x69, 0xd9, 0x37],
    'PE_COFF_LOADER_PROTOCOL_GUID': [0xB323179B, 0x97FB, 0x477E, 0xB0, 0xFE, 0xD8, 0x85, 0x91, 0xFA, 0x11, 0xAB],
    'PLAT_OVER_MNGR_GUID': [0x8614567d, 0x35be, 0x4415, 0x8d, 0x88, 0xbd, 0x7d, 0xc, 0x9c, 0x70, 0xc0],
    'PRE_PI_EXTRACT_GUIDED_SECTION_DATA_GUID': [0x385A982C, 0x2F49, 0x4043, 0xA5, 0x1E, 0x49, 0x01, 0x02, 0x5C, 0x8B, 0x6B],
    'PWD_CREDENTIAL_PROVIDER_GUID': [0x78b9ec8b, 0xc000, 0x46c5, 0xac, 0x93, 0x24, 0xa0, 0xc1, 0xbb, 0x0, 0xce],
    'RECOVERY_ON_DATA_CD_GUID': [0x5cac0099, 0x0dc9, 0x48e5, 0x80, 0x68, 0xbb, 0x95, 0xf5, 0x40, 0x0a, 0x9f],
    'RECOVERY_ON_FAT_FLOPPY_DISK_GUID': [0x2e3d2e75, 0x9b2e, 0x412d, 0xb4, 0xb1, 0x70, 0x41, 0x6b, 0x87, 0x0, 0xff],
    'RECOVERY_ON_FAT_IDE_DISK_GUID': [0xb38573b6, 0x6200, 0x4ac5, 0xb5, 0x1d, 0x82, 0xe6, 0x59, 0x38, 0xd7, 0x83],
    'RECOVERY_ON_FAT_USB_DISK_GUID': [0x0ffbce19, 0x324c, 0x4690, 0xa0, 0x09, 0x98, 0xc6, 0xae, 0x2e, 0xb1, 0x86],
    'SAL_SYSTEM_TABLE_GUID': [0xeb9d2d32, 0x2d88, 0x11d3, 0x9a, 0x16, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'SECUREBOOT_CONFIG_FORM_SET_GUID': [0x5daf50a5, 0xea81, 0x4de2, 0x8f, 0x9b, 0xca, 0xbd, 0xa9, 0xcf, 0x5c, 0x14],
    'SECURITYPKG_TOKEN_SPACE_GUID': [0xd3fb176, 0x9569, 0x4d51, 0xa3, 0xef, 0x7d, 0x61, 0xc6, 0x4f, 0xea, 0xba],
    'SHELLPKG_SHELL_ENV2_EXT_GUID': [0xd2c18636, 0x40e5, 0x4eb5, 0xa3, 0x1b, 0x36, 0x69, 0x5f, 0xd4, 0x2c, 0x87],
    'SHELL_ALIAS_VARIABLE_GUID': [0x0053d9d6, 0x2659, 0x4599, 0xa2, 0x6b, 0xef, 0x45, 0x36, 0xe6, 0x31, 0xa9],
    'SHELL_DEBUG1_HII_GUID': [0x25f200aa, 0xd3cb, 0x470a, 0xbf, 0x51, 0xe7, 0xd1, 0x62, 0xd2, 0x2e, 0x6f],
    'SHELL_DRIVER1_HII_GUID': [0xaf0b742, 0x63ec, 0x45bd, 0x8d, 0xb6, 0x71, 0xad, 0x7f, 0x2f, 0xe8, 0xe8],
    'SHELL_ENVIRONMENT_PROTOCOL_GUID': [0x47c7b221, 0xc42a, 0x11d2, 0x8e, 0x57, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'SHELL_INSTALL1_HII_GUID': [0x7d574d54, 0xd364, 0x4d4a, 0x95, 0xe3, 0x49, 0x45, 0xdb, 0x7a, 0xd3, 0xee],
    'SHELL_INTERFACE_PROTOCOL_GUID': [0x47c7b223, 0xc42a, 0x11d2, 0x8e, 0x57, 0x0, 0xa0, 0xc9, 0x69, 0x72, 0x3b],
    'SHELL_LEVEL1_HII_GUID': [0xdec5daa4, 0x6781, 0x4820, 0x9c, 0x63, 0xa7, 0xb0, 0xe4, 0xf1, 0xdb, 0x31],
    'SHELL_LEVEL2_HII_GUID': [0xf95a7ccc, 0x4c55, 0x4426, 0xa7, 0xb4, 0xdc, 0x89, 0x61, 0x95, 0xb, 0xae],
    'SHELL_LEVEL3_HII_GUID': [0x4344558d, 0x4ef9, 0x4725, 0xb1, 0xe4, 0x33, 0x76, 0xe8, 0xd6, 0x97, 0x4f],
    'SHELL_MAP_GUID': [0x51271e13, 0x7de3, 0x43af, 0x8b, 0xc2, 0x71, 0xad, 0x3b, 0x82, 0x43, 0x25],
    'SHELL_NETWORK1_HII_GUID': [0xf3d301bb, 0xf4a5, 0x45a8, 0xb0, 0xb7, 0xfa, 0x99, 0x9c, 0x62, 0x37, 0xae],
    'SHELL_VARIABLE_GUID': [0x158def5a, 0xf656, 0x419c, 0xb0, 0x27, 0x7a, 0x31, 0x92, 0xc0, 0x79, 0xd2],
    'SMBIOS_TABLE_GUID': [0xeb9d2d31, 0x2d88, 0x11d3, 0x9a, 0x16, 0x0, 0x90, 0x27, 0x3f, 0xc1, 0x4d],
    'SMM_COMMUNICATE_HEADER_GUID': [0xf328e36c, 0x23b6, 0x4a95, 0x85, 0x4b, 0x32, 0xe1, 0x95, 0x34, 0xcd, 0x75],
    'SMM_PERFORMANCE_EX_PROTOCOL_GUID': [0x931fc048, 0xc71d, 0x4455, 0x89, 0x30, 0x47, 0x6, 0x30, 0xe3, 0xe, 0xe5],
    'SMM_PERFORMANCE_PROTOCOL_GUID': [0xf866226a, 0xeaa5, 0x4f5a, 0xa9, 0xa, 0x6c, 0xfb, 0xa5, 0x7c, 0x58, 0x8e],
    'STATUS_CODE_CALLBACK_GUID': [0xe701458c, 0x4900, 0x4ca5, 0xb7, 0x72, 0x3d, 0x37, 0x94, 0x9f, 0x79, 0x27],
    'SYSTEM_ROM_FILE_GUID': [0x1547B4F3, 0x3E8A, 0x4FEF, 0x81, 0xC8, 0x32, 0x8E, 0xD6, 0x47, 0xAB, 0x1A],
    'TCG_CONFIG_FORM_SET_GUID': [0xb0f901e4, 0xc424, 0x45de, 0x90, 0x81, 0x95, 0xe2, 0xb, 0xde, 0x6f, 0xb5],
    'TEMPORARY_RAM_SUPPORT_PPI_GUID': [0xdbe23aa9, 0xa345, 0x4b97, 0x85, 0xb6, 0xb2, 0x26, 0xf1, 0x61, 0x73, 0x89],
    'TIANO_CUSTOM_DECOMPRESS_GUID': [0xA31280AD, 0x481E, 0x41B6, 0x95, 0xE8, 0x12, 0x7F, 0x4C, 0x98, 0x47, 0x79],
    'UNIX_FWH_PPI_GUID': [0xf2f0dc30, 0x8985, 0x11db, 0xa1, 0x5b, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'UNIX_PEI_LOAD_FILE_GUID': [0xf2f48768, 0x8985, 0x11db, 0xb8, 0xda, 0x00, 0x40, 0xd0, 0x2b, 0x18, 0x35],
    'UNKNOWN_DEVICE_GUID': [0xcf31fac5, 0xc24e, 0x11d2, 0x85, 0xf3, 0x0, 0xa0, 0xc9, 0x3e, 0xc9, 0x3b],
    'USB_CREDENTIAL_PROVIDER_GUID': [0xd0849ed1, 0xa88c, 0x4ba6, 0xb1, 0xd6, 0xab, 0x50, 0xe2, 0x80, 0xb7, 0xa9],
    'USB_KEYBOARD_LAYOUT_PACKAGE_GUID': [0xc0f3b43, 0x44de, 0x4907, 0xb4, 0x78, 0x22, 0x5f, 0x6f, 0x62, 0x89, 0xdc],
    'USER_IDENTIFY_MANAGER_GUID': [0x3ccd3dd8, 0x8d45, 0x4fed, 0x96, 0x2d, 0x2b, 0x38, 0xcd, 0x82, 0xb3, 0xc4],
    'USER_PROFILE_MANAGER_GUID': [0xc35f272c, 0x97c2, 0x465a, 0xa2, 0x16, 0x69, 0x6b, 0x66, 0x8a, 0x8c, 0xfe],
    'VIRTUAL_UNCACHED_PAGES_PROTOCOL_GUID': [0xAD651C7D, 0x3C22, 0x4DBF, 0x92, 0xe8, 0x38, 0xa7, 0xcd, 0xae, 0x87, 0xb2],
    'VLAN_CONFIG_FORM_SET_GUID': [0xd79df6b0, 0xef44, 0x43bd, 0x97, 0x97, 0x43, 0xe9, 0x3b, 0xcf, 0x5f, 0xa8],
}
