// Copyright (c) 2016 Daniel Grunwald
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy of this
// software and associated documentation files (the "Software"), to deal in the Software
// without restriction, including without limitation the rights to use, copy, modify, merge,
// publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons
// to whom the Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in all copies or
// substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
// INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
// FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

using System.Collections.Generic;
using System.Linq;
using ICSharpCode.Decompiler.FlowAnalysis;
using ICSharpCode.Decompiler.TypeSystem;

namespace ICSharpCode.Decompiler.IL.Transforms
{
	/// <summary>
	/// Remove <c>HasInitialValue</c> from locals that are definitely assigned before every use
	/// (=the initial value is a dead store).
	/// 
	/// In yield return generators, additionally removes dead 'V = null;' assignments.
	/// 
	/// Additionally infers IType of stack slots that have StackType.Ref
	/// </summary>
	public class RemoveDeadVariableInit : IILTransform
	{
		public void Run(ILFunction function, ILTransformContext context)
		{
			ResetHasInitialValueFlag(function, context);
			// Remove dead stores to variables that are never read from.
			// If the stored value has some side-effect, the value is unwrapped.
			// This is necessary to remove useless stores generated by some compilers, e.g., the F# compiler.
			// In yield return + async, the C# compiler tends to store null/default(T) to variables
			// when the variable goes out of scope.
			if (function.IsAsync || function.IsIterator || context.Settings.RemoveDeadStores) {
				var variableQueue = new Queue<ILVariable>(function.Variables);
				while (variableQueue.Count > 0) {
					var v = variableQueue.Dequeue();
					if (v.Kind != VariableKind.Local && v.Kind != VariableKind.StackSlot)
						continue;
					// Skip variables that are captured in a mcs yield state-machine
					// loads of these will only be visible after DelegateConstruction step.
					if (function.StateMachineCompiledWithMono && v.StateMachineField != null)
						continue;
					if (v.LoadCount != 0 || v.AddressCount != 0)
						continue;
					foreach (var stloc in v.StoreInstructions.OfType<StLoc>().ToArray()) {
						if (stloc.Parent is Block block) {
							if (SemanticHelper.IsPure(stloc.Value.Flags)) {
								block.Instructions.Remove(stloc);
							} else {
								stloc.ReplaceWith(stloc.Value);
							}
							if (stloc.Value is LdLoc ldloc) {
								variableQueue.Enqueue(ldloc.Variable);
							}
						}
					}
				}
			}

			// Try to infer IType of stack slots that are of StackType.Ref:
			foreach (var v in function.Variables) {
				if (v.Kind == VariableKind.StackSlot && v.StackType == StackType.Ref && v.AddressCount == 0) {
					IType newType = null;
					// Multiple store are possible in case of (c ? ref a : ref b) += 1, for example.
					foreach (var stloc in v.StoreInstructions.OfType<StLoc>()) {
						var inferredType = stloc.Value.InferType(context.TypeSystem);
						// cancel, if types of values do not match exactly
						if (newType != null && !newType.Equals(inferredType)) {
							newType = SpecialType.UnknownType;
							break;
						}
						newType = inferredType;
					}
					// Only overwrite existing type, if a "better" type was found.
					if (newType != null && newType != SpecialType.UnknownType)
						v.Type = newType;
				}
			}
		}

		internal static void ResetHasInitialValueFlag(ILFunction function, ILTransformContext context)
		{
			var visitor = new DefiniteAssignmentVisitor(function, context.CancellationToken);
			function.AcceptVisitor(visitor);
			foreach (var v in function.Variables) {
				if (v.Kind != VariableKind.Parameter && !visitor.IsPotentiallyUsedUninitialized(v)) {
					v.HasInitialValue = false;
				}
			}
		}
	}
}
